#' Hierarchical Clustering - Single linkage
#'
#' @description A function that performs hierarchical clustering with single linkage.
#' It can also print the clustering steps and display a dendrogram.
#'
#' @param data Numerical matrix or data frame of observations (rows = observations, columns = variables).
#' @param metric Distance metric to be used (default: "euclidean").
#' @param print.steps If TRUE, the algorithm's steps are printed.
#' @param plot If TRUE, a dendrogram is plotted.
#' @param label.names If TRUE, uses the row names as labels in the dendrogram.
#'
#' @return object of class "hclust".
#' @export
#' @examples
#' y1 <- c(1, 2, 1, 0); y2 <- c(2, 1, 0, 2)
#' y3 <- c(8, 8, 9, 7); y4 <- c(6, 9, 8, 9)
#' Data <- rbind(y1, y2, y3, y4)
#' hc <- hclust_single(Data, metric = "euclidean",
#'                     print.steps = TRUE,
#'                     plot = TRUE,
#'                     label.names = TRUE)


hclust_single <- function(data, metric = "euclidean",
                                      print.steps = TRUE,
                                      plot = TRUE,
                                      label.names = TRUE) {
  # checagens
  if (is.data.frame(data)) data <- as.matrix(data)
  if (!is.matrix(data) || !is.numeric(data)) stop("The data must be a numeric matrix or data frame.")
  n <- nrow(data)
  if (n < 2) stop("A minimum of two observations is needed.")

  # distancias entre todas as observacoes
  base_dist <- as.matrix(dist(data, method = metric))

  # inicializa clusters
  clusters <- lapply(seq_len(n), function(i) i)
  cluster_ids <- -seq_len(n)

  merge_mat <- matrix(0L, nrow = n - 1, ncol = 2)
  heights <- numeric(n - 1)

  if (print.steps) {
    cat("Start: each observation is a cluster\n")
    for (i in seq_along(clusters)) {
      membros <- if (label.names) rownames(data)[clusters[[i]]] else clusters[[i]]
      cat(sprintf(" [%s] members: (%s)\n", cluster_ids[i], paste(membros, collapse = ",")))
    }
    cat("\n")
  }

  for (k in seq_len(n - 1)) {
    # matriz de distancias entre clusters (single linkage)
    m <- length(clusters)
    dmat <- matrix(0, nrow = m, ncol = m)
    for (i in seq_len(m)) {
      for (j in seq_len(m)) {
        if (i == j) {
          dmat[i, j] <- 0
        } else {
          ai <- clusters[[i]]
          bj <- clusters[[j]]
          dmat[i, j] <- min(base_dist[ai, bj, drop = FALSE])
        }
      }
    }
    rownames(dmat) <- colnames(dmat) <- as.character(cluster_ids)

    if (print.steps) {
      cat(sprintf("... Step %d ...\n", k))
      cat("Minimum distance matrix:\n")
      print(round(dmat, 6))
    }

    # escolher par com menor distancia positiva
    dmat[dmat == 0] <- Inf
    minpos <- which(dmat == min(dmat), arr.ind = TRUE)[1, ]
    i <- minpos[1]; j <- minpos[2]

    merge_mat[k, ] <- c(cluster_ids[i], cluster_ids[j])
    heights[k] <- dmat[i, j]

    if (print.steps) {
      membros_i <- if (label.names) rownames(data)[clusters[[i]]] else clusters[[i]]
      membros_j <- if (label.names) rownames(data)[clusters[[j]]] else clusters[[j]]
      cat(sprintf(" -> Merging clusters %s (%s) and %s (%s), dist = %g\n\n",
                  cluster_ids[i], paste(membros_i, collapse = ","),
                  cluster_ids[j], paste(membros_j, collapse = ","),
                  round(dmat[i, j], 6)))
    }

    # formar novo cluster
    new_cluster <- c(clusters[[i]], clusters[[j]])
    clusters <- clusters[-c(i, j)]
    cluster_ids <- cluster_ids[-c(i, j)]
    clusters[[length(clusters) + 1]] <- new_cluster
    cluster_ids <- c(cluster_ids, k)
  }

  # ordem final
  traverse <- function(node) {
    if (node < 0) return(-node)
    left <- merge_mat[node, 1]
    right <- merge_mat[node, 2]
    c(traverse(left), traverse(right))
  }
  order_vec <- traverse(n - 1)

  labels <- if (!is.null(rownames(data))) rownames(data) else as.character(seq_len(n))

  hc <- list(merge = merge_mat,
             height = heights,
             order = as.integer(order_vec),
             labels = labels,
             method = paste0("single linkage (", metric, ")"),
             call = match.call())
  class(hc) <- "hclust"

  if (plot) {
    plot(hc, main = paste0("Dendrogram - Single Linkage (", metric, ")"))
  }

  invisible(hc)
}

