% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kkw.R
\name{rkkw}
\alias{rkkw}
\title{Random Number Generation for the kkw Distribution}
\usage{
rkkw(n, alpha = 1, beta = 1, delta = 0, lambda = 1)
}
\arguments{
\item{n}{Number of observations. If \code{length(n) > 1}, the length is
taken to be the number required. Must be a non-negative integer.}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{delta}{Shape parameter \code{delta} >= 0. Can be a scalar or a vector.
Default: 0.0.}

\item{lambda}{Shape parameter \code{lambda} > 0. Can be a scalar or a vector.
Default: 1.0.}
}
\value{
A vector of length \code{n} containing random deviates from the kkw
distribution. The length of the result is determined by \code{n} and the
recycling rule applied to the parameters (\code{alpha}, \code{beta},
\code{delta}, \code{lambda}). Returns \code{NaN} if parameters
are invalid (e.g., \code{alpha <= 0}, \code{beta <= 0}, \code{delta < 0},
\code{lambda <= 0}).
}
\description{
Generates random deviates from the Kumaraswamy-Kumaraswamy (kkw)
distribution with parameters \code{alpha} (\eqn{\alpha}), \code{beta}
(\eqn{\beta}), \code{delta} (\eqn{\delta}), and \code{lambda} (\eqn{\lambda}).
This distribution is a special case of the Generalized Kumaraswamy (GKw)
distribution where the parameter \eqn{\gamma = 1}.
}
\details{
The generation method uses the inverse transform method based on the quantile
function (\code{\link{qkkw}}). The kkw quantile function is:
\deqn{
Q(p) = \left[ 1 - \left\{ 1 - \left[ 1 - (1 - p)^{1/(\delta+1)} \right]^{1/\lambda} \right\}^{1/\beta} \right]^{1/\alpha}
}
Random deviates are generated by evaluating \eqn{Q(p)} where \eqn{p} is a
random variable following the standard Uniform distribution on (0, 1)
(\code{\link[stats]{runif}}).

This is equivalent to the general method for the GKw distribution
(\code{\link{rgkw}}) specialized for \eqn{\gamma=1}. The GKw method generates
\eqn{W \sim \mathrm{Beta}(\gamma, \delta+1)} and then applies transformations.
When \eqn{\gamma=1}, \eqn{W \sim \mathrm{Beta}(1, \delta+1)}, which can be
generated via \eqn{W = 1 - V^{1/(\delta+1)}} where \eqn{V \sim \mathrm{Unif}(0,1)}.
Substituting this \eqn{W} into the GKw transformation yields the same result
as evaluating \eqn{Q(1-V)} above (noting \eqn{p = 1-V} is also Uniform).
}
\examples{
\donttest{
set.seed(2025) # for reproducibility

# Generate 1000 random values from a specific kkw distribution
alpha_par <- 2.0
beta_par <- 3.0
delta_par <- 0.5
lambda_par <- 1.5

x_sample_kkw <- rkkw(1000, alpha = alpha_par, beta = beta_par,
                       delta = delta_par, lambda = lambda_par)
summary(x_sample_kkw)

# Histogram of generated values compared to theoretical density
hist(x_sample_kkw, breaks = 30, freq = FALSE, # freq=FALSE for density
     main = "Histogram of kkw Sample", xlab = "x", ylim = c(0, 3.5))
curve(dkkw(x, alpha = alpha_par, beta = beta_par, delta = delta_par,
            lambda = lambda_par),
      add = TRUE, col = "red", lwd = 2, n = 201)
legend("topright", legend = "Theoretical PDF", col = "red", lwd = 2, bty = "n")

# Comparing empirical and theoretical quantiles (Q-Q plot)
prob_points <- seq(0.01, 0.99, by = 0.01)
theo_quantiles <- qkkw(prob_points, alpha = alpha_par, beta = beta_par,
                        delta = delta_par, lambda = lambda_par)
emp_quantiles <- quantile(x_sample_kkw, prob_points, type = 7) # type 7 is default

plot(theo_quantiles, emp_quantiles, pch = 16, cex = 0.8,
     main = "Q-Q Plot for kkw Distribution",
     xlab = "Theoretical Quantiles", ylab = "Empirical Quantiles (n=1000)")
abline(a = 0, b = 1, col = "blue", lty = 2)

# Compare summary stats with rgkw(..., gamma=1, ...)
# Note: individual values will differ due to randomness
x_sample_gkw <- rgkw(1000, alpha = alpha_par, beta = beta_par, gamma = 1.0,
                     delta = delta_par, lambda = lambda_par)
print("Summary stats for rkkw sample:")
print(summary(x_sample_kkw))
print("Summary stats for rgkw(gamma=1) sample:")
print(summary(x_sample_gkw)) # Should be similar
}

}
\references{
Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation}

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.

Devroye, L. (1986). \emph{Non-Uniform Random Variate Generation}. Springer-Verlag.
(General methods for random variate generation).
}
\seealso{
\code{\link{rgkw}} (parent distribution random generation),
\code{\link{dkkw}}, \code{\link{pkkw}}, \code{\link{qkkw}},
\code{\link[stats]{runif}}, \code{\link[stats]{rbeta}}
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{random}
