% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hrf-functions.R
\name{hrf_weighted}
\alias{hrf_weighted}
\title{Weighted HRF (No Hemodynamic Delay)}
\usage{
hrf_weighted(
  weights,
  width = NULL,
  times = NULL,
  method = c("constant", "linear"),
  normalize = FALSE
)
}
\arguments{
\item{weights}{Numeric vector of weights. Required.}

\item{width}{Total duration of the window in seconds. If provided without
\code{times}, weights are evenly spaced from 0 to \code{width}.}

\item{times}{Numeric vector of time points (in seconds, relative to t=0) where
weights are specified. Must be strictly increasing and start at 0 for
consistency with other HRFs. If provided, \code{width} is ignored.}

\item{method}{Interpolation method between time points:
\describe{
  \item{"constant"}{Step function - weight is constant until the next time
    point (default). Good for discrete time bins.
  }
  \item{"linear"}{Linear interpolation between points. Good for smooth
    weight transitions.
  }
}}

\item{normalize}{Logical; if \code{TRUE}, weights are scaled so they sum to 1
(for \code{method = "constant"}) or integrate to 1 (for \code{method = "linear"}).
This makes the regression coefficient interpretable as a weighted mean.
Default is \code{FALSE}.}
}
\value{
An HRF object that can be used with \code{regressor()} and other
  fmrihrf functions.
}
\description{
Creates a flexible weighted HRF starting at t=0 with user-specified weights.
Unlike traditional HRFs, this has no built-in hemodynamic delay - it directly
maps weights to time points, allowing for arbitrary temporal response shapes.
}
\details{
This is useful for extracting weighted averages of data at specific time points.
When \code{normalize = TRUE} and the HRF is used in a GLM, the estimated
coefficient represents a weighted mean of the data at the specified times.

There are two ways to specify the temporal structure:
\enumerate{
  \item \code{width + weights}: Weights are evenly spaced from 0 to \code{width}
  \item \code{times + weights}: Explicit time points for each weight (relative to t=0)
}

For delayed windows (not starting at t=0), use \code{\link{lag_hrf}} to shift
the weighted HRF in time.
}
\section{Note on durations}{

The temporal structure (\code{width} or \code{times}) is fixed when the HRF
is created. The \code{duration} parameter in \code{\link{regressor}()} does
\strong{not} modify the weighted HRF's structure---it controls how long the
neural input is sustained (which then gets convolved with this HRF). For
trial-varying weighted HRFs, use a list of HRFs:
\preformatted{
hrf_early <- hrf_weighted(width = 6, weights = c(1, 1, 0, 0), normalize = TRUE)
hrf_late <- hrf_weighted(width = 6, weights = c(0, 0, 1, 1), normalize = TRUE)
reg <- regressor(onsets = c(0, 20), hrf = list(hrf_early, hrf_late))
}
}

\examples{
# Simple: 6s window with 4 evenly-spaced weights (at 0, 2, 4, 6s)
hrf1 <- hrf_weighted(width = 6, weights = c(0.2, 0.5, 0.8, 0.3))
t <- seq(-1, 10, by = 0.1)
plot(t, evaluate(hrf1, t), type = "s", main = "Weighted HRF (width + weights)")

# Explicit times for precise control
hrf2 <- hrf_weighted(
  times = c(0, 1, 3, 5, 6),
  weights = c(0.1, 0.5, 0.8, 0.5, 0.1),
  method = "linear"
)
plot(t, evaluate(hrf2, t), type = "l", main = "Smooth Weighted HRF")

# Normalized weights - coefficient estimates weighted mean of signal
hrf3 <- hrf_weighted(
  width = 8,
  weights = c(1, 2, 2, 1),
  normalize = TRUE
)

# Trial-varying weighted HRFs
hrf_early <- hrf_weighted(width = 6, weights = c(1, 1, 0, 0), normalize = TRUE)
hrf_late <- hrf_weighted(width = 6, weights = c(0, 0, 1, 1), normalize = TRUE)
reg <- regressor(onsets = c(0, 20), hrf = list(hrf_early, hrf_late))

# For delayed windows, use lag_hrf
hrf_delayed <- lag_hrf(hrf_weighted(width = 5, weights = c(1, 2, 1)), lag = 10)
}
\seealso{
\code{\link{hrf_boxcar}} for simple uniform boxcars,
  \code{\link{lag_hrf}} to shift the window in time,
  \code{\link{empirical_hrf}} for HRFs from measured data

Other hrf_functions: 
\code{\link{hrf_basis_lwu}()},
\code{\link{hrf_boxcar}()},
\code{\link{hrf_bspline}()},
\code{\link{hrf_gamma}()},
\code{\link{hrf_gaussian}()},
\code{\link{hrf_inv_logit}()},
\code{\link{hrf_lwu}()},
\code{\link{hrf_mexhat}()},
\code{\link{hrf_sine}()},
\code{\link{hrf_spmg1}()},
\code{\link{hrf_time}()}
}
\concept{hrf_functions}
