#' Stratified Feldt's Coefficient
#'
#' @description
#' Compute the stratified Feldt's coefficient for a test composed of several
#' item strata (e.g., subtests or subscales).
#'
#' @param x A data frame or matrix containing item responses, with rows as
#'   subjects and columns as items. Items are assumed to be ordered by stratum.
#' @param s A numeric vector giving the number of items in each stratum. The
#'   sum of \code{s} must equal \code{ncol(x)}.
#'
#' @details
#' Stratified Feldt's coefficient is an estimate of internal consistency
#' reliability for a composite test formed by multiple strata.
#'
#' @return A named list with:
#' \describe{
#'   \item{stratified.feldt}{Stratified Feldt's coefficient.}
#' }
#'
#' @examples
#' data(data.m)
#' stratified_feldt(data.m, c(13, 12, 6))
#'
#' @export
stratified_feldt <- function(x, s) {
  # basic checks ---------------------------------------------------------------
  if (is.null(x)) {
    stop("`x` must not be NULL.")
  }
  if (!is.data.frame(x) && !is.matrix(x)) {
    stop("`x` must be a data frame or a matrix.")
  }

  x <- stats::na.exclude(x)
  x <- as.matrix(x)
  if (!is.numeric(x)) {
    stop("`x` must contain only numeric values.")
  }

  if (missing(s)) {
    stop("`s` must be supplied as a numeric vector giving items per stratum.")
  }
  if (!is.numeric(s)) {
    stop("`s` must be numeric.")
  }
  if (any(s <= 0L)) {
    stop("All elements of `s` must be positive.")
  }

  ns <- length(s)
  if (ns < 2L) {
    stop("`s` must define at least 2 strata.")
  }
  if (sum(s) != ncol(x)) {
    stop("The sum of `s` must equal the number of columns in `x`.")
  }

  # total score variance -------------------------------------------------------
  total_scores <- rowSums(x)
  vt <- stats::var(total_scores)
  if (isTRUE(all.equal(vt, 0))) {
    stop("Total score variance is zero; stratified Feldt's coefficient is undefined.")
  }

  # CSEM from compound binomial model -----------------------------------------
  cb <- csem_compound_binomial(x, s)
  if (!("csem" %in% names(cb))) {
    stop("`csem.compound.binomial()` must return an object with a `csem` component.")
  }

  # stratified Feldt via CSEM^2 and total-score variance -----------------------
  stratifiedfeldt <- 1 - mean(cb$csem^2) / vt

  return(list(stratified_feldt = stratifiedfeldt))
}
