% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mct.R
\name{multiple_comparisons}
\alias{multiple_comparisons}
\title{Perform Multiple Comparison Tests on a statistical model}
\usage{
multiple_comparisons(
  model.obj,
  classify,
  sig = 0.05,
  int.type = "ci",
  trans = NULL,
  offset = NULL,
  power = NULL,
  decimals = 2,
  descending = FALSE,
  groups = TRUE,
  plot = FALSE,
  label_height = 0.1,
  rotation = 0,
  save = FALSE,
  savename = "predicted_values",
  order,
  pred.obj,
  pred,
  ...
)
}
\arguments{
\item{model.obj}{An ASReml-R or aov model object. Will likely also work with \code{lme} (\code{\link[nlme:lme]{nlme::lme()}}), \code{lmerMod} (\code{\link[lme4:lmer]{lme4::lmer()}}) models as well.}

\item{classify}{Name of predictor variable as string.}

\item{sig}{The significance level, numeric between 0 and 1. Default is 0.05.}

\item{int.type}{The type of confidence interval to calculate. One of \code{ci}, \code{tukey}, \verb{1se}, \verb{2se}, or \code{none}. Default is \code{ci}.}

\item{trans}{Transformation that was applied to the response variable. One of \code{log}, \code{sqrt}, \code{logit}, \code{power}, \code{inverse}, or \code{arcsin}. Default is \code{NULL}.}

\item{offset}{Numeric offset applied to response variable prior to transformation. Default is \code{NULL}. Use 0 if no offset was applied to the transformed data. See Details for more information.}

\item{power}{Numeric power applied to response variable with power transformation. Default is \code{NULL}. See Details for more information.}

\item{decimals}{Controls rounding of decimal places in output. Default is 2 decimal places.}

\item{descending}{Logical (default \code{FALSE}). Order of the output sorted by the predicted value. If \code{TRUE}, largest will be first, through to smallest last.}

\item{groups}{Logical (default \code{TRUE}). If \code{TRUE}, the significance letter groupings will be calculated and displayed. This can get overwhelming for large numbers of comparisons, so can be turned off by setting to \code{FALSE}.}

\item{plot}{Automatically produce a plot of the output of the multiple comparison test? Default is \code{FALSE}. This is maintained for backwards compatibility, but the preferred method now is to use \verb{autoplot(<multiple_comparisons output>)}. See \code{\link[=autoplot.mct]{autoplot.mct()}} for more details.}

\item{label_height}{Height of the text labels above the upper error bar on the plot. Default is 0.1 (10\%) of the difference between upper and lower error bars above the top error bar.}

\item{rotation}{Rotate the text output as Treatments within the plot. Allows for easier reading of long treatment labels. Number between 0 and 360 (inclusive) - default 0}

\item{save}{Logical (default \code{FALSE}). Save the predicted values to a csv file?}

\item{savename}{A file name for the predicted values to be saved to. Default is \code{predicted_values}.}

\item{order}{Deprecated. Use \code{descending} instead.}

\item{pred.obj}{Deprecated. Predicted values are calculated within the function from version 1.0.1 onwards.}

\item{pred}{Deprecated. Use \code{classify} instead.}

\item{...}{Other arguments passed through to \code{\link[=get_predictions]{get_predictions()}}.}
}
\value{
An object of class \code{mct} (a list with class attributes) containing:
\item{predictions}{A data frame with predicted means, standard errors,
confidence interval upper and lower bounds, and significant group
allocations}
\item{pairwise_pvalues}{A symmetric matrix of p-values for all pairwise
comparisons using Tukey's HSD test}
\item{hsd}{The Honest Significant Difference value(s) used in the comparisons.
Either a single numeric value (if constant across comparisons) or a matrix
(if varies by comparison)}
\item{aliased}{Character vector of aliased treatment levels (only present if
some predictions are aliased)}
\item{sig_level}{The significance level used (default 0.05)}
}
\description{
A function for comparing and ranking predicted means with Tukey's Honest Significant Difference (HSD) Test.
}
\details{
\subsection{Offset}{

Some transformations require that data has a small offset to be
applied, otherwise it will cause errors (for example taking a log of 0, or
the square root of negative values). In order to correctly reverse this
offset, if the \code{trans} argument is supplied, a value should also be supplied
in the \code{offset} argument. By default the function assumes no offset was
required for a transformation, implying a value of 0 for the \code{offset}
argument. If an offset value is provided, use the same value as provided in
the model, not the inverse. For example, if adding 0.1 to values for a log
transformation, add 0.1 in the \code{offset} argument.
}

\subsection{Power}{

The power argument allows the specification of arbitrary powers to be
back transformed, if they have been used to attempt to improve normality of
residuals.
}

\subsection{Confidence Intervals & Comparison Intervals}{

The function provides several options for confidence intervals via the \code{int.type} argument:
\itemize{
\item \strong{\code{ci} (default)}: Traditional confidence intervals for individual means. These estimate the precision of each individual mean but may not align with the multiple comparison results. Non-overlapping traditional confidence intervals do not necessarily indicate significant differences in multiple comparison tests.
\item \strong{\code{tukey}}: Tukey comparison intervals that are consistent with the multiple comparison test. These intervals are wider than regular confidence intervals and are designed so that non-overlapping intervals correspond to statistically significant differences in the Tukey HSD test. This ensures visual consistency between the intervals and letter groupings.
\item \strong{\verb{1se}} and \strong{\verb{2se}}: Intervals of ±1 or ±2 standard errors around each mean.
\item \strong{\code{none}}: No confidence intervals will be calculated or displayed in plots.
}

By default, the function displays regular confidence intervals (\code{int.type = "ci"}),
which estimate the precision of individual treatment means. However, when
performing multiple comparisons, these regular confidence intervals may not
align with the letter groupings from Tukey's HSD test. Specifically, you may
observe non-overlapping confidence intervals for treatments that share the
same letter group (indicating no significant difference).

This occurs because regular confidence intervals and Tukey's HSD test serve
different purposes:
\itemize{
\item Regular confidence intervals estimate individual mean precision
\item Tukey's HSD controls the family-wise error rate across all pairwise comparisons
}

To resolve this visual inconsistency, you can use Tukey comparison intervals
(\code{int.type = "tukey"}). These intervals are specifically designed for multiple
comparisons and will be consistent with the letter groupings: non-overlapping
Tukey intervals indicate significant differences, while overlapping intervals
suggest no significant difference.

The function will issue a message if it detects potential inconsistencies
between non-overlapping confidence intervals and letter groupings, suggesting
the use of Tukey intervals for clearer interpretation.
For multiple comparison contexts, Tukey comparison intervals are recommended as they provide visual consistency with the statistical test being performed and avoid the common confusion where traditional confidence intervals don't overlap but groups share the same significance letter.
}
}
\examples{
# Fit aov model
model <- aov(Petal.Length ~ Species, data = iris)

# Display the ANOVA table for the model
anova(model)

# Determine ranking and groups according to Tukey's Test (with Tukey intervals)
pred.out <- multiple_comparisons(model, classify = "Species")

# Display the predicted values table
pred.out

# Access the p-value matrix
pred.out$pairwise_pvalues

# Access the HSD value
pred.out$hsd

# Show the predicted values plot
autoplot(pred.out, label_height = 0.5)

# Use traditional confidence intervals instead of Tukey comparison intervals
pred.out.ci <- multiple_comparisons(model, classify = "Species", int.type = "ci")
pred.out.ci

# Plot without confidence intervals
pred.out.none <- multiple_comparisons(model, classify = "Species", int.type = "none")
autoplot(pred.out.none)

# AOV model example with transformation
my_iris <- iris
my_iris$Petal.Length <- exp(my_iris$Petal.Length) # Create exponential response
exp_model <- aov(Petal.Length ~ Species, data = my_iris)

resplot(exp_model) # Residual plot shows problems

# Fit a new model using a log transformation of the response
log_model <- aov(log(Petal.Length) ~ Species, data = my_iris)

resplot(log_model) # Looks much better

# Display the ANOVA table for the model
anova(log_model)

# Back transform, because the "original" data was exponential
pred.out <- multiple_comparisons(log_model, classify = "Species",
                                   trans = "log")

# Display the predicted values table
pred.out

# Show the predicted values plot
autoplot(pred.out, label_height = 15)

\dontrun{
# ASReml-R Example
library(asreml)

#Fit ASReml Model
model.asr <- asreml(yield ~ Nitrogen + Variety + Nitrogen:Variety,
                    random = ~ Blocks + Blocks:Wplots,
                    residual = ~ units,
                    data = asreml::oats)

wald(model.asr) # Nitrogen main effect significant

#Determine ranking and groups according to Tukey's Test
pred.out <- multiple_comparisons(model.obj = model.asr, classify = "Nitrogen",
                    descending = TRUE, decimals = 5)

pred.out

# Example using a box-cox transformation
set.seed(42) # See the seed for reproducibility
resp <- rnorm(n = 50, 5, 1)^3
trt <- as.factor(sample(rep(LETTERS[1:10], 5), 50))
block <- as.factor(rep(1:5, each = 10))
ex_data <- data.frame(resp, trt, block)

# Change one treatment random values to get significant difference
ex_data$resp[ex_data$trt=="A"] <- rnorm(n = 5, 7, 1)^3

model.asr <- asreml(resp ~ trt,
                    random = ~ block,
                    residual = ~ units,
                    data = ex_data)

resplot(model.asr)

# lambda = 1/3 from MASS::boxcox()
model.asr <- asreml(resp^(1/3) ~ trt,
                    random = ~ block,
                    residual = ~ units,
                    data = ex_data)

resplot(model.asr) # Look much better

pred.out <- multiple_comparisons(model.obj = model.asr,
                                 classify = "trt",
                                 trans = "power", power = (1/3))

pred.out
autoplot(pred.out, label_height = 0.5)
}

}
\references{
Jørgensen, E. & Pedersen, A. R. (1997). How to Obtain Those Nasty
Standard Errors From Transformed Data - and Why They Should Not Be Used.
}
