#' Apply reporting structures to generate retrospective reporting triangles
#'
#' This function applies a reporting structure to a list of truncated reporting
#'   triangles by setting observations to NA row by row from the bottom up based
#'   on the specified structure. This generates the reporting triangles that
#'   would have been available at each retrospective time point. It operates on
#'   each element in the list in order (from most recent retrospective nowcast
#'   time to oldest retrospective nowcast time).
#'
#' @param truncated_reporting_triangles List of `n` truncated reporting triangle
#'   matrices with as many rows as available given the truncation.
#' @inheritParams apply_reporting_structure
#' @returns `reporting_triangles` List of retrospective reporting triangles,
#'   generated by removing the bottom right observations from the truncated
#'   reporting triangle matrices.
#' @importFrom cli cli_abort
#' @family generate_retrospective_data
#' @export
#' @examples
#' # Generate retrospective triangles from truncated triangles
#' trunc_rts <- truncate_to_rows(example_reporting_triangle, n = 2)
#' retro_rts <- apply_reporting_structures(trunc_rts)
#'
#' # With custom structure
#' retro_rts_custom <- apply_reporting_structures(
#'   trunc_rts,
#'   structure = 2
#' )
#' retro_rts_custom
apply_reporting_structures <- function(truncated_reporting_triangles,
                                       structure = 1,
                                       validate = TRUE) {
  # Check that input is a list
  if (!is.list(truncated_reporting_triangles)) {
    cli_abort(message = "`truncated_reporting_triangles` must be a list")
  }

  reporting_triangles <- lapply(
    truncated_reporting_triangles,
    apply_reporting_structure,
    structure = structure,
    validate = validate
  )

  return(reporting_triangles)
}

#' Apply reporting structure to generate a single retrospective reporting
#'   triangle
#'
#' This function applies a reporting structure to a truncated reporting triangle
#'   by setting observations to NA row by row from the bottom up based on the
#'   specified structure. It is the singular version of
#'   `apply_reporting_structures()`.
#'
#' @param truncated_reporting_triangle A single truncated reporting_triangle
#'   object. May or may not contain NAs.
#' @param structure Integer or vector specifying the reporting structure.
#'   If integer, divides columns evenly by that integer (with last possibly
#'   truncated).  If vector, the sum must not be greater than or equal to the
#'   number of columns. Default is 1 (standard triangular structure).
#' @inheritParams assert_reporting_triangle
#' @returns A single retrospective reporting triangle matrix with NAs in the
#'   appropriate positions.
#' @family generate_retrospective_data
#' @export
#' @examples
#' # Standard triangular structure (default)
#' rep_tri <- apply_reporting_structure(example_reporting_triangle)
#' rep_tri
#'
#' # Ragged structure with 2 columns per delay period
#' rep_ragged <- apply_reporting_structure(example_reporting_triangle, 2)
#' rep_ragged
#'
#' # Custom structure with explicit column counts
#' rep_custom <- apply_reporting_structure(example_reporting_triangle, c(1, 2))
#' rep_custom
apply_reporting_structure <- function(truncated_reporting_triangle,
                                      structure = 1,
                                      validate = TRUE) {
  assert_reporting_triangle(truncated_reporting_triangle, validate)

  # Get matrix dimensions
  rows <- nrow(truncated_reporting_triangle)
  cols <- ncol(truncated_reporting_triangle)

  # Convert to matrix for modification
  result <- as.matrix(truncated_reporting_triangle)

  # Process structure parameter
  if (length(structure) == 1) {
    structure_vec <- .expand_structure_vec(structure, cols)
  } else {
    # Vector case
    structure_vec <- structure
    if (any(structure_vec < 0) || !all(is.finite(structure_vec))) {
      cli_abort("All elements of `structure` must be positive integers")
    }
    if (sum(structure_vec) >= cols) {
      cli_abort(
        message = c(
          "Sum of structure vector must not be greater than or equal",
          "to the number of columns in matrix"
        )
      )
    }
  }

  # For each row, determine which columns should be NA
  cutoff_cols <- cumsum(structure_vec) + 1

  for (i in seq_along(structure_vec)) {
    index_row <- rows - i + 1
    start_col <- cutoff_cols[i]
    result[index_row, start_col:cols] <- NA_real_
  }

  # Convert back to reporting_triangle with preserved attributes
  return(.update_triangle_matrix(truncated_reporting_triangle, result))
}

.expand_structure_vec <- function(structure, cols) {
  if (structure <= 0) {
    cli_abort("Structure must be positive")
  }

  if (structure > cols) {
    cli_abort("Structure cannot be larger than number of columns")
  }
  adjusted_cols <- cols - 1

  n_complete_groups <- floor(adjusted_cols / structure)

  structure_vec <- rep(structure, n_complete_groups)

  remainder <- adjusted_cols %% structure

  if (remainder > 0) {
    structure_vec <- c(structure_vec, remainder)
  }

  return(structure_vec)
}
