#' Plot Robustness Decay Curve
#'
#' Visualizes how model performance (robustness) decreases as the level
#' of input noise increases. This "decay curve" is a powerful tool for
#' understanding the sensitivity threshold of a machine learning model.
#'
#' @param predict_fn A function that accepts a numeric matrix and returns
#'   a numeric vector of predictions.
#' @param X A numeric matrix or data.frame of input features.
#' @param levels A numeric vector of noise levels to evaluate.
#'   Default is \code{seq(0, 0.3, by = 0.05)}.
#' @param n_rep Number of repetitions for each noise level. Default is \code{5L}.
#' @param ... Additional arguments passed to \code{\link[graphics]{plot}}.
#'
#' @return A data.frame with columns \code{noise_level} and \code{robustness_score}.
#'
#' @examples
#' # Simple model
#' pred_fn <- function(X) X %*% c(1, -1)
#' X <- matrix(rnorm(200), ncol = 2)
#'
#' # Plot decay
#' plot_robustness(pred_fn, X, main = "Model Robustness Decay")
#'
#' @importFrom graphics plot lines grid
#' @export
plot_robustness <- function(predict_fn, X, levels = seq(0, 0.3, by = 0.05),
                            n_rep = 5L, ...) {
    scores <- vapply(levels, function(lvl) {
        if (lvl == 0) {
            return(1.0)
        }
        robustness_score(predict_fn, X, noise_level = lvl, n_rep = n_rep)
    }, numeric(1L))

    results <- data.frame(
        noise_level = levels,
        robustness_score = scores
    )

    plot(
        results$noise_level, results$robustness_score,
        type = "b", pch = 19,
        xlab = "Noise Level",
        ylab = "Robustness Score",
        ylim = c(0, 1.1),
        ...
    )
    grid()
    lines(results$noise_level, results$robustness_score, lwd = 2, col = "steelblue")

    return(invisible(results))
}
