#' Annotate cell types using features plot with different marker databases
#'
#' @description This function dynamically selects the appropriate annotation method
#'     based on the `gene_list_type` parameter. It supports marker databases from
#'     Cellmarker2, PanglaoDB, Seurat (via `FindAllMarkers`), or Excel files.
#'
#' @param seurat_obj A valid Seurat object with cluster annotations in `meta.data`.
#' @param gene_list A list of data frames containing marker genes and metrics.
#'     Format depends on `gene_list_type`:
#'     - **Cellmarker2**: Generated by `Markers_filter_Cellmarker2()`.
#'     - **PanglaoDB**: Generated by `Markers_filter_PanglaoDB()`.
#'     - **Seurat**: Generated by `read_seurat_markers()`.
#'     - **Excel**: Generated by `read_excel_markers()`.
#' @param gene_list_type Type of marker database to use. Be one of:
#'     `"Cellmarker2"`, `"PanglaoDB"`, `"Seurat"`, or `"Excel"`.
#' @param species Species of the dataset: `"Human"` or `"Mouse"` for gene name standardization.
#' @param cluster_col Column name in `meta.data` defining clusters (default: `"seurat_clusters"`).
#' @param assay Assay layer in the Seurat object (default: `"RNA"`).
#' @param save_path Directory to save output PNGs. Must be explicitly specified.
#' @param min_counts Minimum number of counts for Cellmarker2 annotations (default: `1`).
#' @param metric_names Optional. Change the row name for the input mertics, not recommended unless
#'     necessary. (NULL is used as default parameter; used in "Seurat"/"Excel").
#' @param colour_low Color for lowest expression level. (default = "white")
#' @param colour_high Color for highest expression level. (default = "navy")
#' @param colour_low_mertic Color for lowest mertic level. (default = "white")
#' @param colour_high_mertic Color for highest mertic level. (default = "navy")
#' @param ... Additional parameters passed to the specific annotation function.
#'
#' @returns Saves cell type annotation PNGs in `save_path`. Returns invisibly.
#' @export
#' @family Section_4_Semi_Automated_Annotation
#'
#' @importFrom stats setNames
#' @importFrom dplyr all_of
#' @importFrom Seurat Idents
#' @importFrom ggplot2 theme_minimal element_text element_blank
#' @importFrom patchwork wrap_plots
#' @importFrom cowplot plot_grid
#' @importFrom scales rescale
#'
#' @examples
#' \dontrun{
#' # Example for Cellmarker2
#' Celltype_Annotation_Features(seurat_obj = sce,
#'     gene_list = Markers_list_Cellmarker2,
#'     species = "Human",
#'     cluster_col = "seurat_clusters",
#'     assay = "RNA",
#'     save_path = file.path(tempdir(),"SlimR_Celltype_annotation_Cellmarker2"),
#'     colour_low = "white",
#'     colour_high = "navy",
#'     colour_low_mertic = "white",
#'     colour_high_mertic = "navy",
#'     )
#'
#' # Example for PanglaoDB
#' Celltype_Annotation_Features(seurat_obj = sce,
#'     gene_list = Markers_list_panglaoDB,
#'     species = "Human",
#'     cluster_col = "seurat_clusters",
#'     assay = "RNA",
#'     save_path = file.path(tempdir(),"SlimR_Celltype_annotation_PanglaoDB")
#'     colour_low = "white",
#'     colour_high = "navy",
#'     colour_low_mertic = "white",
#'     colour_high_mertic = "navy",
#'     )
#'
#' # Example for Seurat marker list
#' Celltype_Annotation_Features(seurat_obj = sce,
#'     gene_list = Markers_list_Seurat,
#'     species = "Human",
#'     cluster_col = "seurat_clusters",
#'     assay = "RNA",
#'     save_path = file.path(tempdir(),"SlimR_Celltype_annotation_Seurat")
#'     colour_low = "white",
#'     colour_high = "navy",
#'     colour_low_mertic = "white",
#'     colour_high_mertic = "navy",
#'     )
#'
#' # Example for Excel marker list
#' Celltype_Annotation_Features(seurat_obj = sce,
#'     gene_list = Markers_list_Excel,
#'     species = "Human",
#'     cluster_col = "seurat_clusters",
#'     assay = "RNA",
#'     save_path = file.path(tempdir(),"SlimR_Celltype_annotation_Excel")
#'     colour_low = "white",
#'     colour_high = "navy",
#'     colour_low_mertic = "white",
#'     colour_high_mertic = "navy",
#'     )
#' }
#'
Celltype_Annotation_Features <- function(
    seurat_obj,
    gene_list,
    gene_list_type = 'Default',
    species = NULL,
    cluster_col = "seurat_clusters",
    assay = "RNA",
    save_path = NULL,
    min_counts = 1,
    metric_names = NULL,
    colour_low = "white",
    colour_high = "navy",
    colour_low_mertic = "white",
    colour_high_mertic = "navy",
    ...
) {
  if (!inherits(seurat_obj, "Seurat")) {
    stop("Input must be a Seurat object!", call. = FALSE)
  }
  if (!is.list(gene_list)) {
    stop("gene_list must be a list of data frames!", call. = FALSE)
  }
  if (!all(sapply(gene_list, is.data.frame))) {
    stop("Each element in gene_list must be a data frame!", call. = FALSE)
  }
  if (species != "Human" && species != "Mouse") {
    stop("species must be 'Human' or 'Mouse'", call. = FALSE)
  }
  if (missing(save_path)) {
    stop("save_path must be explicitly specified", call. = FALSE)
  }
  if (!interactive() && !grepl(normalizePath(tempdir()), normalizePath(save_path), fixed = TRUE)) {
    warning("Writing to non-temporary locations is restricted. Using temporary directory instead.", immediate. = TRUE)
    save_path <- file.path(tempdir(), "Celltype_annotation_Dotplot_output")
  }

  dir.create(save_path, showWarnings = FALSE, recursive = TRUE)

  switch(EXPR = gene_list_type,
         "Cellmarker2" = {
           Celltype_annotation_Cellmarker2(
             seurat_obj = seurat_obj,
             gene_list = gene_list,
             species = species,
             cluster_col = cluster_col,
             assay = assay,
             save_path = save_path,
             min_counts = min_counts,
             colour_low = colour_low,
             colour_high = colour_high,
             colour_low_mertic = colour_low_mertic,
             colour_high_mertic = colour_high_mertic,
             ...
           )
         },
         "PanglaoDB" = {
           Celltype_annotation_PanglaoDB(
             seurat_obj = seurat_obj,
             gene_list = gene_list,
             species = species,
             cluster_col = cluster_col,
             assay = assay,
             save_path = save_path,
             colour_low = colour_low,
             colour_high = colour_high,
             colour_low_mertic = colour_low_mertic,
             colour_high_mertic = colour_high_mertic,
             ...
           )
         },
         "Seurat" = {
           Celltype_annotation_Seurat(
             seurat_obj = seurat_obj,
             gene_list = gene_list,
             species = species,
             cluster_col = cluster_col,
             assay = assay,
             save_path = save_path,
             metric_names = metric_names,
             colour_low = colour_low,
             colour_high = colour_high,
             colour_low_mertic = colour_low_mertic,
             colour_high_mertic = colour_high_mertic,
             ...
           )
         },
         "Excel" = {
           Celltype_annotation_Excel(
             seurat_obj = seurat_obj,
             gene_list = gene_list,
             species = species,
             cluster_col = cluster_col,
             assay = assay,
             save_path = save_path,
             metric_names = metric_names,
             colour_low = colour_low,
             colour_high = colour_high,
             colour_low_mertic = colour_low_mertic,
             colour_high_mertic = colour_high_mertic,
             ...
           )
         },
         {message("Input 'gene_list_type' is not recognized. Using 'Default' as fallback.\n")
           Celltype_annotation_Excel(
             seurat_obj = seurat_obj,
             gene_list = gene_list,
             species = species,
             cluster_col = cluster_col,
             assay = assay,
             save_path = save_path,
             metric_names = metric_names,
             colour_low = colour_low,
             colour_high = colour_high,
             colour_low_mertic = colour_low_mertic,
             colour_high_mertic = colour_high_mertic,
             ...
           )
         }
  )

  invisible()
}
