test_that("ddCt calculation is mathematically correct", {

  # --- 1. SETUP PERFECT DATA ---
  # Scenario:
  # - Reference Gene (GAPDH) is constant at Ct=20.
  # - Target Gene in Control (WT) is Ct=25.
  # - Target Gene in Treated is Ct=24 (1 cycle lower = 2x expression).
  #
  # Expected Math:
  # dCt(WT)      = 25 - 20 = 5
  # dCt(Treated) = 24 - 20 = 4
  # ddCt         = 4 - 5 = -1
  # Fold Change  = 2^(-(-1)) = 2^1 = 2.0

  df <- data.frame(
    Sample = c("S1", "S2", "S3", "S4"),
    Group = c("WT", "WT", "Treated", "Treated"),
    Gene = c(rep("GAPDH", 4), rep("Target", 4)),
    Ct = c(20, 20, 20, 20,   # GAPDH (Perfectly constant)
           25, 25, 24, 24)   # Target (Perfect 1 cycle difference)
  )

  # --- 2. RUN FUNCTION ---
  results <- calculate_ddct(df, ref_gene = "GAPDH", control_group = "WT")

  # --- 3. CHECK THE NUMBERS (Fold Change) ---
  # Extract the fold change for the Target gene in the Treated group
  fc_value <- results$fold_change[results$Gene == "Target" & results$Group == "Treated"]

  # The answer MUST be exactly 2
  expect_equal(fc_value[1], 2)
})

test_that("T-test detects significant difference", {

  # Scenario: Distinct groups with no variance overlap
  df <- data.frame(
    Sample = c("C1", "C2", "C3", "T1", "T2", "T3"),
    Group = c(rep("WT", 3), rep("Treated", 3)),
    Gene = c(rep("GAPDH", 6), rep("Target", 6)),
    Ct = c(
      rep(20, 6),           # GAPDH Constant
      25, 25.1, 24.9,       # WT Target (~25)
      20, 20.1, 19.9        # Treated Target (~20) -> Massive difference
    )
  )

  results <- calculate_ddct(df, ref_gene = "GAPDH", control_group = "WT")

  # Extract P-value for treated group
  p_val <- results$p_val[results$Group == "Treated" & results$Gene == "Target"][1]

  # P-value should be extremely small (significant)
  expect_lt(p_val, 0.001)
})
