% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OrgHeatmap.R
\name{OrgHeatmap}
\alias{OrgHeatmap}
\title{Visualization Tool for Human, Mouse and Organelle Data}
\usage{
OrgHeatmap(
  data,
  species = c("human", "mouse", "organelle"),
  system = NULL,
  valid_organs = NULL,
  sort_by_value = TRUE,
  title = NULL,
  showall = FALSE,
  outline = TRUE,
  palette = "YlOrRd",
  color_high = NULL,
  color_low = NULL,
  color_mid = NULL,
  reverse_palette = FALSE,
  fillcolor_outline = NULL,
  fillcolor_organ = "plasma",
  fillcolor_other = "#D3D3D3",
  organbar = TRUE,
  organbar_title = NULL,
  organbar_digit = 4,
  organbar_color = NULL,
  organbar_low = NULL,
  organbar_high = NULL,
  direction = 1,
  save_clean_data = FALSE,
  save_plot = FALSE,
  clean_data_path = file.path(getwd(), "clean_data.rds"),
  plot_path = file.path(getwd(), "organ_plot.png"),
  plot_width = 10,
  plot_height = 8,
  plot_dpi = 300,
  plot_device = "png",
  organ_system_map = NULL,
  organ_name_mapping = NULL,
  aggregate_method = "mean",
  organ_col = "organ",
  value_col = "value"
)
}
\arguments{
\item{data}{Data frame with at least two columns: organ name and corresponding value}

\item{species}{Character, species to visualize, one of \code{"human"} (default), \code{"mouse"}, or \code{"organelle"}.
Determines the default body/cell contour, organ coordinates, and organ-system mapping.}

\item{system}{Optional character vector specifying organ system to display(not applicable for organelles)}

\item{valid_organs}{Optional character vector of valid organ names for filtering}

\item{sort_by_value}{Logical, default TRUE, sorts by value descending}

\item{title}{Optional character vector for plot title}

\item{showall}{Logical, default FALSE. If TRUE, shows all organ outlines (grey) with light grey fill (#EFEFEF) for non-target organs (to provide anatomical context).}

\item{outline}{Logical, default TRUE, draws human/cell outline}

\item{palette}{Character, name of RColorBrewer palette (e.g., "YlOrRd", "PuBuGn") for unified color scheme (applies to both organ heatmap and bar chart if no custom colors are specified).
Ignored if \code{color_low}/\code{color_high} (for heatmap) or \code{organbar_low}/\code{organbar_high} (for bar chart) are specified. Default: "YlOrRd" (suitable for highlighting high values).}

\item{color_high}{Character, custom color for the \strong{maximum value} of the organ heatmap (and bar chart if \code{organbar_high} is not specified). Overrides \code{palette} but is overridden by \code{organbar_high} (highest priority). Default: NULL.}

\item{color_low}{Character, custom color for the \strong{minimum value} of the organ heatmap (and bar chart if \code{organbar_low} is not specified). Overrides \code{palette} but is overridden by \code{organbar_low} (highest priority). Default: NULL.}

\item{color_mid}{Character, optional color for the \strong{middle value} of the organ heatmap (for 3-color gradients). Default: NULL.}

\item{reverse_palette}{Logical, whether to reverse the color order of \code{palette}. Default: FALSE (low=light, high=dark).}

\item{fillcolor_outline}{Character, default "#F5D5B8" for human/mouse, "#F0F8FF" for organelle, fill color for outline}

\item{fillcolor_organ}{Character, fallback color scheme for organs (supports viridis options: "viridis", "plasma", "magma", etc.). Only used if no \code{palette}, \code{color_low/color_high}, or \code{organbar_low/organbar_high} are specified. Default: "plasma".}

\item{fillcolor_other}{Character, default "#D3D3D3", fill color for non-target organelles}

\item{organbar}{Logical, default TRUE, shows value bar chart}

\item{organbar_title}{Optional character, title for bar chart legend}

\item{organbar_digit}{Integer, default 4, digits for bar values}

\item{organbar_color}{Optional character, solid color for bars}

\item{organbar_low}{Character, low end of gradient for \strong{bar chart} (and organ heatmap if \code{color_low} is not specified). Highest priority for color configuration (overrides \code{color_low} and \code{palette}). Default: NULL.}

\item{organbar_high}{Character, high end of gradient for \strong{bar chart} (and organ heatmap if \code{color_high} is not specified). Highest priority for color configuration (overrides \code{color_high} and \code{palette}). Default: NULL.}

\item{direction}{Integer, default 1. Direction of color gradient: 1 = normal (low value → light color, high value → dark color); -1 = reversed (low value → dark color, high value → light color).}

\item{save_clean_data}{Logical, default FALSE, saves cleaned data}

\item{save_plot}{Logical, default FALSE, whether to save the plot}

\item{clean_data_path}{Character, default \code{file.path(getwd(), "clean_data.rds")}, path for cleaned data (default to current working directory)}

\item{plot_path}{Character, default \code{file.path(getwd(), "organ_plot.png")}, path for saving the plot (default to current working directory)}

\item{plot_width}{Numeric, default 10, plot width in inches}

\item{plot_height}{Numeric, default 8, plot height in inches}

\item{plot_dpi}{Numeric, default 300, plot resolution}

\item{plot_device}{Character, default "png", plot format (e.g., "png", "pdf")}

\item{organ_system_map}{Data frame, CSV path, or NULL (default). If NULL, uses species-specific defaults:
\code{human_organ_systems} for humans, \code{mouse_organ_systems} for mice, or \code{organelle_systems} for organelles.}

\item{organ_name_mapping}{Optional: Either a named vector (non-standard → standard names, e.g., c("adrenal" = "adrenal_gland")),
a data frame (must contain columns specified by \code{original_col} and \code{standard_col}), or a CSV path (same column requirement).
Internally processed by \code{create_organ_mapping()} for standardization (lowercase, underscores for spaces).}

\item{aggregate_method}{Character, aggregation method for duplicate organs
(one of "mean" \code{default}, "sum", "count").}

\item{organ_col}{Character, default "organ", column name for organs}

\item{value_col}{Character, default "value", column name for values}
}
\value{
List containing:
\item{plot}{ggplot2 object}
\item{clean_data}{Cleaned data frame}
\item{system_used}{Organ system used}
\item{mapped_organs}{Standardized organ names}
\item{missing_organs}{Organs without coordinates}
\item{total_value}{Sum of all values}
}
\description{
This tool visualizes numerical data (such as gene expression) on a human, mouse or organelle map.
It supports custom color schemes, organ system filtering, and bar charts for quantitative comparison.
}
\details{
The function uses \code{get_component_colors()} (an internal helper function) to generate unified color schemes:
\enumerate{
\item If \code{organbar_low} and \code{organbar_high} are specified by the user, they will be used directly (highest priority);
\item If not, colors are generated from the \code{palette} (RColorBrewer) with optional reversal (\code{reverse_palette});
\item Custom middle color (\code{color_mid}) is supported for 3-color gradients (applied to both heatmap and bar chart).
For valid \code{palette} names, see \code{RColorBrewer::brewer.pal.info}.
}
\subsection{Organ Mapping Logic}{
\itemize{
\item For \code{organ_name_mapping}: Accepts a named vector, data frame, or CSV path.
Internal helper \code{create_organ_mapping()} standardizes names (lowercase, underscores for spaces).
\item For \code{organ_system_map}: Custom tables (data frame/CSV) are processed to align with
built-in \code{human_organ_systems} (for humans), \code{mouse_organ_systems} (for mice), or \code{organelle_systems} (for organelles)
format via \code{create_organ_mapping()}.
}
}
}
\examples{
\donttest{
# Load the package
library(OrgHeatmap)

# Note: Example datasets (example_Data1, example_Data2, example_Data3, expr_data)
# are included in the package's 'extdata' directory.

## Load example data files from extdata (contains example_Data1, example_Data2, example_Data3)
data_path <- system.file("extdata", "exampledata.Rdata", package = "OrgHeatmap")
load(data_path)  

# 1.Plot all organs and save results using internal saving function
result_all <- OrgHeatmap(
  example_Data3, 
  organbar = TRUE,
  save_plot = TRUE,  # Enable plot saving
  plot_path = file.path(tempdir(), "all_system.png"),  
  plot_width = 10,
  plot_height = 8,
  save_clean_data = TRUE,  # Enable cleaned data saving
  clean_data_path = file.path(tempdir(), "all_system_clean_data.rds")
)
print(result_all$plot)  # Print the plot to the console

# 2. Plot circulatory system organs and save results
result_circulatory <- OrgHeatmap(
  example_Data3, 
  system = "circulatory",
 organbar = TRUE,
  save_plot = TRUE,
 plot_path = file.path(tempdir(), "circulatory_system.png"),
 plot_width = 10,
  plot_height = 8,
  plot_device = "png",  # Specify plot format
  save_clean_data = TRUE,
  clean_data_path = file.path(tempdir(), "circulatory_clean_data.rds")
)
print(result_circulatory$plot)  # Print the plot to the console

# 3. Quick color configuration with palette
# Core logic: Trigger internal color_config generation via palette parameters,
# ensuring organ and bar chart colors are synchronized
result_palette <- OrgHeatmap(
example_Data3,
system = "respiratory",  # Respiratory system
palette = "PuBuGn",      # Use RColorBrewer's blue-purple-green palette
reverse_palette = TRUE,  # Reverse palette (low value = dark green, high value = purple)
color_mid = "#87CEEB",   # Custom middle color (sky blue)
title = "Respiratory System (Palette: PuBuGn)",
organbar_title = "Mean Value",
organbar_digit = 2,
showall = TRUE,
save_plot = TRUE,
plot_path = file.path(tempdir(),"respiratory_palette.png")
  # To use solid color for bars, add parameter: organbar_color = "skyblue"
  # (overrides gradient and synchronizes with organ colors)
)
print(result_palette$plot)


 ## Load the example dataset expr_data from extdata
 expr_data_path <- system.file("extdata", "expr_data.rds", package = "OrgHeatmap")
 expr_data <- readRDS(expr_data_path)

# Custom organ mapping (standardize original organ names)
custom_mapping <- c(
  "adrenal" = "adrenal_gland",
  "lymph node" = "lymph_node",
  "soft tissue" = "muscle",
  "peritoneal" = "peritoneum"
)

# Add prostate cancer-specific organs to system mapping
prostate_organ_systems <- rbind(
  human_organ_systems,  # Package's built-in organ system mapping
  data.frame(
    organ = c("prostate", "bone", "lymph_node", "adrenal_gland"),
    system = c("reproductive", "musculoskeletal", "lymphatic", "endocrine"),
    stringsAsFactors = FALSE
  )
)

# Generate TP53 expression heatmap and save results
tp53_plot <- OrgHeatmap(
  data = expr_data,
  value_col = "expression",  # Specify value column name as "expression"
  organ_system_map = prostate_organ_systems,  # Use custom organ system mapping
  organ_name_mapping = custom_mapping,  # Apply organ name mapping
  title = "TP53 Expression in Metastatic Prostate Cancer",
  organbar_title = "Mean Expression(log2) of TP53",
  aggregate_method = "mean",  # Calculate mean by organ
  showall = TRUE,  # Show all organ outlines
  fillcolor_other = "#DCDCDC",  # Fill color for non-target organs
  organbar_digit = 2,  # Keep 2 decimal places for bar values
  direction = -1,  # Reverse color gradient (darker = higher expression)
  save_plot = TRUE,  # Save the plot
  plot_path = file.path(tempdir(), "tp53_expression_metastatic_prostate.png"),
  plot_width = 14,
  plot_height = 10,
  plot_dpi = 300,
  save_clean_data = TRUE,  # Save cleaned data
  clean_data_path = file.path(tempdir(), "tp53_clean_data.rds")
)

# Print the plot
print(tp53_plot$plot)

# 4.Plot mouse digestive system
# Load mouse example data (included in the package)
mouse_data_path <- system.file("extdata", "exampledata.Rdata", package = "OrgHeatmap")
load(mouse_data_path)

# Generate plot for mouse digestive system
mouse_digestive_plot <- OrgHeatmap(
data = example_Data1,
species = "mouse",  # Specify mouse species
system = "digestive",
organbar = TRUE,
palette = "PuBu",
save_plot = TRUE,
plot_path = file.path(tempdir(), "mouse_digestive_plot.png"),
save_clean_data = TRUE
)
print(mouse_digestive_plot$plot)
}

 # 5. Plot organelles 
organelle_data <- data.frame(
 organ = c("mitochondrion", "nucleus", "endoplasmic_reticulum","cell_membrane"),
 value = c(15.2, 8.7, 6.3,6.8)
)

organelle_plot <- OrgHeatmap(
 data = organelle_data,
 species = "organelle",
title = "Organelle Expression Visualization",
organbar_title = "Expression Level",
save_plot = TRUE,
plot_path = file.path(tempdir(), "organelle_expression_plot.png")
)

}
