#' @title Introduces Missing Completely at Random (MCAR) Values into Data Sets.
#' @description
#' This function introduces missing values under the Missing Completely at Random (MCAR) mechanism into previously generated data sets (e.g., those produced by sim.skewed() or sim.normal()).
#' Missing values are inserted at random locations according to user specifications and are denoted as "NA" in the resulting files.
#' The modified data sets are saved as new files, preserving the original data sets.
#' In each data file, the first column contains the sample identifiers, while the subsequent columns show actual data with some entries replaced by NA.
#' Additionally, a file named "MCAR_List.dat" is created, listing the names of all data sets to which missing values were introduced.
#'
#' @author Fatih Orcan
#' @importFrom utils read.table write.table
#' @param misg A numeric vector of 0s and 1s specifying which items will contain missing values.
#' A value of 0 indicates the item will not include missingness, while 1 indicates missing values will be introduced.If omitted, all items are treated as eligible for missingness.
#' @param perct The percentage of missingness to be applied (default = 10 percents).
#' @param dataList The file name containing the list of previously generated data sets (e.g., "Data_List.dat"), either created by this package or by external software.
#' @param f.loc The directory path where both the original data sets and the "dataList" file are located.
#' @export
#' @examples
#'
#' # Step 1: Generate data sets
#' fc<-fcors.value(nf=3, cors=c(1,.5,.6,.5,1,.4,.6,.4,1))
#' fl<-loading.value(nf=3, fl.loads=c(.5,.5,.5,0,0,0,0,0,0,0,0,.6,.6,.6,0,0,0,0,0,0,0,0,.4,.4))
#' floc<-tempdir()
#' sim.normal(nd=10, ss=100, fcors=fc, loading<-fl,  f.loc=floc)
#'
#'  # Step 2: Introduce missing values
#'
#' mis.items<-c(1,1,1,0,0,0,0,0)
#' dl<-"Data_List.dat"  # must be located in the working directory
#' MCAR.data(misg = mis.items, perct = 20, dataList = dl, f.loc=floc)
#'

  MCAR.data<-function(misg=NULL, perct=10, dataList="Data_List.dat", f.loc){

    data.names<-as.matrix(read.table(paste(f.loc, "/", dataList,sep=""), header = FALSE))
    misg.names<-data.names
    veri<-read.table(paste(f.loc,"/", data.names[1,],sep=""))
    colnames(veri)<-c("ID", paste("x",seq(1:(dim(veri)[2]-1)),sep=""))

    nd<-dim(data.names)[1]
    for(i in 1:nd){
    veri<-read.table(paste(f.loc,"/", data.names[i,],sep=""))

    misg.names[i,]<-c(paste("MCAR_",i,".dat", sep=""))
    nitem<-dim(veri)[2]-1
    ss<-dim(veri)[1]

    MCAR.data<-veri
    for(j in 1:nitem){
      if(misg[j]==0){
        MCAR.data[,j+1]<-veri[,j+1]}
      else if(misg[j]==1){
        mis.smpl<-sample(1:ss,(perct/100)*ss,FALSE)
        MCAR.data[mis.smpl,j+1]<-NA
        message(paste("MCAR_",i,".dat was completed", sep=""))
        }
      else {stop("Please use only 0s or 1s to indicate missingness")}}
  write.table(MCAR.data, file= paste(f.loc, "/MCAR_",i,".dat", sep=""), sep = "\t",
                  col.names = FALSE, row.names = FALSE,quote = FALSE)
 }
    write.table(misg.names,file=paste(f.loc,"/MCAR_List.dat", sep = ""),
                  col.names = FALSE, row.names = FALSE, quote = FALSE)
    message("Done!...")
}
