# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_literacy_rate

library(testthat)

test_that("get_japan_literacy_rate() returns a tibble with expected structure", {
  skip_on_cran()

  result <- get_japan_literacy_rate()

  # Must return NULL or tibble
  expect_true(is.null(result) || inherits(result, "tbl_df"))

  if (is.null(result)) return(invisible(TRUE))

  # Structure
  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 4)

  # Column names
  expect_named(result, c("indicator", "country", "year", "value"))

  # Column types – literacy rate currently returns logical NA (World Bank)
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_true(is.double(result$value) || is.logical(result$value))
})

test_that("get_japan_literacy_rate() returns correct number of rows and years", {
  skip_on_cran()

  result <- get_japan_literacy_rate()
  if (is.null(result)) return(invisible(TRUE))

  # Should always be 13 rows (2010–2022)
  expect_equal(nrow(result), 13)

  # Years must be 2010 through 2022
  expect_equal(sort(result$year), 2010:2022)

  # Verify descending order
  expect_true(all(diff(result$year) <= 0))
})

test_that("get_japan_literacy_rate() has consistent metadata", {
  skip_on_cran()

  result <- get_japan_literacy_rate()
  if (is.null(result)) return(invisible(TRUE))

  # Country must always be Japan
  expect_true(all(result$country == "Japan"))

  # Indicator text should match World Bank label
  expect_true(all(grepl("Literacy rate", result$indicator, ignore.case = TRUE)))
})

test_that("get_japan_literacy_rate() allows NA values as valid (World Bank provides no data)", {
  skip_on_cran()

  result <- get_japan_literacy_rate()
  if (is.null(result)) return(invisible(TRUE))

  # All values may legitimately be NA
  expect_true(all(is.na(result$value)))

  # NA values MUST be accepted as valid (per your instruction)
  expect_true(all(is.na(result$value)))
})

test_that("get_japan_literacy_rate() handles API errors safely", {
  skip_on_cran()

  result <- tryCatch(get_japan_literacy_rate(), error = function(e) NULL)

  expect_true(is.null(result) || inherits(result, "tbl_df"))
})
