#' @title Simple Longitudinal Mean (SLM)
#'
#' @description
#' This function detects influential subjects in longitudinal data based on their mean response values.
#' It identifies subjects whose mean response deviates significantly beyond a specified threshold
#' (defined as \code{k} standard deviations from the mean). The function provides a summary of influential subjects,
#' separates the data into influential and non-influential subjects, calculates influence scores, and visualizes the results using \code{ggplot2}.
#'
#' @details
#' The function follows these steps:
#' \itemize{
#'   \item Calculates the mean and standard deviation of the response variable across all subjects.
#'   \item Determines the threshold for influence based on \code{k} standard deviations from the mean.
#'   \item Identifies subjects whose mean response falls outside this threshold.
#'   \item Calculates the Influence Score (IS) for each subject as the absolute deviation of their mean from the overall mean.
#'   \item Calculates the Proportional Influence Score (PIS) for each subject as IS divided by the overall standard deviation.
#'   \item Separates data into influential and non-influential subjects.
#'   \item Visualizes the distribution of responses and highlights influential subjects.
#' }
#'
#' This method is useful for detecting outliers and understanding the impact of extreme values in longitudinal studies.
#'
#' @param data A data frame containing longitudinal data.
#' @param subject_id A column specifying the column name representing subject identifiers.
#' @param time A column specifying different time points that observations are measured.
#' @param response A column specifying the column name representing response values.
#' @param k A numeric value representing the threshold (number of standard deviations from the mean)
#'          to classify a subject as influential.
#' @param verbose Logical; if TRUE, prints informative messages during execution.
#'
#' @return A list containing:
#' \item{influential_subjects}{A vector of subject IDs identified as influential.}
#' \item{influential_data}{A data frame containing data for influential subjects.}
#' \item{non_influential_data}{A data frame containing data for non-influential subjects.}
#' \item{influence_scores}{A data frame with subject IDs, mean response, IS (Influence Score), and PIS (Proportional Influence Score).}
#' \item{mean_plot}{A ggplot object showing mean responses per subject with influential subjects highlighted.}
#' \item{longitudinal_plot}{A ggplot object visualizing longitudinal response trends, with influential subjects highlighted.}
#' \item{IS_table}{A data frame containing the Influence Score (IS) and the Partial Influence Score (PIS) values for each subject.}
#'
#' @examples
#' data(infsdata)
#' infsdata <- infsdata[1:5,]
#' result <- slm(infsdata, "subject_id", "time", "response", 2)
#' print(result$influential_subjects)
#' head(result$influential_data)
#' head(result$non_influential_data)
#' head(result$influence_scores)
#' print(result$mean_plot)
#' print(result$longitudinal_plot)
#'
#' @export
#'
#' @seealso tvm, wlm, sld, rld


slm <- function(data, subject_id, time, response, k = 2, verbose = FALSE) {

  # Step 1: Subject-wise mean response
  data_summary <- aggregate(
    data[[response]],
    by = list(data[[subject_id]]),
    FUN = function(x) mean(x, na.rm = TRUE)
  )
  colnames(data_summary) <- c(subject_id, "mean_response")

  # Step 2: Overall mean and SD
  overall_mean <- mean(data_summary$mean_response, na.rm = TRUE)
  overall_sd   <- stats::sd(data_summary$mean_response, na.rm = TRUE)

  # Step 3: Influence scores
  J <- nrow(data_summary)
  data_summary$IS  <- abs(data_summary$mean_response - overall_mean) / overall_sd
  data_summary$PIS <- (data_summary$mean_response - overall_mean)^2 /
    (J * overall_sd^2)

  # Step 4: Identify influential subjects
  influential_ids <- data_summary[[subject_id]][data_summary$IS > k]

  if (length(influential_ids) == 0) {
    warning("No influential subjects detected based on the given threshold.")
    return(list(
      influential_subjects = integer(0),
      influential_data = data[0, ],
      non_influential_data = data,
      IS_table = data_summary[, c(subject_id, "IS", "PIS")],
      mean_plot = NULL,
      longitudinal_plot = NULL
    ))
  }

  # Step 5: Split data
  influential_data <- data[data[[subject_id]] %in% influential_ids, ]
  non_influential_data <- data[!(data[[subject_id]] %in% influential_ids), ]

  if (verbose) {
    message(
      "Influential subjects detected: ",
      paste(influential_ids, collapse = ", ")
    )
  }

  # Step 6: Mean response plot
  mean_plot <- ggplot2::ggplot(
    data_summary,
    ggplot2::aes(x = .data[[subject_id]], y = mean_response)
  ) +
    ggplot2::geom_point(size = 3) +
    ggplot2::geom_point(
      data = data_summary[data_summary[[subject_id]] %in% influential_ids, ],
      color = "red",
      size = 4
    ) +
    ggplot2::geom_hline(
      yintercept = overall_mean,
      linetype = "dashed"
    ) +
    ggplot2::geom_hline(
      yintercept = overall_mean + k * overall_sd,
      linetype = "dashed",
      color = "red"
    ) +
    ggplot2::geom_hline(
      yintercept = overall_mean - k * overall_sd,
      linetype = "dashed",
      color = "red"
    ) +
    ggplot2::labs(
      title = "Influential Subjects Identification (SLM Method)",
      x = subject_id,
      y = "Mean Response"
    ) +
    ggplot2::theme_minimal()

  # Step 7: Longitudinal plot
  longitudinal_plot <- ggplot2::ggplot(
    data,
    ggplot2::aes(
      x = .data[[time]],
      y = .data[[response]],
      group = .data[[subject_id]]
    )
  ) +
    ggplot2::geom_line(alpha = 0.8) +
    ggplot2::geom_line(
      data = influential_data,
      color = "red",
      linewidth = 1.2
    ) +
    ggplot2::labs(
      title = "Longitudinal Trends with Influential Subjects (SLM Method)",
      x = time,
      y = response
    ) +
    ggplot2::theme_minimal()

  list(
    influential_subjects = influential_ids,
    influential_data = influential_data,
    non_influential_data = non_influential_data,
    IS_table = data_summary[, c(subject_id, "IS", "PIS")],
    mean_plot = mean_plot,
    longitudinal_plot = longitudinal_plot
  )
}

utils::globalVariables(c("mean_response", ".data",'aggregate'))
