# R/recursive.R
# ============================================================
#  SAFE RECURSIVE FORECASTER FOR GACE 1.0.0 (STABILITY PATCH)
# ============================================================

#' Internal helper: recursive GACE forecast generator
#'
#' Safe recursive multi-step forecast with:
#'  - decay of growth,
#'  - damped seasonal factors,
#'  - blended level anchoring,
#'  - stability guardrails (no explosions),
#'  - prevention of negative or zero collapses.
#'
#' @keywords internal
#' @noRd
.gace_recursive_forecast <- function(y_last,
                                     growth_hist,
                                     seasonal_factors,
                                     h,
                                     gamma = 0.9,
                                     beta  = 0.8,
                                     start_season_index = 1L) {
  
  # ----------------------------------------------------------
  # SAFETY 1: Ensure valid starting point
  # ----------------------------------------------------------
  if (!is.finite(y_last) || y_last <= 0) {
    y_last <- max(1e-6, abs(y_last))
  }
  
  # ----------------------------------------------------------
  # SAFETY 2: Seasonal factors clamped to avoid spikes
  # ----------------------------------------------------------
  s <- length(seasonal_factors)
  if (s == 0L) {
    seasonal_factors <- 1.0
    s <- 1L
  }
  
  # clamp extreme seasonal swings
  seasonal_factors <- pmin(pmax(seasonal_factors, 0.7), 1.3)
  
  # ----------------------------------------------------------
  # SAFETY 3: Growth signal stabilization
  # ----------------------------------------------------------
  good_growth <- growth_hist[is.finite(growth_hist)]
  
  if (length(good_growth) == 0L) {
    g_base <- 0
  } else {
    g_base <- stats::median(good_growth, na.rm = TRUE)
  }
  
  # hard cap ±50%
  g_base <- pmin(pmax(g_base, -0.50), 0.50)
  
  # If last observed growth collapses → reduce g_base
  if (tail(growth_hist, 1) < -0.20) {
    g_base <- g_base * 0.5
  }
  
  # ----------------------------------------------------------
  # PREP OUTPUT VECTORS
  fc <- numeric(h)
  last_val <- y_last
  season_pos <- start_season_index
  
  # ----------------------------------------------------------
  # MAIN RECURSIVE LOOP
  # ----------------------------------------------------------
  for (i in seq_len(h)) {
    
    # apply geometric decay to growth rate
    g_t <- g_base * (gamma^(i - 1L))
    
    # seasonal multiplier
    s_t <- seasonal_factors[((season_pos - 1L) %% s) + 1L]
    
    # raw multiplicative next value
    raw_next <- last_val * (1 + g_t) * s_t
    
    # ------------------------------------------------------
    # SAFETY 4: Prevent explosive jumps
    # ------------------------------------------------------
    jump <- (raw_next - last_val) / max(last_val, 1e-6)
    
    if (jump >  0.40) raw_next <- last_val * 1.40      # +40% cap
    if (jump < -0.30) raw_next <- last_val * 0.70      # -30% floor
    
    # ------------------------------------------------------
    # SAFETY 5: Level anchoring (prevents drift runaway)
    # ------------------------------------------------------
    final_next <- beta * raw_next + (1 - beta) * last_val
    
    if (!is.finite(final_next) || final_next <= 0) {
      final_next <- max(1e-6, last_val)
    }
    
    # output & prepare next loop
    fc[i] <- final_next
    last_val <- final_next
    season_pos <- season_pos + 1L
  }
  
  fc
}