#' Summarize outliers detected by three methods
#'
#' Generates a summary of outliers detected by the three-sigma edit rule,
#' the Hampel identifier, and the boxplot rule, including an optional
#' label to identify the outlying points
#'
#' @param x numerical vector in which outliers are to be detected
#' @param label optional identifying tag for outliers (default NULL gives
#' an index into the elements of `x` declared outliers)
#' @param labelName optional name for the `label` column, if specified
#' (default NULL labels this column as Case)
#' @param thresh vector of threshold values for each outlier detection rule (default = c(3, 3, 1.5))
#'
#' @return Data frame with one row for each outlier detected by any of the
#' three methods and these 5 columns:
#'   * Record (or `labelName`) giving the location or label for each outlier
#'   * Value the value detected as an outlier by at least one method
#'   * ThreeSigma 1 if the outlier is detected by the three-sigma rule, 0 otherwise
#'   * Hampel 1 if the outlier is detected by the Hampel identifier, 0 otherwise
#'   * Boxplot 1 if the outlier is detected by the boxplot rule, 0 otherwise
#'
#' Note that this data frame is empty (0 rows) if no outliers are detected by any method
#' @export
#'
#' @examples
#' x <- seq(-1, 1, length = 100)
#' x[1:10] <- 10
#' SummarizeOutliers(x)
SummarizeOutliers <- function(x, label = NULL, labelName = NULL, thresh = c(3, 3, 1.5)){
  #
  stopifnot("x must be numeric"= is.numeric(x))
  stopifnot("All thresholds must be positive"= min(thresh) >= 0)
  stopifnot("thresh must list 3 threshold values"= length(thresh) == 3)
  nonMiss <- length(which(!is.na(x)))
  stopifnot("No non-missing values to characterize"= nonMiss > 0)
  #
  Tindex <- FindOutliers(x, method = "T", t = thresh[1])
  Hindex <- FindOutliers(x, method = "H", t = thresh[2])
  Bindex <- FindOutliers(x, method = "B", t = thresh[3])
  outIndex <- union(Tindex, Hindex)
  outIndex <- union(outIndex, Bindex)
  nOut <- length(outIndex)
  if (nOut == 0){
    outframe <- data.frame(Record = numeric(0),
                           Value = numeric(0),
                           ThreeSigma = numeric(0),
                           Hampel = numeric(0),
                           Boxplot = numeric(0))
  } else {
    if (is.null(label)){
      label <- seq(1, length(x))
      labelName <- "Record"
    } else {
      if (is.null(labelName)){
        labelName <- "Case"
      }
    }
    Tout <- rep(0, nOut)
    Hout <- rep(0, nOut)
    Bout <- rep(0, nOut)
    index <- which(outIndex %in% Tindex)
    Tout[index] <- 1
    index <- which(outIndex %in% Hindex)
    Hout[index] <- 1
    index <- which(outIndex %in% Bindex)
    Bout[index] <- 1
    outframe <- data.frame(label = label[outIndex], x = x[outIndex],
                           Tout = Tout, Hout = Hout, Bout = Bout)
    colnames(outframe) <- c(labelName, "Value", "ThreeSigma", "Hampel", "Boxplot")
  }
  return(outframe)
}

