% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/workflow.R
\name{workflow}
\alias{workflow}
\title{Create a workflow}
\usage{
workflow()
}
\value{
A new \code{workflow} object.
}
\description{
A \code{workflow} is a container object that aggregates information required to
fit and predict from a model. This information might be a recipe used in
preprocessing, specified through \code{\link[=add_recipe]{add_recipe()}}, or the model specification
to fit, specified through \code{\link[=add_model]{add_model()}}.
}
\section{Indicator Variable Details}{
Some modeling functions in R create indicator/dummy variables from
categorical data when you use a model formula, and some do not. When you
specify and fit a model with a \code{workflow()}, parsnip and workflows match
and reproduce the underlying behavior of the user-specified model’s
computational engine.
\subsection{Formula Preprocessor}{

In the \link[modeldata:Sacramento]{modeldata::Sacramento} data set of real
estate prices, the \code{type} variable has three levels: \code{"Residential"},
\code{"Condo"}, and \code{"Multi-Family"}. This base \code{workflow()} contains a
formula added via \code{\link[=add_formula]{add_formula()}} to predict
property price from property type, square footage, number of beds, and
number of baths:\if{html}{\out{<div class="r">}}\preformatted{set.seed(123)

library(parsnip)
library(recipes)
library(workflows)
library(modeldata)

data("Sacramento")

base_wf <- workflow() \%>\%
  add_formula(price ~ type + sqft + beds + baths)
}\if{html}{\out{</div>}}

This first model does create dummy/indicator variables:\if{html}{\out{<div class="r">}}\preformatted{lm_spec <- linear_reg() \%>\%
  set_engine("lm")

base_wf \%>\%
  add_model(lm_spec) \%>\%
  fit(Sacramento)
}\if{html}{\out{</div>}}\preformatted{## == Workflow [trained] ================================================
## Preprocessor: Formula
## Model: linear_reg()
## 
## -- Preprocessor ------------------------------------------------------
## price ~ type + sqft + beds + baths
## 
## -- Model -------------------------------------------------------------
## 
## Call:
## stats::lm(formula = ..y ~ ., data = data)
## 
## Coefficients:
##      (Intercept)  typeMulti_Family   typeResidential  
##          32919.4          -21995.8           33688.6  
##             sqft              beds             baths  
##            156.2          -29788.0            8730.0
}

There are \strong{five} independent variables in the fitted model for this
OLS linear regression. With this model type and engine, the factor
predictor \code{type} of the real estate properties was converted to two
binary predictors, \code{typeMulti_Family} and \code{typeResidential}. (The third
type, for condos, does not need its own column because it is the
baseline level).

This second model does not create dummy/indicator variables:\if{html}{\out{<div class="r">}}\preformatted{rf_spec <- rand_forest() \%>\%
  set_mode("regression") \%>\%
  set_engine("ranger")

base_wf \%>\%
  add_model(rf_spec) \%>\%
  fit(Sacramento)
}\if{html}{\out{</div>}}\preformatted{## == Workflow [trained] ================================================
## Preprocessor: Formula
## Model: rand_forest()
## 
## -- Preprocessor ------------------------------------------------------
## price ~ type + sqft + beds + baths
## 
## -- Model -------------------------------------------------------------
## Ranger result
## 
## Call:
##  ranger::ranger(formula = ..y ~ ., data = data, num.threads = 1,      verbose = FALSE, seed = sample.int(10^5, 1)) 
## 
## Type:                             Regression 
## Number of trees:                  500 
## Sample size:                      932 
## Number of independent variables:  4 
## Mtry:                             2 
## Target node size:                 5 
## Variable importance mode:         none 
## Splitrule:                        variance 
## OOB prediction error (MSE):       7058847504 
## R squared (OOB):                  0.5894647
}

Note that there are \strong{four} independent variables in the fitted model
for this ranger random forest. With this model type and engine,
indicator variables were not created for the \code{type} of real estate
property being sold. Tree-based models such as random forest models can
handle factor predictors directly, and don’t need any conversion to
numeric binary variables.
}

\subsection{Recipe Preprocessor}{

When you specify a model with a \code{workflow()} and a recipe preprocessor
via \code{\link[=add_recipe]{add_recipe()}}, the \emph{recipe} controls whether
dummy variables are created or not; the recipe overrides any underlying
behavior from the model’s computational engine.
}
}

\examples{
library(parsnip)
library(recipes)
library(magrittr)
library(modeldata)

data("attrition")

model <- logistic_reg() \%>\%
  set_engine("glm")

base_wf <- workflow() \%>\%
  add_model(model)

formula_wf <- base_wf \%>\%
  add_formula(Attrition ~ BusinessTravel + YearsSinceLastPromotion + OverTime)

fit(formula_wf, attrition)

recipe <- recipe(Attrition ~ ., attrition) \%>\%
  step_dummy(all_nominal(), -Attrition) \%>\%
  step_corr(all_predictors(), threshold = 0.8)

recipe_wf <- base_wf \%>\%
  add_recipe(recipe)

fit(recipe_wf, attrition)
}
