\name{whiteningMatrix}
\alias{whiteningMatrix}

\title{Compute Whitening Matrix}
\usage{
whiteningMatrix(Sigma, method=c("ZCA", "ZCA-cor", "PCA", "PCA-cor", 
    "Chol-prec", "Chol-cov", "Cholesky"))
}
\arguments{
  \item{Sigma}{Covariance matrix.}
  
  \item{method}{Determines the type of whitening transformation.}

}

\description{
  \code{whiteningMatrix} computes the whitening matrix \eqn{W}.


}

\value{
  \code{whiteningMatrix} returns a square whitening matrix \eqn{W}.  

}

\details{

  Whitening is a linear transformation \eqn{z = W x} where the whitening matrix satisfies the constraint
  \eqn{W'W = Sigma^{-1}}  where \eqn{Sigma = Cov(x)}.

   This function implements various natural whitening transformations discussed in  Kessy, Lewin, and Strimmer (2018).

   The following six different whitening approaches can be selected:

   \code{method="ZCA"}: ZCA whitening, also known as Mahalanobis whitening, ensures that the average covariance between whitened and orginal variables is maximal.  

   \code{method="ZCA-cor"}: Likewise, ZCA-cor whitening leads to whitened variables that are maximally correlated (on average) with the original variables.  

   \code{method="PCA"}: In contrast, PCA whitening lead to maximally compressed whitened variables, as measured by squared covariance.

  \code{method="PCA-cor"}: PCA-cor whitening is similar to PCA whitening but uses squared correlations.  

  \code{method="Chol-prec"} and \code{method="Cholesky"}: computes a whitening matrix by applying Cholesky decomposition on the precision matrix.  This yields an upper triangular positive diagonal whitening matrix and lower triangular positive diagonal cross-covariance and cross-correlation matrices.
   
  \code{method="Chol-cov"}: computes a whitening matrix by applying Cholesky decomposition on the precision matrix.  This yields a lower triangular positive diagonal whitening matrix and upper triangular positive diagonal cross-covariance and cross-correlation matrices.
   

   ZCA-cor whitening is implicitely employed in computing CAT and CAR scores used for variable selection in classification and regression, see the functions \code{catscore} in the \code{sda} package and \code{carscore} in the \code{care} package.

   In both PCA and PCA-cor whitening there is a sign-ambiguity in the eigenvector matrices. In order to resolve the sign-ambiguity we use eigenvector matrices with a positive diagonal.  This has the effect to make cross-correlations and cross-correlations positive diagonal for PCA and PCA-cor.

   For details see Kessy, Lewin, and Strimmer (2018).
 
   Canonical correlation analysis (CCA) can also be understood as a
   special form form of whitening.
}

\author{
  Korbinian Strimmer (\url{https://strimmerlab.github.io}) with Agnan Kessy and Alex Lewin.
}
\seealso{\code{\link{whiteningCrossCov}}, \code{\link{scca}}.}


\references{
Kessy, A., A. Lewin, and K. Strimmer. 2018.
Optimal whitening and decorrelation. The American Statistician. 72: 309-314.
<DOI:10.1080/00031305.2016.1277159>

}


\examples{
# load whitening library
library("whitening")

######

# example data set
# E. Anderson. 1935.  The irises of the Gaspe Peninsula.
# Bull. Am. Iris Soc. 59: 2--5
data("iris")
X = as.matrix(iris[,1:4])
d = ncol(X) # 4
n = nrow(X) # 150
colnames(X) # "Sepal.Length" "Sepal.Width"  "Petal.Length" "Petal.Width"

# estimate covariance
S = cov(X)

# ZCA-cor whitening matrix
W.ZCAcor = whiteningMatrix(S, method="ZCA-cor")

# check of constraint on whitening matrix
crossprod(W.ZCAcor)
solve(S)
}
\keyword{multivariate}
