% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv_vennLasso.R
\name{cv.vennLasso}
\alias{cv.vennLasso}
\title{Cross Validation for the vennLasso}
\usage{
cv.vennLasso(x, y, groups, lambda = NULL, compute.se = FALSE,
  conf.int = NULL, type.measure = c("mse", "deviance", "class", "auc",
  "mae", "brier"), nfolds = 10, foldid, grouped = TRUE, keep = FALSE,
  parallel = FALSE, ...)
}
\arguments{
\item{x}{input matrix or SparseMatrix of dimension nobs x nvars. Each row is an observation,
each column corresponds to a covariate}

\item{y}{numeric response vector of length nobs}

\item{groups}{A list of length equal to the number of groups containing vectors of integers
indicating the variable IDs for each group. For example, groups=list(c(1,2), c(2,3), c(3,4,5)) specifies
that Group 1 contains variables 1 and 2, Group 2 contains variables 2 and 3, and Group 3 contains
variables 3, 4, and 5. Can also be a matrix of 0s and 1s with the number of columns equal to the
number of groups and the number of rows equal to the number of variables. A value of 1 in row i and
column j indicates that variable i is in group j and 0 indicates that variable i is not in group j.}

\item{lambda}{A user-specified sequence of lambda values. Left unspecified, the a sequence of lambda values is
automatically computed, ranging uniformly on the log scale over the relevant range of lambda values.}

\item{compute.se}{logical flag. If \code{TRUE}, standard errors will be computed, otherwise if \code{FALSE} they will not}

\item{conf.int}{value between 0 and 1 indicating the level of the confidence intervals to be computed. For example
if \code{conf.int = 0.95}, 95 percent confidence intervals will be computed.}

\item{type.measure}{One of \code{c("mse","deviance","class","auc","mae","brier")} indicating measure to evaluate for cross-validation. The default is \code{type.measure = "deviance"}, 
which uses squared-error for gaussian models (a.k.a \code{type.measure = "mse"} there), deviance for logistic
regression. \code{type.measure = "class"} applies to binomial only. \code{type.measure = "auc"} is for two-class logistic 
regression only. \code{type.measure = "mse"} or \code{type.measure = "mae"} (mean absolute error) can be used by all models;
they measure the deviation from the fitted mean to the response. \code{type.measure = "brier"} is for models with 
\code{family = "coxph"} and will compute the Brier score.}

\item{nfolds}{number of folds for cross-validation. default is 10. 3 is smallest value allowed.}

\item{foldid}{an optional vector of values between 1 and nfold specifying which fold each observation belongs to.}

\item{grouped}{Like in \pkg{glmnet}, this is an experimental argument, with default \code{TRUE}, and can be ignored by most users. 
For all models, this refers to computing nfolds separate statistics, and then using their mean and estimated standard 
error to describe the CV curve. If \code{grouped = FALSE}, an error matrix is built up at the observation level from the 
predictions from the \code{nfold} fits, and then summarized (does not apply to \code{type.measure = "auc"}).}

\item{keep}{If \code{keep = TRUE}, a prevalidated list of arrasy is returned containing fitted values for each observation 
and each value of lambda for each model. This means these fits are computed with this observation and the rest of its
fold omitted. The folid vector is also returned. Default is \code{keep = FALSE}}

\item{parallel}{If TRUE, use parallel foreach to fit each fold. Must register parallel before hand, such as \pkg{doMC}.}

\item{...}{parameters to be passed to vennLasso}
}
\value{
An object with S3 class "cv.vennLasso"
}
\description{
Cross Validation for the vennLasso
}
\examples{

library(Matrix)

set.seed(123)
n.obs <- 150
n.vars <- 25

true.beta.mat <- array(NA, dim = c(3, n.vars))
true.beta.mat[1,] <- c(-0.5, -1, 0, 0, 2, rep(0, n.vars - 5))
true.beta.mat[2,] <- c(0.5, 0.5, -0.5, -0.5, 1, -1, rep(0, n.vars - 6))
true.beta.mat[3,] <- c(0, 0, 1, 1, -1, rep(0, n.vars - 5))
rownames(true.beta.mat) <- c("1,0", "1,1", "0,1")
true.beta <- as.vector(t(true.beta.mat))

x.sub1 <- matrix(rnorm(n.obs * n.vars), n.obs, n.vars)
x.sub2 <- matrix(rnorm(n.obs * n.vars), n.obs, n.vars)
x.sub3 <- matrix(rnorm(n.obs * n.vars), n.obs, n.vars)

x <- as.matrix(rbind(x.sub1, x.sub2, x.sub3))

conditions <- as.matrix(cbind(c(rep(1, 2 * n.obs), rep(0, n.obs)),
                              c(rep(0, n.obs), rep(1, 2 * n.obs))))

y <- rnorm(n.obs * 3, sd = 3) + drop(as.matrix(bdiag(x.sub1, x.sub2, x.sub3)) \%*\% true.beta)

fit <- cv.vennLasso(x = x, y = y, groups = conditions, nfolds = 3)

fitted.coef <- predict(fit$vennLasso.fit, type = "coefficients", s = fit$lambda.min)
(true.coef <- true.beta.mat[match(dimnames(fit$vennLasso.fit$beta)[[1]], 
                                  rownames(true.beta.mat)),])
round(fitted.coef, 2)

## effects smaller for logistic regression
true.beta.mat <- true.beta.mat / 2
true.beta <- true.beta / 2
# logistic regression example#'
y <- rbinom(n.obs * 3, 1, 
       prob = 1 / (1 + exp(-drop(as.matrix(bdiag(x.sub1, x.sub2, x.sub3)) \%*\% true.beta))))

bfit <- cv.vennLasso(x = x, y = y, groups = conditions, family = "binomial",
                     nfolds = 3)

fitted.coef <- predict(bfit$vennLasso.fit, type = "coefficients", s = bfit$lambda.min)
(true.coef <- true.beta.mat[match(dimnames(bfit$vennLasso.fit$beta)[[1]], 
                                  rownames(true.beta.mat)),])
round(fitted.coef, 2)

}
