#' Coefficient Array of an Estimated tvVAR
#'
#' Returns the system estimated coefficients as an array.
#'
#' Given an estimated time varying VAR of the form:
#' \deqn{\hat{{y}}_t = \hat{A}_{1t} {y}_{t-1} + \ldots + \hat{A}_{pt} {y}_{t-p} + \hat{C}_tD_t}
#' the function returns a list for each equation with
#' \eqn{(\hat{A}_{1t} | \ldots | \hat{A}_{pt} | \hat{C}_t )} set of arrays .
#'
#' @param x An object of class 'tvvar', generated by \code{\link{tvVAR}}.
#' @return A list object with coefficient arrays for the lagged endogenous variables without including the intercept.
#' @rdname tvBcoef
#' @export
tvBcoef <- function (x) UseMethod("tvBcoef", x)

#' @rdname tvBcoef
#' @inheritParams tvBcoef
#' @examples
#' data(Canada, package="vars")
#' var.2p <- vars::VAR(Canada, p = 2, type = "const")
#' tvvar.2p <- tvVAR(Canada, p=2, type= "const")
#' B <- vars::Bcoef(var.2p)
#' tvB <- tvBcoef(tvvar.2p)
#'
#' @method tvBcoef tvvar
#' @export
tvBcoef.tvvar<-function(x)
{
  if (!(class(x) == "tvvar")) {
    stop("\nPlease provide an object of class 'tvvar', generated by tvVAR().\n")
  }
  neq <- x$neq
  y.names <- colnames(x$datamat[, c(1:neq)])
  Z <- x$datamat[, -c(1:neq)]
  B <- array(0, dim=c(nrow(Z), neq, ncol(Z)))
  if (is.null(x$R))
  {
    for (i in 1:neq)
    {
      B[,i, ] <- x$tvcoef[[i]]
    }
  }
  else if (!(is.null(x$R)))
  {#fix this
    for (i in 1:neq)
    {
      restrictions <- x$R
      restrictions[i, restrictions[i, ] == TRUE] <- (x$tvcoef[[i]])
      temp <- restrictions[i, ]
      B[,i, ] <- temp
    }
  }
  dimnames(B)<-list(NULL, y.names, colnames(Z))
  return(B)

}
