
#' Visualize tipping point analysis
#'
#' @description
#' Uses a data frame created by \code{createTippingPointData()} to visualize the tipping point analysis.
#'
#' @param tippingPointData A data frame containing tipping point data, generated by \code{createTippingPointData()}.
#' @param targetPopLabel A label for the trial in the target population.
#' @param yRange An optional argument specifying range of the y axis.
#' @param yBreaks An optional vector specifying breaks on the y axis.
#' @param title The plot title.
#' @param ylab The label for the y axis. Defaults to "Endpoint".
#' @param xlab The label for the x axis. Defaults to "Weight on informative component".
#' @param mapPriorLabel The label for the MAP prior. Defaults to "MAP prior"
#' @param metaAnalysisLabel An optional label for a meta-analysis (if included).
#' @param legendLabel An optional label for the plot legend. Defaults to "Posterior quantile".
#' @param nullTreatmentEffect The null treatment effect, determining where tipping points are calculated. Defaults to 0.
#'
#' @export
#' @seealso \code{\link{createTippingPointData}}
#' @return A \code{ggplot} object of the tipping point plot
#' @examples
#'
#' tipdat <- loadTipmapData("tipdat.rds")
#'
#' tippingPointPlot(tipdat)
tippingPointPlot <- function(tippingPointData, nullTreatmentEffect = 0, targetPopLabel = "Trial in target\n population",
                             yRange = NULL, yBreaks = NULL, title = NULL, ylab = "Endpoint", xlab = "Weight on informative component",
                             mapPriorLabel = "MAP \nprior", metaAnalysisLabel = "MA", legendLabel = "Posterior quantile") {
  if (!(is.numeric(nullTreatmentEffect))) stop("nullTreatmentEffect must be numeric")
  if (!(is.data.frame(tippingPointData))) stop("tippingPointData must be a data frame. Use createTippingPointData()")

  if (nrow(dplyr::filter(tippingPointData, x.col == "prior")) == 2) {
    x.labels <- c(targetPopLabel, seq(from = 0, to = 1, by = .1), mapPriorLabel, metaAnalysisLabel)
    x.breaks <- c(-0.15, seq(from = 0, to = 1, by = .1), 1.15, 1.35)
  } else if (nrow(dplyr::filter(tippingPointData, x.col == "prior")) == 1) {
    x.labels <- c(targetPopLabel, seq(from = 0, to = 1, by = .1), mapPriorLabel)
    x.breaks <- c(-0.15, seq(from = 0, to = 1, by = .1), 1.15)
  }


  if ((tippingPointData$t.0.025[tippingPointData$x.at == 1.15] - nullTreatmentEffect) *
    (tippingPointData$t.0.975[tippingPointData$x.at == 1.15] - nullTreatmentEffect) <= 0) {
    message("95% credible interval for MAP prior includes null treatment effect")
  }

  if ((tippingPointData$t.0.025[tippingPointData$x.at == -0.15] - nullTreatmentEffect) *
    (tippingPointData$t.0.975[tippingPointData$x.at == -0.15] - nullTreatmentEffect) > 0) {
    message("Treatment effect in target population without borrowing")
  }

  # base plot
  tpaPlot <- ggplot2::ggplot(
    data = tippingPointData,
    ggplot2::aes(
      x = x.at, y = t.est,
      colour = x.col
    )
  ) +
    ggplot2::ggtitle(title) +
    # draw horizontal line for no treatment effect
    ggplot2::geom_hline(yintercept = nullTreatmentEffect) +
    # draw mean for target population, MAP prior and meta-analysis
    ggplot2::geom_point(
      data = dplyr::filter(tippingPointData, x.col %in% c("new.obs", "prior")),
      colour = tipmapDarkBlue
    ) +
    # draw errorbars for target population, MAP prior and meta-analysis
    ggplot2::geom_errorbar(
      data = dplyr::filter(tippingPointData, x.col %in% c("new.obs", "prior")),
      ggplot2::aes(ymin = t.0.025, ymax = t.0.975), colour = tipmapDarkBlue, width = .04
    )
  tippingPointData <- dplyr::filter(tippingPointData, x.col == "post")
  # draw line for combined treatment effect estimate
  tpaPlot <- tpaPlot +
    ggplot2::geom_line(data = tippingPointData, ggplot2::aes(y = t.est), colour = tipmapDarkBlue) +
    # draw dotted line for combined 0.025-quantile of treatment effect
    ggplot2::geom_line(
      data = tippingPointData, ggplot2::aes(y = t.0.025, linetype = "2.5%/97.5%", colour = "2.5%/97.5%")
    ) +
    # draw dashed line for combined 0.05-quantile of treatment effect
    ggplot2::geom_line(
      data = tippingPointData, ggplot2::aes(y = t.0.05, linetype = "5%/95%", colour = "5%/95%")
    ) +
    # draw two-dashed line for combined 0.1-quantile of treatment effect
    ggplot2::geom_line(
      data = tippingPointData, ggplot2::aes(y = t.0.1, linetype = "10%/90%", colour = "10%/90%")
    ) +
    # draw long-dashed line for combined 0.2-quantile of treatment effect
    ggplot2::geom_line(
      data = tippingPointData, ggplot2::aes(y = t.0.2, linetype = "20%/80%", colour = "20%/80%")
    ) +
    # draw long-dashed line for combined 0.8-quantile of treatment effect
    ggplot2::geom_line(
      data = tippingPointData, ggplot2::aes(y = t.0.8, linetype = "20%/80%", colour = "20%/80%")
    ) +
    # draw two-dashed line for combined 0.9-quantile of treatment effect
    ggplot2::geom_line(
      data = tippingPointData, ggplot2::aes(y = t.0.9, linetype = "10%/90%", colour = "10%/90%")
    ) +
    # draw dashed line for combined 0.95-quantile of treatment effect
    ggplot2::geom_line(
      data = tippingPointData, ggplot2::aes(y = t.0.95, linetype = "5%/95%", colour = "5%/95%")
    ) +
    # draw dotted line for combined 0.975-quantile of treatment effect
    ggplot2::geom_line(
      data = tippingPointData, ggplot2::aes(y = t.0.975, linetype = "2.5%/97.5%", colour = "2.5%/97.5%")
    )

  # for positive treatment effect in MAP prior

  if (abs((min(abs(unlist(tippingPointData$t.0.025)), na.rm = TRUE) - nullTreatmentEffect)) <
    (abs(min(abs(unlist(tippingPointData$t.0.975)), na.rm = TRUE) - nullTreatmentEffect))) {
    tippingPoint.025 <- getTippingPoint(tippingPointData,
      quantile = 0.025,
      nullTreatmentEffect = nullTreatmentEffect
    )
    tippingPoint.05 <- getTippingPoint(tippingPointData,
      quantile = 0.05,
      nullTreatmentEffect = nullTreatmentEffect
    )
    tippingPoint.1 <- getTippingPoint(tippingPointData,
      quantile = 0.1,
      nullTreatmentEffect = nullTreatmentEffect
    )
    tippingPoint.2 <- getTippingPoint(tippingPointData,
      quantile = 0.2,
      nullTreatmentEffect = nullTreatmentEffect
    )

    if (tippingPoint.025 != 0 & tippingPoint.025 != 1) {
      tpaPlot <- tpaPlot +
        ggplot2::geom_point(
          data = tippingPointData,
          ggplot2::aes(x = tippingPoint.025, y = t.0.025[x.at == tippingPoint.025]),
          colour = tipmapLightRed
        ) +
        ggplot2::geom_vline(ggplot2::aes(xintercept = tippingPoint.025),
          colour = tipmapLightRed, linetype = "dotted"
        )
    }
    if (tippingPoint.05 != 0 & tippingPoint.05 != 1) {
      tpaPlot <- tpaPlot +
        ggplot2::geom_point(
          data = tippingPointData,
          ggplot2::aes(x = tippingPoint.05, y = t.0.05[x.at == tippingPoint.05]),
          colour = tipmapLightRed
        ) +
        ggplot2::geom_vline(ggplot2::aes(xintercept = tippingPoint.05),
          colour = tipmapLightRed, linetype = "dotted"
        )
    }
    if (tippingPoint.1 != 0 & tippingPoint.1 != 1) {
      tpaPlot <- tpaPlot +
        ggplot2::geom_point(
          data = tippingPointData,
          ggplot2::aes(x = tippingPoint.1, y = t.0.1[x.at == tippingPoint.1]),
          colour = tipmapLightRed
        ) +
        ggplot2::geom_vline(ggplot2::aes(xintercept = tippingPoint.1),
          colour = tipmapLightRed, linetype = "dotted"
        )
    }
    if (tippingPoint.2 != 0 & tippingPoint.2 != 1) {
      tpaPlot <- tpaPlot +
        ggplot2::geom_point(data = tippingPointData, ggplot2::aes(
          x = tippingPoint.2,
          y = t.0.2[x.at == tippingPoint.2]
        ), colour = tipmapLightRed) +
        ggplot2::geom_vline(ggplot2::aes(xintercept = tippingPoint.2),
          colour = tipmapLightRed, linetype = "dotted"
        )
    }
  } else {
    tippingPoint.975 <- getTippingPoint(tippingPointData, quantile = 0.975, nullTreatmentEffect = nullTreatmentEffect)
    tippingPoint.95 <- getTippingPoint(tippingPointData, quantile = 0.95, nullTreatmentEffect = nullTreatmentEffect)
    tippingPoint.9 <- getTippingPoint(tippingPointData, quantile = 0.9, nullTreatmentEffect = nullTreatmentEffect)
    tippingPoint.8 <- getTippingPoint(tippingPointData, quantile = 0.8, nullTreatmentEffect = nullTreatmentEffect)

    if (tippingPoint.975 != 0 & tippingPoint.975 != 1) {
      tpaPlot <- tpaPlot +
        ggplot2::geom_point(data = tippingPointData, ggplot2::aes(
          x = tippingPoint.975,
          y = t.0.975[x.at == tippingPoint.975]
        ), colour = tipmapLightRed) +
        ggplot2::geom_vline(ggplot2::aes(xintercept = tippingPoint.975),
          colour = tipmapLightRed, linetype = "dotted"
        )
    }
    if (tippingPoint.95 != 0 & tippingPoint.95 != 1) {
      tpaPlot <- tpaPlot +
        ggplot2::geom_point(data = tippingPointData, ggplot2::aes(
          x = tippingPoint.95,
          y = t.0.95[x.at == tippingPoint.95]
        ), colour = tipmapLightRed) +
        ggplot2::geom_vline(ggplot2::aes(xintercept = tippingPoint.95),
          colour = tipmapLightRed, linetype = "dotted"
        )
    }
    if (tippingPoint.9 != 0 & tippingPoint.9 != 1) {
      tpaPlot <- tpaPlot +
        ggplot2::geom_point(data = tippingPointData, ggplot2::aes(
          x = tippingPoint.9,
          y = t.0.9[x.at == tippingPoint.9]
        ), colour = tipmapLightRed) +
        ggplot2::geom_vline(ggplot2::aes(xintercept = tippingPoint.9),
          colour = tipmapLightRed, linetype = "dotted"
        )
    }
    if (tippingPoint.8 != 0 & tippingPoint.8 != 1) {
      tpaPlot <- tpaPlot +
        ggplot2::geom_point(data = tippingPointData, ggplot2::aes(
          x = tippingPoint.8,
          y = t.0.8[x.at == tippingPoint.8]
        ), colour = tipmapLightRed) +
        ggplot2::geom_vline(ggplot2::aes(xintercept = tippingPoint.8),
          colour = tipmapLightRed, linetype = "dotted"
        )
    }
  }
  # final twists and tweaks
  tpaPlot <- tpaPlot +
    ggplot2::scale_x_continuous(breaks = x.breaks, labels = x.labels) +
    ggplot2::theme_bw() +
    ggplot2::theme(
      panel.grid.minor.x = ggplot2::element_blank(),
      panel.grid.major.x = ggplot2::element_blank(),
      legend.position = "right"
    ) +
    ggplot2::xlab(xlab) +
    ggplot2::ylab(ylab) +
    ggplot2::scale_colour_manual(
      name = legendLabel,
      values = c("2.5%/97.5%" = tipmapDarkBlue, "5%/95%" = tipmapDarkBlue, "10%/90%" = tipmapDarkBlue, "20%/80%" = tipmapDarkBlue), aesthetics = "colour"
    ) +
    ggplot2::scale_linetype_manual(
      name = legendLabel,
      values = c("2.5%/97.5%" = "dotted", "5%/95%" = "dashed", "10%/90%" = "twodash", "20%/80%" = "longdash")
    )

  if (!(missing(yRange))) {
    tpaPlot <- tpaPlot +
      ggplot2::coord_cartesian(ylim = yRange)
  }
  if (!(missing(yBreaks))) {
    tpaPlot <- tpaPlot +
      ggplot2::scale_y_continuous(breaks = yBreaks)
  }

  return(tpaPlot)
}
