#' Apply SMOTE algorithm
#'
#' `step_smote` creates a *specification* of a recipe
#'  step that generate new examples of the  minority class using nearest
#'  neighbors of these cases.
#'
#' @inheritParams recipes::step_center
#' @inheritParams step_upsample
#' @param ... One or more selector functions to choose which
#'  variable is used to sample the data. See [selections()]
#'  for more details. The selection should result in _single
#'  factor variable_. For the `tidy` method, these are not
#'  currently used.
#' @param role Not used by this step since no new variables are
#'  created.
#' @param column A character string of the variable name that will
#'  be populated (eventually) by the `...` selectors.
#' @param neighbors An integer. Number of nearest neighbours that are used
#'  to generate the new examples of the minority class.
#' @param seed An integer that will be used as the seed when
#' smote-ing.
#' @return An updated version of `recipe` with the new step
#'  added to the sequence of existing steps (if any). For the
#'  `tidy` method, a tibble with columns `terms` which is
#'  the variable used to sample.
#'
#' @details
#' The parameter `neighbors` controls the way the new examples are created.
#' For each currently existing minority class example X new examples will be
#' created (this is controlled by the parameter `over_ratio` as mentioned
#' above). These examples will be generated by using the information from the
#' `neighbors` nearest neighbours of each example of the minority class.
#' The parameter `neighbors` controls how many of these neighbours are used.
#'
#' All columns in the data are sampled and returned by [juice()]
#'  and [bake()].
#'
#' All columns used in this step must be numeric with no missing data.
#'
#' When used in modeling, users should strongly consider using the
#'  option `skip = TRUE` so that the extra sampling is _not_
#'  conducted outside of the training set.
#'
#' @references Chawla, N. V., Bowyer, K. W., Hall, L. O., and Kegelmeyer,
#'  W. P. (2002). Smote: Synthetic minority over-sampling technique.
#'  Journal of Artificial Intelligence Research, 16:321-357.
#'
#' @keywords datagen
#' @concept preprocessing
#' @concept subsampling
#' @export
#' @examples
#' library(recipes)
#' library(modeldata)
#' data(credit_data)
#'
#' sort(table(credit_data$Status, useNA = "always"))
#'
#' ds_rec <- recipe(Status ~ Age + Income + Assets, data = credit_data) %>%
#'   step_meanimpute(all_predictors()) %>%
#'   step_smote(Status) %>%
#'   prep()
#'
#' sort(table(juice(ds_rec)$Status, useNA = "always"))
#'
#' # since `skip` defaults to TRUE, baking the step has no effect
#' baked_okc <- bake(ds_rec, new_data = credit_data)
#' table(baked_okc$Status, useNA = "always")
#'
#' ds_rec2 <- recipe(Status ~ Age + Income + Assets, data = credit_data) %>%
#'   step_meanimpute(all_predictors()) %>%
#'   step_smote(Status, over_ratio = 0.2) %>%
#'   prep()
#'
#' table(juice(ds_rec2)$Status, useNA = "always")
#'
#' library(ggplot2)
#'
#' ggplot(circle_example, aes(x, y, color = class)) +
#'   geom_point() +
#'   labs(title = "Without SMOTE")
#'
#' recipe(class ~ ., data = circle_example) %>%
#'   step_smote(class) %>%
#'   prep() %>%
#'   juice() %>%
#'   ggplot(aes(x, y, color = class)) +
#'   geom_point() +
#'   labs(title = "With SMOTE")
#'
#' @importFrom recipes rand_id add_step ellipse_check
step_smote <-
  function(recipe, ..., role = NA, trained = FALSE,
           column = NULL, over_ratio = 1, neighbors = 5,
           skip = TRUE, seed = sample.int(10^5, 1), id = rand_id("smote")) {

    add_step(recipe,
             step_smote_new(
               terms = ellipse_check(...),
               role = role,
               trained = trained,
               column = column,
               over_ratio = over_ratio,
               neighbors = neighbors,
               skip = skip,
               seed = seed,
               id = id
             ))
  }

#' @importFrom recipes step
step_smote_new <-
  function(terms, role, trained, column, over_ratio, neighbors, skip,
           seed, id) {
    step(
      subclass = "smote",
      terms = terms,
      role = role,
      trained = trained,
      column = column,
      over_ratio = over_ratio,
      neighbors = neighbors,
      skip = skip,
      id = id,
      seed = seed,
      id = id
    )
  }

#' @importFrom recipes bake prep check_type
#' @importFrom dplyr select
#' @export
prep.step_smote <- function(x, training, info = NULL, ...) {

  col_name <- terms_select(x$terms, info = info)
  if (length(col_name) != 1)
    rlang::abort("Please select a single factor variable.")
  if (!is.factor(training[[col_name]]))
    rlang::abort(paste0(col_name, " should be a factor variable."))

  check_type(select(training, -col_name), TRUE)
  check_na(select(training, -col_name), "step_smote")

  step_smote_new(
    terms = x$terms,
    role = x$role,
    trained = TRUE,
    column = col_name,
    over_ratio = x$over_ratio,
    neighbors = x$neighbors,
    skip = x$skip,
    seed = x$seed,
    id = x$id
  )
}

#' @importFrom tibble as_tibble tibble
#' @importFrom withr with_seed
#' @export
bake.step_smote <- function(object, new_data, ...) {

  new_data <- as.data.frame(new_data)
  # smote with seed for reproducibility
  with_seed(
    seed = object$seed,
    code = {
      new_data <- smote(new_data, object$column,
                        k = object$neighbors, over_ratio = object$over_ratio)
    }
  )

  as_tibble(new_data)
}

#' @importFrom recipes printer terms_select
#' @export
print.step_smote <-
  function(x, width = max(20, options()$width - 26), ...) {
    cat("SMOTE based on ", sep = "")
    printer(x$column, x$terms, x$trained, width = width)
    invisible(x)
  }

#' @rdname step_smote
#' @param x A `step_smote` object.
#' @importFrom generics tidy
#' @importFrom recipes sel2char is_trained
#' @export
tidy.step_smote <- function(x, ...) {
  if (is_trained(x)) {
    res <- tibble(terms = x$column)
  }
  else {
    term_names <- sel2char(x$terms)
    res <- tibble(terms = unname(term_names))
  }
  res$id <- x$id
  res
}
