#' Server logic for a surveydown survey
#'
#' @description
#' This function defines the server-side logic for a 'shiny' application used in
#' surveydown. It handles various operations such as conditional display,
#' progress tracking, page navigation, database updates for survey responses,
#' and exit survey functionality.
#'
#' @param db A list containing database connection information created using
#'   `sd_database()` function. Defaults to `NULL`. If `NULL`, will be auto-detected
#'   from the calling environment or remain `NULL` (ignore mode).
#' @param required_questions Vector of character strings. The IDs of questions
#'   that must be answered. Defaults to `NULL` (no required questions).
#' @param all_questions_required Logical. If `TRUE`, all questions in the
#'   survey will be required. Defaults to `FALSE`.
#' @param start_page Character string. The ID of the page to start on.
#'   Defaults to `NULL` (first page).
#' @param auto_scroll Logical. Whether to enable auto-scrolling to the next
#'   question after answering. Defaults to `FALSE`.
#' @param rate_survey Logical. If `TRUE`, shows a rating question when exiting
#'   the survey. If `FALSE`, shows a simple confirmation dialog.
#'   Defaults to `FALSE`.
#' @param system_language Set the language for the survey system messages. Include
#'   your own in a `translations.yml` file, or choose a built in one from
#'   the following list: English (`"en"`), German (`"de"`), Spanish (`"es"`),
#'   French (`"fr"`), Italian (`"it"`), Simplified Chinese (`"zh-CN"`).
#'   Defaults to `"en"`. Note: The deprecated `language` parameter is still
#'   supported for backward compatibility.
#' @param use_cookies Logical. If `TRUE`, enables cookie-based session management
#'   for storing and restoring survey progress. Defaults to `TRUE`. Can be
#'   overridden by `use_cookies` setting in the survey.qmd YAML header.
#' @param highlight_unanswered Logical. If `TRUE`, enables highlighting
#'   of all unanswered questions on page display. Defaults to `TRUE`.
#' @param highlight_color Character string. Color for highlighting unanswered
#'   questions. Options are "blue", "orange", "green", "purple", "gray", or "grey".
#'   Defaults to "gray".
#' @param capture_metadata Logical. If `TRUE`, automatically captures and stores
#'   browser information (browser name, version, and OS), IP address, and
#'   screen resolution.
#'   Defaults to `TRUE`.
#' @param language Deprecated as of v0.13.0. Use `system_language` instead.
#' This parameter. is maintained for backward compatibility only.
#'
#' @details
#' The function performs the following tasks:
#' \itemize{
#'   \item Initializes variables and reactive values.
#'   \item Implements conditional display logic for questions.
#'   \item Tracks answered questions and updates the progress bar.
#'   \item Handles page navigation and skip logic.
#'   \item Manages required questions.
#'   \item Performs database operation.
#'   \item Controls auto-scrolling behavior based on the `auto_scroll` argument.
#'   \item Uses sweetalert for warning messages when required questions are not
#'         answered.
#'   \item Handles the exit survey process based on the `rate_survey` argument.
#' }
#'
#' @section Progress Bar:
#' The progress bar is updated based on the last answered question. It will jump
#' to the percentage corresponding to the last answered question and will never
#' decrease, even if earlier questions are answered later. The progress is
#' calculated as the ratio of the last answered question's index to the total
#' number of questions.
#'
#' @section Database Operations:
#' If `db` is provided, the function will update the database with survey
#' responses. If `db` is `NULL` (ignore mode), responses will be saved to a local
#' CSV file.
#'
#' @section Auto-Scrolling:
#' When `auto_scroll` is `TRUE`, the survey will automatically scroll to the
#' next question after the current question is answered. This behavior can be
#' disabled by setting `auto_scroll = FALSE`.
#'
#' @section Exit Survey:
#' When `rate_survey = TRUE`, the function will show a rating question when
#' the user attempts to exit the survey. When `FALSE`, it will show a simple
#' confirmation dialog. The rating, if provided, is saved with the survey data.
#'
#' @return
#' This function does not return a value; it sets up the server-side logic for
#' the 'shiny' application.
#'
#' @examples
#' if (interactive()) {
#'   library(surveydown)
#'
#'   # Get path to example survey file
#'   survey_path <- system.file("examples", "basic_survey.qmd",
#'                              package = "surveydown")
#'
#'   # Copy to a temporary directory
#'   temp_dir <- tempdir()
#'   file.copy(survey_path, file.path(temp_dir, "survey.qmd"))
#'   orig_dir <- getwd()
#'   setwd(temp_dir)
#'
#'   # Define a minimal server
#'   server <- function(input, output, session) {
#'
#'     # sd_server() accepts these following parameters
#'     sd_server(
#'       db = NULL,
#'       required_questions = NULL,
#'       all_questions_required = NULL,
#'       start_page = NULL,
#'       auto_scroll = NULL,
#'       rate_survey = NULL,
#'       system_language = "en",
#'       use_cookies = NULL,
#'       highlight_unanswered = NULL,
#'       highlight_color = NULL,
#'       capture_metadata = NULL
#'     )
#'   }
#'
#'   # Run the app
#'   shiny::shinyApp(ui = sd_ui(), server = server)
#'
#'   # Clean up
#'   setwd(orig_dir)
#' }
#'
#' @seealso
#' `sd_database()`, `sd_ui()`
#'
#' @export
sd_server <- function(
    db = NULL,
    required_questions = NULL,
    all_questions_required = FALSE,
    start_page = NULL,
    auto_scroll = FALSE,
    rate_survey = FALSE,
    system_language = "en",
    use_cookies = TRUE,
    highlight_unanswered = TRUE,
    highlight_color = "gray",
    capture_metadata = TRUE,
    language = NULL
) {
    # 1. Initialize local variables ----

    # Get input, output, and session from the parent environment
    parent_env <- parent.frame()
    input <- get("input", envir = parent_env)
    output <- get("output", envir = parent_env)
    session <- get("session", envir = parent_env)

    # Auto-detect db from calling environment if not provided
    if (is.null(db)) {
        # Only look for 'db' variable in the calling environment
        if (exists("db", envir = parent_env)) {
            db <- get("db", envir = parent_env)
        }
    }

    session$userData$db <- db

    # Tag start time
    time_start <- get_utc_timestamp()

    # Get any skip or show conditions
    show_if <- shiny::getDefaultReactiveDomain()$userData$show_if
    skip_forward <- shiny::getDefaultReactiveDomain()$userData$skip_forward

    # Handle backward compatibility for deprecated 'language' argument
    if ("language" %in% names(match.call())) {
        system_language <- language
        warning(
            "The 'language' argument is deprecated as of v0.13.0. Use 'system_language' instead."
        )
    }

    # Track which parameters were explicitly provided
    explicit_params <- list(
        use_cookies = !missing(use_cookies),
        auto_scroll = !missing(auto_scroll),
        rate_survey = !missing(rate_survey),
        all_questions_required = !missing(all_questions_required),
        start_page = !missing(start_page),
        system_language = !missing(system_language),
        highlight_unanswered = !missing(highlight_unanswered),
        highlight_color = !missing(highlight_color),
        capture_metadata = !missing(capture_metadata),
        required_questions = !missing(required_questions),
        language = !missing(language)
    )

    # Run the configuration settings
    config <- run_config(
        required_questions,
        all_questions_required,
        start_page,
        skip_forward,
        show_if,
        rate_survey,
        system_language
    )

    # Now read settings from _survey/settings.yml (created in sd_ui)
    # Priority: sd_server() parameters > YAML values > defaults
    # Only use YAML values if sd_server() parameters were not explicitly provided
    settings <- read_settings_yaml()

    # Apply YAML overrides for parameters that weren't explicitly provided
    yaml_params <- c(
        "use_cookies",
        "auto_scroll",
        "rate_survey",
        "all_questions_required",
        "start_page",
        "system_language",
        "highlight_unanswered",
        "highlight_color",
        "capture_metadata",
        "required_questions"
    )

    for (param in yaml_params) {
        if (!explicit_params[[param]] && !is.null(settings[[param]])) {
            assign(param, settings[[param]])
        }
    }

    # Normalize color spelling
    if (highlight_color == "grey") {
        highlight_color <- "gray"
    }

    # Update translations if system_language was resolved from YAML or differs from run_config()
    # This ensures the translation system uses the final resolved language
    if (
        (!explicit_params$system_language &&
            !is.null(settings$system_language)) ||
            (explicit_params$system_language && system_language != "en")
    ) {
        paths <- get_paths()
        set_translations(paths, system_language)
    }

    # Update settings.yml with final resolved parameters
    resolved_params <- list(
        use_cookies = use_cookies,
        auto_scroll = auto_scroll,
        rate_survey = rate_survey,
        all_questions_required = all_questions_required,
        start_page = start_page,
        system_language = system_language,
        highlight_unanswered = highlight_unanswered,
        highlight_color = highlight_color,
        capture_metadata = capture_metadata,
        required_questions = required_questions
    )
    update_settings_yaml(resolved_params)

    # Create local objects from config file
    pages <- config$pages
    page_ids <- config$page_ids
    question_ids <- config$question_ids
    question_structure <- config$question_structure

    # Don't overwrite start_page if it was resolved from YAML settings
    # Only use config$start_page if start_page is still NULL
    if (is.null(start_page)) {
        start_page <- config$start_page
    }

    # Handle all_questions_required and required_questions logic
    # This mirrors the logic in run_config() but uses YAML-resolved values
    # Priority: explicit sd_server() parameters > YAML values > config defaults
    if (all_questions_required) {
        matrix_question_ids <- names(which(sapply(
            question_structure,
            `[[`,
            "is_matrix"
        )))
        question_required <- setdiff(question_ids, matrix_question_ids)
    } else if (
        explicit_params$required_questions && !is.null(required_questions)
    ) {
        # Use explicitly provided required_questions from sd_server()
        question_required <- required_questions
    } else if (
        !explicit_params$required_questions && !is.null(required_questions)
    ) {
        # Use YAML-resolved required_questions (when sd_server() didn't provide them)
        question_required <- required_questions
    } else {
        # Fall back to config-determined required questions
        question_required <- config$question_required
    }

    # Update each page's required_questions to reflect final resolved settings
    # This is necessary because pages were created before final parameter resolution
    # Apply this logic when we have any required questions different from config defaults
    if (
        all_questions_required ||
            (explicit_params$required_questions &&
                !is.null(required_questions)) ||
            (!explicit_params$required_questions &&
                !is.null(required_questions) &&
                length(required_questions) > 0)
    ) {
        for (i in seq_along(pages)) {
            page_question_ids <- pages[[i]]$questions
            # Find which questions on this page are in the global required list
            page_required <- intersect(page_question_ids, question_required)
            pages[[i]]$required_questions <- page_required

            # Update asterisks in the HTML content for newly required questions
            if (length(page_required) > 0) {
                # Parse the page content as HTML
                page_html <- xml2::read_html(pages[[i]]$content)

                for (q_id in page_required) {
                    # Find the question container for this question
                    container_selector <- paste0(
                        "[data-question-id='",
                        q_id,
                        "']"
                    )
                    container <- rvest::html_element(
                        page_html,
                        container_selector
                    )

                    if (!is.na(container)) {
                        # Check if it's a matrix question
                        is_matrix <- length(rvest::html_elements(
                            container,
                            ".matrix-question"
                        )) >
                            0

                        if (is_matrix) {
                            # Show asterisks for matrix subquestions
                            sub_asterisks <- rvest::html_elements(
                                container,
                                ".matrix-question td .hidden-asterisk"
                            )
                            for (asterisk in sub_asterisks) {
                                xml2::xml_attr(
                                    asterisk,
                                    "style"
                                ) <- "display: inline;"
                            }
                        } else {
                            # Show asterisk for regular questions
                            asterisk <- rvest::html_element(
                                container,
                                ".hidden-asterisk"
                            )
                            if (!is.na(asterisk)) {
                                xml2::xml_attr(
                                    asterisk,
                                    "style"
                                ) <- "display: inline;"
                            }
                        }
                    }
                }

                # Update the page content with the modified HTML
                pages[[i]]$content <- as.character(page_html)
            }
        }
    }
    page_id_to_index <- stats::setNames(seq_along(page_ids), page_ids)

    # Pre-compute timestamp IDs
    page_ts_ids <- paste0("time_p_", page_ids)
    question_ts_ids <- paste0("time_q_", question_ids)
    start_page_ts_id <- page_ts_ids[which(page_ids == start_page)]
    all_ids <- c('time_end', question_ids, question_ts_ids, page_ts_ids)

    # Create current_page_id reactive value
    current_page_id <- shiny::reactiveVal(start_page)

    # Progress bar
    max_progress <- shiny::reactiveVal(0)
    last_answered_question <- shiny::reactiveVal(0)
    update_progress_bar <- function(index) {
        if (index > last_answered_question()) {
            last_answered_question(index)
            current_progress <- index / length(question_ids)
            max_progress(max(max_progress(), current_progress))
            session$sendCustomMessage("updateProgressBar", max_progress() * 100)
        }
    }

    # Capture metadata (browser, IP address, and screen resolution) if enabled
    if (capture_metadata) {
        # Initialize stored_values if needed
        if (is.null(session$userData$stored_values)) {
            session$userData$stored_values <- list()
        }

        # Capture browser information
        user_agent <- session$request$HTTP_USER_AGENT
        if (!is.null(user_agent)) {
            parsed_ua <- parse_user_agent(user_agent)
            browser_info <- paste0(
                parsed_ua$browser,
                " v",
                parsed_ua$version,
                ", ",
                parsed_ua$os
            )
            session$userData$stored_values[["browser"]] <- browser_info
        }

        # Capture IP address (try multiple headers for proxy/load balancer scenarios)
        ip_address <- get_client_ip(session$request)
        if (!is.null(ip_address)) {
            session$userData$stored_values[["ip_address"]] <- ip_address
        }

        # Screen resolution will be requested after all_data is initialized
    }

    # Initialize session handling and session_id
    session_id <- session$token
    session_id <- handle_sessions(
        session_id,
        db,
        session,
        input,
        time_start,
        start_page,
        current_page_id,
        question_ids,
        question_ts_ids,
        update_progress_bar,
        use_cookies
    )
    # Auto scroll
    session$sendCustomMessage(
        "updateSurveydownConfig",
        list(autoScrollEnabled = auto_scroll)
    )

    # Check if db is NULL (either blank or specified with ignore = TRUE)
    ignore_mode <- is.null(db)

    # Initialize translations list (from '_survey/translations.yml' file)
    translations <- get_translations()$translations

    # Keep-alive observer - this will be triggered every 60 seconds
    shiny::observeEvent(input$keepAlive, {
        cat(
            "Session keep-alive at",
            format(Sys.time(), "%m/%d/%Y %H:%M:%S"),
            "\n"
        )
    })

    # 2. show_if conditions ----

    # Separate page and question conditions
    separated_conditions <- if (!is.null(show_if)) {
        separate_show_if_conditions(show_if$conditions, page_ids, question_ids)
    } else {
        list(page_conditions = list(), question_conditions = list())
    }
    page_conditions <- separated_conditions$page_conditions
    question_conditions <- separated_conditions$question_conditions

    # Store page conditions for use in navigation
    session$userData$page_conditions <- page_conditions

    # Reactive to store visibility status of all questions
    question_visibility <- shiny::reactiveVal(
        stats::setNames(rep(TRUE, length(question_ids)), question_ids)
    )

    # Observer to apply show_if conditions and update question_visibility (questions only)
    if (length(question_conditions) > 0) {
        # Create a modified show_if object with only question conditions
        question_show_if <- list(conditions = question_conditions)

        shiny::observe({
            shiny::reactiveValuesToList(input)
            show_if_results <- set_show_if_conditions(question_show_if)()
            current_visibility <- question_visibility()
            for (target in names(show_if_results)) {
                current_visibility[target] <- show_if_results[[target]]
                if (show_if_results[[target]]) {
                    shinyjs::show(paste0('container-', target))
                } else {
                    shinyjs::hide(paste0('container-', target))
                }
            }
            question_visibility(current_visibility)
        })
    }

    # 3. Update data ----

    update_data <- function(time_last = FALSE) {
        data_list <- latest_data()
        fields <- changed_fields()

        # Only update fields that have actually changed and have values
        if (length(fields) > 0) {
            # Filter out fields with empty values unless explicitly changed
            valid_fields <- character(0)
            for (field in fields) {
                if (!is.null(data_list[[field]]) && data_list[[field]] != "") {
                    valid_fields <- c(valid_fields, field)
                }
            }
            fields <- valid_fields
        } else {
            # On initial load or restoration, use all non-empty fields
            fields <- names(data_list)[sapply(data_list, function(x) {
                !is.null(x) && x != ""
            })]
        }

        if (time_last) {
            data_list[['time_end']] <- get_utc_timestamp()
            fields <- unique(c(fields, 'time_end'))
        }

        # Local data handling
        if (ignore_mode) {
            if (file.access('.', 2) == 0) {
                tryCatch(
                    {
                        # Read existing data
                        existing_data <- if (file.exists("preview_data.csv")) {
                            utils::read.csv(
                                "preview_data.csv",
                                stringsAsFactors = FALSE
                            )
                        } else {
                            data.frame()
                        }

                        # Convert current data_list to data frame
                        new_data <- as.data.frame(
                            data_list,
                            stringsAsFactors = FALSE
                        )

                        # If there is existing data, update or append based on session_id
                        if (nrow(existing_data) > 0) {
                            # Find if this session_id already exists
                            session_idx <- which(
                                existing_data$session_id == data_list$session_id
                            )

                            if (length(session_idx) > 0) {
                                # Update existing session data
                                for (field in fields) {
                                    if (field %in% names(existing_data)) {
                                        existing_data[
                                            session_idx,
                                            field
                                        ] <- data_list[[field]]
                                    } else {
                                        # Add new column with NAs, then update the specific row
                                        existing_data[[field]] <- NA
                                        existing_data[
                                            session_idx,
                                            field
                                        ] <- data_list[[field]]
                                    }
                                }
                                updated_data <- existing_data
                            } else {
                                # Ensure all columns from existing_data are in new_data
                                missing_cols <- setdiff(
                                    names(existing_data),
                                    names(new_data)
                                )
                                for (col in missing_cols) {
                                    new_data[[col]] <- NA
                                }
                                # Ensure all columns from new_data are in existing_data
                                missing_cols <- setdiff(
                                    names(new_data),
                                    names(existing_data)
                                )
                                for (col in missing_cols) {
                                    existing_data[[col]] <- NA
                                }
                                # Now both data frames should have the same columns
                                updated_data <- rbind(
                                    existing_data,
                                    new_data[names(existing_data)]
                                )
                            }
                        } else {
                            # If no existing data, use new data
                            updated_data <- new_data
                        }

                        # Write updated data back to file
                        utils::write.csv(
                            updated_data,
                            "preview_data.csv",
                            row.names = FALSE,
                            na = ""
                        )
                    },
                    error = function(e) {
                        warning(
                            "Unable to write to preview_data.csv: ",
                            e$message
                        )
                        message("Error details: ", e$message)
                    }
                )
            } else {
                message("Running in a non-writable environment.")
            }
        } else {
            database_uploading(data_list, db$db, db$table, fields)
        }

        # Only reset changed fields that were actually processed
        changed_fields(setdiff(changed_fields(), fields))
    }

    # 4. Data tracking ----

    # First check and initialize table if needed
    if (!ignore_mode) {
        # Create a minimal initial data just for table creation
        min_initial_data <- list(
            session_id = character(0),
            time_start = character(0),
            time_end = character(0)
        )

        table_exists <- pool::poolWithTransaction(db$db, function(conn) {
            DBI::dbExistsTable(conn, db$table)
        })
        if (!table_exists) {
            create_table(min_initial_data, db$db, db$table)
        }
    }

    # Now handle session and get proper initial data
    initial_data <- get_initial_data(
        session,
        session_id,
        time_start,
        all_ids,
        start_page_ts_id
    )
    all_data <- do.call(shiny::reactiveValues, initial_data)

    # Reactive expression that returns a list of the latest data
    latest_data <- shiny::reactive({
        # Convert reactiveValues to a regular list
        data <- shiny::reactiveValuesToList(all_data)

        # Ensure all elements are of length 1, use "" for empty or NULL values
        data <- lapply(data, function(x) {
            if (
                length(x) == 0 || is.null(x) || (is.na(x) && !is.character(x))
            ) {
                ""
            } else {
                as.character(x)[1]
            }
        })

        data[names(data) != ""]
    })

    # Reactive value to track which fields have changed
    changed_fields <- shiny::reactiveVal(names(initial_data))

    # Expose all_data and changed_fields to session's userData for use by sd_store_value
    session$userData$all_data <- all_data
    session$userData$changed_fields <- changed_fields

    # Update checkpoint 1 - when session starts
    shiny::isolate({
        update_data()
    })

    # 5. Main question observers ----

    lapply(seq_along(question_ids), function(index) {
        local({
            local_id <- question_ids[index]
            local_ts_id <- question_ts_ids[index]

            shiny::observeEvent(
                input[[local_id]],
                {
                    # Tag event time and update value
                    timestamp <- get_utc_timestamp()
                    value <- input[[local_id]]
                    formatted_value <- format_question_value(value)
                    all_data[[local_id]] <- formatted_value

                    # Update timestamp and progress if interacted
                    changed <- local_id
                    if (!is.null(input[[paste0(local_id, "_interacted")]])) {
                        all_data[[local_ts_id]] <- timestamp
                        changed <- c(changed, local_ts_id)
                        update_progress_bar(index)
                    }

                    # Update tracker of which fields changed
                    changed_fields(c(changed_fields(), changed))

                    # Get question labels and values from question structure
                    question_info <- question_structure[[local_id]]
                    label_question <- question_info$label
                    options <- question_info$options
                    label_options <- names(options)

                    # For the selected value(s), get the corresponding label(s)
                    if (length(options) == length(label_options)) {
                        names(options) <- label_options
                    }
                    label_option <- if (is.null(value) || length(value) == 0) {
                        ""
                    } else {
                        options[options %in% value] |>
                            names() |>
                            paste(collapse = ", ")
                    }

                    # Store the values and labels in output
                    output[[paste0(local_id, "_value")]] <- shiny::renderText({
                        formatted_value
                    })
                    output[[paste0(
                        local_id,
                        "_label_option"
                    )]] <- shiny::renderText({
                        label_option
                    })
                    output[[paste0(
                        local_id,
                        "_label_question"
                    )]] <- shiny::renderText({
                        label_question
                    })
                },
                ignoreNULL = FALSE,
                ignoreInit = TRUE
            )
        })
    })

    # Manual range observers for range sliders auto-save
    lapply(seq_along(question_ids), function(index) {
        local({
            local_id <- question_ids[index]
            local_ts_id <- question_ts_ids[index]
            manual_id <- paste0(local_id, "_manual_range")

            shiny::observeEvent(
                input[[manual_id]],
                {
                    # Tag event time and update value
                    timestamp <- get_utc_timestamp()
                    value <- input[[manual_id]]
                    formatted_value <- format_question_value(value)
                    all_data[[local_id]] <- formatted_value

                    # Always update timestamp for manual range (auto-save scenario)
                    changed <- local_id
                    all_data[[local_ts_id]] <- timestamp
                    changed <- c(changed, local_ts_id)

                    # Update progress if interacted
                    if (!is.null(input[[paste0(local_id, "_interacted")]])) {
                        update_progress_bar(index)
                    }

                    # Update tracker of which fields changed
                    changed_fields(c(changed_fields(), changed))

                    # Get question labels and values from question structure
                    question_info <- question_structure[[local_id]]
                    label_question <- question_info$label
                    options <- question_info$options
                    label_options <- names(options)

                    # For the selected value(s), get the corresponding label(s)
                    if (length(options) == length(label_options)) {
                        names(options) <- label_options
                    }
                    label_option <- if (is.null(value) || length(value) == 0) {
                        ""
                    } else {
                        options[options %in% value] |>
                            names() |>
                            paste(collapse = ", ")
                    }

                    # Store the values and labels in output
                    output[[paste0(local_id, "_value")]] <- shiny::renderText({
                        formatted_value
                    })
                    output[[paste0(
                        local_id,
                        "_label_option"
                    )]] <- shiny::renderText({
                        label_option
                    })
                    output[[paste0(
                        local_id,
                        "_label_question"
                    )]] <- shiny::renderText({
                        label_question
                    })
                },
                ignoreNULL = FALSE,
                ignoreInit = TRUE
            )
        })
    })

    # Auto-save timestamp observers
    lapply(seq_along(question_ids), function(index) {
        local({
            local_id <- question_ids[index]
            local_ts_id <- question_ts_ids[index]
            autosave_ts_id <- paste0(local_id, "_autosave_timestamp")

            shiny::observeEvent(
                input[[autosave_ts_id]],
                {
                    # Force timestamp update for auto-saved questions
                    if (!is.null(input[[paste0(local_id, "_interacted")]])) {
                        timestamp <- get_utc_timestamp()
                        all_data[[local_ts_id]] <- timestamp
                        changed_fields(c(changed_fields(), local_ts_id))
                    }
                },
                ignoreNULL = TRUE,
                ignoreInit = TRUE
            )
        })
    })

    # Observer to update cookies with answers
    shiny::observe({
        # Get current page ID
        page_id <- current_page_id()

        # Get all questions for current page
        page_questions <- names(input)[names(input) %in% question_ids]

        # Create answers list
        answers <- list()
        last_timestamp <- NULL
        max_index <- 0

        for (q_id in page_questions) {
            # Get question value
            val <- input[[q_id]]
            if (!is.null(val)) {
                answers[[q_id]] <- val

                # If question was interacted with, check its position
                if (!is.null(input[[paste0(q_id, "_interacted")]])) {
                    # Find this question's index in the overall sequence
                    current_index <- match(q_id, question_ids)
                    if (!is.na(current_index) && current_index > max_index) {
                        max_index <- current_index
                        last_timestamp <- list(
                            id = paste0("time_q_", q_id),
                            time = get_utc_timestamp()
                        )
                    }
                }
            }
        }

        # Send to client to update cookie
        if (length(answers) > 0) {
            # Update cookies in both database and local modes
            page_data <- list(
                answers = answers,
                last_timestamp = last_timestamp
            )
            session$sendCustomMessage(
                "setAnswerData",
                list(pageId = page_id, pageData = page_data)
            )
        }
    })

    # 6. Page rendering ----

    # Create reactive values for the start page ID
    get_current_page <- shiny::reactive({
        pages[[which(sapply(pages, function(p) p$id == current_page_id()))]]
    })

    # Render main page content when current page changes
    output$main <- shiny::renderUI({
        current_page <- get_current_page()
        shiny::tagList(
            shiny::tags$div(
                class = "content",
                shiny::tags$div(
                    class = "page-columns page-rows-contents page-layout-article",
                    shiny::tags$div(
                        id = "quarto-content",
                        role = "main",
                        shiny::HTML(current_page$content)
                    )
                )
            )
        )
    })

    # Observer to trigger gray highlighting for unanswered questions when page changes
    shiny::observe({
        if (highlight_unanswered) {
            current_page <- get_current_page()
            if (!is.null(current_page)) {
                # Use JavaScript to delay highlighting until after DOM is ready and widgets initialized
                session$sendCustomMessage(
                    "delayedHighlightCheck",
                    list(
                        delay = 100, # 0.1 second delay
                        page_id = current_page$id
                    )
                )
            }
        }
    })

    # Observer for delayed highlighting check triggered by JavaScript
    shiny::observeEvent(input$delayed_highlight_trigger, {
        if (highlight_unanswered) {
            current_page <- get_current_page()
            if (!is.null(current_page)) {
                unanswered_all <- get_unanswered_all(current_page)

                # Send highlighting for all unanswered questions with specified color
                if (length(unanswered_all) > 0) {
                    session$sendCustomMessage(
                        "highlightUnansweredQuestions",
                        list(
                            questions = unanswered_all,
                            color = highlight_color
                        )
                    )
                } else {
                    # Clear unanswered highlighting if no unanswered questions
                    session$sendCustomMessage(
                        "clearUnansweredHighlights",
                        list()
                    )
                }
            }
        }
    })

    # 7. Page navigation ----

    check_required <- function(page) {
        required_questions <- page$required_questions
        is_visible <- question_visibility()[required_questions]
        result <- all(vapply(
            required_questions,
            function(q) {
                if (!is_visible[q]) {
                    return(TRUE)
                }
                if (question_structure[[q]]$is_matrix) {
                    all(sapply(question_structure[[q]]$row, function(r) {
                        check_answer(
                            paste0(q, "_", r),
                            input,
                            question_structure
                        )
                    }))
                } else {
                    check_answer(q, input, question_structure)
                }
            },
            logical(1)
        ))
        return(result)
    }

    get_unanswered_required <- function(page) {
        required_questions <- page$required_questions
        if (is.null(required_questions) || length(required_questions) == 0) {
            return(character(0))
        }

        is_visible <- question_visibility()[required_questions]
        unanswered <- character(0)

        for (q in required_questions) {
            if (!is_visible[q]) {
                next
            }

            if (question_structure[[q]]$is_matrix) {
                # For matrix questions, check each subquestion individually
                for (r in question_structure[[q]]$row) {
                    subq_id <- paste0(q, "_", r)
                    if (!check_answer(subq_id, input, question_structure)) {
                        unanswered <- c(unanswered, subq_id)
                    }
                }
            } else {
                if (!check_answer(q, input, question_structure)) {
                    unanswered <- c(unanswered, q)
                }
            }
        }

        return(unanswered)
    }

    get_unanswered_all <- function(page) {
        page_questions <- page$questions
        if (is.null(page_questions) || length(page_questions) == 0) {
            return(character(0))
        }

        is_visible <- question_visibility()[page_questions]
        unanswered <- character(0)

        for (q in page_questions) {
            if (!is_visible[q]) {
                next
            }

            if (question_structure[[q]]$is_matrix) {
                # For matrix questions, check each subquestion individually
                for (r in question_structure[[q]]$row) {
                    subq_id <- paste0(q, "_", r)
                    if (
                        !check_answer_for_highlighting(
                            subq_id,
                            input,
                            question_structure
                        )
                    ) {
                        unanswered <- c(unanswered, subq_id)
                    }
                }
            } else {
                if (
                    !check_answer_for_highlighting(q, input, question_structure)
                ) {
                    unanswered <- c(unanswered, q)
                }
            }
        }

        return(unanswered)
    }

    # Determine which page is next, then update current_page_id() to it
    shiny::observe({
        lapply(pages, function(page) {
            shiny::observeEvent(input[[page$next_button_id]], {
                shiny::isolate({
                    # Grab the time stamp of the page turn
                    timestamp <- get_utc_timestamp()

                    # Figure out page ids
                    current_page_id_val <- page$id
                    next_page_id <- get_default_next_page(
                        page,
                        page_ids,
                        page_id_to_index
                    )
                    next_page_id <- handle_skip_logic(
                        input,
                        skip_forward,
                        current_page_id_val,
                        next_page_id,
                        page_id_to_index
                    )

                    # Handle page conditions - check if target page should be shown
                    if (!is.null(next_page_id) && length(page_conditions) > 0) {
                        # First check if the target page (from skip_forward or default) should be shown
                        if (
                            !should_show_page(
                                next_page_id,
                                page_conditions,
                                session
                            )
                        ) {
                            # If target page shouldn't be shown, find next eligible page from that point
                            eligible_page <- find_next_eligible_page_from_target(
                                next_page_id,
                                page_ids,
                                page_conditions,
                                session
                            )

                            if (!is.null(eligible_page)) {
                                next_page_id <- eligible_page
                            }
                        }
                    }

                    # Save current data before validation
                    update_data()

                    if (!is.null(next_page_id) && check_required(page)) {
                        # Clear any existing highlights before navigating
                        session$sendCustomMessage(
                            "clearRequiredHighlights",
                            list()
                        )

                        # Set the current page as the next page
                        current_page_id(next_page_id)

                        # Update the page time stamp
                        next_ts_id <- page_ts_ids[which(
                            page_ids == next_page_id
                        )]
                        all_data[[next_ts_id]] <- timestamp

                        # Save the current page to all_data
                        all_data[["current_page"]] <- next_page_id

                        # Update tracker of which fields changed
                        changed_fields(c(
                            changed_fields(),
                            next_ts_id,
                            "current_page"
                        ))

                        # Save navigation data to database
                        update_data()
                    } else if (!is.null(next_page_id)) {
                        # Get list of unanswered required questions
                        unanswered_questions <- get_unanswered_required(page)

                        # Always send as character vector, even if empty
                        # This ensures consistent JSON formatting
                        if (length(unanswered_questions) == 0) {
                            unanswered_questions <- character(0)
                        }

                        # Send list to JavaScript for highlighting
                        session$sendCustomMessage(
                            "highlightRequiredQuestions",
                            list(questions = unanswered_questions)
                        )

                        # Show warning alert
                        shinyWidgets::sendSweetAlert(
                            session = session,
                            title = translations[["warning"]],
                            text = translations[["required"]],
                            type = "warning"
                        )
                    }
                })
            })
        })
    })

    # Observer to max out the progress bar when we reach the last page
    shiny::observe({
        page <- get_current_page()
        if (is.null(page$next_page_id)) {
            # Check if there are questions on this page
            page_questions <- page$questions
            if (is.null(page_questions) || length(page_questions) == 0) {
                # No questions on this page, set progress to 100%
                update_progress_bar(length(question_ids))
            } else {
                # There are questions on this page, check if all questions are answered
                # Use the same logic as get_unanswered_all to ensure consistency
                unanswered <- get_unanswered_all(page)
                if (length(unanswered) == 0) {
                    # All questions are answered, set progress to 100%
                    update_progress_bar(length(question_ids))
                }
            }
        }
    })

    # 8. Survey rating and exit ----

    # Observer for the exit survey modal
    shiny::observeEvent(input$show_exit_modal, {
        # Get current page for required question validation
        page <- get_current_page()

        # Save current data before validation
        update_data()

        # Check required questions before allowing exit
        if (check_required(page)) {
            # Clear any existing highlights before proceeding
            session$sendCustomMessage("clearRequiredHighlights", list())

            # Proceed with exit modal
            if (rate_survey) {
                shiny::showModal(shiny::modalDialog(
                    title = translations[["rating_title"]],
                    sd_question(
                        type = 'mc_buttons',
                        id = 'survey_rating',
                        label = glue::glue(
                            "{translations[['rating_text']]}:<br><small>({translations[['rating_scale']]})</small>"
                        ),
                        option = c(
                            "1" = "1",
                            "2" = "2",
                            "3" = "3",
                            "4" = "4",
                            "5" = "5"
                        )
                    ),
                    footer = shiny::tagList(
                        shiny::modalButton(translations[["cancel"]]),
                        shiny::actionButton(
                            "submit_rating",
                            translations[["submit_exit"]]
                        )
                    )
                ))
            } else {
                shiny::showModal(shiny::modalDialog(
                    title = translations[["confirm_exit"]],
                    translations[["sure_exit"]],
                    footer = shiny::tagList(
                        shiny::modalButton(translations[["cancel"]]),
                        shiny::actionButton(
                            "confirm_exit",
                            translations[["exit"]]
                        )
                    )
                ))
            }
        } else {
            # Required questions validation failed - same logic as Next button
            # Get list of unanswered required questions
            unanswered_questions <- get_unanswered_required(page)

            # Always send as character vector, even if empty
            # This ensures consistent JSON formatting
            if (length(unanswered_questions) == 0) {
                unanswered_questions <- character(0)
            }

            # Send list to JavaScript for highlighting
            session$sendCustomMessage(
                "highlightRequiredQuestions",
                list(questions = unanswered_questions)
            )

            # Show warning alert
            shinyWidgets::sendSweetAlert(
                session = session,
                title = translations[["warning"]],
                text = translations[["required"]],
                type = "warning"
            )
        }
    })

    # Observer to handle the rating submission or exit confirmation
    shiny::observeEvent(input$submit_rating, {
        # Save the rating
        rating <- input$survey_rating
        all_data[['exit_survey_rating']] <- rating
        changed_fields(c(changed_fields(), 'exit_survey_rating'))
        # Update checkpoint 3 - when submitting rating
        shiny::isolate({
            update_data(time_last = TRUE)
        })
        # Close the modal and the window
        shiny::removeModal()
        session$sendCustomMessage("closeWindow", list())
    })

    shiny::observeEvent(input$confirm_exit, {
        # Update checkpoint 4 - when exiting survey
        shiny::isolate({
            update_data(time_last = TRUE)
        })
        # Close the modal and the window
        shiny::removeModal()
        session$sendCustomMessage("closeWindow", list())
    })

    # Update checkpoint 5 - when session ends
    shiny::onSessionEnded(function() {
        shiny::isolate({
            update_data(time_last = TRUE)
        })
    })
}

#' Define forward skip conditions for survey pages
#'
#' @description
#' This function is used to define conditions under which certain pages in the
#' survey should be skipped ahead to (forward only). It takes one or more formulas
#' where the left-hand side is the condition and the right-hand side is the target page ID.
#'
#' @param ... One or more formulas defining skip conditions.
#'   The left-hand side of each formula should be a condition based on input
#'   values, and the right-hand side should be the ID of the page to skip to if
#'   the condition is met. Only forward skipping (to pages later in the sequence) is allowed.
#'
#' @return A list of parsed conditions, where each element contains the
#' condition and the target page ID.
#'
#' @examples
#' if (interactive()) {
#'   library(surveydown)
#'
#'   # Get path to example survey file
#'   survey_path <- system.file("examples", "sd_skip_forward.qmd",
#'                              package = "surveydown")
#'
#'   # Copy to a temporary directory
#'   temp_dir <- tempdir()
#'   file.copy(survey_path, file.path(temp_dir, "survey.qmd"))
#'   orig_dir <- getwd()
#'   setwd(temp_dir)
#'
#'   # Define a minimal server
#'   server <- function(input, output, session) {
#'
#'     # Skip forward to specific pages based on fruit selection
#'     sd_skip_forward(
#'       input$fav_fruit == "apple" ~ "apple_page",
#'       input$fav_fruit == "orange" ~ "orange_page",
#'       input$fav_fruit == "other" ~ "other_page"
#'     )
#'
#'     sd_server()
#'   }
#'
#'   # Run the app
#'   shiny::shinyApp(ui = sd_ui(), server = server)
#'
#'   # Clean up
#'   setwd(orig_dir)
#' }
#'
#' @seealso `sd_show_if()`
#'
#' @export
sd_skip_forward <- function(...) {
    conditions <- parse_conditions(...)
    calling_env <- parent.frame()

    # Process each condition
    processed_conditions <- lapply(conditions, function(rule) {
        tryCatch(
            {
                # Store the original condition for use with function calls
                rule$original_condition <- rule$condition

                # Extract any reactive expressions that might be called
                # We're storing the environment for potential evaluation later
                rule$calling_env <- calling_env

                # # For debugging
                # cat("Captured condition: ", deparse(rule$condition), "\n")

                return(rule)
            },
            error = function(e) {
                warning("Error processing condition: ", e$message)
                return(rule)
            }
        )
    })

    # Store in userData
    shiny::isolate({
        session <- shiny::getDefaultReactiveDomain()
        if (is.null(session)) {
            stop(
                "sd_skip_forward must be called within a Shiny reactive context"
            )
        }
        if (is.null(session$userData$skip_forward)) {
            session$userData$skip_forward <- list()
        }
        session$userData$skip_forward$conditions <- processed_conditions
        session$userData$skip_forward$targets <- get_unique_targets(
            processed_conditions
        )
    })
}

#' Define skip conditions for survey pages (Deprecated)
#'
#' @description
#' This function is deprecated. Please use `sd_skip_forward()` instead.
#'
#' This function is used to define conditions under which certain pages in the
#' survey should be skipped. It now behaves like `sd_skip_forward()` where only forward
#' skipping is allowed to prevent navigation loops.
#'
#' @param ... One or more formulas defining skip conditions.
#'   The left-hand side of each formula should be a condition based on input
#'   values, and the right-hand side should be the ID of the page to skip to if
#'   the condition is met.
#'
#' @return A list of parsed conditions, where each element contains the
#' condition and the target page ID.
#'
#' @export
sd_skip_if <- function(...) {
    # v0.9.0
    .Deprecated("sd_skip_forward()")

    sd_skip_forward(...)
}

#' Define show conditions for survey questions and pages
#'
#' @description
#' This function is used to define conditions under which certain questions or pages in the survey should be shown.
#' It takes one or more formulas where the left-hand side is the condition and the right-hand side is the target question ID or page ID.
#' If called with no arguments, it will return `NULL` and set no conditions.
#'
#' @param ... One or more formulas defining show conditions.
#'   The left-hand side of each formula should be a condition based on input values,
#'   and the right-hand side should be the ID of the question or page to show if the condition is met.
#'
#' @return A list of parsed conditions, where each element contains the condition and the target question or page ID.
#'   Returns `NULL` if no conditions are provided.
#'
#' @examples
#' if (interactive()) {
#'   library(surveydown)
#'
#'   # Get path to example survey file
#'   survey_path <- system.file("examples", "sd_show_if.qmd",
#'                              package = "surveydown")
#'
#'   # Copy to a temporary directory
#'   temp_dir <- tempdir()
#'   file.copy(survey_path, file.path(temp_dir, "survey.qmd"))
#'   orig_dir <- getwd()
#'   setwd(temp_dir)
#'
#'   # Define a minimal server
#'   server <- function(input, output, session) {
#'
#'     sd_show_if(
#'       # If "Other" is chosen, show the conditional question
#'       input$fav_fruit == "other" ~ "fav_fruit_other",
#'       # If condition is met, show specific page
#'       input$category == "advanced" ~ "advanced_page"
#'     )
#'
#'     sd_server()
#'   }
#'
#'   # Run the app
#'   shiny::shinyApp(ui = sd_ui(), server = server)
#'
#'   # Clean up
#'   setwd(orig_dir)
#' }
#'
#' @seealso `sd_skip_forward()`
#'
#' @export
sd_show_if <- function(...) {
    conditions <- parse_conditions(...)
    calling_env <- parent.frame()

    # Process each condition
    processed_conditions <- lapply(conditions, function(rule) {
        tryCatch(
            {
                # Store the original condition for use with function calls
                rule$original_condition <- rule$condition

                # Store the calling environment for later evaluation
                rule$calling_env <- calling_env

                # # For debugging
                # cat("Captured show_if condition: ", deparse(rule$condition), "\n")

                return(rule)
            },
            error = function(e) {
                warning("Error processing show_if condition: ", e$message)
                return(rule)
            }
        )
    })

    # Create a list in userData to store the show_if targets
    shiny::isolate({
        session <- shiny::getDefaultReactiveDomain()
        if (is.null(session)) {
            stop("sd_show_if must be called within a Shiny reactive context")
        }
        if (is.null(session$userData$show_if)) {
            session$userData$show_if <- list()
        }
        session$userData$show_if$conditions <- processed_conditions
        session$userData$show_if$targets <- get_unique_targets(
            processed_conditions
        )
    })
}

#' Set password for surveydown survey
#'
#' This function sets your surveydown password, which is used to access
#' the 'PostgreSQL' data (e.g. Supabase). The password is saved in a `.Renviron`
#' file and adds `.Renviron` to `.gitignore`.
#'
#' @param password Character string. The password to be set for the database
#'   connection.
#'
#' @details The function performs the following actions:
#'   1. Creates a `.Renviron` file in the root directory if it doesn't exist.
#'   2. Adds or updates the `SURVEYDOWN_PASSWORD` entry in the `.Renviron` file.
#'   3. Adds `.Renviron` to `.gitignore` if it's not already there.
#'
#' @return None. The function is called for its side effects.
#'
#' @examples
#' \dontrun{
#'   # Set a temporary password for demonstration
#'   temp_password <- paste0(sample(letters, 10, replace = TRUE), collapse = "")
#'
#'   # Set the password
#'   sd_set_password(temp_password)
#'
#'   # After restarting R, verify the password was set
#'   cat("Password is :", Sys.getenv('SURVEYDOWN_PASSWORD'))
#' }
#'
#' @export
sd_set_password <- function(password) {
    # v0.8.0
    .Deprecated("sd_db_config")

    # Define the path to .Renviron file
    renviron_path <- file.path(getwd(), ".Renviron")

    # Check if .Renviron file exists, if not create it
    if (!file.exists(renviron_path)) {
        file.create(renviron_path)
    }

    # Read existing content
    existing_content <- readLines(renviron_path)

    # Check if SURVEYDOWN_PASSWORD is already defined
    password_line_index <- grep("^SURVEYDOWN_PASSWORD=", existing_content)

    # Prepare the new password line
    new_password_line <- paste0("SURVEYDOWN_PASSWORD=", password)

    # If SURVEYDOWN_PASSWORD is already defined, replace it; otherwise, append it
    if (length(password_line_index) > 0) {
        existing_content[password_line_index] <- new_password_line
    } else {
        existing_content <- c(existing_content, new_password_line)
    }

    # Write the updated content back to .Renviron
    writeLines(existing_content, renviron_path)

    # Add .Renviron to .gitignore if not already there
    gitignore_path <- file.path(getwd(), ".gitignore")
    if (file.exists(gitignore_path)) {
        gitignore_content <- readLines(gitignore_path)
        if (!".Renviron" %in% gitignore_content) {
            # Remove any trailing empty lines
            while (
                length(gitignore_content) > 0 &&
                    gitignore_content[length(gitignore_content)] == ""
            ) {
                gitignore_content <- gitignore_content[
                    -length(gitignore_content)
                ]
            }
            # Add .Renviron to the end without an extra newline
            gitignore_content <- c(gitignore_content, ".Renviron")
            writeLines(gitignore_content, gitignore_path)
        }
    } else {
        writeLines(".Renviron", gitignore_path)
    }

    message("Password set successfully and .Renviron added to .gitignore.")
}

#' Create a reactive value that is also stored in survey data
#'
#' This function creates a reactive value similar to Shiny's reactive() function,
#' but also automatically stores the calculated value in the survey data.
#'
#' @param id Character string. The id (name) of the value to be stored in the data.
#' @param expr An expression that calculates a value based on inputs
#' @param blank_na Logical. If TRUE, NA values are converted to empty strings. Default is TRUE.
#'
#' @return A reactive expression that can be called like a function
#'
#' @examples
#' # This example shows how sd_reactive would be used in the app.R file
#' if (interactive()) {
#'   library(surveydown)
#'   library(shiny)
#'
#'   # Demo app setup
#'   server <- function(input, output, session) {
#'     # Create a reactive value that is stored in survey data
#'     product <- sd_reactive("product", {
#'       as.numeric(input$first_number) * as.numeric(input$second_number)
#'     })
#'
#'     # Display the result
#'     output$result <- renderText({
#'       paste("The product is:", product())
#'     })
#'
#'     # The rest of your survey setup...
#'     sd_server()
#'   }
#'
#'   # In your survey.qmd file, you would use:
#'   # The product is: `r sd_output("product", type = "value")`
#' }
#'
#' @export
sd_reactive <- function(id, expr, blank_na = TRUE) {
    # Validate id
    if (!is.character(id) || length(id) != 1) {
        stop("'id' must be a single character string")
    }

    # Capture the expression and its environment
    expr_call <- substitute(expr)
    expr_env <- parent.frame()

    # Create a reactive expression
    reactive_expr <- shiny::reactive({
        # Get current session
        session <- shiny::getDefaultReactiveDomain()
        if (is.null(session)) {
            warning(
                "sd_reactive() must be called within a Shiny reactive context"
            )
            return(NULL)
        }

        # Use tryCatch to safely evaluate the expression
        tryCatch(
            {
                # Evaluate the expression in its original environment
                result <- eval(expr_call, envir = expr_env)

                # Store the value in the survey data
                if (is.null(result) || (length(result) == 1 && is.na(result))) {
                    sd_store_value("", id)
                    return(if (blank_na) "" else result)
                } else {
                    sd_store_value(result, id)
                    return(result)
                }
            },
            error = function(e) {
                warning("Error in sd_reactive for ", id, ": ", e$message)
                sd_store_value("", id)
                return(if (blank_na) "" else NULL)
            }
        )
    })

    # Auto-trigger the evaluation once to ensure value is available
    # This creates an observer that will run once when the session initializes
    shiny::observeEvent(
        shiny::getDefaultReactiveDomain()$clientData,
        {
            # This forces the reactive to run once right away
            reactive_expr()
        },
        once = TRUE
    )

    # Create a separate observer that will monitor the reactive expression
    shiny::observe({
        # Wrap in tryCatch to prevent errors from crashing the app
        tryCatch(
            {
                reactive_expr()
            },
            error = function(e) {
                warning(
                    "Error in sd_reactive observer for ",
                    id,
                    ": ",
                    e$message
                )
            }
        )
    })

    return(reactive_expr)
}

#' Create a copy of a value
#'
#' This function creates a copy of an input value and makes it available as a
#' new output. The new output can then be displayed using `sd_output()`.
#'
#' @param id Character string. The ID of the input value to copy.
#' @param id_copy Character string. The ID for the new copy (must be different
#'   from `id`).
#'
#' @return `NULL` invisibly. This function is called for its side effects.
#'
#' @examples
#' if (interactive()) {
#'   library(surveydown)
#'
#'   # Get path to example survey file
#'   survey_path <- system.file("examples", "sd_ui.qmd",
#'                              package = "surveydown")
#'
#'   # Copy to a temporary directory
#'   temp_dir <- tempdir()
#'   file.copy(survey_path, file.path(temp_dir, "sd_copy_value.qmd"))
#'   orig_dir <- getwd()
#'   setwd(temp_dir)
#'
#'   # Define a minimal server
#'   server <- function(input, output, session) {
#'
#'     # Make a copy of the "name" variable to call its value a second time
#'     sd_copy_value(id = "name", id_copy = "name_copy")
#'
#'     sd_server()
#'   }
#'
#'   # Run the app
#'   shiny::shinyApp(ui = sd_ui(), server = server)
#'
#'   # Clean up
#'   setwd(orig_dir)
#' }
#'
#' @seealso `sd_output()` for displaying the copied value
#'
#' @export
sd_copy_value <- function(id, id_copy) {
    if (id == id_copy) {
        stop("The 'id_copy' must be different from the 'id'")
    }
    shiny::isolate({
        output <- shiny::getDefaultReactiveDomain()$output
        input <- shiny::getDefaultReactiveDomain()$input
        output_id <- paste0(id_copy, "_value")
        if (!is.null(output)) {
            output[[output_id]] <- shiny::renderText({
                input[[id]]
            })
        } else {
            warning(
                "sd_copy_value was not called within a Shiny reactive context"
            )
        }
    })
    invisible(NULL)
}

#' Check if a question is answered
#'
#' This function checks if a given question has been answered by the user.
#' For matrix questions, it checks if all sub-questions (rows) are answered.
#'
#' @param question_id The ID of the question to check.
#' @return A logical value: `TRUE` if the question is answered, `FALSE`
#' otherwise.
#'
#' @examples
#' if (interactive()) {
#'   library(surveydown)
#'
#'   # Get path to example survey file
#'   survey_path <- system.file("examples", "sd_is_answered.qmd",
#'                              package = "surveydown")
#'
#'   # Copy to a temporary directory
#'   temp_dir <- tempdir()
#'   file.copy(survey_path, file.path(temp_dir, "survey.qmd"))
#'   orig_dir <- getwd()
#'   setwd(temp_dir)
#'
#'   # Define a minimal server
#'   server <- function(input, output, session) {
#'
#'     sd_show_if(
#'       # If "apple_text" is answered, show the conditional question
#'       sd_is_answered("apple_text") ~ "other_fruit"
#'     )
#'
#'     sd_server()
#'   }
#'
#'   # Run the app
#'   shiny::shinyApp(ui = sd_ui(), server = server)
#'
#'   # Clean up
#'   setwd(orig_dir)
#' }
#'
#' @export
sd_is_answered <- function(question_id) {
    # Get the Shiny session
    session <- shiny::getDefaultReactiveDomain()

    if (is.null(session)) {
        stop(
            "sd_is_answered() must be called from within a Shiny reactive context"
        )
    }

    # Access the input object from the session
    input <- session$input

    # Check if it's a matrix question (ends with a number)
    if (!grepl("_\\d+$", question_id)) {
        # It's potentially a matrix question, check all sub-questions
        sub_questions <- grep(
            paste0("^", question_id, "_"),
            names(input),
            value = TRUE
        )

        if (length(sub_questions) > 0) {
            # It's confirmed to be a matrix question
            return(all(sapply(sub_questions, function(sq) {
                !is.null(input[[sq]]) && nzchar(input[[sq]])
            })))
        }
    }

    # For non-matrix questions or individual sub-questions
    if (is.null(input[[question_id]])) {
        return(FALSE)
    }

    if (is.list(input[[question_id]])) {
        # For questions that can have multiple answers (e.g., checkboxes)
        return(
            length(input[[question_id]]) > 0 &&
                any(nzchar(unlist(input[[question_id]])))
        )
    } else {
        # For single-answer questions
        return(!is.null(input[[question_id]]) && nzchar(input[[question_id]]))
    }
}

# Helper functions ----

# Helper function to find the next eligible page starting from a specific target
find_next_eligible_page_from_target <- function(
    target_page_id,
    page_ids,
    page_conditions,
    session
) {
    target_index <- which(page_ids == target_page_id)

    if (length(target_index) == 0) {
        return(NULL) # Target page not found
    }

    # Look for the next eligible page starting from the target
    for (i in target_index:length(page_ids)) {
        candidate_page <- page_ids[i]
        if (should_show_page(candidate_page, page_conditions, session)) {
            return(candidate_page)
        }
    }

    return(NULL) # No eligible page found
}

# Helper function to separate page and question conditions
separate_show_if_conditions <- function(
    show_if_conditions,
    page_ids,
    question_ids
) {
    if (is.null(show_if_conditions) || length(show_if_conditions) == 0) {
        return(list(page_conditions = list(), question_conditions = list()))
    }

    page_conditions <- list()
    question_conditions <- list()

    for (condition in show_if_conditions) {
        target <- condition$target
        if (target %in% page_ids) {
            page_conditions <- c(page_conditions, list(condition))
        } else if (target %in% question_ids) {
            question_conditions <- c(question_conditions, list(condition))
        } else {
            warning(sprintf(
                "Target '%s' is neither a page nor a question ID",
                target
            ))
        }
    }

    return(list(
        page_conditions = page_conditions,
        question_conditions = question_conditions
    ))
}

# Helper function to evaluate if a page should be shown based on conditions
should_show_page <- function(page_id, page_conditions, session) {
    if (length(page_conditions) == 0) {
        return(TRUE) # No conditions, always show
    }

    # Check if this page has any conditions
    page_condition <- NULL
    for (condition in page_conditions) {
        if (condition$target == page_id) {
            page_condition <- condition
            break
        }
    }

    # If no condition for this page, show it
    if (is.null(page_condition)) {
        return(TRUE)
    }

    # Evaluate the condition
    tryCatch(
        {
            result <- eval(
                page_condition$original_condition,
                envir = page_condition$calling_env
            )
            return(isTRUE(result))
        },
        error = function(e) {
            warning(
                "Error evaluating page condition for ",
                page_id,
                ": ",
                e$message
            )
            return(TRUE) # Default to showing the page if evaluation fails
        }
    )
}

# Helper function to find the next eligible page to show
find_next_eligible_page <- function(
    current_page_id,
    config,
    page_conditions,
    session
) {
    page_ids <- config$page_ids
    current_index <- which(page_ids == current_page_id)

    if (length(current_index) == 0) {
        return(NULL) # Current page not found
    }

    # Look for the next eligible page
    for (i in (current_index + 1):length(page_ids)) {
        if (i > length(page_ids)) {
            break
        }

        candidate_page <- page_ids[i]
        if (should_show_page(candidate_page, page_conditions, session)) {
            return(candidate_page)
        }
    }

    return(NULL) # No eligible next page found
}

set_show_if_conditions <- function(show_if) {
    if (is.null(show_if) || length(show_if$conditions) == 0) {
        return(shiny::reactive(list()))
    }
    shiny::reactive({
        results <- lapply(show_if$conditions, function(rule) {
            result <- tryCatch(
                {
                    evaluate_condition(rule)
                },
                error = function(e) {
                    warning(sprintf(
                        "Error in show_if condition for target '%s', condition '%s': %s",
                        rule$target,
                        deparse(rule$condition),
                        conditionMessage(e)
                    ))
                    FALSE
                }
            )
            stats::setNames(list(result), rule$target)
        })
        do.call(c, results)
    })
}

get_unique_targets <- function(a) {
    return(unique(sapply(a, function(x) x$target)))
}

parse_conditions <- function(...) {
    conditions <- list(...)
    lapply(conditions, function(cond) {
        if (!inherits(cond, "formula")) {
            stop("Each condition must be a formula (condition ~ target)")
        }
        list(
            condition = cond[[2]], # Left-hand side of the formula
            target = eval(cond[[3]]) # Right-hand side of the formula
        )
    })
}

evaluate_condition <- function(rule) {
    # Create a safe evaluation environment that can handle reactive expressions
    session <- shiny::getDefaultReactiveDomain()
    eval_env <- list(input = session$input)

    # Try to evaluate using the original condition (which might have function calls)
    tryCatch(
        {
            # Use both the original calling environment and the input
            result <- eval(
                rule$original_condition,
                envir = rule$calling_env,
                enclos = environment()
            )
            return(isTRUE(result))
        },
        error = function(e) {
            warning("Error in condition evaluation: ", e$message)
            return(FALSE)
        }
    )
}

get_stored_vals <- function(session) {
    shiny::isolate({
        if (is.null(session)) {
            stop(
                "get_stored_vals must be called from within a Shiny reactive context"
            )
        }
        stored_vals <- session$userData$stored_values
        if (is.null(stored_vals)) {
            return(NULL)
        }

        # Format stored values as a list
        formatted_vals <- lapply(stored_vals, function(val) {
            if (is.null(val)) "" else val
        })

        return(formatted_vals)
    })
}

get_utc_timestamp <- function() {
    return(format(Sys.time(), tz = "UTC", usetz = TRUE))
}

get_initial_data <- function(
    session,
    session_id,
    time_start,
    all_ids,
    start_page_ts_id
) {
    # Initialize with static data
    data <- c(
        list(session_id = session_id, time_start = time_start),
        get_stored_vals(session)
    )

    # Process deferred values with session persistence check
    if (!is.null(session$userData$deferred_values)) {
        db <- session$userData$db

        # If we have a database connection, check for existing values
        if (!is.null(db)) {
            # Get current session ID for persistence check
            current_session_id <- session$token
            persistent_session_id <- shiny::isolate(
                session$input$stored_session_id
            )

            search_session_id <- if (
                !is.null(persistent_session_id) &&
                    nchar(persistent_session_id) > 0
            ) {
                persistent_session_id
            } else {
                current_session_id
            }

            # Get existing data for this session
            existing_data <- get_db_data(db, search_session_id)

            # Check which values should be skipped for database updates
            skip_db_values <- session$userData$deferred_skip_db
            if (is.null(skip_db_values)) {
                skip_db_values <- character(0)
            }

            # Process each deferred value
            for (id in names(session$userData$deferred_values)) {
                # Check if value already exists in database
                value_exists <- if (
                    !is.null(existing_data) && nrow(existing_data) > 0
                ) {
                    id %in%
                        names(existing_data) &&
                        !is.na(existing_data[[id]]) &&
                        existing_data[[id]] != ""
                } else {
                    FALSE
                }

                # Always add to initial data, but check if we should skip database updates
                data[[id]] <- session$userData$deferred_values[[id]]
            }
        } else {
            # Local CSV mode - check for existing values in preview_data.csv
            search_session_id <- get_session_id(session, NULL)

            # Get existing data from local CSV
            all_local_data <- get_local_data()
            existing_data <- if (!is.null(all_local_data)) {
                all_local_data[all_local_data$session_id == search_session_id, ]
            } else {
                NULL
            }

            # Check which values should be skipped for local updates
            skip_db_values <- session$userData$deferred_skip_db
            if (is.null(skip_db_values)) {
                skip_db_values <- character(0)
            }

            # Process each deferred value
            for (id in names(session$userData$deferred_values)) {
                # Check if value already exists in local CSV
                value_exists <- if (
                    !is.null(existing_data) && nrow(existing_data) > 0
                ) {
                    id %in%
                        names(existing_data) &&
                        !is.na(existing_data[[id]]) &&
                        existing_data[[id]] != ""
                } else {
                    FALSE
                }

                if (value_exists) {
                    # Use existing value from CSV for session persistence
                    data[[id]] <- existing_data[[id]]
                } else {
                    # Use new deferred value
                    data[[id]] <- session$userData$deferred_values[[id]]
                }
            }
        }
    }

    # Initialize question & timestamp values
    for (id in all_ids) {
        data[[id]] <- ""
    }
    data[['time_start']] <- time_start
    data[[start_page_ts_id]] <- time_start
    data[['time_end']] <- ""

    return(data)
}

# Helper function to format a single question value
format_question_value <- function(val) {
    if (is.null(val) || identical(val, NA) || identical(val, "NA")) {
        return("")
    } else if (length(val) > 1) {
        return(paste(val, collapse = ", "))
    } else {
        return(as.character(val))
    }
}

get_default_next_page <- function(page, page_ids, page_id_to_index) {
    if (is.null(page$next_page_id)) {
        return(NULL)
    }
    next_page_id <- page$next_page_id
    if (next_page_id == "") {
        index <- page_id_to_index[page$id] + 1
        if (index <= length(page_ids)) {
            return(page_ids[index])
        } else {
            return(NULL)
        }
    }
    return(next_page_id)
}

handle_skip_logic <- function(
    input,
    skip_forward,
    current_page_id,
    next_page_id,
    page_id_to_index
) {
    if (is.null(next_page_id) | is.null(skip_forward)) {
        return(next_page_id)
    }

    # Get the current page index and page object
    current_page_index <- page_id_to_index[current_page_id]

    # Loop through each skip forward logic condition
    if (!is.null(skip_forward) && !is.null(skip_forward$conditions)) {
        conditions <- skip_forward$conditions
        for (i in seq_along(conditions)) {
            rule <- conditions[[i]]

            # Ignore the condition if already on target page
            if (current_page_id == rule$target) {
                next
            }

            # Ignore the condition if not a forward direction skip
            target_page_index <- page_id_to_index[rule$target]
            if (target_page_index <= current_page_index) {
                next
            }

            # Evaluate the condition
            condition_result <- tryCatch(
                {
                    evaluate_condition(rule)
                },
                error = function(e) {
                    warning(sprintf(
                        "Error in sd_skip_forward condition for target '%s': %s",
                        rule$target,
                        conditionMessage(e)
                    ))
                    FALSE
                }
            )

            # Check if the condition is met
            if (condition_result) {
                return(rule$target)
            }
        }
    }

    return(next_page_id)
}

# Check if a single question is answered
check_answer <- function(q, input, question_structure = NULL) {
    answer <- input[[q]]
    if (is.null(answer)) {
        return(FALSE)
    }

    # For question types that have default values, check if user has actually interacted
    # These types often have default values that shouldn't count as "answered"
    interacted <- input[[paste0(q, "_interacted")]]

    # Also check for auto-save timestamp (indicates auto-save occurred)
    autosave_timestamp <- input[[paste0(q, "_autosave_timestamp")]]
    if (is.null(interacted) && !is.null(autosave_timestamp)) {
        interacted <- TRUE
    }

    # Smart auto-save detection: ONLY apply if normal validation would fail
    # This ensures we don't interfere with normal user interaction tracking
    if (is.null(interacted) && !is.null(answer)) {
        # Get question type from question_structure if available
        if (!is.null(question_structure) && q %in% names(question_structure)) {
            q_type_raw <- question_structure[[q]]$type

            # Map raw HTML classes to proper question types (same as in write_question_structure_yaml)
            type_replacement <- c(
                'shiny-input-text form-control' = 'text',
                'shiny-input-textarea form-control' = 'textarea',
                'shiny-input-number form-control' = 'numeric',
                'form-group shiny-input-radiogroup shiny-input-container' = 'mc',
                'radio-group-buttons' = 'mc_buttons',
                'form-group shiny-input-checkboxgroup shiny-input-container' = 'mc_multiple',
                'checkbox-group-buttons' = 'mc_multiple_buttons',
                'shiny-input-select' = 'select',
                'js-range-slider sw-slider-text' = 'slider',
                'js-range-slider' = 'slider_numeric',
                'shiny-date-input form-group shiny-input-container' = 'date',
                'shiny-date-range-input form-group shiny-input-container' = 'daterange'
            )

            # Map the raw type to the proper type
            q_type <- type_replacement[q_type_raw]
            if (is.na(q_type)) {
                q_type <- q_type_raw
            } # Fallback to raw if no mapping

            # ONLY apply smart detection for auto-save supported types
            # and ONLY when there's no interaction (meaning this is default value scenario)
            if (
                !is.null(q_type) &&
                    q_type %in%
                        c("slider", "slider_numeric", "date", "daterange")
            ) {
                interacted <- TRUE
            }
        }
    }

    # Get question type from question_structure if available
    q_type <- NULL
    if (!is.null(question_structure) && q %in% names(question_structure)) {
        q_type <- question_structure[[q]]$type
    }

    # Handle based on question type if available
    if (!is.null(q_type)) {
        # Check if this is a slider type (could be "slider", "slider_numeric", or contain slider-related classes)
        is_slider <- grepl("slider", q_type, ignore.case = TRUE) ||
            q_type %in% c("slider", "slider_numeric")

        if (is_slider) {
            # Sliders always have defaults, require interaction tracking
            if (!is.null(interacted)) {
                return(isTRUE(interacted))
            } else {
                return(FALSE)
            }
        }
    }

    if (is.character(answer)) {
        # For text/textarea inputs, require both interaction AND non-empty content
        if (!is.null(interacted) && isTRUE(interacted)) {
            return(any(nzchar(answer)))
        } else {
            return(any(nzchar(answer)))
        }
    } else if (is.numeric(answer) || is.logical(answer)) {
        # For numeric text inputs (sliders are already handled above), require both interaction AND valid content
        if (!is.null(interacted) && isTRUE(interacted)) {
            # User interacted - check if they provided actual content
            if (is.logical(answer) || all(is.na(answer))) {
                return(FALSE) # No valid content provided
            }
            return(TRUE) # Valid content provided
        } else {
            # Without interaction tracking, check for valid content
            if (is.logical(answer) || all(is.na(answer))) {
                return(FALSE)
            }
            return(TRUE)
        }
    } else if (inherits(answer, "Date")) {
        # For date inputs, require user interaction if available
        if (!is.null(interacted)) {
            return(isTRUE(interacted))
        } else {
            # Fallback: check if date is not NA and not today's date (common default)
            return(any(!is.na(answer)) && !all(answer == Sys.Date()))
        }
    } else if (is.list(answer)) {
        # For date ranges and other list inputs
        if (!is.null(interacted)) {
            return(isTRUE(interacted))
        } else {
            return(any(!sapply(answer, is.null)))
        }
    }

    return(TRUE) # Default to true for unknown types
}

# Check if a question should be highlighted (based on interaction only, no smart detection)
check_answer_for_highlighting <- function(q, input, question_structure = NULL) {
    # For highlighting purposes, only check actual user interaction
    # Do NOT use smart detection - we want to show gray for untouched questions
    interacted <- input[[paste0(q, "_interacted")]]

    # If user has explicitly interacted, don't highlight
    if (!is.null(interacted) && isTRUE(interacted)) {
        return(TRUE) # Interacted = answered for highlighting purposes
    }

    # For all question types, if no interaction flag, show as unanswered for highlighting
    return(FALSE)
}

get_local_data <- function() {
    if (file.exists("preview_data.csv")) {
        tryCatch(
            {
                return(utils::read.csv(
                    "preview_data.csv",
                    stringsAsFactors = FALSE
                ))
            },
            error = function(e) {
                warning("Error reading preview_data.csv: ", e$message)
                return(NULL)
            }
        )
    }
    return(NULL)
}

get_cookie_data <- function(session, current_page_id) {
    # Get stored answer data from input
    answer_data <- session$input$stored_answer_data

    if (is.null(answer_data) || !length(answer_data)) {
        return(NULL)
    }

    # Extract data for current page
    page_data <- answer_data[[current_page_id]]
    if (is.null(page_data)) {
        return(NULL)
    }

    # Return the full page data structure including answers and timestamps
    return(page_data)
}

restore_current_page_values <- function(
    restore_data,
    session,
    page_filter = NULL
) {
    for (col in names(restore_data)) {
        # Skip special columns
        if (
            !col %in% c("session_id", "current_page", "time_start", "time_end")
        ) {
            val <- restore_data[[col]]
            if (!is.null(val) && !is.na(val) && val != "") {
                session$sendInputMessage(
                    col,
                    list(value = val, priority = "event")
                )
            }
        }
    }
}

handle_data_restoration <- function(
    session_id,
    db,
    session,
    current_page_id,
    start_page,
    question_ids,
    question_ts_ids,
    progress_updater
) {
    if (is.null(session_id)) {
        return(NULL)
    }

    # Get data based on source
    if (!is.null(db)) {
        all_data <- sd_get_data(db)
    } else {
        all_data <- get_local_data()
    }

    # If no data available, return NULL
    if (is.null(all_data)) {
        return(NULL)
    }

    restore_data <- all_data[all_data$session_id == session_id, ]

    if (nrow(restore_data) == 0) {
        return(NULL)
    }

    shiny::isolate({
        # 1. Restore page state (using restore_data)
        if ("current_page" %in% names(restore_data)) {
            restored_page <- restore_data[["current_page"]]
            if (
                !is.null(restored_page) &&
                    !is.na(restored_page) &&
                    nchar(restored_page) > 0
            ) {
                current_page_id(restored_page)
            } else {
                current_page_id(start_page)
            }
        } else {
            current_page_id(start_page)
        }

        # Get cookie data after page state is set
        answer_data <- NULL
        if (!is.null(db)) {
            answer_data <- get_cookie_data(session, current_page_id())
        }

        # 2. Find the last answered question for progress bar
        last_index <- 0
        if (
            !is.null(db) &&
                !is.null(answer_data) &&
                !is.null(answer_data$last_timestamp)
        ) {
            # Use last timestamp from cookie data in DB mode
            last_ts_id <- answer_data$last_timestamp$id
            # Find the index of this timestamp ID in our question_ts_ids
            last_index <- match(last_ts_id, question_ts_ids)
            if (is.na(last_index)) last_index <- 0
        } else {
            # Use restore_data for local CSV mode
            for (i in seq_along(question_ids)) {
                ts_id <- question_ts_ids[i]
                if (ts_id %in% names(restore_data)) {
                    ts_val <- restore_data[[ts_id]]
                    if (
                        length(ts_val) == 1 &&
                            !is.null(ts_val) &&
                            !is.na(ts_val) &&
                            ts_val != ""
                    ) {
                        last_index <- i
                    }
                }
            }
        }

        if (last_index > 0) {
            progress_updater(last_index)
        }

        # 3. Restore question values
        if (
            !is.null(db) &&
                !is.null(answer_data) &&
                !is.null(answer_data$answers)
        ) {
            # Use answer data from cookies for current page
            for (col in names(answer_data$answers)) {
                val <- answer_data$answers[[col]]
                if (!is.null(val) && !identical(val, "")) {
                    session$sendInputMessage(
                        col,
                        list(value = val, priority = "event")
                    )
                }
            }
        } else {
            # Fall back to restore_data
            restore_current_page_values(restore_data, session)
        }
    })
    return(restore_data)
}

handle_sessions <- function(
    session_id,
    db = NULL,
    session,
    input,
    time_start,
    start_page,
    current_page_id,
    question_ids,
    question_ts_ids,
    progress_updater,
    use_cookies = TRUE
) {
    # Note: Cookies can work in both database and local modes
    # No need to disable cookies when db is NULL

    # Check 2: Cookies enabled?
    if (!use_cookies) {
        return(session_id)
    }

    # Create a variable to store the final ID
    final_session_id <- session_id

    # Do the cookie check synchronously in a reactive context
    shiny::isolate({
        # Check 3: Cookie exists and is valid?
        stored_id <- shiny::reactiveValuesToList(input)$stored_session_id
        if (
            !is.null(stored_id) &&
                nchar(stored_id) > 0 &&
                # Check 4: Either DB connection exists or preview_data.csv is writable
                (!is.null(db) ||
                    (file.exists("preview_data.csv") &&
                        file.access("preview_data.csv", 2) == 0))
        ) {
            # Check 5: Session exists in DB or preview data?
            restore_data <- handle_data_restoration(
                stored_id,
                db,
                session,
                current_page_id,
                start_page,
                question_ids,
                question_ts_ids,
                progress_updater
            )

            if (!is.null(restore_data)) {
                # All checks passed - use stored session
                final_session_id <- stored_id
                session$sendCustomMessage(
                    "setCookie",
                    list(sessionId = stored_id)
                )
            } else {
                # Session not in DB - use new session
                session$sendCustomMessage(
                    "setCookie",
                    list(sessionId = session_id)
                )
            }
        } else {
            # No cookie or no DB connection - use new session
            session$sendCustomMessage("setCookie", list(sessionId = session_id))
        }
    })

    return(final_session_id)
}
