#' S3 class TD
#'
#' \code{createTD}\cr
#' Function for creating objects of S3 class TD (Trial Data). The function
#' converts a data.frame to an object of class TD in the following steps:
#' \itemize{
#' \item{Check input data}
#' \item{Rename columns to default column names - default column names:
#' genotype, trial, loc, year, repId, subBlock, rowCoord, colCoord, rowId,
#' colId, checkId}
#' \item{Convert column types to default column types - rowCoord and colCoord
#' are converted to numeric columns, all other renamed columns to factor
#' columns. Columns other than the default columns, e.g. traits or other
#' covariates will be included in the output unchanged.}
#' \item{Split input data by trial - each trial in the input data will become
#' a list item in the output.}
#' \item{Add meta data - the trial meta data are added as attributes to the
#' different output items. The function parameters starting with "tr" provide
#' the meta data. Their values will be recycled if needed, so by setting a
#' single "trDesign", all trials will get the same design. The meta data can be
#' changed later on using \code{getMeta} and \code{setMeta}}
#' }
#' \code{addTD}\cr
#' Function for adding extra trial data to an existing object of class TD. The
#' data for the new trials will be added after the data for existing trials. It
#' is possible to add data for an already existing trial, but this will cause
#' multiple items in the output with identical names, which might cause problems
#' later on in the analysis. Therefore a warning will be issued in this
#' case.\cr\cr
#' \code{dropTD}\cr
#' Function for removing data for selected trials from an existing object of
#' class TD.\cr\cr
#' \code{\link{summary.TD}} and \code{\link{plot.TD}} methods are available.
#'
#' @param data A data.frame containing trial data with at least a column for
#' genotype. The data.frame should be in a wide format, i.e. all available
#' phenotypic data should be in a separate column within the data.frame.
#' @param genotype An optional character string indicating the column in
#' \code{data} that contains genotypes.
#' @param trial An optional character string indicating the column in
#' \code{data} that contains trials.
#' @param loc An optional character string indicating the column in
#' \code{data} that contains trial locations.
#' @param year An optional character string indicating the column in \code{data}
#' that contains years.
#' @param repId An optional character string indicating the column in
#' \code{data} that contains replicates.
#' @param subBlock An optional character string indicating the column in
#' \code{data} that contains sub blocks.
#' @param plotId An optional character string indicating the column in
#' \code{data} that contains plots. This column will be combined with trial
#' to a single output factor.
#' @param rowCoord An optional character string indicating the column in
#' \code{data} that contains the row coordinates.
#' @param colCoord An optional character string indicating the column in
#' \code{data} that contains the column coordinates.
#' @param rowId An optional character string indicating the column in
#' \code{data} that contains field rows. If not supplied, this is assumed to
#' be the same as rowCoord.
#' @param colId An optional character string indicating the column in
#' \code{data} that contains field columns. If not supplied, this is assumed to
#' be the same as colCoord.
#' @param checkId An optional character string indicating the column in
#' \code{data} that contains the check IDs.
#' @param trLocation An optional character vector indicating the locations of
#' the trials. This will be used as default names when creating plots and
#' summaries. If no locations are provided, first the column loc is considered.
#' If this contains one unique value for a trial this is used as trLocation.
#' Otherwise the trialname is used.
#' @param trDate An optional date vector indicating the dates of the trials.
#' @param trDesign An optional character vector indicating the designs of the
#' trials. Either "none" (no (known) design), "ibd" (incomplete-block design),
#' "res.ibd" (resolvable incomplete-block design), "rcbd" (randomized complete
#' block design), "rowcol" (row-column design) or "res.rowcol" (resolvable
#' row-column design).
#' @param trLat An optional numerical vector indicating the latitudes of the
#' trials on a scale of -90 to 90.
#' @param trLong An optional numerical vector indicating the longitudes of the
#' trials on a scale of -180 to 180.
#' @param trPlWidth An optional positive numerical vector indicating the
#' widths of the plots.
#' @param trPlLength An optional positive numerical vector indicating the
#' lengths of the plots.
#'
#' @return An object of class TD, a list of data.frames with renamed columns
#' and an attribute \code{renamedCols} containing an overview of renamed
#' columns. For each unique value of trial, the output has a data.frame in
#' the list with the same name as the trial. These data.frames have attributes
#' containing the metadata for the corresponding trial. If there is no column
#' for trial, the list will contain one item named after the input data.
#'
#' @examples
#' ## Create a data.frame with to be converted to TD object.
#' ## The data consists of genotype, trial, row and column information and
#' ## two traits, yield and flowering time.
#' datT1 <- data.frame(geno = paste0("G", 1:10), tr = "T1",
#'                     row = rep(1:5, each = 2), col = rep(1:2, times = 5),
#'                     yield = 1:10, flowering = 3:12)
#'
#' ## Convert data.frame to TD object.
#' TDT1 <- createTD(data = datT1, genotype = "geno", trial = "tr",
#'                  rowCoord = "row", colCoord = "col")
#'
#' ## Create a second data.frame similar to the first with data for a second trial.
#' datT2<- data.frame(geno = paste0("G", 1:10), tr = "T2",
#'                     row = rep(1:2, each = 5), col = rep(1:5, times = 2),
#'                     yield = 10:1, flowering = 12:3)
#'
#' ## Add this data to the TD object created above.
#' TDTot <- addTD(TD = TDT1, data = datT2, genotype = "geno", trial = "tr",
#'                rowCoord = "row", colCoord = "col")
#'
#' ## Drop the data for the first trial from the object.
#' TDT2 <- dropTD(TD = TDTot, rmTrials = "T1")
#'
#' @family functions for TD objects
#'
#' @importFrom utils hasName
#'
#' @name TD
NULL

#' @rdname TD
#' @export
createTD <- function(data,
                     genotype = NULL,
                     trial = NULL,
                     loc = NULL,
                     year = NULL,
                     repId = NULL,
                     subBlock = NULL,
                     plotId = NULL,
                     rowCoord = NULL,
                     colCoord = NULL,
                     rowId = rowCoord,
                     colId = colCoord,
                     checkId = NULL,
                     trLocation = NULL,
                     trDate = NULL,
                     trDesign = NULL,
                     trLat = NULL,
                     trLong = NULL,
                     trPlWidth = NULL,
                     trPlLength = NULL) {
  ## Save name of original data for naming output.
  dataName <- deparse(substitute(data))
  if (length(dataName) > 1) {
    dataName <- "dat"
  }
  ## Checks.
  if (missing(data) || !is.data.frame(data)) {
    stop("data has to be a data.frame.\n")
  }
  ## Convert input to data.frame. This needs to be done to be able to handle
  ## tibbles and possibly other data structures in the future.
  data <- as.data.frame(data)
  cols <- colnames(data)
  for (param in c(genotype, trial, loc, year, repId, subBlock, plotId,
                  rowId, colId, rowCoord, colCoord, checkId)) {
    if (!is.null(param) && (!is.character(param) || length(param) > 1 ||
                            !hasName(data, param))) {
      stop(deparse(param), " has to be NULL or a column in data.\n")
    }
  }
  checkTDMeta(trDesign = trDesign, trLat = trLat, trLong = trLong,
              trPlWidth = trPlWidth, trPlLength = trPlLength)
  ## Create list of reserved column names for renaming columns.
  renameCols <- c("genotype", "trial", "loc", "year", "repId", "plotId",
                  "subBlock", "rowId", "colId", "rowCoord", "colCoord",
                  "checkId")
  ## First rename duplicate colums and add duplicated columns to data
  renameFrom <- as.character(sapply(X = renameCols, FUN = function(x) {
    get(x)
  }))
  ## Create a data.frame with renamed cols to add to TD as an attribute.
  renamed <- data.frame(orig = renameFrom[renameFrom != "NULL"],
                        new = renameCols[renameFrom != "NULL"],
                        stringsAsFactors = FALSE)
  ## Get duplicate columns.
  dupCols <- which(duplicated(renameFrom) & renameFrom != "NULL")
  for (dupCol in dupCols) {
    ## Copy original column as extra column in data for each duplicate.
    tempName <- paste0(".temp", dupCol)
    data[tempName] <- data[, colnames(data) == renameFrom[dupCol]]
    ## Add new replacementname to cols and renameFrom.
    cols[length(cols) + 1] <- tempName
    renameFrom[dupCol] <- tempName
  }
  ## Rename columns.
  for (i in 1:length(renameCols)) {
    cols[cols == renameFrom[i]] <- renameCols[i]
  }
  colnames(data) <- cols
  ## Convert columns to factor if neccessary.
  factorCols <-  c("genotype", "trial", "loc", "year", "repId", "subBlock",
                   "plotId", "rowId", "colId", "checkId")
  for (factorCol in factorCols) {
    if (hasName(data, factorCol)) {
      data[cols == factorCol] <- as.factor(data[, cols == factorCol])
    }
  }
  ## Combine plotId and trial into a single factor if both are available.
  ## If trial is not available plotId itself was converted to factor in the
  ## previous step.
  if (all(hasName(data, c("trial", "plotId")))) {
    data$plotId <- interaction(data$trial, data$plotId, sep = "_")
  }
  ## Convert columns to numeric if neccessary.
  numCols <- c("rowCoord", "colCoord")
  for (numCol in numCols) {
    if (hasName(data, numCol) && !is.numeric(data[cols == numCol])) {
      data[cols == numCol] <- as.numeric(data[, cols == numCol])
    }
  }
  ## Sort data by rowCoord and colCoord.
  ## This is only needed for spatial modeling with asreml but doesn't harm
  ## always doing so.
  if (all(hasName(data, c("rowCoord", "colCoord")))) {
    data <- data[order(data[["rowCoord"]], data[["colCoord"]]), ]
  }
  if (hasName(data, "trial")) {
    listData <- split(x = data, f = droplevels(data$trial))
  } else {
    listData <- setNames(list(data), dataName)
  }
  ## Define meta data to set from input variables.
  meta <- c("trLocation", "trDate", "trDesign", "trLat", "trLong",
            "trPlWidth", "trPlLength")
  ## Expand input values for meta variables to number of trials.
  metaVals <- sapply(X = meta, FUN = function(m) {
    if (!is.null(get(m))) {
      metaVal <- rep(x = get(m), length.out = length(listData))
      if (is.null(names(metaVal)) || !all(hasName(listData, names(metaVal)))) {
        names(metaVal) <- names(listData)
      }
      return(metaVal)
    } else {
      NULL
    }
  }, simplify = FALSE)
  ## Set meta for all trials in data.
  for (tr in names(listData)) {
    for (m in meta) {
      ## Set meta data. Set to NULL if not in input so meta variable is
      ## left out meta data. This to avoid a list of NULL.
      attr(x = listData[[tr]], which = m) <- unname(metaVals[[m]][tr])
    }
    ## Location should always be filled since it is used in plot titles as
    ## well. Use trial name as default value.
    if (is.null(trLocation)) {
      if (hasName(x = listData[[tr]], name = "loc") &
          length(unique(listData[[tr]][["loc"]])) == 1) {
        attr(x = listData[[tr]],
             which = "trLocation") <- as.character(listData[[tr]][["loc"]][1])
      } else {
        attr(x = listData[[tr]], which = "trLocation") <- tr
      }
    }
    ## Add a list of columns that have been renamed as attribute to TD.
    attr(x = listData[[tr]], which = "renamedCols") <-
      if (nrow(renamed) > 0) renamed else NULL
  }
  TD <- structure(listData,
                  class = c("TD", "list"))
  return(TD)
}

#' @param TD An object of class TD which should be modified.
#'
#' @rdname TD
#' @export
addTD <- function(TD,
                  data,
                  genotype = NULL,
                  trial = NULL,
                  loc = NULL,
                  year = NULL,
                  repId = NULL,
                  subBlock = NULL,
                  plotId = NULL,
                  rowCoord = NULL,
                  colCoord = NULL,
                  rowId = rowCoord,
                  colId = colCoord,
                  checkId = NULL,
                  trLocation = NULL,
                  trDate = NULL,
                  trDesign = NULL,
                  trLat = NULL,
                  trLong = NULL,
                  trPlWidth = NULL,
                  trPlLength = NULL) {
  TDNw <- createTD(data = data, genotype = genotype, trial = trial,
                   loc = loc, year = year, repId = repId,
                   subBlock = subBlock, plotId = plotId, rowCoord = rowCoord,
                   colCoord = colCoord, rowId = rowId, colId = colId,
                   checkId = checkId, trLocation = trLocation, trDate = trDate,
                   trDesign = trDesign, trLat = trLat, trLong = trLong,
                   trPlWidth = trPlWidth, trPlLength = trPlLength)
  dupTrials <- names(TDNw)[names(TDNw) %in% names(TD)]
  if (length(dupTrials) > 0) {
    warning("The following trials already existed in TD and will be added ",
            "again: ", paste(dupTrials, collapse = ", "), ".\n", call. = FALSE)
  }
  TDTot <- c(TD, TDNw)
  class(TDTot) <- c("TD", "list")
  return(TDTot)
}

#' @param rmTrials A character vector of trials that should be removed.
#'
#' @rdname TD
#' @export
dropTD <- function(TD,
                   rmTrials) {
  naTrials <- rmTrials[!rmTrials %in% names(TD)]
  if (length(naTrials) > 0) {
    warning("The following trials are not in TD: ",
            paste(naTrials, collapse = ", "), ".\n", call. = FALSE)
  }
  leftTrials <- names(TD)[!names(TD) %in% rmTrials]
  if (length(leftTrials) == 0) {
    warning("All trials have been removed from TD.\n", call. = FALSE)
  }
  return(TD[!names(TD) %in% rmTrials])
}

#' Summarizing objects of class \code{TD}
#'
#' \code{summary} method for class \code{TD}.
#'
#' @param object An object of class TD.
#' @param ... Further arguments - currently not used.
#' @param trial A character string specifying the trial to be summarised.
#' @param traits A character vector specifying the traits to be summarised.
#' @param groupBy A character string specifying a column in TD by which the
#' summary should be grouped. If \code{NULL}, no grouping is done.
#' @param what A character vector indicating which summary statistics should be
#' computed. If \code{what = "all"}, all available statistics are computed.\cr
#' Possible options are:
#' \describe{
#' \item{nVals}{The number of values, i.e. non-missing + missing values.}
#' \item{nObs}{The number of non-missing observations.}
#' \item{nMiss}{The number of missing values.}
#' \item{mean}{The mean.}
#' \item{median}{The median.}
#' \item{min}{The minimum.}
#' \item{max}{The maximum.}
#' \item{range}{The range (maximum - minimum).}
#' \item{firstQ}{The first (25\%) quantile.}
#' \item{thirdQ}{The third (75\%) quantile.}
#' \item{sd}{The standard deviation.}
#' \item{seMean}{The standard error of mean.}
#' \item{var}{The variance.}
#' \item{seVar}{The standard error of variance.}
#' \item{CV}{The coefficient of variation.}
#' \item{sum}{The sum.}
#' \item{sumSq}{The sum of squares.}
#' \item{uncorSumSq}{The uncorrected sum of squares.}
#' \item{skew}{The skewness.}
#' \item{seSkew}{The standard error of the skewness.}
#' \item{kurt}{The kurtosis.}
#' \item{seKurt}{The standard error of the kurtosis.}
#' \item{all}{All summary statistics.}
#' }
#'
#' @return A table containing the selected summary statistics.
#'
#' @family functions for TD objects
#'
#' @examples
#' ## Summarize TDHeat05.
#' summary(TDHeat05, traits = "yield")
#'
#' ## Summarize TDHeat05 grouping by repId.
#' summary(TDHeat05, traits = "yield", groupBy = "repId")
#'
#' @export
summary.TD <- function(object,
                       ...,
                       trial = names(object),
                       traits,
                       groupBy = NULL,
                       what = if (!is.null(groupBy)) {
                         c("nObs", "mean", "sd")
                       } else {
                         c("nObs", "nMiss", "mean", "median", "min", "max",
                           "firstQ", "thirdQ", "var")
                       }) {
  allStat <- data.frame(stat = c("nVals", "nObs", "nMiss", "mean", "median",
                                 "min","max", "range", "firstQ", "thirdQ", "sd",
                                 "seMean", "var", "seVar", "CV", "sum", "sumSq",
                                 "uncorSumSq", "skew", "seSkew", "kurt",
                                 "seKurt"),
                        name = c("Number of values", "Number of observations",
                                 "Number of missing values", "Mean", "Median",
                                 "Min", "Max", "Range", "First quantile",
                                 "Third quantile", "Standard deviation",
                                 "Standard error of mean", "Variance",
                                 "Standard error of variance",
                                 "Coefficient of variation", "Sum of values",
                                 "Sum of squares", "Uncorrected sum of squares",
                                 "Skewness", "Standard Error of Skewness",
                                 "Kurtosis", "Standard Error of Kurtosis"),
                        stringsAsFactors = FALSE)
  ## Checks.
  if (!is.character(trial) || length(trial) > 1 ||
      !hasName(x = object, name = trial)) {
    stop("trial should be a single character string in ",
         deparse(substitute(object)), ".\n")
  }
  trDat <- object[[trial]]
  if (!is.character(traits) || !all(hasName(x = trDat, name = traits))) {
    stop("All traits should be columns in trial.\n")
  }
  if (!is.null(groupBy) && (!is.character(groupBy) || length(groupBy) > 1 ||
                            !hasName(x = trDat, name = groupBy))) {
    stop("groupBy should be a single character string indicating ",
         "a column in trial")
  }
  if (what[[1]] == "all") {
    what <- allStat[["stat"]]
  }
  if (!is.character(what) || all(!what %in% allStat[["stat"]])) {
    stop("At least one statistic should be chosen.\n")
  }
  whichStat <- which(allStat[["stat"]] %in% what)
  what <- allStat[whichStat, "stat"]
  if (!is.null(groupBy)) {
    groups <- unique(trDat[[groupBy]])
  } else {
    trDat[[".tot"]] <- 1
    groupBy <- ".tot"
    groups <- 1
  }
  ## Create an array to store the values.
  stats <- array(dim = c(length(what), length(traits), length(groups)),
                 dimnames = list(what, traits, groups))
  for (i in seq_along(traits)) {
    for (j in seq_along(groups)) {
      trDatGr <- trDat[trDat[[groupBy]] == groups[j], traits[i]]
      if ("nVals" %in% what) {
        stats["nVals", i, j] <- length(trDatGr)
      }
      if ("nObs" %in% what) {
        stats["nObs", i, j] <- length(na.omit(trDatGr))
      }
      if ("nMiss" %in% what) {
        stats["nMiss", i, j] <- sum(is.na(trDatGr))
      }
      if ("mean" %in% what) {
        stats["mean", i, j] <- mean(trDatGr, na.rm = TRUE)
      }
      if ("median" %in% what) {
        stats["median", i, j] <- median(trDatGr, na.rm = TRUE)
      }
      if ("min" %in% what) {
        stats["min", i, j] <- min(trDatGr, na.rm = TRUE)
      }
      if ("max" %in% what) {
        stats["max", i, j] <- max(trDatGr, na.rm = TRUE)
      }
      if ("range" %in% what) {
        stats["range", i, j] <- diff(range(trDatGr, na.rm = TRUE))
      }
      if ("firstQ" %in% what) {
        stats["firstQ", i, j] <- quantile(trDatGr, prob = .25, na.rm = TRUE)
      }
      if ("thirdQ" %in% what) {
        stats["thirdQ", i, j] <- quantile(trDatGr, prob = .75, na.rm = TRUE)
      }
      if ("sd" %in% what) {
        stats["sd", i, j] <- sd(trDatGr, na.rm = TRUE)
      }
      if ("seMean" %in% what) {
        stats["seMean", i, j] <- sd(trDatGr, na.rm = TRUE) /
          sqrt(length(na.omit(trDatGr)))
      }
      if ("var" %in% what) {
        stats["var", i, j] <- var(trDatGr, na.rm = TRUE)
      }
      if ("seVar" %in% what) {
        stats["seVar", i, j] <- seVar(trDatGr, na.rm = TRUE)
      }
      if ("CV" %in% what) {
        stats["CV", i, j] <- 100 * sd(trDatGr, na.rm = TRUE) /
          mean(trDatGr, na.rm = TRUE)
      }
      if ("sum" %in% what) {
        stats["sum", i, j] <- sum(trDatGr, na.rm = TRUE)
      }
      if ("sumSq" %in% what) {
        stats["sumSq", i, j] <- sum((na.omit(trDatGr) -
                                       mean(trDatGr, na.rm = TRUE)) ^ 2)
      }
      if ("uncorSumSq" %in% what) {
        stats["uncorSumSq", i, j] <- sum(trDatGr ^ 2, na.rm = TRUE)
      }
      if ("skew" %in% what) {
        stats["skew", i, j] <- skewness(trDatGr, na.rm = TRUE)
      }
      if ("seSkew" %in% what) {
        stats["seSkew", i, j] <- seSkewness(length(na.omit(trDatGr)))
      }
      if ("kurt" %in% what) {
        stats["kurt", i, j] <- kurtosis(trDatGr, na.rm = TRUE)
      }
      if ("seKurt" %in% what) {
        stats["seKurt", i, j] <- seKurtosis(length(na.omit(trDatGr)))
      }
    }
  }
  rownames(stats) <- allStat[whichStat, "name"]
  attr(x = stats, which = "whichStat") <- whichStat
  return(structure(stats,
                   class = c("summary.TD", "array"),
                   trial = trial,
                   group = if (groupBy != ".tot") groupBy else NULL))
}

#' @export
print.summary.TD <- function(x, ...) {
  whichStat <- attr(x, "whichStat")
  groupBy  <- attr(x, "group")
  decimals <- c(rep(x = 0, times = 3), rep(x = 2, times = 7),
                rep(x = 3, times = 5), rep(x = 2, times = 3),
                rep(x = 3, times = 4))[whichStat]
  xPrint <- x
  for (i in seq_along(decimals)) {
    xPrint[i, , ] <- format(x[i, , ], digits = decimals[i],
                            nsmall = decimals[i])
  }
  for (i in 1:ncol(xPrint)) {
    trait <- colnames(xPrint)[i]
    cat(paste("\nSummary statistics for", trait, "in", attr(x, "trial"),
              if (!is.null(groupBy)) paste("grouped by", groupBy), "\n\n"))
    if (dim(xPrint)[3] > 1) {
      xPrintM <- matrix(nrow = nrow(xPrint), ncol = dim(xPrint)[3])
      for (j in 1:nrow(xPrint)) {
        xPrintM[j, ] <- xPrint[j, i, ]
      }
      dimnames(xPrintM) <- list(rownames(xPrint), dimnames(xPrint)[[3]])
      print(xPrintM, quote = FALSE, right = TRUE)
    } else {
      xPrintM <- as.matrix(xPrint[, i , 1])
      dimnames(xPrintM) <- list(rownames(xPrint), trait)
      print(xPrintM, quote = FALSE, right = TRUE)
    }
  }
  cat("\n")
}

#' Plot function for class TD
#'
#' Plotting function for objects of class TD. Plots either the layout of the
#' different trials within the TD object or locates the trials on a map. Also a
#' boxplot can be made for selected traits and trials, a plot of correlations
#' between trials and a scatter plot matrix. A detailed description and optional
#' extra parameters of the different plots is given in the sections below.
#'
#' @section Layout Plot:
#' Plots the layout of the selected trials (all available trials by default).
#' This plot can only be made for trials that contain both row (\code{rowCoord})
#' and column (\code{colCoord}) information. If either one of those is missing
#' the trial is skipped with a warning. If blocks (\code{subBlock}) are
#' available for a trial these are indicated in different colors per block,
#' otherwise all plots are colored in grey. If replicates (\code{repId}) are
#' available a black line is plotted between diffent replicates. Missing plots
#' are indicated in white. This can either be single plots in a trial or
#' complete missing columns or rows.\cr
#' Extra parameter options:
#' \describe{
#' \item{showGeno}{Should individual genotypes be indicated in the plot?
#' Defaults to \code{FALSE}}
#' \item{highlight}{A character vector of genotypes to be highlighted in the
#' plot.}
#' \item{colorSubBlock}{Should subBlocks be colored with a different color per
#' subBlock? Defaults to \code{FALSE}. \code{colorSubBlock} is ignored when
#' highlight is used to highlight genotypes.}
#' }
#'
#' @section Map Plot:
#' A map is plotted with the locations of the trials in the TD object.
#' Mapping the trials is done based on lattitude and longitude that can be
#' added when creating an object of class TD. Trials without latitude and/or
#' longitude available are skipped with a warning message. The countries in
#' which the trials are located will be plotted on a single map and the
#' location of the trials will be indicated on this map. The actual plot is
#' made using ggplot, but for getting the data for the borders of the countries
#' the maps package is needed.\cr
#' Extra parameter options:
#' \describe{
#' \item{minLatRange}{A positive numerical value indicating the minimum range
#' (in degrees) for the latitude on the plotted map. Defaults to 10.}
#' \item{minLongRange}{A positive numerical value indicating the minimum range
#' (in degrees) for the longitud on the plotted map. Defaults to 5.}
#' }
#'
#' @section Box Plot:
#' Creates a boxplot per selected trait grouped by trial. Extra parameter
#' options:
#' \describe{
#' \item{groupBy}{A character string indicating a column in \code{TD} by which
#' the boxes in the plot should be grouped. By default the boxes are grouped
#' per trial.}
#' \item{colorBy}{A character string indicating a column in \code{TD} by which
#' the boxes are colored. Coloring will be done within the groups indicated by
#' the \code{groupBy} parameter.}
#' \item{orderBy}{A character string indicating the way the boxes should be
#' ordered. Either "alphabetic" for alphabetical ordering of the groups,
#' "ascending" for ordering by ascending mean, or "descending" for ordering by
#' descending mean. Default boxes are ordered alphabetically.}
#' }
#'
#' @section Correlation Plot:
#' Draws a heatmap of correlations between trials per selected trait. If
#' genotypes are replicated within trials genotypic means are taken before
#' computing correlations. The order of the trials in the heatmap is determined
#' by clustering them.
#'
#' @section Scatter Plot:
#' Draws a scatter plot matrix per selected trait. If genotypes are replicated
#' within trials genotypic means are taken before plotting. The lower left of
#' the matix contains scatter plots between trials. The diagonal contains
#' histograms of the data per trial.\cr
#' Extra parameter options:
#' \describe{
#' \item{colorBy}{A character string indicating a column in \code{TD} by which
#' the genotypes in the scatter plots are colored.}
#' \item{trialOrder}{A character vector indicating the order of the trials in
#' the plot matrix (left to right and top to bottom). This vector should be a
#' permutation of all trials plotted.}
#' \item{addCorr}{A character string indicating the position of the correlation
#' diplayed in each plot, either "tl" for top left, "tr", for top right, "bl"
#' for bottom left or "br" for bottom right. If \code{NULL}, the default, then
#' no correlation is added to the plot.}
#' }
#'
#' @param x An object of class TD.
#' @param ... Extra plot options. Described per plotType in their respective
#' section.
#' @param plotType A single character string indicating which plot should be
#' made. See the sections below for a detailed explanation of the plots.
#' @param trials A character vector indicating the trials to be plotted when
#' plotting field layouts. Only used if \code{plotType} = "layout" or "box".
#' @param traits A character vector indicating the traits to be plotted in
#' a boxplot. Only used if \code{plotType} = "box" or "cor".
#' @param output Should the plot be output to the current device? If
#' \code{FALSE} only a list of ggplot objects is invisibly returned.
#'
#' @family functions for TD objects
#'
#' @examples
#' data("wheatChl")
#'
#' ## Create a TD object.
#' wheatTD <- createTD(data = wheatChl, genotype = "trt", repId = "rep",
#'                     subBlock = "bl", rowCoord = "row", colCoord = "col")
#'
#' ## Add meta data to be able to plot locations on a map.
#' wheatMeta <- getMeta(wheatTD)
#' wheatMeta$trLocation <- c("Cauquenes", rep("Santa Rosa", times = 4))
#' wheatMeta$trLat <- c(-35.58, rep(-36.32, times = 4))
#' wheatMeta$trLong <- c(-72.17, rep(-71.55, times = 4))
#' wheatTD <- setMeta(wheatTD, wheatMeta)
#'
#' ### Layout plot.
#'
#' ## Plot the layout of one of the trials.
#' plot(wheatTD, trials = "C_SWS_12")
#'
#' ## Highlight some of the genotypes in the layout.
#' plot(wheatTD, trials = "C_SWS_12", highlight = c("G001", "G002"))
#'
#' ### Map plot.
#'
#' ## Plot the location of the trials on the map.
#' plot(wheatTD, plotType = "map")
#'
#' ### Box plot.
#'
#' ## Create a box plot for GY.
#' plot(wheatTD, plotType = "box", traits = "GY")
#'
#' ## Add coloring by repId to the boxes.
#' plot(wheatTD, plotType = "box", traits = "GY", colorBy = "repId")
#'
#' ## Sort the boxes in descending order.
#' plot(wheatTD, plotType = "box", traits = "GY", orderBy = "descending")
#'
#' ### Correlation plot.
#'
#' ## Plot the correlations between trials for GY.
#' plot(wheatTD, plotType = "cor", traits = "GY")
#'
#' ### Scatter plot
#'
#' ## Plot scatter plot for GY.
#' plot(wheatTD, plotType = "scatter", traits = "GY")
#'
#' ## Add correlations to top left corner of plots.
#' plot(wheatTD, plotType = "scatter", traits = "GY", addCorr = "tl")
#'
#' @importFrom grDevices hcl.colors hcl.pals
#' @export
plot.TD <- function(x,
                    ...,
                    plotType = c("layout", "map", "box", "cor", "scatter"),
                    trials = names(x),
                    traits = NULL,
                    output = TRUE) {
  ## Checks.
  trials <- chkTrials(trials, x)
  plotType <- match.arg(plotType)
  dotArgs <- list(...)
  if (plotType == "layout") {
    showGeno <- isTRUE(dotArgs$showGeno)
    highlight <- dotArgs$highlight
    colorSubBlock <- isTRUE(dotArgs$colorSubBlock)
    if (!is.null(highlight)) {
      chkChar(highlight, null = FALSE)
    }
    p <- setNames(vector(mode = "list", length = length(trials)), trials)
    for (trial in trials) {
      trDat <- x[[trial]]
      if (!chkRowCol(trDat)) next
      if (length(highlight) > 0) {
        trDat[["highlight."]] <- ifelse(trDat[["genotype"]] %in% highlight,
                                        as.character(trDat[["genotype"]]), NA)
      }
      trLoc <- attr(trDat, "trLocation")
      plotRep <- hasName(x = trDat, name = "repId")
      plotSubBlock <- hasName(x = trDat, name = "subBlock")
      ## Compute min and max for row and column coordinates.
      yMin <- min(trDat[["rowCoord"]])
      yMax <- max(trDat[["rowCoord"]])
      xMin <- min(trDat[["colCoord"]])
      xMax <- max(trDat[["colCoord"]])
      ## Create data.frame with all rows columns in field.
      ## Full missing rows/columns are included.
      ## If not geom_tile just fills the empty columns by expanding the
      ## neighbouring colums (or rows).
      fullGrid <- expand.grid(colCoord = xMin:xMax, rowCoord = yMin:yMax)
      trDat <- merge(fullGrid, trDat, all.x = TRUE)
      trDat[!is.na(trDat[["rowId"]]), "color."] <- "grey75"
      ## Compute aspect for proper depiction of field size. If no information
      ## is available plots are assumed to be square.
      ylen <- attr(trDat, "trPlLength")
      xlen <- attr(trDat, "trPlWidth")
      if (is.null(ylen) || is.null(xlen)) {
        aspect <- length(unique(trDat[["colCoord"]])) /
          length(unique(trDat[["rowCoord"]]))
      } else {
        aspect <- ylen / xlen
      }
      ## Create data for lines between replicates.
      if (plotRep) {
        repBord <- calcPlotBorders(trDat = trDat, bordVar = "repId")
      }
      ## Create base plot.
      pTr <- ggplot(data = trDat, aes_string(x = "colCoord", y = "rowCoord")) +
        coord_fixed(ratio = aspect,
                    xlim = range(trDat[["colCoord"]]) + c(-0.5, 0.5),
                    ylim = range(trDat[["rowCoord"]]) + c(-0.5, 0.5),
                    clip = "off") +
        theme(panel.background = element_blank(),
              plot.title = element_text(hjust = 0.5)) +
        ## Move ticks to edge of the plot.
        scale_x_continuous(breaks = scales::pretty_breaks(), expand = c(0, 0)) +
        scale_y_continuous(breaks = scales::pretty_breaks(), expand = c(0, 0)) +
        ggtitle(trLoc)
      if (sum(!is.na(trDat[["highlight."]])) > 0) {
        ## Genotypes to be highlighted get a color.
        ## Everything else the NA color.
        pTr <- pTr + geom_tile(aes_string(fill = "highlight.",
                                          color = "color.")) +
          scale_color_manual(values = "grey75", na.translate = FALSE,
                             na.value = "transparant") +
          ## Remove NA from scale.
          scale_fill_discrete(na.translate = FALSE) +
          labs(fill = "Highlighted") +
          guides(color = "none")
      } else if (plotSubBlock && colorSubBlock) {
        ## Color tiles by subblock.
        pTr <- pTr + geom_tile(aes_string(fill = "subBlock",
                                          color = "color.")) +
          scale_color_manual(values = "grey75", na.translate = FALSE,
                             na.value = "transparant") +
          guides(fill = guide_legend(ncol = 3), color = "none")
      } else {
        ## No subblocks and no hightlights so just a single fill color.
        pTr <- pTr + geom_tile(aes_string(color = "color."), fill = "white") +
          scale_color_manual(values = "grey75", na.translate = FALSE,
                             na.value = "transparant") +
          guides(color = "none")
      }
      ## Create data for lines between subBlocks.
      if (plotSubBlock) {
        subBlockBord <- calcPlotBorders(trDat = trDat, bordVar = "subBlock")
        ## Add horizontal and vertical lines as segment.
        ## adding/subtracting 0.5 assures plotting at the borders of
        ## the tiles.
        pTr <- pTr + geom_segment(aes_string(x = "x - 0.5", xend = "x - 0.5",
                                             y = "y - 0.5", yend = "y + 0.5",
                                             linetype = "'subBlocks'"),
                                  data = subBlockBord$vertW, size = 0.4) +
          geom_segment(aes_string(x = "x - 0.5", xend = "x + 0.5",
                                  y = "y - 0.5", yend = "y - 0.5"),
                       data = subBlockBord$horW, size = 0.4)
      }
      if (showGeno) {
        ## Add names of genotypes to the center of the tiles.
        pTr <- pTr + geom_text(aes_string(label = "genotype"),
                               size = 2, check_overlap = TRUE)
      }
      if (plotRep) {
        ## Add lines for replicates.
        ## Add horizontal and vertical lines as segment.
        ## adding/subtracting 0.5 assures plotting at the borders of
        ## the tiles.
        pTr <- pTr +  geom_segment(aes_string(x = "x - 0.5", xend = "x - 0.5",
                                              y = "y - 0.5", yend = "y + 0.5",
                                              linetype = "'replicates'"),
                                   data = repBord$vertW, size = 1) +
          geom_segment(aes_string(x = "x - 0.5", xend = "x + 0.5",
                                  y = "y - 0.5", yend = "y - 0.5"),
                       data = repBord$horW, size = 1)
      }
      if (plotSubBlock || plotRep) {
        shwVals <- c(plotRep, plotSubBlock)
        pTr <- pTr +
          ## Add a legend entry for replicates and subBlocks.
          scale_linetype_manual(c("replicates", "subBlocks")[shwVals],
                                values = c("replicates" = "solid",
                                           "subBlocks" = "solid")[shwVals],
                                name = element_blank()) +
          guides(linetype = guide_legend(override.aes =
                                           list(size = c(1, 0.4)[shwVals])))
      }
      p[[trial]] <- pTr
      if (output) {
        plot(pTr)
      }
    }
  } else if (plotType == "map") {
    ## Create a data.frame for plotting trials.
    ## Population has a random value but if left out nothing is plotted.
    locs <- setNames(getMeta(x)[c("trLocation", "trLat", "trLong")],
                     c("name", "lat", "long"))
    locs <- unique(locs[!is.na(locs$lat) & !is.na(locs$long), ])
    if (nrow(locs) == 0) {
      stop("At least one trial should have latitude and longitude ",
           "for plotting on map.\n")
    }
    minLatRange <- dotArgs$minLatRange
    minLongRange <- dotArgs$minLongRange
    if (!is.null(minLatRange) && (!is.numeric(minLatRange) ||
                                  length(minLatRange) > 1)) {
      stop("minLatRange should be a single numerical value.\n")
    }
    if (!is.null(minLongRange) && (!is.numeric(minLongRange) ||
                                   length(minLongRange) > 1)) {
      stop("minLongRange should be a single numerical value.\n")
    }
    if (is.null(minLatRange)) {
      minLatRange <- 10
    }
    if (is.null(minLongRange)) {
      minLongRange <- 5
    }
    ## Set minimum range for latitude and longitude.
    latR <- range(locs$lat)
    latR <- latR +
      (diff(latR) < minLatRange) * c(-1, 1) * (minLatRange - diff(latR)) / 2
    longR <- range(locs$long)
    longR <- longR +
      (diff(longR) < minLongRange) * c(-1, 1) * (minLongRange - diff(longR)) / 2
    ## Add 10% to edges of map so locations are not on the absolute edge.
    longR <- longR + c(-0.1, 0.1) * diff(longR)
    latR <- latR + c(-0.1, 0.1) * diff(latR)
    ## Create data usable by ggplot geom_polygon.
    mapDat <- mapData(xLim = longR, yLim = latR)
    p <- ggplot(mapDat, aes_string(x = "long", y = "lat")) +
      geom_polygon(aes_string(group = "group"), fill = "white",
                   color = "black") +
      ## Add a proper map projection.
      coord_map(clip = "on", xlim = longR, ylim = latR) +
      ## Add trial locations.
      geom_point(data = locs) +
      ggrepel::geom_text_repel(aes_string(label = "name"), data = locs,
                               color = "red", size = 3,
                               nudge_x = 0.01 * diff(longR),
                               nudge_y = 0.04 * diff(latR)) +
      theme(plot.title = element_text(hjust = 0.5),
            panel.grid.major = element_blank(),
            panel.grid.minor = element_blank(),
            ## Empty space left represents water areas. Color blue.
            panel.background = element_rect(fill = "steelblue2")) +
      ggtitle("Trial locations")
    if (output) {
      plot(p)
    }
  } else if (plotType == "box") {
    chkChar(traits, null = FALSE)
    groupBy <- dotArgs$groupBy
    if (!is.null(groupBy)) {
      chkChar(groupBy, len = 1, null = FALSE)
    }
    if (!is.null(groupBy) && !all(sapply(X = x, FUN = function(trial) {
      hasName(x = trial, name = groupBy)
    }))) {
      stop("groupBy should be a column in TD.\n")
    }
    colorBy <- dotArgs$colorBy
    if (!is.null(colorBy)) {
      chkChar(colorBy, len = 1, null = FALSE)
    }
    if (!is.null(colorBy) && !all(sapply(X = x, FUN = function(trial) {
      hasName(x = trial, name = colorBy)
    }))) {
      stop("colorBy should be a column in TD.\n")
    }
    orderBy <- dotArgs$orderBy
    if (!is.null(orderBy)) {
      orderBy <- match.arg(orderBy, choices = c("alphabetic", "ascending",
                                                "descending"))
    } else {
      orderBy <- "alphabetic"
    }
    p <- setNames(vector(mode = "list", length = length(traits)), traits)
    for (trait in traits) {
      ## Create a single data.frame from x with only columns trial, trait and
      ## genotype.
      ## Genotype is needed to be able to display hovering info (in GUI).
      ## trials where trait is not measured/available are removed by setting
      ## them to NULL.
      xVar <- if (is.null(groupBy)) "trial" else groupBy
      plotDat <- Reduce(f = rbind, x = lapply(X = x[trials], function(trial) {
        if (!hasName(x = trial, name = trait)) {
          NULL
        } else {
          if (!hasName(x = trial, name = "trial")) {
            trial[["trial"]] <- names(x)
          }
          trial[c(trait, "genotype", xVar, if (!is.null(colorBy)) colorBy)]
        }
      }))
      if (is.null(plotDat)) {
        warning(trait, " isn't a column in any of the trials.\n",
                "Plot skipped.\n", call. = FALSE)
        next
      }
      if (orderBy != "alphabetic") {
        ## Reorder levels in trial so plotting is done according to orderBy.
        ## do.call needed since order doesn't accept a vector as input.
        levNw <- reorder(x = plotDat[[xVar]], X = plotDat[[trait]],
                         FUN = mean, na.rm = TRUE, order = TRUE)
        if (orderBy == "ascending") {
          plotDat[xVar] <- factor(plotDat[[xVar]], levels = levels(levNw))
        } else {
          plotDat[xVar] <- factor(plotDat[[xVar]], levels = rev(levels(levNw)))
        }
      }
      ## Colorby is ignored in plot if it is not a factor.
      if (!is.null(colorBy) && !is.factor(plotDat[colorBy])) {
        plotDat[colorBy] <- factor(plotDat[[colorBy]])
      }
      ## Create boxplot.
      pTr <- ggplot(plotDat, aes_string(x = paste0("`", xVar, "`"),
                                        y = paste0("`", trait, "`"),
                                        fill = if (is.null(colorBy)) 1 else
                                          paste0("`", colorBy, "`"))) +
        theme(axis.text.x = element_text(angle = 90, vjust = 0.5, hjust = 1),
              panel.background = element_blank(),
              panel.grid = element_blank(),
              panel.border = element_rect(color = "black", fill = NA)) +
        labs(x = xVar, y = trait)
        if (is.null(colorBy)) {
          pTr <- pTr + geom_boxplot(na.rm = TRUE, fill = "darkgrey")
        } else {
          pTr <- pTr + geom_boxplot(na.rm = TRUE)
        }
      p[[trait]] <- pTr
      if (output) {
        plot(pTr)
      }
    }
  } else if (plotType == "cor") {
    if (length(trials) == 1) {
      stop("At least two trials requiered for a correlation plot.\n")
    }
    chkChar(traits, null = FALSE)
    p <- setNames(vector(mode = "list", length = length(traits)), traits)
    for (trait in traits) {
      ## Create a single data.frame from x with only columns genotype, trial
      ## and trait.
      ## trials where trait is not measured/available are removed by setting
      ## them to NULL.
      plotDat <- Reduce(f = rbind, x = lapply(X = x, FUN = function(trial) {
        if (!hasName(x = trial, name = trait)) {
          NULL
        } else {
          trial[c("genotype", "trial", trait)]
        }
      }))
      if (is.null(plotDat)) {
        warning(trait, " isn't a column in any of the trials.\n",
                "Plot skipped.\n", call. = FALSE)
        next
      }
      ## Create table with values trait per genotype per trial.
      ## If TD already contains BLUEs/BLUPs taking means doesn't do anything
      ## but it is needed for raw data where there can be replicates.
      plotTab <- tapply(plotDat[[trait]],
                        INDEX = list(plotDat$genotype, plotDat$trial),
                        FUN = mean, na.rm = TRUE)
      ## Create a correlation matrix.
      corMat <- cor(plotTab, use = "pairwise.complete.obs")
      ## Remove rows and columns with only NA.
      corKeep <- sapply(X = 1:ncol(corMat), FUN = function(i) {
        any(!is.na(corMat[, i]))
      })
      corMat <- corMat[corKeep, corKeep, drop = FALSE]
      ## Determine ordering according to clustering of trials.
      corClust <- hclust(as.dist(1 - corMat), method = "ward.D2")
      ordClust <- order.dendrogram(as.dendrogram(corClust))
      ## Reorder according to clusters.
      corMat <- corMat[ordClust, ordClust]
      ## Convert corMat to data.frame to prevent crash when reshaping.
      corMat <- as.data.frame(corMat)
      ## Convert correlation matrix to long format for ggplot.
      meltedCorMat <- reshape(corMat, direction = "long",
                              varying = list(genotype = colnames(corMat)),
                              ids = rownames(corMat), idvar = "trial1",
                              times = colnames(corMat), timevar = "trial2",
                              v.names = "cor")
      ## Reshape converts trial columns to character.
      ## This gives problems with plotting, so reconvert them to factor.
      meltedCorMat[["trial1"]] <- factor(meltedCorMat[["trial1"]],
                                         levels = rownames(corMat))
      meltedCorMat[["trial2"]] <- factor(meltedCorMat[["trial2"]],
                                         levels = rownames(corMat))
      ## Select bottom right triangle for correlations and top for variances.
      meltedCorMatLow <- meltedCorMat[as.numeric(meltedCorMat[["trial1"]]) >
                                        as.numeric(meltedCorMat[["trial2"]]), ]
      ## Create plot.
      pTr <- ggplot(data = meltedCorMatLow, aes_string("trial1", "trial2")) +
        geom_tile(aes_string(fill = "cor"), color = "grey50") +
        ## Create a gradient scale.
        scale_fill_gradient2(low = "blue", high = "red", mid = "white",
                             na.value = "grey", limit = c(-1, 1)) +
        ## Move y-axis to the right for easier reading.
        scale_y_discrete(position = "right") +
        ## Remove grid behind empty bit of triangle.
        theme(panel.background = element_blank(),
              panel.grid = element_blank(),
              axis.ticks = element_blank(),
              axis.text.x = element_text(angle = 45, vjust = 1, size = 6,
                                         hjust = 1),
              axis.text.y = element_text(size = 6),
              ## Center title.
              plot.title = element_text(hjust = 0.5)) +
        ## No axis and legend titles.
        labs(title = paste("Correlations of trials for", trait),
             x = "", y = "", fill = "") +
        ## Equal coordinates to get a square sized plot.
        coord_equal()
      p[[trait]] <- pTr
      if (output) {
        plot(pTr)
      }
    }
  } else if (plotType == "scatter") {
    if (length(trials) == 1) {
      stop("At least two trials requiered for a scatter plot.\n")
    }
    chkChar(traits, null = FALSE)
    colorBy <- dotArgs$colorBy
    if (!is.null(colorBy)) {
      chkChar(colorBy, len = 1, null = FALSE)
    }
    if (!is.null(colorBy) && !all(sapply(X = x, FUN = function(trial) {
      hasName(x = trial, name = colorBy)
    }))) {
      stop("colorBy should be a column in TD.\n")
    }
    trialOrder <- dotArgs$trialOrder
    if (!is.null(trialOrder) &&
        (!all(trialOrder %in% trials) || !all(trials %in% trialOrder))) {
      stop("trials and trialOrder should contain exactly the same trials.\n")
    }
    addCorr <- dotArgs$addCorr
    if (!is.null(addCorr)) {
      addCorr <- match.arg(addCorr, choices = c("tl", "bl", "tr", "br"))
    }
    ## Create list of colors for histograms.
    ## Outside trait loop to assure identical coloring of trials.
    histCols <- setNames(hcl.colors(length(trials),
                                    palette = hcl.pals("qualitative")[1]),
                         paste0("t", trials))
    p <- setNames(vector(mode = "list", length = length(traits)), traits)
    for (trait in traits) {
      ## Create plot title.
      plotTitle <- ifelse(!is.null(dotArgs$title), dotArgs$title,
                          paste("Scatterplots of trials for", trait))
      ## Create a single data.frame from x with only columns genotype, trial
      ## and trait.
      ## trials where trait is not measured/available are removed by setting
      ## them to NULL.
      plotDat <- Reduce(f = rbind, x = lapply(X = x, FUN = function(trial) {
        if (!hasName(x = trial, name = trait) || all(is.na(trial[[trait]]))) {
          NULL
        } else {
          trial[c("genotype", "trial", trait, colorBy)]
        }
      }))
      if (!is.null(plotDat)) {
        plotDat <- droplevels(plotDat)
      }
      if (is.null(plotDat) || nlevels(plotDat[["trial"]]) < 2) {
        warning(trait, " has no valid observations for a least two trials.\n",
                "Plot skipped.\n", call. = FALSE)
        next
      }
      if (!is.null(trialOrder)) {
        ## Reorder trials.
        ## First restrict reordering to trials left after removing NA trials.
        trialOrderTr <- trialOrder[trialOrder %in% levels(plotDat[["trial"]])]
        plotDat[["trial"]] <- factor(plotDat[["trial"]], trialOrderTr)
      }
      ## Create table with values trait per genotype per trial.
      ## If TD already contains BLUEs/BLUPs taking means doesn't do anything
      ## but it is needed for raw data where there can be replicates.
      plotTab <- as.data.frame(tapply(plotDat[[trait]],
                                      INDEX = list(plotDat[["genotype"]],
                                                   plotDat[["trial"]]),
                                      FUN = mean, na.rm = TRUE))
      plotRange <- range(unlist(plotTab), na.rm = TRUE)
      if (!is.null(addCorr)) {
        ## Compute correlations for annotation.
        corMat <- cor(plotTab, use = "pairwise.complete.obs")
        ## Convert corMat to data.frame to prevent crash when reshaping.
        corMat <- as.data.frame(corMat)
        ## Convert correlation matrix to long format for ggplot.
        meltedCorMat <- reshape(corMat, direction = "long",
                                varying = list(genotype = colnames(corMat)),
                                ids = rownames(corMat), idvar = "trial1",
                                times = colnames(corMat), timevar = "trial2",
                                v.names = "cor")
        ## Reshape converts trial columns to character.
        ## This gives problems with plotting, so reconvert them to factor.
        meltedCorMat[["trial1"]] <- factor(meltedCorMat[["trial1"]],
                                           levels = rownames(corMat))
        meltedCorMat[["trial2"]] <- factor(meltedCorMat[["trial2"]],
                                           levels = rownames(corMat))
        ## Set position for annotation.
        ## Using Inf and -Inf for positions independent of scale.
        minPos <- plotRange[1] + 0.03 * plotRange[1] * sign(plotRange[1])
        maxPos <- plotRange[2] - 0.03 * plotRange[2] * sign(plotRange[2])
        meltedCorMat[["x"]] <- ifelse(addCorr %in% c("tl", "bl"), minPos, maxPos)
        meltedCorMat[["y"]] <- ifelse(addCorr %in% c("br", "bl"), minPos, maxPos)
        colnames(meltedCorMat)[colnames(meltedCorMat) == "trial1"] <- "trial.x"
        colnames(meltedCorMat)[colnames(meltedCorMat) == "trial2"] <- "trial.y"
      }
      ## Create plots containing histograms.
      ## Used further on to replace diagonal plot in plot matrix.
      histVars <- paste0("t", colnames(plotTab))
      histPlots <- lapply(X = histVars, FUN = function(trial) {
        colnames(plotTab) <- paste0("t", colnames(plotTab))
        binWidth <- diff(range(plotTab[[trial]], na.rm = TRUE)) / 10
        ggplot(plotTab, aes_string(x = trial,
                                   y = "(..count..)/sum(..count..)")) +
          geom_histogram(na.rm = TRUE, binwidth = binWidth, boundary = 0,
                         fill = histCols[trial], color = "grey50") +
          scale_x_continuous(limits = range(plotTab, na.rm = TRUE)) +
          theme(panel.background = element_blank(),
                panel.grid = element_blank(),
                panel.border = element_rect(color = "black", fill = NA))
      })
      ## Y-axis should be the same for all histograms.
      ## Build histograms and extract axis information.
      yMax <- max(sapply(X = histPlots, FUN = function(hp) {
        max(ggplot_build(hp)$data[[1]][["ymax"]])
      }))
      ## Add scaling for y-axis to histograms
      ## Convert to grobs for easier use later on.
      histGrobs <- lapply(X = histPlots, FUN = function(hp) {
        hp <- hp + scale_y_continuous(expand = c(0, 0, 0, 0.05),
                                      labels = function(x) {
                                        paste0(100 * x, "%")
                                      }, limits = c(0, yMax))
        ggplotGrob(hp)
      })
      ## Reshape to get data in format suitable for ggplot.
      plotTab <- reshape(plotTab, direction = "long",
                         varying = colnames(plotTab),
                         timevar = "trial", times = colnames(plotTab),
                         idvar = "genotype", ids = rownames(plotTab),
                         v.names = trait)
      if (!is.null(colorBy)) {
        plotTab <- merge(plotTab, unique(plotDat[c("genotype", colorBy)]))
      }
      ## Merge to itself to create a full data set.
      plotTab <- merge(plotTab, plotTab, by = c("genotype", colorBy))
      ## Create a facet plot containing only scatterplots.
      scatterBase <- ggplot(data = plotTab,
                            aes_string(x = paste0(trait, ".x"),
                                       y = paste0(trait, ".y"),
                                       color = if (is.null(colorBy)) NULL else
                                         paste0("`", colorBy, "`"))) +
        scale_x_continuous(breaks = scales::breaks_extended(n = 3)) +
        scale_y_continuous(breaks = scales::breaks_extended(n = 3)) +
        facet_grid(facets = c("trial.y", "trial.x")) +
        labs(title = plotTitle, x = "", y = "") +
        theme(plot.title = element_text(hjust = 0.5),
              legend.position = c(1, 1),
              legend.justification = c(1.5, 1.5),
              aspect.ratio = 1,
              panel.background = element_rect(fill = "white"),
              panel.grid = element_blank(),
              panel.border = element_rect(color = "black", fill = NA))
      if (is.null(colorBy)) {
        scatterBase <- scatterBase +
          geom_point(na.rm = TRUE, color = "darkgrey", shape = 1)
      } else {
        scatterBase <- scatterBase + geom_point(na.rm = TRUE, shape = 1)
      }
      if (!is.null(addCorr)) {
        ## Add correlation annotated in the corner of the plot.
        scatterBase <- scatterBase +
          geom_text(data = meltedCorMat,
                    aes_string(x = "x", y = "y",
                               label = "paste('rho ==', round(cor, 2))"),
                    color = "red", hjust = "inward", vjust = "inward",
                    parse = TRUE, inherit.aes = FALSE)
      }
      ## Convert to grobs to enable modifying.
      scatterGrob <- ggplotGrob(scatterBase)
      ## Get grobs containing plot panels.
      panels <- scatterGrob$layout$name[grepl(pattern = "panel",
                                              x = scatterGrob$layout$name)]
      splitPanels <- strsplit(x = panels, split = "-")
      ## Upper right panels need to be set to zeroGrob to make them empty.
      nullPanels <- panels[sapply(X = splitPanels, FUN = function(pan) {
        as.numeric(pan[2]) < as.numeric(pan[3])
      })]
      for (np in nullPanels) {
        scatterGrob$grobs[[which(scatterGrob$layout$name == np)]] <- zeroGrob()
      }
      ## Set diagonal panels to histograms calculated before.
      histPanels <- panels[sapply(X = splitPanels, FUN = function(pan) {
        as.numeric(pan[2]) == as.numeric(pan[3])
      })]
      for (i in seq_along(histPanels)) {
        hg <- histGrobs[[i]]
        ## Replace grob by panel grob from histogram.
        scatterGrob$grobs[[which(scatterGrob$layout$name == histPanels[i])]] <-
          hg$grobs[[which(hg$layout$name == "panel")]]
      }
      ## Replace top left axis in the matrix by y axis from the first histogram.
      scatterGrob$grobs[[which(scatterGrob$layout$name == "axis-l-1")]] <-
        histGrobs[[1]]$grobs[[which(histGrobs[[1]]$layout$name == "axis-l")]]
      p[[trait]] <- scatterGrob
      if (output) {
        grid::grid.newpage()
        grid::grid.draw(scatterGrob)
      }
    }
  }
  invisible(p)
}

#' Get and set metadata for TD objects
#'
#' Functions for extracting and adding metadata for objects of class TD.\cr
#' \code{getMeta} extracts a data.frame with location, date, design, latitude,
#' longitude, plotWidth and plotLength for all trials in TD.\cr\cr
#' \code{setMeta} adds metadata from a data.frame to an object of class TD. See
#' details for the specifications of the data.frame.\cr\cr
#' The most common use case is extracting metadata from a TD object, modifying
#' the content and then adding it back to the TD object.\cr
#' Information in the metadata of a TD object is used in plotting functions
#' (e.g. latitude and longitude for a map plot) and when fitting models on the
#' data (the trial design).
#'
#' When setting metadata, metadata has to be a data.frame with rownames
#' corresponding to the trials in \code{TD}. The data.frame should contain one
#' or more of the following columns:
#' \describe{
#' \item{trLocation}{The location of the trial. Used as default name when
#' creating plots and summaries.}
#' \item{trDate}{The date of the trial.}
#' \item{trDesign}{The design of the trial. One of "none" (no (known) design),
#' "ibd" (incomplete-block design), "res.ibd" (resolvable incomplete-block
#' design), "rcbd" (randomized complete block design),
#' "rowcol" (row-column design) or "res.rowcol" (resolvable row-column design).
#' Used when fitting models.}
#' \item{trLat}{The latitude of the trial on a scale of -90 to 90. Used when
#' plotting the trials on a map.}
#' \item{trLong}{The longitude of the trial on a scale of -180 to 180. Used
#' when plotting the trials on a map.}
#' \item{trPlWidth}{The width of the plot. Used in combination with trPlLength
#' to determine the size of the plots in a layout plot of a trial.}
#' \item{trPlLength}{The length of the plot. Used in combination with
#' trPlWidth to determine the size of the plots in a layout plot of a trial.}
#' }
#' The values of the metadata of TD will be set to the values in the
#' corresponding column in \code{meta}. Existing values will be overwritten,
#' but \code{NA} will be ignored so setting a value to \code{NA} won't result
#' in accidentally removing it.
#'
#' @param TD An object of class TD.
#'
#' @family functions for TD objects
#'
#' @examples
#' data("wheatChl")
#'
#' ## Create a TD object.
#' wheatTD <- createTD(data = wheatChl, genotype = "trt", repId = "rep",
#'                     subBlock = "bl", rowCoord = "row", colCoord = "col")
#'
#' ## Get meta data from wheatTD
#' (wheatMeta <- getMeta(wheatTD))
#'
#' ## Add location names and latitude/longitude to meta data.
#' wheatMeta$trLocation <- c("Cauquenes", rep("Santa Rosa", times = 4))
#' wheatMeta$trLat <- c(-35.58, rep(-36.32, times = 4))
#' wheatMeta$trLong <- c(-72.17, rep(-71.55, times = 4))
#'
#' ## Add back meta data to wheatTD.
#' wheatTD <- setMeta(wheatTD, wheatMeta)
#'
#' @export
getMeta <- function(TD) {
  if (missing(TD) || !inherits(TD, "TD")) {
    stop("TD should be an object of class TD.\n")
  }
  metaVars <- c("trLocation", "trDate", "trDesign", "trLat", "trLong",
                "trPlWidth", "trPlLength")
  meta <- as.data.frame(matrix(nrow = length(TD), ncol = length(metaVars),
                               dimnames = list(names(TD), metaVars)))
  for (mv in metaVars) {
    meta[mv] <- sapply(X = TD, FUN = function(tr) {
      mvTr <- attr(tr, which = mv)
      ## Replace NULL by NA to ensure correct output format for inserting in df.
      if (!is.null(mvTr)) {
        return(mvTr)
      } else {
        return(NA)
      }
    })
  }
  class(meta$trDate) <- "Date"
  return(meta)
}

#' @param TD An object of class TD.
#' @param meta A data.frame containing metadata.
#'
#' @rdname getMeta
#' @export
setMeta <- function(TD,
                    meta) {
  if (missing(TD) || !inherits(TD, "TD")) {
    stop("TD should be an object of class TD.\n")
  }
  if (missing(meta) || !inherits(meta, "data.frame")) {
    stop("meta should be a data.frame.\n")
  }
  naTr <- rownames(meta)[!rownames(meta) %in% names(TD)]
  if (length(naTr) > 0) {
    warning("The following trials in meta are not in TD: ",
            paste(naTr, collapse = ", "), ".\n", call. = FALSE)
  }
  metaVars <- c("trLocation", "trDate", "trDesign", "trLat", "trLong",
                "trPlWidth", "trPlLength")
  ## Set metadata for trials in meta that are also in TD.
  for (tr in rownames(meta)[rownames(meta) %in% names(TD)]) {
    for (mv in metaVars) {
      mvTr <- meta[tr, mv]
      if (!is.na(mvTr)) {
        chk <- try(do.call(what = checkTDMeta, args = setNames(list(mvTr), mv)),
                   silent = TRUE)
        if (inherits(chk, "try-error")) {
          ## Get message from check function but remove first 8 chars to
          ## prevent having an error text with 3x error in it.
          stop("\nError for ", tr, ":\n", substring(text = chk, first = 9))
        }
        attr(TD[[tr]], which = mv) <- mvTr
      }
    }
  }
  return(TD)
}

#' Function for extracting for objects of class TD that keeps class.
#'
#' @noRd
#' @keywords internal
`[.TD` <- function(x, i, ...) {
  r <- NextMethod("[")
  attr(r, "class") <- attr(x, "class")
  return(r)
}

#' Helper function for checking metadata structure for TD objects.
#'
#' @noRd
#' @keywords internal
checkTDMeta <- function(trLocation = NULL,
                        trDate = NULL,
                        trDesign = NULL,
                        trLat = NULL,
                        trLong = NULL,
                        trPlWidth = NULL,
                        trPlLength = NULL) {
  if (!is.null(trDesign)) {
    trDesign <- match.arg(trDesign, choices = c("none", "ibd", "res.ibd",
                                                "rcbd", "rowcol", "res.rowcol"),
                          several.ok = TRUE)
  }
  if (!is.null(trLat) && (!is.numeric(trLat) || any(abs(trLat) > 90))) {
    stop("trLat should be a numerical vector between -90 and 90.\n",
         call. = FALSE)
  }
  if (!is.null(trLong) && (!is.numeric(trLong) || any(abs(trLong) > 180))) {
    stop("trLat should be a numerical vector between -180 and 180.\n",
         call. = FALSE)
  }
  if (!is.null(trLat) && !is.null(trLong)) {
    locLen <- max(length(trLat), length(trLong))
    ## Check that coordinates point to a proper location so plotting can be done.
    loc <- maps::map.where(x = rep(x = trLong, length.out = locLen),
                           y = rep(x = trLat, length.out = locLen))
    if (length(loc) > 0 && anyNA(loc)) {
      warning("Values for trLat and trLong should all match a known land ",
              "location.\n", call. = FALSE)
    }
  }
  if (!is.null(trPlWidth) && (!is.numeric(trPlWidth) || any(trPlWidth < 0))) {
    stop("trPlWidth should be a positive numerical vector.\n", call. = FALSE)
  }
  if (!is.null(trPlLength) && (!is.numeric(trPlLength) || any(trPlLength < 0))) {
    stop("trPlLength should be a positive numerical vector.\n", call. = FALSE)
  }
}
