% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/edge_nn.R
\name{edge_nn}
\alias{edge_nn}
\title{Nearest neighbour based edge lists}
\usage{
edge_nn(DT = NULL, id = NULL, coords = NULL, timegroup = NULL,
  splitBy = NULL, threshold = NULL)
}
\arguments{
\item{DT}{input data.table}

\item{id}{Character string of ID column name}

\item{coords}{Character vector of X coordinate and Y coordinate column names}

\item{timegroup}{(optional) timegroup field in the DT upon which the grouping will be calculated}

\item{splitBy}{(optional) character string or vector of grouping column name(s) upon which the grouping will be calculated}

\item{threshold}{(optional) spatial distance threshold to set maximum distance between an individual and their neighbour.}
}
\value{
\code{edge_nn} returns a \code{data.table}  with three columns: timegroup, ID and NN.

The ID and NN columns represent the edges defined by the nearest neighbours (and temporal thresholds with \code{group_times}).

If an individual was alone in a timegroup or splitBy, or did not have any neighbours within the threshold distance, they are assigned NA for nearest neighbour.
}
\description{
\code{edge_nn} returns edge lists defined by the nearest neighbour. The function accepts a \code{data.table} with relocation data, individual identifiers and a threshold argument. The threshold argument is used to specify the criteria for distance between points which defines a group. Relocation data should be in two columns representing the X and Y coordinates.
}
\details{
The \code{DT} must be a \code{data.table}. If your data is a \code{data.frame}, you can convert it by reference using \code{\link[data.table:setDT]{data.table::setDT}}.

The \code{id}, \code{coords} (and optional \code{timegroup} and \code{splitBy}) arguments expect the names of a column in \code{DT} which correspond to the individual identifier, X and Y coordinates, timegroup (generated by \code{group_times}) and additional grouping columns.

The \code{threshold} must be provided in the units of the coordinates. The \code{threshold} must be larger than 0. The coordinates must be planar coordinates (e.g.: UTM). In the case of UTM, a \code{threshold} = 50 would indicate a 50m distance threshold.

The \code{timegroup} argument is optional, but recommended to pair with \code{\link{group_times}}. The intended framework is to group rows temporally with \code{\link{group_times}} then spatially with \code{edge_nn} (or grouping functions).

The \code{splitBy} argument offers further control over grouping. If within your \code{DT}, you have multiple populations, subgroups or other distinct parts, you can provide the name of the column which identifies them to \code{splitBy}. \code{edge_nn} will only consider rows within each \code{splitBy} subgroup.
}
\examples{
# Load data.table
library(data.table)

# Read example data
DT <- fread(system.file("extdata", "DT.csv", package = "spatsoc"))

# Cast the character column to POSIXct
DT[, datetime := as.POSIXct(datetime, tz = 'UTC')]

# Temporal grouping
group_times(DT, datetime = 'datetime', threshold = '20 minutes')

# Edge list generation
edge_nn(DT, id = 'ID', coords = c('X', 'Y'),
        timegroup = 'timegroup')

# Edge list generation using maximum distance threshold
edge_nn(DT, id = 'ID', coords = c('X', 'Y'),
        timegroup = 'timegroup', threshold = 100)
}
\seealso{
Other Edge-list generation: \code{\link{edge_dist}}
}
\concept{Edge-list generation}
