#' Analyses Netlogo simulations generated for a latin-hypercube based sensitivity analysis
#'
#' Takes each parameter value set generated by the hypercube in turn, and
#' analyses the Netlogo simulation results.  For each parameter set, there
#' will be n simulation results. This method goes through these results,
#' producing a file containing the median of each output measure for each
#' of the n runs. Thus, if a Netlogo simulation was replicated 10 times,
#' the median file will contain 10 medians for each simulation output measure.
#' Once this has been created, the user should run
#' \code{lhc_generateLHCSummary}, \code{lhc_generatePRCoEffs}, and
#' \code{lhc_graphMeasuresForParameterChange} as per analysing any data
#' in spartan that was not generated by Netlogo
#'
#' @param FILEPATH Directory where either the simulation runs can be found
#' @param LHCSAMPLE_RESULTFILENAME Name of the result file generated by Netlogo, for a LHC parameter sample.
#' @param SPARTAN_PARAMETER_FILE Location of the file output by the latin-hypercube sampling method, containing the parameters on which this experiment was performed
#' @param NUMSAMPLES The number of parameter subsets that were generated in the LHC design
#' @param MEASURES Array containing the names of the Netlogo output measures which are used to analyse the simulation.
#' @param LHC_ALL_SIM_RESULTS_FILE Name of the LHC Summary file to be generated. Contains each parameter set alongside the result gained when the simulation was run under that criteria.
#' @param TIMESTEP The timestep of the Netlogo simulation being analysed
#'
#' @export
lhc_process_netlogo_result <- function(FILEPATH, LHCSAMPLE_RESULTFILENAME,
                                       SPARTAN_PARAMETER_FILE, NUMSAMPLES,
                                       MEASURES, LHC_ALL_SIM_RESULTS_FILE,
                                       TIMESTEP) {

  # READ IN THE SPARTAN PARAMETER FILE
  LHCTABLE <- read.csv(paste(FILEPATH, "/", SPARTAN_PARAMETER_FILE, sep = ""),
                       header = TRUE, check.names = FALSE)

  # NOW ALL THE MEDIANS ARE HELD TOGETHER, ACCOMPANIED BY THEIR SIMULATION
  # PARAMETERS BEING ANALYSED
  ALL_SIM_MEDIAN_RESULTS <- NULL

  for (SAMPLE in 1:NUMSAMPLES) {
    if (file.exists(paste(FILEPATH, "/", SAMPLE, "/", LHCSAMPLE_RESULTFILENAME,
                          SAMPLE, ".csv", sep = ""))) {
      print(paste("Processing LHC Results for Sample: ", SAMPLE, sep = ""))

      # READ IN THE RESULT FILE
      # SKIP THE FIRST 6 LINES AS NONE OF THIS INFORMATION IS REQUIRED
      NL_RESULT <- read.csv(paste(FILEPATH, "/", SAMPLE, "/",
                                  LHCSAMPLE_RESULTFILENAME, SAMPLE,
                                  ".csv", sep = ""), sep = ",",
                            skip = 6, check.names = FALSE)

      # ORDER IT BY RUN FOR EFFICIENCY LATER
      NL_RESULT_ORDERED <- NL_RESULT[order(NL_RESULT[, 1]), ]

      TIMESTEP_RESULTS <- subset(NL_RESULT_ORDERED,
                                 NL_RESULT_ORDERED["[step]"] == TIMESTEP)

      # NOW TO CREATE THE RESULTS FOR THIS SAMPLE SET
      # NETLOGO DOES GIVE THE OPTION OF RUNNING REPLICATES OF THE SAME
      # EXPERIMENT. SO THERE MAY BE A FEW ROWS HERE. THE SUMMARY METHOD
      # WILL TAKE CARE OF THESE REPLICATES
      # FIRST LETS SET UP THE NUMBER OF PARAMETER ROWS
      param_set <- LHCTABLE[SAMPLE, ]

      # Make duplicates of the parameters to match the number of replicate runs
      PARAMS <- NULL
      for (paramval in 1:ncol(param_set)) {
        PARAMS <- cbind(PARAMS, param_set[[paramval]])
      }

      DUP_PARAMS <- NULL
      for (r in 1:nrow(TIMESTEP_RESULTS) - 1) {
        DUP_PARAMS <- rbind(DUP_PARAMS, PARAMS)
      }

      # NOW WE CAN ADD THE RESULTS FOR EACH NETLOGO RUN
      for (RESPONSE in 1:length(MEASURES)) {
        DUP_PARAMS <- cbind(DUP_PARAMS,
                            TIMESTEP_RESULTS[MEASURES[RESPONSE]][, 1])
      }

      ALL_SIM_MEDIAN_RESULTS <- rbind(ALL_SIM_MEDIAN_RESULTS, DUP_PARAMS)

    } else  {
      print(paste("ERROR: Results for Sample ", SAMPLE, " not found",
                  sep = ""))

    }
  }
  # NOW OUTPUT ALL THE MEDIAN RESULTS TO THE SPECIFIED FILEPATH
  colnames(ALL_SIM_MEDIAN_RESULTS) <- cbind(t(names(LHCTABLE)),
                                            t(MEASURES))

  # OUTPUT IF THE RESULTS ARE NOT BLANK
  if (!is.null(ALL_SIM_MEDIAN_RESULTS)) {
    RESULTSFILE <- paste(FILEPATH, "/", LHC_ALL_SIM_RESULTS_FILE,
                         sep = "")
    print(paste("Writing Median Results to CSV File: ", RESULTSFILE,
                sep = ""))
    write.csv(ALL_SIM_MEDIAN_RESULTS, RESULTSFILE, quote = FALSE,
              row.names = FALSE)
  }
}
