% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/icc.R
\name{icc}
\alias{icc}
\title{Intraclass-Correlation Coefficient}
\usage{
icc(x, ...)
}
\arguments{
\item{x}{Fitted mixed effects model (of class \code{merMod}, \code{glmmTMB},
\code{stanreg} or \code{brmsfit}).}

\item{...}{More fitted model objects, to compute multiple intraclass-correlation
coefficients at once.}
}
\value{
A numeric vector with all random intercept intraclass-correlation-coefficients,
          or a list of numeric vectors, when more than one model were used
          as arguments. Furthermore, between- and within-group variances as well
          as random-slope variance are returned as attributes.
}
\description{
This function calculates the intraclass-correlation
               (icc) - sometimes also called \emph{variance partition coefficient}
               (vpc) - for random intercepts of mixed effects models.
               Currently, \code{\link[lme4]{merMod}}, \code{\link[glmmTMB]{glmmTMB}}
               \code{stanreg} and \code{\link[brms]{brmsfit}}
               objects are supported.
}
\details{
The ICC is calculated by dividing the between-group-variance (random
         intercept variance) by the total variance (i.e. sum of between-group-variance
         and within-group (residual) variance). \cr \cr
      The calculation of the ICC for generalized linear mixed models with binary outcome is based on
      \cite{Wu et al. (2012)}. For Poisson multilevel models, please refere to \cite{Stryhn et al. (2006)}.
      \cite{Aly et al. (2014)} describe computation of ICC for negative binomial models.
      \cr \cr
      There is a \code{print}-method that prints the variance parameters using
      the \code{comp}-argument set to \code{"var"}: \code{print(x, comp = "var")}
      (see 'Examples'). The \code{\link{re_var}}-function is a convenient wrapper.
      \cr \cr
      The random effect variances indicate the between- and within-group
        variances as well as random-slope variance and random-slope-intercept
        correlation. The components are denoted as following:
        \itemize{
         \item Within-group (residual) variance: sigma_2
         \item Between-group-variance: tau.00 (variation between individual intercepts and average intercept)
         \item Random-slope-variance: tau.11 (variation between individual slopes and average slope)
         \item Random-Intercept-Slope-covariance: tau.01
         \item Random-Intercept-Slope-correlation: rho.01
        }
}
\note{
Some notes on why the ICC is useful, based on \cite{Grace-Martin}:
      \itemize{
       \item It can help you determine whether or not a linear mixed model is even necessary. If you find that the correlation is zero, that means the observations within clusters are no more similar than observations from different clusters. Go ahead and use a simpler analysis technique.
       \item It can be theoretically meaningful to understand how much of the overall variation in the response is explained simply by clustering. For example, in a repeated measures psychological study you can tell to what extent mood is a trait (varies among people, but not within a person on different occasions) or state (varies little on average among people, but varies a lot across occasions).
       \item It can also be meaningful to see how the ICC (as well as the between and within cluster variances) changes as variable are added to the model.
      }
      In short, the ICC can be interpreted as \dQuote{the proportion of the variance
      explained by the grouping structure in the population} \cite{(Hox 2002: 15)}.
      \cr \cr
      Usually, the ICC is calculated for the null model ("unconditional model").
      However, according to \cite{Raudenbush and Bryk (2002)} or
      \cite{Rabe-Hesketh and Skrondal (2012)} it is also feasible to compute the ICC
      for full models with covariates ("conditional models") and compare how
      much a level-2 variable explains the portion of variation in the grouping
      structure (random intercept).
      \cr \cr
      \strong{Caution:} For three-level-models, depending on the nested structure
      of the model, the ICC only reports the proportion of variance explained
      for each grouping level. However, the proportion of variance for specific
      levels related to each other (e.g., similarity of level-1-units within
      level-2-units or level-2-units within level-3-units) must be computed
      manually. Use \code{\link{get_re_var}} to get the between-group-variances
      and residual variance of the model, and calculate the ICC for the various level
      correlations.
      \cr \cr
      For example, for the ICC between level 1 and 2: \cr
      \code{sum(get_re_var(fit)) / (sum(get_re_var(fit)) + get_re_var(fit, "sigma_2"))}
      \cr \cr
      or for the ICC between level 2 and 3: \cr
      \code{get_re_var(fit)[2] / sum(get_re_var(fit))}
}
\examples{
library(lme4)
fit0 <- lmer(Reaction ~ 1 + (1 | Subject), sleepstudy)
icc(fit0)

fit1 <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy)
icc(fit1)

sleepstudy$mygrp <- sample(1:45, size = 180, replace = TRUE)
fit2 <- lmer(Reaction ~ Days + (1 | mygrp) + (Days | Subject), sleepstudy)
icc(fit2)

# return icc for all models at once
icc(fit0, fit1, fit2)

icc1 <- icc(fit1)
icc2 <- icc(fit2)

print(icc1, comp = "var")
print(icc2, comp = "var")


}
\references{
\itemize{
              \item Aguinis H, Gottfredson RK, Culpepper SA. 2013. Best-Practice Recommendations for Estimating Cross-Level Interaction Effects Using Multilevel Modeling. Journal of Management 39(6): 1490–1528 (\doi{10.1177/0149206313478188})
              \item Aly SS, Zhao J, Li B, Jiang J. 2014. Reliability of environmental sampling culture results using the negative binomial intraclass correlation coefficient. Springerplus 14(3) (\doi{10.1186/2193-1801-3-40})
              \item Grace-Martion K. The Intraclass Correlation Coefficient in Mixed Models, \href{http://www.theanalysisfactor.com/the-intraclass-correlation-coefficient-in-mixed-models/}{web}
              \item Hox J. 2002. Multilevel analysis: techniques and applications. Mahwah, NJ: Erlbaum
              \item Rabe-Hesketh S, Skrondal A. 2012. Multilevel and longitudinal modeling using Stata. 3rd ed. College Station, Tex: Stata Press Publication
              \item Raudenbush SW, Bryk AS. 2002. Hierarchical linear models: applications and data analysis methods. 2nd ed. Thousand Oaks: Sage Publications
              \item Stryhn H, Sanchez J, Morley P, Booker C, Dohoo IR. 2006. Interpretation of variance parameters in multilevel Poisson regression models. Proceedings of the 11th International Symposium on Veterinary Epidemiology and Economics, 2006 Available at \url{http://www.sciquest.org.nz/node/64294}
              \item Wu S, Crespi CM, Wong WK. 2012. Comparison of methods for estimating the intraclass correlation coefficient for binary responses in cancer prevention cluster randomized trials. Contempory Clinical Trials 33: 869-880 (\doi{10.1016/j.cct.2012.05.004})
            }
            Further helpful online-ressources:
            \itemize{
              \item \href{http://stats.stackexchange.com/questions/18088/intraclass-correlation-icc-for-an-interaction/28100#28100}{CrossValidated (2012) \emph{Intraclass correlation (ICC) for an interaction?}}
              \item \href{http://stats.stackexchange.com/questions/113577/interpreting-the-random-effect-in-a-mixed-effect-model/113825#113825}{CrossValidated (2014) \emph{Interpreting the random effect in a mixed-effect model}}
              \item \href{http://stats.stackexchange.com/questions/67247/how-to-partition-the-variance-explained-at-group-level-and-individual-level/67356#67356}{CrossValidated (2014) \emph{how to partition the variance explained at group level and individual level}}
            }
}
\seealso{
\code{\link{re_var}}
}
