% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sits_accuracy.R
\name{sits_accuracy}
\alias{sits_accuracy}
\alias{sits_accuracy.sits}
\alias{sits_accuracy.class_cube}
\title{Assess classification accuracy (area-weighted method)}
\usage{
sits_accuracy(data, ...)

\method{sits_accuracy}{sits}(data, ...)

\method{sits_accuracy}{class_cube}(data, validation = NULL, ..., validation_csv = NULL)
}
\arguments{
\item{data}{Either a data cube with classified images or
a set of time series}

\item{\dots}{Specific parameters}

\item{validation}{Samples for validation (see below)
Only required when data is a data cube.}

\item{validation_csv}{CSV file with samples (deprecated)}
}
\value{
A list of lists: The error_matrix, the class_areas, the unbiased
estimated areas, the standard error areas, confidence interval 95% areas,
and the accuracy (user, producer, and overall), or NULL if the data is empty.
A confusion matrix assessment produced by the caret package.
}
\description{
This function calculates the accuracy of the classification
result. For a set of time series, it creates a confusion matrix and then
calculates the resulting statistics using the R package "caret". The time
series needs to be classified using \code{\link[sits]{sits_classify}}.

Classified images are generated using \code{\link[sits]{sits_classify}}
followed by \code{\link[sits]{sits_label_classification}}.
For a classified image, the function uses an area-weighted technique
proposed by Olofsson et al. according to [1-3] to produce more reliable
accuracy estimates at 95% confidence level.

In both cases, it provides an accuracy assessment of the classified,
including Overall Accuracy, Kappa, User's Accuracy, Producer's Accuracy
and error matrix (confusion matrix)
}
\note{
The validation data needs to contain the following columns: "latitude",
 "longitude", "start_date", "end_date", and "label". It can be either a
 path to a CSV file, a sits tibble or a data frame.
}
\examples{
if (sits_run_examples()) {
    # show accuracy for a set of samples
    train_data <- sits_sample(samples_modis_ndvi, n = 200)
    test_data <- sits_sample(samples_modis_ndvi, n = 200)
    rfor_model <- sits_train(train_data, sits_rfor())
    points_class <- sits_classify(
        data = test_data, ml_model = rfor_model
    )
    acc <- sits_accuracy(points_class)

    # show accuracy for a data cube classification
    # create a random forest model
    rfor_model <- sits_train(samples_modis_ndvi, sits_rfor())
    # create a data cube from local files
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6",
        data_dir = data_dir
    )
    # classify a data cube
    probs_cube <- sits_classify(
        data = cube, ml_model = rfor_model, output_dir = tempdir()
    )
    # label the probability cube
    label_cube <- sits_label_classification(
        probs_cube, output_dir = tempdir()
    )
    # obtain the ground truth for accuracy assessment
    ground_truth <- system.file("extdata/samples/samples_sinop_crop.csv",
        package = "sits"
    )
    # make accuracy assessment
    as <- sits_accuracy(label_cube, validation = ground_truth)
}
}
\references{
[1] Olofsson, P., Foody, G.M., Stehman, S.V., Woodcock, C.E. (2013).
Making better use of accuracy data in land change studies: Estimating
accuracy and area and quantifying uncertainty using stratified estimation.
Remote Sensing of Environment, 129, pp.122-131.

[2] Olofsson, P., Foody G.M., Herold M., Stehman, S.V.,
Woodcock, C.E., Wulder, M.A. (2014)
Good practices for estimating area and assessing accuracy of land change.
Remote Sensing of Environment, 148, pp. 42-57.

[3] FAO, Map Accuracy Assessment and Area Estimation: A Practical Guide.
National forest monitoring assessment working paper No.46/E, 2016.
}
\author{
Rolf Simoes, \email{rolf.simoes@inpe.br}

Alber Sanchez, \email{alber.ipia@inpe.br}
}
