% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clade_phyglm.R
\name{clade_phyglm}
\alias{clade_phyglm}
\title{Influential clade detection - Phylogenetic Logistic Regression}
\usage{
clade_phyglm(formula, data, phy, btol = 50, track = TRUE, clade.col,
  n.species = 5, times = 100, ...)
}
\arguments{
\item{formula}{The model formula}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{btol}{Bound on searching space. For details see \code{phyloglm}}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{clade.col}{The name of a column in the provided data frame with clades 
specification (a character vector with clade names).}

\item{n.species}{Minimum number of species in the clade in order to include
this clade in the leave-one-out deletion analyis. Default is \code{5}.}

\item{times}{Number of simulations for the randomization test.}

\item{...}{Further arguments to be passed to \code{phyloglm}}
}
\value{
The function \code{clade_phyglm} returns a list with the following
components:

\code{formula}: The formula

\code{full.model.estimates}: Coefficients, aic and the optimised
value of the phylogenetic parameter (e.g. \code{alpha}) for the full model
without deleted species.

\code{clade.model.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade. Columns report the calculated
regression intercept (\code{intercept}), difference between simulation
intercept and full model intercept (\code{DFintercept}), the percentage of change
in intercept compared to the full model (\code{intercept.perc}) and intercept
p-value (\code{pval.intercept}). All these parameters are also reported for the regression
slope (\code{DFslope} etc.). Additionally, model aic value (\code{AIC}) and
the optimised value (\code{optpar}) of the phylogenetic parameter are
reported.

\code{data}: Original full dataset.

\code{errors}: Clades where deletion resulted in errors.
}
\description{
Estimate the impact on model estimates of phylogenetic logistic regression after 
removing clades from the analysis.
}
\details{
Currently only logistic regression using the "logistic_MPLE"-method from
\code{phyloglm} is implemented.

\code{clade_phyglm} detects influential clades based on
difference in intercept and/or slope when removing a given clade compared
to the full model including all species.

#' Additionally, to account for the influence of the number of species on each 
clade (clade sample size), this function also estimate a null distribution of slopes
expected for the number of species in a given clade. This is done by fitting
 models without the same number of species in the given clade. 
 The number of simulations to be performed is set by 'times'. To test if the 
 clade influence differs from the null expectation, a randomization test can
 be performed using 'summary(x)'. 

Currently, this function can only implement simple linear models (i.e. 
\eqn{y = a + bx}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontrun{
# Simulate Data:
set.seed(6987)
phy = rtree(150)
x = rTrait(n=1,phy=phy)
X = cbind(rep(1,150),x)
y = rbinTrait(n=1,phy=phy, beta=c(-1,0.5), alpha=.7 ,X=X)
cla <- rep(c("A","B","C","D","E"), each = 30)
dat = data.frame(y, x, cla)
# Run sensitivity analysis:
clade <- clade_phyglm(y ~ x, phy = phy, data = dat, times = 30, clade.col = "cla")
# To check summary results and most influential clades:
summary(clade)
# Visual diagnostics for clade removal:
sensi_plot(clade)
# Specify which clade removal to plot:
sensi_plot(clade, "B")
sensi_plot(clade, "C")
sensi_plot(clade, "D") #The clade with the largest effect on slope and intercept
}
}
\author{
Gustavo Paterno & Gijsbert Werner
}
\references{
Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phyloglm}}, \code{\link[sensiPhy]{clade_phylm}},
 \code{\link{influ_phyglm}}, \code{\link{sensi_plot}}
}

