#' runArgusBatchFile
#' @param obj \code{NULL} or an object of class \code{\link{sdcProblem-class}} that was used to generate the batchfile for argus. If not \code{NULL},
#' this object is used to create correct variable names. Else, only the output from tau-Argus is read and returned as a \code{data.table}. In this case
#' it is possible to run tau-Argus on arbitrarily created batch-files.
#' @param batchF a filepath to an batch-input file created by e.g. \code{\link{createArgusInput}}.
#' @param exe (character) file-path to tau-argus executable
#'
#' @return a \code{data.table} containing the protected table or an error in case the batch-file was not solved correctly
#' @export
runArgusBatchFile <- function(obj=NULL, batchF, exe="C:\\Tau\\TauArgus.exe") {
  ## checks
  if(!file.exists(batchF)) {
    stop("argument 'batchF' does not exist!\n")
  }
  ## checks
  if (!is.null(obj)) {
    if(!class(obj)=="sdcProblem") {
      stop("argument 'obj' must be of class 'sdcProblem'!\n")
    }
    ## check if it was written by sdcTable
    if (readLines(batchF)[1]!="//This batch file was generated by sdcTable") {
      stop("the provided batch-file was not created from sdcTable!\n")
    }
    logf <- "arguslog.log"
  } else {
    # restriction in <WRITEABLE> section in case of "free" batch input, otherwise reading outputs won't work
    inpO <- readLines(batchF)
    inp <- inpO[grep("WRITETABLE", inpO)]
    if (length("inp")!=1) {
      stop("the provided batch-input file must contain a single <WRITETABLE> section!\n")
    }
    if (substr(inp, 1, 27)!="<WRITETABLE> (1, 3, AS+FL+,") {
      stop("The <WRITETABLE> directive must start with '<WRITETABLE> (1, 3, AS+FL+,'!\n")
    }
    logf <- infoFromBatch(batchF, typ="LOGBOOK")
  }

  if (!file.exists(exe)) {
    stop("the supplied patch to the tau-argus executable does not exist!\n")
  }

  basedir_tauexe <- dirname(exe)
  logf <- infoFromBatch(batchF, typ="LOGBOOK")
  if (dirname(logf)==".") {
    logf <- file.path(basedir_tauexe, logf)
  }
  outtab <- infoFromBatch(batchF, typ="WRITETABLE")
  # relativPath
  if (dirname(outtab)==".") {
    outtab <- file.path(basedir_tauexe, outtab)
  }

  ## run and check for success
  cmd <- paste(exe, batchF)
  res <- suppressWarnings(system(cmd, intern=TRUE, ignore.stdout=TRUE, ignore.stderr=FALSE))
  s <- attributes(res)$status
  if (!is.null(s) && s!=0) {
    stop(paste0("An error has occured. Please have a look at the logfile located at ", dQuote(logf),".\n"))
  }

  ## everything ok, we can read the actual output from argus
  out <- read_ArgusSolution(outtab)
  if (!is.null(out)) {
    out <- combineInputs(obj=obj, out, batchF=batchF)
  }
  return(out[])
}
