% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rvine_structure.R
\name{rvine_structure}
\alias{rvine_structure}
\alias{rvine_matrix}
\alias{is.rvine_structure}
\alias{is.rvine_matrix}
\alias{dvine_structure}
\alias{cvine_structure}
\title{R-vine structure}
\usage{
rvine_structure(order, struct_array = list(), is_natural_order = FALSE)

cvine_structure(order, trunc_lvl = Inf)

dvine_structure(order, trunc_lvl = Inf)

rvine_matrix(matrix)
}
\arguments{
\item{order}{a vector of positive integers.}

\item{struct_array}{a list of vectors of positive integers. The vectors
represent rows of the r-rvine structure and the number of elements have to
be compatible with the \code{order} vector. If empty, the model is 0-truncated.}

\item{is_natural_order}{whether \code{struct_array} is assumed to be provided in
natural order already (a structure is in natural order if the anti-
diagonal is 1, .., d from bottom left to top right).}

\item{trunc_lvl}{the truncation level}

\item{matrix}{an R-vine matrix, see \emph{Details}.}
}
\value{
Either an \code{rvine_structure} or an \code{rvine_matrix}.
}
\description{
R-vine structures are compressed representations encoding the tree structure
of the vine, i.e. the conditioned/conditioning variables of each edge. The
functions \verb{[cvine_structure()]} or \verb{[dvine_structure()]} give a simpler way
to construct C-vines (every tree is a star) and D-vines (every tree is a
path), respectively (see \emph{Examples}).
}
\details{
The R-vine structure is essentially a lower-triangular matrix/triangular
array, with a notation that differs from the one in the VineCopula package.
An example array is\preformatted{4 4 4 4
3 3 3
2 2
1
}

which encodes the following pair-copulas:

\tabular{lll}{
tree \tab  edge \tab pair-copulas   \cr
0    \tab  0    \tab \verb{(1, 4)}       \cr
\tab  1    \tab \verb{(2, 4)}       \cr
\tab  2    \tab \verb{(3, 4)}       \cr
1    \tab  0    \tab \verb{(1, 3; 4)}    \cr
\tab  1    \tab \verb{(2, 3; 4)}    \cr
2    \tab  0    \tab \verb{(1, 2; 3, 4)}
}

An R-vine structure can be converted to an R-vine matrix using
\code{\link[=as_rvine_matrix]{as_rvine_matrix()}}, which encodes the same model with a square matrix filled
with zeros. For instance, the matrix corresponding to the structure above is:\preformatted{4 4 4 4
3 3 3 0
2 2 0 0
1 0 0 0
}

Similarly, an R-vine matrix can be converted to an R-vine structure using
\code{\link[=as_rvine_structure]{as_rvine_structure()}}.

Denoting by \code{M[i, j]} the array entry in row \code{i} and column \code{j} (the
pair-copula index for edge \code{e} in tree \code{t} of a \code{d} dimensional vine is
\verb{(M[d + 1 - e, e], M[t, e]; M[t - 1, e], ..., M[1, e])}. Less formally,
\enumerate{
\item Start with the counter-diagonal element of column \code{e} (first conditioned
variable).
\item Jump up to the element in row \code{t} (second conditioned variable).
\item Gather all entries further up in column \code{e} (conditioning set).
}

Internally, the diagonal is stored separately from the off-diagonal elements,
which are stored as a triangular array. For instance, the off-diagonal
elements off the structure above are stored as\preformatted{4 4 4
3 3
2
}

for the structure above. The reason is that it allows for parsimonious
representations of truncated models. For instance, the 2-truncated model is
represented by the same diagonal and the following truncated triangular
array:\preformatted{4 4 4
3 3
}

A valid R-vine structure or matrix must satisfy several conditions which are
checked when \code{\link[=rvine_structure]{rvine_structure()}}, \code{\link[=rvine_matrix]{rvine_matrix()}}, or some coercion methods
(see \code{\link[=as_rvine_structure]{as_rvine_structure()}} and \verb{as_rvine_matrix(}) are called:
\enumerate{
\item It can only contain numbers between 1 and d (and additionally zeros for
R-vine matrices).
\item The anti-diagonal must contain the numbers 1, ..., d.
\item The anti-diagonal entry of a column must not be contained in any column
further to the right.
\item The entries of a column must be contained in all columns to the left.
\item The proximity condition must hold: For all t = 1, ..., d - 2 and e = 1,
..., d - t there must exist an index j > d, such that
\verb{(M[t, e], \{M[1, e], ..., M[t - 1, e]\})} equals either
\verb{(M[d + 1 - j, j], \{M[1, j], ..., M[t - 1, j]\})} or
\verb{(M[t - 1, j], \{M[d + 1 - j, j], M[1, j], ..., M[t - 2, j]\})}.
}

Condition 5 already implies conditions 2-4, but is more difficult to check by
hand.
}
\examples{

# R-vine structures can be constructed from the order vector and struct_array
rvine_structure(order = 1:4, struct_array = list(
  c(4, 4, 4),
  c(3, 3),
  2
))

# R-vine matrices can be constructed from standard matrices
mat <- matrix(c(4, 3, 2, 1, 4, 3, 2, 0, 4, 3, 0, 0, 4, 0, 0, 0), 4, 4)
rvine_matrix(mat)

# coerce to R-vine structure
str(as_rvine_structure(mat))

# truncate and construct the R-vine matrix
mat[3, 1] <- 0
rvine_matrix(mat)

# or use directly the R-vine structure constructor
rvine_structure(order = 1:4, struct_array = list(
  c(4, 4, 4),
  c(3, 3)
))

# throws an error
mat[3, 1] <- 5
try(rvine_matrix(mat))

# C-vine structure
cvine <- cvine_structure(1:5)
cvine
plot(cvine)

# D-vine structure
dvine <- dvine_structure(c(1, 4, 2, 3, 5))
dvine
plot(dvine)

}
\seealso{
\code{\link[=as_rvine_structure]{as_rvine_structure()}}, \code{\link[=as_rvine_matrix]{as_rvine_matrix()}},
\code{\link[=plot.rvine_structure]{plot.rvine_structure()}}, \code{\link[=plot.rvine_matrix]{plot.rvine_matrix()}},
\code{\link[=rvine_structure_sim]{rvine_structure_sim()}}, \code{\link[=rvine_matrix_sim]{rvine_matrix_sim()}}
}
