#' @title Run trial simulations
#' @export
#' @seealso [get_facts_file_example()], [run_flfll()],
#'   [get_facts_engine()], [prep_param_files()]
#' @description For fine control over trial simulations,
#'   you must first call [run_flfll()] and then call either [run_engine()]
#'   or one of the specific engine functions (such as [run_engine_contin()]).
#'   The engines read the `*.param` files generated by
#'   [run_flfll()], run the trial simulations, and save output
#'   to a bunch of CSV files. You can find these CSV output files next to
#'   the `*.param` files.
#' @details If you need to repeatedly invoke an engine, as with most
#'   trial execution mode workflows, [run_engine()] is slow.
#'   Instead, consider running [prep_param_files()]
#'   and then passing the result to one of the individual engine
#'   functions (such as [run_engine_contin()]).
#' @return Nothing.
#' @inheritParams get_facts_version
#' @param ... Named arguments to the appropriate inner engine function,
#'   such as [run_engine_contin()]. Use [get_facts_engine()]
#'   to identify the appropriate engine function for your FACTS file.
#'   Then, open the help file of that function to read about the arguments.
#' @examples
#' # Can only run if system dependencies are configured:
#' if (file.exists(Sys.getenv("RFACTS_PATHS"))) {
#' facts_file <- get_facts_file_example("contin.facts") # example FACTS file
#' out <- run_flfll(facts_file, verbose = FALSE) # Generate param files.
#' # Run the simulations.
#' run_engine(
#'   facts_file,
#'   param_files = out,
#'   n_sims = 1,
#'   verbose = FALSE
#' )
#' read_patients(out)
#' }
run_engine <- function(facts_file, ...) {
  facts_file <- facts_xml(facts_file)
  version <- get_facts_version(facts_file)
  version <- choose_version(version, facts_versions())
  engine_name <- get_engine_name(facts_file)
  class(engine_name) <- engine_name
  run_engine_impl(engine_name = engine_name, version = version, ...)
  invisible()
}

run_engine_impl <- function(engine_name, version, ...) {
  UseMethod("run_engine_impl")
}

run_engine_impl.default <- function(engine_name, version) {
  stop0("unsupported engine ", engine_name)
}

run_engine_impl.aipf_contin <- function(engine_name, version, ...) {
  run_engine_aipf_contin(version = version, ...)
}

run_engine_impl.aipf_dichot <- function(engine_name, version, ...) {
  run_engine_aipf_dichot(version = version, ...)
}

run_engine_impl.aipf_tte <- function(engine_name, version, ...) {
  run_engine_aipf_tte(version = version, ...)
}

run_engine_impl.contin <- function(engine_name, version, ...) {
  run_engine_contin(version = version, ...)
}

run_engine_impl.crm <- function(engine_name, version, ...) {
  run_engine_crm(version = version, ...)
}

run_engine_impl.dichot <- function(engine_name, version, ...) {
  run_engine_dichot(version = version, ...)
}

run_engine_impl.multep <- function(engine_name, version, ...) {
  run_engine_multep(version = version, ...)
}

run_engine_impl.tte <- function(engine_name, version, ...) {
  run_engine_tte(version = version, ...)
}
