% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_pcrit.R
\encoding{UTF-8}
\name{plot_pcrit}
\alias{plot_pcrit}
\title{Plot Pcrit}
\usage{
plot_pcrit(
  po2,
  mo2,
  mo2_data,
  method = "Breakpoint",
  avg_top_n = 1,
  level = 0.95,
  iqr = 1.5,
  NLR_m = 0.065,
  MR = NULL,
  mo2_threshold = Inf,
  showNLRs = FALSE,
  ...
)
}
\arguments{
\item{po2}{a vector of PO2 values. Any unit of measurement should work, but the NLR calculation was optimized using kPa. If the NLR metric is giving you trouble, try converting to kPa using \code{\link{conv_o2}}.}

\item{mo2}{a vector of metabolic rate values. Must be the same length and corresponding to \code{po2}.}

\item{mo2_data}{for convenience, the output of \code{\link{calc_MO2}} can be entered here, as an alternative to specifying the \code{po2} and \code{mo2} parameters (optional).}

\item{method}{Over the years, many different methods of analysis have been proposed to quantify Pcrit. You must choose one of the following: Alpha, Breakpoint (default), LLO, NLR, Sub_PI, All. If in doubt, try "All".}

\item{avg_top_n}{applies to the \code{alpha} metric only (only when \code{method == "Alpha"} or \code{"All"}). A numeric value representing the number of top \eqn{\alpha0} (MO2/PO2) values to average together to estimate \eqn{\alpha}. Default is 1. We recommend no more than 3 to avoid diminishing the \eqn{\alpha} value with sub-maximal observations.}

\item{level}{applies to the \code{Sub_PI} metric only (only when \code{method == "Sub_PI"} or \code{"All"}). Percentage at which the prediction interval should be constructed. Default is 0.95.}

\item{iqr}{applies to the \code{Sub_PI} metric only (only when \code{method == "Sub_PI"} or \code{"All"}). Removes \code{mo2} observations that are this many interquartile ranges away from the mean value for the oxyregulating portion of the trial. If this filtering is not desired, set to infinity. To visualize which observations will be removed by this parameter, use \code{\link{plot_pcrit}}. Default is 1.5.}

\item{NLR_m}{applies to the \code{NLR} metric only (only when \code{method == "NLR"} or \code{"All"}). Pcrit is defined as the PO2 at which the slope of the best fitting function equals \code{NLR_m} (after the MO2 data are normalized to the 90\% quantile). Default is 0.065.}

\item{MR}{applies to the \code{alpha} and \code{LLO} metrics only (only when \code{method == "Alpha", "LLO"} or \code{"All"}). A numeric value for the metabolic rate at which \code{pcrit_alpha} and \code{pcrit_LLO} should be returned. If not supplied by the user, then the mean MO2 of the "oxyregulating" portion of the curve is applied for \code{pcrit_alpha} and \code{NA} is returned for \code{pcrit_LLO}.}

\item{mo2_threshold}{applies to the \code{alpha} metric only (only when \code{method == "Alpha"} or \code{"All"}). A single numeric value above which \code{mo2} values are ignored for \code{alpha} Pcrit estimation. Useful to removing obviously erroneous values. Default is \code{Inf}.}

\item{showNLRs}{logical. Should all the NLR functions be plotted in a second plot? If \code{FALSE} then only the best fit NLR function will be plotted.}

\item{...}{arguments to be passed to \code{\link[segmented]{plot.segmented}}.}
}
\value{
A plot is created showing the relationship between PO2 and MO2. Based on the \code{method} used, the alpha, breakpoint, LLO, NLR, and/or sub-PI Pcrit values are shown in the title and on the plot by inverted triangles.

For breakpoint and sub-PI methods, the broken-stick regression is shown by black lines. The gray bands represent the confidence interval (defaults to 95\% but will change with \code{level}). 

For the sub-PI method, the dashed red curves signify the prediction interval used. Black circles represent oxyregulating observations used in the generation of the prediction interval, while grey circles represent both the oxyconforming observations and those observations outside the IQR threshold (defined by \code{iqr}). 

For the NLR method, the green curve represents the best fitting NLR function and the green inverted triangle represents the NLR Pcrit (modified by \code{NLR_m}) 

For the Alpha method, the blue line represents alpha, which was fit based on the blue circle observation(s). If \code{MR} is not defined by the user, then the black points are those that were averaged to choose \code{MR}. These are the "oxyregulating" observations based on the breakpoint method.

If \code{showNLRs = TRUE}, then a second plot is generated which shows all the NLR functions that converged. Vertical lines represent the Pcrit values corresponding to each curve.

    Black = Michaelis-Menten

    Red = Power

    Green = Hyperbola

    Blue = Pareto

    Cyan = Weibull with intercept.
}
\description{
Creates a Pcrit plot (the threshold below which oxygen consumption rate can no longer be sustained) based on paired PO2 and MO2 values. Five Pcrit metrics are plotted: the traditional breakpoint metric (broken stick regression, black), the nonlinear regression metric (Marshall et al. 2013, green), the sub-prediction interval metric (Birk et al. 2019, red), the alpha-based Pcrit method (Seibel et al., 2021, blue), and the linear low O2 (LLO) method (Reemeyer & Rees 2019, purple). For details on how the Pcrit values are calculated, see \code{\link{calc_pcrit}}.
}
\details{
\describe{
\item{Alpha Pcrit}{Alpha is calculated from \code{\link{calc_alpha}} and the Pcrit corresponding to \code{MR} is returned. This determine's the animal's oxygen supply capacity and calculates the Pcrit at any given metabolic rate of interest. If no \code{MR} is provided, then it defaults to the mean MO2 value from the oxyregulating portion of the curve (as defined by the broken-stick regression).}
\item{Breakpoint Pcrit}{Data are fit to a broken-stick regression using \code{\link[segmented]{segmented}}.}
\item{LLO Pcrit}{A subset of observations are chosen only from those with an MO2 < \code{MR}. Then, a linear model is fit through the observations and Pcrit is calculated as the PO2 at which the line reaches \code{MR}.}
\item{NLR Pcrit}{Data are fit to the following functions: Michaelis-Menten, Power, Hyperbola, Pareto, and Weibull with intercept. Following the method developed by Marshall et al. 2013, the function that best fits the data (smallest AIC) is chosen and the Pcrit is determined as the PO2 at which the slope of the function is \code{NLR_m} (by default = 0.065 following the authors' suggestion).}
\item{Sub_PI Pcrit}{This metric builds off the \code{Breakpoint} metric and results in a systematically lower Pcrit value. This is useful for applications where it is important to ensure that Pcrit is not being overestimated. It represents a reasonable lower bounded estimate of the Pcrit value for a given trial. Once the \code{Breakpoint} Pcrit is calculated, a 95\% prediction interval (can be changed with the \code{level} argument) is calculated around the oxyregulating region (i.e. using PO2 values > breakpoint Pcrit). By default, \code{iqr} provides some filtering of abberant observations to prevent their influence on the calculated prediction interval. Finally, the Sub_PI Pcrit value is returned at the intersection of the oxyconforming line and the lower limit of the oxyregulating prediction interval.}
}
}
\examples{
raw_data <- system.file('extdata/pcrit_run/', package = 'respirometry')
o2_data <- import_pyroscience_workbench(folder = raw_data)
mo2_data <- calc_MO2(duration = o2_data$DURATION, o2 = o2_data$CH_1_O2, bin_width = 10, vol = 3)
plot_pcrit(mo2_data = mo2_data)

par(mfrow = c(2, 1))
plot_pcrit(po2 = mo2_data$O2_MEAN, mo2 = mo2_data$MO2, method = 'All', MR = 100, showNLRs = TRUE)

}
\references{
Birk, Matthew A., K.A.S. Mislan, Karen F. Wishner, and Brad A. Seibel. 2019. “Metabolic Adaptations of the Pelagic Octopod Japetella Diaphana to Oxygen Minimum Zones.” Deep-Sea Research Part I 148: 123–31.

Marshall, Dustin J., Michael Bode, and Craig R. White. 2013. “Estimating Physiological Tolerances - a Comparison of Traditional Approaches to Nonlinear Regression Techniques.” Journal of Experimental Biology 216(12): 2176–82.

Reemeyer, Jessica E., and Bernard B. Rees. 2019. “Standardizing the Determination and Interpretation of Pcrit in Fishes.” Journal of Experimental Biology 222(18): jeb210633.

Seibel, B. A., A. Andres, M. A. Birk, A. L. Burns, C. T. Shaw, A. W. Timpe, C. J. Welsh. 2021. “Oxygen supply capacity breathes new life into the critical oxygen partial pressure (Pcrit).” Journal of Experimental Biology.
}
\seealso{
\code{\link{calc_pcrit}}, \code{\link{calc_alpha}}
}
\author{
Matthew A. Birk, \email{matthewabirk@gmail.com}
}
