################################################################################
### Tests for internal functions
##
## Created on: 2020-03-09
## Author: Kazuki Yoshida
################################################################################

## Load testthat in case this is run in isolation.
library(testthat)
library(survival)
library(tidyverse)


###
### Internal function for yreg model fitting (logistic)
################################################################################

describe("fit_yreg poisson (no interaction)", {

    data(pbc)

    ## Missing data should be warned in validate_args()
    pbc_cc <- pbc[complete.cases(pbc),] %>%
        mutate(male = if_else(sex == "m", 1L, 0L))

    it("fits a correct model with no covariates", {
        ## No covariates
        yreg_fit0 <- fit_yreg(yreg = "poisson",
                              data = pbc_cc,
                              yvar = "platelet",
                              avar = "trt",
                              mvar = "bili",
                              cvar = NULL,
                              interaction = FALSE,
                              eventvar = NULL)
        ref_fit0 <- glm(formula = platelet ~ trt + bili,
                        family = poisson(link = "log"),
                        data = pbc_cc)
        ## Same classes
        expect_equal(class(yreg_fit0),
                     class(ref_fit0))
        ## Same formula
        expect_equal(as.character(yreg_fit0$call$formula),
                     as.character(ref_fit0$call$formula))
        ## Same coef
        expect_equal(coef(yreg_fit0),
                     coef(ref_fit0))
        ## Same vcov
        expect_equal(vcov(yreg_fit0),
                     vcov(ref_fit0))
    })

    it("fits a correct model with one covariate", {
        ## One covariates
        yreg_fit1 <- fit_yreg(yreg = "poisson",
                              data = pbc_cc,
                              yvar = "platelet",
                              avar = "trt",
                              mvar = "bili",
                              cvar = c("age"),
                              interaction = FALSE,
                              eventvar = NULL)
        ref_fit1 <- glm(formula = platelet ~ trt + bili + age,
                        family = poisson(link = "log"),
                        data = pbc_cc)
        ## Same classes
        expect_equal(class(yreg_fit1),
                     class(ref_fit1))
        ## Same formula
        expect_equal(as.character(yreg_fit1$call$formula),
                     as.character(ref_fit1$call$formula))
        ## Same coef
        expect_equal(coef(yreg_fit1),
                     coef(ref_fit1))
        ## Same vcov
        expect_equal(vcov(yreg_fit1),
                     vcov(ref_fit1))
    })

    it("fits a correct model with three covariates", {
        ## Three covariates
        yreg_fit3 <- fit_yreg(yreg = "poisson",
                              data = pbc_cc,
                              yvar = "platelet",
                              avar = "trt",
                              mvar = "bili",
                              cvar = c("age","male","stage"),
                              interaction = FALSE,
                              eventvar = NULL)
        ref_fit3 <- glm(formula = platelet ~ trt + bili + age + male + stage,
                        family = poisson(link = "log"),
                        data = pbc_cc)
        ## Same classes
        expect_equal(class(yreg_fit3),
                     class(ref_fit3))
        ## Same formula
        expect_equal(as.character(yreg_fit3$call$formula),
                     as.character(ref_fit3$call$formula))
        ## Same coef
        expect_equal(coef(yreg_fit3),
                     coef(ref_fit3))
        ## Same vcov
        expect_equal(vcov(yreg_fit3),
                     vcov(ref_fit3))
    })

})


describe("fit_yreg poisson (interaction)", {

    data(pbc)

    ## Missing data should be warned in validate_args()
    pbc_cc <- pbc[complete.cases(pbc),] %>%
        mutate(male = if_else(sex == "m", 1L, 0L))

    it("fits a correct model with no covariates", {
        ## No covariates
        yreg_fit0 <- fit_yreg(yreg = "poisson",
                              data = pbc_cc,
                              yvar = "platelet",
                              avar = "trt",
                              mvar = "bili",
                              cvar = NULL,
                              interaction = TRUE,
                              eventvar = NULL)
        ref_fit0 <- glm(formula = platelet ~ trt*bili,
                        family = poisson(link = "log"),
                        data = pbc_cc)
        ## Same classes
        expect_equal(class(yreg_fit0),
                     class(ref_fit0))
        ## Same formula
        expect_equal(as.character(yreg_fit0$call$formula),
                     as.character(ref_fit0$call$formula))
        ## Same coef
        expect_equal(coef(yreg_fit0),
                     coef(ref_fit0))
        ## Same vcov
        expect_equal(vcov(yreg_fit0),
                     vcov(ref_fit0))
    })

    it("fits a correct model with one covariate", {
        ## One covariates
        yreg_fit1 <- fit_yreg(yreg = "poisson",
                              data = pbc_cc,
                              yvar = "platelet",
                              avar = "trt",
                              mvar = "bili",
                              cvar = c("age"),
                              interaction = TRUE,
                              eventvar = NULL)
        ref_fit1 <- glm(formula = platelet ~ trt*bili + age,
                        family = poisson(link = "log"),
                        data = pbc_cc)
        ## Same classes
        expect_equal(class(yreg_fit1),
                     class(ref_fit1))
        ## Same formula
        expect_equal(as.character(yreg_fit1$call$formula),
                     as.character(ref_fit1$call$formula))
        ## Same coef
        expect_equal(coef(yreg_fit1),
                     coef(ref_fit1))
        ## Same vcov
        expect_equal(vcov(yreg_fit1),
                     vcov(ref_fit1))
    })

    it("fits a correct model with three covariates", {
        ## Three covariates
        yreg_fit3 <- fit_yreg(yreg = "poisson",
                              data = pbc_cc,
                              yvar = "platelet",
                              avar = "trt",
                              mvar = "bili",
                              cvar = c("age","male","stage"),
                              interaction = TRUE,
                              eventvar = NULL)
        ref_fit3 <- glm(formula = platelet ~ trt*bili + age + male + stage,
                        family = poisson(link = "log"),
                        data = pbc_cc)
        ## Same classes
        expect_equal(class(yreg_fit3),
                     class(ref_fit3))
        ## Same formula
        expect_equal(as.character(yreg_fit3$call$formula),
                     as.character(ref_fit3$call$formula))
        ## Same coef
        expect_equal(coef(yreg_fit3),
                     coef(ref_fit3))
        ## Same vcov
        expect_equal(vcov(yreg_fit3),
                     vcov(ref_fit3))
    })

})
