% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misc_functions.R
\name{combine_query.aurum}
\alias{combine_query.aurum}
\title{Combine a CPRD aurum database query with a cohort.}
\usage{
\method{combine_query}{aurum}(
  db_query,
  cohort,
  query_type,
  time_prev = Inf,
  time_post = Inf,
  lower_bound = -Inf,
  upper_bound = Inf,
  numobs = 1,
  value_na_rm = TRUE,
  earliest_values = FALSE,
  reduce_output = TRUE
)
}
\arguments{
\item{db_query}{Output from database query (ideally obtained through \code{\link{db_query}}).}

\item{cohort}{Cohort to combine with the database query.}

\item{query_type}{Type of query}

\item{time_prev}{Number of days prior to index date to look for codes.}

\item{time_post}{Number of days after index date to look for codes.}

\item{lower_bound}{Lower bound for returned values when \code{query_type = "test"}.}

\item{upper_bound}{Upper bound for returned values when \code{query_type = "test"}.}

\item{numobs}{Number of observations to be returned.}

\item{value_na_rm}{If TRUE will remove data with NA in the \code{value} column of the queried data and remove values outside of \code{lower_bound} and \code{upper_bound} when \code{query_type = "test"}.}

\item{earliest_values}{If TRUE will return the earliest values as opposed to most recent.}

\item{reduce_output}{If TRUE will reduce output to just \code{patid}, event date, medical/product code, and test \code{value}.}
}
\value{
A data.table with observations that meet specified criteria.
}
\description{
An S3 method that can be used on database queries from Aurum extracts.
Combine a database query with a cohort, only retaining observations between \code{time_prev} days prior to \code{indexdt}, and \code{time_post} days after
\code{indexdt}, and for test data with values between \code{lower_bound} and \code{upper_bound}. The most recent \code{numobs} observations will be returned.
\code{cohort} must contain variables \code{patid} and \code{indexdt}. The \code{type} of query must be specified for appropriate data manipulation. Input \code{type = med} if
interested in medical diagnoses from the observation file, and \code{type = test} if interseted in test data from the observation file.
}
\details{
\code{value_na_rm = FALSE} may be of use when extracting variables like smoking status, where we want test data for number of cigarettes per day,
but do not want to remove all observations with NA in the \code{value} column, because the medcodeid itself may indicate smoking status.
}
\examples{
## Create connection to a temporary database
aurum_extract <- connect_database(file.path(tempdir(), "temp.sqlite"))

## Add observation data from all observation files in specified directory
cprd_extract(db = aurum_extract,
filepath = system.file("aurum_data", package = "rcprd"),
filetype = "observation")

## Query database for a specific medcode
db_query <- db_query(db_open = aurum_extract,
tab ="observation",
codelist_vector = "187341000000114")

## Define cohort
pat<-extract_cohort(filepath = system.file("aurum_data", package = "rcprd"))

### Add an index date to pat
pat$indexdt <- as.Date("01/01/2020", format = "\%d/\%m/\%Y")

## Combine query with cohort retaining most recent three records
combine_query(cohort = pat,
db_query = db_query,
query_type = "med",
numobs = 3)

## clean up
RSQLite::dbDisconnect(aurum_extract)
unlink(file.path(tempdir(), "temp.sqlite"))

}
