############################################################################################
## Hexaploid likelihood, LOD and recombination frequency functions
## Assumes random bivalent pairing. 
## Peter Bourke, Plant Breeding, Wageningen University and Research. January 2017
############################################################################################


#' Calculate recombination frequency, LOD and log-likelihood from frequency tables in a random pairing hexaploid
#' @description This group of functions is called by \code{\link{linkage}}.
#' @param x A frequency table of the different classes of dosages in the progeny. The column names start with \code{"n_"}. Followed by the dosage of the first marker and then of the second.
#' @return
#' A list with the following items:
#' \item{r_mat}{A matrix with recombination frequencies for the different phases}
#' \item{LOD_mat}{A matrix with LOD scores for the different phases}
#' \item{logL_mat}{A matrix with log likelihood ratios for the different phases}
#' \item{phasing_strategy}{A character string specifying the phasing strategy. \code{"MLL"} for maximum likelihood en \code{"MINR"} for minimum recombination frequency.}
#' \item{possible_phases}{The phases between markers that are possible. Same order and length as column names of output matrices.}
#' @name r6_functions
NULL

#' @rdname r6_functions
r6_1.0_1.0 <- function(x){
r_c <- (x[,"n_01"] + x[,"n_10"])/(x[,"n_00"] + x[,"n_01"] + x[,"n_10"] + x[,"n_11"])
logL_c <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_10"] - x[,"n_11"])*log(2) + (x[,"n_00"] + x[,"n_11"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_10"])*log(pmax(1e-6,r_c))
LOD_c <- (x[,"n_01"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_11"])*log10(2) + (x[,"n_00"] + x[,"n_11"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_10"])*log10(pmax(1e-6,r_c))
r_r <- (3*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_10"] + 3*x[,"n_11"])/(x[,"n_00"] + x[,"n_01"] + x[,"n_10"] + x[,"n_11"])
logL_r <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_10"] - x[,"n_11"])*(log(2) + log(5)) + (x[,"n_01"] + x[,"n_10"])*log(3 - r_r) + (x[,"n_00"] + x[,"n_11"])*log(2 + r_r)
LOD_r <- -((x[,"n_01"] + x[,"n_10"])*(-log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_11"])*(-log10(2) + log10(5)) + (x[,"n_01"] + x[,"n_10"])*log10(3 - r_r) + (x[,"n_00"] + x[,"n_11"])*log10(2 + r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_1.1 <- function(x){
r_c <- (x[,"n_02"] + x[,"n_10"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])
logL_c <- 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"])*log(2) + (x[,"n_00"] + x[,"n_12"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_10"])*log(pmax(1e-6,r_c))
LOD_c <- (x[,"n_02"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_12"])*log10(2) + (x[,"n_00"] + x[,"n_12"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_10"])*log10(pmax(1e-6,r_c))
r_r <- (3*x[,"n_00"] - 2*x[,"n_02"] - 2*x[,"n_10"] + 3*x[,"n_12"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])
logL_r <- 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"])*log(2) + (-x[,"n_00"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"])*log(5) + (x[,"n_02"] + x[,"n_10"])*log(3 - r_r) + (x[,"n_00"] + x[,"n_12"])*log(2 + r_r)
LOD_r <- -((x[,"n_02"] + x[,"n_10"])*(-log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_12"])*(-log10(2) + log10(5)) + (x[,"n_02"] + x[,"n_10"])*log10(3 - r_r) + (x[,"n_00"] + x[,"n_12"])*log10(2 + r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_1.2 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n10,n11,n12,n13) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(log(2) + log(5)) + (n01 + n12)*log(3 - 2*r) + (n00 + n13)*log(pmax(1e-6,1 - r)) + (n03 + n10)*log(pmax(1e-6,r)) + (n02 + n11)*log(1 + 2*r)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n10,n11,n12,n13) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"])
LOD_c <- (x[,"n_03"] + x[,"n_10"])*log10(2) - (x[,"n_02"] + x[,"n_11"])*log10(2) - (x[,"n_01"] + x[,"n_12"])*log10(2) + (x[,"n_00"] + x[,"n_13"])*log10(2) + (x[,"n_01"] + x[,"n_12"])*log10(3 - 2*r_c) + (x[,"n_00"] + x[,"n_13"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_03"] + x[,"n_10"])*log10(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_11"])*log10(1 + 2*r_c)
logL_c <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"])*(log(2) + log(5)) + (x[,"n_01"] + x[,"n_12"])*log(3 - 2*r_c) + (x[,"n_00"] + x[,"n_13"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_03"] + x[,"n_10"])*log(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_11"])*log(1 + 2*r_c)
logL_r <- function(r,n00,n01,n02,n03,n10,n11,n12,n13) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(log(2) + 2*log(5)) + (n02 + n11)*log(11 - 2*r) + (n03 + n10)*log(3 - r) + (n00 + n13)*log(2 + r) + (n01 + n12)*log(9 + 2*r)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n10,n11,n12,n13) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"])
LOD_r <- -((x[,"n_03"] + x[,"n_10"])*(-log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_13"])*(-log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_11"])*(log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_12"])*(log10(2) + log10(5)) + (x[,"n_02"] + x[,"n_11"])*log10(11 - 2*r_r) + (x[,"n_03"] + x[,"n_10"])*log10(3 - r_r) + (x[,"n_00"] + x[,"n_13"])*log10(2 + r_r) + (x[,"n_01"] + x[,"n_12"])*log10(9 + 2*r_r)
logL_r <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"])*(log(2) + 2*log(5)) + (x[,"n_02"] + x[,"n_11"])*log(11 - 2*r_r) + (x[,"n_03"] + x[,"n_10"])*log(3 - r_r) + (x[,"n_00"] + x[,"n_13"])*log(2 + r_r) + (x[,"n_01"] + x[,"n_12"])*log(9 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_1.3 <- function(x){
logL_c <- function(r,n02,n12,n00,n01,n03,n04,n10,n11,n13,n14) {
L <- 2*(n02 + n12)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*(3*log(2) + log(5)) + (n01 + n13)*log(9 - 8*r) + (n00 + n14)*log(pmax(1e-6,1 - r)) + (n04 + n10)*log(pmax(1e-6,r)) + (n03 + n11)*log(1 + 8*r)
return(L)}
interlogL_c <- function(n02,n12,n00,n01,n03,n04,n10,n11,n13,n14) {
optimize(logL_c,c(0,0.5), n02,n12,n00,n01,n03,n04,n10,n11,n13,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_02"],x[,"n_12"],x[,"n_00"],x[,"n_01"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_13"],x[,"n_14"])
LOD_c <- (x[,"n_04"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_14"])*log10(2) - (x[,"n_03"] + x[,"n_11"])*log10(5) - (x[,"n_01"] + x[,"n_13"])*log10(5) + (x[,"n_01"] + x[,"n_13"])*log10(9 - 8*r_c) + (x[,"n_00"] + x[,"n_14"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_10"])*log10(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_11"])*log10(1 + 8*r_c)
logL_c <- 2*(x[,"n_02"] + x[,"n_12"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*(3*log(2) + log(5)) + (x[,"n_01"] + x[,"n_13"])*log(9 - 8*r_c) + (x[,"n_00"] + x[,"n_14"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_10"])*log(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_11"])*log(1 + 8*r_c)
logL_r <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14) {
L <- 3*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(2) + 2*(n02 + n12)*log(3) + (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - n12 - 2*n13 - 2*n14)*log(5) + (n03 + n11)*log(29 - 8*r) + (n04 + n10)*log(3 - r) + (n00 + n14)*log(2 + r) + (n01 + n13)*log(21 + 8*r)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"])
LOD_r <- -2*(x[,"n_03"] + x[,"n_11"])*log10(5) - 2*(x[,"n_01"] + x[,"n_13"])*log10(5) - (x[,"n_04"] + x[,"n_10"])*(-log10(2) + log10(5)) - (x[,"n_00"] + x[,"n_14"])*(-log10(2) + log10(5)) + (x[,"n_03"] + x[,"n_11"])*log10(29 - 8*r_r) + (x[,"n_04"] + x[,"n_10"])*log10(3 - r_r) + (x[,"n_00"] + x[,"n_14"])*log10(2 + r_r) + (x[,"n_01"] + x[,"n_13"])*log10(21 + 8*r_r)
logL_r <- 3*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*log(2) + 2*(x[,"n_02"] + x[,"n_12"])*log(3) + (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"])*log(5) + (x[,"n_03"] + x[,"n_11"])*log(29 - 8*r_r) + (x[,"n_04"] + x[,"n_10"])*log(3 - r_r) + (x[,"n_00"] + x[,"n_14"])*log(2 + r_r) + (x[,"n_01"] + x[,"n_13"])*log(21 + 8*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_1.4 <- function(x){
logL_c <- function(r,n01,n02,n03,n04,n11,n12,n13,n14) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14)*(log(2) + log(5)) + (n02 + n13)*log(3 - 2*r) + (n01 + n14)*log(pmax(1e-6,1 - r)) + (n04 + n11)*log(pmax(1e-6,r)) + (n03 + n12)*log(1 + 2*r)
return(L)}
interlogL_c <- function(n01,n02,n03,n04,n11,n12,n13,n14) {
optimize(logL_c,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"])
LOD_c <- (x[,"n_04"] + x[,"n_11"])*log10(2) - (x[,"n_03"] + x[,"n_12"])*log10(2) - (x[,"n_02"] + x[,"n_13"])*log10(2) + (x[,"n_01"] + x[,"n_14"])*log10(2) + (x[,"n_02"] + x[,"n_13"])*log10(3 - 2*r_c) + (x[,"n_01"] + x[,"n_14"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_11"])*log10(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_12"])*log10(1 + 2*r_c)
logL_c <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*(log(2) + log(5)) + (x[,"n_02"] + x[,"n_13"])*log(3 - 2*r_c) + (x[,"n_01"] + x[,"n_14"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_11"])*log(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_12"])*log(1 + 2*r_c)
logL_r <- function(r,n01,n02,n03,n04,n11,n12,n13,n14) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14)*(log(2) + 2*log(5)) + (n03 + n12)*log(11 - 2*r) + (n04 + n11)*log(3 - r) + (n01 + n14)*log(2 + r) + (n02 + n13)*log(9 + 2*r)
return(L)}
interlogL_r <- function(n01,n02,n03,n04,n11,n12,n13,n14) {
optimize(logL_r,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"])
LOD_r <- -((x[,"n_04"] + x[,"n_11"])*(-log10(2) + log10(5))) - (x[,"n_01"] + x[,"n_14"])*(-log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_12"])*(log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_13"])*(log10(2) + log10(5)) + (x[,"n_03"] + x[,"n_12"])*log10(11 - 2*r_r) + (x[,"n_04"] + x[,"n_11"])*log10(3 - r_r) + (x[,"n_01"] + x[,"n_14"])*log10(2 + r_r) + (x[,"n_02"] + x[,"n_13"])*log10(9 + 2*r_r)
logL_r <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*(log(2) + 2*log(5)) + (x[,"n_03"] + x[,"n_12"])*log(11 - 2*r_r) + (x[,"n_04"] + x[,"n_11"])*log(3 - r_r) + (x[,"n_01"] + x[,"n_14"])*log(2 + r_r) + (x[,"n_02"] + x[,"n_13"])*log(9 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_1.5 <- function(x){
r_c <- (x[,"n_04"] + x[,"n_12"])/(x[,"n_02"] + x[,"n_04"] + x[,"n_12"] + x[,"n_14"])
logL_c <- 2*(-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*log(2) + (x[,"n_02"] + x[,"n_14"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_12"])*log(pmax(1e-6,r_c))
LOD_c <- (x[,"n_04"] + x[,"n_12"])*log10(2) + (x[,"n_02"] + x[,"n_14"])*log10(2) + (x[,"n_02"] + x[,"n_14"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_12"])*log10(pmax(1e-6,r_c))
r_r <- (3*x[,"n_02"] - 2*x[,"n_04"] - 2*x[,"n_12"] + 3*x[,"n_14"])/(x[,"n_02"] + x[,"n_04"] + x[,"n_12"] + x[,"n_14"])
logL_r <- 2*(-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*log(2) + (-x[,"n_02"] - x[,"n_04"] - x[,"n_12"] - x[,"n_14"])*log(5) + (x[,"n_04"] + x[,"n_12"])*log(3 - r_r) + (x[,"n_02"] + x[,"n_14"])*log(2 + r_r)
LOD_r <- -((x[,"n_04"] + x[,"n_12"])*(-log10(2) + log10(5))) - (x[,"n_02"] + x[,"n_14"])*(-log10(2) + log10(5)) + (x[,"n_04"] + x[,"n_12"])*log10(3 - r_r) + (x[,"n_02"] + x[,"n_14"])*log10(2 + r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_2.0 <- function(x){
r_c <- (x[,"n_02"] + x[,"n_10"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])
logL_c <- -((-x[,"n_01"] - x[,"n_11"])*(-log(2) + log(3))) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"])*log(5) + (x[,"n_00"] + x[,"n_12"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_10"])*log(pmax(1e-6,r_c))
LOD_c <- (x[,"n_02"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_12"])*log10(2) + (x[,"n_00"] + x[,"n_12"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_10"])*log10(pmax(1e-6,r_c))
r_r <- (3*x[,"n_00"] - x[,"n_02"] - x[,"n_10"] + 3*x[,"n_12"])/(2*(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"]))
logL_r <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_10"] - x[,"n_11"] - 2*x[,"n_12"])*log(2) + (x[,"n_01"] + x[,"n_11"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"])*log(5) + (x[,"n_02"] + x[,"n_10"])*log(3 - 2*r_r) + (x[,"n_00"] + x[,"n_12"])*log(1 + 2*r_r)
LOD_r <- -((x[,"n_02"] + x[,"n_10"])*log10(2)) - (x[,"n_00"] + x[,"n_12"])*log10(2) + (x[,"n_02"] + x[,"n_10"])*log10(3 - 2*r_r) + (x[,"n_00"] + x[,"n_12"])*log10(1 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_2.1 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n10,n11,n12,n13) {
L <- (-n00 - 2*n01 - 2*n02 - n03 - n10 - 2*n11 - 2*n12 - n13)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*log(5) + (n01 + n12)*log(5 - 2*r) + (n00 + n13)*log(pmax(1e-6,1 - r)) + (n03 + n10)*log(pmax(1e-6,r)) + (n02 + n11)*log(3 + 2*r)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n10,n11,n12,n13) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"])
LOD_c <- (x[,"n_03"] + x[,"n_10"])*log10(2) - 2*(x[,"n_02"] + x[,"n_11"])*log10(2) - 2*(x[,"n_01"] + x[,"n_12"])*log10(2) + (x[,"n_00"] + x[,"n_13"])*log10(2) + (x[,"n_01"] + x[,"n_12"])*log10(5 - 2*r_c) + (x[,"n_00"] + x[,"n_13"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_03"] + x[,"n_10"])*log10(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_11"])*log10(3 + 2*r_c)
logL_c <- (-x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - x[,"n_13"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"])*log(5) + (x[,"n_01"] + x[,"n_12"])*log(5 - 2*r_c) + (x[,"n_00"] + x[,"n_13"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_03"] + x[,"n_10"])*log(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_11"])*log(3 + 2*r_c)
logL_r <- function(r,n00,n01,n02,n03,n10,n11,n12,n13) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(3*log(2) + log(5)) + (n03 + n10)*log(3 - 2*r) + (n02 + n11)*log(9 - 2*r) + (n00 + n13)*log(1 + 2*r) + (n01 + n12)*log(7 + 2*r)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n10,n11,n12,n13) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"])
LOD_r <- -((x[,"n_03"] + x[,"n_10"])*log10(2)) - 3*(x[,"n_02"] + x[,"n_11"])*log10(2) - 3*(x[,"n_01"] + x[,"n_12"])*log10(2) - (x[,"n_00"] + x[,"n_13"])*log10(2) + (x[,"n_03"] + x[,"n_10"])*log10(3 - 2*r_r) + (x[,"n_02"] + x[,"n_11"])*log10(9 - 2*r_r) + (x[,"n_00"] + x[,"n_13"])*log10(1 + 2*r_r) + (x[,"n_01"] + x[,"n_12"])*log10(7 + 2*r_r)
logL_r <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"])*(3*log(2) + log(5)) + (x[,"n_03"] + x[,"n_10"])*log(3 - 2*r_r) + (x[,"n_02"] + x[,"n_11"])*log(9 - 2*r_r) + (x[,"n_00"] + x[,"n_13"])*log(1 + 2*r_r) + (x[,"n_01"] + x[,"n_12"])*log(7 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_2.2 <- function(x){
logL_c <- function(r,n01,n02,n03,n11,n12,n13,n00,n04,n10,n14) {
L <- (-n01 - n02 - n03 - n11 - n12 - n13)*log(2) + (n01 + n03 + n11 + n13)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(5) + (n02 + n12)*log(11) + (n01 + n13)*log(3 - 2*r) + (n00 + n14)*log(pmax(1e-6,1 - r)) + (n04 + n10)*log(pmax(1e-6,r)) + (n03 + n11)*log(1 + 2*r)
return(L)}
interlogL_c <- function(n01,n02,n03,n11,n12,n13,n00,n04,n10,n14) {
optimize(logL_c,c(0,0.5), n01,n02,n03,n11,n12,n13,n00,n04,n10,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_00"],x[,"n_04"],x[,"n_10"],x[,"n_14"])
LOD_c <- (x[,"n_04"] + x[,"n_10"])*log10(2) - (x[,"n_03"] + x[,"n_11"])*log10(2) - (x[,"n_01"] + x[,"n_13"])*log10(2) + (x[,"n_00"] + x[,"n_14"])*log10(2) + (x[,"n_01"] + x[,"n_13"])*log10(3 - 2*r_c) + (x[,"n_00"] + x[,"n_14"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_10"])*log10(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_11"])*log10(1 + 2*r_c)
logL_c <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"])*log(2) + (x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*log(5) + (x[,"n_02"] + x[,"n_12"])*log(11) + (x[,"n_01"] + x[,"n_13"])*log(3 - 2*r_c) + (x[,"n_00"] + x[,"n_14"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_10"])*log(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_11"])*log(1 + 2*r_c)
logL_r <- function(r,n00,n01,n02,n04,n10,n12,n13,n14,n03,n11) {
L <- (-2*n00 - 2*n01 - n02 - 2*n04 - 2*n10 - n12 - 2*n13 - 2*n14)*log(2) + (n01 + n13)*log(3) + 2*(-n00 - n01 - n02 - n04 - n10 - n12 - n13 - n14)*log(5) + (n02 + n12)*log(11) + (n04 + n10)*log(3 - 2*r) + (n03 + n11)*log(3/20 - (3*r)/50) + (n00 + n14)*log(1 + 2*r) + (n01 + n13)*log(3 + 2*r)
return(L)}
interlogL_r <- function(n00,n01,n02,n04,n10,n12,n13,n14,n03,n11) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n04,n10,n12,n13,n14,n03,n11, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_04"],x[,"n_10"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_03"],x[,"n_11"])
LOD_r <- -((x[,"n_04"] + x[,"n_10"])*log10(2)) - 2*(x[,"n_01"] + x[,"n_13"])*log10(2) - (x[,"n_00"] + x[,"n_14"])*log10(2) + (x[,"n_03"] + x[,"n_11"])*(-log10(3) + 2*log10(5)) + (x[,"n_04"] + x[,"n_10"])*log10(3 - 2*r_r) + (x[,"n_03"] + x[,"n_11"])*log10(3/20 - (3*r_r)/50) + (x[,"n_00"] + x[,"n_14"])*log10(1 + 2*r_r) + (x[,"n_01"] + x[,"n_13"])*log10(3 + 2*r_r)
logL_r <- (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_04"] - 2*x[,"n_10"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"])*log(2) + (x[,"n_01"] + x[,"n_13"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_04"] - x[,"n_10"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*log(5) + (x[,"n_02"] + x[,"n_12"])*log(11) + (x[,"n_04"] + x[,"n_10"])*log(3 - 2*r_r) + (x[,"n_03"] + x[,"n_11"])*log(3/20 - (3*r_r)/50) + (x[,"n_00"] + x[,"n_14"])*log(1 + 2*r_r) + (x[,"n_01"] + x[,"n_13"])*log(3 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_2.3 <- function(x){
logL_c <- function(r,n00,n01,n03,n04,n05,n10,n11,n12,n14,n15,n02,n13) {
L <- (-2*n00 - 3*n01 - 3*n03 - 3*n04 - 2*n05 - 2*n10 - 3*n11 - 3*n12 - 3*n14 - 2*n15)*log(2) + (n01 + n04 + n11 + n14)*log(3) + 2*(-n00 - n01 - n03 - n04 - n05 - n10 - n11 - n12 - n14 - n15)*log(5) + (n01 + n14)*log(7 - 6*r) + (n00 + n15)*log(pmax(1e-6,1 - r)) + (n02 + n13)*log(9/40 - (2*r)/25) + (n05 + n10)*log(pmax(1e-6,r)) + (n04 + n11)*log(1 + 6*r) + (n03 + n12)*log(29 + 16*r)
return(L)}
interlogL_c <- function(n00,n01,n03,n04,n05,n10,n11,n12,n14,n15,n02,n13) {
optimize(logL_c,c(0,0.5), n00,n01,n03,n04,n05,n10,n11,n12,n14,n15,n02,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_14"],x[,"n_15"],x[,"n_02"],x[,"n_13"])
LOD_c <- (x[,"n_05"] + x[,"n_10"])*log10(2) - 2*(x[,"n_04"] + x[,"n_11"])*log10(2) - 2*(x[,"n_01"] + x[,"n_14"])*log10(2) + (x[,"n_00"] + x[,"n_15"])*log10(2) + (x[,"n_02"] + x[,"n_13"])*(3*log10(2) + 2*log10(5) - log10(37)) - (x[,"n_03"] + x[,"n_12"])*log10(37) + (x[,"n_01"] + x[,"n_14"])*log10(7 - 6*r_c) + (x[,"n_00"] + x[,"n_15"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_13"])*log10(9/40 - (2*r_c)/25) + (x[,"n_05"] + x[,"n_10"])*log10(pmax(1e-6,r_c)) + (x[,"n_04"] + x[,"n_11"])*log10(1 + 6*r_c) + (x[,"n_03"] + x[,"n_12"])*log10(29 + 16*r_c)
logL_c <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_14"] - 2*x[,"n_15"])*log(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_14"] - x[,"n_15"])*log(5) + (x[,"n_01"] + x[,"n_14"])*log(7 - 6*r_c) + (x[,"n_00"] + x[,"n_15"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_13"])*log(9/40 - (2*r_c)/25) + (x[,"n_05"] + x[,"n_10"])*log(pmax(1e-6,r_c)) + (x[,"n_04"] + x[,"n_11"])*log(1 + 6*r_c) + (x[,"n_03"] + x[,"n_12"])*log(29 + 16*r_c)
logL_r <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15) {
L <- (-4*n00 - 4*n01 - 3*n02 - 3*n03 - 4*n04 - 4*n05 - 4*n10 - 4*n11 - 3*n12 - 3*n13 - 4*n14 - 4*n15)*log(2) + (n01 + n04 + n11 + n14)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15)*log(5) + (n03 + n12)*log(41 - 8*r) + (n04 + n11)*log(11 - 6*r) + (n05 + n10)*log(3 - 2*r) + (n00 + n15)*log(1 + 2*r) + (n01 + n14)*log(5 + 6*r) + (n02 + n13)*log(33 + 8*r)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"])
LOD_r <- -((x[,"n_05"] + x[,"n_10"])*log10(2)) - 3*(x[,"n_04"] + x[,"n_11"])*log10(2) - 3*(x[,"n_01"] + x[,"n_14"])*log10(2) - (x[,"n_00"] + x[,"n_15"])*log10(2) - (x[,"n_03"] + x[,"n_12"])*log10(37) - (x[,"n_02"] + x[,"n_13"])*log10(37) + (x[,"n_03"] + x[,"n_12"])*log10(41 - 8*r_r) + (x[,"n_04"] + x[,"n_11"])*log10(11 - 6*r_r) + (x[,"n_05"] + x[,"n_10"])*log10(3 - 2*r_r) + (x[,"n_00"] + x[,"n_15"])*log10(1 + 2*r_r) + (x[,"n_01"] + x[,"n_14"])*log10(5 + 6*r_r) + (x[,"n_02"] + x[,"n_13"])*log10(33 + 8*r_r)
logL_r <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_15"])*log(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"])*log(5) + (x[,"n_03"] + x[,"n_12"])*log(41 - 8*r_r) + (x[,"n_04"] + x[,"n_11"])*log(11 - 6*r_r) + (x[,"n_05"] + x[,"n_10"])*log(3 - 2*r_r) + (x[,"n_00"] + x[,"n_15"])*log(1 + 2*r_r) + (x[,"n_01"] + x[,"n_14"])*log(5 + 6*r_r) + (x[,"n_02"] + x[,"n_13"])*log(33 + 8*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_2.4 <- function(x){
logL_c <- function(r,n02,n03,n04,n12,n13,n14,n01,n05,n11,n15) {
L <- (-n02 - n03 - n04 - n12 - n13 - n14)*log(2) + (n02 + n04 + n12 + n14)*log(3) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15)*log(5) + (n03 + n13)*log(11) + (n02 + n14)*log(3 - 2*r) + (n01 + n15)*log(pmax(1e-6,1 - r)) + (n05 + n11)*log(pmax(1e-6,r)) + (n04 + n12)*log(1 + 2*r)
return(L)}
interlogL_c <- function(n02,n03,n04,n12,n13,n14,n01,n05,n11,n15) {
optimize(logL_c,c(0,0.5), n02,n03,n04,n12,n13,n14,n01,n05,n11,n15, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_01"],x[,"n_05"],x[,"n_11"],x[,"n_15"])
LOD_c <- (x[,"n_05"] + x[,"n_11"])*log10(2) - (x[,"n_04"] + x[,"n_12"])*log10(2) - (x[,"n_02"] + x[,"n_14"])*log10(2) + (x[,"n_01"] + x[,"n_15"])*log10(2) + (x[,"n_02"] + x[,"n_14"])*log10(3 - 2*r_c) + (x[,"n_01"] + x[,"n_15"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_05"] + x[,"n_11"])*log10(pmax(1e-6,r_c)) + (x[,"n_04"] + x[,"n_12"])*log10(1 + 2*r_c)
logL_c <- (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*log(2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_12"] + x[,"n_14"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"])*log(5) + (x[,"n_03"] + x[,"n_13"])*log(11) + (x[,"n_02"] + x[,"n_14"])*log(3 - 2*r_c) + (x[,"n_01"] + x[,"n_15"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_05"] + x[,"n_11"])*log(pmax(1e-6,r_c)) + (x[,"n_04"] + x[,"n_12"])*log(1 + 2*r_c)
logL_r <- function(r,n01,n02,n03,n05,n11,n13,n14,n15,n04,n12) {
L <- (-2*n01 - 2*n02 - n03 - 2*n05 - 2*n11 - n13 - 2*n14 - 2*n15)*log(2) + (n02 + n14)*log(3) + 2*(-n01 - n02 - n03 - n05 - n11 - n13 - n14 - n15)*log(5) + (n03 + n13)*log(11) + (n05 + n11)*log(3 - 2*r) + (n04 + n12)*log(3/20 - (3*r)/50) + (n01 + n15)*log(1 + 2*r) + (n02 + n14)*log(3 + 2*r)
return(L)}
interlogL_r <- function(n01,n02,n03,n05,n11,n13,n14,n15,n04,n12) {
optimize(logL_r,c(0,0.5), n01,n02,n03,n05,n11,n13,n14,n15,n04,n12, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_05"],x[,"n_11"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_04"],x[,"n_12"])
LOD_r <- -((x[,"n_05"] + x[,"n_11"])*log10(2)) - 2*(x[,"n_02"] + x[,"n_14"])*log10(2) - (x[,"n_01"] + x[,"n_15"])*log10(2) + (x[,"n_04"] + x[,"n_12"])*(-log10(3) + 2*log10(5)) + (x[,"n_05"] + x[,"n_11"])*log10(3 - 2*r_r) + (x[,"n_04"] + x[,"n_12"])*log10(3/20 - (3*r_r)/50) + (x[,"n_01"] + x[,"n_15"])*log10(1 + 2*r_r) + (x[,"n_02"] + x[,"n_14"])*log10(3 + 2*r_r)
logL_r <- (-2*x[,"n_01"] - 2*x[,"n_02"] - x[,"n_03"] - 2*x[,"n_05"] - 2*x[,"n_11"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"])*log(2) + (x[,"n_02"] + x[,"n_14"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_05"] - x[,"n_11"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"])*log(5) + (x[,"n_03"] + x[,"n_13"])*log(11) + (x[,"n_05"] + x[,"n_11"])*log(3 - 2*r_r) + (x[,"n_04"] + x[,"n_12"])*log(3/20 - (3*r_r)/50) + (x[,"n_01"] + x[,"n_15"])*log(1 + 2*r_r) + (x[,"n_02"] + x[,"n_14"])*log(3 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_3.0 <- function(x){
logL_c <- function(r,n01,n02,n11,n12,n00,n03,n10,n13) {
L <- (n01 + n02 + n11 + n12)*log(3) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(2*log(2) + log(5)) + (n00 + n13)*log(pmax(1e-6,1 - r)) + (n01 + n12)*log(2 - r) + (n03 + n10)*log(pmax(1e-6,r)) + (n02 + n11)*log(1 + r)
return(L)}
interlogL_c <- function(n01,n02,n11,n12,n00,n03,n10,n13) {
optimize(logL_c,c(0,0.5), n01,n02,n11,n12,n00,n03,n10,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_01"],x[,"n_02"],x[,"n_11"],x[,"n_12"],x[,"n_00"],x[,"n_03"],x[,"n_10"],x[,"n_13"])
LOD_c <- (x[,"n_03"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_13"])*log10(2) - (x[,"n_02"] + x[,"n_11"])*(-log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_12"])*(-log10(2) + log10(3)) + (x[,"n_00"] + x[,"n_13"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_12"])*log10(2 - r_c) + (x[,"n_03"] + x[,"n_10"])*log10(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_11"])*log10(1 + r_c)
logL_c <- (x[,"n_01"] + x[,"n_02"] + x[,"n_11"] + x[,"n_12"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"])*(2*log(2) + log(5)) + (x[,"n_00"] + x[,"n_13"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_12"])*log(2 - r_c) + (x[,"n_03"] + x[,"n_10"])*log(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_11"])*log(1 + r_c)
logL_r <- function(r,n01,n02,n11,n12,n00,n03,n10,n13) {
L <- (n01 + n02 + n11 + n12)*log(3) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(2*log(2) + log(5)) + (n03 + n10)*log(pmax(1e-6,1 - r)) + (n02 + n11)*log(2 - r) + (n00 + n13)*log(pmax(1e-6,r)) + (n01 + n12)*log(1 + r)
return(L)}
interlogL_r <- function(n01,n02,n11,n12,n00,n03,n10,n13) {
optimize(logL_r,c(0,0.5), n01,n02,n11,n12,n00,n03,n10,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_02"],x[,"n_11"],x[,"n_12"],x[,"n_00"],x[,"n_03"],x[,"n_10"],x[,"n_13"])
LOD_r <- (x[,"n_03"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_13"])*log10(2) - (x[,"n_02"] + x[,"n_11"])*(-log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_12"])*(-log10(2) + log10(3)) + (x[,"n_03"] + x[,"n_10"])*log10(pmax(1e-6,1 - r_r)) + (x[,"n_02"] + x[,"n_11"])*log10(2 - r_r) + (x[,"n_00"] + x[,"n_13"])*log10(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_12"])*log10(1 + r_r)
logL_r <- (x[,"n_01"] + x[,"n_02"] + x[,"n_11"] + x[,"n_12"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"])*(2*log(2) + log(5)) + (x[,"n_03"] + x[,"n_10"])*log(pmax(1e-6,1 - r_r)) + (x[,"n_02"] + x[,"n_11"])*log(2 - r_r) + (x[,"n_00"] + x[,"n_13"])*log(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_12"])*log(1 + r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_3.1 <- function(x){
logL_c <- function(r,n02,n12,n00,n01,n03,n04,n10,n11,n13,n14) {
L <- 2*(n02 + n12)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*(3*log(2) + log(5)) + (n01 + n13)*log(7 - 4*r) + (n00 + n14)*log(pmax(1e-6,1 - r)) + (n04 + n10)*log(pmax(1e-6,r)) + (n03 + n11)*log(3 + 4*r)
return(L)}
interlogL_c <- function(n02,n12,n00,n01,n03,n04,n10,n11,n13,n14) {
optimize(logL_c,c(0,0.5), n02,n12,n00,n01,n03,n04,n10,n11,n13,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_02"],x[,"n_12"],x[,"n_00"],x[,"n_01"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_13"],x[,"n_14"])
LOD_c <- (x[,"n_04"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_14"])*log10(2) - (x[,"n_03"] + x[,"n_11"])*log10(5) - (x[,"n_01"] + x[,"n_13"])*log10(5) + (x[,"n_01"] + x[,"n_13"])*log10(7 - 4*r_c) + (x[,"n_00"] + x[,"n_14"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_10"])*log10(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_11"])*log10(3 + 4*r_c)
logL_c <- 2*(x[,"n_02"] + x[,"n_12"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*(3*log(2) + log(5)) + (x[,"n_01"] + x[,"n_13"])*log(7 - 4*r_c) + (x[,"n_00"] + x[,"n_14"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_10"])*log(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_11"])*log(3 + 4*r_c)
logL_r <- function(r,n02,n12,n00,n01,n03,n04,n10,n11,n13,n14) {
L <- 2*(n02 + n12)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*(3*log(2) + log(5)) + (n03 + n11)*log(7 - 4*r) + (n04 + n10)*log(pmax(1e-6,1 - r)) + (n00 + n14)*log(pmax(1e-6,r)) + (n01 + n13)*log(3 + 4*r)
return(L)}
interlogL_r <- function(n02,n12,n00,n01,n03,n04,n10,n11,n13,n14) {
optimize(logL_r,c(0,0.5), n02,n12,n00,n01,n03,n04,n10,n11,n13,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_02"],x[,"n_12"],x[,"n_00"],x[,"n_01"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_13"],x[,"n_14"])
LOD_r <- (x[,"n_04"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_14"])*log10(2) - (x[,"n_03"] + x[,"n_11"])*log10(5) - (x[,"n_01"] + x[,"n_13"])*log10(5) + (x[,"n_03"] + x[,"n_11"])*log10(7 - 4*r_r) + (x[,"n_04"] + x[,"n_10"])*log10(pmax(1e-6,1 - r_r)) + (x[,"n_00"] + x[,"n_14"])*log10(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_13"])*log10(3 + 4*r_r)
logL_r <- 2*(x[,"n_02"] + x[,"n_12"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*(3*log(2) + log(5)) + (x[,"n_03"] + x[,"n_11"])*log(7 - 4*r_r) + (x[,"n_04"] + x[,"n_10"])*log(pmax(1e-6,1 - r_r)) + (x[,"n_00"] + x[,"n_14"])*log(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_13"])*log(3 + 4*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_3.2 <- function(x){
logL_c <- function(r,n01,n04,n11,n14,n00,n02,n03,n05,n10,n12,n13,n15) {
L <- (n01 + n04 + n11 + n14)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15)*(2*log(2) + 2*log(5)) + (n02 + n13)*log(22 - 7*r) + (n01 + n14)*log(3 - 2*r) + (n00 + n15)*log(pmax(1e-6,1 - r)) + (n05 + n10)*log(pmax(1e-6,r)) + (n04 + n11)*log(1 + 2*r) + (n03 + n12)*log(15 + 7*r)
return(L)}
interlogL_c <- function(n01,n04,n11,n14,n00,n02,n03,n05,n10,n12,n13,n15) {
optimize(logL_c,c(0,0.5), n01,n04,n11,n14,n00,n02,n03,n05,n10,n12,n13,n15, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_01"],x[,"n_04"],x[,"n_11"],x[,"n_14"],x[,"n_00"],x[,"n_02"],x[,"n_03"],x[,"n_05"],x[,"n_10"],x[,"n_12"],x[,"n_13"],x[,"n_15"])
LOD_c <- (x[,"n_05"] + x[,"n_10"])*log10(2) - (x[,"n_04"] + x[,"n_11"])*log10(2) - (x[,"n_01"] + x[,"n_14"])*log10(2) + (x[,"n_00"] + x[,"n_15"])*log10(2) - (x[,"n_03"] + x[,"n_12"])*(-log10(2) + log10(37)) - (x[,"n_02"] + x[,"n_13"])*(-log10(2) + log10(37)) + (x[,"n_02"] + x[,"n_13"])*log10(22 - 7*r_c) + (x[,"n_01"] + x[,"n_14"])*log10(3 - 2*r_c) + (x[,"n_00"] + x[,"n_15"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_05"] + x[,"n_10"])*log10(pmax(1e-6,r_c)) + (x[,"n_04"] + x[,"n_11"])*log10(1 + 2*r_c) + (x[,"n_03"] + x[,"n_12"])*log10(15 + 7*r_c)
logL_c <- (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"])*(2*log(2) + 2*log(5)) + (x[,"n_02"] + x[,"n_13"])*log(22 - 7*r_c) + (x[,"n_01"] + x[,"n_14"])*log(3 - 2*r_c) + (x[,"n_00"] + x[,"n_15"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_05"] + x[,"n_10"])*log(pmax(1e-6,r_c)) + (x[,"n_04"] + x[,"n_11"])*log(1 + 2*r_c) + (x[,"n_03"] + x[,"n_12"])*log(15 + 7*r_c)
logL_r <- function(r,n01,n04,n11,n14,n00,n02,n03,n05,n10,n12,n13,n15) {
L <- (n01 + n04 + n11 + n14)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15)*(2*log(2) + 2*log(5)) + (n03 + n12)*log(22 - 7*r) + (n04 + n11)*log(3 - 2*r) + (n05 + n10)*log(pmax(1e-6,1 - r)) + (n00 + n15)*log(pmax(1e-6,r)) + (n01 + n14)*log(1 + 2*r) + (n02 + n13)*log(15 + 7*r)
return(L)}
interlogL_r <- function(n01,n04,n11,n14,n00,n02,n03,n05,n10,n12,n13,n15) {
optimize(logL_r,c(0,0.5), n01,n04,n11,n14,n00,n02,n03,n05,n10,n12,n13,n15, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_04"],x[,"n_11"],x[,"n_14"],x[,"n_00"],x[,"n_02"],x[,"n_03"],x[,"n_05"],x[,"n_10"],x[,"n_12"],x[,"n_13"],x[,"n_15"])
LOD_r <- (x[,"n_05"] + x[,"n_10"])*log10(2) - (x[,"n_04"] + x[,"n_11"])*log10(2) - (x[,"n_01"] + x[,"n_14"])*log10(2) + (x[,"n_00"] + x[,"n_15"])*log10(2) - (x[,"n_03"] + x[,"n_12"])*(-log10(2) + log10(37)) - (x[,"n_02"] + x[,"n_13"])*(-log10(2) + log10(37)) + (x[,"n_03"] + x[,"n_12"])*log10(22 - 7*r_r) + (x[,"n_04"] + x[,"n_11"])*log10(3 - 2*r_r) + (x[,"n_05"] + x[,"n_10"])*log10(pmax(1e-6,1 - r_r)) + (x[,"n_00"] + x[,"n_15"])*log10(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_14"])*log10(1 + 2*r_r) + (x[,"n_02"] + x[,"n_13"])*log10(15 + 7*r_r)
logL_r <- (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"])*(2*log(2) + 2*log(5)) + (x[,"n_03"] + x[,"n_12"])*log(22 - 7*r_r) + (x[,"n_04"] + x[,"n_11"])*log(3 - 2*r_r) + (x[,"n_05"] + x[,"n_10"])*log(pmax(1e-6,1 - r_r)) + (x[,"n_00"] + x[,"n_15"])*log(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_14"])*log(1 + 2*r_r) + (x[,"n_02"] + x[,"n_13"])*log(15 + 7*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_3.3 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16) {
L <- (-4*n00 - 4*n01 - 4*n02 - 3*n03 - 4*n04 - 4*n05 - 4*n06 - 4*n10 - 4*n11 - 4*n12 - 3*n13 - 4*n14 - 4*n15 - 4*n16)*log(2) + (n01 + n02 + n04 + n05 + n11 + n12 + n14 + n15)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16)*log(5) + (n02 + n04 + n12 + n14)*log(11) + (n03 + n13)*log(41) + (n01 + n15)*log(5 - 4*r) + (n00 + n16)*log(pmax(1e-6,1 - r)) + (n02 + n14)*log(2 - r) + (n06 + n10)*log(pmax(1e-6,r)) + (n04 + n12)*log(1 + r) + (n05 + n11)*log(1 + 4*r)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"])
LOD_c <- (x[,"n_06"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_16"])*log10(2) - (x[,"n_05"] + x[,"n_11"])*log10(3) - (x[,"n_01"] + x[,"n_15"])*log10(3) - (x[,"n_04"] + x[,"n_12"])*(-log10(2) + log10(3)) - (x[,"n_02"] + x[,"n_14"])*(-log10(2) + log10(3)) + (x[,"n_01"] + x[,"n_15"])*log10(5 - 4*r_c) + (x[,"n_00"] + x[,"n_16"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_14"])*log10(2 - r_c) + (x[,"n_06"] + x[,"n_10"])*log10(pmax(1e-6,r_c)) + (x[,"n_04"] + x[,"n_12"])*log10(1 + r_c) + (x[,"n_05"] + x[,"n_11"])*log10(1 + 4*r_c)
logL_c <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_06"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_15"] - 4*x[,"n_16"])*log(2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_04"] + x[,"n_05"] + x[,"n_11"] + x[,"n_12"] + x[,"n_14"] + x[,"n_15"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"])*log(5) + (x[,"n_02"] + x[,"n_04"] + x[,"n_12"] + x[,"n_14"])*log(11) + (x[,"n_03"] + x[,"n_13"])*log(41) + (x[,"n_01"] + x[,"n_15"])*log(5 - 4*r_c) + (x[,"n_00"] + x[,"n_16"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_14"])*log(2 - r_c) + (x[,"n_06"] + x[,"n_10"])*log(pmax(1e-6,r_c)) + (x[,"n_04"] + x[,"n_12"])*log(1 + r_c) + (x[,"n_05"] + x[,"n_11"])*log(1 + 4*r_c)
logL_r <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16) {
L <- (-4*n00 - 4*n01 - 4*n02 - 3*n03 - 4*n04 - 4*n05 - 4*n06 - 4*n10 - 4*n11 - 4*n12 - 3*n13 - 4*n14 - 4*n15 - 4*n16)*log(2) + (n01 + n02 + n04 + n05 + n11 + n12 + n14 + n15)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16)*log(5) + (n02 + n04 + n12 + n14)*log(11) + (n03 + n13)*log(41) + (n05 + n11)*log(5 - 4*r) + (n06 + n10)*log(pmax(1e-6,1 - r)) + (n04 + n12)*log(2 - r) + (n00 + n16)*log(pmax(1e-6,r)) + (n02 + n14)*log(1 + r) + (n01 + n15)*log(1 + 4*r)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"])
LOD_r <- (x[,"n_06"] + x[,"n_10"])*log10(2) + (x[,"n_00"] + x[,"n_16"])*log10(2) - (x[,"n_05"] + x[,"n_11"])*log10(3) - (x[,"n_01"] + x[,"n_15"])*log10(3) - (x[,"n_04"] + x[,"n_12"])*(-log10(2) + log10(3)) - (x[,"n_02"] + x[,"n_14"])*(-log10(2) + log10(3)) + (x[,"n_05"] + x[,"n_11"])*log10(5 - 4*r_r) + (x[,"n_06"] + x[,"n_10"])*log10(pmax(1e-6,1 - r_r)) + (x[,"n_04"] + x[,"n_12"])*log10(2 - r_r) + (x[,"n_00"] + x[,"n_16"])*log10(pmax(1e-6,r_r)) + (x[,"n_02"] + x[,"n_14"])*log10(1 + r_r) + (x[,"n_01"] + x[,"n_15"])*log10(1 + 4*r_r)
logL_r <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_06"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_15"] - 4*x[,"n_16"])*log(2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_04"] + x[,"n_05"] + x[,"n_11"] + x[,"n_12"] + x[,"n_14"] + x[,"n_15"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"])*log(5) + (x[,"n_02"] + x[,"n_04"] + x[,"n_12"] + x[,"n_14"])*log(11) + (x[,"n_03"] + x[,"n_13"])*log(41) + (x[,"n_05"] + x[,"n_11"])*log(5 - 4*r_r) + (x[,"n_06"] + x[,"n_10"])*log(pmax(1e-6,1 - r_r)) + (x[,"n_04"] + x[,"n_12"])*log(2 - r_r) + (x[,"n_00"] + x[,"n_16"])*log(pmax(1e-6,r_r)) + (x[,"n_02"] + x[,"n_14"])*log(1 + r_r) + (x[,"n_01"] + x[,"n_15"])*log(1 + 4*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.0_4.1 <- function(x){
logL_c <- function(r,n01,n02,n03,n04,n11,n12,n13,n14) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14)*(3*log(2) + log(5)) + (n01 + n14)*log(3 - 2*r) + (n02 + n13)*log(9 - 2*r) + (n04 + n11)*log(1 + 2*r) + (n03 + n12)*log(7 + 2*r)
return(L)}
interlogL_c <- function(n01,n02,n03,n04,n11,n12,n13,n14) {
optimize(logL_c,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"])
LOD_c <- -((x[,"n_04"] + x[,"n_11"])*log10(2)) - 3*(x[,"n_03"] + x[,"n_12"])*log10(2) - 3*(x[,"n_02"] + x[,"n_13"])*log10(2) - (x[,"n_01"] + x[,"n_14"])*log10(2) + (x[,"n_01"] + x[,"n_14"])*log10(3 - 2*r_c) + (x[,"n_02"] + x[,"n_13"])*log10(9 - 2*r_c) + (x[,"n_04"] + x[,"n_11"])*log10(1 + 2*r_c) + (x[,"n_03"] + x[,"n_12"])*log10(7 + 2*r_c)
logL_c <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*(3*log(2) + log(5)) + (x[,"n_01"] + x[,"n_14"])*log(3 - 2*r_c) + (x[,"n_02"] + x[,"n_13"])*log(9 - 2*r_c) + (x[,"n_04"] + x[,"n_11"])*log(1 + 2*r_c) + (x[,"n_03"] + x[,"n_12"])*log(7 + 2*r_c)
logL_r <- function(r,n01,n02,n03,n04,n11,n12,n13,n14) {
L <- (-n01 - 2*n02 - 2*n03 - n04 - n11 - 2*n12 - 2*n13 - n14)*log(2) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14)*log(5) + (n03 + n12)*log(5 - 2*r) + (n04 + n11)*log(pmax(1e-6,1 - r)) + (n01 + n14)*log(pmax(1e-6,r)) + (n02 + n13)*log(3 + 2*r)
return(L)}
interlogL_r <- function(n01,n02,n03,n04,n11,n12,n13,n14) {
optimize(logL_r,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"])
LOD_r <- (x[,"n_04"] + x[,"n_11"])*log10(2) - 2*(x[,"n_03"] + x[,"n_12"])*log10(2) - 2*(x[,"n_02"] + x[,"n_13"])*log10(2) + (x[,"n_01"] + x[,"n_14"])*log10(2) + (x[,"n_03"] + x[,"n_12"])*log10(5 - 2*r_r) + (x[,"n_04"] + x[,"n_11"])*log10(pmax(1e-6,1 - r_r)) + (x[,"n_01"] + x[,"n_14"])*log10(pmax(1e-6,r_r)) + (x[,"n_02"] + x[,"n_13"])*log10(3 + 2*r_r)
logL_r <- (-x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"])*log(5) + (x[,"n_03"] + x[,"n_12"])*log(5 - 2*r_r) + (x[,"n_04"] + x[,"n_11"])*log(pmax(1e-6,1 - r_r)) + (x[,"n_01"] + x[,"n_14"])*log(pmax(1e-6,r_r)) + (x[,"n_02"] + x[,"n_13"])*log(3 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_1.1 <- function(x){
logL_cc <- function(r,n00,n01,n02,n10,n12,n20,n21,n22,n11) {
L <- (-2*n00 - n01 - 2*n02 - n10 - n12 - 2*n20 - n21 - 2*n22)*log(2) + 2*(n00 + n22)*log(pmax(1e-6,1 - r)) + 2*(n02 + n20)*log(pmax(1e-6,r)) + (n01 + n10 + n12 + n21)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n11*log(1/2 - r + r^2)
return(L)}
interlogL_cc <- function(n00,n01,n02,n10,n12,n20,n21,n22,n11) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n10,n12,n20,n21,n22,n11, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_11"])
LOD_cc <- 2*x[,"n_11"]*log10(2) + 2*(x[,"n_02"] + x[,"n_20"])*log10(2) + 2*(x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(2) + 2*(x[,"n_00"] + x[,"n_22"])*log10(2) + 2*(x[,"n_00"] + x[,"n_22"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_02"] + x[,"n_20"])*log10(pmax(1e-6,r_cc)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + x[,"n_11"]*log10(1/2 - r_cc + r_cc^2)
logL_cc <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - x[,"n_10"] - x[,"n_12"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"])*log(2) + 2*(x[,"n_00"] + x[,"n_22"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_02"] + x[,"n_20"])*log(pmax(1e-6,r_cc)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + x[,"n_11"]*log(1/2 - r_cc + r_cc^2)
logL_cr <- function(r,n00,n01,n02,n10,n12,n20,n21,n22,n11) {
L <- 2*(-n00 - n01 - n02 - n10 - n12 - n20 - n21 - n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(5) + (n02 + n20)*(log(3 - r) + log(pmax(1e-6,r))) + (n00 + n22)*(log(pmax(1e-6,1 - r)) + log(2 + r)) + n11*log(1 + r - r^2) + (n01 + n10 + n12 + n21)*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_cr <- function(n00,n01,n02,n10,n12,n20,n21,n22,n11) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n10,n12,n20,n21,n22,n11, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_11"])
LOD_cr <- -(x[,"n_11"]*(-2*log10(2) + log10(5))) - (x[,"n_02"] + x[,"n_20"])*(-2*log10(2) + log10(5)) - (x[,"n_00"] + x[,"n_22"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(-log10(2) + log10(5)) + (x[,"n_02"] + x[,"n_20"])*(log10(3 - r_cr) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_cr)) + log10(2 + r_cr)) + x[,"n_11"]*log10(1 + r_cr - r_cr^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(3 - 2*r_cr + 2*r_cr^2)
logL_cr <- 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(5) + (x[,"n_02"] + x[,"n_20"])*(log(3 - r_cr) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_cr)) + log(2 + r_cr)) + x[,"n_11"]*log(1 + r_cr - r_cr^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(3 - 2*r_cr + 2*r_cr^2)
logL_rr <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22) {
L <- (-2*n00 - n01 - 2*n02 - n10 - n11 - n12 - 2*n20 - n21 - 2*n22)*log(2) + 2*(-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(5) + 2*(n02 + n20)*log(3 - r) + 2*(n00 + n22)*log(2 + r) + (n01 + n10 + n12 + n21)*(log(3 - r) + log(2 + r)) + n11*log(13 - 2*r + 2*r^2)
return(L)}
interlogL_rr <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"])
LOD_rr <- -2*(x[,"n_02"] + x[,"n_20"])*(-log10(2) + log10(5)) - 2*(x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(-log10(2) + log10(5)) - 2*(x[,"n_00"] + x[,"n_22"])*(-log10(2) + log10(5)) - x[,"n_11"]*(-log10(2) + 2*log10(5)) + 2*(x[,"n_02"] + x[,"n_20"])*log10(3 - r_rr) + 2*(x[,"n_00"] + x[,"n_22"])*log10(2 + r_rr) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(3 - r_rr) + log10(2 + r_rr)) + x[,"n_11"]*log10(13 - 2*r_rr + 2*r_rr^2)
logL_rr <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(5) + 2*(x[,"n_02"] + x[,"n_20"])*log(3 - r_rr) + 2*(x[,"n_00"] + x[,"n_22"])*log(2 + r_rr) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log(3 - r_rr) + log(2 + r_rr)) + x[,"n_11"]*log(13 - 2*r_rr + 2*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_1.2 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n11,n12,n20,n21,n22,n23,n10,n13) {
L <- (-n00 - 2*n01 - 2*n02 - n03 - 2*n11 - 2*n12 - n20 - 2*n21 - 2*n22 - n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(5) + 2*(n00 + n23)*log(pmax(1e-6,1 - r)) + 2*(n03 + n20)*log(pmax(1e-6,r)) + (n02 + n21)*(log(5 - 2*r) + log(pmax(1e-6,r))) + (n10 + n13)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n22)*log(3 - r - 2*r^2) + (n11 + n12)*log(5 - 4*r + 4*r^2)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n11,n12,n20,n21,n22,n23,n10,n13) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n11,n12,n20,n21,n22,n23,n10,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_10"],x[,"n_13"])
LOD_cc <- -2*(x[,"n_11"] + x[,"n_12"])*log10(2) + 2*(x[,"n_10"] + x[,"n_13"])*log10(2) + 2*(x[,"n_03"] + x[,"n_20"])*log10(2) - (x[,"n_02"] + x[,"n_21"])*log10(2) - (x[,"n_01"] + x[,"n_22"])*log10(2) + 2*(x[,"n_00"] + x[,"n_23"])*log10(2) + 2*(x[,"n_00"] + x[,"n_23"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_03"] + x[,"n_20"])*log10(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_21"])*(log10(5 - 2*r_cc) + log10(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_13"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_22"])*log10(3 - r_cc - 2*r_cc^2) + (x[,"n_11"] + x[,"n_12"])*log10(5 - 4*r_cc + 4*r_cc^2)
logL_cc <- (-x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - x[,"n_03"] - 2*x[,"n_11"] - 2*x[,"n_12"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(5) + 2*(x[,"n_00"] + x[,"n_23"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_03"] + x[,"n_20"])*log(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_21"])*(log(5 - 2*r_cc) + log(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_13"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_22"])*log(3 - r_cc - 2*r_cc^2) + (x[,"n_11"] + x[,"n_12"])*log(5 - 4*r_cc + 4*r_cc^2)
logL_cr <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(3*log(2) + log(5)) + (n03 + n20)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n00 + n23)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n11 + n12)*log(7 + 4*r - 4*r^2) + (n01 + n22)*log(6 - 5*r + 2*r^2) + (n02 + n21)*log(3 + r + 2*r^2) + (n10 + n13)*log(3 - 4*r + 4*r^2)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_cr <- -3*(x[,"n_11"] + x[,"n_12"])*log10(2) - (x[,"n_10"] + x[,"n_13"])*log10(2) - 2*(x[,"n_02"] + x[,"n_21"])*log10(2) - 2*(x[,"n_01"] + x[,"n_22"])*log10(2) + (x[,"n_03"] + x[,"n_20"])*(log10(3 - 2*r_cr) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cr)) + log10(1 + 2*r_cr)) + (x[,"n_11"] + x[,"n_12"])*log10(7 + 4*r_cr - 4*r_cr^2) + (x[,"n_01"] + x[,"n_22"])*log10(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_02"] + x[,"n_21"])*log10(3 + r_cr + 2*r_cr^2) + (x[,"n_10"] + x[,"n_13"])*log10(3 - 4*r_cr + 4*r_cr^2)
logL_cr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(3*log(2) + log(5)) + (x[,"n_03"] + x[,"n_20"])*(log(3 - 2*r_cr) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cr)) + log(1 + 2*r_cr)) + (x[,"n_11"] + x[,"n_12"])*log(7 + 4*r_cr - 4*r_cr^2) + (x[,"n_01"] + x[,"n_22"])*log(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_02"] + x[,"n_21"])*log(3 + r_cr + 2*r_cr^2) + (x[,"n_10"] + x[,"n_13"])*log(3 - 4*r_cr + 4*r_cr^2)
logL_rc <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
L <- (-n00 - 2*n01 - 2*n02 - n03 - n10 - 2*n11 - 2*n12 - n13 - n20 - 2*n21 - 2*n22 - n23)*log(2) + 2*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(5) + (n03 + n20)*(log(3 - r) + log(pmax(1e-6,r))) + (n00 + n23)*(log(pmax(1e-6,1 - r)) + log(2 + r)) + (n11 + n12)*log(19 + 4*r - 4*r^2) + (n01 + n22)*log(12 - 5*r + 2*r^2) + (n10 + n13)*log(3 - 2*r + 2*r^2) + (n02 + n21)*log(9 + r + 2*r^2)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_rc <- -((x[,"n_03"] + x[,"n_20"])*(-2*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_23"])*(-2*log10(2) + log10(5)) - (x[,"n_10"] + x[,"n_13"])*(-log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_21"])*(log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_22"])*(log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_12"])*(2*log10(2) + log10(5)) + (x[,"n_03"] + x[,"n_20"])*(log10(3 - r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_rc)) + log10(2 + r_rc)) + (x[,"n_11"] + x[,"n_12"])*log10(19 + 4*r_rc - 4*r_rc^2) + (x[,"n_01"] + x[,"n_22"])*log10(12 - 5*r_rc + 2*r_rc^2) + (x[,"n_10"] + x[,"n_13"])*log10(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_02"] + x[,"n_21"])*log10(9 + r_rc + 2*r_rc^2)
logL_rc <- (-x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - x[,"n_23"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(5) + (x[,"n_03"] + x[,"n_20"])*(log(3 - r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_rc)) + log(2 + r_rc)) + (x[,"n_11"] + x[,"n_12"])*log(19 + 4*r_rc - 4*r_rc^2) + (x[,"n_01"] + x[,"n_22"])*log(12 - 5*r_rc + 2*r_rc^2) + (x[,"n_10"] + x[,"n_13"])*log(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_02"] + x[,"n_21"])*log(9 + r_rc + 2*r_rc^2)
logL_rr <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(3*log(2) + 2*log(5)) + (n03 + n20)*(log(3 - 2*r) + log(3 - r)) + (n00 + n23)*(log(2 + r) + log(1 + 2*r)) + (n10 + n13)*log(9 + 4*r - 4*r^2) + (n02 + n21)*log(24 - 7*r - 2*r^2) + (n01 + n22)*log(15 + 11*r - 2*r^2) + (n11 + n12)*log(41 - 4*r + 4*r^2)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_rr <- -((x[,"n_03"] + x[,"n_20"])*log10(5)) - (x[,"n_00"] + x[,"n_23"])*log10(5) - (x[,"n_10"] + x[,"n_13"])*(log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_21"])*(2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_22"])*(2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_12"])*(3*log10(2) + log10(5)) + (x[,"n_03"] + x[,"n_20"])*(log10(3 - 2*r_rr) + log10(3 - r_rr)) + (x[,"n_00"] + x[,"n_23"])*(log10(2 + r_rr) + log10(1 + 2*r_rr)) + (x[,"n_10"] + x[,"n_13"])*log10(9 + 4*r_rr - 4*r_rr^2) + (x[,"n_02"] + x[,"n_21"])*log10(24 - 7*r_rr - 2*r_rr^2) + (x[,"n_01"] + x[,"n_22"])*log10(15 + 11*r_rr - 2*r_rr^2) + (x[,"n_11"] + x[,"n_12"])*log10(41 - 4*r_rr + 4*r_rr^2)
logL_rr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(3*log(2) + 2*log(5)) + (x[,"n_03"] + x[,"n_20"])*(log(3 - 2*r_rr) + log(3 - r_rr)) + (x[,"n_00"] + x[,"n_23"])*(log(2 + r_rr) + log(1 + 2*r_rr)) + (x[,"n_10"] + x[,"n_13"])*log(9 + 4*r_rr - 4*r_rr^2) + (x[,"n_02"] + x[,"n_21"])*log(24 - 7*r_rr - 2*r_rr^2) + (x[,"n_01"] + x[,"n_22"])*log(15 + 11*r_rr - 2*r_rr^2) + (x[,"n_11"] + x[,"n_12"])*log(41 - 4*r_rr + 4*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_1.3 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-3*n00 - 2*n01 - 3*n02 - 2*n03 - 3*n04 - 2*n10 - n11 - n12 - n13 - 2*n14 - 3*n20 - 2*n21 - 3*n22 - 2*n23 - 3*n24)*log(2) + (n02 + n12 + n22)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + 2*(n00 + n24)*log(pmax(1e-6,1 - r)) + 2*(n04 + n20)*log(pmax(1e-6,r)) + (n10 + n14)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n21)*(log(pmax(1e-6,r)) + log(2 + r)) + (n02 + n22)*log(1 + 2*r - 2*r^2) + (n11 + n13)*log(1 + r - r^2) + (n01 + n23)*log(3 - 4*r + r^2) + n12*log(1 - r + r^2)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_cc <- 2*(x[,"n_10"] + x[,"n_14"])*log10(2) + 2*(x[,"n_04"] + x[,"n_20"])*log10(2) + 2*(x[,"n_00"] + x[,"n_24"])*log10(2) + x[,"n_12"]*(2*log10(2) - log10(3)) - (x[,"n_02"] + x[,"n_22"])*(-log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_13"])*(-2*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_21"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_23"])*(-2*log10(2) + log10(5)) + 2*(x[,"n_00"] + x[,"n_24"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_20"])*log10(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_14"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_02"] + x[,"n_22"])*log10(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_11"] + x[,"n_13"])*log10(1 + r_cc - r_cc^2) + (x[,"n_01"] + x[,"n_23"])*log10(3 - 4*r_cc + r_cc^2) + x[,"n_12"]*log10(1 - r_cc + r_cc^2)
logL_cc <- (-3*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_20"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_22"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + 2*(x[,"n_00"] + x[,"n_24"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_20"])*log(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_14"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_02"] + x[,"n_22"])*log(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_11"] + x[,"n_13"])*log(1 + r_cc - r_cc^2) + (x[,"n_01"] + x[,"n_23"])*log(3 - 4*r_cc + r_cc^2) + x[,"n_12"]*log(1 - r_cc + r_cc^2)
logL_cr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-3*n00 - 3*n01 - 2*n02 - 3*n03 - 3*n04 - 3*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n14 - 3*n20 - 3*n21 - 2*n22 - 3*n23 - 3*n24)*log(2) + (n02 + n12 + n22)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + (n00 + n04 + n20 + n24)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n23)*log(3 - 2*r^2) + n12*log(1 + 2*r - 2*r^2) + (n03 + n21)*log(1 + 4*r - 2*r^2) + (n02 + n22)*log(1 - r + r^2) + (n10 + n14)*log(1 - 2*r + 2*r^2) + (n11 + n13)*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_cr <- (x[,"n_10"] + x[,"n_14"])*log10(2) + 2*(x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*log10(2) + (x[,"n_02"] + x[,"n_22"])*(2*log10(2) - log10(3)) - x[,"n_12"]*(-log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_13"])*(-log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(5)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_23"])*log10(3 - 2*r_cr^2) + x[,"n_12"]*log10(1 + 2*r_cr - 2*r_cr^2) + (x[,"n_03"] + x[,"n_21"])*log10(1 + 4*r_cr - 2*r_cr^2) + (x[,"n_02"] + x[,"n_22"])*log10(1 - r_cr + r_cr^2) + (x[,"n_10"] + x[,"n_14"])*log10(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_11"] + x[,"n_13"])*log10(3 - 2*r_cr + 2*r_cr^2)
logL_cr <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 2*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_22"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_23"])*log(3 - 2*r_cr^2) + x[,"n_12"]*log(1 + 2*r_cr - 2*r_cr^2) + (x[,"n_03"] + x[,"n_21"])*log(1 + 4*r_cr - 2*r_cr^2) + (x[,"n_02"] + x[,"n_22"])*log(1 - r_cr + r_cr^2) + (x[,"n_10"] + x[,"n_14"])*log(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_11"] + x[,"n_13"])*log(3 - 2*r_cr + 2*r_cr^2)
logL_rc <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-3*n00 - 3*n01 - 2*n02 - 3*n03 - 3*n04 - 3*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n14 - 3*n20 - 3*n21 - 2*n22 - 3*n23 - 3*n24)*log(2) + (n02 + n12 + n22)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + (n04 + n20)*(log(3 - r) + log(pmax(1e-6,r))) + (n00 + n24)*(log(pmax(1e-6,1 - r)) + log(2 + r)) + (n01 + n23)*log(15 - 4*r - 2*r^2) + n12*log(7 + 2*r - 2*r^2) + (n03 + n21)*log(9 + 8*r - 2*r^2) + (n02 + n22)*log(4 - r + r^2) + (n10 + n14)*log(3 - 2*r + 2*r^2) + (n11 + n13)*log(13 - 2*r + 2*r^2)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_rc <- -((x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_22"])*(-2*log10(2) + log10(3) + log10(5)) - x[,"n_12"]*(-log10(2) + log10(3) + log10(5)) - (x[,"n_11"] + x[,"n_13"])*(-log10(2) + 2*log10(5)) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + 2*log10(5)) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + 2*log10(5)) + (x[,"n_04"] + x[,"n_20"])*(log10(3 - r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_24"])*(log10(pmax(1e-6,1 - r_rc)) + log10(2 + r_rc)) + (x[,"n_01"] + x[,"n_23"])*log10(15 - 4*r_rc - 2*r_rc^2) + x[,"n_12"]*log10(7 + 2*r_rc - 2*r_rc^2) + (x[,"n_03"] + x[,"n_21"])*log10(9 + 8*r_rc - 2*r_rc^2) + (x[,"n_02"] + x[,"n_22"])*log10(4 - r_rc + r_rc^2) + (x[,"n_10"] + x[,"n_14"])*log10(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_11"] + x[,"n_13"])*log10(13 - 2*r_rc + 2*r_rc^2)
logL_rc <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 2*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_22"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_04"] + x[,"n_20"])*(log(3 - r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_24"])*(log(pmax(1e-6,1 - r_rc)) + log(2 + r_rc)) + (x[,"n_01"] + x[,"n_23"])*log(15 - 4*r_rc - 2*r_rc^2) + x[,"n_12"]*log(7 + 2*r_rc - 2*r_rc^2) + (x[,"n_03"] + x[,"n_21"])*log(9 + 8*r_rc - 2*r_rc^2) + (x[,"n_02"] + x[,"n_22"])*log(4 - r_rc + r_rc^2) + (x[,"n_10"] + x[,"n_14"])*log(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_11"] + x[,"n_13"])*log(13 - 2*r_rc + 2*r_rc^2)
logL_rr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-3*n00 - 2*n01 - 3*n02 - 2*n03 - 3*n04 - 2*n10 - n11 - n12 - n13 - 2*n14 - 3*n20 - 2*n21 - 3*n22 - 2*n23 - 3*n24)*log(2) + (n02 + n12 + n22)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + (n04 + n20)*(log(pmax(1e-6,1 - r)) + log(3 - r)) + (n00 + n24)*(log(pmax(1e-6,r)) + log(2 + r)) + (n02 + n22)*log(7 + 2*r - 2*r^2) + (n10 + n14)*log(1 + r - r^2) + (n11 + n13)*log(6 + r - r^2) + (n03 + n21)*log(10 - 8*r + r^2) + n12*log(4 - r + r^2) + (n01 + n23)*log(3 + 6*r + r^2)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_rr <- -((x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(5))) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(5)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - x[,"n_12"]*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_22"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_11"] + x[,"n_13"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_03"] + x[,"n_21"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_01"] + x[,"n_23"])*(-2*log10(2) + 2*log10(5)) + (x[,"n_04"] + x[,"n_20"])*(log10(pmax(1e-6,1 - r_rr)) + log10(3 - r_rr)) + (x[,"n_00"] + x[,"n_24"])*(log10(pmax(1e-6,r_rr)) + log10(2 + r_rr)) + (x[,"n_02"] + x[,"n_22"])*log10(7 + 2*r_rr - 2*r_rr^2) + (x[,"n_10"] + x[,"n_14"])*log10(1 + r_rr - r_rr^2) + (x[,"n_11"] + x[,"n_13"])*log10(6 + r_rr - r_rr^2) + (x[,"n_03"] + x[,"n_21"])*log10(10 - 8*r_rr + r_rr^2) + x[,"n_12"]*log10(4 - r_rr + r_rr^2) + (x[,"n_01"] + x[,"n_23"])*log10(3 + 6*r_rr + r_rr^2)
logL_rr <- (-3*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_20"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_22"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_04"] + x[,"n_20"])*(log(pmax(1e-6,1 - r_rr)) + log(3 - r_rr)) + (x[,"n_00"] + x[,"n_24"])*(log(pmax(1e-6,r_rr)) + log(2 + r_rr)) + (x[,"n_02"] + x[,"n_22"])*log(7 + 2*r_rr - 2*r_rr^2) + (x[,"n_10"] + x[,"n_14"])*log(1 + r_rr - r_rr^2) + (x[,"n_11"] + x[,"n_13"])*log(6 + r_rr - r_rr^2) + (x[,"n_03"] + x[,"n_21"])*log(10 - 8*r_rr + r_rr^2) + x[,"n_12"]*log(4 - r_rr + r_rr^2) + (x[,"n_01"] + x[,"n_23"])*log(3 + 6*r_rr + r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_1.4 <- function(x){
logL_cc <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24)*(3*log(2) + log(5)) + (n01 + n24)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n04 + n21)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n11 + n14)*log(1 + 4*r - 4*r^2) + (n02 + n23)*log(6 - 3*r - 2*r^2) + (n03 + n22)*log(1 + 7*r - 2*r^2) + (n12 + n13)*log(9 - 4*r + 4*r^2)
return(L)}
interlogL_cc <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
optimize(logL_cc,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_cc <- -3*(x[,"n_12"] + x[,"n_13"])*log10(2) - (x[,"n_11"] + x[,"n_14"])*log10(2) - 2*(x[,"n_03"] + x[,"n_22"])*log10(2) - 2*(x[,"n_02"] + x[,"n_23"])*log10(2) + (x[,"n_01"] + x[,"n_24"])*(log10(3 - 2*r_cc) + log10(pmax(1e-6,1 - r_cc))) + (x[,"n_04"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(1 + 2*r_cc)) + (x[,"n_11"] + x[,"n_14"])*log10(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_02"] + x[,"n_23"])*log10(6 - 3*r_cc - 2*r_cc^2) + (x[,"n_03"] + x[,"n_22"])*log10(1 + 7*r_cc - 2*r_cc^2) + (x[,"n_12"] + x[,"n_13"])*log10(9 - 4*r_cc + 4*r_cc^2)
logL_cc <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*(3*log(2) + log(5)) + (x[,"n_01"] + x[,"n_24"])*(log(3 - 2*r_cc) + log(pmax(1e-6,1 - r_cc))) + (x[,"n_04"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(1 + 2*r_cc)) + (x[,"n_11"] + x[,"n_14"])*log(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_02"] + x[,"n_23"])*log(6 - 3*r_cc - 2*r_cc^2) + (x[,"n_03"] + x[,"n_22"])*log(1 + 7*r_cc - 2*r_cc^2) + (x[,"n_12"] + x[,"n_13"])*log(9 - 4*r_cc + 4*r_cc^2)
logL_cr <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
L <- (-n01 - 2*n02 - 2*n03 - n04 - n11 - 2*n12 - 2*n13 - n14 - n21 - 2*n22 - 2*n23 - n24)*log(2) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24)*log(5) + (n01 + n04 + n21 + n24)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n13)*log(3 + 4*r - 4*r^2) + (n02 + n23)*log(3 - 3*r + 2*r^2) + (n11 + n14)*log(1 - 2*r + 2*r^2) + (n03 + n22)*log(2 - r + 2*r^2)
return(L)}
interlogL_cr <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
optimize(logL_cr,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_cr <- -2*(x[,"n_12"] + x[,"n_13"])*log10(2) + (x[,"n_11"] + x[,"n_14"])*log10(2) - (x[,"n_03"] + x[,"n_22"])*log10(2) - (x[,"n_02"] + x[,"n_23"])*log10(2) + 2*(x[,"n_01"] + x[,"n_04"] + x[,"n_21"] + x[,"n_24"])*log10(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_21"] + x[,"n_24"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_12"] + x[,"n_13"])*log10(3 + 4*r_cr - 4*r_cr^2) + (x[,"n_02"] + x[,"n_23"])*log10(3 - 3*r_cr + 2*r_cr^2) + (x[,"n_11"] + x[,"n_14"])*log10(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_03"] + x[,"n_22"])*log10(2 - r_cr + 2*r_cr^2)
logL_cr <- (-x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_01"] + x[,"n_04"] + x[,"n_21"] + x[,"n_24"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_12"] + x[,"n_13"])*log(3 + 4*r_cr - 4*r_cr^2) + (x[,"n_02"] + x[,"n_23"])*log(3 - 3*r_cr + 2*r_cr^2) + (x[,"n_11"] + x[,"n_14"])*log(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_03"] + x[,"n_22"])*log(2 - r_cr + 2*r_cr^2)
logL_rc <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24)*(3*log(2) + 2*log(5)) + (n01 + n24)*(log(3 - 2*r) + log(2 + r)) + (n04 + n21)*(log(3 - r) + log(1 + 2*r)) + (n12 + n13)*log(39 + 4*r - 4*r^2) + (n02 + n23)*log(21 - 3*r + 2*r^2) + (n03 + n22)*log(20 - r + 2*r^2) + (n11 + n14)*log(11 - 4*r + 4*r^2)
return(L)}
interlogL_rc <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
optimize(logL_rc,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_rc <- -((x[,"n_04"] + x[,"n_21"])*log10(5)) - (x[,"n_01"] + x[,"n_24"])*log10(5) - (x[,"n_11"] + x[,"n_14"])*(log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_22"])*(2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_23"])*(2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_13"])*(3*log10(2) + log10(5)) + (x[,"n_01"] + x[,"n_24"])*(log10(3 - 2*r_rc) + log10(2 + r_rc)) + (x[,"n_04"] + x[,"n_21"])*(log10(3 - r_rc) + log10(1 + 2*r_rc)) + (x[,"n_12"] + x[,"n_13"])*log10(39 + 4*r_rc - 4*r_rc^2) + (x[,"n_02"] + x[,"n_23"])*log10(21 - 3*r_rc + 2*r_rc^2) + (x[,"n_03"] + x[,"n_22"])*log10(20 - r_rc + 2*r_rc^2) + (x[,"n_11"] + x[,"n_14"])*log10(11 - 4*r_rc + 4*r_rc^2)
logL_rc <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*(3*log(2) + 2*log(5)) + (x[,"n_01"] + x[,"n_24"])*(log(3 - 2*r_rc) + log(2 + r_rc)) + (x[,"n_04"] + x[,"n_21"])*(log(3 - r_rc) + log(1 + 2*r_rc)) + (x[,"n_12"] + x[,"n_13"])*log(39 + 4*r_rc - 4*r_rc^2) + (x[,"n_02"] + x[,"n_23"])*log(21 - 3*r_rc + 2*r_rc^2) + (x[,"n_03"] + x[,"n_22"])*log(20 - r_rc + 2*r_rc^2) + (x[,"n_11"] + x[,"n_14"])*log(11 - 4*r_rc + 4*r_rc^2)
logL_rr <- function(r,n01,n02,n03,n04,n12,n13,n21,n22,n23,n24,n11,n14) {
L <- (-n01 - 2*n02 - 2*n03 - n04 - 2*n12 - 2*n13 - n21 - 2*n22 - 2*n23 - n24)*log(2) + 2*(-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24)*log(5) + (n04 + n21)*(log(pmax(1e-6,1 - r)) + log(3 - r)) + (n01 + n24)*(log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n22)*log(13 - 5*r - 2*r^2) + (n02 + n23)*log(6 + 9*r - 2*r^2) + (n11 + n14)*log(1 + r - r^2) + (n12 + n13)*log(21 - 4*r + 4*r^2)
return(L)}
interlogL_rr <- function(n01,n02,n03,n04,n12,n13,n21,n22,n23,n24,n11,n14) {
optimize(logL_rr,c(0,0.5), n01,n02,n03,n04,n12,n13,n21,n22,n23,n24,n11,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_11"],x[,"n_14"])
LOD_rr <- -((x[,"n_11"] + x[,"n_14"])*(-2*log10(2) + log10(5))) - (x[,"n_04"] + x[,"n_21"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_22"])*(log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_23"])*(log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_13"])*(2*log10(2) + log10(5)) + (x[,"n_04"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr)) + log10(3 - r_rr)) + (x[,"n_01"] + x[,"n_24"])*(log10(pmax(1e-6,r_rr)) + log10(2 + r_rr)) + (x[,"n_03"] + x[,"n_22"])*log10(13 - 5*r_rr - 2*r_rr^2) + (x[,"n_02"] + x[,"n_23"])*log10(6 + 9*r_rr - 2*r_rr^2) + (x[,"n_11"] + x[,"n_14"])*log10(1 + r_rr - r_rr^2) + (x[,"n_12"] + x[,"n_13"])*log10(21 - 4*r_rr + 4*r_rr^2)
logL_rr <- (-x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - x[,"n_04"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_04"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr)) + log(3 - r_rr)) + (x[,"n_01"] + x[,"n_24"])*(log(pmax(1e-6,r_rr)) + log(2 + r_rr)) + (x[,"n_03"] + x[,"n_22"])*log(13 - 5*r_rr - 2*r_rr^2) + (x[,"n_02"] + x[,"n_23"])*log(6 + 9*r_rr - 2*r_rr^2) + (x[,"n_11"] + x[,"n_14"])*log(1 + r_rr - r_rr^2) + (x[,"n_12"] + x[,"n_13"])*log(21 - 4*r_rr + 4*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_1.5 <- function(x){
logL_cc <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24) {
L <- (-2*n02 - n03 - 2*n04 - n12 - n13 - n14 - 2*n22 - n23 - 2*n24)*log(2) + (-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24)*log(5) + (n02 + n24)*(log(pmax(1e-6,1 - r)) + log(3 - r)) + (n04 + n22)*(log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n12 + n14 + n23)*log(1 + r - r^2) + n13*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_cc <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24) {
optimize(logL_cc,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_cc <- -((x[,"n_04"] + x[,"n_22"])*(-2*log10(2) + log10(5))) - (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - x[,"n_13"]*(-log10(2) + log10(5)) + (x[,"n_02"] + x[,"n_24"])*(log10(pmax(1e-6,1 - r_cc)) + log10(3 - r_cc)) + (x[,"n_04"] + x[,"n_22"])*(log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*log10(1 + r_cc - r_cc^2) + x[,"n_13"]*log10(3 - 2*r_cc + 2*r_cc^2)
logL_cc <- (-2*x[,"n_02"] - x[,"n_03"] - 2*x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - 2*x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"])*log(2) + (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_02"] + x[,"n_24"])*(log(pmax(1e-6,1 - r_cc)) + log(3 - r_cc)) + (x[,"n_04"] + x[,"n_22"])*(log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*log(1 + r_cc - r_cc^2) + x[,"n_13"]*log(3 - 2*r_cc + 2*r_cc^2)
logL_cr <- function(r,n02,n03,n04,n12,n14,n22,n23,n24,n13) {
L <- 2*(-n02 - n03 - n04 - n12 - n14 - n22 - n23 - n24)*log(2) + (n02 + n04 + n13 + n22 + n24)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n12 + n14 + n23)*log(1 - 2*r + 2*r^2)
return(L)}
interlogL_cr <- function(n02,n03,n04,n12,n14,n22,n23,n24,n13) {
optimize(logL_cr,c(0,0.5), n02,n03,n04,n12,n14,n22,n23,n24,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_13"])
LOD_cr <- (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*log10(2) + 2*(x[,"n_02"] + x[,"n_04"] + x[,"n_13"] + x[,"n_22"] + x[,"n_24"])*log10(2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_13"] + x[,"n_22"] + x[,"n_24"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*log10(1 - 2*r_cr + 2*r_cr^2)
logL_cr <- 2*(-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_13"] + x[,"n_22"] + x[,"n_24"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*log(1 - 2*r_cr + 2*r_cr^2)
logL_rc <- function(r,n02,n03,n04,n12,n14,n22,n23,n24,n13) {
L <- 2*(-n02 - n03 - n04 - n12 - n14 - n22 - n23 - n24)*log(2) + 2*(-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24)*log(5) + (n02 + n04 + n13 + n22 + n24)*(log(3 - r) + log(2 + r)) + (n03 + n12 + n14 + n23)*log(13 - 2*r + 2*r^2)
return(L)}
interlogL_rc <- function(n02,n03,n04,n12,n14,n22,n23,n24,n13) {
optimize(logL_rc,c(0,0.5), n02,n03,n04,n12,n14,n22,n23,n24,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_13"])
LOD_rc <- -2*(x[,"n_02"] + x[,"n_04"] + x[,"n_13"] + x[,"n_22"] + x[,"n_24"])*(-log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*(-log10(2) + 2*log10(5)) + (x[,"n_02"] + x[,"n_04"] + x[,"n_13"] + x[,"n_22"] + x[,"n_24"])*(log10(3 - r_rc) + log10(2 + r_rc)) + (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*log10(13 - 2*r_rc + 2*r_rc^2)
logL_rc <- 2*(-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(2) + 2*(-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_02"] + x[,"n_04"] + x[,"n_13"] + x[,"n_22"] + x[,"n_24"])*(log(3 - r_rc) + log(2 + r_rc)) + (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*log(13 - 2*r_rc + 2*r_rc^2)
logL_rr <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24) {
L <- (-2*n02 - n03 - 2*n04 - n12 - n13 - n14 - 2*n22 - n23 - 2*n24)*log(2) + (-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24)*log(5) + (n04 + n22)*(log(pmax(1e-6,1 - r)) + log(3 - r)) + (n02 + n24)*(log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n12 + n14 + n23)*log(1 + r - r^2) + n13*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_rr <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24) {
optimize(logL_rr,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_rr <- -((x[,"n_04"] + x[,"n_22"])*(-2*log10(2) + log10(5))) - (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - x[,"n_13"]*(-log10(2) + log10(5)) + (x[,"n_04"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_rr)) + log10(3 - r_rr)) + (x[,"n_02"] + x[,"n_24"])*(log10(pmax(1e-6,r_rr)) + log10(2 + r_rr)) + (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*log10(1 + r_rr - r_rr^2) + x[,"n_13"]*log10(3 - 2*r_rr + 2*r_rr^2)
logL_rr <- (-2*x[,"n_02"] - x[,"n_03"] - 2*x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - 2*x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"])*log(2) + (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_04"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_rr)) + log(3 - r_rr)) + (x[,"n_02"] + x[,"n_24"])*(log(pmax(1e-6,r_rr)) + log(2 + r_rr)) + (x[,"n_03"] + x[,"n_12"] + x[,"n_14"] + x[,"n_23"])*log(1 + r_rr - r_rr^2) + x[,"n_13"]*log(3 - 2*r_rr + 2*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_2.0 <- function(x){
logL_c <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22) {
L <- (-n00 - 2*n01 - n02 - n10 - n11 - n12 - n20 - 2*n21 - n22)*log(2) + (n01 + n11 + n21)*log(3) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(5) + (n00 + n22)*log(pmax(1e-6,1 - r)) + (n02 + n20)*log(pmax(1e-6,r))
return(L)}
interlogL_c <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"])
LOD_c <- (x[,"n_02"] + x[,"n_20"])*log10(2) + (x[,"n_00"] + x[,"n_22"])*log10(2) + (x[,"n_00"] + x[,"n_22"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_20"])*log10(pmax(1e-6,r_c))
logL_c <- (-x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"])*log(2) + (x[,"n_01"] + x[,"n_11"] + x[,"n_21"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(5) + (x[,"n_00"] + x[,"n_22"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_20"])*log(pmax(1e-6,r_c))
logL_r <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22) {
L <- (-3*n00 - 2*n01 - 3*n02 - n10 - n11 - n12 - 3*n20 - 2*n21 - 3*n22)*log(2) + (n01 + n11 + n21)*log(3) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(5) + (n02 + n20)*log(3 - 2*r) + (n00 + n22)*log(1 + 2*r)
return(L)}
interlogL_r <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"])
LOD_r <- -((x[,"n_02"] + x[,"n_20"])*log10(2)) - (x[,"n_00"] + x[,"n_22"])*log10(2) + (x[,"n_02"] + x[,"n_20"])*log10(3 - 2*r_r) + (x[,"n_00"] + x[,"n_22"])*log10(1 + 2*r_r)
logL_r <- (-3*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - 3*x[,"n_20"] - 2*x[,"n_21"] - 3*x[,"n_22"])*log(2) + (x[,"n_01"] + x[,"n_11"] + x[,"n_21"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(5) + (x[,"n_02"] + x[,"n_20"])*log(3 - 2*r_r) + (x[,"n_00"] + x[,"n_22"])*log(1 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_2.2 <- function(x){
logL_cc <- function(r,n02,n10,n12,n14,n22,n01,n03,n11,n13,n21,n23,n00,n04,n20,n24) {
L <- (-2*n02 + n10 - n12 + n14 - 2*n22)*log(2) + (n01 + n03 + n11 + n13 + n21 + n23)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + (n01 + n23)*log(pmax(1e-6,1 - r)) + 2*(n00 + n24)*log(pmax(1e-6,1 - r)) + 2*(n04 + n20)*log(pmax(1e-6,r)) + (n03 + n21)*log(pmax(1e-6,r)) + (n10 + n14)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n22)*log(9 + 8*r - 8*r^2) + n12*log(13 - 8*r + 8*r^2)
return(L)}
interlogL_cc <- function(n02,n10,n12,n14,n22,n01,n03,n11,n13,n21,n23,n00,n04,n20,n24) {
optimize(logL_cc,c(0,0.5), n02,n10,n12,n14,n22,n01,n03,n11,n13,n21,n23,n00,n04,n20,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_02"],x[,"n_10"],x[,"n_12"],x[,"n_14"],x[,"n_22"],x[,"n_01"],x[,"n_03"],x[,"n_11"],x[,"n_13"],x[,"n_21"],x[,"n_23"],x[,"n_00"],x[,"n_04"],x[,"n_20"],x[,"n_24"])
LOD_cc <- 2*(x[,"n_10"] + x[,"n_14"])*log10(2) + 2*(x[,"n_04"] + x[,"n_20"])*log10(2) + (x[,"n_03"] + x[,"n_21"])*log10(2) + (x[,"n_01"] + x[,"n_23"])*log10(2) + 2*(x[,"n_00"] + x[,"n_24"])*log10(2) - x[,"n_12"]*log10(11) - (x[,"n_02"] + x[,"n_22"])*log10(11) + (x[,"n_01"] + x[,"n_23"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_00"] + x[,"n_24"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_20"])*log10(pmax(1e-6,r_cc)) + (x[,"n_03"] + x[,"n_21"])*log10(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_14"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_02"] + x[,"n_22"])*log10(9 + 8*r_cc - 8*r_cc^2) + x[,"n_12"]*log10(13 - 8*r_cc + 8*r_cc^2)
logL_cc <- (-2*x[,"n_02"] + x[,"n_10"] - x[,"n_12"] + x[,"n_14"] - 2*x[,"n_22"])*log(2) + (x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"] + x[,"n_21"] + x[,"n_23"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_01"] + x[,"n_23"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_00"] + x[,"n_24"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_20"])*log(pmax(1e-6,r_cc)) + (x[,"n_03"] + x[,"n_21"])*log(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_14"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_02"] + x[,"n_22"])*log(9 + 8*r_cc - 8*r_cc^2) + x[,"n_12"]*log(13 - 8*r_cc + 8*r_cc^2)
logL_cr <- function(r,n00,n01,n03,n04,n10,n14,n20,n21,n23,n24,n11,n13,n02,n12,n22) {
L <- (-2*n00 - 3*n01 - 3*n03 - 2*n04 - 2*n10 - 2*n14 - 2*n20 - 3*n21 - 3*n23 - 2*n24)*log(2) + (n01 + n03 + n11 + n13 + n21 + n23)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + (n01 + n23)*log(5 - 2*r) + (n04 + n20)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n00 + n24)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n03 + n21)*log(3 + 2*r) + n12*log(5 + 2*r - 2*r^2) + (n02 + n22)*log(3 - r + r^2) + (n10 + n14)*log(3 - 4*r + 4*r^2)
return(L)}
interlogL_cr <- function(n00,n01,n03,n04,n10,n14,n20,n21,n23,n24,n11,n13,n02,n12,n22) {
optimize(logL_cr,c(0,0.5), n00,n01,n03,n04,n10,n14,n20,n21,n23,n24,n11,n13,n02,n12,n22, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_23"],x[,"n_24"],x[,"n_11"],x[,"n_13"],x[,"n_02"],x[,"n_12"],x[,"n_22"])
LOD_cr <- -((x[,"n_10"] + x[,"n_14"])*log10(2)) - 2*(x[,"n_03"] + x[,"n_21"])*log10(2) - 2*(x[,"n_01"] + x[,"n_23"])*log10(2) - (x[,"n_02"] + x[,"n_22"])*(-2*log10(2) + log10(11)) - x[,"n_12"]*(-log10(2) + log10(11)) + (x[,"n_01"] + x[,"n_23"])*log10(5 - 2*r_cr) + (x[,"n_04"] + x[,"n_20"])*(log10(3 - 2*r_cr) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_24"])*(log10(pmax(1e-6,1 - r_cr)) + log10(1 + 2*r_cr)) + (x[,"n_03"] + x[,"n_21"])*log10(3 + 2*r_cr) + x[,"n_12"]*log10(5 + 2*r_cr - 2*r_cr^2) + (x[,"n_02"] + x[,"n_22"])*log10(3 - r_cr + r_cr^2) + (x[,"n_10"] + x[,"n_14"])*log10(3 - 4*r_cr + 4*r_cr^2)
logL_cr <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_23"] - 2*x[,"n_24"])*log(2) + (x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"] + x[,"n_21"] + x[,"n_23"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_01"] + x[,"n_23"])*log(5 - 2*r_cr) + (x[,"n_04"] + x[,"n_20"])*(log(3 - 2*r_cr) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_24"])*(log(pmax(1e-6,1 - r_cr)) + log(1 + 2*r_cr)) + (x[,"n_03"] + x[,"n_21"])*log(3 + 2*r_cr) + x[,"n_12"]*log(5 + 2*r_cr - 2*r_cr^2) + (x[,"n_02"] + x[,"n_22"])*log(3 - r_cr + r_cr^2) + (x[,"n_10"] + x[,"n_14"])*log(3 - 4*r_cr + 4*r_cr^2)
logL_rr <- function(r,n00,n01,n02,n03,n04,n10,n12,n14,n20,n21,n22,n23,n24,n11,n13) {
L <- (-4*n00 - 2*n01 - 3*n02 - 2*n03 - 4*n04 - 3*n10 - 2*n12 - 3*n14 - 4*n20 - 2*n21 - 3*n22 - 2*n23 - 4*n24)*log(2) + (n01 + n03 + n11 + n13 + n21 + n23)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + 2*(n04 + n20)*log(3 - 2*r) + (n03 + n21)*log(3 - 2*r) + (n01 + n23)*log(1 + 2*r) + 2*(n00 + n24)*log(1 + 2*r) + (n10 + n14)*(log(3 - 2*r) + log(1 + 2*r)) + (n02 + n22)*log(21 + 4*r - 4*r^2) + n12*log(23 - 4*r + 4*r^2)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n10,n12,n14,n20,n21,n22,n23,n24,n11,n13) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n10,n12,n14,n20,n21,n22,n23,n24,n11,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_12"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_11"],x[,"n_13"])
LOD_rr <- -2*(x[,"n_10"] + x[,"n_14"])*log10(2) - 2*(x[,"n_04"] + x[,"n_20"])*log10(2) - (x[,"n_03"] + x[,"n_21"])*log10(2) - (x[,"n_01"] + x[,"n_23"])*log10(2) - 2*(x[,"n_00"] + x[,"n_24"])*log10(2) - x[,"n_12"]*(log10(2) + log10(11)) - (x[,"n_02"] + x[,"n_22"])*(log10(2) + log10(11)) + 2*(x[,"n_04"] + x[,"n_20"])*log10(3 - 2*r_rr) + (x[,"n_03"] + x[,"n_21"])*log10(3 - 2*r_rr) + (x[,"n_01"] + x[,"n_23"])*log10(1 + 2*r_rr) + 2*(x[,"n_00"] + x[,"n_24"])*log10(1 + 2*r_rr) + (x[,"n_10"] + x[,"n_14"])*(log10(3 - 2*r_rr) + log10(1 + 2*r_rr)) + (x[,"n_02"] + x[,"n_22"])*log10(21 + 4*r_rr - 4*r_rr^2) + x[,"n_12"]*log10(23 - 4*r_rr + 4*r_rr^2)
logL_rr <- (-4*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 2*x[,"n_12"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"])*log(2) + (x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"] + x[,"n_21"] + x[,"n_23"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + 2*(x[,"n_04"] + x[,"n_20"])*log(3 - 2*r_rr) + (x[,"n_03"] + x[,"n_21"])*log(3 - 2*r_rr) + (x[,"n_01"] + x[,"n_23"])*log(1 + 2*r_rr) + 2*(x[,"n_00"] + x[,"n_24"])*log(1 + 2*r_rr) + (x[,"n_10"] + x[,"n_14"])*(log(3 - 2*r_rr) + log(1 + 2*r_rr)) + (x[,"n_02"] + x[,"n_22"])*log(21 + 4*r_rr - 4*r_rr^2) + x[,"n_12"]*log(23 - 4*r_rr + 4*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_2.3 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- (-2*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 2*n05 - n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25)*log(2) + (n01 + n04 + n11 + n14 + n21 + n24)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(5) + 2*(n00 + n25)*log(pmax(1e-6,1 - r)) + 2*(n05 + n20)*log(pmax(1e-6,r)) + (n10 + n15)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n21)*(log(pmax(1e-6,r)) + log(3 + 2*r)) + (n02 + n23)*log(24 - 7*r - 8*r^2) + (n03 + n22)*log(9 + 23*r - 8*r^2) + (n11 + n14)*log(3 + 4*r - 4*r^2) + (n01 + n24)*log(5 - 7*r + 2*r^2) + (n12 + n13)*log(41 - 16*r + 16*r^2)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_cc <- -2*(x[,"n_11"] + x[,"n_14"])*log10(2) + 2*(x[,"n_10"] + x[,"n_15"])*log10(2) + 2*(x[,"n_05"] + x[,"n_20"])*log10(2) - (x[,"n_04"] + x[,"n_21"])*log10(2) - (x[,"n_01"] + x[,"n_24"])*log10(2) + 2*(x[,"n_00"] + x[,"n_25"])*log10(2) - (x[,"n_12"] + x[,"n_13"])*log10(37) - (x[,"n_03"] + x[,"n_22"])*(-log10(2) + log10(37)) - (x[,"n_02"] + x[,"n_23"])*(-log10(2) + log10(37)) + 2*(x[,"n_00"] + x[,"n_25"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_05"] + x[,"n_20"])*log10(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_15"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(3 + 2*r_cc)) + (x[,"n_02"] + x[,"n_23"])*log10(24 - 7*r_cc - 8*r_cc^2) + (x[,"n_03"] + x[,"n_22"])*log10(9 + 23*r_cc - 8*r_cc^2) + (x[,"n_11"] + x[,"n_14"])*log10(3 + 4*r_cc - 4*r_cc^2) + (x[,"n_01"] + x[,"n_24"])*log10(5 - 7*r_cc + 2*r_cc^2) + (x[,"n_12"] + x[,"n_13"])*log10(41 - 16*r_cc + 16*r_cc^2)
logL_cc <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"])*log(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"] + x[,"n_21"] + x[,"n_24"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + 2*(x[,"n_00"] + x[,"n_25"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_05"] + x[,"n_20"])*log(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_15"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(3 + 2*r_cc)) + (x[,"n_02"] + x[,"n_23"])*log(24 - 7*r_cc - 8*r_cc^2) + (x[,"n_03"] + x[,"n_22"])*log(9 + 23*r_cc - 8*r_cc^2) + (x[,"n_11"] + x[,"n_14"])*log(3 + 4*r_cc - 4*r_cc^2) + (x[,"n_01"] + x[,"n_24"])*log(5 - 7*r_cc + 2*r_cc^2) + (x[,"n_12"] + x[,"n_13"])*log(41 - 16*r_cc + 16*r_cc^2)
logL_cr <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- (-2*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 2*n05 - 2*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25)*log(2) + (n01 + n04 + n11 + n14 + n21 + n24)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(5) + (n00 + n05 + n20 + n25)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n13)*log(33 + 16*r - 16*r^2) + (n01 + n24)*log(2 + r - 2*r^2) + (n04 + n21)*log(1 + 3*r - 2*r^2) + (n10 + n15)*log(1 - 2*r + 2*r^2) + (n11 + n14)*log(5 - 4*r + 4*r^2) + (n02 + n23)*log(21 - 9*r + 8*r^2) + (n03 + n22)*log(20 - 7*r + 8*r^2)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_cr <- -2*(x[,"n_11"] + x[,"n_14"])*log10(2) + (x[,"n_10"] + x[,"n_15"])*log10(2) - (x[,"n_04"] + x[,"n_21"])*log10(2) - (x[,"n_01"] + x[,"n_24"])*log10(2) + 2*(x[,"n_00"] + x[,"n_05"] + x[,"n_20"] + x[,"n_25"])*log10(2) - (x[,"n_12"] + x[,"n_13"])*log10(37) - (x[,"n_03"] + x[,"n_22"])*(-log10(2) + log10(37)) - (x[,"n_02"] + x[,"n_23"])*(-log10(2) + log10(37)) + (x[,"n_00"] + x[,"n_05"] + x[,"n_20"] + x[,"n_25"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_12"] + x[,"n_13"])*log10(33 + 16*r_cr - 16*r_cr^2) + (x[,"n_01"] + x[,"n_24"])*log10(2 + r_cr - 2*r_cr^2) + (x[,"n_04"] + x[,"n_21"])*log10(1 + 3*r_cr - 2*r_cr^2) + (x[,"n_10"] + x[,"n_15"])*log10(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_11"] + x[,"n_14"])*log10(5 - 4*r_cr + 4*r_cr^2) + (x[,"n_02"] + x[,"n_23"])*log10(21 - 9*r_cr + 8*r_cr^2) + (x[,"n_03"] + x[,"n_22"])*log10(20 - 7*r_cr + 8*r_cr^2)
logL_cr <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"])*log(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"] + x[,"n_21"] + x[,"n_24"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_00"] + x[,"n_05"] + x[,"n_20"] + x[,"n_25"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_12"] + x[,"n_13"])*log(33 + 16*r_cr - 16*r_cr^2) + (x[,"n_01"] + x[,"n_24"])*log(2 + r_cr - 2*r_cr^2) + (x[,"n_04"] + x[,"n_21"])*log(1 + 3*r_cr - 2*r_cr^2) + (x[,"n_10"] + x[,"n_15"])*log(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_11"] + x[,"n_14"])*log(5 - 4*r_cr + 4*r_cr^2) + (x[,"n_02"] + x[,"n_23"])*log(21 - 9*r_cr + 8*r_cr^2) + (x[,"n_03"] + x[,"n_22"])*log(20 - 7*r_cr + 8*r_cr^2)
logL_rc <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- (-4*n00 - 4*n01 - 3*n02 - 3*n03 - 4*n04 - 4*n05 - 4*n10 - 4*n11 - 3*n12 - 3*n13 - 4*n14 - 4*n15 - 4*n20 - 4*n21 - 3*n22 - 3*n23 - 4*n24 - 4*n25)*log(2) + (n01 + n04 + n11 + n14 + n21 + n24)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(5) + (n05 + n20)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n00 + n25)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n12 + n13)*log(35 + 8*r - 8*r^2) + (n01 + n24)*log(4 + r - 2*r^2) + (n04 + n21)*log(3 + 3*r - 2*r^2) + (n02 + n23)*log(21 - 7*r + 4*r^2) + (n10 + n15)*log(3 - 4*r + 4*r^2) + (n11 + n14)*log(9 - 4*r + 4*r^2) + (n03 + n22)*log(18 - r + 4*r^2)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_rc <- -3*(x[,"n_11"] + x[,"n_14"])*log10(2) - (x[,"n_10"] + x[,"n_15"])*log10(2) - 2*(x[,"n_04"] + x[,"n_21"])*log10(2) - 2*(x[,"n_01"] + x[,"n_24"])*log10(2) - (x[,"n_12"] + x[,"n_13"])*log10(37) - (x[,"n_03"] + x[,"n_22"])*(-log10(2) + log10(37)) - (x[,"n_02"] + x[,"n_23"])*(-log10(2) + log10(37)) + (x[,"n_05"] + x[,"n_20"])*(log10(3 - 2*r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_25"])*(log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_12"] + x[,"n_13"])*log10(35 + 8*r_rc - 8*r_rc^2) + (x[,"n_01"] + x[,"n_24"])*log10(4 + r_rc - 2*r_rc^2) + (x[,"n_04"] + x[,"n_21"])*log10(3 + 3*r_rc - 2*r_rc^2) + (x[,"n_02"] + x[,"n_23"])*log10(21 - 7*r_rc + 4*r_rc^2) + (x[,"n_10"] + x[,"n_15"])*log10(3 - 4*r_rc + 4*r_rc^2) + (x[,"n_11"] + x[,"n_14"])*log10(9 - 4*r_rc + 4*r_rc^2) + (x[,"n_03"] + x[,"n_22"])*log10(18 - r_rc + 4*r_rc^2)
logL_rc <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_15"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"])*log(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"] + x[,"n_21"] + x[,"n_24"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_05"] + x[,"n_20"])*(log(3 - 2*r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_25"])*(log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_12"] + x[,"n_13"])*log(35 + 8*r_rc - 8*r_rc^2) + (x[,"n_01"] + x[,"n_24"])*log(4 + r_rc - 2*r_rc^2) + (x[,"n_04"] + x[,"n_21"])*log(3 + 3*r_rc - 2*r_rc^2) + (x[,"n_02"] + x[,"n_23"])*log(21 - 7*r_rc + 4*r_rc^2) + (x[,"n_10"] + x[,"n_15"])*log(3 - 4*r_rc + 4*r_rc^2) + (x[,"n_11"] + x[,"n_14"])*log(9 - 4*r_rc + 4*r_rc^2) + (x[,"n_03"] + x[,"n_22"])*log(18 - r_rc + 4*r_rc^2)
logL_rr <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- (-4*n00 - 4*n01 - 3*n02 - 3*n03 - 4*n04 - 4*n05 - 4*n10 - 4*n11 - 3*n12 - 3*n13 - 4*n14 - 4*n15 - 4*n20 - 4*n21 - 3*n22 - 3*n23 - 4*n24 - 4*n25)*log(2) + (n01 + n04 + n11 + n14 + n21 + n24)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(5) + (n05 + n20)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n00 + n25)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n03 + n22)*log(23 - 7*r - 4*r^2) + (n10 + n15)*log(1 + 4*r - 4*r^2) + (n11 + n14)*log(7 + 4*r - 4*r^2) + (n02 + n23)*log(12 + 15*r - 4*r^2) + (n04 + n21)*log(8 - 9*r + 2*r^2) + (n01 + n24)*log(1 + 5*r + 2*r^2) + (n12 + n13)*log(39 - 8*r + 8*r^2)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_rr <- -3*(x[,"n_11"] + x[,"n_14"])*log10(2) - (x[,"n_10"] + x[,"n_15"])*log10(2) - 2*(x[,"n_04"] + x[,"n_21"])*log10(2) - 2*(x[,"n_01"] + x[,"n_24"])*log10(2) - (x[,"n_12"] + x[,"n_13"])*log10(37) - (x[,"n_03"] + x[,"n_22"])*(-log10(2) + log10(37)) - (x[,"n_02"] + x[,"n_23"])*(-log10(2) + log10(37)) + (x[,"n_05"] + x[,"n_20"])*(log10(3 - 2*r_rr) + log10(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_25"])*(log10(pmax(1e-6,r_rr)) + log10(1 + 2*r_rr)) + (x[,"n_03"] + x[,"n_22"])*log10(23 - 7*r_rr - 4*r_rr^2) + (x[,"n_10"] + x[,"n_15"])*log10(1 + 4*r_rr - 4*r_rr^2) + (x[,"n_11"] + x[,"n_14"])*log10(7 + 4*r_rr - 4*r_rr^2) + (x[,"n_02"] + x[,"n_23"])*log10(12 + 15*r_rr - 4*r_rr^2) + (x[,"n_04"] + x[,"n_21"])*log10(8 - 9*r_rr + 2*r_rr^2) + (x[,"n_01"] + x[,"n_24"])*log10(1 + 5*r_rr + 2*r_rr^2) + (x[,"n_12"] + x[,"n_13"])*log10(39 - 8*r_rr + 8*r_rr^2)
logL_rr <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_15"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"])*log(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"] + x[,"n_21"] + x[,"n_24"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_05"] + x[,"n_20"])*(log(3 - 2*r_rr) + log(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_25"])*(log(pmax(1e-6,r_rr)) + log(1 + 2*r_rr)) + (x[,"n_03"] + x[,"n_22"])*log(23 - 7*r_rr - 4*r_rr^2) + (x[,"n_10"] + x[,"n_15"])*log(1 + 4*r_rr - 4*r_rr^2) + (x[,"n_11"] + x[,"n_14"])*log(7 + 4*r_rr - 4*r_rr^2) + (x[,"n_02"] + x[,"n_23"])*log(12 + 15*r_rr - 4*r_rr^2) + (x[,"n_04"] + x[,"n_21"])*log(8 - 9*r_rr + 2*r_rr^2) + (x[,"n_01"] + x[,"n_24"])*log(1 + 5*r_rr + 2*r_rr^2) + (x[,"n_12"] + x[,"n_13"])*log(39 - 8*r_rr + 8*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_2.4 <- function(x){
logL_cc <- function(r,n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n12,n14) {
L <- (-2*n01 - 3*n02 - n03 - 3*n04 - 2*n05 - 2*n11 + n13 - 2*n15 - 2*n21 - 3*n22 - n23 - 3*n24 - 2*n25)*log(2) + (n02 + n04 + n12 + n14 + n22 + n24)*log(3) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25)*log(5) + (n02 + n24)*log(7 - 6*r) + (n01 + n25)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n05 + n21)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n04 + n22)*log(1 + 6*r) + (n11 + n15)*log(1 + 4*r - 4*r^2) + (n03 + n23)*log(5 + 2*r - 2*r^2) + n13*log(3 - r + r^2)
return(L)}
interlogL_cc <- function(n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n12,n14) {
optimize(logL_cc,c(0,0.5), n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n12,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_13"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_12"],x[,"n_14"])
LOD_cc <- -((x[,"n_11"] + x[,"n_15"])*log10(2)) - 2*(x[,"n_04"] + x[,"n_22"])*log10(2) - 2*(x[,"n_02"] + x[,"n_24"])*log10(2) - x[,"n_13"]*(-2*log10(2) + log10(11)) - (x[,"n_03"] + x[,"n_23"])*(-log10(2) + log10(11)) + (x[,"n_02"] + x[,"n_24"])*log10(7 - 6*r_cc) + (x[,"n_01"] + x[,"n_25"])*(log10(3 - 2*r_cc) + log10(pmax(1e-6,1 - r_cc))) + (x[,"n_05"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(1 + 2*r_cc)) + (x[,"n_04"] + x[,"n_22"])*log10(1 + 6*r_cc) + (x[,"n_11"] + x[,"n_15"])*log10(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_03"] + x[,"n_23"])*log10(5 + 2*r_cc - 2*r_cc^2) + x[,"n_13"]*log10(3 - r_cc + r_cc^2)
logL_cc <- (-2*x[,"n_01"] - 3*x[,"n_02"] - x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_11"] + x[,"n_13"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 3*x[,"n_22"] - x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"])*log(2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_12"] + x[,"n_14"] + x[,"n_22"] + x[,"n_24"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_02"] + x[,"n_24"])*log(7 - 6*r_cc) + (x[,"n_01"] + x[,"n_25"])*(log(3 - 2*r_cc) + log(pmax(1e-6,1 - r_cc))) + (x[,"n_05"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(1 + 2*r_cc)) + (x[,"n_04"] + x[,"n_22"])*log(1 + 6*r_cc) + (x[,"n_11"] + x[,"n_15"])*log(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_03"] + x[,"n_23"])*log(5 + 2*r_cc - 2*r_cc^2) + x[,"n_13"]*log(3 - r_cc + r_cc^2)
logL_cr <- function(r,n02,n03,n04,n13,n22,n23,n24,n12,n14,n01,n05,n11,n15,n21,n25) {
L <- (-n02 - 2*n03 - n04 - n13 - n22 - 2*n23 - n24)*log(2) + (n02 + n04 + n12 + n14 + n22 + n24)*log(3) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25)*log(5) + (n01 + n05 + n21 + n25)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n13*log(9 + 8*r - 8*r^2) + (n11 + n15)*log(1 - 2*r + 2*r^2) + (n03 + n23)*log(13 - 8*r + 8*r^2)
return(L)}
interlogL_cr <- function(n02,n03,n04,n13,n22,n23,n24,n12,n14,n01,n05,n11,n15,n21,n25) {
optimize(logL_cr,c(0,0.5), n02,n03,n04,n13,n22,n23,n24,n12,n14,n01,n05,n11,n15,n21,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_13"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_12"],x[,"n_14"],x[,"n_01"],x[,"n_05"],x[,"n_11"],x[,"n_15"],x[,"n_21"],x[,"n_25"])
LOD_cr <- (x[,"n_11"] + x[,"n_15"])*log10(2) + 2*(x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*log10(2) - x[,"n_13"]*log10(11) - (x[,"n_03"] + x[,"n_23"])*log10(11) + (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + x[,"n_13"]*log10(9 + 8*r_cr - 8*r_cr^2) + (x[,"n_11"] + x[,"n_15"])*log10(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_03"] + x[,"n_23"])*log10(13 - 8*r_cr + 8*r_cr^2)
logL_cr <- (-x[,"n_02"] - 2*x[,"n_03"] - x[,"n_04"] - x[,"n_13"] - x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"])*log(2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_12"] + x[,"n_14"] + x[,"n_22"] + x[,"n_24"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + x[,"n_13"]*log(9 + 8*r_cr - 8*r_cr^2) + (x[,"n_11"] + x[,"n_15"])*log(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_03"] + x[,"n_23"])*log(13 - 8*r_cr + 8*r_cr^2)
logL_rc <- function(r,n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n12,n14) {
L <- (-4*n01 - n02 - 3*n03 - n04 - 4*n05 - 3*n11 - 2*n13 - 3*n15 - 4*n21 - n22 - 3*n23 - n24 - 4*n25)*log(2) + (n02 + n04 + n12 + n14 + n22 + n24)*log(3) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25)*log(5) + (n01 + n05 + n21 + n25)*(log(3 - 2*r) + log(1 + 2*r)) + n13*log(21 + 4*r - 4*r^2) + (n11 + n15)*log(5 - 4*r + 4*r^2) + (n03 + n23)*log(23 - 4*r + 4*r^2)
return(L)}
interlogL_rc <- function(n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n12,n14) {
optimize(logL_rc,c(0,0.5), n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n12,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_13"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_12"],x[,"n_14"])
LOD_rc <- -2*(x[,"n_11"] + x[,"n_15"])*log10(2) - 2*(x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*log10(2) - x[,"n_13"]*(log10(2) + log10(11)) - (x[,"n_03"] + x[,"n_23"])*(log10(2) + log10(11)) + (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*(log10(3 - 2*r_rc) + log10(1 + 2*r_rc)) + x[,"n_13"]*log10(21 + 4*r_rc - 4*r_rc^2) + (x[,"n_11"] + x[,"n_15"])*log10(5 - 4*r_rc + 4*r_rc^2) + (x[,"n_03"] + x[,"n_23"])*log10(23 - 4*r_rc + 4*r_rc^2)
logL_rc <- (-4*x[,"n_01"] - x[,"n_02"] - 3*x[,"n_03"] - x[,"n_04"] - 4*x[,"n_05"] - 3*x[,"n_11"] - 2*x[,"n_13"] - 3*x[,"n_15"] - 4*x[,"n_21"] - x[,"n_22"] - 3*x[,"n_23"] - x[,"n_24"] - 4*x[,"n_25"])*log(2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_12"] + x[,"n_14"] + x[,"n_22"] + x[,"n_24"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*(log(3 - 2*r_rc) + log(1 + 2*r_rc)) + x[,"n_13"]*log(21 + 4*r_rc - 4*r_rc^2) + (x[,"n_11"] + x[,"n_15"])*log(5 - 4*r_rc + 4*r_rc^2) + (x[,"n_03"] + x[,"n_23"])*log(23 - 4*r_rc + 4*r_rc^2)
logL_rr <- function(r,n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n12,n14) {
L <- (-2*n01 - 3*n02 - n03 - 3*n04 - 2*n05 - 2*n11 + n13 - 2*n15 - 2*n21 - 3*n22 - n23 - 3*n24 - 2*n25)*log(2) + (n02 + n04 + n12 + n14 + n22 + n24)*log(3) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25)*log(5) + (n04 + n22)*log(7 - 6*r) + (n05 + n21)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n01 + n25)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n02 + n24)*log(1 + 6*r) + (n11 + n15)*log(1 + 4*r - 4*r^2) + (n03 + n23)*log(5 + 2*r - 2*r^2) + n13*log(3 - r + r^2)
return(L)}
interlogL_rr <- function(n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n12,n14) {
optimize(logL_rr,c(0,0.5), n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n12,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_13"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_12"],x[,"n_14"])
LOD_rr <- -((x[,"n_11"] + x[,"n_15"])*log10(2)) - 2*(x[,"n_04"] + x[,"n_22"])*log10(2) - 2*(x[,"n_02"] + x[,"n_24"])*log10(2) - x[,"n_13"]*(-2*log10(2) + log10(11)) - (x[,"n_03"] + x[,"n_23"])*(-log10(2) + log10(11)) + (x[,"n_04"] + x[,"n_22"])*log10(7 - 6*r_rr) + (x[,"n_05"] + x[,"n_21"])*(log10(3 - 2*r_rr) + log10(pmax(1e-6,1 - r_rr))) + (x[,"n_01"] + x[,"n_25"])*(log10(pmax(1e-6,r_rr)) + log10(1 + 2*r_rr)) + (x[,"n_02"] + x[,"n_24"])*log10(1 + 6*r_rr) + (x[,"n_11"] + x[,"n_15"])*log10(1 + 4*r_rr - 4*r_rr^2) + (x[,"n_03"] + x[,"n_23"])*log10(5 + 2*r_rr - 2*r_rr^2) + x[,"n_13"]*log10(3 - r_rr + r_rr^2)
logL_rr <- (-2*x[,"n_01"] - 3*x[,"n_02"] - x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_11"] + x[,"n_13"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 3*x[,"n_22"] - x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"])*log(2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_12"] + x[,"n_14"] + x[,"n_22"] + x[,"n_24"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_04"] + x[,"n_22"])*log(7 - 6*r_rr) + (x[,"n_05"] + x[,"n_21"])*(log(3 - 2*r_rr) + log(pmax(1e-6,1 - r_rr))) + (x[,"n_01"] + x[,"n_25"])*(log(pmax(1e-6,r_rr)) + log(1 + 2*r_rr)) + (x[,"n_02"] + x[,"n_24"])*log(1 + 6*r_rr) + (x[,"n_11"] + x[,"n_15"])*log(1 + 4*r_rr - 4*r_rr^2) + (x[,"n_03"] + x[,"n_23"])*log(5 + 2*r_rr - 2*r_rr^2) + x[,"n_13"]*log(3 - r_rr + r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_3.0 <- function(x){
logL_c <- function(r,n01,n02,n11,n12,n21,n22,n00,n03,n10,n13,n20,n23) {
L <- (n01 + n02 + 2*n11 + 2*n12 + n21 + n22)*log(3) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(3*log(2) + log(5)) + (n00 + n23)*log(pmax(1e-6,1 - r)) + (n01 + n22)*log(2 - r) + (n03 + n20)*log(pmax(1e-6,r)) + (n02 + n21)*log(1 + r)
return(L)}
interlogL_c <- function(n01,n02,n11,n12,n21,n22,n00,n03,n10,n13,n20,n23) {
optimize(logL_c,c(0,0.5), n01,n02,n11,n12,n21,n22,n00,n03,n10,n13,n20,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_01"],x[,"n_02"],x[,"n_11"],x[,"n_12"],x[,"n_21"],x[,"n_22"],x[,"n_00"],x[,"n_03"],x[,"n_10"],x[,"n_13"],x[,"n_20"],x[,"n_23"])
LOD_c <- (x[,"n_03"] + x[,"n_20"])*log10(2) + (x[,"n_00"] + x[,"n_23"])*log10(2) - (x[,"n_02"] + x[,"n_21"])*(-log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_22"])*(-log10(2) + log10(3)) + (x[,"n_00"] + x[,"n_23"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_22"])*log10(2 - r_c) + (x[,"n_03"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_21"])*log10(1 + r_c)
logL_c <- (x[,"n_01"] + x[,"n_02"] + 2*x[,"n_11"] + 2*x[,"n_12"] + x[,"n_21"] + x[,"n_22"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(3*log(2) + log(5)) + (x[,"n_00"] + x[,"n_23"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_22"])*log(2 - r_c) + (x[,"n_03"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_21"])*log(1 + r_c)
logL_r <- function(r,n01,n02,n11,n12,n21,n22,n00,n03,n10,n13,n20,n23) {
L <- (n01 + n02 + 2*n11 + 2*n12 + n21 + n22)*log(3) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(3*log(2) + log(5)) + (n03 + n20)*log(pmax(1e-6,1 - r)) + (n02 + n21)*log(2 - r) + (n00 + n23)*log(pmax(1e-6,r)) + (n01 + n22)*log(1 + r)
return(L)}
interlogL_r <- function(n01,n02,n11,n12,n21,n22,n00,n03,n10,n13,n20,n23) {
optimize(logL_r,c(0,0.5), n01,n02,n11,n12,n21,n22,n00,n03,n10,n13,n20,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_02"],x[,"n_11"],x[,"n_12"],x[,"n_21"],x[,"n_22"],x[,"n_00"],x[,"n_03"],x[,"n_10"],x[,"n_13"],x[,"n_20"],x[,"n_23"])
LOD_r <- (x[,"n_03"] + x[,"n_20"])*log10(2) + (x[,"n_00"] + x[,"n_23"])*log10(2) - (x[,"n_02"] + x[,"n_21"])*(-log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_22"])*(-log10(2) + log10(3)) + (x[,"n_03"] + x[,"n_20"])*log10(pmax(1e-6,1 - r_r)) + (x[,"n_02"] + x[,"n_21"])*log10(2 - r_r) + (x[,"n_00"] + x[,"n_23"])*log10(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_22"])*log10(1 + r_r)
logL_r <- (x[,"n_01"] + x[,"n_02"] + 2*x[,"n_11"] + 2*x[,"n_12"] + x[,"n_21"] + x[,"n_22"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(3*log(2) + log(5)) + (x[,"n_03"] + x[,"n_20"])*log(pmax(1e-6,1 - r_r)) + (x[,"n_02"] + x[,"n_21"])*log(2 - r_r) + (x[,"n_00"] + x[,"n_23"])*log(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_22"])*log(1 + r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.1_3.3 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
L <- (-4*n00 - 3*n01 - 4*n02 - 2*n03 - 4*n04 - 3*n05 - 4*n06 - 3*n10 - 3*n11 - 3*n12 - 2*n13 - 3*n14 - 3*n15 - 3*n16 - 4*n20 - 3*n21 - 4*n22 - 2*n23 - 4*n24 - 3*n25 - 4*n26)*log(2) + (n01 + n02 + n04 + n05 + n11 + n12 + n14 + n15 + n21 + n22 + n24 + n25)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26)*log(5) + 2*(n00 + n26)*log(pmax(1e-6,1 - r)) + (n01 + n25)*(log(pmax(1e-6,1 - r)) + log(2 - r)) + 2*(n06 + n20)*log(pmax(1e-6,r)) + (n10 + n16)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n05 + n21)*(log(pmax(1e-6,r)) + log(1 + r)) + (n03 + n23)*log(9 + 5*r - 5*r^2) + (n11 + n15)*log(1 + 2*r - 2*r^2) + (n12 + n14)*log(8 + r - r^2) + (n02 + n24)*log(14 - 12*r + r^2) + (n04 + n22)*log(3 + 10*r + r^2) + n13*log(23 - 10*r + 10*r^2)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"])
LOD_cc <- 2*(x[,"n_10"] + x[,"n_16"])*log10(2) + 2*(x[,"n_06"] + x[,"n_20"])*log10(2) + 2*(x[,"n_00"] + x[,"n_26"])*log10(2) - (x[,"n_05"] + x[,"n_21"])*(-2*log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_25"])*(-2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_15"])*(-log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_14"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_22"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_02"] + x[,"n_24"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_03"] + x[,"n_23"])*(-2*log10(2) + log10(41)) - x[,"n_13"]*(-log10(2) + log10(41)) + 2*(x[,"n_00"] + x[,"n_26"])*log10(pmax(1e-6,1 - r_cc)) + (x[,"n_01"] + x[,"n_25"])*(log10(pmax(1e-6,1 - r_cc)) + log10(2 - r_cc)) + 2*(x[,"n_06"] + x[,"n_20"])*log10(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_16"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_05"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(1 + r_cc)) + (x[,"n_03"] + x[,"n_23"])*log10(9 + 5*r_cc - 5*r_cc^2) + (x[,"n_11"] + x[,"n_15"])*log10(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_12"] + x[,"n_14"])*log10(8 + r_cc - r_cc^2) + (x[,"n_02"] + x[,"n_24"])*log10(14 - 12*r_cc + r_cc^2) + (x[,"n_04"] + x[,"n_22"])*log10(3 + 10*r_cc + r_cc^2) + x[,"n_13"]*log10(23 - 10*r_cc + 10*r_cc^2)
logL_cc <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 2*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_05"] - 4*x[,"n_06"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 3*x[,"n_16"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_25"] - 4*x[,"n_26"])*log(2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_04"] + x[,"n_05"] + x[,"n_11"] + x[,"n_12"] + x[,"n_14"] + x[,"n_15"] + x[,"n_21"] + x[,"n_22"] + x[,"n_24"] + x[,"n_25"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"])*log(5) + 2*(x[,"n_00"] + x[,"n_26"])*log(pmax(1e-6,1 - r_cc)) + (x[,"n_01"] + x[,"n_25"])*(log(pmax(1e-6,1 - r_cc)) + log(2 - r_cc)) + 2*(x[,"n_06"] + x[,"n_20"])*log(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_16"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_05"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(1 + r_cc)) + (x[,"n_03"] + x[,"n_23"])*log(9 + 5*r_cc - 5*r_cc^2) + (x[,"n_11"] + x[,"n_15"])*log(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_12"] + x[,"n_14"])*log(8 + r_cc - r_cc^2) + (x[,"n_02"] + x[,"n_24"])*log(14 - 12*r_cc + r_cc^2) + (x[,"n_04"] + x[,"n_22"])*log(3 + 10*r_cc + r_cc^2) + x[,"n_13"]*log(23 - 10*r_cc + 10*r_cc^2)
logL_cr <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
L <- (-4*n00 - 4*n01 - 4*n02 - 3*n03 - 4*n04 - 4*n05 - 4*n06 - 4*n10 - 2*n11 - 4*n12 - n13 - 4*n14 - 2*n15 - 4*n16 - 4*n20 - 4*n21 - 4*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n26)*log(2) + (n01 + n02 + n04 + n05 + n11 + n12 + n14 + n15 + n21 + n22 + n24 + n25)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26)*log(5) + (n00 + n06 + n20 + n26)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n13*log(9 + 5*r - 5*r^2) + (n01 + n05 + n21 + n25)*log(1 + 2*r - 2*r^2) + (n02 + n04 + n22 + n24)*log(8 + r - r^2) + (n11 + n15)*log(1 - r + r^2) + (n10 + n16)*log(1 - 2*r + 2*r^2) + (n12 + n14)*log(17 - 2*r + 2*r^2) + (n03 + n23)*log(23 - 10*r + 10*r^2)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"])
LOD_cr <- (x[,"n_10"] + x[,"n_16"])*log10(2) + 2*(x[,"n_00"] + x[,"n_06"] + x[,"n_20"] + x[,"n_26"])*log10(2) + (x[,"n_11"] + x[,"n_15"])*(2*log10(2) - log10(3)) - (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*(-log10(2) + log10(3)) - (x[,"n_02"] + x[,"n_04"] + x[,"n_22"] + x[,"n_24"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_12"] + x[,"n_14"])*(-log10(2) + log10(3) + log10(11)) - x[,"n_13"]*(-2*log10(2) + log10(41)) - (x[,"n_03"] + x[,"n_23"])*(-log10(2) + log10(41)) + (x[,"n_00"] + x[,"n_06"] + x[,"n_20"] + x[,"n_26"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + x[,"n_13"]*log10(9 + 5*r_cr - 5*r_cr^2) + (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*log10(1 + 2*r_cr - 2*r_cr^2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_22"] + x[,"n_24"])*log10(8 + r_cr - r_cr^2) + (x[,"n_11"] + x[,"n_15"])*log10(1 - r_cr + r_cr^2) + (x[,"n_10"] + x[,"n_16"])*log10(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_12"] + x[,"n_14"])*log10(17 - 2*r_cr + 2*r_cr^2) + (x[,"n_03"] + x[,"n_23"])*log10(23 - 10*r_cr + 10*r_cr^2)
logL_cr <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_06"] - 4*x[,"n_10"] - 2*x[,"n_11"] - 4*x[,"n_12"] - x[,"n_13"] - 4*x[,"n_14"] - 2*x[,"n_15"] - 4*x[,"n_16"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"])*log(2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_04"] + x[,"n_05"] + x[,"n_11"] + x[,"n_12"] + x[,"n_14"] + x[,"n_15"] + x[,"n_21"] + x[,"n_22"] + x[,"n_24"] + x[,"n_25"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"])*log(5) + (x[,"n_00"] + x[,"n_06"] + x[,"n_20"] + x[,"n_26"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + x[,"n_13"]*log(9 + 5*r_cr - 5*r_cr^2) + (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*log(1 + 2*r_cr - 2*r_cr^2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_22"] + x[,"n_24"])*log(8 + r_cr - r_cr^2) + (x[,"n_11"] + x[,"n_15"])*log(1 - r_cr + r_cr^2) + (x[,"n_10"] + x[,"n_16"])*log(1 - 2*r_cr + 2*r_cr^2) + (x[,"n_12"] + x[,"n_14"])*log(17 - 2*r_cr + 2*r_cr^2) + (x[,"n_03"] + x[,"n_23"])*log(23 - 10*r_cr + 10*r_cr^2)
logL_rr <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
L <- (-4*n00 - 3*n01 - 4*n02 - 2*n03 - 4*n04 - 3*n05 - 4*n06 - 3*n10 - 3*n11 - 3*n12 - 2*n13 - 3*n14 - 3*n15 - 3*n16 - 4*n20 - 3*n21 - 4*n22 - 2*n23 - 4*n24 - 3*n25 - 4*n26)*log(2) + (n01 + n02 + n04 + n05 + n11 + n12 + n14 + n15 + n21 + n22 + n24 + n25)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26)*log(5) + 2*(n06 + n20)*log(pmax(1e-6,1 - r)) + (n05 + n21)*(log(pmax(1e-6,1 - r)) + log(2 - r)) + 2*(n00 + n26)*log(pmax(1e-6,r)) + (n10 + n16)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n25)*(log(pmax(1e-6,r)) + log(1 + r)) + (n03 + n23)*log(9 + 5*r - 5*r^2) + (n11 + n15)*log(1 + 2*r - 2*r^2) + (n12 + n14)*log(8 + r - r^2) + (n04 + n22)*log(14 - 12*r + r^2) + (n02 + n24)*log(3 + 10*r + r^2) + n13*log(23 - 10*r + 10*r^2)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"])
LOD_rr <- 2*(x[,"n_10"] + x[,"n_16"])*log10(2) + 2*(x[,"n_06"] + x[,"n_20"])*log10(2) + 2*(x[,"n_00"] + x[,"n_26"])*log10(2) - (x[,"n_05"] + x[,"n_21"])*(-2*log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_25"])*(-2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_15"])*(-log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_14"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_22"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_02"] + x[,"n_24"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_03"] + x[,"n_23"])*(-2*log10(2) + log10(41)) - x[,"n_13"]*(-log10(2) + log10(41)) + 2*(x[,"n_06"] + x[,"n_20"])*log10(pmax(1e-6,1 - r_rr)) + (x[,"n_05"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr)) + log10(2 - r_rr)) + 2*(x[,"n_00"] + x[,"n_26"])*log10(pmax(1e-6,r_rr)) + (x[,"n_10"] + x[,"n_16"])*(log10(pmax(1e-6,1 - r_rr)) + log10(pmax(1e-6,r_rr))) + (x[,"n_01"] + x[,"n_25"])*(log10(pmax(1e-6,r_rr)) + log10(1 + r_rr)) + (x[,"n_03"] + x[,"n_23"])*log10(9 + 5*r_rr - 5*r_rr^2) + (x[,"n_11"] + x[,"n_15"])*log10(1 + 2*r_rr - 2*r_rr^2) + (x[,"n_12"] + x[,"n_14"])*log10(8 + r_rr - r_rr^2) + (x[,"n_04"] + x[,"n_22"])*log10(14 - 12*r_rr + r_rr^2) + (x[,"n_02"] + x[,"n_24"])*log10(3 + 10*r_rr + r_rr^2) + x[,"n_13"]*log10(23 - 10*r_rr + 10*r_rr^2)
logL_rr <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 2*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_05"] - 4*x[,"n_06"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 3*x[,"n_16"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_25"] - 4*x[,"n_26"])*log(2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_04"] + x[,"n_05"] + x[,"n_11"] + x[,"n_12"] + x[,"n_14"] + x[,"n_15"] + x[,"n_21"] + x[,"n_22"] + x[,"n_24"] + x[,"n_25"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"])*log(5) + 2*(x[,"n_06"] + x[,"n_20"])*log(pmax(1e-6,1 - r_rr)) + (x[,"n_05"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr)) + log(2 - r_rr)) + 2*(x[,"n_00"] + x[,"n_26"])*log(pmax(1e-6,r_rr)) + (x[,"n_10"] + x[,"n_16"])*(log(pmax(1e-6,1 - r_rr)) + log(pmax(1e-6,r_rr))) + (x[,"n_01"] + x[,"n_25"])*(log(pmax(1e-6,r_rr)) + log(1 + r_rr)) + (x[,"n_03"] + x[,"n_23"])*log(9 + 5*r_rr - 5*r_rr^2) + (x[,"n_11"] + x[,"n_15"])*log(1 + 2*r_rr - 2*r_rr^2) + (x[,"n_12"] + x[,"n_14"])*log(8 + r_rr - r_rr^2) + (x[,"n_04"] + x[,"n_22"])*log(14 - 12*r_rr + r_rr^2) + (x[,"n_02"] + x[,"n_24"])*log(3 + 10*r_rr + r_rr^2) + x[,"n_13"]*log(23 - 10*r_rr + 10*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_1.2 <- function(x){
logL_cc <- function(r,n01,n02,n10,n13,n20,n23,n31,n32,n00,n03,n11,n12,n21,n22,n30,n33) {
L <- (n01 + n02 + n10 + n13 + n20 + n23 + n31 + n32)*log(3) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(log(2) + log(5)) + 3*(n00 + n33)*log(pmax(1e-6,1 - r)) + 3*(n03 + n30)*log(pmax(1e-6,r)) + (n01 + n10 + n23 + n32)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n13 + n20 + n31)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n12 + n21)*(log(pmax(1e-6,r)) + log(7 - 12*r + 9*r^2)) + (n11 + n22)*log(4 - 10*r + 15*r^2 - 9*r^3)
return(L)}
interlogL_cc <- function(n01,n02,n10,n13,n20,n23,n31,n32,n00,n03,n11,n12,n21,n22,n30,n33) {
optimize(logL_cc,c(0,0.5), n01,n02,n10,n13,n20,n23,n31,n32,n00,n03,n11,n12,n21,n22,n30,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_13"],x[,"n_20"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_00"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_21"],x[,"n_22"],x[,"n_30"],x[,"n_33"])
LOD_cc <- 3*(x[,"n_03"] + x[,"n_30"])*log10(2) + 3*(x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log10(2) + 3*(x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(2) + 3*(x[,"n_00"] + x[,"n_33"])*log10(2) - (x[,"n_12"] + x[,"n_21"])*(-3*log10(2) + log10(13)) - (x[,"n_11"] + x[,"n_22"])*(-3*log10(2) + log10(13)) + 3*(x[,"n_00"] + x[,"n_33"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_03"] + x[,"n_30"])*log10(pmax(1e-6,r_cc)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_12"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(7 - 12*r_cc + 9*r_cc^2)) + (x[,"n_11"] + x[,"n_22"])*log10(4 - 10*r_cc + 15*r_cc^2 - 9*r_cc^3)
logL_cc <- (x[,"n_01"] + x[,"n_02"] + x[,"n_10"] + x[,"n_13"] + x[,"n_20"] + x[,"n_23"] + x[,"n_31"] + x[,"n_32"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(log(2) + log(5)) + 3*(x[,"n_00"] + x[,"n_33"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_03"] + x[,"n_30"])*log(pmax(1e-6,r_cc)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_12"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(7 - 12*r_cc + 9*r_cc^2)) + (x[,"n_11"] + x[,"n_22"])*log(4 - 10*r_cc + 15*r_cc^2 - 9*r_cc^3)
logL_cm <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(3*log(2) + log(5)) + (n03 + n30)*(log(2 - r) + 2*log(pmax(1e-6,r))) + (n00 + n33)*(log(pmax(1e-6,1 - r)) + log(1 - r^2)) + (n02 + n13 + n20 + n31)*(log(pmax(1e-6,r)) + log(5 - 5*r + 3*r^2)) + (n12 + n21)*log(6 - 4*r + 14*r^2 - 9*r^3) + (n01 + n10 + n23 + n32)*log(3 - 4*r + 4*r^2 - 3*r^3) + (n11 + n22)*log(7 + 3*r - 13*r^2 + 9*r^3)
return(L)}
interlogL_cm <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
optimize(logL_cm,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"])
LOD_cm <- -((x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(3))) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(3)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-3*log10(2) + log10(13)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(13)) - (x[,"n_12"] + x[,"n_21"])*(-3*log10(2) + log10(3) + log10(17)) - (x[,"n_11"] + x[,"n_22"])*(-3*log10(2) + log10(3) + log10(17)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - r_cm) + 2*log10(pmax(1e-6,r_cm))) + (x[,"n_00"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cm)) + log10(1 - r_cm^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm)) + log10(5 - 5*r_cm + 3*r_cm^2)) + (x[,"n_12"] + x[,"n_21"])*log10(6 - 4*r_cm + 14*r_cm^2 - 9*r_cm^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(3 - 4*r_cm + 4*r_cm^2 - 3*r_cm^3) + (x[,"n_11"] + x[,"n_22"])*log10(7 + 3*r_cm - 13*r_cm^2 + 9*r_cm^3)
logL_cm <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(3*log(2) + log(5)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - r_cm) + 2*log(pmax(1e-6,r_cm))) + (x[,"n_00"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cm)) + log(1 - r_cm^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(pmax(1e-6,r_cm)) + log(5 - 5*r_cm + 3*r_cm^2)) + (x[,"n_12"] + x[,"n_21"])*log(6 - 4*r_cm + 14*r_cm^2 - 9*r_cm^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log(3 - 4*r_cm + 4*r_cm^2 - 3*r_cm^3) + (x[,"n_11"] + x[,"n_22"])*log(7 + 3*r_cm - 13*r_cm^2 + 9*r_cm^3)
logL_cr <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(2*log(2) + log(3) + log(5)) + (n00 + n33)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n30)*(log(pmax(1e-6,r)) + log(3 - 4*r + r^2)) + (n11 + n22)*log(12 - 8*r + 11*r^2 - 9*r^3) + (n02 + n13 + n20 + n31)*log(3 - 4*r + 7*r^2 - 3*r^3) + (n01 + n10 + n23 + n32)*log(3 - r - 2*r^2 + 3*r^3) + (n12 + n21)*log(6 + 13*r - 16*r^2 + 9*r^3)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"])
LOD_cr <- -((x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_21"])*(-3*log10(2) + log10(7) + log10(11)) - (x[,"n_11"] + x[,"n_22"])*(-3*log10(2) + log10(7) + log10(11)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-3*log10(2) + log10(19)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(19)) + (x[,"n_00"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr)) + log10(2 + r_cr)) + (x[,"n_03"] + x[,"n_30"])*(log10(pmax(1e-6,r_cr)) + log10(3 - 4*r_cr + r_cr^2)) + (x[,"n_11"] + x[,"n_22"])*log10(12 - 8*r_cr + 11*r_cr^2 - 9*r_cr^3) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log10(3 - 4*r_cr + 7*r_cr^2 - 3*r_cr^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(3 - r_cr - 2*r_cr^2 + 3*r_cr^3) + (x[,"n_12"] + x[,"n_21"])*log10(6 + 13*r_cr - 16*r_cr^2 + 9*r_cr^3)
logL_cr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(2*log(2) + log(3) + log(5)) + (x[,"n_00"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr)) + log(2 + r_cr)) + (x[,"n_03"] + x[,"n_30"])*(log(pmax(1e-6,r_cr)) + log(3 - 4*r_cr + r_cr^2)) + (x[,"n_11"] + x[,"n_22"])*log(12 - 8*r_cr + 11*r_cr^2 - 9*r_cr^3) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log(3 - 4*r_cr + 7*r_cr^2 - 3*r_cr^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log(3 - r_cr - 2*r_cr^2 + 3*r_cr^3) + (x[,"n_12"] + x[,"n_21"])*log(6 + 13*r_cr - 16*r_cr^2 + 9*r_cr^3)
logL_rc <- function(r,n01,n02,n10,n13,n20,n23,n31,n32,n00,n03,n11,n12,n21,n22,n30,n33) {
L <- (n01 + n02 + n10 + n13 + n20 + n23 + n31 + n32)*log(3) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(log(2) + 2*log(5)) + (n03 + n30)*(log(3 - r) + 2*log(pmax(1e-6,r))) + (n00 + n33)*(2*log(pmax(1e-6,1 - r)) + log(2 + r)) + (n02 + n13 + n20 + n31)*(log(pmax(1e-6,r)) + log(2 - 2*r + r^2)) + (n12 + n21)*log(9 - 7*r + 15*r^2 - 9*r^3) + (n01 + n10 + n23 + n32)*log(1 - r + r^2 - r^3) + (n11 + n22)*log(8 + 4*r - 12*r^2 + 9*r^3)
return(L)}
interlogL_rc <- function(n01,n02,n10,n13,n20,n23,n31,n32,n00,n03,n11,n12,n21,n22,n30,n33) {
optimize(logL_rc,c(0,0.5), n01,n02,n10,n13,n20,n23,n31,n32,n00,n03,n11,n12,n21,n22,n30,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_13"],x[,"n_20"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_00"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_21"],x[,"n_22"],x[,"n_30"],x[,"n_33"])
LOD_rc <- (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(3*log10(2) - log10(5)) - (x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-3*log10(2) + log10(5)) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_21"])*(-3*log10(2) + log10(5) + log10(13)) - (x[,"n_11"] + x[,"n_22"])*(-3*log10(2) + log10(5) + log10(13)) + (x[,"n_03"] + x[,"n_30"])*(log10(3 - r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_33"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(2 + r_rc)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 2*r_rc + r_rc^2)) + (x[,"n_12"] + x[,"n_21"])*log10(9 - 7*r_rc + 15*r_rc^2 - 9*r_rc^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(1 - r_rc + r_rc^2 - r_rc^3) + (x[,"n_11"] + x[,"n_22"])*log10(8 + 4*r_rc - 12*r_rc^2 + 9*r_rc^3)
logL_rc <- (x[,"n_01"] + x[,"n_02"] + x[,"n_10"] + x[,"n_13"] + x[,"n_20"] + x[,"n_23"] + x[,"n_31"] + x[,"n_32"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(log(2) + 2*log(5)) + (x[,"n_03"] + x[,"n_30"])*(log(3 - r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_33"])*(2*log(pmax(1e-6,1 - r_rc)) + log(2 + r_rc)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(2 - 2*r_rc + r_rc^2)) + (x[,"n_12"] + x[,"n_21"])*log(9 - 7*r_rc + 15*r_rc^2 - 9*r_rc^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log(1 - r_rc + r_rc^2 - r_rc^3) + (x[,"n_11"] + x[,"n_22"])*log(8 + 4*r_rc - 12*r_rc^2 + 9*r_rc^3)
logL_rm <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(3*log(2) + 2*log(5)) + (n03 + n30)*(log(2 - r) + log(3 - r) + log(pmax(1e-6,r))) + (n00 + n33)*(log(2 + r) + log(1 - r^2)) + (n11 + n22)*log(32 - 3*r + 10*r^2 - 9*r^3) + (n02 + n13 + n20 + n31)*log(9 - 5*r + 8*r^2 - 3*r^3) + (n01 + n10 + n23 + n32)*log(9 - 2*r - r^2 + 3*r^3) + (n12 + n21)*log(30 + 10*r - 17*r^2 + 9*r^3)
return(L)}
interlogL_rm <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
optimize(logL_rm,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"])
LOD_rm <- -((x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(3) + log10(5))) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-3*log10(2) + log10(5) + log10(13)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(5) + log10(13)) - (x[,"n_12"] + x[,"n_21"])*(-3*log10(2) + log10(3) + log10(5) + log10(17)) - (x[,"n_11"] + x[,"n_22"])*(-3*log10(2) + log10(3) + log10(5) + log10(17)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - r_rm) + log10(3 - r_rm) + log10(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_33"])*(log10(2 + r_rm) + log10(1 - r_rm^2)) + (x[,"n_11"] + x[,"n_22"])*log10(32 - 3*r_rm + 10*r_rm^2 - 9*r_rm^3) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log10(9 - 5*r_rm + 8*r_rm^2 - 3*r_rm^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(9 - 2*r_rm - r_rm^2 + 3*r_rm^3) + (x[,"n_12"] + x[,"n_21"])*log10(30 + 10*r_rm - 17*r_rm^2 + 9*r_rm^3)
logL_rm <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(3*log(2) + 2*log(5)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - r_rm) + log(3 - r_rm) + log(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_33"])*(log(2 + r_rm) + log(1 - r_rm^2)) + (x[,"n_11"] + x[,"n_22"])*log(32 - 3*r_rm + 10*r_rm^2 - 9*r_rm^3) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log(9 - 5*r_rm + 8*r_rm^2 - 3*r_rm^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log(9 - 2*r_rm - r_rm^2 + 3*r_rm^3) + (x[,"n_12"] + x[,"n_21"])*log(30 + 10*r_rm - 17*r_rm^2 + 9*r_rm^3)
logL_rr <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(2*log(2) + log(3) + 2*log(5)) + (n00 + n33)*(log(pmax(1e-6,r)) + 2*log(2 + r)) + (n03 + n30)*(log(3 - r) + log(3 - 4*r + r^2)) + (n12 + n21)*log(57 - 25*r + 19*r^2 - 9*r^3) + (n01 + n10 + n23 + n32)*log(6 + 13*r - r^2 - 3*r^3) + (n02 + n13 + n20 + n31)*log(15 - 2*r - 10*r^2 + 3*r^3) + (n11 + n22)*log(42 + 14*r - 8*r^2 + 9*r^3)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"])
LOD_rr <- -((x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + 2*log10(5))) - (x[,"n_00"] + x[,"n_33"])*(-log10(2) + 2*(-log10(2) + log10(5))) - (x[,"n_12"] + x[,"n_21"])*(-3*log10(2) + log10(5) + log10(7) + log10(11)) - (x[,"n_11"] + x[,"n_22"])*(-3*log10(2) + log10(5) + log10(7) + log10(11)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-3*log10(2) + log10(5) + log10(19)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(5) + log10(19)) + (x[,"n_00"] + x[,"n_33"])*(log10(pmax(1e-6,r_rr)) + 2*log10(2 + r_rr)) + (x[,"n_03"] + x[,"n_30"])*(log10(3 - r_rr) + log10(3 - 4*r_rr + r_rr^2)) + (x[,"n_12"] + x[,"n_21"])*log10(57 - 25*r_rr + 19*r_rr^2 - 9*r_rr^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(6 + 13*r_rr - r_rr^2 - 3*r_rr^3) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log10(15 - 2*r_rr - 10*r_rr^2 + 3*r_rr^3) + (x[,"n_11"] + x[,"n_22"])*log10(42 + 14*r_rr - 8*r_rr^2 + 9*r_rr^3)
logL_rr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(2*log(2) + log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_33"])*(log(pmax(1e-6,r_rr)) + 2*log(2 + r_rr)) + (x[,"n_03"] + x[,"n_30"])*(log(3 - r_rr) + log(3 - 4*r_rr + r_rr^2)) + (x[,"n_12"] + x[,"n_21"])*log(57 - 25*r_rr + 19*r_rr^2 - 9*r_rr^3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log(6 + 13*r_rr - r_rr^2 - 3*r_rr^3) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log(15 - 2*r_rr - 10*r_rr^2 + 3*r_rr^3) + (x[,"n_11"] + x[,"n_22"])*log(42 + 14*r_rr - 8*r_rr^2 + 9*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_1.3 <- function(x){
logL_cc <- function(r,n10,n14,n20,n24,n00,n01,n02,n03,n04,n11,n12,n13,n21,n22,n23,n30,n31,n32,n33,n34) {
L <- (n10 + n14 + n20 + n24)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(3*log(2) + log(5)) + (n02 + n32)*log(7) + 3*(n00 + n34)*log(pmax(1e-6,1 - r)) + 3*(n04 + n30)*log(pmax(1e-6,r)) + (n02 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n31)*(log(5 - 2*r) + 2*log(pmax(1e-6,r))) + (n14 + n20)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n33)*(2*log(pmax(1e-6,1 - r)) + log(3 + 2*r)) + (n13 + n21)*(log(pmax(1e-6,r)) + log(12 - 11*r + 6*r^2)) + (n12 + n22)*log(9 - 7*r + 7*r^2) + (n11 + n23)*log(7 - 8*r + 7*r^2 - 6*r^3)
return(L)}
interlogL_cc <- function(n10,n14,n20,n24,n00,n01,n02,n03,n04,n11,n12,n13,n21,n22,n23,n30,n31,n32,n33,n34) {
optimize(logL_cc,c(0,0.5), n10,n14,n20,n24,n00,n01,n02,n03,n04,n11,n12,n13,n21,n22,n23,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_10"],x[,"n_14"],x[,"n_20"],x[,"n_24"],x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cc <- 3*(x[,"n_14"] + x[,"n_20"])*log10(2) - 2*(x[,"n_13"] + x[,"n_21"])*log10(2) - 2*(x[,"n_11"] + x[,"n_23"])*log10(2) + 3*(x[,"n_10"] + x[,"n_24"])*log10(2) + 3*(x[,"n_04"] + x[,"n_30"])*log10(2) + 2*(x[,"n_02"] + x[,"n_32"])*log10(2) + 3*(x[,"n_00"] + x[,"n_34"])*log10(2) - (x[,"n_12"] + x[,"n_22"])*(-2*log10(2) + log10(29)) + 3*(x[,"n_00"] + x[,"n_34"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_04"] + x[,"n_30"])*log10(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_24"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_31"])*(log10(5 - 2*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_20"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_33"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(3 + 2*r_cc)) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(12 - 11*r_cc + 6*r_cc^2)) + (x[,"n_12"] + x[,"n_22"])*log10(9 - 7*r_cc + 7*r_cc^2) + (x[,"n_11"] + x[,"n_23"])*log10(7 - 8*r_cc + 7*r_cc^2 - 6*r_cc^3)
logL_cc <- (x[,"n_10"] + x[,"n_14"] + x[,"n_20"] + x[,"n_24"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(3*log(2) + log(5)) + (x[,"n_02"] + x[,"n_32"])*log(7) + 3*(x[,"n_00"] + x[,"n_34"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_04"] + x[,"n_30"])*log(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_24"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_31"])*(log(5 - 2*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_20"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_33"])*(2*log(pmax(1e-6,1 - r_cc)) + log(3 + 2*r_cc)) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(12 - 11*r_cc + 6*r_cc^2)) + (x[,"n_12"] + x[,"n_22"])*log(9 - 7*r_cc + 7*r_cc^2) + (x[,"n_11"] + x[,"n_23"])*log(7 - 8*r_cc + 7*r_cc^2 - 6*r_cc^3)
logL_cm <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n12 - n14 - n20 - n22 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(5) + (n00 + n34)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n30)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n12 + n22)*log(21 + 2*r - 2*r^2) + (n02 + n32)*log(3 - r + r^2) + (n03 + n31)*(log(pmax(1e-6,r)) + log(4 - 3*r + 2*r^2)) + (n14 + n20)*(log(pmax(1e-6,r)) + log(5 - 8*r + 6*r^2)) + (n10 + n24)*log(3 - 7*r + 10*r^2 - 6*r^3) + (n13 + n21)*log(3 + 6*r^2 - 4*r^3) + (n01 + n33)*log(3 - 4*r + 3*r^2 - 2*r^3) + (n11 + n23)*log(5 - 6*r^2 + 4*r^3)
return(L)}
interlogL_cm <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_cm,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cm <- -2*(x[,"n_13"] + x[,"n_21"])*log10(2) - 2*(x[,"n_11"] + x[,"n_23"])*log10(2) + 3*(x[,"n_04"] + x[,"n_30"])*log10(2) + 3*(x[,"n_00"] + x[,"n_34"])*log10(2) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_20"])*(-2*log10(2) + log10(5)) - (x[,"n_10"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(11)) - (x[,"n_12"] + x[,"n_22"])*(-log10(2) + log10(43)) + (x[,"n_00"] + x[,"n_34"])*(2*log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_cm)) + 2*log10(pmax(1e-6,r_cm))) + (x[,"n_12"] + x[,"n_22"])*log10(21 + 2*r_cm - 2*r_cm^2) + (x[,"n_02"] + x[,"n_32"])*log10(3 - r_cm + r_cm^2) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm)) + log10(4 - 3*r_cm + 2*r_cm^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(pmax(1e-6,r_cm)) + log10(5 - 8*r_cm + 6*r_cm^2)) + (x[,"n_10"] + x[,"n_24"])*log10(3 - 7*r_cm + 10*r_cm^2 - 6*r_cm^3) + (x[,"n_13"] + x[,"n_21"])*log10(3 + 6*r_cm^2 - 4*r_cm^3) + (x[,"n_01"] + x[,"n_33"])*log10(3 - 4*r_cm + 3*r_cm^2 - 2*r_cm^3) + (x[,"n_11"] + x[,"n_23"])*log10(5 - 6*r_cm^2 + 4*r_cm^3)
logL_cm <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_12"] - x[,"n_14"] - x[,"n_20"] - x[,"n_22"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + (x[,"n_00"] + x[,"n_34"])*(2*log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_cm)) + 2*log(pmax(1e-6,r_cm))) + (x[,"n_12"] + x[,"n_22"])*log(21 + 2*r_cm - 2*r_cm^2) + (x[,"n_02"] + x[,"n_32"])*log(3 - r_cm + r_cm^2) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,r_cm)) + log(4 - 3*r_cm + 2*r_cm^2)) + (x[,"n_14"] + x[,"n_20"])*(log(pmax(1e-6,r_cm)) + log(5 - 8*r_cm + 6*r_cm^2)) + (x[,"n_10"] + x[,"n_24"])*log(3 - 7*r_cm + 10*r_cm^2 - 6*r_cm^3) + (x[,"n_13"] + x[,"n_21"])*log(3 + 6*r_cm^2 - 4*r_cm^3) + (x[,"n_01"] + x[,"n_33"])*log(3 - 4*r_cm + 3*r_cm^2 - 2*r_cm^3) + (x[,"n_11"] + x[,"n_23"])*log(5 - 6*r_cm^2 + 4*r_cm^3)
logL_cr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(3*log(2) + log(5)) + (n04 + n30)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n34)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n12 + n22)*log(6 + 5*r - 5*r^2) + (n01 + n33)*(log(pmax(1e-6,r)) + log(4 - 5*r + 2*r^2)) + (n10 + n24)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n02 + n32)*log(3 - 5*r + 5*r^2) + (n11 + n23)*log(6 - 8*r + 11*r^2 - 6*r^3) + (n14 + n20)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n03 + n31)*log(1 + r^2 - 2*r^3) + (n13 + n21)*log(3 + 4*r - 7*r^2 + 6*r^3)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cr <- -2*(x[,"n_13"] + x[,"n_21"])*log10(2) - 2*(x[,"n_11"] + x[,"n_23"])*log10(2) + 3*(x[,"n_04"] + x[,"n_30"])*log10(2) + 3*(x[,"n_00"] + x[,"n_34"])*log10(2) + (x[,"n_14"] + x[,"n_20"])*(3*log10(2) - log10(3)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(7)) - (x[,"n_12"] + x[,"n_22"])*(-2*log10(2) + log10(29)) + (x[,"n_04"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cr)) + 2*log10(pmax(1e-6,r_cr))) + (x[,"n_12"] + x[,"n_22"])*log10(6 + 5*r_cr - 5*r_cr^2) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,r_cr)) + log10(4 - 5*r_cr + 2*r_cr^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(pmax(1e-6,r_cr)) + log10(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_02"] + x[,"n_32"])*log10(3 - 5*r_cr + 5*r_cr^2) + (x[,"n_11"] + x[,"n_23"])*log10(6 - 8*r_cr + 11*r_cr^2 - 6*r_cr^3) + (x[,"n_14"] + x[,"n_20"])*log10(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_03"] + x[,"n_31"])*log10(1 + r_cr^2 - 2*r_cr^3) + (x[,"n_13"] + x[,"n_21"])*log10(3 + 4*r_cr - 7*r_cr^2 + 6*r_cr^3)
logL_cr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(3*log(2) + log(5)) + (x[,"n_04"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_cr)) + 2*log(pmax(1e-6,r_cr))) + (x[,"n_12"] + x[,"n_22"])*log(6 + 5*r_cr - 5*r_cr^2) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,r_cr)) + log(4 - 5*r_cr + 2*r_cr^2)) + (x[,"n_10"] + x[,"n_24"])*(log(pmax(1e-6,r_cr)) + log(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_02"] + x[,"n_32"])*log(3 - 5*r_cr + 5*r_cr^2) + (x[,"n_11"] + x[,"n_23"])*log(6 - 8*r_cr + 11*r_cr^2 - 6*r_cr^3) + (x[,"n_14"] + x[,"n_20"])*log(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_03"] + x[,"n_31"])*log(1 + r_cr^2 - 2*r_cr^3) + (x[,"n_13"] + x[,"n_21"])*log(3 + 4*r_cr - 7*r_cr^2 + 6*r_cr^3)
logL_rc <- function(r,n10,n14,n20,n24,n00,n01,n02,n03,n04,n11,n12,n13,n21,n22,n23,n30,n31,n32,n33,n34) {
L <- (n10 + n14 + n20 + n24)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(3*log(2) + 2*log(5)) + (n04 + n30)*(log(3 - r) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(2*log(pmax(1e-6,1 - r)) + log(2 + r)) + (n12 + n22)*log(36 + r - r^2) + (n14 + n20)*(log(pmax(1e-6,r)) + log(2 - 2*r + r^2)) + (n02 + n32)*log(9 - r + r^2) + (n03 + n31)*(log(pmax(1e-6,r)) + log(12 - 5*r + 2*r^2)) + (n13 + n21)*log(18 + 11*r^2 - 6*r^3) + (n01 + n33)*log(9 - 8*r + r^2 - 2*r^3) + (n10 + n24)*log(1 - r + r^2 - r^3) + (n11 + n23)*log(23 - 4*r - 7*r^2 + 6*r^3)
return(L)}
interlogL_rc <- function(n10,n14,n20,n24,n00,n01,n02,n03,n04,n11,n12,n13,n21,n22,n23,n30,n31,n32,n33,n34) {
optimize(logL_rc,c(0,0.5), n10,n14,n20,n24,n00,n01,n02,n03,n04,n11,n12,n13,n21,n22,n23,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_10"],x[,"n_14"],x[,"n_20"],x[,"n_24"],x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rc <- (x[,"n_10"] + x[,"n_24"])*(3*log10(2) - log10(5)) - (x[,"n_03"] + x[,"n_31"])*log10(5) - (x[,"n_01"] + x[,"n_33"])*log10(5) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(5)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(5)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_21"])*(2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_23"])*(2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(5) + log10(7)) - (x[,"n_12"] + x[,"n_22"])*(-2*log10(2) + log10(5) + log10(29)) + (x[,"n_04"] + x[,"n_30"])*(log10(3 - r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_34"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(2 + r_rc)) + (x[,"n_12"] + x[,"n_22"])*log10(36 + r_rc - r_rc^2) + (x[,"n_14"] + x[,"n_20"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 2*r_rc + r_rc^2)) + (x[,"n_02"] + x[,"n_32"])*log10(9 - r_rc + r_rc^2) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(12 - 5*r_rc + 2*r_rc^2)) + (x[,"n_13"] + x[,"n_21"])*log10(18 + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_01"] + x[,"n_33"])*log10(9 - 8*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_10"] + x[,"n_24"])*log10(1 - r_rc + r_rc^2 - r_rc^3) + (x[,"n_11"] + x[,"n_23"])*log10(23 - 4*r_rc - 7*r_rc^2 + 6*r_rc^3)
logL_rc <- (x[,"n_10"] + x[,"n_14"] + x[,"n_20"] + x[,"n_24"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(3*log(2) + 2*log(5)) + (x[,"n_04"] + x[,"n_30"])*(log(3 - r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_34"])*(2*log(pmax(1e-6,1 - r_rc)) + log(2 + r_rc)) + (x[,"n_12"] + x[,"n_22"])*log(36 + r_rc - r_rc^2) + (x[,"n_14"] + x[,"n_20"])*(log(pmax(1e-6,r_rc)) + log(2 - 2*r_rc + r_rc^2)) + (x[,"n_02"] + x[,"n_32"])*log(9 - r_rc + r_rc^2) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(12 - 5*r_rc + 2*r_rc^2)) + (x[,"n_13"] + x[,"n_21"])*log(18 + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_01"] + x[,"n_33"])*log(9 - 8*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_10"] + x[,"n_24"])*log(1 - r_rc + r_rc^2 - r_rc^3) + (x[,"n_11"] + x[,"n_23"])*log(23 - 4*r_rc - 7*r_rc^2 + 6*r_rc^3)
logL_rm <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n12 - n14 - n20 - n22 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - n12 - 2*n13 - 2*n14 - 2*n20 - 2*n21 - n22 - 2*n23 - 2*n24 - 2*n30 - 2*n31 - n32 - 2*n33 - 2*n34)*log(5) + (n04 + n30)*(log(pmax(1e-6,1 - r)) + log(3 - r) + log(pmax(1e-6,r))) + (n00 + n34)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 + r)) + (n12 + n22)*log(21 + 2*r - 2*r^2) + (n02 + n32)*log(3 - r + r^2) + (n14 + n20)*log(9 - 11*r + 14*r^2 - 6*r^3) + (n11 + n23)*log(19 + 6*r^2 - 4*r^3) + (n03 + n31)*log(9 - 4*r + 3*r^2 - 2*r^3) + (n01 + n33)*log(6 + 4*r - 3*r^2 + 2*r^3) + (n13 + n21)*log(21 - 6*r^2 + 4*r^3) + (n10 + n24)*log(6 + r - 4*r^2 + 6*r^3)
return(L)}
interlogL_rm <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_rm,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rm <- -((x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_21"])*(2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_23"])*(2*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_14"] + x[,"n_20"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_10"] + x[,"n_24"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(11)) - (x[,"n_12"] + x[,"n_22"])*(-log10(2) + log10(43)) + (x[,"n_04"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_rm)) + log10(3 - r_rm) + log10(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_rm)) + log10(pmax(1e-6,r_rm)) + log10(2 + r_rm)) + (x[,"n_12"] + x[,"n_22"])*log10(21 + 2*r_rm - 2*r_rm^2) + (x[,"n_02"] + x[,"n_32"])*log10(3 - r_rm + r_rm^2) + (x[,"n_14"] + x[,"n_20"])*log10(9 - 11*r_rm + 14*r_rm^2 - 6*r_rm^3) + (x[,"n_11"] + x[,"n_23"])*log10(19 + 6*r_rm^2 - 4*r_rm^3) + (x[,"n_03"] + x[,"n_31"])*log10(9 - 4*r_rm + 3*r_rm^2 - 2*r_rm^3) + (x[,"n_01"] + x[,"n_33"])*log10(6 + 4*r_rm - 3*r_rm^2 + 2*r_rm^3) + (x[,"n_13"] + x[,"n_21"])*log10(21 - 6*r_rm^2 + 4*r_rm^3) + (x[,"n_10"] + x[,"n_24"])*log10(6 + r_rm - 4*r_rm^2 + 6*r_rm^3)
logL_rm <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_12"] - x[,"n_14"] - x[,"n_20"] - x[,"n_22"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(5) + (x[,"n_04"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_rm)) + log(3 - r_rm) + log(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_rm)) + log(pmax(1e-6,r_rm)) + log(2 + r_rm)) + (x[,"n_12"] + x[,"n_22"])*log(21 + 2*r_rm - 2*r_rm^2) + (x[,"n_02"] + x[,"n_32"])*log(3 - r_rm + r_rm^2) + (x[,"n_14"] + x[,"n_20"])*log(9 - 11*r_rm + 14*r_rm^2 - 6*r_rm^3) + (x[,"n_11"] + x[,"n_23"])*log(19 + 6*r_rm^2 - 4*r_rm^3) + (x[,"n_03"] + x[,"n_31"])*log(9 - 4*r_rm + 3*r_rm^2 - 2*r_rm^3) + (x[,"n_01"] + x[,"n_33"])*log(6 + 4*r_rm - 3*r_rm^2 + 2*r_rm^3) + (x[,"n_13"] + x[,"n_21"])*log(21 - 6*r_rm^2 + 4*r_rm^3) + (x[,"n_10"] + x[,"n_24"])*log(6 + r_rm - 4*r_rm^2 + 6*r_rm^3)
logL_rr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(3*log(2) + 2*log(5)) + (n04 + n30)*(2*log(pmax(1e-6,1 - r)) + log(3 - r)) + (n00 + n34)*(2*log(pmax(1e-6,r)) + log(2 + r)) + (n02 + n32)*log(6 + 11*r - 11*r^2) + (n10 + n24)*(log(pmax(1e-6,r)) + log(4 + r - 3*r^2)) + (n01 + n33)*(log(pmax(1e-6,r)) + log(8 + 5*r - 2*r^2)) + (n12 + n22)*log(39 - 11*r + 11*r^2) + (n13 + n21)*log(27 - 16*r + 7*r^2 - 6*r^3) + (n03 + n31)*log(11 - 12*r - r^2 + 2*r^3) + (n14 + n20)*log(2 + 3*r - 8*r^2 + 3*r^3) + (n11 + n23)*log(12 + 20*r - 11*r^2 + 6*r^3)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rr <- -((x[,"n_03"] + x[,"n_31"])*log10(5)) - (x[,"n_01"] + x[,"n_33"])*log10(5) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(5)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_21"])*(2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_23"])*(2*log10(2) + log10(5)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(5) + log10(7)) - (x[,"n_12"] + x[,"n_22"])*(-2*log10(2) + log10(5) + log10(29)) + (x[,"n_04"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(3 - r_rr)) + (x[,"n_00"] + x[,"n_34"])*(2*log10(pmax(1e-6,r_rr)) + log10(2 + r_rr)) + (x[,"n_02"] + x[,"n_32"])*log10(6 + 11*r_rr - 11*r_rr^2) + (x[,"n_10"] + x[,"n_24"])*(log10(pmax(1e-6,r_rr)) + log10(4 + r_rr - 3*r_rr^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,r_rr)) + log10(8 + 5*r_rr - 2*r_rr^2)) + (x[,"n_12"] + x[,"n_22"])*log10(39 - 11*r_rr + 11*r_rr^2) + (x[,"n_13"] + x[,"n_21"])*log10(27 - 16*r_rr + 7*r_rr^2 - 6*r_rr^3) + (x[,"n_03"] + x[,"n_31"])*log10(11 - 12*r_rr - r_rr^2 + 2*r_rr^3) + (x[,"n_14"] + x[,"n_20"])*log10(2 + 3*r_rr - 8*r_rr^2 + 3*r_rr^3) + (x[,"n_11"] + x[,"n_23"])*log10(12 + 20*r_rr - 11*r_rr^2 + 6*r_rr^3)
logL_rr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(3*log(2) + 2*log(5)) + (x[,"n_04"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_rr)) + log(3 - r_rr)) + (x[,"n_00"] + x[,"n_34"])*(2*log(pmax(1e-6,r_rr)) + log(2 + r_rr)) + (x[,"n_02"] + x[,"n_32"])*log(6 + 11*r_rr - 11*r_rr^2) + (x[,"n_10"] + x[,"n_24"])*(log(pmax(1e-6,r_rr)) + log(4 + r_rr - 3*r_rr^2)) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,r_rr)) + log(8 + 5*r_rr - 2*r_rr^2)) + (x[,"n_12"] + x[,"n_22"])*log(39 - 11*r_rr + 11*r_rr^2) + (x[,"n_13"] + x[,"n_21"])*log(27 - 16*r_rr + 7*r_rr^2 - 6*r_rr^3) + (x[,"n_03"] + x[,"n_31"])*log(11 - 12*r_rr - r_rr^2 + 2*r_rr^3) + (x[,"n_14"] + x[,"n_20"])*log(2 + 3*r_rr - 8*r_rr^2 + 3*r_rr^3) + (x[,"n_11"] + x[,"n_23"])*log(12 + 20*r_rr - 11*r_rr^2 + 6*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_1.4 <- function(x){
logL_cc <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*(2*log(2) + log(3) + log(5)) + (n04 + n31)*(2*log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n14 + n21 + n32)*(log(pmax(1e-6,r)) + log(5 + r - 3*r^2)) + (n01 + n34)*(log(pmax(1e-6,1 - r)) + log(3 - 4*r + r^2)) + (n12 + n23)*log(15 - 17*r + 17*r^2 - 9*r^3) + (n02 + n11 + n24 + n33)*log(3 + 2*r - 8*r^2 + 3*r^3) + (n13 + n22)*log(6 + 10*r - 10*r^2 + 9*r^3)
return(L)}
interlogL_cc <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
optimize(logL_cc,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cc <- -((x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(5))) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_22"])*(-3*log10(2) + log10(7) + log10(11)) - (x[,"n_12"] + x[,"n_23"])*(-3*log10(2) + log10(7) + log10(11)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*(-3*log10(2) + log10(19)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(-3*log10(2) + log10(19)) + (x[,"n_04"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(5 + r_cc - 3*r_cc^2)) + (x[,"n_01"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cc)) + log10(3 - 4*r_cc + r_cc^2)) + (x[,"n_12"] + x[,"n_23"])*log10(15 - 17*r_cc + 17*r_cc^2 - 9*r_cc^3) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*log10(3 + 2*r_cc - 8*r_cc^2 + 3*r_cc^3) + (x[,"n_13"] + x[,"n_22"])*log10(6 + 10*r_cc - 10*r_cc^2 + 9*r_cc^3)
logL_cc <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(2*log(2) + log(3) + log(5)) + (x[,"n_04"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(5 + r_cc - 3*r_cc^2)) + (x[,"n_01"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_cc)) + log(3 - 4*r_cc + r_cc^2)) + (x[,"n_12"] + x[,"n_23"])*log(15 - 17*r_cc + 17*r_cc^2 - 9*r_cc^3) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*log(3 + 2*r_cc - 8*r_cc^2 + 3*r_cc^3) + (x[,"n_13"] + x[,"n_22"])*log(6 + 10*r_cc - 10*r_cc^2 + 9*r_cc^3)
logL_cm <- function(r,n12,n13,n22,n23,n01,n02,n03,n04,n11,n14,n21,n24,n31,n32,n33,n34) {
L <- (n12 + n13 + n22 + n23)*log(3) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*(3*log(2) + log(5)) + (n01 + n34)*(log(pmax(1e-6,1 - r)) + log(2 - r) + log(pmax(1e-6,r))) + (n04 + n31)*(log(pmax(1e-6,r)) + log(1 - r^2)) + (n13 + n22)*log(2 - r + 4*r^2 - 3*r^3) + (n02 + n11 + n24 + n33)*log(3 - 5*r + 6*r^2 - 3*r^3) + (n12 + n23)*log(2 + 2*r - 5*r^2 + 3*r^3) + (n03 + n14 + n21 + n32)*log(1 + 2*r - 3*r^2 + 3*r^3)
return(L)}
interlogL_cm <- function(n12,n13,n22,n23,n01,n02,n03,n04,n11,n14,n21,n24,n31,n32,n33,n34) {
optimize(logL_cm,c(0,0.5), n12,n13,n22,n23,n01,n02,n03,n04,n11,n14,n21,n24,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_12"],x[,"n_13"],x[,"n_22"],x[,"n_23"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_14"],x[,"n_21"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cm <- -((x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(3))) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*(-3*log10(2) + log10(13)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(-3*log10(2) + log10(13)) - (x[,"n_13"] + x[,"n_22"])*(-3*log10(2) + log10(17)) - (x[,"n_12"] + x[,"n_23"])*(-3*log10(2) + log10(17)) + (x[,"n_01"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cm)) + log10(2 - r_cm) + log10(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm)) + log10(1 - r_cm^2)) + (x[,"n_13"] + x[,"n_22"])*log10(2 - r_cm + 4*r_cm^2 - 3*r_cm^3) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*log10(3 - 5*r_cm + 6*r_cm^2 - 3*r_cm^3) + (x[,"n_12"] + x[,"n_23"])*log10(2 + 2*r_cm - 5*r_cm^2 + 3*r_cm^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log10(1 + 2*r_cm - 3*r_cm^2 + 3*r_cm^3)
logL_cm <- (x[,"n_12"] + x[,"n_13"] + x[,"n_22"] + x[,"n_23"])*log(3) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(3*log(2) + log(5)) + (x[,"n_01"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_cm)) + log(2 - r_cm) + log(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_31"])*(log(pmax(1e-6,r_cm)) + log(1 - r_cm^2)) + (x[,"n_13"] + x[,"n_22"])*log(2 - r_cm + 4*r_cm^2 - 3*r_cm^3) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*log(3 - 5*r_cm + 6*r_cm^2 - 3*r_cm^3) + (x[,"n_12"] + x[,"n_23"])*log(2 + 2*r_cm - 5*r_cm^2 + 3*r_cm^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log(1 + 2*r_cm - 3*r_cm^2 + 3*r_cm^3)
logL_cr <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*(log(2) + log(5)) + (n04 + n31)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n34)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n02 + n11 + n24 + n33)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n13 + n22)*(log(pmax(1e-6,r)) + log(8 - 14*r + 9*r^2)) + (n12 + n23)*log(3 - 7*r + 13*r^2 - 9*r^3) + (n03 + n14 + n21 + n32)*log(1 - 3*r + 5*r^2 - 3*r^3)
return(L)}
interlogL_cr <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
optimize(logL_cr,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cr <- 3*(x[,"n_04"] + x[,"n_31"])*log10(2) + 3*(x[,"n_01"] + x[,"n_34"])*log10(2) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*(3*log10(2) - log10(3)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(-3*log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_22"])*(-3*log10(2) + log10(13)) - (x[,"n_12"] + x[,"n_23"])*(-3*log10(2) + log10(13)) + (x[,"n_04"] + x[,"n_31"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cr)) + 2*log10(pmax(1e-6,r_cr))) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(log10(pmax(1e-6,r_cr)) + log10(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_13"] + x[,"n_22"])*(log10(pmax(1e-6,r_cr)) + log10(8 - 14*r_cr + 9*r_cr^2)) + (x[,"n_12"] + x[,"n_23"])*log10(3 - 7*r_cr + 13*r_cr^2 - 9*r_cr^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log10(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3)
logL_cr <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(log(2) + log(5)) + (x[,"n_04"] + x[,"n_31"])*(2*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_cr)) + 2*log(pmax(1e-6,r_cr))) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(log(pmax(1e-6,r_cr)) + log(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_13"] + x[,"n_22"])*(log(pmax(1e-6,r_cr)) + log(8 - 14*r_cr + 9*r_cr^2)) + (x[,"n_12"] + x[,"n_23"])*log(3 - 7*r_cr + 13*r_cr^2 - 9*r_cr^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3)
logL_rc <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*(2*log(2) + log(3) + 2*log(5)) + (n04 + n31)*(log(3 - r) + log(pmax(1e-6,r)) + log(2 + r)) + (n01 + n34)*(log(2 + r) + log(3 - 4*r + r^2)) + (n13 + n22)*log(48 - 4*r + 13*r^2 - 9*r^3) + (n02 + n11 + n24 + n33)*log(15 - 8*r + 5*r^2 - 3*r^3) + (n03 + n14 + n21 + n32)*log(9 + 7*r - 4*r^2 + 3*r^3) + (n12 + n23)*log(48 + 5*r - 14*r^2 + 9*r^3)
return(L)}
interlogL_rc <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
optimize(logL_rc,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rc <- -((x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + 2*log10(5))) - (x[,"n_04"] + x[,"n_31"])*(-log10(2) + 2*(-log10(2) + log10(5))) - (x[,"n_13"] + x[,"n_22"])*(-3*log10(2) + log10(5) + log10(7) + log10(11)) - (x[,"n_12"] + x[,"n_23"])*(-3*log10(2) + log10(5) + log10(7) + log10(11)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*(-3*log10(2) + log10(5) + log10(19)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(-3*log10(2) + log10(5) + log10(19)) + (x[,"n_04"] + x[,"n_31"])*(log10(3 - r_rc) + log10(pmax(1e-6,r_rc)) + log10(2 + r_rc)) + (x[,"n_01"] + x[,"n_34"])*(log10(2 + r_rc) + log10(3 - 4*r_rc + r_rc^2)) + (x[,"n_13"] + x[,"n_22"])*log10(48 - 4*r_rc + 13*r_rc^2 - 9*r_rc^3) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*log10(15 - 8*r_rc + 5*r_rc^2 - 3*r_rc^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log10(9 + 7*r_rc - 4*r_rc^2 + 3*r_rc^3) + (x[,"n_12"] + x[,"n_23"])*log10(48 + 5*r_rc - 14*r_rc^2 + 9*r_rc^3)
logL_rc <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(2*log(2) + log(3) + 2*log(5)) + (x[,"n_04"] + x[,"n_31"])*(log(3 - r_rc) + log(pmax(1e-6,r_rc)) + log(2 + r_rc)) + (x[,"n_01"] + x[,"n_34"])*(log(2 + r_rc) + log(3 - 4*r_rc + r_rc^2)) + (x[,"n_13"] + x[,"n_22"])*log(48 - 4*r_rc + 13*r_rc^2 - 9*r_rc^3) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*log(15 - 8*r_rc + 5*r_rc^2 - 3*r_rc^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log(9 + 7*r_rc - 4*r_rc^2 + 3*r_rc^3) + (x[,"n_12"] + x[,"n_23"])*log(48 + 5*r_rc - 14*r_rc^2 + 9*r_rc^3)
logL_rm <- function(r,n12,n13,n22,n23,n01,n02,n03,n04,n11,n14,n21,n24,n31,n32,n33,n34) {
L <- (n12 + n13 + n22 + n23)*log(3) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*(3*log(2) + 2*log(5)) + (n01 + n34)*(log(2 - r) + log(pmax(1e-6,r)) + log(2 + r)) + (n04 + n31)*(log(3 - r) + log(1 - r^2)) + (n12 + n23)*log(10 + 4*r^2 - 3*r^3) + (n03 + n14 + n21 + n32)*log(11 - 8*r + 6*r^2 - 3*r^3) + (n13 + n22)*log(11 + r - 5*r^2 + 3*r^3) + (n02 + n11 + n24 + n33)*log(6 + 5*r - 3*r^2 + 3*r^3)
return(L)}
interlogL_rm <- function(n12,n13,n22,n23,n01,n02,n03,n04,n11,n14,n21,n24,n31,n32,n33,n34) {
optimize(logL_rm,c(0,0.5), n12,n13,n22,n23,n01,n02,n03,n04,n11,n14,n21,n24,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_12"],x[,"n_13"],x[,"n_22"],x[,"n_23"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_14"],x[,"n_21"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rm <- -((x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(3) + log10(5))) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*(-3*log10(2) + log10(5) + log10(13)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(-3*log10(2) + log10(5) + log10(13)) - (x[,"n_13"] + x[,"n_22"])*(-3*log10(2) + log10(5) + log10(17)) - (x[,"n_12"] + x[,"n_23"])*(-3*log10(2) + log10(5) + log10(17)) + (x[,"n_01"] + x[,"n_34"])*(log10(2 - r_rm) + log10(pmax(1e-6,r_rm)) + log10(2 + r_rm)) + (x[,"n_04"] + x[,"n_31"])*(log10(3 - r_rm) + log10(1 - r_rm^2)) + (x[,"n_12"] + x[,"n_23"])*log10(10 + 4*r_rm^2 - 3*r_rm^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log10(11 - 8*r_rm + 6*r_rm^2 - 3*r_rm^3) + (x[,"n_13"] + x[,"n_22"])*log10(11 + r_rm - 5*r_rm^2 + 3*r_rm^3) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*log10(6 + 5*r_rm - 3*r_rm^2 + 3*r_rm^3)
logL_rm <- (x[,"n_12"] + x[,"n_13"] + x[,"n_22"] + x[,"n_23"])*log(3) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(3*log(2) + 2*log(5)) + (x[,"n_01"] + x[,"n_34"])*(log(2 - r_rm) + log(pmax(1e-6,r_rm)) + log(2 + r_rm)) + (x[,"n_04"] + x[,"n_31"])*(log(3 - r_rm) + log(1 - r_rm^2)) + (x[,"n_12"] + x[,"n_23"])*log(10 + 4*r_rm^2 - 3*r_rm^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log(11 - 8*r_rm + 6*r_rm^2 - 3*r_rm^3) + (x[,"n_13"] + x[,"n_22"])*log(11 + r_rm - 5*r_rm^2 + 3*r_rm^3) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*log(6 + 5*r_rm - 3*r_rm^2 + 3*r_rm^3)
logL_rr <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*(log(2) + 2*log(5)) + (n04 + n31)*(2*log(pmax(1e-6,1 - r)) + log(3 - r)) + (n01 + n34)*(2*log(pmax(1e-6,r)) + log(2 + r)) + (n02 + n11 + n24 + n33)*(log(pmax(1e-6,r)) + log(4 + r - 3*r^2)) + (n13 + n22)*log(12 - 14*r + 17*r^2 - 9*r^3) + (n03 + n14 + n21 + n32)*log(2 + 3*r - 8*r^2 + 3*r^3) + (n12 + n23)*log(6 + 7*r - 10*r^2 + 9*r^3)
return(L)}
interlogL_rr <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
optimize(logL_rr,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rr <- -((x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(5))) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_13"] + x[,"n_22"])*(-3*log10(2) + log10(5) + log10(13)) - (x[,"n_12"] + x[,"n_23"])*(-3*log10(2) + log10(5) + log10(13)) + (x[,"n_04"] + x[,"n_31"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(3 - r_rr)) + (x[,"n_01"] + x[,"n_34"])*(2*log10(pmax(1e-6,r_rr)) + log10(2 + r_rr)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(log10(pmax(1e-6,r_rr)) + log10(4 + r_rr - 3*r_rr^2)) + (x[,"n_13"] + x[,"n_22"])*log10(12 - 14*r_rr + 17*r_rr^2 - 9*r_rr^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log10(2 + 3*r_rr - 8*r_rr^2 + 3*r_rr^3) + (x[,"n_12"] + x[,"n_23"])*log10(6 + 7*r_rr - 10*r_rr^2 + 9*r_rr^3)
logL_rr <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(log(2) + 2*log(5)) + (x[,"n_04"] + x[,"n_31"])*(2*log(pmax(1e-6,1 - r_rr)) + log(3 - r_rr)) + (x[,"n_01"] + x[,"n_34"])*(2*log(pmax(1e-6,r_rr)) + log(2 + r_rr)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_24"] + x[,"n_33"])*(log(pmax(1e-6,r_rr)) + log(4 + r_rr - 3*r_rr^2)) + (x[,"n_13"] + x[,"n_22"])*log(12 - 14*r_rr + 17*r_rr^2 - 9*r_rr^3) + (x[,"n_03"] + x[,"n_14"] + x[,"n_21"] + x[,"n_32"])*log(2 + 3*r_rr - 8*r_rr^2 + 3*r_rr^3) + (x[,"n_12"] + x[,"n_23"])*log(6 + 7*r_rr - 10*r_rr^2 + 9*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_1.5 <- function(x){
logL_cc <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34) {
L <- (-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34)*(3*log(2) + log(5)) + (n02 + n34)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n04 + n32)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n03 + n33)*log(1 + 4*r - 4*r^2) + (n12 + n24)*log(6 - 3*r - 2*r^2) + (n14 + n22)*log(1 + 7*r - 2*r^2) + (n13 + n23)*log(9 - 4*r + 4*r^2)
return(L)}
interlogL_cc <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34) {
optimize(logL_cc,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cc <- -2*(x[,"n_14"] + x[,"n_22"])*log10(2) - 3*(x[,"n_13"] + x[,"n_23"])*log10(2) - 2*(x[,"n_12"] + x[,"n_24"])*log10(2) - (x[,"n_03"] + x[,"n_33"])*log10(2) + (x[,"n_02"] + x[,"n_34"])*(log10(3 - 2*r_cc) + log10(pmax(1e-6,1 - r_cc))) + (x[,"n_04"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(1 + 2*r_cc)) + (x[,"n_03"] + x[,"n_33"])*log10(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_12"] + x[,"n_24"])*log10(6 - 3*r_cc - 2*r_cc^2) + (x[,"n_14"] + x[,"n_22"])*log10(1 + 7*r_cc - 2*r_cc^2) + (x[,"n_13"] + x[,"n_23"])*log10(9 - 4*r_cc + 4*r_cc^2)
logL_cc <- (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(3*log(2) + log(5)) + (x[,"n_02"] + x[,"n_34"])*(log(3 - 2*r_cc) + log(pmax(1e-6,1 - r_cc))) + (x[,"n_04"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(1 + 2*r_cc)) + (x[,"n_03"] + x[,"n_33"])*log(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_12"] + x[,"n_24"])*log(6 - 3*r_cc - 2*r_cc^2) + (x[,"n_14"] + x[,"n_22"])*log(1 + 7*r_cc - 2*r_cc^2) + (x[,"n_13"] + x[,"n_23"])*log(9 - 4*r_cc + 4*r_cc^2)
logL_cm <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34) {
L <- (-n02 - n03 - n04 - 2*n12 - 2*n13 - 2*n14 - 2*n22 - 2*n23 - 2*n24 - n32 - n33 - n34)*log(2) + (-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34)*log(5) + (n02 + n04 + n32 + n34)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n13 + n23)*log(3 + 4*r - 4*r^2) + (n12 + n24)*log(3 - 3*r + 2*r^2) + (n03 + n33)*log(1 - 2*r + 2*r^2) + (n14 + n22)*log(2 - r + 2*r^2)
return(L)}
interlogL_cm <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34) {
optimize(logL_cm,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cm <- -((x[,"n_14"] + x[,"n_22"])*log10(2)) - 2*(x[,"n_13"] + x[,"n_23"])*log10(2) - (x[,"n_12"] + x[,"n_24"])*log10(2) + (x[,"n_03"] + x[,"n_33"])*log10(2) + 2*(x[,"n_02"] + x[,"n_04"] + x[,"n_32"] + x[,"n_34"])*log10(2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_32"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + (x[,"n_13"] + x[,"n_23"])*log10(3 + 4*r_cm - 4*r_cm^2) + (x[,"n_12"] + x[,"n_24"])*log10(3 - 3*r_cm + 2*r_cm^2) + (x[,"n_03"] + x[,"n_33"])*log10(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_14"] + x[,"n_22"])*log10(2 - r_cm + 2*r_cm^2)
logL_cm <- (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + (x[,"n_02"] + x[,"n_04"] + x[,"n_32"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + (x[,"n_13"] + x[,"n_23"])*log(3 + 4*r_cm - 4*r_cm^2) + (x[,"n_12"] + x[,"n_24"])*log(3 - 3*r_cm + 2*r_cm^2) + (x[,"n_03"] + x[,"n_33"])*log(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_14"] + x[,"n_22"])*log(2 - r_cm + 2*r_cm^2)
logL_rc <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34) {
L <- (-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34)*(3*log(2) + 2*log(5)) + (n02 + n34)*(log(3 - 2*r) + log(2 + r)) + (n04 + n32)*(log(3 - r) + log(1 + 2*r)) + (n13 + n23)*log(39 + 4*r - 4*r^2) + (n12 + n24)*log(21 - 3*r + 2*r^2) + (n14 + n22)*log(20 - r + 2*r^2) + (n03 + n33)*log(11 - 4*r + 4*r^2)
return(L)}
interlogL_rc <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34) {
optimize(logL_rc,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rc <- -((x[,"n_04"] + x[,"n_32"])*log10(5)) - (x[,"n_02"] + x[,"n_34"])*log10(5) - (x[,"n_03"] + x[,"n_33"])*(log10(2) + log10(5)) - (x[,"n_14"] + x[,"n_22"])*(2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_24"])*(2*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_23"])*(3*log10(2) + log10(5)) + (x[,"n_02"] + x[,"n_34"])*(log10(3 - 2*r_rc) + log10(2 + r_rc)) + (x[,"n_04"] + x[,"n_32"])*(log10(3 - r_rc) + log10(1 + 2*r_rc)) + (x[,"n_13"] + x[,"n_23"])*log10(39 + 4*r_rc - 4*r_rc^2) + (x[,"n_12"] + x[,"n_24"])*log10(21 - 3*r_rc + 2*r_rc^2) + (x[,"n_14"] + x[,"n_22"])*log10(20 - r_rc + 2*r_rc^2) + (x[,"n_03"] + x[,"n_33"])*log10(11 - 4*r_rc + 4*r_rc^2)
logL_rc <- (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(3*log(2) + 2*log(5)) + (x[,"n_02"] + x[,"n_34"])*(log(3 - 2*r_rc) + log(2 + r_rc)) + (x[,"n_04"] + x[,"n_32"])*(log(3 - r_rc) + log(1 + 2*r_rc)) + (x[,"n_13"] + x[,"n_23"])*log(39 + 4*r_rc - 4*r_rc^2) + (x[,"n_12"] + x[,"n_24"])*log(21 - 3*r_rc + 2*r_rc^2) + (x[,"n_14"] + x[,"n_22"])*log(20 - r_rc + 2*r_rc^2) + (x[,"n_03"] + x[,"n_33"])*log(11 - 4*r_rc + 4*r_rc^2)
logL_rm <- function(r,n02,n04,n12,n13,n14,n22,n23,n24,n32,n34,n03,n33) {
L <- (-n02 - n04 - 2*n12 - 2*n13 - 2*n14 - 2*n22 - 2*n23 - 2*n24 - n32 - n34)*log(2) + 2*(-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34)*log(5) + (n04 + n32)*(log(pmax(1e-6,1 - r)) + log(3 - r)) + (n02 + n34)*(log(pmax(1e-6,r)) + log(2 + r)) + (n14 + n22)*log(13 - 5*r - 2*r^2) + (n12 + n24)*log(6 + 9*r - 2*r^2) + (n03 + n33)*log(1 + r - r^2) + (n13 + n23)*log(21 - 4*r + 4*r^2)
return(L)}
interlogL_rm <- function(n02,n04,n12,n13,n14,n22,n23,n24,n32,n34,n03,n33) {
optimize(logL_rm,c(0,0.5), n02,n04,n12,n13,n14,n22,n23,n24,n32,n34,n03,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_02"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_34"],x[,"n_03"],x[,"n_33"])
LOD_rm <- -((x[,"n_04"] + x[,"n_32"])*(-2*log10(2) + log10(5))) - (x[,"n_03"] + x[,"n_33"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_34"])*(-2*log10(2) + log10(5)) - (x[,"n_14"] + x[,"n_22"])*(log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_24"])*(log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_23"])*(2*log10(2) + log10(5)) + (x[,"n_04"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_rm)) + log10(3 - r_rm)) + (x[,"n_02"] + x[,"n_34"])*(log10(pmax(1e-6,r_rm)) + log10(2 + r_rm)) + (x[,"n_14"] + x[,"n_22"])*log10(13 - 5*r_rm - 2*r_rm^2) + (x[,"n_12"] + x[,"n_24"])*log10(6 + 9*r_rm - 2*r_rm^2) + (x[,"n_03"] + x[,"n_33"])*log10(1 + r_rm - r_rm^2) + (x[,"n_13"] + x[,"n_23"])*log10(21 - 4*r_rm + 4*r_rm^2)
logL_rm <- (-x[,"n_02"] - x[,"n_04"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_32"] - x[,"n_34"])*log(2) + 2*(-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + (x[,"n_04"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_rm)) + log(3 - r_rm)) + (x[,"n_02"] + x[,"n_34"])*(log(pmax(1e-6,r_rm)) + log(2 + r_rm)) + (x[,"n_14"] + x[,"n_22"])*log(13 - 5*r_rm - 2*r_rm^2) + (x[,"n_12"] + x[,"n_24"])*log(6 + 9*r_rm - 2*r_rm^2) + (x[,"n_03"] + x[,"n_33"])*log(1 + r_rm - r_rm^2) + (x[,"n_13"] + x[,"n_23"])*log(21 - 4*r_rm + 4*r_rm^2)

return(list(
r_mat = cbind(r_cc,r_cm,r_rc,r_rm,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_rc,LOD_rm,0),
logL_mat = cbind(logL_cc,logL_cm,logL_rc,logL_rm,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","repulsion coupling","repulsion mixed","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_2.1 <- function(x){
logL_cc <- function(r,n01,n02,n10,n11,n12,n13,n20,n21,n22,n23,n31,n32,n00,n03,n30,n33) {
L <- (-n01 - n02 - n10 - 2*n11 - 2*n12 - n13 - n20 - 2*n21 - 2*n22 - n23 - n31 - n32)*log(2) + 2*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(5) + 2*(n00 + n33)*log(pmax(1e-6,1 - r)) + 2*(n03 + n30)*log(pmax(1e-6,r)) + (n02 + n13 + n20 + n31)*(log(5 - 2*r) + log(pmax(1e-6,r))) + (n01 + n10 + n23 + n32)*log(3 - r - 2*r^2) + (n11 + n22)*log(21 - 12*r + 4*r^2) + (n12 + n21)*log(13 + 4*r + 4*r^2)
return(L)}
interlogL_cc <- function(n01,n02,n10,n11,n12,n13,n20,n21,n22,n23,n31,n32,n00,n03,n30,n33) {
optimize(logL_cc,c(0,0.5), n01,n02,n10,n11,n12,n13,n20,n21,n22,n23,n31,n32,n00,n03,n30,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_00"],x[,"n_03"],x[,"n_30"],x[,"n_33"])
LOD_cc <- -4*(x[,"n_12"] + x[,"n_21"])*log10(2) - 4*(x[,"n_11"] + x[,"n_22"])*log10(2) + 2*(x[,"n_03"] + x[,"n_30"])*log10(2) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log10(2) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(2) + 2*(x[,"n_00"] + x[,"n_33"])*log10(2) + 2*(x[,"n_00"] + x[,"n_33"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_03"] + x[,"n_30"])*log10(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(5 - 2*r_cc) + log10(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(3 - r_cc - 2*r_cc^2) + (x[,"n_11"] + x[,"n_22"])*log10(21 - 12*r_cc + 4*r_cc^2) + (x[,"n_12"] + x[,"n_21"])*log10(13 + 4*r_cc + 4*r_cc^2)
logL_cc <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(5) + 2*(x[,"n_00"] + x[,"n_33"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_03"] + x[,"n_30"])*log(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(5 - 2*r_cc) + log(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log(3 - r_cc - 2*r_cc^2) + (x[,"n_11"] + x[,"n_22"])*log(21 - 12*r_cc + 4*r_cc^2) + (x[,"n_12"] + x[,"n_21"])*log(13 + 4*r_cc + 4*r_cc^2)
logL_cr <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
L <- (-2*n00 - 3*n01 - 3*n02 - 2*n03 - 2*n10 - 3*n11 - 3*n12 - 2*n13 - 2*n20 - 3*n21 - 3*n22 - 2*n23 - 2*n30 - 3*n31 - 3*n32 - 2*n33)*log(2) + 2*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(5) + (n03 + n30)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n00 + n33)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n11 + n22)*log(33 - 4*r^2) + (n12 + n21)*log(29 + 8*r - 4*r^2) + (n10 + n23)*log(6 - 5*r + 2*r^2) + (n13 + n20)*log(3 + r + 2*r^2) + (n01 + n02 + n31 + n32)*log(9 - 4*r + 4*r^2)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"])
LOD_cr <- -2*(x[,"n_13"] + x[,"n_20"])*log10(2) - 5*(x[,"n_12"] + x[,"n_21"])*log10(2) - 5*(x[,"n_11"] + x[,"n_22"])*log10(2) - 2*(x[,"n_10"] + x[,"n_23"])*log10(2) - 3*(x[,"n_01"] + x[,"n_02"] + x[,"n_31"] + x[,"n_32"])*log10(2) + (x[,"n_03"] + x[,"n_30"])*(log10(3 - 2*r_cr) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cr)) + log10(1 + 2*r_cr)) + (x[,"n_11"] + x[,"n_22"])*log10(33 - 4*r_cr^2) + (x[,"n_12"] + x[,"n_21"])*log10(29 + 8*r_cr - 4*r_cr^2) + (x[,"n_10"] + x[,"n_23"])*log10(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_13"] + x[,"n_20"])*log10(3 + r_cr + 2*r_cr^2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_31"] + x[,"n_32"])*log10(9 - 4*r_cr + 4*r_cr^2)
logL_cr <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 2*x[,"n_33"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(5) + (x[,"n_03"] + x[,"n_30"])*(log(3 - 2*r_cr) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cr)) + log(1 + 2*r_cr)) + (x[,"n_11"] + x[,"n_22"])*log(33 - 4*r_cr^2) + (x[,"n_12"] + x[,"n_21"])*log(29 + 8*r_cr - 4*r_cr^2) + (x[,"n_10"] + x[,"n_23"])*log(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_13"] + x[,"n_20"])*log(3 + r_cr + 2*r_cr^2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_31"] + x[,"n_32"])*log(9 - 4*r_cr + 4*r_cr^2)
logL_rc <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n20 - 3*n21 - 3*n22 - 3*n23 - 2*n30 - 2*n31 - 2*n32 - 2*n33)*log(2) + 2*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(5) + (n03 + n30)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n00 + n33)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n11 + n22)*log(33 - 4*r^2) + (n12 + n21)*log(29 + 8*r - 4*r^2) + (n01 + n32)*log(6 - 5*r + 2*r^2) + (n02 + n31)*log(3 + r + 2*r^2) + (n10 + n13 + n20 + n23)*log(9 - 4*r + 4*r^2)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"])
LOD_rc <- -5*(x[,"n_12"] + x[,"n_21"])*log10(2) - 5*(x[,"n_11"] + x[,"n_22"])*log10(2) - 3*(x[,"n_10"] + x[,"n_13"] + x[,"n_20"] + x[,"n_23"])*log10(2) - 2*(x[,"n_02"] + x[,"n_31"])*log10(2) - 2*(x[,"n_01"] + x[,"n_32"])*log10(2) + (x[,"n_03"] + x[,"n_30"])*(log10(3 - 2*r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_11"] + x[,"n_22"])*log10(33 - 4*r_rc^2) + (x[,"n_12"] + x[,"n_21"])*log10(29 + 8*r_rc - 4*r_rc^2) + (x[,"n_01"] + x[,"n_32"])*log10(6 - 5*r_rc + 2*r_rc^2) + (x[,"n_02"] + x[,"n_31"])*log10(3 + r_rc + 2*r_rc^2) + (x[,"n_10"] + x[,"n_13"] + x[,"n_20"] + x[,"n_23"])*log10(9 - 4*r_rc + 4*r_rc^2)
logL_rc <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(5) + (x[,"n_03"] + x[,"n_30"])*(log(3 - 2*r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_11"] + x[,"n_22"])*log(33 - 4*r_rc^2) + (x[,"n_12"] + x[,"n_21"])*log(29 + 8*r_rc - 4*r_rc^2) + (x[,"n_01"] + x[,"n_32"])*log(6 - 5*r_rc + 2*r_rc^2) + (x[,"n_02"] + x[,"n_31"])*log(3 + r_rc + 2*r_rc^2) + (x[,"n_10"] + x[,"n_13"] + x[,"n_20"] + x[,"n_23"])*log(9 - 4*r_rc + 4*r_rc^2)
logL_rr <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(5)) + 2*(n03 + n30)*log(3 - 2*r) + 2*(n00 + n33)*log(1 + 2*r) + (n02 + n13 + n20 + n31)*log(21 - 8*r - 4*r^2) + (n01 + n10 + n23 + n32)*log(9 + 16*r - 4*r^2) + (n12 + n21)*log(73 - 20*r + 4*r^2) + (n11 + n22)*log(57 + 12*r + 4*r^2)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"])
LOD_rr <- -6*(x[,"n_12"] + x[,"n_21"])*log10(2) - 6*(x[,"n_11"] + x[,"n_22"])*log10(2) - 2*(x[,"n_03"] + x[,"n_30"])*log10(2) - 4*(x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log10(2) - 4*(x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(2) - 2*(x[,"n_00"] + x[,"n_33"])*log10(2) + 2*(x[,"n_03"] + x[,"n_30"])*log10(3 - 2*r_rr) + 2*(x[,"n_00"] + x[,"n_33"])*log10(1 + 2*r_rr) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log10(21 - 8*r_rr - 4*r_rr^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log10(9 + 16*r_rr - 4*r_rr^2) + (x[,"n_12"] + x[,"n_21"])*log10(73 - 20*r_rr + 4*r_rr^2) + (x[,"n_11"] + x[,"n_22"])*log10(57 + 12*r_rr + 4*r_rr^2)
logL_rr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(5)) + 2*(x[,"n_03"] + x[,"n_30"])*log(3 - 2*r_rr) + 2*(x[,"n_00"] + x[,"n_33"])*log(1 + 2*r_rr) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*log(21 - 8*r_rr - 4*r_rr^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*log(9 + 16*r_rr - 4*r_rr^2) + (x[,"n_12"] + x[,"n_21"])*log(73 - 20*r_rr + 4*r_rr^2) + (x[,"n_11"] + x[,"n_22"])*log(57 + 12*r_rr + 4*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_2.2 <- function(x){
logL_cc <- function(r,n01,n02,n03,n11,n12,n13,n21,n22,n23,n31,n32,n33,n10,n14,n20,n24,n00,n04,n30,n34) {
L <- (-n01 + 2*n02 - n03 - n11 - n12 - n13 - n21 - n22 - n23 - n31 + 2*n32 - n33)*log(2) + (n10 + n14 + n20 + n24)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(5) + 3*(n00 + n34)*log(pmax(1e-6,1 - r)) + 3*(n04 + n30)*log(pmax(1e-6,r)) + (n02 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n31)*(log(7 - 4*r) + 2*log(pmax(1e-6,r))) + (n14 + n20)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n33)*(2*log(pmax(1e-6,1 - r)) + log(3 + 4*r)) + (n12 + n22)*log(11 - 8*r + 8*r^2) + (n13 + n21)*(log(pmax(1e-6,r)) + log(16 - 19*r + 12*r^2)) + (n11 + n23)*log(9 - 14*r + 17*r^2 - 12*r^3)
return(L)}
interlogL_cc <- function(n01,n02,n03,n11,n12,n13,n21,n22,n23,n31,n32,n33,n10,n14,n20,n24,n00,n04,n30,n34) {
optimize(logL_cc,c(0,0.5), n01,n02,n03,n11,n12,n13,n21,n22,n23,n31,n32,n33,n10,n14,n20,n24,n00,n04,n30,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_10"],x[,"n_14"],x[,"n_20"],x[,"n_24"],x[,"n_00"],x[,"n_04"],x[,"n_30"],x[,"n_34"])
LOD_cc <- 3*(x[,"n_14"] + x[,"n_20"])*log10(2) + 3*(x[,"n_10"] + x[,"n_24"])*log10(2) + 3*(x[,"n_04"] + x[,"n_30"])*log10(2) + 2*(x[,"n_02"] + x[,"n_32"])*log10(2) + 3*(x[,"n_00"] + x[,"n_34"])*log10(2) - 2*(x[,"n_12"] + x[,"n_22"])*log10(3) - (x[,"n_03"] + x[,"n_31"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_33"])*(-2*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_21"])*(-2*log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_23"])*(-2*log10(2) + log10(19)) + 3*(x[,"n_00"] + x[,"n_34"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_04"] + x[,"n_30"])*log10(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_24"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_31"])*(log10(7 - 4*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_20"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_33"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(3 + 4*r_cc)) + (x[,"n_12"] + x[,"n_22"])*log10(11 - 8*r_cc + 8*r_cc^2) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(16 - 19*r_cc + 12*r_cc^2)) + (x[,"n_11"] + x[,"n_23"])*log10(9 - 14*r_cc + 17*r_cc^2 - 12*r_cc^3)
logL_cc <- (-x[,"n_01"] + 2*x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] + 2*x[,"n_32"] - x[,"n_33"])*log(2) + (x[,"n_10"] + x[,"n_14"] + x[,"n_20"] + x[,"n_24"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + 3*(x[,"n_00"] + x[,"n_34"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_04"] + x[,"n_30"])*log(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_24"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_31"])*(log(7 - 4*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_20"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_33"])*(2*log(pmax(1e-6,1 - r_cc)) + log(3 + 4*r_cc)) + (x[,"n_12"] + x[,"n_22"])*log(11 - 8*r_cc + 8*r_cc^2) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(16 - 19*r_cc + 12*r_cc^2)) + (x[,"n_11"] + x[,"n_23"])*log(9 - 14*r_cc + 17*r_cc^2 - 12*r_cc^3)
logL_cm <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-2*n00 - 3*n01 - 3*n02 - 3*n03 - 2*n04 - 2*n10 - 3*n11 - 3*n12 - 3*n13 - 2*n14 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 2*n24 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 2*n34)*log(2) + 2*(n02 + n32)*log(3) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - n12 - 2*n13 - 2*n14 - 2*n20 - 2*n21 - n22 - 2*n23 - 2*n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(5) + (n12 + n22)*log(7) + (n04 + n30)*(log(2 - r) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(pmax(1e-6,1 - r)) + log(1 - r^2)) + (n14 + n20)*(log(pmax(1e-6,r)) + log(5 - 5*r + 3*r^2)) + (n03 + n31)*(log(pmax(1e-6,r)) + log(12 - 7*r + 4*r^2)) + (n13 + n21)*log(15 + 2*r + 19*r^2 - 12*r^3) + (n01 + n33)*log(9 - 10*r + 5*r^2 - 4*r^3) + (n10 + n24)*log(3 - 4*r + 4*r^2 - 3*r^3) + (n11 + n23)*log(24 - 4*r - 17*r^2 + 12*r^3)
return(L)}
interlogL_cm <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_cm,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cm <- -((x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(3))) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_21"])*(-2*log10(2) + log10(7) + log10(11)) - (x[,"n_11"] + x[,"n_23"])*(-2*log10(2) + log10(7) + log10(11)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(13)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(13)) - (x[,"n_03"] + x[,"n_31"])*(-2*log10(2) + log10(19)) - (x[,"n_01"] + x[,"n_33"])*(-2*log10(2) + log10(19)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - r_cm) + 2*log10(pmax(1e-6,r_cm))) + (x[,"n_00"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cm)) + log10(1 - r_cm^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(pmax(1e-6,r_cm)) + log10(5 - 5*r_cm + 3*r_cm^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm)) + log10(12 - 7*r_cm + 4*r_cm^2)) + (x[,"n_13"] + x[,"n_21"])*log10(15 + 2*r_cm + 19*r_cm^2 - 12*r_cm^3) + (x[,"n_01"] + x[,"n_33"])*log10(9 - 10*r_cm + 5*r_cm^2 - 4*r_cm^3) + (x[,"n_10"] + x[,"n_24"])*log10(3 - 4*r_cm + 4*r_cm^2 - 3*r_cm^3) + (x[,"n_11"] + x[,"n_23"])*log10(24 - 4*r_cm - 17*r_cm^2 + 12*r_cm^3)
logL_cm <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 2*x[,"n_34"])*log(2) + 2*(x[,"n_02"] + x[,"n_32"])*log(3) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(5) + (x[,"n_12"] + x[,"n_22"])*log(7) + (x[,"n_04"] + x[,"n_30"])*(log(2 - r_cm) + 2*log(pmax(1e-6,r_cm))) + (x[,"n_00"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_cm)) + log(1 - r_cm^2)) + (x[,"n_14"] + x[,"n_20"])*(log(pmax(1e-6,r_cm)) + log(5 - 5*r_cm + 3*r_cm^2)) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,r_cm)) + log(12 - 7*r_cm + 4*r_cm^2)) + (x[,"n_13"] + x[,"n_21"])*log(15 + 2*r_cm + 19*r_cm^2 - 12*r_cm^3) + (x[,"n_01"] + x[,"n_33"])*log(9 - 10*r_cm + 5*r_cm^2 - 4*r_cm^3) + (x[,"n_10"] + x[,"n_24"])*log(3 - 4*r_cm + 4*r_cm^2 - 3*r_cm^3) + (x[,"n_11"] + x[,"n_23"])*log(24 - 4*r_cm - 17*r_cm^2 + 12*r_cm^3)
logL_cr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-n00 - 2*n01 - 2*n02 - 2*n03 - n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - n30 - 2*n31 - 2*n32 - 2*n33 - n34)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(log(3) + 2*log(5)) + (n00 + n34)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 + r)) + (n12 + n22)*log(51 + 8*r - 8*r^2) + (n04 + n30)*(log(pmax(1e-6,r)) + log(3 - 4*r + r^2)) + (n02 + n32)*log(15 - 8*r + 8*r^2) + (n11 + n23)*log(33 - 14*r + 17*r^2 - 12*r^3) + (n03 + n31)*log(9 - 6*r + 7*r^2 - 4*r^3) + (n14 + n20)*log(3 - 4*r + 7*r^2 - 3*r^3) + (n10 + n24)*log(3 - r - 2*r^2 + 3*r^3) + (n01 + n33)*log(6 + 4*r - 5*r^2 + 4*r^3) + (n13 + n21)*log(24 + 16*r - 19*r^2 + 12*r^3)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cr <- -((x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_32"])*log10(13) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(19)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(19)) - (x[,"n_13"] + x[,"n_21"])*(-2*log10(2) + log10(5) + log10(23)) - (x[,"n_11"] + x[,"n_23"])*(-2*log10(2) + log10(5) + log10(23)) - (x[,"n_03"] + x[,"n_31"])*(-2*log10(2) + log10(29)) - (x[,"n_01"] + x[,"n_33"])*(-2*log10(2) + log10(29)) - (x[,"n_12"] + x[,"n_22"])*log10(53) + (x[,"n_00"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr)) + log10(2 + r_cr)) + (x[,"n_12"] + x[,"n_22"])*log10(51 + 8*r_cr - 8*r_cr^2) + (x[,"n_04"] + x[,"n_30"])*(log10(pmax(1e-6,r_cr)) + log10(3 - 4*r_cr + r_cr^2)) + (x[,"n_02"] + x[,"n_32"])*log10(15 - 8*r_cr + 8*r_cr^2) + (x[,"n_11"] + x[,"n_23"])*log10(33 - 14*r_cr + 17*r_cr^2 - 12*r_cr^3) + (x[,"n_03"] + x[,"n_31"])*log10(9 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3) + (x[,"n_14"] + x[,"n_20"])*log10(3 - 4*r_cr + 7*r_cr^2 - 3*r_cr^3) + (x[,"n_10"] + x[,"n_24"])*log10(3 - r_cr - 2*r_cr^2 + 3*r_cr^3) + (x[,"n_01"] + x[,"n_33"])*log10(6 + 4*r_cr - 5*r_cr^2 + 4*r_cr^3) + (x[,"n_13"] + x[,"n_21"])*log10(24 + 16*r_cr - 19*r_cr^2 + 12*r_cr^3)
logL_cr <- (-x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - x[,"n_34"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr)) + log(2 + r_cr)) + (x[,"n_12"] + x[,"n_22"])*log(51 + 8*r_cr - 8*r_cr^2) + (x[,"n_04"] + x[,"n_30"])*(log(pmax(1e-6,r_cr)) + log(3 - 4*r_cr + r_cr^2)) + (x[,"n_02"] + x[,"n_32"])*log(15 - 8*r_cr + 8*r_cr^2) + (x[,"n_11"] + x[,"n_23"])*log(33 - 14*r_cr + 17*r_cr^2 - 12*r_cr^3) + (x[,"n_03"] + x[,"n_31"])*log(9 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3) + (x[,"n_14"] + x[,"n_20"])*log(3 - 4*r_cr + 7*r_cr^2 - 3*r_cr^3) + (x[,"n_10"] + x[,"n_24"])*log(3 - r_cr - 2*r_cr^2 + 3*r_cr^3) + (x[,"n_01"] + x[,"n_33"])*log(6 + 4*r_cr - 5*r_cr^2 + 4*r_cr^3) + (x[,"n_13"] + x[,"n_21"])*log(24 + 16*r_cr - 19*r_cr^2 + 12*r_cr^3)
logL_rc <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-2*n00 - n01 - 2*n02 - n03 - 2*n04 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24 - 2*n30 - n31 - 2*n32 - n33 - 2*n34)*log(2) + (n10 + n14 + n20 + n24)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(5) + (n04 + n30)*(log(3 - 2*r) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(2*log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n02 + n32)*log(3 + 4*r - 4*r^2) + (n14 + n20)*(log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n03 + n31)*(log(pmax(1e-6,r)) + log(3 - 2*r + 2*r^2)) + (n12 + n22)*log(19 - 4*r + 4*r^2) + (n13 + n21)*log(9 - 4*r + 16*r^2 - 12*r^3) + (n10 + n24)*log(1 - 2*r + 3*r^2 - 2*r^3) + (n01 + n33)*log(3 - 5*r + 4*r^2 - 2*r^3) + (n11 + n23)*log(9 + 8*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rc <- (x[,"n_14"] + x[,"n_20"])*log10(2) + (x[,"n_10"] + x[,"n_24"])*log10(2) + (x[,"n_04"] + x[,"n_30"])*log10(2) - 2*(x[,"n_02"] + x[,"n_32"])*log10(2) + (x[,"n_00"] + x[,"n_34"])*log10(2) - (x[,"n_12"] + x[,"n_22"])*(log10(2) + 2*log10(3)) - (x[,"n_03"] + x[,"n_31"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_33"])*(-2*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(19)) + (x[,"n_04"] + x[,"n_30"])*(log10(3 - 2*r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_34"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_02"] + x[,"n_32"])*log10(3 + 4*r_rc - 4*r_rc^2) + (x[,"n_14"] + x[,"n_20"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 3*r_rc + 2*r_rc^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(3 - 2*r_rc + 2*r_rc^2)) + (x[,"n_12"] + x[,"n_22"])*log10(19 - 4*r_rc + 4*r_rc^2) + (x[,"n_13"] + x[,"n_21"])*log10(9 - 4*r_rc + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_10"] + x[,"n_24"])*log10(1 - 2*r_rc + 3*r_rc^2 - 2*r_rc^3) + (x[,"n_01"] + x[,"n_33"])*log10(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3) + (x[,"n_11"] + x[,"n_23"])*log10(9 + 8*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rc <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - x[,"n_31"] - 2*x[,"n_32"] - x[,"n_33"] - 2*x[,"n_34"])*log(2) + (x[,"n_10"] + x[,"n_14"] + x[,"n_20"] + x[,"n_24"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + (x[,"n_04"] + x[,"n_30"])*(log(3 - 2*r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_34"])*(2*log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_02"] + x[,"n_32"])*log(3 + 4*r_rc - 4*r_rc^2) + (x[,"n_14"] + x[,"n_20"])*(log(pmax(1e-6,r_rc)) + log(2 - 3*r_rc + 2*r_rc^2)) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(3 - 2*r_rc + 2*r_rc^2)) + (x[,"n_12"] + x[,"n_22"])*log(19 - 4*r_rc + 4*r_rc^2) + (x[,"n_13"] + x[,"n_21"])*log(9 - 4*r_rc + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_10"] + x[,"n_24"])*log(1 - 2*r_rc + 3*r_rc^2 - 2*r_rc^3) + (x[,"n_01"] + x[,"n_33"])*log(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3) + (x[,"n_11"] + x[,"n_23"])*log(9 + 8*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rm <- function(r,n02,n32,n00,n01,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n33,n34) {
L <- (n02 + n32)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(4*log(2) + 2*log(5)) + (n04 + n30)*(log(3 - 2*r) + log(2 - r) + log(pmax(1e-6,r))) + (n12 + n22)*log(67 + 12*r - 12*r^2) + (n00 + n34)*(log(1 + 2*r) + log(1 - r^2)) + (n02 + n32)*log(7 - 4*r + 4*r^2) + (n11 + n23)*log(39 - 8*r + 20*r^2 - 12*r^3) + (n14 + n20)*log(9 - 10*r + 13*r^2 - 6*r^3) + (n03 + n31)*log(9 + 4*r^2 - 4*r^3) + (n01 + n33)*log(9 + 4*r - 8*r^2 + 4*r^3) + (n10 + n24)*log(6 + 2*r - 5*r^2 + 6*r^3) + (n13 + n21)*log(39 + 4*r - 16*r^2 + 12*r^3)
return(L)}
interlogL_rm <- function(n02,n32,n00,n01,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n33,n34) {
optimize(logL_rm,c(0,0.5), n02,n32,n00,n01,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_02"],x[,"n_32"],x[,"n_00"],x[,"n_01"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_33"],x[,"n_34"])
LOD_rm <- -((x[,"n_04"] + x[,"n_30"])*(-log10(2) + log10(3))) - (x[,"n_00"] + x[,"n_34"])*(-log10(2) + log10(3)) - (x[,"n_02"] + x[,"n_32"])*(log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_22"])*(log10(2) + log10(5) + log10(7)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(7) + log10(11)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(7) + log10(11)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(13)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(13)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(19)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(19)) + (x[,"n_04"] + x[,"n_30"])*(log10(3 - 2*r_rm) + log10(2 - r_rm) + log10(pmax(1e-6,r_rm))) + (x[,"n_12"] + x[,"n_22"])*log10(67 + 12*r_rm - 12*r_rm^2) + (x[,"n_00"] + x[,"n_34"])*(log10(1 + 2*r_rm) + log10(1 - r_rm^2)) + (x[,"n_02"] + x[,"n_32"])*log10(7 - 4*r_rm + 4*r_rm^2) + (x[,"n_11"] + x[,"n_23"])*log10(39 - 8*r_rm + 20*r_rm^2 - 12*r_rm^3) + (x[,"n_14"] + x[,"n_20"])*log10(9 - 10*r_rm + 13*r_rm^2 - 6*r_rm^3) + (x[,"n_03"] + x[,"n_31"])*log10(9 + 4*r_rm^2 - 4*r_rm^3) + (x[,"n_01"] + x[,"n_33"])*log10(9 + 4*r_rm - 8*r_rm^2 + 4*r_rm^3) + (x[,"n_10"] + x[,"n_24"])*log10(6 + 2*r_rm - 5*r_rm^2 + 6*r_rm^3) + (x[,"n_13"] + x[,"n_21"])*log10(39 + 4*r_rm - 16*r_rm^2 + 12*r_rm^3)
logL_rm <- (x[,"n_02"] + x[,"n_32"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(4*log(2) + 2*log(5)) + (x[,"n_04"] + x[,"n_30"])*(log(3 - 2*r_rm) + log(2 - r_rm) + log(pmax(1e-6,r_rm))) + (x[,"n_12"] + x[,"n_22"])*log(67 + 12*r_rm - 12*r_rm^2) + (x[,"n_00"] + x[,"n_34"])*(log(1 + 2*r_rm) + log(1 - r_rm^2)) + (x[,"n_02"] + x[,"n_32"])*log(7 - 4*r_rm + 4*r_rm^2) + (x[,"n_11"] + x[,"n_23"])*log(39 - 8*r_rm + 20*r_rm^2 - 12*r_rm^3) + (x[,"n_14"] + x[,"n_20"])*log(9 - 10*r_rm + 13*r_rm^2 - 6*r_rm^3) + (x[,"n_03"] + x[,"n_31"])*log(9 + 4*r_rm^2 - 4*r_rm^3) + (x[,"n_01"] + x[,"n_33"])*log(9 + 4*r_rm - 8*r_rm^2 + 4*r_rm^3) + (x[,"n_10"] + x[,"n_24"])*log(6 + 2*r_rm - 5*r_rm^2 + 6*r_rm^3) + (x[,"n_13"] + x[,"n_21"])*log(39 + 4*r_rm - 16*r_rm^2 + 12*r_rm^3)
logL_rr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(3*log(2) + log(3) + 2*log(5)) + (n00 + n34)*(log(pmax(1e-6,r)) + log(2 + r) + log(1 + 2*r)) + (n02 + n32)*log(21 + 20*r - 20*r^2) + (n04 + n30)*(log(3 - 2*r) + log(3 - 4*r + r^2)) + (n12 + n22)*log(111 - 20*r + 20*r^2) + (n13 + n21)*log(75 - 40*r + 16*r^2 - 12*r^3) + (n10 + n24)*log(3 + 14*r + r^2 - 6*r^3) + (n01 + n33)*log(3 + 20*r + 8*r^2 - 4*r^3) + (n03 + n31)*log(27 - 24*r - 4*r^2 + 4*r^3) + (n14 + n20)*log(12 + 2*r - 17*r^2 + 6*r^3) + (n11 + n23)*log(39 + 44*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rr <- -((x[,"n_04"] + x[,"n_30"])*(-log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_34"])*(-log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_32"])*(log10(2) + log10(13)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(19)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(19)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(5) + log10(23)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(5) + log10(23)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(29)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(29)) - (x[,"n_12"] + x[,"n_22"])*(log10(2) + log10(53)) + (x[,"n_00"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(2 + r_rr) + log10(1 + 2*r_rr)) + (x[,"n_02"] + x[,"n_32"])*log10(21 + 20*r_rr - 20*r_rr^2) + (x[,"n_04"] + x[,"n_30"])*(log10(3 - 2*r_rr) + log10(3 - 4*r_rr + r_rr^2)) + (x[,"n_12"] + x[,"n_22"])*log10(111 - 20*r_rr + 20*r_rr^2) + (x[,"n_13"] + x[,"n_21"])*log10(75 - 40*r_rr + 16*r_rr^2 - 12*r_rr^3) + (x[,"n_10"] + x[,"n_24"])*log10(3 + 14*r_rr + r_rr^2 - 6*r_rr^3) + (x[,"n_01"] + x[,"n_33"])*log10(3 + 20*r_rr + 8*r_rr^2 - 4*r_rr^3) + (x[,"n_03"] + x[,"n_31"])*log10(27 - 24*r_rr - 4*r_rr^2 + 4*r_rr^3) + (x[,"n_14"] + x[,"n_20"])*log10(12 + 2*r_rr - 17*r_rr^2 + 6*r_rr^3) + (x[,"n_11"] + x[,"n_23"])*log10(39 + 44*r_rr - 20*r_rr^2 + 12*r_rr^3)
logL_rr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(3*log(2) + log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(2 + r_rr) + log(1 + 2*r_rr)) + (x[,"n_02"] + x[,"n_32"])*log(21 + 20*r_rr - 20*r_rr^2) + (x[,"n_04"] + x[,"n_30"])*(log(3 - 2*r_rr) + log(3 - 4*r_rr + r_rr^2)) + (x[,"n_12"] + x[,"n_22"])*log(111 - 20*r_rr + 20*r_rr^2) + (x[,"n_13"] + x[,"n_21"])*log(75 - 40*r_rr + 16*r_rr^2 - 12*r_rr^3) + (x[,"n_10"] + x[,"n_24"])*log(3 + 14*r_rr + r_rr^2 - 6*r_rr^3) + (x[,"n_01"] + x[,"n_33"])*log(3 + 20*r_rr + 8*r_rr^2 - 4*r_rr^3) + (x[,"n_03"] + x[,"n_31"])*log(27 - 24*r_rr - 4*r_rr^2 + 4*r_rr^3) + (x[,"n_14"] + x[,"n_20"])*log(12 + 2*r_rr - 17*r_rr^2 + 6*r_rr^3) + (x[,"n_11"] + x[,"n_23"])*log(39 + 44*r_rr - 20*r_rr^2 + 12*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_2.3 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-2*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 2*n05 - 2*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35)*log(2) + (n10 + n15 + n20 + n25)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*log(5) + 3*(n00 + n35)*log(pmax(1e-6,1 - r)) + 3*(n05 + n30)*log(pmax(1e-6,r)) + (n10 + n25)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n31)*(log(11 - 2*r) + 2*log(pmax(1e-6,r))) + (n15 + n20)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n34)*(2*log(pmax(1e-6,1 - r)) + log(9 + 2*r)) + (n03 + n32)*(log(pmax(1e-6,r)) + log(18 - 5*r - 4*r^2)) + (n14 + n21)*(log(pmax(1e-6,r)) + log(26 - 17*r + 6*r^2)) + (n12 + n23)*log(41 - 34*r + 29*r^2 - 12*r^3) + (n11 + n24)*log(15 - 10*r + r^2 - 6*r^3) + (n02 + n33)*log(9 + 4*r - 17*r^2 + 4*r^3) + (n13 + n22)*log(24 + 12*r - 7*r^2 + 12*r^3)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_cc <- 3*(x[,"n_15"] + x[,"n_20"])*log10(2) + 3*(x[,"n_10"] + x[,"n_25"])*log10(2) + 3*(x[,"n_05"] + x[,"n_30"])*log10(2) + 3*(x[,"n_00"] + x[,"n_35"])*log10(2) - (x[,"n_04"] + x[,"n_31"])*(-log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_34"])*(-log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_22"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_12"] + x[,"n_23"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_14"] + x[,"n_21"])*(-log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_24"])*(-log10(2) + log10(19)) - (x[,"n_03"] + x[,"n_32"])*(-2*log10(2) + log10(29)) - (x[,"n_02"] + x[,"n_33"])*(-2*log10(2) + log10(29)) + 3*(x[,"n_00"] + x[,"n_35"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_05"] + x[,"n_30"])*log10(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_25"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_31"])*(log10(11 - 2*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_20"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_34"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(9 + 2*r_cc)) + (x[,"n_03"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(18 - 5*r_cc - 4*r_cc^2)) + (x[,"n_14"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(26 - 17*r_cc + 6*r_cc^2)) + (x[,"n_12"] + x[,"n_23"])*log10(41 - 34*r_cc + 29*r_cc^2 - 12*r_cc^3) + (x[,"n_11"] + x[,"n_24"])*log10(15 - 10*r_cc + r_cc^2 - 6*r_cc^3) + (x[,"n_02"] + x[,"n_33"])*log10(9 + 4*r_cc - 17*r_cc^2 + 4*r_cc^3) + (x[,"n_13"] + x[,"n_22"])*log10(24 + 12*r_cc - 7*r_cc^2 + 12*r_cc^3)
logL_cc <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"])*log(2) + (x[,"n_10"] + x[,"n_15"] + x[,"n_20"] + x[,"n_25"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(5) + 3*(x[,"n_00"] + x[,"n_35"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_05"] + x[,"n_30"])*log(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_25"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_31"])*(log(11 - 2*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_20"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_34"])*(2*log(pmax(1e-6,1 - r_cc)) + log(9 + 2*r_cc)) + (x[,"n_03"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(18 - 5*r_cc - 4*r_cc^2)) + (x[,"n_14"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(26 - 17*r_cc + 6*r_cc^2)) + (x[,"n_12"] + x[,"n_23"])*log(41 - 34*r_cc + 29*r_cc^2 - 12*r_cc^3) + (x[,"n_11"] + x[,"n_24"])*log(15 - 10*r_cc + r_cc^2 - 6*r_cc^3) + (x[,"n_02"] + x[,"n_33"])*log(9 + 4*r_cc - 17*r_cc^2 + 4*r_cc^3) + (x[,"n_13"] + x[,"n_22"])*log(24 + 12*r_cc - 7*r_cc^2 + 12*r_cc^3)
logL_cm <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n05 - 2*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - n35)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*log(5) + (n00 + n35)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n05 + n30)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n31)*(log(pmax(1e-6,r)) + log(9 - 5*r + 2*r^2)) + (n15 + n20)*(log(pmax(1e-6,r)) + log(5 - 8*r + 6*r^2)) + (n14 + n21)*log(21 + 8*r + 22*r^2 - 12*r^3) + (n13 + n22)*log(27 + 2*r + 10*r^2 - 8*r^3) + (n10 + n25)*log(3 - 7*r + 10*r^2 - 6*r^3) + (n02 + n33)*log(15 - 11*r + 9*r^2 - 4*r^3) + (n01 + n34)*log(6 - 5*r + r^2 - 2*r^3) + (n03 + n32)*log(9 + 5*r - 3*r^2 + 4*r^3) + (n12 + n23)*log(31 + 2*r - 14*r^2 + 8*r^3) + (n11 + n24)*log(39 - 16*r - 14*r^2 + 12*r^3)
return(L)}
interlogL_cm <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_cm,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_cm <- 3*(x[,"n_05"] + x[,"n_30"])*log10(2) + 3*(x[,"n_00"] + x[,"n_35"])*log10(2) - (x[,"n_15"] + x[,"n_20"])*(-2*log10(2) + log10(5)) - (x[,"n_10"] + x[,"n_25"])*(-2*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_32"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_33"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_04"] + x[,"n_31"])*(-log10(2) + log10(7)) - (x[,"n_01"] + x[,"n_34"])*(-log10(2) + log10(7)) - (x[,"n_14"] + x[,"n_21"])*log10(29) - (x[,"n_11"] + x[,"n_24"])*log10(29) - (x[,"n_13"] + x[,"n_22"])*(-log10(2) + log10(59)) - (x[,"n_12"] + x[,"n_23"])*(-log10(2) + log10(59)) + (x[,"n_00"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + (x[,"n_05"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_cm)) + 2*log10(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm)) + log10(9 - 5*r_cm + 2*r_cm^2)) + (x[,"n_15"] + x[,"n_20"])*(log10(pmax(1e-6,r_cm)) + log10(5 - 8*r_cm + 6*r_cm^2)) + (x[,"n_14"] + x[,"n_21"])*log10(21 + 8*r_cm + 22*r_cm^2 - 12*r_cm^3) + (x[,"n_13"] + x[,"n_22"])*log10(27 + 2*r_cm + 10*r_cm^2 - 8*r_cm^3) + (x[,"n_10"] + x[,"n_25"])*log10(3 - 7*r_cm + 10*r_cm^2 - 6*r_cm^3) + (x[,"n_02"] + x[,"n_33"])*log10(15 - 11*r_cm + 9*r_cm^2 - 4*r_cm^3) + (x[,"n_01"] + x[,"n_34"])*log10(6 - 5*r_cm + r_cm^2 - 2*r_cm^3) + (x[,"n_03"] + x[,"n_32"])*log10(9 + 5*r_cm - 3*r_cm^2 + 4*r_cm^3) + (x[,"n_12"] + x[,"n_23"])*log10(31 + 2*r_cm - 14*r_cm^2 + 8*r_cm^3) + (x[,"n_11"] + x[,"n_24"])*log10(39 - 16*r_cm - 14*r_cm^2 + 12*r_cm^3)
logL_cm <- (-x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_05"] - 2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - x[,"n_35"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(5) + (x[,"n_00"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + (x[,"n_05"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_cm)) + 2*log(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_31"])*(log(pmax(1e-6,r_cm)) + log(9 - 5*r_cm + 2*r_cm^2)) + (x[,"n_15"] + x[,"n_20"])*(log(pmax(1e-6,r_cm)) + log(5 - 8*r_cm + 6*r_cm^2)) + (x[,"n_14"] + x[,"n_21"])*log(21 + 8*r_cm + 22*r_cm^2 - 12*r_cm^3) + (x[,"n_13"] + x[,"n_22"])*log(27 + 2*r_cm + 10*r_cm^2 - 8*r_cm^3) + (x[,"n_10"] + x[,"n_25"])*log(3 - 7*r_cm + 10*r_cm^2 - 6*r_cm^3) + (x[,"n_02"] + x[,"n_33"])*log(15 - 11*r_cm + 9*r_cm^2 - 4*r_cm^3) + (x[,"n_01"] + x[,"n_34"])*log(6 - 5*r_cm + r_cm^2 - 2*r_cm^3) + (x[,"n_03"] + x[,"n_32"])*log(9 + 5*r_cm - 3*r_cm^2 + 4*r_cm^3) + (x[,"n_12"] + x[,"n_23"])*log(31 + 2*r_cm - 14*r_cm^2 + 8*r_cm^3) + (x[,"n_11"] + x[,"n_24"])*log(39 - 16*r_cm - 14*r_cm^2 + 12*r_cm^3)
logL_cr <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-2*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 2*n05 - 2*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*log(5) + (n05 + n30)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n35)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n34)*(log(pmax(1e-6,r)) + log(8 - 7*r + 2*r^2)) + (n10 + n25)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n12 + n23)*log(30 - 4*r + 13*r^2 - 12*r^3) + (n11 + n24)*log(12 - 10*r + 13*r^2 - 6*r^3) + (n03 + n32)*log(11 - 12*r + 11*r^2 - 4*r^3) + (n15 + n20)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n04 + n31)*log(3 - r^2 - 2*r^3) + (n02 + n33)*log(6 + 2*r - r^2 + 4*r^3) + (n14 + n21)*log(9 + 2*r - 5*r^2 + 6*r^3) + (n13 + n22)*log(27 + 14*r - 23*r^2 + 12*r^3)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_cr <- 3*(x[,"n_05"] + x[,"n_30"])*log10(2) + 3*(x[,"n_00"] + x[,"n_35"])*log10(2) + (x[,"n_15"] + x[,"n_20"])*(3*log10(2) - log10(3)) - (x[,"n_10"] + x[,"n_25"])*(-3*log10(2) + log10(3)) - (x[,"n_04"] + x[,"n_31"])*(-log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_34"])*(-log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_22"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_12"] + x[,"n_23"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_14"] + x[,"n_21"])*(-log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_24"])*(-log10(2) + log10(19)) - (x[,"n_03"] + x[,"n_32"])*(-2*log10(2) + log10(29)) - (x[,"n_02"] + x[,"n_33"])*(-2*log10(2) + log10(29)) + (x[,"n_05"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_cr)) + 2*log10(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_34"])*(log10(pmax(1e-6,r_cr)) + log10(8 - 7*r_cr + 2*r_cr^2)) + (x[,"n_10"] + x[,"n_25"])*(log10(pmax(1e-6,r_cr)) + log10(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_12"] + x[,"n_23"])*log10(30 - 4*r_cr + 13*r_cr^2 - 12*r_cr^3) + (x[,"n_11"] + x[,"n_24"])*log10(12 - 10*r_cr + 13*r_cr^2 - 6*r_cr^3) + (x[,"n_03"] + x[,"n_32"])*log10(11 - 12*r_cr + 11*r_cr^2 - 4*r_cr^3) + (x[,"n_15"] + x[,"n_20"])*log10(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_04"] + x[,"n_31"])*log10(3 - r_cr^2 - 2*r_cr^3) + (x[,"n_02"] + x[,"n_33"])*log10(6 + 2*r_cr - r_cr^2 + 4*r_cr^3) + (x[,"n_14"] + x[,"n_21"])*log10(9 + 2*r_cr - 5*r_cr^2 + 6*r_cr^3) + (x[,"n_13"] + x[,"n_22"])*log10(27 + 14*r_cr - 23*r_cr^2 + 12*r_cr^3)
logL_cr <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(5) + (x[,"n_05"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_cr)) + 2*log(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_34"])*(log(pmax(1e-6,r_cr)) + log(8 - 7*r_cr + 2*r_cr^2)) + (x[,"n_10"] + x[,"n_25"])*(log(pmax(1e-6,r_cr)) + log(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_12"] + x[,"n_23"])*log(30 - 4*r_cr + 13*r_cr^2 - 12*r_cr^3) + (x[,"n_11"] + x[,"n_24"])*log(12 - 10*r_cr + 13*r_cr^2 - 6*r_cr^3) + (x[,"n_03"] + x[,"n_32"])*log(11 - 12*r_cr + 11*r_cr^2 - 4*r_cr^3) + (x[,"n_15"] + x[,"n_20"])*log(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_04"] + x[,"n_31"])*log(3 - r_cr^2 - 2*r_cr^3) + (x[,"n_02"] + x[,"n_33"])*log(6 + 2*r_cr - r_cr^2 + 4*r_cr^3) + (x[,"n_14"] + x[,"n_21"])*log(9 + 2*r_cr - 5*r_cr^2 + 6*r_cr^3) + (x[,"n_13"] + x[,"n_22"])*log(27 + 14*r_cr - 23*r_cr^2 + 12*r_cr^3)
logL_rc <- function(r,n10,n15,n20,n25,n00,n01,n02,n03,n04,n05,n11,n12,n13,n14,n21,n22,n23,n24,n30,n31,n32,n33,n34,n35) {
L <- (n10 + n15 + n20 + n25)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*(4*log(2) + 2*log(5)) + (n05 + n30)*(log(3 - 2*r) + 2*log(pmax(1e-6,r))) + (n00 + n35)*(2*log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n04 + n31)*(log(pmax(1e-6,r)) + log(12 - 5*r + 2*r^2)) + (n15 + n20)*(log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n13 + n22)*log(57 + 16*r^2 - 12*r^3) + (n14 + n21)*log(18 - 2*r + 11*r^2 - 6*r^3) + (n02 + n33)*log(21 - 16*r + 8*r^2 - 4*r^3) + (n01 + n34)*log(9 - 8*r + r^2 - 2*r^3) + (n10 + n25)*log(1 - 2*r + 3*r^2 - 2*r^3) + (n03 + n32)*log(9 + 12*r - 4*r^2 + 4*r^3) + (n11 + n24)*log(21 - 2*r - 7*r^2 + 6*r^3) + (n12 + n23)*log(61 + 4*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rc <- function(n10,n15,n20,n25,n00,n01,n02,n03,n04,n05,n11,n12,n13,n14,n21,n22,n23,n24,n30,n31,n32,n33,n34,n35) {
optimize(logL_rc,c(0,0.5), n10,n15,n20,n25,n00,n01,n02,n03,n04,n05,n11,n12,n13,n14,n21,n22,n23,n24,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_10"],x[,"n_15"],x[,"n_20"],x[,"n_25"],x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_rc <- (x[,"n_15"] + x[,"n_20"])*log10(2) + (x[,"n_10"] + x[,"n_25"])*log10(2) + (x[,"n_05"] + x[,"n_30"])*log10(2) + (x[,"n_00"] + x[,"n_35"])*log10(2) - (x[,"n_04"] + x[,"n_31"])*log10(5) - (x[,"n_01"] + x[,"n_34"])*log10(5) - (x[,"n_13"] + x[,"n_22"])*(-log10(2) + log10(7) + log10(17)) - (x[,"n_12"] + x[,"n_23"])*(-log10(2) + log10(7) + log10(17)) - (x[,"n_14"] + x[,"n_21"])*log10(19) - (x[,"n_11"] + x[,"n_24"])*log10(19) - (x[,"n_03"] + x[,"n_32"])*(-log10(2) + log10(29)) - (x[,"n_02"] + x[,"n_33"])*(-log10(2) + log10(29)) + (x[,"n_05"] + x[,"n_30"])*(log10(3 - 2*r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_04"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(12 - 5*r_rc + 2*r_rc^2)) + (x[,"n_15"] + x[,"n_20"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 3*r_rc + 2*r_rc^2)) + (x[,"n_13"] + x[,"n_22"])*log10(57 + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_14"] + x[,"n_21"])*log10(18 - 2*r_rc + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_02"] + x[,"n_33"])*log10(21 - 16*r_rc + 8*r_rc^2 - 4*r_rc^3) + (x[,"n_01"] + x[,"n_34"])*log10(9 - 8*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_10"] + x[,"n_25"])*log10(1 - 2*r_rc + 3*r_rc^2 - 2*r_rc^3) + (x[,"n_03"] + x[,"n_32"])*log10(9 + 12*r_rc - 4*r_rc^2 + 4*r_rc^3) + (x[,"n_11"] + x[,"n_24"])*log10(21 - 2*r_rc - 7*r_rc^2 + 6*r_rc^3) + (x[,"n_12"] + x[,"n_23"])*log10(61 + 4*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rc <- (x[,"n_10"] + x[,"n_15"] + x[,"n_20"] + x[,"n_25"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*(4*log(2) + 2*log(5)) + (x[,"n_05"] + x[,"n_30"])*(log(3 - 2*r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_04"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(12 - 5*r_rc + 2*r_rc^2)) + (x[,"n_15"] + x[,"n_20"])*(log(pmax(1e-6,r_rc)) + log(2 - 3*r_rc + 2*r_rc^2)) + (x[,"n_13"] + x[,"n_22"])*log(57 + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_14"] + x[,"n_21"])*log(18 - 2*r_rc + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_02"] + x[,"n_33"])*log(21 - 16*r_rc + 8*r_rc^2 - 4*r_rc^3) + (x[,"n_01"] + x[,"n_34"])*log(9 - 8*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_10"] + x[,"n_25"])*log(1 - 2*r_rc + 3*r_rc^2 - 2*r_rc^3) + (x[,"n_03"] + x[,"n_32"])*log(9 + 12*r_rc - 4*r_rc^2 + 4*r_rc^3) + (x[,"n_11"] + x[,"n_24"])*log(21 - 2*r_rc - 7*r_rc^2 + 6*r_rc^3) + (x[,"n_12"] + x[,"n_23"])*log(61 + 4*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rm <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 3*n05 - 4*n10 - 4*n11 - 2*n12 - 2*n13 - 4*n14 - 4*n15 - 4*n20 - 4*n21 - 2*n22 - 2*n23 - 4*n24 - 4*n25 - 3*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 3*n35)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*log(5) + (n05 + n30)*(log(3 - 2*r) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n35)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(1 + 2*r)) + (n15 + n20)*log(9 - 16*r + 22*r^2 - 12*r^3) + (n11 + n24)*log(51 + 4*r + 26*r^2 - 12*r^3) + (n03 + n32)*log(27 - 14*r + 12*r^2 - 4*r^3) + (n04 + n31)*log(9 - 3*r - r^2 - 2*r^3) + (n12 + n23)*log(14 + r + 2*r^2 - 2*r^3) + (n01 + n34)*log(3 + 11*r - 7*r^2 + 2*r^3) + (n13 + n22)*log(15 + r - 4*r^2 + 2*r^3) + (n02 + n33)*log(21 + 2*r + 4*r^3) + (n10 + n25)*log(3 + 8*r - 14*r^2 + 12*r^3) + (n14 + n21)*log(69 - 20*r - 10*r^2 + 12*r^3)
return(L)}
interlogL_rm <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_rm,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_rm <- (x[,"n_05"] + x[,"n_30"])*log10(2) + (x[,"n_00"] + x[,"n_35"])*log10(2) - (x[,"n_15"] + x[,"n_20"])*log10(5) - (x[,"n_10"] + x[,"n_25"])*log10(5) - (x[,"n_03"] + x[,"n_32"])*(-log10(2) + 2*log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_33"])*(-log10(2) + 2*log10(3) + log10(5)) - (x[,"n_04"] + x[,"n_31"])*log10(7) - (x[,"n_01"] + x[,"n_34"])*log10(7) - (x[,"n_14"] + x[,"n_21"])*(log10(2) + log10(29)) - (x[,"n_11"] + x[,"n_24"])*(log10(2) + log10(29)) - (x[,"n_13"] + x[,"n_22"])*(-2*log10(2) + log10(59)) - (x[,"n_12"] + x[,"n_23"])*(-2*log10(2) + log10(59)) + (x[,"n_05"] + x[,"n_30"])*(log10(3 - 2*r_rm) + log10(pmax(1e-6,1 - r_rm)) + log10(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_rm)) + log10(pmax(1e-6,r_rm)) + log10(1 + 2*r_rm)) + (x[,"n_15"] + x[,"n_20"])*log10(9 - 16*r_rm + 22*r_rm^2 - 12*r_rm^3) + (x[,"n_11"] + x[,"n_24"])*log10(51 + 4*r_rm + 26*r_rm^2 - 12*r_rm^3) + (x[,"n_03"] + x[,"n_32"])*log10(27 - 14*r_rm + 12*r_rm^2 - 4*r_rm^3) + (x[,"n_04"] + x[,"n_31"])*log10(9 - 3*r_rm - r_rm^2 - 2*r_rm^3) + (x[,"n_12"] + x[,"n_23"])*log10(14 + r_rm + 2*r_rm^2 - 2*r_rm^3) + (x[,"n_01"] + x[,"n_34"])*log10(3 + 11*r_rm - 7*r_rm^2 + 2*r_rm^3) + (x[,"n_13"] + x[,"n_22"])*log10(15 + r_rm - 4*r_rm^2 + 2*r_rm^3) + (x[,"n_02"] + x[,"n_33"])*log10(21 + 2*r_rm + 4*r_rm^3) + (x[,"n_10"] + x[,"n_25"])*log10(3 + 8*r_rm - 14*r_rm^2 + 12*r_rm^3) + (x[,"n_14"] + x[,"n_21"])*log10(69 - 20*r_rm - 10*r_rm^2 + 12*r_rm^3)
logL_rm <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_05"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_15"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_35"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(5) + (x[,"n_05"] + x[,"n_30"])*(log(3 - 2*r_rm) + log(pmax(1e-6,1 - r_rm)) + log(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_rm)) + log(pmax(1e-6,r_rm)) + log(1 + 2*r_rm)) + (x[,"n_15"] + x[,"n_20"])*log(9 - 16*r_rm + 22*r_rm^2 - 12*r_rm^3) + (x[,"n_11"] + x[,"n_24"])*log(51 + 4*r_rm + 26*r_rm^2 - 12*r_rm^3) + (x[,"n_03"] + x[,"n_32"])*log(27 - 14*r_rm + 12*r_rm^2 - 4*r_rm^3) + (x[,"n_04"] + x[,"n_31"])*log(9 - 3*r_rm - r_rm^2 - 2*r_rm^3) + (x[,"n_12"] + x[,"n_23"])*log(14 + r_rm + 2*r_rm^2 - 2*r_rm^3) + (x[,"n_01"] + x[,"n_34"])*log(3 + 11*r_rm - 7*r_rm^2 + 2*r_rm^3) + (x[,"n_13"] + x[,"n_22"])*log(15 + r_rm - 4*r_rm^2 + 2*r_rm^3) + (x[,"n_02"] + x[,"n_33"])*log(21 + 2*r_rm + 4*r_rm^3) + (x[,"n_10"] + x[,"n_25"])*log(3 + 8*r_rm - 14*r_rm^2 + 12*r_rm^3) + (x[,"n_14"] + x[,"n_21"])*log(69 - 20*r_rm - 10*r_rm^2 + 12*r_rm^3)
logL_rr <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*(4*log(2) + 2*log(5)) + (n05 + n30)*(log(3 - 2*r) + 2*log(pmax(1e-6,1 - r))) + (n00 + n35)*(2*log(pmax(1e-6,r)) + log(1 + 2*r)) + (n10 + n25)*(log(pmax(1e-6,r)) + log(2 + 5*r - 6*r^2)) + (n01 + n34)*(log(pmax(1e-6,r)) + log(4 + 13*r - 2*r^2)) + (n13 + n22)*log(75 - 44*r + 32*r^2 - 12*r^3) + (n14 + n21)*log(27 - 14*r - r^2 - 6*r^3) + (n02 + n33)*log(3 + 28*r - 8*r^2 - 4*r^3) + (n04 + n31)*log(15 - 24*r + 7*r^2 + 2*r^3) + (n03 + n32)*log(19 - 20*r^2 + 4*r^3) + (n11 + n24)*log(6 + 34*r - 19*r^2 + 6*r^3) + (n15 + n20)*log(1 + 6*r - 13*r^2 + 6*r^3) + (n12 + n23)*log(51 + 16*r - 4*r^2 + 12*r^3)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_rr <- (x[,"n_05"] + x[,"n_30"])*log10(2) + (x[,"n_00"] + x[,"n_35"])*log10(2) - (x[,"n_15"] + x[,"n_20"])*(-log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_25"])*(-log10(2) + log10(3)) - (x[,"n_04"] + x[,"n_31"])*log10(5) - (x[,"n_01"] + x[,"n_34"])*log10(5) - (x[,"n_13"] + x[,"n_22"])*(-log10(2) + log10(7) + log10(17)) - (x[,"n_12"] + x[,"n_23"])*(-log10(2) + log10(7) + log10(17)) - (x[,"n_14"] + x[,"n_21"])*log10(19) - (x[,"n_11"] + x[,"n_24"])*log10(19) - (x[,"n_03"] + x[,"n_32"])*(-log10(2) + log10(29)) - (x[,"n_02"] + x[,"n_33"])*(-log10(2) + log10(29)) + (x[,"n_05"] + x[,"n_30"])*(log10(3 - 2*r_rr) + 2*log10(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_35"])*(2*log10(pmax(1e-6,r_rr)) + log10(1 + 2*r_rr)) + (x[,"n_10"] + x[,"n_25"])*(log10(pmax(1e-6,r_rr)) + log10(2 + 5*r_rr - 6*r_rr^2)) + (x[,"n_01"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(4 + 13*r_rr - 2*r_rr^2)) + (x[,"n_13"] + x[,"n_22"])*log10(75 - 44*r_rr + 32*r_rr^2 - 12*r_rr^3) + (x[,"n_14"] + x[,"n_21"])*log10(27 - 14*r_rr - r_rr^2 - 6*r_rr^3) + (x[,"n_02"] + x[,"n_33"])*log10(3 + 28*r_rr - 8*r_rr^2 - 4*r_rr^3) + (x[,"n_04"] + x[,"n_31"])*log10(15 - 24*r_rr + 7*r_rr^2 + 2*r_rr^3) + (x[,"n_03"] + x[,"n_32"])*log10(19 - 20*r_rr^2 + 4*r_rr^3) + (x[,"n_11"] + x[,"n_24"])*log10(6 + 34*r_rr - 19*r_rr^2 + 6*r_rr^3) + (x[,"n_15"] + x[,"n_20"])*log10(1 + 6*r_rr - 13*r_rr^2 + 6*r_rr^3) + (x[,"n_12"] + x[,"n_23"])*log10(51 + 16*r_rr - 4*r_rr^2 + 12*r_rr^3)
logL_rr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*(4*log(2) + 2*log(5)) + (x[,"n_05"] + x[,"n_30"])*(log(3 - 2*r_rr) + 2*log(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_35"])*(2*log(pmax(1e-6,r_rr)) + log(1 + 2*r_rr)) + (x[,"n_10"] + x[,"n_25"])*(log(pmax(1e-6,r_rr)) + log(2 + 5*r_rr - 6*r_rr^2)) + (x[,"n_01"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(4 + 13*r_rr - 2*r_rr^2)) + (x[,"n_13"] + x[,"n_22"])*log(75 - 44*r_rr + 32*r_rr^2 - 12*r_rr^3) + (x[,"n_14"] + x[,"n_21"])*log(27 - 14*r_rr - r_rr^2 - 6*r_rr^3) + (x[,"n_02"] + x[,"n_33"])*log(3 + 28*r_rr - 8*r_rr^2 - 4*r_rr^3) + (x[,"n_04"] + x[,"n_31"])*log(15 - 24*r_rr + 7*r_rr^2 + 2*r_rr^3) + (x[,"n_03"] + x[,"n_32"])*log(19 - 20*r_rr^2 + 4*r_rr^3) + (x[,"n_11"] + x[,"n_24"])*log(6 + 34*r_rr - 19*r_rr^2 + 6*r_rr^3) + (x[,"n_15"] + x[,"n_20"])*log(1 + 6*r_rr - 13*r_rr^2 + 6*r_rr^3) + (x[,"n_12"] + x[,"n_23"])*log(51 + 16*r_rr - 4*r_rr^2 + 12*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_2.4 <- function(x){
logL_cc <- function(r,n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
L <- (-n01 - 2*n02 - 2*n03 - 2*n04 - n05 - n11 - 2*n12 - 2*n13 - 2*n14 - n15 - n21 - 2*n22 - 2*n23 - 2*n24 - n25 - n31 - 2*n32 - 2*n33 - 2*n34 - n35)*log(2) + (-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35)*(log(3) + 2*log(5)) + (n05 + n31)*(2*log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n33)*log(9 + 16*r - 16*r^2) + (n04 + n32)*(log(pmax(1e-6,r)) + log(12 + 7*r - 4*r^2)) + (n15 + n21)*(log(pmax(1e-6,r)) + log(5 + r - 3*r^2)) + (n01 + n35)*(log(pmax(1e-6,1 - r)) + log(3 - 4*r + r^2)) + (n13 + n23)*log(57 - 16*r + 16*r^2) + (n12 + n24)*log(42 - 32*r + 17*r^2 - 12*r^3) + (n11 + n25)*log(3 + 2*r - 8*r^2 + 3*r^3) + (n02 + n34)*log(15 - 14*r - 5*r^2 + 4*r^3) + (n14 + n22)*log(15 + 34*r - 19*r^2 + 12*r^3)
return(L)}
interlogL_cc <- function(n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
optimize(logL_cc,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_cc <- -((x[,"n_05"] + x[,"n_31"])*(-3*log10(2) + log10(5))) - (x[,"n_01"] + x[,"n_35"])*(-3*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_33"])*log10(13) - (x[,"n_15"] + x[,"n_21"])*(-3*log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_25"])*(-3*log10(2) + log10(19)) - (x[,"n_14"] + x[,"n_22"])*(-2*log10(2) + log10(5) + log10(23)) - (x[,"n_12"] + x[,"n_24"])*(-2*log10(2) + log10(5) + log10(23)) - (x[,"n_04"] + x[,"n_32"])*(-2*log10(2) + log10(29)) - (x[,"n_02"] + x[,"n_34"])*(-2*log10(2) + log10(29)) - (x[,"n_13"] + x[,"n_23"])*log10(53) + (x[,"n_05"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_03"] + x[,"n_33"])*log10(9 + 16*r_cc - 16*r_cc^2) + (x[,"n_04"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(12 + 7*r_cc - 4*r_cc^2)) + (x[,"n_15"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(5 + r_cc - 3*r_cc^2)) + (x[,"n_01"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_cc)) + log10(3 - 4*r_cc + r_cc^2)) + (x[,"n_13"] + x[,"n_23"])*log10(57 - 16*r_cc + 16*r_cc^2) + (x[,"n_12"] + x[,"n_24"])*log10(42 - 32*r_cc + 17*r_cc^2 - 12*r_cc^3) + (x[,"n_11"] + x[,"n_25"])*log10(3 + 2*r_cc - 8*r_cc^2 + 3*r_cc^3) + (x[,"n_02"] + x[,"n_34"])*log10(15 - 14*r_cc - 5*r_cc^2 + 4*r_cc^3) + (x[,"n_14"] + x[,"n_22"])*log10(15 + 34*r_cc - 19*r_cc^2 + 12*r_cc^3)
logL_cc <- (-x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - x[,"n_35"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*(log(3) + 2*log(5)) + (x[,"n_05"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_03"] + x[,"n_33"])*log(9 + 16*r_cc - 16*r_cc^2) + (x[,"n_04"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(12 + 7*r_cc - 4*r_cc^2)) + (x[,"n_15"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(5 + r_cc - 3*r_cc^2)) + (x[,"n_01"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_cc)) + log(3 - 4*r_cc + r_cc^2)) + (x[,"n_13"] + x[,"n_23"])*log(57 - 16*r_cc + 16*r_cc^2) + (x[,"n_12"] + x[,"n_24"])*log(42 - 32*r_cc + 17*r_cc^2 - 12*r_cc^3) + (x[,"n_11"] + x[,"n_25"])*log(3 + 2*r_cc - 8*r_cc^2 + 3*r_cc^3) + (x[,"n_02"] + x[,"n_34"])*log(15 - 14*r_cc - 5*r_cc^2 + 4*r_cc^3) + (x[,"n_14"] + x[,"n_22"])*log(15 + 34*r_cc - 19*r_cc^2 + 12*r_cc^3)
logL_cm <- function(r,n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
L <- (-2*n01 - 3*n02 - 3*n03 - 3*n04 - 2*n05 - 2*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35)*log(2) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35)*log(5) + (n01 + n35)*(log(pmax(1e-6,1 - r)) + log(2 - r) + log(pmax(1e-6,r))) + (n13 + n23)*log(33 + 8*r - 8*r^2) + (n05 + n31)*(log(pmax(1e-6,r)) + log(1 - r^2)) + (n03 + n33)*log(11 - 8*r + 8*r^2) + (n14 + n22)*log(18 - 4*r + 19*r^2 - 12*r^3) + (n02 + n34)*log(6 - 4*r + 5*r^2 - 4*r^3) + (n11 + n25)*log(3 - 5*r + 6*r^2 - 3*r^3) + (n15 + n21)*log(1 + 2*r - 3*r^2 + 3*r^3) + (n04 + n32)*log(3 + 6*r - 7*r^2 + 4*r^3) + (n12 + n24)*log(21 + 2*r - 17*r^2 + 12*r^3)
return(L)}
interlogL_cm <- function(n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
optimize(logL_cm,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_cm <- -2*(x[,"n_03"] + x[,"n_33"])*log10(3) - (x[,"n_05"] + x[,"n_31"])*(-3*log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_35"])*(-3*log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_23"])*(log10(5) + log10(7)) - (x[,"n_14"] + x[,"n_22"])*(-2*log10(2) + log10(7) + log10(11)) - (x[,"n_12"] + x[,"n_24"])*(-2*log10(2) + log10(7) + log10(11)) - (x[,"n_15"] + x[,"n_21"])*(-3*log10(2) + log10(13)) - (x[,"n_11"] + x[,"n_25"])*(-3*log10(2) + log10(13)) - (x[,"n_04"] + x[,"n_32"])*(-2*log10(2) + log10(19)) - (x[,"n_02"] + x[,"n_34"])*(-2*log10(2) + log10(19)) + (x[,"n_01"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_cm)) + log10(2 - r_cm) + log10(pmax(1e-6,r_cm))) + (x[,"n_13"] + x[,"n_23"])*log10(33 + 8*r_cm - 8*r_cm^2) + (x[,"n_05"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm)) + log10(1 - r_cm^2)) + (x[,"n_03"] + x[,"n_33"])*log10(11 - 8*r_cm + 8*r_cm^2) + (x[,"n_14"] + x[,"n_22"])*log10(18 - 4*r_cm + 19*r_cm^2 - 12*r_cm^3) + (x[,"n_02"] + x[,"n_34"])*log10(6 - 4*r_cm + 5*r_cm^2 - 4*r_cm^3) + (x[,"n_11"] + x[,"n_25"])*log10(3 - 5*r_cm + 6*r_cm^2 - 3*r_cm^3) + (x[,"n_15"] + x[,"n_21"])*log10(1 + 2*r_cm - 3*r_cm^2 + 3*r_cm^3) + (x[,"n_04"] + x[,"n_32"])*log10(3 + 6*r_cm - 7*r_cm^2 + 4*r_cm^3) + (x[,"n_12"] + x[,"n_24"])*log10(21 + 2*r_cm - 17*r_cm^2 + 12*r_cm^3)
logL_cm <- (-2*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(5) + (x[,"n_01"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_cm)) + log(2 - r_cm) + log(pmax(1e-6,r_cm))) + (x[,"n_13"] + x[,"n_23"])*log(33 + 8*r_cm - 8*r_cm^2) + (x[,"n_05"] + x[,"n_31"])*(log(pmax(1e-6,r_cm)) + log(1 - r_cm^2)) + (x[,"n_03"] + x[,"n_33"])*log(11 - 8*r_cm + 8*r_cm^2) + (x[,"n_14"] + x[,"n_22"])*log(18 - 4*r_cm + 19*r_cm^2 - 12*r_cm^3) + (x[,"n_02"] + x[,"n_34"])*log(6 - 4*r_cm + 5*r_cm^2 - 4*r_cm^3) + (x[,"n_11"] + x[,"n_25"])*log(3 - 5*r_cm + 6*r_cm^2 - 3*r_cm^3) + (x[,"n_15"] + x[,"n_21"])*log(1 + 2*r_cm - 3*r_cm^2 + 3*r_cm^3) + (x[,"n_04"] + x[,"n_32"])*log(3 + 6*r_cm - 7*r_cm^2 + 4*r_cm^3) + (x[,"n_12"] + x[,"n_24"])*log(21 + 2*r_cm - 17*r_cm^2 + 12*r_cm^3)
logL_cr <- function(r,n02,n04,n12,n13,n14,n22,n23,n24,n32,n34,n01,n03,n05,n11,n15,n21,n25,n31,n33,n35) {
L <- (-n02 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n34)*log(2) + 2*(n13 + n23)*log(3) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35)*log(5) + (n05 + n31)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n35)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n11 + n25)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n02 + n34)*(log(pmax(1e-6,r)) + log(4 - 5*r + 4*r^2)) + (n12 + n24)*log(6 - 8*r + 17*r^2 - 12*r^3) + (n04 + n32)*log(3 - 6*r + 7*r^2 - 4*r^3) + (n15 + n21)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n14 + n22)*log(3 + 10*r - 19*r^2 + 12*r^3)
return(L)}
interlogL_cr <- function(n02,n04,n12,n13,n14,n22,n23,n24,n32,n34,n01,n03,n05,n11,n15,n21,n25,n31,n33,n35) {
optimize(logL_cr,c(0,0.5), n02,n04,n12,n13,n14,n22,n23,n24,n32,n34,n01,n03,n05,n11,n15,n21,n25,n31,n33,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_02"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_34"],x[,"n_01"],x[,"n_03"],x[,"n_05"],x[,"n_11"],x[,"n_15"],x[,"n_21"],x[,"n_25"],x[,"n_31"],x[,"n_33"],x[,"n_35"])
LOD_cr <- 3*(x[,"n_05"] + x[,"n_31"])*log10(2) + 3*(x[,"n_01"] + x[,"n_35"])*log10(2) + (x[,"n_15"] + x[,"n_21"])*(3*log10(2) - log10(3)) - (x[,"n_11"] + x[,"n_25"])*(-3*log10(2) + log10(3)) - (x[,"n_04"] + x[,"n_32"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_34"])*(-2*log10(2) + log10(5)) - (x[,"n_14"] + x[,"n_22"])*(-2*log10(2) + log10(19)) - (x[,"n_12"] + x[,"n_24"])*(-2*log10(2) + log10(19)) + (x[,"n_05"] + x[,"n_31"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_cr)) + 2*log10(pmax(1e-6,r_cr))) + (x[,"n_11"] + x[,"n_25"])*(log10(pmax(1e-6,r_cr)) + log10(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_02"] + x[,"n_34"])*(log10(pmax(1e-6,r_cr)) + log10(4 - 5*r_cr + 4*r_cr^2)) + (x[,"n_12"] + x[,"n_24"])*log10(6 - 8*r_cr + 17*r_cr^2 - 12*r_cr^3) + (x[,"n_04"] + x[,"n_32"])*log10(3 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3) + (x[,"n_15"] + x[,"n_21"])*log10(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_14"] + x[,"n_22"])*log10(3 + 10*r_cr - 19*r_cr^2 + 12*r_cr^3)
logL_cr <- (-x[,"n_02"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_34"])*log(2) + 2*(x[,"n_13"] + x[,"n_23"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(5) + (x[,"n_05"] + x[,"n_31"])*(2*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_cr)) + 2*log(pmax(1e-6,r_cr))) + (x[,"n_11"] + x[,"n_25"])*(log(pmax(1e-6,r_cr)) + log(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_02"] + x[,"n_34"])*(log(pmax(1e-6,r_cr)) + log(4 - 5*r_cr + 4*r_cr^2)) + (x[,"n_12"] + x[,"n_24"])*log(6 - 8*r_cr + 17*r_cr^2 - 12*r_cr^3) + (x[,"n_04"] + x[,"n_32"])*log(3 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3) + (x[,"n_15"] + x[,"n_21"])*log(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_14"] + x[,"n_22"])*log(3 + 10*r_cr - 19*r_cr^2 + 12*r_cr^3)
logL_rc <- function(r,n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
L <- (-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35)*(3*log(2) + log(3) + 2*log(5)) + (n05 + n31)*(log(3 - 2*r) + log(pmax(1e-6,r)) + log(2 + r)) + (n13 + n23)*log(105 + 4*r - 4*r^2) + (n01 + n35)*(log(1 + 2*r) + log(3 - 4*r + r^2)) + (n03 + n33)*log(27 - 4*r + 4*r^2) + (n14 + n22)*log(57 - 4*r + 16*r^2 - 12*r^3) + (n11 + n25)*log(12 - 10*r + 13*r^2 - 6*r^3) + (n02 + n34)*log(21 - 16*r + 8*r^2 - 4*r^3) + (n04 + n32)*log(9 + 12*r - 4*r^2 + 4*r^3) + (n15 + n21)*log(9 + 2*r - 5*r^2 + 6*r^3) + (n12 + n24)*log(57 + 8*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rc <- function(n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
optimize(logL_rc,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_rc <- -((x[,"n_05"] + x[,"n_31"])*(-log10(2) + log10(5))) - (x[,"n_01"] + x[,"n_35"])*(-log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_33"])*(log10(2) + log10(13)) - (x[,"n_15"] + x[,"n_21"])*(-log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_25"])*(-log10(2) + log10(19)) - (x[,"n_14"] + x[,"n_22"])*(-log10(2) + log10(5) + log10(23)) - (x[,"n_12"] + x[,"n_24"])*(-log10(2) + log10(5) + log10(23)) - (x[,"n_04"] + x[,"n_32"])*(-log10(2) + log10(29)) - (x[,"n_02"] + x[,"n_34"])*(-log10(2) + log10(29)) - (x[,"n_13"] + x[,"n_23"])*(log10(2) + log10(53)) + (x[,"n_05"] + x[,"n_31"])*(log10(3 - 2*r_rc) + log10(pmax(1e-6,r_rc)) + log10(2 + r_rc)) + (x[,"n_13"] + x[,"n_23"])*log10(105 + 4*r_rc - 4*r_rc^2) + (x[,"n_01"] + x[,"n_35"])*(log10(1 + 2*r_rc) + log10(3 - 4*r_rc + r_rc^2)) + (x[,"n_03"] + x[,"n_33"])*log10(27 - 4*r_rc + 4*r_rc^2) + (x[,"n_14"] + x[,"n_22"])*log10(57 - 4*r_rc + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_11"] + x[,"n_25"])*log10(12 - 10*r_rc + 13*r_rc^2 - 6*r_rc^3) + (x[,"n_02"] + x[,"n_34"])*log10(21 - 16*r_rc + 8*r_rc^2 - 4*r_rc^3) + (x[,"n_04"] + x[,"n_32"])*log10(9 + 12*r_rc - 4*r_rc^2 + 4*r_rc^3) + (x[,"n_15"] + x[,"n_21"])*log10(9 + 2*r_rc - 5*r_rc^2 + 6*r_rc^3) + (x[,"n_12"] + x[,"n_24"])*log10(57 + 8*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rc <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*(3*log(2) + log(3) + 2*log(5)) + (x[,"n_05"] + x[,"n_31"])*(log(3 - 2*r_rc) + log(pmax(1e-6,r_rc)) + log(2 + r_rc)) + (x[,"n_13"] + x[,"n_23"])*log(105 + 4*r_rc - 4*r_rc^2) + (x[,"n_01"] + x[,"n_35"])*(log(1 + 2*r_rc) + log(3 - 4*r_rc + r_rc^2)) + (x[,"n_03"] + x[,"n_33"])*log(27 - 4*r_rc + 4*r_rc^2) + (x[,"n_14"] + x[,"n_22"])*log(57 - 4*r_rc + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_11"] + x[,"n_25"])*log(12 - 10*r_rc + 13*r_rc^2 - 6*r_rc^3) + (x[,"n_02"] + x[,"n_34"])*log(21 - 16*r_rc + 8*r_rc^2 - 4*r_rc^3) + (x[,"n_04"] + x[,"n_32"])*log(9 + 12*r_rc - 4*r_rc^2 + 4*r_rc^3) + (x[,"n_15"] + x[,"n_21"])*log(9 + 2*r_rc - 5*r_rc^2 + 6*r_rc^3) + (x[,"n_12"] + x[,"n_24"])*log(57 + 8*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rm <- function(r,n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
L <- (-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35)*(4*log(2) + 2*log(5)) + (n01 + n35)*(log(2 - r) + log(pmax(1e-6,r)) + log(1 + 2*r)) + (n13 + n23)*log(69 + 4*r - 4*r^2) + (n05 + n31)*(log(3 - 2*r) + log(1 - r^2)) + (n03 + n33)*log(19 - 4*r + 4*r^2) + (n12 + n24)*log(33 + 4*r + 20*r^2 - 12*r^3) + (n15 + n21)*log(10 - 10*r + 9*r^2 - 6*r^3) + (n04 + n32)*log(15 - 12*r + 4*r^2 - 4*r^3) + (n02 + n34)*log(3 + 16*r - 8*r^2 + 4*r^3) + (n11 + n25)*log(3 + 10*r - 9*r^2 + 6*r^3) + (n14 + n22)*log(45 - 8*r - 16*r^2 + 12*r^3)
return(L)}
interlogL_rm <- function(n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
optimize(logL_rm,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_rm <- -((x[,"n_05"] + x[,"n_31"])*(-log10(2) + log10(3))) - (x[,"n_01"] + x[,"n_35"])*(-log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_33"])*(log10(2) + 2*log10(3)) - (x[,"n_13"] + x[,"n_23"])*(log10(2) + log10(5) + log10(7)) - (x[,"n_14"] + x[,"n_22"])*(-log10(2) + log10(7) + log10(11)) - (x[,"n_12"] + x[,"n_24"])*(-log10(2) + log10(7) + log10(11)) - (x[,"n_15"] + x[,"n_21"])*(-log10(2) + log10(13)) - (x[,"n_11"] + x[,"n_25"])*(-log10(2) + log10(13)) - (x[,"n_04"] + x[,"n_32"])*(-log10(2) + log10(19)) - (x[,"n_02"] + x[,"n_34"])*(-log10(2) + log10(19)) + (x[,"n_01"] + x[,"n_35"])*(log10(2 - r_rm) + log10(pmax(1e-6,r_rm)) + log10(1 + 2*r_rm)) + (x[,"n_13"] + x[,"n_23"])*log10(69 + 4*r_rm - 4*r_rm^2) + (x[,"n_05"] + x[,"n_31"])*(log10(3 - 2*r_rm) + log10(1 - r_rm^2)) + (x[,"n_03"] + x[,"n_33"])*log10(19 - 4*r_rm + 4*r_rm^2) + (x[,"n_12"] + x[,"n_24"])*log10(33 + 4*r_rm + 20*r_rm^2 - 12*r_rm^3) + (x[,"n_15"] + x[,"n_21"])*log10(10 - 10*r_rm + 9*r_rm^2 - 6*r_rm^3) + (x[,"n_04"] + x[,"n_32"])*log10(15 - 12*r_rm + 4*r_rm^2 - 4*r_rm^3) + (x[,"n_02"] + x[,"n_34"])*log10(3 + 16*r_rm - 8*r_rm^2 + 4*r_rm^3) + (x[,"n_11"] + x[,"n_25"])*log10(3 + 10*r_rm - 9*r_rm^2 + 6*r_rm^3) + (x[,"n_14"] + x[,"n_22"])*log10(45 - 8*r_rm - 16*r_rm^2 + 12*r_rm^3)
logL_rm <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*(4*log(2) + 2*log(5)) + (x[,"n_01"] + x[,"n_35"])*(log(2 - r_rm) + log(pmax(1e-6,r_rm)) + log(1 + 2*r_rm)) + (x[,"n_13"] + x[,"n_23"])*log(69 + 4*r_rm - 4*r_rm^2) + (x[,"n_05"] + x[,"n_31"])*(log(3 - 2*r_rm) + log(1 - r_rm^2)) + (x[,"n_03"] + x[,"n_33"])*log(19 - 4*r_rm + 4*r_rm^2) + (x[,"n_12"] + x[,"n_24"])*log(33 + 4*r_rm + 20*r_rm^2 - 12*r_rm^3) + (x[,"n_15"] + x[,"n_21"])*log(10 - 10*r_rm + 9*r_rm^2 - 6*r_rm^3) + (x[,"n_04"] + x[,"n_32"])*log(15 - 12*r_rm + 4*r_rm^2 - 4*r_rm^3) + (x[,"n_02"] + x[,"n_34"])*log(3 + 16*r_rm - 8*r_rm^2 + 4*r_rm^3) + (x[,"n_11"] + x[,"n_25"])*log(3 + 10*r_rm - 9*r_rm^2 + 6*r_rm^3) + (x[,"n_14"] + x[,"n_22"])*log(45 - 8*r_rm - 16*r_rm^2 + 12*r_rm^3)
logL_rr <- function(r,n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
L <- (-2*n01 - n02 - 2*n03 - n04 - 2*n05 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n21 - 2*n22 - 2*n23 - 2*n24 - 2*n25 - 2*n31 - n32 - 2*n33 - n34 - 2*n35)*log(2) + (n13 + n23)*log(3) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35)*log(5) + (n05 + n31)*(log(3 - 2*r) + 2*log(pmax(1e-6,1 - r))) + (n01 + n35)*(2*log(pmax(1e-6,r)) + log(1 + 2*r)) + (n03 + n33)*log(1 + 12*r - 12*r^2) + (n11 + n25)*(log(pmax(1e-6,r)) + log(2 + 5*r - 6*r^2)) + (n13 + n23)*log(7 - 4*r + 4*r^2) + (n14 + n22)*log(15 - 16*r + 16*r^2 - 12*r^3) + (n02 + n34)*log(r + 4*r^2 - 2*r^3) + (n04 + n32)*log(3 - 3*r - 2*r^2 + 2*r^3) + (n15 + n21)*log(1 + 6*r - 13*r^2 + 6*r^3) + (n12 + n24)*log(3 + 20*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rr <- function(n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35) {
optimize(logL_rr,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_rr <- (x[,"n_05"] + x[,"n_31"])*log10(2) - 2*(x[,"n_03"] + x[,"n_33"])*log10(2) + (x[,"n_01"] + x[,"n_35"])*log10(2) - (x[,"n_15"] + x[,"n_21"])*(-log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_25"])*(-log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_23"])*(log10(2) + log10(3)) - (x[,"n_04"] + x[,"n_32"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_34"])*(-2*log10(2) + log10(5)) - (x[,"n_14"] + x[,"n_22"])*(-log10(2) + log10(19)) - (x[,"n_12"] + x[,"n_24"])*(-log10(2) + log10(19)) + (x[,"n_05"] + x[,"n_31"])*(log10(3 - 2*r_rr) + 2*log10(pmax(1e-6,1 - r_rr))) + (x[,"n_01"] + x[,"n_35"])*(2*log10(pmax(1e-6,r_rr)) + log10(1 + 2*r_rr)) + (x[,"n_03"] + x[,"n_33"])*log10(1 + 12*r_rr - 12*r_rr^2) + (x[,"n_11"] + x[,"n_25"])*(log10(pmax(1e-6,r_rr)) + log10(2 + 5*r_rr - 6*r_rr^2)) + (x[,"n_13"] + x[,"n_23"])*log10(7 - 4*r_rr + 4*r_rr^2) + (x[,"n_14"] + x[,"n_22"])*log10(15 - 16*r_rr + 16*r_rr^2 - 12*r_rr^3) + (x[,"n_02"] + x[,"n_34"])*log10(r_rr + 4*r_rr^2 - 2*r_rr^3) + (x[,"n_04"] + x[,"n_32"])*log10(3 - 3*r_rr - 2*r_rr^2 + 2*r_rr^3) + (x[,"n_15"] + x[,"n_21"])*log10(1 + 6*r_rr - 13*r_rr^2 + 6*r_rr^3) + (x[,"n_12"] + x[,"n_24"])*log10(3 + 20*r_rr - 20*r_rr^2 + 12*r_rr^3)
logL_rr <- (-2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_31"] - x[,"n_32"] - 2*x[,"n_33"] - x[,"n_34"] - 2*x[,"n_35"])*log(2) + (x[,"n_13"] + x[,"n_23"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(5) + (x[,"n_05"] + x[,"n_31"])*(log(3 - 2*r_rr) + 2*log(pmax(1e-6,1 - r_rr))) + (x[,"n_01"] + x[,"n_35"])*(2*log(pmax(1e-6,r_rr)) + log(1 + 2*r_rr)) + (x[,"n_03"] + x[,"n_33"])*log(1 + 12*r_rr - 12*r_rr^2) + (x[,"n_11"] + x[,"n_25"])*(log(pmax(1e-6,r_rr)) + log(2 + 5*r_rr - 6*r_rr^2)) + (x[,"n_13"] + x[,"n_23"])*log(7 - 4*r_rr + 4*r_rr^2) + (x[,"n_14"] + x[,"n_22"])*log(15 - 16*r_rr + 16*r_rr^2 - 12*r_rr^3) + (x[,"n_02"] + x[,"n_34"])*log(r_rr + 4*r_rr^2 - 2*r_rr^3) + (x[,"n_04"] + x[,"n_32"])*log(3 - 3*r_rr - 2*r_rr^2 + 2*r_rr^3) + (x[,"n_15"] + x[,"n_21"])*log(1 + 6*r_rr - 13*r_rr^2 + 6*r_rr^3) + (x[,"n_12"] + x[,"n_24"])*log(3 + 20*r_rr - 20*r_rr^2 + 12*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_3.0 <- function(x){
logL_c <- function(r,n01,n02,n11,n12,n21,n22,n31,n32,n00,n03,n10,n13,n20,n23,n30,n33) {
L <- (n01 + n02 + n11 + n12 + n21 + n22 + n31 + n32)*log(3) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(2*log(2) + 2*log(5)) + (n10 + n23)*log(3 - 2*r) + (n11 + n22)*log(7 - 2*r) + (n00 + n33)*log(pmax(1e-6,1 - r)) + (n01 + n32)*log(2 - r) + (n03 + n30)*log(pmax(1e-6,r)) + (n02 + n31)*log(1 + r) + (n13 + n20)*log(1 + 2*r) + (n12 + n21)*log(5 + 2*r)
return(L)}
interlogL_c <- function(n01,n02,n11,n12,n21,n22,n31,n32,n00,n03,n10,n13,n20,n23,n30,n33) {
optimize(logL_c,c(0,0.5), n01,n02,n11,n12,n21,n22,n31,n32,n00,n03,n10,n13,n20,n23,n30,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_01"],x[,"n_02"],x[,"n_11"],x[,"n_12"],x[,"n_21"],x[,"n_22"],x[,"n_31"],x[,"n_32"],x[,"n_00"],x[,"n_03"],x[,"n_10"],x[,"n_13"],x[,"n_20"],x[,"n_23"],x[,"n_30"],x[,"n_33"])
LOD_c <- -((x[,"n_13"] + x[,"n_20"])*log10(2)) - (x[,"n_10"] + x[,"n_23"])*log10(2) + (x[,"n_03"] + x[,"n_30"])*log10(2) + (x[,"n_00"] + x[,"n_33"])*log10(2) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_21"])*(log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_22"])*(log10(2) + log10(3)) + (x[,"n_10"] + x[,"n_23"])*log10(3 - 2*r_c) + (x[,"n_11"] + x[,"n_22"])*log10(7 - 2*r_c) + (x[,"n_00"] + x[,"n_33"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_32"])*log10(2 - r_c) + (x[,"n_03"] + x[,"n_30"])*log10(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_31"])*log10(1 + r_c) + (x[,"n_13"] + x[,"n_20"])*log10(1 + 2*r_c) + (x[,"n_12"] + x[,"n_21"])*log10(5 + 2*r_c)
logL_c <- (x[,"n_01"] + x[,"n_02"] + x[,"n_11"] + x[,"n_12"] + x[,"n_21"] + x[,"n_22"] + x[,"n_31"] + x[,"n_32"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(2*log(2) + 2*log(5)) + (x[,"n_10"] + x[,"n_23"])*log(3 - 2*r_c) + (x[,"n_11"] + x[,"n_22"])*log(7 - 2*r_c) + (x[,"n_00"] + x[,"n_33"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_32"])*log(2 - r_c) + (x[,"n_03"] + x[,"n_30"])*log(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_31"])*log(1 + r_c) + (x[,"n_13"] + x[,"n_20"])*log(1 + 2*r_c) + (x[,"n_12"] + x[,"n_21"])*log(5 + 2*r_c)
logL_r <- function(r,n01,n02,n11,n12,n21,n22,n31,n32,n00,n03,n10,n13,n20,n23,n30,n33) {
L <- (n01 + n02 + n11 + n12 + n21 + n22 + n31 + n32)*log(3) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(2*log(2) + 2*log(5)) + (n13 + n20)*log(3 - 2*r) + (n12 + n21)*log(7 - 2*r) + (n03 + n30)*log(pmax(1e-6,1 - r)) + (n02 + n31)*log(2 - r) + (n00 + n33)*log(pmax(1e-6,r)) + (n01 + n32)*log(1 + r) + (n10 + n23)*log(1 + 2*r) + (n11 + n22)*log(5 + 2*r)
return(L)}
interlogL_r <- function(n01,n02,n11,n12,n21,n22,n31,n32,n00,n03,n10,n13,n20,n23,n30,n33) {
optimize(logL_r,c(0,0.5), n01,n02,n11,n12,n21,n22,n31,n32,n00,n03,n10,n13,n20,n23,n30,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_02"],x[,"n_11"],x[,"n_12"],x[,"n_21"],x[,"n_22"],x[,"n_31"],x[,"n_32"],x[,"n_00"],x[,"n_03"],x[,"n_10"],x[,"n_13"],x[,"n_20"],x[,"n_23"],x[,"n_30"],x[,"n_33"])
LOD_r <- -((x[,"n_13"] + x[,"n_20"])*log10(2)) - (x[,"n_10"] + x[,"n_23"])*log10(2) + (x[,"n_03"] + x[,"n_30"])*log10(2) + (x[,"n_00"] + x[,"n_33"])*log10(2) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_21"])*(log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_22"])*(log10(2) + log10(3)) + (x[,"n_13"] + x[,"n_20"])*log10(3 - 2*r_r) + (x[,"n_12"] + x[,"n_21"])*log10(7 - 2*r_r) + (x[,"n_03"] + x[,"n_30"])*log10(pmax(1e-6,1 - r_r)) + (x[,"n_02"] + x[,"n_31"])*log10(2 - r_r) + (x[,"n_00"] + x[,"n_33"])*log10(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_32"])*log10(1 + r_r) + (x[,"n_10"] + x[,"n_23"])*log10(1 + 2*r_r) + (x[,"n_11"] + x[,"n_22"])*log10(5 + 2*r_r)
logL_r <- (x[,"n_01"] + x[,"n_02"] + x[,"n_11"] + x[,"n_12"] + x[,"n_21"] + x[,"n_22"] + x[,"n_31"] + x[,"n_32"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(2*log(2) + 2*log(5)) + (x[,"n_13"] + x[,"n_20"])*log(3 - 2*r_r) + (x[,"n_12"] + x[,"n_21"])*log(7 - 2*r_r) + (x[,"n_03"] + x[,"n_30"])*log(pmax(1e-6,1 - r_r)) + (x[,"n_02"] + x[,"n_31"])*log(2 - r_r) + (x[,"n_00"] + x[,"n_33"])*log(pmax(1e-6,r_r)) + (x[,"n_01"] + x[,"n_32"])*log(1 + r_r) + (x[,"n_10"] + x[,"n_23"])*log(1 + 2*r_r) + (x[,"n_11"] + x[,"n_22"])*log(5 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_3.1 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-2*n00 - n01 - 2*n02 - n03 - 2*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n30 - n31 - 2*n32 - n33 - 2*n34)*log(2) + (n02 + n12 + n22 + n32)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(5) + 2*(n00 + n34)*log(pmax(1e-6,1 - r)) + 2*(n04 + n30)*log(pmax(1e-6,r)) + (n14 + n20)*(log(5 - 2*r) + log(pmax(1e-6,r))) + (n03 + n31)*(log(pmax(1e-6,r)) + log(2 + r)) + (n11 + n23)*log(27 - 12*r - 4*r^2) + (n13 + n21)*log(11 + 20*r - 4*r^2) + (n10 + n24)*log(3 - r - 2*r^2) + (n02 + n32)*log(1 + 2*r - 2*r^2) + (n01 + n33)*log(3 - 4*r + r^2) + (n12 + n22)*log(13 - 4*r + 4*r^2)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cc <- -((x[,"n_14"] + x[,"n_20"])*log10(2)) - (x[,"n_10"] + x[,"n_24"])*log10(2) + 2*(x[,"n_04"] + x[,"n_30"])*log10(2) + 2*(x[,"n_00"] + x[,"n_34"])*log10(2) - (x[,"n_02"] + x[,"n_32"])*(-log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_22"])*(2*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_31"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_33"])*(-2*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_21"])*(2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_23"])*(2*log10(2) + log10(5)) + 2*(x[,"n_00"] + x[,"n_34"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_30"])*log10(pmax(1e-6,r_cc)) + (x[,"n_14"] + x[,"n_20"])*(log10(5 - 2*r_cc) + log10(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_11"] + x[,"n_23"])*log10(27 - 12*r_cc - 4*r_cc^2) + (x[,"n_13"] + x[,"n_21"])*log10(11 + 20*r_cc - 4*r_cc^2) + (x[,"n_10"] + x[,"n_24"])*log10(3 - r_cc - 2*r_cc^2) + (x[,"n_02"] + x[,"n_32"])*log10(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_01"] + x[,"n_33"])*log10(3 - 4*r_cc + r_cc^2) + (x[,"n_12"] + x[,"n_22"])*log10(13 - 4*r_cc + 4*r_cc^2)
logL_cc <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - x[,"n_03"] - 2*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_30"] - x[,"n_31"] - 2*x[,"n_32"] - x[,"n_33"] - 2*x[,"n_34"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_22"] + x[,"n_32"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + 2*(x[,"n_00"] + x[,"n_34"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_30"])*log(pmax(1e-6,r_cc)) + (x[,"n_14"] + x[,"n_20"])*(log(5 - 2*r_cc) + log(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_11"] + x[,"n_23"])*log(27 - 12*r_cc - 4*r_cc^2) + (x[,"n_13"] + x[,"n_21"])*log(11 + 20*r_cc - 4*r_cc^2) + (x[,"n_10"] + x[,"n_24"])*log(3 - r_cc - 2*r_cc^2) + (x[,"n_02"] + x[,"n_32"])*log(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_01"] + x[,"n_33"])*log(3 - 4*r_cc + r_cc^2) + (x[,"n_12"] + x[,"n_22"])*log(13 - 4*r_cc + 4*r_cc^2)
logL_cr <- function(r,n02,n12,n22,n32,n00,n01,n03,n04,n10,n11,n13,n14,n20,n21,n23,n24,n30,n31,n33,n34) {
L <- (n02 + n12 + n22 + n32)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(4*log(2) + 2*log(5)) + (n04 + n30)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n00 + n34)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n01 + n03 + n31 + n33)*log(9 + 4*r - 4*r^2) + (n12 + n22)*log(23 + 4*r - 4*r^2) + (n10 + n24)*log(6 - 5*r + 2*r^2) + (n14 + n20)*log(3 + r + 2*r^2) + (n11 + n23)*log(45 - 12*r + 4*r^2) + (n02 + n32)*log(7 - 4*r + 4*r^2) + (n13 + n21)*log(37 + 4*r + 4*r^2)
return(L)}
interlogL_cr <- function(n02,n12,n22,n32,n00,n01,n03,n04,n10,n11,n13,n14,n20,n21,n23,n24,n30,n31,n33,n34) {
optimize(logL_cr,c(0,0.5), n02,n12,n22,n32,n00,n01,n03,n04,n10,n11,n13,n14,n20,n21,n23,n24,n30,n31,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_02"],x[,"n_12"],x[,"n_22"],x[,"n_32"],x[,"n_00"],x[,"n_01"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_33"],x[,"n_34"])
LOD_cr <- -2*(x[,"n_14"] + x[,"n_20"])*log10(2) - 2*(x[,"n_10"] + x[,"n_24"])*log10(2) - (x[,"n_02"] + x[,"n_32"])*(log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_22"])*(3*log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_03"] + x[,"n_31"] + x[,"n_33"])*(log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_21"])*(3*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_23"])*(3*log10(2) + log10(5)) + (x[,"n_04"] + x[,"n_30"])*(log10(3 - 2*r_cr) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cr)) + log10(1 + 2*r_cr)) + (x[,"n_01"] + x[,"n_03"] + x[,"n_31"] + x[,"n_33"])*log10(9 + 4*r_cr - 4*r_cr^2) + (x[,"n_12"] + x[,"n_22"])*log10(23 + 4*r_cr - 4*r_cr^2) + (x[,"n_10"] + x[,"n_24"])*log10(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_14"] + x[,"n_20"])*log10(3 + r_cr + 2*r_cr^2) + (x[,"n_11"] + x[,"n_23"])*log10(45 - 12*r_cr + 4*r_cr^2) + (x[,"n_02"] + x[,"n_32"])*log10(7 - 4*r_cr + 4*r_cr^2) + (x[,"n_13"] + x[,"n_21"])*log10(37 + 4*r_cr + 4*r_cr^2)
logL_cr <- (x[,"n_02"] + x[,"n_12"] + x[,"n_22"] + x[,"n_32"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(4*log(2) + 2*log(5)) + (x[,"n_04"] + x[,"n_30"])*(log(3 - 2*r_cr) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_cr)) + log(1 + 2*r_cr)) + (x[,"n_01"] + x[,"n_03"] + x[,"n_31"] + x[,"n_33"])*log(9 + 4*r_cr - 4*r_cr^2) + (x[,"n_12"] + x[,"n_22"])*log(23 + 4*r_cr - 4*r_cr^2) + (x[,"n_10"] + x[,"n_24"])*log(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_14"] + x[,"n_20"])*log(3 + r_cr + 2*r_cr^2) + (x[,"n_11"] + x[,"n_23"])*log(45 - 12*r_cr + 4*r_cr^2) + (x[,"n_02"] + x[,"n_32"])*log(7 - 4*r_cr + 4*r_cr^2) + (x[,"n_13"] + x[,"n_21"])*log(37 + 4*r_cr + 4*r_cr^2)
logL_rc <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
L <- (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n30 - 2*n31 - n32 - 2*n33 - 2*n34)*log(2) + (n02 + n12 + n22 + n32)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(5) + (n00 + n04 + n30 + n34)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n22)*log(11 + 4*r - 4*r^2) + (n01 + n33)*log(3 - 2*r^2) + (n03 + n31)*log(1 + 4*r - 2*r^2) + (n02 + n32)*log(1 - r + r^2) + (n14 + n20)*log(3 - 3*r + 2*r^2) + (n10 + n24)*log(2 - r + 2*r^2) + (n11 + n13 + n21 + n23)*log(21 - 4*r + 4*r^2)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rc <- -((x[,"n_14"] + x[,"n_20"])*log10(2)) - (x[,"n_10"] + x[,"n_24"])*log10(2) + 2*(x[,"n_00"] + x[,"n_04"] + x[,"n_30"] + x[,"n_34"])*log10(2) + (x[,"n_02"] + x[,"n_32"])*(2*log10(2) - log10(3)) - (x[,"n_12"] + x[,"n_22"])*(2*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_13"] + x[,"n_21"] + x[,"n_23"])*(2*log10(2) + log10(5)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_30"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc))) + (x[,"n_12"] + x[,"n_22"])*log10(11 + 4*r_rc - 4*r_rc^2) + (x[,"n_01"] + x[,"n_33"])*log10(3 - 2*r_rc^2) + (x[,"n_03"] + x[,"n_31"])*log10(1 + 4*r_rc - 2*r_rc^2) + (x[,"n_02"] + x[,"n_32"])*log10(1 - r_rc + r_rc^2) + (x[,"n_14"] + x[,"n_20"])*log10(3 - 3*r_rc + 2*r_rc^2) + (x[,"n_10"] + x[,"n_24"])*log10(2 - r_rc + 2*r_rc^2) + (x[,"n_11"] + x[,"n_13"] + x[,"n_21"] + x[,"n_23"])*log10(21 - 4*r_rc + 4*r_rc^2)
logL_rc <- (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_22"] + x[,"n_32"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + (x[,"n_00"] + x[,"n_04"] + x[,"n_30"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc))) + (x[,"n_12"] + x[,"n_22"])*log(11 + 4*r_rc - 4*r_rc^2) + (x[,"n_01"] + x[,"n_33"])*log(3 - 2*r_rc^2) + (x[,"n_03"] + x[,"n_31"])*log(1 + 4*r_rc - 2*r_rc^2) + (x[,"n_02"] + x[,"n_32"])*log(1 - r_rc + r_rc^2) + (x[,"n_14"] + x[,"n_20"])*log(3 - 3*r_rc + 2*r_rc^2) + (x[,"n_10"] + x[,"n_24"])*log(2 - r_rc + 2*r_rc^2) + (x[,"n_11"] + x[,"n_13"] + x[,"n_21"] + x[,"n_23"])*log(21 - 4*r_rc + 4*r_rc^2)
logL_rr <- function(r,n02,n12,n22,n32,n00,n01,n03,n04,n10,n11,n13,n14,n20,n21,n23,n24,n30,n31,n33,n34) {
L <- (n02 + n12 + n22 + n32)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(4*log(2) + 2*log(5)) + (n04 + n30)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n00 + n34)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n13 + n21)*log(51 - 20*r - 4*r^2) + (n02 + n32)*log(5 + 4*r - 4*r^2) + (n11 + n23)*log(27 + 28*r - 4*r^2) + (n14 + n20)*log(6 - 3*r - 2*r^2) + (n10 + n24)*log(1 + 7*r - 2*r^2) + (n03 + n31)*log(19 - 20*r + 4*r^2) + (n12 + n22)*log(25 - 4*r + 4*r^2) + (n01 + n33)*log(3 + 12*r + 4*r^2)
return(L)}
interlogL_rr <- function(n02,n12,n22,n32,n00,n01,n03,n04,n10,n11,n13,n14,n20,n21,n23,n24,n30,n31,n33,n34) {
optimize(logL_rr,c(0,0.5), n02,n12,n22,n32,n00,n01,n03,n04,n10,n11,n13,n14,n20,n21,n23,n24,n30,n31,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_02"],x[,"n_12"],x[,"n_22"],x[,"n_32"],x[,"n_00"],x[,"n_01"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_33"],x[,"n_34"])
LOD_rr <- -2*(x[,"n_14"] + x[,"n_20"])*log10(2) - 2*(x[,"n_10"] + x[,"n_24"])*log10(2) - (x[,"n_02"] + x[,"n_32"])*(log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_22"])*(3*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_31"])*(log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_33"])*(log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_21"])*(3*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_23"])*(3*log10(2) + log10(5)) + (x[,"n_04"] + x[,"n_30"])*(log10(3 - 2*r_rr) + log10(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(1 + 2*r_rr)) + (x[,"n_13"] + x[,"n_21"])*log10(51 - 20*r_rr - 4*r_rr^2) + (x[,"n_02"] + x[,"n_32"])*log10(5 + 4*r_rr - 4*r_rr^2) + (x[,"n_11"] + x[,"n_23"])*log10(27 + 28*r_rr - 4*r_rr^2) + (x[,"n_14"] + x[,"n_20"])*log10(6 - 3*r_rr - 2*r_rr^2) + (x[,"n_10"] + x[,"n_24"])*log10(1 + 7*r_rr - 2*r_rr^2) + (x[,"n_03"] + x[,"n_31"])*log10(19 - 20*r_rr + 4*r_rr^2) + (x[,"n_12"] + x[,"n_22"])*log10(25 - 4*r_rr + 4*r_rr^2) + (x[,"n_01"] + x[,"n_33"])*log10(3 + 12*r_rr + 4*r_rr^2)
logL_rr <- (x[,"n_02"] + x[,"n_12"] + x[,"n_22"] + x[,"n_32"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(4*log(2) + 2*log(5)) + (x[,"n_04"] + x[,"n_30"])*(log(3 - 2*r_rr) + log(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(1 + 2*r_rr)) + (x[,"n_13"] + x[,"n_21"])*log(51 - 20*r_rr - 4*r_rr^2) + (x[,"n_02"] + x[,"n_32"])*log(5 + 4*r_rr - 4*r_rr^2) + (x[,"n_11"] + x[,"n_23"])*log(27 + 28*r_rr - 4*r_rr^2) + (x[,"n_14"] + x[,"n_20"])*log(6 - 3*r_rr - 2*r_rr^2) + (x[,"n_10"] + x[,"n_24"])*log(1 + 7*r_rr - 2*r_rr^2) + (x[,"n_03"] + x[,"n_31"])*log(19 - 20*r_rr + 4*r_rr^2) + (x[,"n_12"] + x[,"n_22"])*log(25 - 4*r_rr + 4*r_rr^2) + (x[,"n_01"] + x[,"n_33"])*log(3 + 12*r_rr + 4*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_3.2 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n05 - 2*n10 - 2*n11 - n12 - n13 - 2*n14 - 2*n15 - 2*n20 - 2*n21 - n22 - n23 - 2*n24 - 2*n25 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35)*log(2) + (n10 + n15 + n20 + n25)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*log(5) + 3*(n00 + n35)*log(pmax(1e-6,1 - r)) + (n01 + n34)*(2*log(pmax(1e-6,1 - r)) + log(6 - r)) + 3*(n05 + n30)*log(pmax(1e-6,r)) + (n10 + n25)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n15 + n20)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n31)*(2*log(pmax(1e-6,r)) + log(5 + r)) + (n03 + n32)*(log(pmax(1e-6,r)) + log(7 + 4*r - 8*r^2)) + (n14 + n21)*(log(pmax(1e-6,r)) + log(11 - 2*r - 3*r^2)) + (n12 + n23)*log(12 - 17*r + 22*r^2 - 12*r^3) + (n11 + n24)*log(6 + 2*r - 11*r^2 + 3*r^3) + (n02 + n33)*log(3 + 9*r - 20*r^2 + 8*r^3) + (n13 + n22)*log(5 + 9*r - 14*r^2 + 12*r^3)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_cc <- 3*(x[,"n_15"] + x[,"n_20"])*log10(2) + 3*(x[,"n_10"] + x[,"n_25"])*log10(2) + 3*(x[,"n_05"] + x[,"n_30"])*log10(2) + 3*(x[,"n_00"] + x[,"n_35"])*log10(2) - (x[,"n_13"] + x[,"n_22"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_12"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_03"] + x[,"n_32"])*(-log10(2) + log10(7)) - (x[,"n_02"] + x[,"n_33"])*(-log10(2) + log10(7)) - (x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(11)) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_21"])*(-3*log10(2) + log10(37)) - (x[,"n_11"] + x[,"n_24"])*(-3*log10(2) + log10(37)) + 3*(x[,"n_00"] + x[,"n_35"])*log10(pmax(1e-6,1 - r_cc)) + (x[,"n_01"] + x[,"n_34"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(6 - r_cc)) + 3*(x[,"n_05"] + x[,"n_30"])*log10(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_25"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_20"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc)) + log10(5 + r_cc)) + (x[,"n_03"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(7 + 4*r_cc - 8*r_cc^2)) + (x[,"n_14"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(11 - 2*r_cc - 3*r_cc^2)) + (x[,"n_12"] + x[,"n_23"])*log10(12 - 17*r_cc + 22*r_cc^2 - 12*r_cc^3) + (x[,"n_11"] + x[,"n_24"])*log10(6 + 2*r_cc - 11*r_cc^2 + 3*r_cc^3) + (x[,"n_02"] + x[,"n_33"])*log10(3 + 9*r_cc - 20*r_cc^2 + 8*r_cc^3) + (x[,"n_13"] + x[,"n_22"])*log10(5 + 9*r_cc - 14*r_cc^2 + 12*r_cc^3)
logL_cc <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"])*log(2) + (x[,"n_10"] + x[,"n_15"] + x[,"n_20"] + x[,"n_25"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(5) + 3*(x[,"n_00"] + x[,"n_35"])*log(pmax(1e-6,1 - r_cc)) + (x[,"n_01"] + x[,"n_34"])*(2*log(pmax(1e-6,1 - r_cc)) + log(6 - r_cc)) + 3*(x[,"n_05"] + x[,"n_30"])*log(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_25"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_20"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc)) + log(5 + r_cc)) + (x[,"n_03"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(7 + 4*r_cc - 8*r_cc^2)) + (x[,"n_14"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(11 - 2*r_cc - 3*r_cc^2)) + (x[,"n_12"] + x[,"n_23"])*log(12 - 17*r_cc + 22*r_cc^2 - 12*r_cc^3) + (x[,"n_11"] + x[,"n_24"])*log(6 + 2*r_cc - 11*r_cc^2 + 3*r_cc^3) + (x[,"n_02"] + x[,"n_33"])*log(3 + 9*r_cc - 20*r_cc^2 + 8*r_cc^3) + (x[,"n_13"] + x[,"n_22"])*log(5 + 9*r_cc - 14*r_cc^2 + 12*r_cc^3)
logL_cm <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*(4*log(2) + 2*log(5)) + (n05 + n30)*(log(2 - r) + 2*log(pmax(1e-6,r))) + (n00 + n35)*(log(pmax(1e-6,1 - r)) + log(1 - r^2)) + (n04 + n31)*(log(pmax(1e-6,r)) + log(9 + r - r^2)) + (n15 + n20)*(log(pmax(1e-6,r)) + log(5 - 5*r + 3*r^2)) + (n13 + n22)*log(55 - 4*r + 36*r^2 - 24*r^3) + (n02 + n33)*log(21 - 16*r + 12*r^2 - 8*r^3) + (n10 + n25)*log(3 - 4*r + 4*r^2 - 3*r^3) + (n11 + n24)*log(27 - 17*r + 5*r^2 - 3*r^3) + (n01 + n34)*log(9 - 8*r - 2*r^2 + r^3) + (n14 + n21)*log(12 + 16*r - 4*r^2 + 3*r^3) + (n03 + n32)*log(9 + 16*r - 12*r^2 + 8*r^3) + (n12 + n23)*log(63 + 4*r - 36*r^2 + 24*r^3)
return(L)}
interlogL_cm <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_cm,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_cm <- -((x[,"n_05"] + x[,"n_30"])*(-3*log10(2) + log10(3))) - (x[,"n_00"] + x[,"n_35"])*(-3*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_32"])*(log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_33"])*(log10(3) + log10(5)) - (x[,"n_15"] + x[,"n_20"])*(-3*log10(2) + log10(13)) - (x[,"n_10"] + x[,"n_25"])*(-3*log10(2) + log10(13)) - (x[,"n_14"] + x[,"n_21"])*(-3*log10(2) + log10(5) + log10(31)) - (x[,"n_11"] + x[,"n_24"])*(-3*log10(2) + log10(5) + log10(31)) - (x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(37)) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(37)) - (x[,"n_13"] + x[,"n_22"])*log10(59) - (x[,"n_12"] + x[,"n_23"])*log10(59) + (x[,"n_05"] + x[,"n_30"])*(log10(2 - r_cm) + 2*log10(pmax(1e-6,r_cm))) + (x[,"n_00"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_cm)) + log10(1 - r_cm^2)) + (x[,"n_04"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm)) + log10(9 + r_cm - r_cm^2)) + (x[,"n_15"] + x[,"n_20"])*(log10(pmax(1e-6,r_cm)) + log10(5 - 5*r_cm + 3*r_cm^2)) + (x[,"n_13"] + x[,"n_22"])*log10(55 - 4*r_cm + 36*r_cm^2 - 24*r_cm^3) + (x[,"n_02"] + x[,"n_33"])*log10(21 - 16*r_cm + 12*r_cm^2 - 8*r_cm^3) + (x[,"n_10"] + x[,"n_25"])*log10(3 - 4*r_cm + 4*r_cm^2 - 3*r_cm^3) + (x[,"n_11"] + x[,"n_24"])*log10(27 - 17*r_cm + 5*r_cm^2 - 3*r_cm^3) + (x[,"n_01"] + x[,"n_34"])*log10(9 - 8*r_cm - 2*r_cm^2 + r_cm^3) + (x[,"n_14"] + x[,"n_21"])*log10(12 + 16*r_cm - 4*r_cm^2 + 3*r_cm^3) + (x[,"n_03"] + x[,"n_32"])*log10(9 + 16*r_cm - 12*r_cm^2 + 8*r_cm^3) + (x[,"n_12"] + x[,"n_23"])*log10(63 + 4*r_cm - 36*r_cm^2 + 24*r_cm^3)
logL_cm <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*(4*log(2) + 2*log(5)) + (x[,"n_05"] + x[,"n_30"])*(log(2 - r_cm) + 2*log(pmax(1e-6,r_cm))) + (x[,"n_00"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_cm)) + log(1 - r_cm^2)) + (x[,"n_04"] + x[,"n_31"])*(log(pmax(1e-6,r_cm)) + log(9 + r_cm - r_cm^2)) + (x[,"n_15"] + x[,"n_20"])*(log(pmax(1e-6,r_cm)) + log(5 - 5*r_cm + 3*r_cm^2)) + (x[,"n_13"] + x[,"n_22"])*log(55 - 4*r_cm + 36*r_cm^2 - 24*r_cm^3) + (x[,"n_02"] + x[,"n_33"])*log(21 - 16*r_cm + 12*r_cm^2 - 8*r_cm^3) + (x[,"n_10"] + x[,"n_25"])*log(3 - 4*r_cm + 4*r_cm^2 - 3*r_cm^3) + (x[,"n_11"] + x[,"n_24"])*log(27 - 17*r_cm + 5*r_cm^2 - 3*r_cm^3) + (x[,"n_01"] + x[,"n_34"])*log(9 - 8*r_cm - 2*r_cm^2 + r_cm^3) + (x[,"n_14"] + x[,"n_21"])*log(12 + 16*r_cm - 4*r_cm^2 + 3*r_cm^3) + (x[,"n_03"] + x[,"n_32"])*log(9 + 16*r_cm - 12*r_cm^2 + 8*r_cm^3) + (x[,"n_12"] + x[,"n_23"])*log(63 + 4*r_cm - 36*r_cm^2 + 24*r_cm^3)
logL_cr <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*(3*log(2) + log(3) + 2*log(5)) + (n00 + n35)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 + r)) + (n05 + n30)*(log(pmax(1e-6,r)) + log(3 - 4*r + r^2)) + (n12 + n23)*log(93 - 16*r + 28*r^2 - 24*r^3) + (n03 + n32)*log(27 - 18*r + 20*r^2 - 8*r^3) + (n15 + n20)*log(3 - 4*r + 7*r^2 - 3*r^3) + (n14 + n21)*log(30 - 7*r + 10*r^2 - 3*r^3) + (n01 + n34)*log(3 + 11*r - 4*r^2 - r^3) + (n04 + n31)*log(9 - 7*r^2 + r^3) + (n10 + n25)*log(3 - r - 2*r^2 + 3*r^3) + (n11 + n24)*log(30 - 4*r + r^2 + 3*r^3) + (n02 + n33)*log(21 + 2*r - 4*r^2 + 8*r^3) + (n13 + n22)*log(81 + 32*r - 44*r^2 + 24*r^3)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_cr <- -((x[,"n_05"] + x[,"n_30"])*(-3*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_35"])*(-3*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_32"])*(log10(2) + log10(11)) - (x[,"n_02"] + x[,"n_33"])*(log10(2) + log10(11)) - (x[,"n_15"] + x[,"n_20"])*(-3*log10(2) + log10(19)) - (x[,"n_10"] + x[,"n_25"])*(-3*log10(2) + log10(19)) - (x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(59)) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(59)) - (x[,"n_13"] + x[,"n_22"])*log10(89) - (x[,"n_12"] + x[,"n_23"])*log10(89) - (x[,"n_14"] + x[,"n_21"])*(-3*log10(2) + log10(229)) - (x[,"n_11"] + x[,"n_24"])*(-3*log10(2) + log10(229)) + (x[,"n_00"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr)) + log10(2 + r_cr)) + (x[,"n_05"] + x[,"n_30"])*(log10(pmax(1e-6,r_cr)) + log10(3 - 4*r_cr + r_cr^2)) + (x[,"n_12"] + x[,"n_23"])*log10(93 - 16*r_cr + 28*r_cr^2 - 24*r_cr^3) + (x[,"n_03"] + x[,"n_32"])*log10(27 - 18*r_cr + 20*r_cr^2 - 8*r_cr^3) + (x[,"n_15"] + x[,"n_20"])*log10(3 - 4*r_cr + 7*r_cr^2 - 3*r_cr^3) + (x[,"n_14"] + x[,"n_21"])*log10(30 - 7*r_cr + 10*r_cr^2 - 3*r_cr^3) + (x[,"n_01"] + x[,"n_34"])*log10(3 + 11*r_cr - 4*r_cr^2 - r_cr^3) + (x[,"n_04"] + x[,"n_31"])*log10(9 - 7*r_cr^2 + r_cr^3) + (x[,"n_10"] + x[,"n_25"])*log10(3 - r_cr - 2*r_cr^2 + 3*r_cr^3) + (x[,"n_11"] + x[,"n_24"])*log10(30 - 4*r_cr + r_cr^2 + 3*r_cr^3) + (x[,"n_02"] + x[,"n_33"])*log10(21 + 2*r_cr - 4*r_cr^2 + 8*r_cr^3) + (x[,"n_13"] + x[,"n_22"])*log10(81 + 32*r_cr - 44*r_cr^2 + 24*r_cr^3)
logL_cr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*(3*log(2) + log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr)) + log(2 + r_cr)) + (x[,"n_05"] + x[,"n_30"])*(log(pmax(1e-6,r_cr)) + log(3 - 4*r_cr + r_cr^2)) + (x[,"n_12"] + x[,"n_23"])*log(93 - 16*r_cr + 28*r_cr^2 - 24*r_cr^3) + (x[,"n_03"] + x[,"n_32"])*log(27 - 18*r_cr + 20*r_cr^2 - 8*r_cr^3) + (x[,"n_15"] + x[,"n_20"])*log(3 - 4*r_cr + 7*r_cr^2 - 3*r_cr^3) + (x[,"n_14"] + x[,"n_21"])*log(30 - 7*r_cr + 10*r_cr^2 - 3*r_cr^3) + (x[,"n_01"] + x[,"n_34"])*log(3 + 11*r_cr - 4*r_cr^2 - r_cr^3) + (x[,"n_04"] + x[,"n_31"])*log(9 - 7*r_cr^2 + r_cr^3) + (x[,"n_10"] + x[,"n_25"])*log(3 - r_cr - 2*r_cr^2 + 3*r_cr^3) + (x[,"n_11"] + x[,"n_24"])*log(30 - 4*r_cr + r_cr^2 + 3*r_cr^3) + (x[,"n_02"] + x[,"n_33"])*log(21 + 2*r_cr - 4*r_cr^2 + 8*r_cr^3) + (x[,"n_13"] + x[,"n_22"])*log(81 + 32*r_cr - 44*r_cr^2 + 24*r_cr^3)
logL_rc <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n05 - 2*n10 - 2*n11 - n12 - n13 - 2*n14 - 2*n15 - 2*n20 - 2*n21 - n22 - n23 - 2*n24 - 2*n25 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35)*log(2) + (n12 + n13 + n22 + n23)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*log(5) + (n00 + n35)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n05 + n30)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n31)*(log(pmax(1e-6,r)) + log(2 + 2*r - r^2)) + (n15 + n20)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n02 + n33)*log(6 - 9*r + 12*r^2 - 8*r^3) + (n13 + n22)*log(3 - 3*r + 6*r^2 - 4*r^3) + (n11 + n24)*log(6 - 4*r + 4*r^2 - 3*r^3) + (n10 + n25)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n01 + n34)*log(3 - 3*r - r^2 + r^3) + (n14 + n21)*log(3 + 5*r - 5*r^2 + 3*r^3) + (n12 + n23)*log(2 + 3*r - 6*r^2 + 4*r^3) + (n03 + n32)*log(1 + 9*r - 12*r^2 + 8*r^3)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_rc <- 3*(x[,"n_05"] + x[,"n_30"])*log10(2) + 3*(x[,"n_00"] + x[,"n_35"])*log10(2) + (x[,"n_10"] + x[,"n_25"])*(3*log10(2) - log10(3)) - (x[,"n_15"] + x[,"n_20"])*(-3*log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_22"])*(-log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_23"])*(-log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_32"])*(-log10(2) + log10(7)) - (x[,"n_02"] + x[,"n_33"])*(-log10(2) + log10(7)) - (x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(11)) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_21"])*(-3*log10(2) + log10(37)) - (x[,"n_11"] + x[,"n_24"])*(-3*log10(2) + log10(37)) + (x[,"n_00"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc))) + (x[,"n_05"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_rc)) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_04"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(2 + 2*r_rc - r_rc^2)) + (x[,"n_15"] + x[,"n_20"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 4*r_rc + 3*r_rc^2)) + (x[,"n_02"] + x[,"n_33"])*log10(6 - 9*r_rc + 12*r_rc^2 - 8*r_rc^3) + (x[,"n_13"] + x[,"n_22"])*log10(3 - 3*r_rc + 6*r_rc^2 - 4*r_rc^3) + (x[,"n_11"] + x[,"n_24"])*log10(6 - 4*r_rc + 4*r_rc^2 - 3*r_rc^3) + (x[,"n_10"] + x[,"n_25"])*log10(1 - 3*r_rc + 5*r_rc^2 - 3*r_rc^3) + (x[,"n_01"] + x[,"n_34"])*log10(3 - 3*r_rc - r_rc^2 + r_rc^3) + (x[,"n_14"] + x[,"n_21"])*log10(3 + 5*r_rc - 5*r_rc^2 + 3*r_rc^3) + (x[,"n_12"] + x[,"n_23"])*log10(2 + 3*r_rc - 6*r_rc^2 + 4*r_rc^3) + (x[,"n_03"] + x[,"n_32"])*log10(1 + 9*r_rc - 12*r_rc^2 + 8*r_rc^3)
logL_rc <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"])*log(2) + (x[,"n_12"] + x[,"n_13"] + x[,"n_22"] + x[,"n_23"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*log(5) + (x[,"n_00"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc))) + (x[,"n_05"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_rc)) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_04"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(2 + 2*r_rc - r_rc^2)) + (x[,"n_15"] + x[,"n_20"])*(log(pmax(1e-6,r_rc)) + log(2 - 4*r_rc + 3*r_rc^2)) + (x[,"n_02"] + x[,"n_33"])*log(6 - 9*r_rc + 12*r_rc^2 - 8*r_rc^3) + (x[,"n_13"] + x[,"n_22"])*log(3 - 3*r_rc + 6*r_rc^2 - 4*r_rc^3) + (x[,"n_11"] + x[,"n_24"])*log(6 - 4*r_rc + 4*r_rc^2 - 3*r_rc^3) + (x[,"n_10"] + x[,"n_25"])*log(1 - 3*r_rc + 5*r_rc^2 - 3*r_rc^3) + (x[,"n_01"] + x[,"n_34"])*log(3 - 3*r_rc - r_rc^2 + r_rc^3) + (x[,"n_14"] + x[,"n_21"])*log(3 + 5*r_rc - 5*r_rc^2 + 3*r_rc^3) + (x[,"n_12"] + x[,"n_23"])*log(2 + 3*r_rc - 6*r_rc^2 + 4*r_rc^3) + (x[,"n_03"] + x[,"n_32"])*log(1 + 9*r_rc - 12*r_rc^2 + 8*r_rc^3)
logL_rm <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*(4*log(2) + 2*log(5)) + (n05 + n30)*(log(pmax(1e-6,1 - r)) + log(2 - r) + log(pmax(1e-6,r))) + (n00 + n35)*(log(pmax(1e-6,r)) + log(1 - r^2)) + (n12 + n23)*log(57 - 4*r + 28*r^2 - 24*r^3) + (n03 + n32)*log(19 - 16*r + 20*r^2 - 8*r^3) + (n15 + n20)*log(3 - 5*r + 6*r^2 - 3*r^3) + (n14 + n21)*log(24 - 14*r + 11*r^2 - 3*r^3) + (n01 + n34)*log(3 + 6*r - 5*r^2 - r^3) + (n04 + n31)*log(3 + 7*r - 8*r^2 + r^3) + (n10 + n25)*log(1 + 2*r - 3*r^2 + 3*r^3) + (n11 + n24)*log(18 + r + 2*r^2 + 3*r^3) + (n02 + n33)*log(15 - 4*r^2 + 8*r^3) + (n13 + n22)*log(57 + 20*r - 44*r^2 + 24*r^3)
return(L)}
interlogL_rm <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_rm,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_rm <- -((x[,"n_05"] + x[,"n_30"])*(-3*log10(2) + log10(3))) - (x[,"n_00"] + x[,"n_35"])*(-3*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_32"])*(log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_33"])*(log10(3) + log10(5)) - (x[,"n_15"] + x[,"n_20"])*(-3*log10(2) + log10(13)) - (x[,"n_10"] + x[,"n_25"])*(-3*log10(2) + log10(13)) - (x[,"n_14"] + x[,"n_21"])*(-3*log10(2) + log10(5) + log10(31)) - (x[,"n_11"] + x[,"n_24"])*(-3*log10(2) + log10(5) + log10(31)) - (x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(37)) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(37)) - (x[,"n_13"] + x[,"n_22"])*log10(59) - (x[,"n_12"] + x[,"n_23"])*log10(59) + (x[,"n_05"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_rm)) + log10(2 - r_rm) + log10(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_35"])*(log10(pmax(1e-6,r_rm)) + log10(1 - r_rm^2)) + (x[,"n_12"] + x[,"n_23"])*log10(57 - 4*r_rm + 28*r_rm^2 - 24*r_rm^3) + (x[,"n_03"] + x[,"n_32"])*log10(19 - 16*r_rm + 20*r_rm^2 - 8*r_rm^3) + (x[,"n_15"] + x[,"n_20"])*log10(3 - 5*r_rm + 6*r_rm^2 - 3*r_rm^3) + (x[,"n_14"] + x[,"n_21"])*log10(24 - 14*r_rm + 11*r_rm^2 - 3*r_rm^3) + (x[,"n_01"] + x[,"n_34"])*log10(3 + 6*r_rm - 5*r_rm^2 - r_rm^3) + (x[,"n_04"] + x[,"n_31"])*log10(3 + 7*r_rm - 8*r_rm^2 + r_rm^3) + (x[,"n_10"] + x[,"n_25"])*log10(1 + 2*r_rm - 3*r_rm^2 + 3*r_rm^3) + (x[,"n_11"] + x[,"n_24"])*log10(18 + r_rm + 2*r_rm^2 + 3*r_rm^3) + (x[,"n_02"] + x[,"n_33"])*log10(15 - 4*r_rm^2 + 8*r_rm^3) + (x[,"n_13"] + x[,"n_22"])*log10(57 + 20*r_rm - 44*r_rm^2 + 24*r_rm^3)
logL_rm <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*(4*log(2) + 2*log(5)) + (x[,"n_05"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_rm)) + log(2 - r_rm) + log(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_35"])*(log(pmax(1e-6,r_rm)) + log(1 - r_rm^2)) + (x[,"n_12"] + x[,"n_23"])*log(57 - 4*r_rm + 28*r_rm^2 - 24*r_rm^3) + (x[,"n_03"] + x[,"n_32"])*log(19 - 16*r_rm + 20*r_rm^2 - 8*r_rm^3) + (x[,"n_15"] + x[,"n_20"])*log(3 - 5*r_rm + 6*r_rm^2 - 3*r_rm^3) + (x[,"n_14"] + x[,"n_21"])*log(24 - 14*r_rm + 11*r_rm^2 - 3*r_rm^3) + (x[,"n_01"] + x[,"n_34"])*log(3 + 6*r_rm - 5*r_rm^2 - r_rm^3) + (x[,"n_04"] + x[,"n_31"])*log(3 + 7*r_rm - 8*r_rm^2 + r_rm^3) + (x[,"n_10"] + x[,"n_25"])*log(1 + 2*r_rm - 3*r_rm^2 + 3*r_rm^3) + (x[,"n_11"] + x[,"n_24"])*log(18 + r_rm + 2*r_rm^2 + 3*r_rm^3) + (x[,"n_02"] + x[,"n_33"])*log(15 - 4*r_rm^2 + 8*r_rm^3) + (x[,"n_13"] + x[,"n_22"])*log(57 + 20*r_rm - 44*r_rm^2 + 24*r_rm^3)
logL_rr <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35)*(3*log(2) + log(3) + 2*log(5)) + (n00 + n35)*(2*log(pmax(1e-6,r)) + log(2 + r)) + (n10 + n25)*(log(pmax(1e-6,r)) + log(5 + r - 3*r^2)) + (n05 + n30)*(log(pmax(1e-6,1 - r)) + log(3 - 4*r + r^2)) + (n01 + n34)*(log(pmax(1e-6,r)) + log(9 + 11*r + r^2)) + (n13 + n22)*log(111 - 64*r + 52*r^2 - 24*r^3) + (n02 + n33)*log(9 + 30*r - 4*r^2 - 8*r^3) + (n11 + n24)*log(12 + 38*r - 8*r^2 - 3*r^3) + (n04 + n31)*log(21 - 34*r + 14*r^2 - r^3) + (n14 + n21)*log(39 - 13*r - 17*r^2 + 3*r^3) + (n15 + n20)*log(3 + 2*r - 8*r^2 + 3*r^3) + (n03 + n32)*log(27 + 2*r - 28*r^2 + 8*r^3) + (n12 + n23)*log(75 + 32*r - 20*r^2 + 24*r^3)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"])
LOD_rr <- -((x[,"n_05"] + x[,"n_30"])*(-3*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_35"])*(-3*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_32"])*(log10(2) + log10(11)) - (x[,"n_02"] + x[,"n_33"])*(log10(2) + log10(11)) - (x[,"n_15"] + x[,"n_20"])*(-3*log10(2) + log10(19)) - (x[,"n_10"] + x[,"n_25"])*(-3*log10(2) + log10(19)) - (x[,"n_04"] + x[,"n_31"])*(-3*log10(2) + log10(59)) - (x[,"n_01"] + x[,"n_34"])*(-3*log10(2) + log10(59)) - (x[,"n_13"] + x[,"n_22"])*log10(89) - (x[,"n_12"] + x[,"n_23"])*log10(89) - (x[,"n_14"] + x[,"n_21"])*(-3*log10(2) + log10(229)) - (x[,"n_11"] + x[,"n_24"])*(-3*log10(2) + log10(229)) + (x[,"n_00"] + x[,"n_35"])*(2*log10(pmax(1e-6,r_rr)) + log10(2 + r_rr)) + (x[,"n_10"] + x[,"n_25"])*(log10(pmax(1e-6,r_rr)) + log10(5 + r_rr - 3*r_rr^2)) + (x[,"n_05"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_rr)) + log10(3 - 4*r_rr + r_rr^2)) + (x[,"n_01"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(9 + 11*r_rr + r_rr^2)) + (x[,"n_13"] + x[,"n_22"])*log10(111 - 64*r_rr + 52*r_rr^2 - 24*r_rr^3) + (x[,"n_02"] + x[,"n_33"])*log10(9 + 30*r_rr - 4*r_rr^2 - 8*r_rr^3) + (x[,"n_11"] + x[,"n_24"])*log10(12 + 38*r_rr - 8*r_rr^2 - 3*r_rr^3) + (x[,"n_04"] + x[,"n_31"])*log10(21 - 34*r_rr + 14*r_rr^2 - r_rr^3) + (x[,"n_14"] + x[,"n_21"])*log10(39 - 13*r_rr - 17*r_rr^2 + 3*r_rr^3) + (x[,"n_15"] + x[,"n_20"])*log10(3 + 2*r_rr - 8*r_rr^2 + 3*r_rr^3) + (x[,"n_03"] + x[,"n_32"])*log10(27 + 2*r_rr - 28*r_rr^2 + 8*r_rr^3) + (x[,"n_12"] + x[,"n_23"])*log10(75 + 32*r_rr - 20*r_rr^2 + 24*r_rr^3)
logL_rr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"])*(3*log(2) + log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_35"])*(2*log(pmax(1e-6,r_rr)) + log(2 + r_rr)) + (x[,"n_10"] + x[,"n_25"])*(log(pmax(1e-6,r_rr)) + log(5 + r_rr - 3*r_rr^2)) + (x[,"n_05"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_rr)) + log(3 - 4*r_rr + r_rr^2)) + (x[,"n_01"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(9 + 11*r_rr + r_rr^2)) + (x[,"n_13"] + x[,"n_22"])*log(111 - 64*r_rr + 52*r_rr^2 - 24*r_rr^3) + (x[,"n_02"] + x[,"n_33"])*log(9 + 30*r_rr - 4*r_rr^2 - 8*r_rr^3) + (x[,"n_11"] + x[,"n_24"])*log(12 + 38*r_rr - 8*r_rr^2 - 3*r_rr^3) + (x[,"n_04"] + x[,"n_31"])*log(21 - 34*r_rr + 14*r_rr^2 - r_rr^3) + (x[,"n_14"] + x[,"n_21"])*log(39 - 13*r_rr - 17*r_rr^2 + 3*r_rr^3) + (x[,"n_15"] + x[,"n_20"])*log(3 + 2*r_rr - 8*r_rr^2 + 3*r_rr^3) + (x[,"n_03"] + x[,"n_32"])*log(27 + 2*r_rr - 28*r_rr^2 + 8*r_rr^3) + (x[,"n_12"] + x[,"n_23"])*log(75 + 32*r_rr - 20*r_rr^2 + 24*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_3.3 <- function(x){
logL_cc <- function(r,n10,n16,n20,n26,n00,n01,n02,n03,n04,n05,n06,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n36) {
L <- (n10 + n16 + n20 + n26)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36)*(4*log(2) + 2*log(5)) + 3*(n00 + n36)*log(pmax(1e-6,1 - r)) + (n01 + n35)*(log(9 - 2*r) + 2*log(pmax(1e-6,1 - r))) + 3*(n06 + n30)*log(pmax(1e-6,r)) + (n10 + n26)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n16 + n20)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n05 + n31)*(2*log(pmax(1e-6,r)) + log(7 + 2*r)) + (n03 + n33)*log(9 + 28*r - 28*r^2) + (n04 + n32)*(log(pmax(1e-6,r)) + log(15 + 13*r - 7*r^2)) + (n15 + n21)*(log(pmax(1e-6,r)) + log(16 - r - 6*r^2)) + (n13 + n23)*log(73 - 28*r + 28*r^2) + (n12 + n24)*log(57 - 44*r + 29*r^2 - 21*r^3) + (n11 + n25)*log(9 + 4*r - 19*r^2 + 6*r^3) + (n02 + n34)*log(21 - 20*r - 8*r^2 + 7*r^3) + (n14 + n22)*log(21 + 49*r - 34*r^2 + 21*r^3)
return(L)}
interlogL_cc <- function(n10,n16,n20,n26,n00,n01,n02,n03,n04,n05,n06,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n36) {
optimize(logL_cc,c(0,0.5), n10,n16,n20,n26,n00,n01,n02,n03,n04,n05,n06,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n36, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_10"],x[,"n_16"],x[,"n_20"],x[,"n_26"],x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"])
LOD_cc <- 3*(x[,"n_16"] + x[,"n_20"])*log10(2) + 3*(x[,"n_10"] + x[,"n_26"])*log10(2) + 3*(x[,"n_06"] + x[,"n_30"])*log10(2) - (x[,"n_05"] + x[,"n_31"])*log10(2) - 4*(x[,"n_03"] + x[,"n_33"])*log10(2) - (x[,"n_01"] + x[,"n_35"])*log10(2) + 3*(x[,"n_00"] + x[,"n_36"])*log10(2) - (x[,"n_15"] + x[,"n_21"])*log10(7) - (x[,"n_11"] + x[,"n_25"])*log10(7) - (x[,"n_13"] + x[,"n_23"])*(log10(2) + log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_32"])*(-3*log10(2) + log10(79)) - (x[,"n_02"] + x[,"n_34"])*(-3*log10(2) + log10(79)) - (x[,"n_14"] + x[,"n_22"])*(-3*log10(2) + log10(317)) - (x[,"n_12"] + x[,"n_24"])*(-3*log10(2) + log10(317)) + 3*(x[,"n_00"] + x[,"n_36"])*log10(pmax(1e-6,1 - r_cc)) + (x[,"n_01"] + x[,"n_35"])*(log10(9 - 2*r_cc) + 2*log10(pmax(1e-6,1 - r_cc))) + 3*(x[,"n_06"] + x[,"n_30"])*log10(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_26"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_16"] + x[,"n_20"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_05"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc)) + log10(7 + 2*r_cc)) + (x[,"n_03"] + x[,"n_33"])*log10(9 + 28*r_cc - 28*r_cc^2) + (x[,"n_04"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(15 + 13*r_cc - 7*r_cc^2)) + (x[,"n_15"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc)) + log10(16 - r_cc - 6*r_cc^2)) + (x[,"n_13"] + x[,"n_23"])*log10(73 - 28*r_cc + 28*r_cc^2) + (x[,"n_12"] + x[,"n_24"])*log10(57 - 44*r_cc + 29*r_cc^2 - 21*r_cc^3) + (x[,"n_11"] + x[,"n_25"])*log10(9 + 4*r_cc - 19*r_cc^2 + 6*r_cc^3) + (x[,"n_02"] + x[,"n_34"])*log10(21 - 20*r_cc - 8*r_cc^2 + 7*r_cc^3) + (x[,"n_14"] + x[,"n_22"])*log10(21 + 49*r_cc - 34*r_cc^2 + 21*r_cc^3)
logL_cc <- (x[,"n_10"] + x[,"n_16"] + x[,"n_20"] + x[,"n_26"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"])*(4*log(2) + 2*log(5)) + 3*(x[,"n_00"] + x[,"n_36"])*log(pmax(1e-6,1 - r_cc)) + (x[,"n_01"] + x[,"n_35"])*(log(9 - 2*r_cc) + 2*log(pmax(1e-6,1 - r_cc))) + 3*(x[,"n_06"] + x[,"n_30"])*log(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_26"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_16"] + x[,"n_20"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_05"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc)) + log(7 + 2*r_cc)) + (x[,"n_03"] + x[,"n_33"])*log(9 + 28*r_cc - 28*r_cc^2) + (x[,"n_04"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(15 + 13*r_cc - 7*r_cc^2)) + (x[,"n_15"] + x[,"n_21"])*(log(pmax(1e-6,r_cc)) + log(16 - r_cc - 6*r_cc^2)) + (x[,"n_13"] + x[,"n_23"])*log(73 - 28*r_cc + 28*r_cc^2) + (x[,"n_12"] + x[,"n_24"])*log(57 - 44*r_cc + 29*r_cc^2 - 21*r_cc^3) + (x[,"n_11"] + x[,"n_25"])*log(9 + 4*r_cc - 19*r_cc^2 + 6*r_cc^3) + (x[,"n_02"] + x[,"n_34"])*log(21 - 20*r_cc - 8*r_cc^2 + 7*r_cc^3) + (x[,"n_14"] + x[,"n_22"])*log(21 + 49*r_cc - 34*r_cc^2 + 21*r_cc^3)
logL_cm <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n13,n23) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 3*n05 - 3*n06 - 4*n10 - 4*n11 - 4*n12 - 4*n14 - 4*n15 - 4*n16 - 4*n20 - 4*n21 - 4*n22 - 4*n24 - 4*n25 - 4*n26 - 3*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 3*n35 - 3*n36)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36)*(log(3) + 2*log(5)) + (n00 + n36)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n06 + n30)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n05 + n31)*(log(pmax(1e-6,r)) + log(6 - r - 2*r^2)) + (n13 + n23)*log(12 + r - r^2) + (n16 + n20)*(log(pmax(1e-6,r)) + log(5 - 8*r + 6*r^2)) + (n03 + n33)*log(27 - 8*r + 8*r^2) + (n14 + n22)*log(105 + 7*r + 62*r^2 - 42*r^3) + (n11 + n25)*log(33 - 32*r + 26*r^2 - 12*r^3) + (n02 + n34)*log(21 - 16*r + 11*r^2 - 7*r^3) + (n10 + n26)*log(3 - 7*r + 10*r^2 - 6*r^3) + (n01 + n35)*log(3 + 2*r - 7*r^2 + 2*r^3) + (n04 + n32)*log(9 + 15*r - 10*r^2 + 7*r^3) + (n15 + n21)*log(15 + 16*r - 10*r^2 + 12*r^3) + (n12 + n24)*log(132 - 5*r - 64*r^2 + 42*r^3)
return(L)}
interlogL_cm <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n13,n23) {
optimize(logL_cm,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n13,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_13"],x[,"n_23"])
LOD_cm <- 3*(x[,"n_06"] + x[,"n_30"])*log10(2) + 3*(x[,"n_00"] + x[,"n_36"])*log10(2) - 2*(x[,"n_03"] + x[,"n_33"])*log10(5) - (x[,"n_16"] + x[,"n_20"])*(-2*log10(2) + log10(5)) - (x[,"n_10"] + x[,"n_26"])*(-2*log10(2) + log10(5)) - (x[,"n_05"] + x[,"n_31"])*(-log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_35"])*(-log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_23"])*(-2*log10(2) + 2*log10(7)) - (x[,"n_15"] + x[,"n_21"])*(log10(2) + log10(11)) - (x[,"n_11"] + x[,"n_25"])*(log10(2) + log10(11)) - (x[,"n_04"] + x[,"n_32"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_02"] + x[,"n_34"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_14"] + x[,"n_22"])*(-2*log10(2) + 2*log10(5) + log10(19)) - (x[,"n_12"] + x[,"n_24"])*(-2*log10(2) + 2*log10(5) + log10(19)) + (x[,"n_00"] + x[,"n_36"])*(2*log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + (x[,"n_06"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_cm)) + 2*log10(pmax(1e-6,r_cm))) + (x[,"n_05"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm)) + log10(6 - r_cm - 2*r_cm^2)) + (x[,"n_13"] + x[,"n_23"])*log10(12 + r_cm - r_cm^2) + (x[,"n_16"] + x[,"n_20"])*(log10(pmax(1e-6,r_cm)) + log10(5 - 8*r_cm + 6*r_cm^2)) + (x[,"n_03"] + x[,"n_33"])*log10(27 - 8*r_cm + 8*r_cm^2) + (x[,"n_14"] + x[,"n_22"])*log10(105 + 7*r_cm + 62*r_cm^2 - 42*r_cm^3) + (x[,"n_11"] + x[,"n_25"])*log10(33 - 32*r_cm + 26*r_cm^2 - 12*r_cm^3) + (x[,"n_02"] + x[,"n_34"])*log10(21 - 16*r_cm + 11*r_cm^2 - 7*r_cm^3) + (x[,"n_10"] + x[,"n_26"])*log10(3 - 7*r_cm + 10*r_cm^2 - 6*r_cm^3) + (x[,"n_01"] + x[,"n_35"])*log10(3 + 2*r_cm - 7*r_cm^2 + 2*r_cm^3) + (x[,"n_04"] + x[,"n_32"])*log10(9 + 15*r_cm - 10*r_cm^2 + 7*r_cm^3) + (x[,"n_15"] + x[,"n_21"])*log10(15 + 16*r_cm - 10*r_cm^2 + 12*r_cm^3) + (x[,"n_12"] + x[,"n_24"])*log10(132 - 5*r_cm - 64*r_cm^2 + 42*r_cm^3)
logL_cm <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_05"] - 3*x[,"n_06"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_14"] - 4*x[,"n_15"] - 4*x[,"n_16"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_35"] - 3*x[,"n_36"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"])*(log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_36"])*(2*log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + (x[,"n_06"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_cm)) + 2*log(pmax(1e-6,r_cm))) + (x[,"n_05"] + x[,"n_31"])*(log(pmax(1e-6,r_cm)) + log(6 - r_cm - 2*r_cm^2)) + (x[,"n_13"] + x[,"n_23"])*log(12 + r_cm - r_cm^2) + (x[,"n_16"] + x[,"n_20"])*(log(pmax(1e-6,r_cm)) + log(5 - 8*r_cm + 6*r_cm^2)) + (x[,"n_03"] + x[,"n_33"])*log(27 - 8*r_cm + 8*r_cm^2) + (x[,"n_14"] + x[,"n_22"])*log(105 + 7*r_cm + 62*r_cm^2 - 42*r_cm^3) + (x[,"n_11"] + x[,"n_25"])*log(33 - 32*r_cm + 26*r_cm^2 - 12*r_cm^3) + (x[,"n_02"] + x[,"n_34"])*log(21 - 16*r_cm + 11*r_cm^2 - 7*r_cm^3) + (x[,"n_10"] + x[,"n_26"])*log(3 - 7*r_cm + 10*r_cm^2 - 6*r_cm^3) + (x[,"n_01"] + x[,"n_35"])*log(3 + 2*r_cm - 7*r_cm^2 + 2*r_cm^3) + (x[,"n_04"] + x[,"n_32"])*log(9 + 15*r_cm - 10*r_cm^2 + 7*r_cm^3) + (x[,"n_15"] + x[,"n_21"])*log(15 + 16*r_cm - 10*r_cm^2 + 12*r_cm^3) + (x[,"n_12"] + x[,"n_24"])*log(132 - 5*r_cm - 64*r_cm^2 + 42*r_cm^3)
logL_cr <- function(r,n13,n23,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36) {
L <- (n13 + n23)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36)*(4*log(2) + 2*log(5)) + (n06 + n30)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n36)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n13 + n23)*log(21 + 4*r - 4*r^2) + (n01 + n35)*(log(pmax(1e-6,r)) + log(4 + r - 2*r^2)) + (n10 + n26)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n03 + n33)*log(19 - 12*r + 12*r^2) + (n12 + n24)*log(42 - 17*r + 35*r^2 - 21*r^3) + (n04 + n32)*log(15 - 12*r + 7*r^2 - 7*r^3) + (n15 + n21)*log(9 - 8*r + 11*r^2 - 6*r^3) + (n16 + n20)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n05 + n31)*log(3 - 5*r^2 + 2*r^3) + (n11 + n25)*log(6 + 4*r - 7*r^2 + 6*r^3) + (n02 + n34)*log(3 + 19*r - 14*r^2 + 7*r^3) + (n14 + n22)*log(39 + 10*r - 28*r^2 + 21*r^3)
return(L)}
interlogL_cr <- function(n13,n23,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36) {
optimize(logL_cr,c(0,0.5), n13,n23,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_13"],x[,"n_23"],x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"])
LOD_cr <- 3*(x[,"n_06"] + x[,"n_30"])*log10(2) - (x[,"n_05"] + x[,"n_31"])*log10(2) - 4*(x[,"n_03"] + x[,"n_33"])*log10(2) - (x[,"n_01"] + x[,"n_35"])*log10(2) + 3*(x[,"n_00"] + x[,"n_36"])*log10(2) + (x[,"n_16"] + x[,"n_20"])*(3*log10(2) - log10(3)) - (x[,"n_10"] + x[,"n_26"])*(-3*log10(2) + log10(3)) - (x[,"n_15"] + x[,"n_21"])*log10(7) - (x[,"n_11"] + x[,"n_25"])*log10(7) - (x[,"n_13"] + x[,"n_23"])*(log10(2) + log10(11)) - (x[,"n_04"] + x[,"n_32"])*(-3*log10(2) + log10(79)) - (x[,"n_02"] + x[,"n_34"])*(-3*log10(2) + log10(79)) - (x[,"n_14"] + x[,"n_22"])*(-3*log10(2) + log10(317)) - (x[,"n_12"] + x[,"n_24"])*(-3*log10(2) + log10(317)) + (x[,"n_06"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_36"])*(log10(pmax(1e-6,1 - r_cr)) + 2*log10(pmax(1e-6,r_cr))) + (x[,"n_13"] + x[,"n_23"])*log10(21 + 4*r_cr - 4*r_cr^2) + (x[,"n_01"] + x[,"n_35"])*(log10(pmax(1e-6,r_cr)) + log10(4 + r_cr - 2*r_cr^2)) + (x[,"n_10"] + x[,"n_26"])*(log10(pmax(1e-6,r_cr)) + log10(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_03"] + x[,"n_33"])*log10(19 - 12*r_cr + 12*r_cr^2) + (x[,"n_12"] + x[,"n_24"])*log10(42 - 17*r_cr + 35*r_cr^2 - 21*r_cr^3) + (x[,"n_04"] + x[,"n_32"])*log10(15 - 12*r_cr + 7*r_cr^2 - 7*r_cr^3) + (x[,"n_15"] + x[,"n_21"])*log10(9 - 8*r_cr + 11*r_cr^2 - 6*r_cr^3) + (x[,"n_16"] + x[,"n_20"])*log10(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_05"] + x[,"n_31"])*log10(3 - 5*r_cr^2 + 2*r_cr^3) + (x[,"n_11"] + x[,"n_25"])*log10(6 + 4*r_cr - 7*r_cr^2 + 6*r_cr^3) + (x[,"n_02"] + x[,"n_34"])*log10(3 + 19*r_cr - 14*r_cr^2 + 7*r_cr^3) + (x[,"n_14"] + x[,"n_22"])*log10(39 + 10*r_cr - 28*r_cr^2 + 21*r_cr^3)
logL_cr <- (x[,"n_13"] + x[,"n_23"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"])*(4*log(2) + 2*log(5)) + (x[,"n_06"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_36"])*(log(pmax(1e-6,1 - r_cr)) + 2*log(pmax(1e-6,r_cr))) + (x[,"n_13"] + x[,"n_23"])*log(21 + 4*r_cr - 4*r_cr^2) + (x[,"n_01"] + x[,"n_35"])*(log(pmax(1e-6,r_cr)) + log(4 + r_cr - 2*r_cr^2)) + (x[,"n_10"] + x[,"n_26"])*(log(pmax(1e-6,r_cr)) + log(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_03"] + x[,"n_33"])*log(19 - 12*r_cr + 12*r_cr^2) + (x[,"n_12"] + x[,"n_24"])*log(42 - 17*r_cr + 35*r_cr^2 - 21*r_cr^3) + (x[,"n_04"] + x[,"n_32"])*log(15 - 12*r_cr + 7*r_cr^2 - 7*r_cr^3) + (x[,"n_15"] + x[,"n_21"])*log(9 - 8*r_cr + 11*r_cr^2 - 6*r_cr^3) + (x[,"n_16"] + x[,"n_20"])*log(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_05"] + x[,"n_31"])*log(3 - 5*r_cr^2 + 2*r_cr^3) + (x[,"n_11"] + x[,"n_25"])*log(6 + 4*r_cr - 7*r_cr^2 + 6*r_cr^3) + (x[,"n_02"] + x[,"n_34"])*log(3 + 19*r_cr - 14*r_cr^2 + 7*r_cr^3) + (x[,"n_14"] + x[,"n_22"])*log(39 + 10*r_cr - 28*r_cr^2 + 21*r_cr^3)
logL_rc <- function(r,n13,n23,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36) {
L <- (n13 + n23)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36)*(4*log(2) + 2*log(5)) + (n00 + n36)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n06 + n30)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n13 + n23)*log(21 + 4*r - 4*r^2) + (n05 + n31)*(log(pmax(1e-6,r)) + log(4 + r - 2*r^2)) + (n16 + n20)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n03 + n33)*log(19 - 12*r + 12*r^2) + (n14 + n22)*log(42 - 17*r + 35*r^2 - 21*r^3) + (n02 + n34)*log(15 - 12*r + 7*r^2 - 7*r^3) + (n11 + n25)*log(9 - 8*r + 11*r^2 - 6*r^3) + (n10 + n26)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n01 + n35)*log(3 - 5*r^2 + 2*r^3) + (n15 + n21)*log(6 + 4*r - 7*r^2 + 6*r^3) + (n04 + n32)*log(3 + 19*r - 14*r^2 + 7*r^3) + (n12 + n24)*log(39 + 10*r - 28*r^2 + 21*r^3)
return(L)}
interlogL_rc <- function(n13,n23,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36) {
optimize(logL_rc,c(0,0.5), n13,n23,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_13"],x[,"n_23"],x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"])
LOD_rc <- 3*(x[,"n_06"] + x[,"n_30"])*log10(2) - (x[,"n_05"] + x[,"n_31"])*log10(2) - 4*(x[,"n_03"] + x[,"n_33"])*log10(2) - (x[,"n_01"] + x[,"n_35"])*log10(2) + 3*(x[,"n_00"] + x[,"n_36"])*log10(2) + (x[,"n_10"] + x[,"n_26"])*(3*log10(2) - log10(3)) - (x[,"n_16"] + x[,"n_20"])*(-3*log10(2) + log10(3)) - (x[,"n_15"] + x[,"n_21"])*log10(7) - (x[,"n_11"] + x[,"n_25"])*log10(7) - (x[,"n_13"] + x[,"n_23"])*(log10(2) + log10(11)) - (x[,"n_04"] + x[,"n_32"])*(-3*log10(2) + log10(79)) - (x[,"n_02"] + x[,"n_34"])*(-3*log10(2) + log10(79)) - (x[,"n_14"] + x[,"n_22"])*(-3*log10(2) + log10(317)) - (x[,"n_12"] + x[,"n_24"])*(-3*log10(2) + log10(317)) + (x[,"n_00"] + x[,"n_36"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc))) + (x[,"n_06"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_rc)) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_13"] + x[,"n_23"])*log10(21 + 4*r_rc - 4*r_rc^2) + (x[,"n_05"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(4 + r_rc - 2*r_rc^2)) + (x[,"n_16"] + x[,"n_20"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 4*r_rc + 3*r_rc^2)) + (x[,"n_03"] + x[,"n_33"])*log10(19 - 12*r_rc + 12*r_rc^2) + (x[,"n_14"] + x[,"n_22"])*log10(42 - 17*r_rc + 35*r_rc^2 - 21*r_rc^3) + (x[,"n_02"] + x[,"n_34"])*log10(15 - 12*r_rc + 7*r_rc^2 - 7*r_rc^3) + (x[,"n_11"] + x[,"n_25"])*log10(9 - 8*r_rc + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_10"] + x[,"n_26"])*log10(1 - 3*r_rc + 5*r_rc^2 - 3*r_rc^3) + (x[,"n_01"] + x[,"n_35"])*log10(3 - 5*r_rc^2 + 2*r_rc^3) + (x[,"n_15"] + x[,"n_21"])*log10(6 + 4*r_rc - 7*r_rc^2 + 6*r_rc^3) + (x[,"n_04"] + x[,"n_32"])*log10(3 + 19*r_rc - 14*r_rc^2 + 7*r_rc^3) + (x[,"n_12"] + x[,"n_24"])*log10(39 + 10*r_rc - 28*r_rc^2 + 21*r_rc^3)
logL_rc <- (x[,"n_13"] + x[,"n_23"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"])*(4*log(2) + 2*log(5)) + (x[,"n_00"] + x[,"n_36"])*(2*log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc))) + (x[,"n_06"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_rc)) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_13"] + x[,"n_23"])*log(21 + 4*r_rc - 4*r_rc^2) + (x[,"n_05"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(4 + r_rc - 2*r_rc^2)) + (x[,"n_16"] + x[,"n_20"])*(log(pmax(1e-6,r_rc)) + log(2 - 4*r_rc + 3*r_rc^2)) + (x[,"n_03"] + x[,"n_33"])*log(19 - 12*r_rc + 12*r_rc^2) + (x[,"n_14"] + x[,"n_22"])*log(42 - 17*r_rc + 35*r_rc^2 - 21*r_rc^3) + (x[,"n_02"] + x[,"n_34"])*log(15 - 12*r_rc + 7*r_rc^2 - 7*r_rc^3) + (x[,"n_11"] + x[,"n_25"])*log(9 - 8*r_rc + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_10"] + x[,"n_26"])*log(1 - 3*r_rc + 5*r_rc^2 - 3*r_rc^3) + (x[,"n_01"] + x[,"n_35"])*log(3 - 5*r_rc^2 + 2*r_rc^3) + (x[,"n_15"] + x[,"n_21"])*log(6 + 4*r_rc - 7*r_rc^2 + 6*r_rc^3) + (x[,"n_04"] + x[,"n_32"])*log(3 + 19*r_rc - 14*r_rc^2 + 7*r_rc^3) + (x[,"n_12"] + x[,"n_24"])*log(39 + 10*r_rc - 28*r_rc^2 + 21*r_rc^3)
logL_rm <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n13,n23) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 3*n05 - 3*n06 - 4*n10 - 4*n11 - 4*n12 - 4*n14 - 4*n15 - 4*n16 - 4*n20 - 4*n21 - 4*n22 - 4*n24 - 4*n25 - 4*n26 - 3*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 3*n35 - 3*n36)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36)*(log(3) + 2*log(5)) + (n06 + n30)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n36)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n35)*(log(pmax(1e-6,r)) + log(6 - r - 2*r^2)) + (n13 + n23)*log(12 + r - r^2) + (n10 + n26)*(log(pmax(1e-6,r)) + log(5 - 8*r + 6*r^2)) + (n03 + n33)*log(27 - 8*r + 8*r^2) + (n12 + n24)*log(105 + 7*r + 62*r^2 - 42*r^3) + (n15 + n21)*log(33 - 32*r + 26*r^2 - 12*r^3) + (n04 + n32)*log(21 - 16*r + 11*r^2 - 7*r^3) + (n16 + n20)*log(3 - 7*r + 10*r^2 - 6*r^3) + (n05 + n31)*log(3 + 2*r - 7*r^2 + 2*r^3) + (n02 + n34)*log(9 + 15*r - 10*r^2 + 7*r^3) + (n11 + n25)*log(15 + 16*r - 10*r^2 + 12*r^3) + (n14 + n22)*log(132 - 5*r - 64*r^2 + 42*r^3)
return(L)}
interlogL_rm <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n13,n23) {
optimize(logL_rm,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n13,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_13"],x[,"n_23"])
LOD_rm <- 3*(x[,"n_06"] + x[,"n_30"])*log10(2) + 3*(x[,"n_00"] + x[,"n_36"])*log10(2) - 2*(x[,"n_03"] + x[,"n_33"])*log10(5) - (x[,"n_16"] + x[,"n_20"])*(-2*log10(2) + log10(5)) - (x[,"n_10"] + x[,"n_26"])*(-2*log10(2) + log10(5)) - (x[,"n_05"] + x[,"n_31"])*(-log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_35"])*(-log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_23"])*(-2*log10(2) + 2*log10(7)) - (x[,"n_15"] + x[,"n_21"])*(log10(2) + log10(11)) - (x[,"n_11"] + x[,"n_25"])*(log10(2) + log10(11)) - (x[,"n_04"] + x[,"n_32"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_02"] + x[,"n_34"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_14"] + x[,"n_22"])*(-2*log10(2) + 2*log10(5) + log10(19)) - (x[,"n_12"] + x[,"n_24"])*(-2*log10(2) + 2*log10(5) + log10(19)) + (x[,"n_06"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_rm)) + log10(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_36"])*(log10(pmax(1e-6,1 - r_rm)) + 2*log10(pmax(1e-6,r_rm))) + (x[,"n_01"] + x[,"n_35"])*(log10(pmax(1e-6,r_rm)) + log10(6 - r_rm - 2*r_rm^2)) + (x[,"n_13"] + x[,"n_23"])*log10(12 + r_rm - r_rm^2) + (x[,"n_10"] + x[,"n_26"])*(log10(pmax(1e-6,r_rm)) + log10(5 - 8*r_rm + 6*r_rm^2)) + (x[,"n_03"] + x[,"n_33"])*log10(27 - 8*r_rm + 8*r_rm^2) + (x[,"n_12"] + x[,"n_24"])*log10(105 + 7*r_rm + 62*r_rm^2 - 42*r_rm^3) + (x[,"n_15"] + x[,"n_21"])*log10(33 - 32*r_rm + 26*r_rm^2 - 12*r_rm^3) + (x[,"n_04"] + x[,"n_32"])*log10(21 - 16*r_rm + 11*r_rm^2 - 7*r_rm^3) + (x[,"n_16"] + x[,"n_20"])*log10(3 - 7*r_rm + 10*r_rm^2 - 6*r_rm^3) + (x[,"n_05"] + x[,"n_31"])*log10(3 + 2*r_rm - 7*r_rm^2 + 2*r_rm^3) + (x[,"n_02"] + x[,"n_34"])*log10(9 + 15*r_rm - 10*r_rm^2 + 7*r_rm^3) + (x[,"n_11"] + x[,"n_25"])*log10(15 + 16*r_rm - 10*r_rm^2 + 12*r_rm^3) + (x[,"n_14"] + x[,"n_22"])*log10(132 - 5*r_rm - 64*r_rm^2 + 42*r_rm^3)
logL_rm <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_05"] - 3*x[,"n_06"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_14"] - 4*x[,"n_15"] - 4*x[,"n_16"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_35"] - 3*x[,"n_36"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"])*(log(3) + 2*log(5)) + (x[,"n_06"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_rm)) + log(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_36"])*(log(pmax(1e-6,1 - r_rm)) + 2*log(pmax(1e-6,r_rm))) + (x[,"n_01"] + x[,"n_35"])*(log(pmax(1e-6,r_rm)) + log(6 - r_rm - 2*r_rm^2)) + (x[,"n_13"] + x[,"n_23"])*log(12 + r_rm - r_rm^2) + (x[,"n_10"] + x[,"n_26"])*(log(pmax(1e-6,r_rm)) + log(5 - 8*r_rm + 6*r_rm^2)) + (x[,"n_03"] + x[,"n_33"])*log(27 - 8*r_rm + 8*r_rm^2) + (x[,"n_12"] + x[,"n_24"])*log(105 + 7*r_rm + 62*r_rm^2 - 42*r_rm^3) + (x[,"n_15"] + x[,"n_21"])*log(33 - 32*r_rm + 26*r_rm^2 - 12*r_rm^3) + (x[,"n_04"] + x[,"n_32"])*log(21 - 16*r_rm + 11*r_rm^2 - 7*r_rm^3) + (x[,"n_16"] + x[,"n_20"])*log(3 - 7*r_rm + 10*r_rm^2 - 6*r_rm^3) + (x[,"n_05"] + x[,"n_31"])*log(3 + 2*r_rm - 7*r_rm^2 + 2*r_rm^3) + (x[,"n_02"] + x[,"n_34"])*log(9 + 15*r_rm - 10*r_rm^2 + 7*r_rm^3) + (x[,"n_11"] + x[,"n_25"])*log(15 + 16*r_rm - 10*r_rm^2 + 12*r_rm^3) + (x[,"n_14"] + x[,"n_22"])*log(132 - 5*r_rm - 64*r_rm^2 + 42*r_rm^3)
logL_rr <- function(r,n10,n16,n20,n26,n00,n01,n02,n03,n04,n05,n06,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n36) {
L <- (n10 + n16 + n20 + n26)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36)*(4*log(2) + 2*log(5)) + 3*(n06 + n30)*log(pmax(1e-6,1 - r)) + (n05 + n31)*(log(9 - 2*r) + 2*log(pmax(1e-6,1 - r))) + 3*(n00 + n36)*log(pmax(1e-6,r)) + (n16 + n20)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n26)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n35)*(2*log(pmax(1e-6,r)) + log(7 + 2*r)) + (n03 + n33)*log(9 + 28*r - 28*r^2) + (n02 + n34)*(log(pmax(1e-6,r)) + log(15 + 13*r - 7*r^2)) + (n11 + n25)*(log(pmax(1e-6,r)) + log(16 - r - 6*r^2)) + (n13 + n23)*log(73 - 28*r + 28*r^2) + (n14 + n22)*log(57 - 44*r + 29*r^2 - 21*r^3) + (n15 + n21)*log(9 + 4*r - 19*r^2 + 6*r^3) + (n04 + n32)*log(21 - 20*r - 8*r^2 + 7*r^3) + (n12 + n24)*log(21 + 49*r - 34*r^2 + 21*r^3)
return(L)}
interlogL_rr <- function(n10,n16,n20,n26,n00,n01,n02,n03,n04,n05,n06,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n36) {
optimize(logL_rr,c(0,0.5), n10,n16,n20,n26,n00,n01,n02,n03,n04,n05,n06,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n36, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_10"],x[,"n_16"],x[,"n_20"],x[,"n_26"],x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"])
LOD_rr <- 3*(x[,"n_16"] + x[,"n_20"])*log10(2) + 3*(x[,"n_10"] + x[,"n_26"])*log10(2) + 3*(x[,"n_06"] + x[,"n_30"])*log10(2) - (x[,"n_05"] + x[,"n_31"])*log10(2) - 4*(x[,"n_03"] + x[,"n_33"])*log10(2) - (x[,"n_01"] + x[,"n_35"])*log10(2) + 3*(x[,"n_00"] + x[,"n_36"])*log10(2) - (x[,"n_15"] + x[,"n_21"])*log10(7) - (x[,"n_11"] + x[,"n_25"])*log10(7) - (x[,"n_13"] + x[,"n_23"])*(log10(2) + log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_32"])*(-3*log10(2) + log10(79)) - (x[,"n_02"] + x[,"n_34"])*(-3*log10(2) + log10(79)) - (x[,"n_14"] + x[,"n_22"])*(-3*log10(2) + log10(317)) - (x[,"n_12"] + x[,"n_24"])*(-3*log10(2) + log10(317)) + 3*(x[,"n_06"] + x[,"n_30"])*log10(pmax(1e-6,1 - r_rr)) + (x[,"n_05"] + x[,"n_31"])*(log10(9 - 2*r_rr) + 2*log10(pmax(1e-6,1 - r_rr))) + 3*(x[,"n_00"] + x[,"n_36"])*log10(pmax(1e-6,r_rr)) + (x[,"n_16"] + x[,"n_20"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(pmax(1e-6,r_rr))) + (x[,"n_10"] + x[,"n_26"])*(log10(pmax(1e-6,1 - r_rr)) + 2*log10(pmax(1e-6,r_rr))) + (x[,"n_01"] + x[,"n_35"])*(2*log10(pmax(1e-6,r_rr)) + log10(7 + 2*r_rr)) + (x[,"n_03"] + x[,"n_33"])*log10(9 + 28*r_rr - 28*r_rr^2) + (x[,"n_02"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(15 + 13*r_rr - 7*r_rr^2)) + (x[,"n_11"] + x[,"n_25"])*(log10(pmax(1e-6,r_rr)) + log10(16 - r_rr - 6*r_rr^2)) + (x[,"n_13"] + x[,"n_23"])*log10(73 - 28*r_rr + 28*r_rr^2) + (x[,"n_14"] + x[,"n_22"])*log10(57 - 44*r_rr + 29*r_rr^2 - 21*r_rr^3) + (x[,"n_15"] + x[,"n_21"])*log10(9 + 4*r_rr - 19*r_rr^2 + 6*r_rr^3) + (x[,"n_04"] + x[,"n_32"])*log10(21 - 20*r_rr - 8*r_rr^2 + 7*r_rr^3) + (x[,"n_12"] + x[,"n_24"])*log10(21 + 49*r_rr - 34*r_rr^2 + 21*r_rr^3)
logL_rr <- (x[,"n_10"] + x[,"n_16"] + x[,"n_20"] + x[,"n_26"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"])*(4*log(2) + 2*log(5)) + 3*(x[,"n_06"] + x[,"n_30"])*log(pmax(1e-6,1 - r_rr)) + (x[,"n_05"] + x[,"n_31"])*(log(9 - 2*r_rr) + 2*log(pmax(1e-6,1 - r_rr))) + 3*(x[,"n_00"] + x[,"n_36"])*log(pmax(1e-6,r_rr)) + (x[,"n_16"] + x[,"n_20"])*(2*log(pmax(1e-6,1 - r_rr)) + log(pmax(1e-6,r_rr))) + (x[,"n_10"] + x[,"n_26"])*(log(pmax(1e-6,1 - r_rr)) + 2*log(pmax(1e-6,r_rr))) + (x[,"n_01"] + x[,"n_35"])*(2*log(pmax(1e-6,r_rr)) + log(7 + 2*r_rr)) + (x[,"n_03"] + x[,"n_33"])*log(9 + 28*r_rr - 28*r_rr^2) + (x[,"n_02"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(15 + 13*r_rr - 7*r_rr^2)) + (x[,"n_11"] + x[,"n_25"])*(log(pmax(1e-6,r_rr)) + log(16 - r_rr - 6*r_rr^2)) + (x[,"n_13"] + x[,"n_23"])*log(73 - 28*r_rr + 28*r_rr^2) + (x[,"n_14"] + x[,"n_22"])*log(57 - 44*r_rr + 29*r_rr^2 - 21*r_rr^3) + (x[,"n_15"] + x[,"n_21"])*log(9 + 4*r_rr - 19*r_rr^2 + 6*r_rr^3) + (x[,"n_04"] + x[,"n_32"])*log(21 - 20*r_rr - 8*r_rr^2 + 7*r_rr^3) + (x[,"n_12"] + x[,"n_24"])*log(21 + 49*r_rr - 34*r_rr^2 + 21*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.2_4.1 <- function(x){
logL_cc <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
L <- (-2*n01 - 2*n02 - 2*n03 - 2*n04 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(2) + 2*(-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*log(5) + (n01 + n34)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n04 + n31)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n11 + n24)*log(11 - 4*r - 4*r^2) + (n14 + n21)*log(3 + 12*r - 4*r^2) + (n02 + n33)*log(6 - 3*r - 2*r^2) + (n03 + n32)*log(1 + 7*r - 2*r^2) + (n12 + n23)*log(39 - 16*r + 4*r^2) + (n13 + n22)*log(27 + 8*r + 4*r^2)
return(L)}
interlogL_cc <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
optimize(logL_cc,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cc <- -3*(x[,"n_14"] + x[,"n_21"])*log10(2) - 5*(x[,"n_13"] + x[,"n_22"])*log10(2) - 5*(x[,"n_12"] + x[,"n_23"])*log10(2) - 3*(x[,"n_11"] + x[,"n_24"])*log10(2) - 2*(x[,"n_03"] + x[,"n_32"])*log10(2) - 2*(x[,"n_02"] + x[,"n_33"])*log10(2) + (x[,"n_01"] + x[,"n_34"])*(log10(3 - 2*r_cc) + log10(pmax(1e-6,1 - r_cc))) + (x[,"n_04"] + x[,"n_31"])*(log10(pmax(1e-6,r_cc)) + log10(1 + 2*r_cc)) + (x[,"n_11"] + x[,"n_24"])*log10(11 - 4*r_cc - 4*r_cc^2) + (x[,"n_14"] + x[,"n_21"])*log10(3 + 12*r_cc - 4*r_cc^2) + (x[,"n_02"] + x[,"n_33"])*log10(6 - 3*r_cc - 2*r_cc^2) + (x[,"n_03"] + x[,"n_32"])*log10(1 + 7*r_cc - 2*r_cc^2) + (x[,"n_12"] + x[,"n_23"])*log10(39 - 16*r_cc + 4*r_cc^2) + (x[,"n_13"] + x[,"n_22"])*log10(27 + 8*r_cc + 4*r_cc^2)
logL_cc <- (-2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + (x[,"n_01"] + x[,"n_34"])*(log(3 - 2*r_cc) + log(pmax(1e-6,1 - r_cc))) + (x[,"n_04"] + x[,"n_31"])*(log(pmax(1e-6,r_cc)) + log(1 + 2*r_cc)) + (x[,"n_11"] + x[,"n_24"])*log(11 - 4*r_cc - 4*r_cc^2) + (x[,"n_14"] + x[,"n_21"])*log(3 + 12*r_cc - 4*r_cc^2) + (x[,"n_02"] + x[,"n_33"])*log(6 - 3*r_cc - 2*r_cc^2) + (x[,"n_03"] + x[,"n_32"])*log(1 + 7*r_cc - 2*r_cc^2) + (x[,"n_12"] + x[,"n_23"])*log(39 - 16*r_cc + 4*r_cc^2) + (x[,"n_13"] + x[,"n_22"])*log(27 + 8*r_cc + 4*r_cc^2)
logL_cr <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*(4*log(2) + 2*log(5)) + (n01 + n04 + n31 + n34)*(log(3 - 2*r) + log(1 + 2*r)) + (n12 + n13 + n22 + n23)*log(63 + 4*r - 4*r^2) + (n02 + n14 + n21 + n33)*log(15 + 4*r^2) + (n03 + n11 + n24 + n32)*log(19 - 8*r + 4*r^2)
return(L)}
interlogL_cr <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
optimize(logL_cr,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cr <- -6*(x[,"n_12"] + x[,"n_13"] + x[,"n_22"] + x[,"n_23"])*log10(2) - 4*(x[,"n_03"] + x[,"n_11"] + x[,"n_24"] + x[,"n_32"])*log10(2) - 4*(x[,"n_02"] + x[,"n_14"] + x[,"n_21"] + x[,"n_33"])*log10(2) - 2*(x[,"n_01"] + x[,"n_04"] + x[,"n_31"] + x[,"n_34"])*log10(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_31"] + x[,"n_34"])*(log10(3 - 2*r_cr) + log10(1 + 2*r_cr)) + (x[,"n_12"] + x[,"n_13"] + x[,"n_22"] + x[,"n_23"])*log10(63 + 4*r_cr - 4*r_cr^2) + (x[,"n_02"] + x[,"n_14"] + x[,"n_21"] + x[,"n_33"])*log10(15 + 4*r_cr^2) + (x[,"n_03"] + x[,"n_11"] + x[,"n_24"] + x[,"n_32"])*log10(19 - 8*r_cr + 4*r_cr^2)
logL_cr <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(4*log(2) + 2*log(5)) + (x[,"n_01"] + x[,"n_04"] + x[,"n_31"] + x[,"n_34"])*(log(3 - 2*r_cr) + log(1 + 2*r_cr)) + (x[,"n_12"] + x[,"n_13"] + x[,"n_22"] + x[,"n_23"])*log(63 + 4*r_cr - 4*r_cr^2) + (x[,"n_02"] + x[,"n_14"] + x[,"n_21"] + x[,"n_33"])*log(15 + 4*r_cr^2) + (x[,"n_03"] + x[,"n_11"] + x[,"n_24"] + x[,"n_32"])*log(19 - 8*r_cr + 4*r_cr^2)
logL_rc <- function(r,n02,n03,n11,n12,n13,n14,n21,n22,n23,n24,n32,n33,n01,n04,n31,n34) {
L <- (-n02 - n03 - n11 - 2*n12 - 2*n13 - n14 - n21 - 2*n22 - 2*n23 - n24 - n32 - n33)*log(2) + 2*(-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*log(5) + (n01 + n04 + n31 + n34)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n13 + n22 + n23)*log(15 + 4*r - 4*r^2) + (n02 + n14 + n21 + n33)*log(3 - 3*r + 2*r^2) + (n03 + n11 + n24 + n32)*log(2 - r + 2*r^2)
return(L)}
interlogL_rc <- function(n02,n03,n11,n12,n13,n14,n21,n22,n23,n24,n32,n33,n01,n04,n31,n34) {
optimize(logL_rc,c(0,0.5), n02,n03,n11,n12,n13,n14,n21,n22,n23,n24,n32,n33,n01,n04,n31,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_01"],x[,"n_04"],x[,"n_31"],x[,"n_34"])
LOD_rc <- -4*(x[,"n_12"] + x[,"n_13"] + x[,"n_22"] + x[,"n_23"])*log10(2) - (x[,"n_03"] + x[,"n_11"] + x[,"n_24"] + x[,"n_32"])*log10(2) - (x[,"n_02"] + x[,"n_14"] + x[,"n_21"] + x[,"n_33"])*log10(2) + 2*(x[,"n_01"] + x[,"n_04"] + x[,"n_31"] + x[,"n_34"])*log10(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_31"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc))) + (x[,"n_12"] + x[,"n_13"] + x[,"n_22"] + x[,"n_23"])*log10(15 + 4*r_rc - 4*r_rc^2) + (x[,"n_02"] + x[,"n_14"] + x[,"n_21"] + x[,"n_33"])*log10(3 - 3*r_rc + 2*r_rc^2) + (x[,"n_03"] + x[,"n_11"] + x[,"n_24"] + x[,"n_32"])*log10(2 - r_rc + 2*r_rc^2)
logL_rc <- (-x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + (x[,"n_01"] + x[,"n_04"] + x[,"n_31"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc))) + (x[,"n_12"] + x[,"n_13"] + x[,"n_22"] + x[,"n_23"])*log(15 + 4*r_rc - 4*r_rc^2) + (x[,"n_02"] + x[,"n_14"] + x[,"n_21"] + x[,"n_33"])*log(3 - 3*r_rc + 2*r_rc^2) + (x[,"n_03"] + x[,"n_11"] + x[,"n_24"] + x[,"n_32"])*log(2 - r_rc + 2*r_rc^2)
logL_rr <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
L <- (-2*n01 - 3*n02 - 3*n03 - 2*n04 - 2*n11 - 3*n12 - 3*n13 - 2*n14 - 2*n21 - 3*n22 - 3*n23 - 2*n24 - 2*n31 - 3*n32 - 3*n33 - 2*n34)*log(2) + 2*(-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34)*log(5) + (n04 + n31)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n01 + n34)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n03 + n32)*log(11 - 4*r - 4*r^2) + (n02 + n33)*log(3 + 12*r - 4*r^2) + (n14 + n21)*log(6 - 3*r - 2*r^2) + (n11 + n24)*log(1 + 7*r - 2*r^2) + (n13 + n22)*log(39 - 16*r + 4*r^2) + (n12 + n23)*log(27 + 8*r + 4*r^2)
return(L)}
interlogL_rr <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34) {
optimize(logL_rr,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_rr <- -2*(x[,"n_14"] + x[,"n_21"])*log10(2) - 5*(x[,"n_13"] + x[,"n_22"])*log10(2) - 5*(x[,"n_12"] + x[,"n_23"])*log10(2) - 2*(x[,"n_11"] + x[,"n_24"])*log10(2) - 3*(x[,"n_03"] + x[,"n_32"])*log10(2) - 3*(x[,"n_02"] + x[,"n_33"])*log10(2) + (x[,"n_04"] + x[,"n_31"])*(log10(3 - 2*r_rr) + log10(pmax(1e-6,1 - r_rr))) + (x[,"n_01"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(1 + 2*r_rr)) + (x[,"n_03"] + x[,"n_32"])*log10(11 - 4*r_rr - 4*r_rr^2) + (x[,"n_02"] + x[,"n_33"])*log10(3 + 12*r_rr - 4*r_rr^2) + (x[,"n_14"] + x[,"n_21"])*log10(6 - 3*r_rr - 2*r_rr^2) + (x[,"n_11"] + x[,"n_24"])*log10(1 + 7*r_rr - 2*r_rr^2) + (x[,"n_13"] + x[,"n_22"])*log10(39 - 16*r_rr + 4*r_rr^2) + (x[,"n_12"] + x[,"n_23"])*log10(27 + 8*r_rr + 4*r_rr^2)
logL_rr <- (-2*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 2*x[,"n_34"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + (x[,"n_04"] + x[,"n_31"])*(log(3 - 2*r_rr) + log(pmax(1e-6,1 - r_rr))) + (x[,"n_01"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(1 + 2*r_rr)) + (x[,"n_03"] + x[,"n_32"])*log(11 - 4*r_rr - 4*r_rr^2) + (x[,"n_02"] + x[,"n_33"])*log(3 + 12*r_rr - 4*r_rr^2) + (x[,"n_14"] + x[,"n_21"])*log(6 - 3*r_rr - 2*r_rr^2) + (x[,"n_11"] + x[,"n_24"])*log(1 + 7*r_rr - 2*r_rr^2) + (x[,"n_13"] + x[,"n_22"])*log(39 - 16*r_rr + 4*r_rr^2) + (x[,"n_12"] + x[,"n_23"])*log(27 + 8*r_rr + 4*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_1.3 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n10,n11,n13,n14,n20,n22,n24,n30,n31,n33,n34,n40,n41,n42,n43,n44,n12,n21,n23,n32) {
L <- (-3*n00 - n01 - 2*n02 - n03 - 3*n04 - n10 - 2*n11 - 2*n13 - n14 - 2*n20 - 2*n22 - 2*n24 - n30 - 2*n31 - 2*n33 - n34 - 3*n40 - n41 - 2*n42 - n43 - 3*n44)*log(2) + (n02 + n12 + n20 + n21 + n22 + n23 + n24 + n32 + n42)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + 4*(n00 + n44)*log(pmax(1e-6,1 - r)) + 4*(n04 + n40)*log(pmax(1e-6,r)) + (n01 + n10 + n34 + n43)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n20 + n24 + n42)*(2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n14 + n30 + n41)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n13 + n31)*(2*log(pmax(1e-6,r)) + log(9 - 12*r + 8*r^2)) + (n11 + n33)*(2*log(pmax(1e-6,1 - r)) + log(5 - 4*r + 8*r^2)) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,r)) + log(1 - 2*r + 2*r^2 - r^3)) + n22*log(3 - 8*r + 14*r^2 - 12*r^3 + 6*r^4)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n10,n11,n13,n14,n20,n22,n24,n30,n31,n33,n34,n40,n41,n42,n43,n44,n12,n21,n23,n32) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n13,n14,n20,n22,n24,n30,n31,n33,n34,n40,n41,n42,n43,n44,n12,n21,n23,n32, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_22"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_12"],x[,"n_21"],x[,"n_23"],x[,"n_32"])
LOD_cc <- 4*(x[,"n_04"] + x[,"n_40"])*log10(2) + 4*(x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*log10(2) + 4*(x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(2) + 4*(x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*log10(2) + 4*(x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-4*log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_31"])*(-2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-2*log10(2) + log10(5)) - x[,"n_22"]*(-3*log10(2) + log10(11)) + 4*(x[,"n_00"] + x[,"n_44"])*log10(pmax(1e-6,1 - r_cc)) + 4*(x[,"n_04"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(3*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(2*log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log10(pmax(1e-6,1 - r_cc)) + 3*log10(pmax(1e-6,r_cc))) + (x[,"n_13"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc)) + log10(9 - 12*r_cc + 8*r_cc^2)) + (x[,"n_11"] + x[,"n_33"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(5 - 4*r_cc + 8*r_cc^2)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(1 - 2*r_cc + 2*r_cc^2 - r_cc^3)) + x[,"n_22"]*log10(3 - 8*r_cc + 14*r_cc^2 - 12*r_cc^3 + 6*r_cc^4)
logL_cc <- (-3*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - x[,"n_03"] - 3*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_13"] - x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_22"] - 2*x[,"n_24"] - x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_33"] - x[,"n_34"] - 3*x[,"n_40"] - x[,"n_41"] - 2*x[,"n_42"] - x[,"n_43"] - 3*x[,"n_44"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_32"] + x[,"n_42"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + 4*(x[,"n_00"] + x[,"n_44"])*log(pmax(1e-6,1 - r_cc)) + 4*(x[,"n_04"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(3*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(2*log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log(pmax(1e-6,1 - r_cc)) + 3*log(pmax(1e-6,r_cc))) + (x[,"n_13"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc)) + log(9 - 12*r_cc + 8*r_cc^2)) + (x[,"n_11"] + x[,"n_33"])*(2*log(pmax(1e-6,1 - r_cc)) + log(5 - 4*r_cc + 8*r_cc^2)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(1 - 2*r_cc + 2*r_cc^2 - r_cc^3)) + x[,"n_22"]*log(3 - 8*r_cc + 14*r_cc^2 - 12*r_cc^3 + 6*r_cc^4)
logL_cm2c <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 3*n10 - n11 - 3*n12 - n13 - 3*n14 - 3*n20 - 3*n21 - 2*n22 - 3*n23 - 3*n24 - 3*n30 - n31 - 3*n32 - n33 - 3*n34 - 3*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n13 - n14 - n20 - n22 - n24 - n30 - n31 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n00 + n44)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n03 + n14 + n30 + n41)*(2*log(pmax(1e-6,r)) + log(5 - 6*r + 4*r^2)) + (n01 + n10 + n34 + n43)*(2*log(pmax(1e-6,1 - r)) + log(3 - 2*r + 4*r^2)) + (n02 + n20 + n24 + n42)*(log(pmax(1e-6,r)) + log(7 - 13*r + 12*r^2 - 6*r^3)) + (n13 + n31)*(log(pmax(1e-6,r)) + log(5 - 5*r + 7*r^2 - 4*r^3)) + n22*log(12 + 5*r - 23*r^2 + 36*r^3 - 18*r^4) + (n11 + n33)*log(3 - 8*r^2 + 9*r^3 - 4*r^4) + (n12 + n21 + n23 + n32)*log(5 - 4*r + 12*r^2 - 16*r^3 + 8*r^4)
return(L)}
interlogL_cm2c <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_cm2c,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cm2c <- 4*(x[,"n_04"] + x[,"n_40"])*log10(2) + 4*(x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-2*log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-2*log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + 2*log10(3)) - (x[,"n_13"] + x[,"n_31"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-3*log10(2) + log10(11)) - x[,"n_22"]*(-3*log10(2) + log10(97)) + (x[,"n_00"] + x[,"n_44"])*(3*log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c))) + (x[,"n_04"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_cm2c)) + 3*log10(pmax(1e-6,r_cm2c))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(2*log10(pmax(1e-6,r_cm2c)) + log10(5 - 6*r_cm2c + 4*r_cm2c^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(2*log10(pmax(1e-6,1 - r_cm2c)) + log10(3 - 2*r_cm2c + 4*r_cm2c^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(pmax(1e-6,r_cm2c)) + log10(7 - 13*r_cm2c + 12*r_cm2c^2 - 6*r_cm2c^3)) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm2c)) + log10(5 - 5*r_cm2c + 7*r_cm2c^2 - 4*r_cm2c^3)) + x[,"n_22"]*log10(12 + 5*r_cm2c - 23*r_cm2c^2 + 36*r_cm2c^3 - 18*r_cm2c^4) + (x[,"n_11"] + x[,"n_33"])*log10(3 - 8*r_cm2c^2 + 9*r_cm2c^3 - 4*r_cm2c^4) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(5 - 4*r_cm2c + 12*r_cm2c^2 - 16*r_cm2c^3 + 8*r_cm2c^4)
logL_cm2c <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_10"] - x[,"n_11"] - 3*x[,"n_12"] - x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 3*x[,"n_30"] - x[,"n_31"] - 3*x[,"n_32"] - x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_22"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_00"] + x[,"n_44"])*(3*log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c))) + (x[,"n_04"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_cm2c)) + 3*log(pmax(1e-6,r_cm2c))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(2*log(pmax(1e-6,r_cm2c)) + log(5 - 6*r_cm2c + 4*r_cm2c^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(2*log(pmax(1e-6,1 - r_cm2c)) + log(3 - 2*r_cm2c + 4*r_cm2c^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(pmax(1e-6,r_cm2c)) + log(7 - 13*r_cm2c + 12*r_cm2c^2 - 6*r_cm2c^3)) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_cm2c)) + log(5 - 5*r_cm2c + 7*r_cm2c^2 - 4*r_cm2c^3)) + x[,"n_22"]*log(12 + 5*r_cm2c - 23*r_cm2c^2 + 36*r_cm2c^3 - 18*r_cm2c^4) + (x[,"n_11"] + x[,"n_33"])*log(3 - 8*r_cm2c^2 + 9*r_cm2c^3 - 4*r_cm2c^4) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(5 - 4*r_cm2c + 12*r_cm2c^2 - 16*r_cm2c^3 + 8*r_cm2c^4)
logL_cm1c <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-3*n00 - n01 - 3*n02 - n03 - 3*n04 - n10 - 2*n11 - n12 - 2*n13 - n14 - 3*n20 - n21 - n22 - n23 - 3*n24 - n30 - 2*n31 - n32 - 2*n33 - n34 - 3*n40 - n41 - 3*n42 - n43 - 3*n44)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n13 - n14 - n20 - n22 - n24 - n30 - n31 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n00 + n04 + n40 + n44)*(2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n03 + n10 + n14 + n30 + n34 + n41 + n43)*(log(pmax(1e-6,r)) + log(1 - 2*r + 2*r^2 - r^3)) + (n12 + n21 + n23 + n32)*log(1 + r - 3*r^2 + 4*r^3 - 2*r^4) + (n02 + n20 + n24 + n42)*log(3 - 8*r + 14*r^2 - 12*r^3 + 6*r^4) + (n11 + n33)*log(6 - 8*r + 13*r^2 - 16*r^3 + 8*r^4) + (n13 + n31)*log(3 - 2*r + 13*r^2 - 16*r^3 + 8*r^4) + n22*log(6 - 2*r + 11*r^2 - 18*r^3 + 9*r^4)
return(L)}
interlogL_cm1c <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_cm1c,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm1c <- mapply(interlogL_cm1c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cm1c <- 4*(x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*log10(2) - (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(-4*log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + 2*log10(3)) - (x[,"n_13"] + x[,"n_31"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-3*log10(2) + log10(11)) - x[,"n_22"]*(-4*log10(2) + log10(97)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_cm1c)) + 2*log10(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log10(pmax(1e-6,r_cm1c)) + log10(1 - 2*r_cm1c + 2*r_cm1c^2 - r_cm1c^3)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(1 + r_cm1c - 3*r_cm1c^2 + 4*r_cm1c^3 - 2*r_cm1c^4) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(3 - 8*r_cm1c + 14*r_cm1c^2 - 12*r_cm1c^3 + 6*r_cm1c^4) + (x[,"n_11"] + x[,"n_33"])*log10(6 - 8*r_cm1c + 13*r_cm1c^2 - 16*r_cm1c^3 + 8*r_cm1c^4) + (x[,"n_13"] + x[,"n_31"])*log10(3 - 2*r_cm1c + 13*r_cm1c^2 - 16*r_cm1c^3 + 8*r_cm1c^4) + x[,"n_22"]*log10(6 - 2*r_cm1c + 11*r_cm1c^2 - 18*r_cm1c^3 + 9*r_cm1c^4)
logL_cm1c <- (-3*x[,"n_00"] - x[,"n_01"] - 3*x[,"n_02"] - x[,"n_03"] - 3*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - 3*x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - 3*x[,"n_24"] - x[,"n_30"] - 2*x[,"n_31"] - x[,"n_32"] - 2*x[,"n_33"] - x[,"n_34"] - 3*x[,"n_40"] - x[,"n_41"] - 3*x[,"n_42"] - x[,"n_43"] - 3*x[,"n_44"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_22"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_cm1c)) + 2*log(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log(pmax(1e-6,r_cm1c)) + log(1 - 2*r_cm1c + 2*r_cm1c^2 - r_cm1c^3)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(1 + r_cm1c - 3*r_cm1c^2 + 4*r_cm1c^3 - 2*r_cm1c^4) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log(3 - 8*r_cm1c + 14*r_cm1c^2 - 12*r_cm1c^3 + 6*r_cm1c^4) + (x[,"n_11"] + x[,"n_33"])*log(6 - 8*r_cm1c + 13*r_cm1c^2 - 16*r_cm1c^3 + 8*r_cm1c^4) + (x[,"n_13"] + x[,"n_31"])*log(3 - 2*r_cm1c + 13*r_cm1c^2 - 16*r_cm1c^3 + 8*r_cm1c^4) + x[,"n_22"]*log(6 - 2*r_cm1c + 11*r_cm1c^2 - 18*r_cm1c^3 + 9*r_cm1c^4)
logL_cr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 3*n10 - n11 - 3*n12 - n13 - 3*n14 - 3*n20 - 3*n21 - 2*n22 - 3*n23 - 3*n24 - 3*n30 - n31 - 3*n32 - n33 - 3*n34 - 3*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44)*log(2) + (n02 + n12 + n20 + n21 + n22 + n23 + n24 + n32 + n42)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n04 + n40)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n44)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n01 + n10 + n34 + n43)*(2*log(pmax(1e-6,r)) + log(3 - 6*r + 4*r^2)) + (n03 + n14 + n30 + n41)*(2*log(pmax(1e-6,1 - r)) + log(1 - 2*r + 4*r^2)) + n22*(log(pmax(1e-6,r)) + log(7 - 13*r + 12*r^2 - 6*r^3)) + (n11 + n33)*(log(pmax(1e-6,r)) + log(3 - 6*r + 7*r^2 - 4*r^3)) + (n13 + n31)*(log(pmax(1e-6,r)) + log(4 - 9*r + 9*r^2 - 4*r^3)) + (n02 + n20 + n24 + n42)*(log(pmax(1e-6,r)) + log(1 - 3*r + 4*r^2 - 2*r^3)) + (n12 + n21 + n23 + n32)*log(3 - 8*r + 16*r^2 - 16*r^3 + 8*r^4)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cr <- 4*(x[,"n_04"] + x[,"n_40"])*log10(2) + 2*(x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*log10(2) + 3*(x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(2) + 2*(x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*log10(2) + 4*(x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_31"])*(-3*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-3*log10(2) + log10(5)) - x[,"n_22"]*(-3*log10(2) + log10(11)) + (x[,"n_04"] + x[,"n_40"])*(3*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cr)) + 3*log10(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(2*log10(pmax(1e-6,r_cr)) + log10(3 - 6*r_cr + 4*r_cr^2)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(1 - 2*r_cr + 4*r_cr^2)) + x[,"n_22"]*(log10(pmax(1e-6,r_cr)) + log10(7 - 13*r_cr + 12*r_cr^2 - 6*r_cr^3)) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,r_cr)) + log10(3 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3)) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_cr)) + log10(4 - 9*r_cr + 9*r_cr^2 - 4*r_cr^3)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(pmax(1e-6,r_cr)) + log10(1 - 3*r_cr + 4*r_cr^2 - 2*r_cr^3)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(3 - 8*r_cr + 16*r_cr^2 - 16*r_cr^3 + 8*r_cr^4)
logL_cr <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_10"] - x[,"n_11"] - 3*x[,"n_12"] - x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 3*x[,"n_30"] - x[,"n_31"] - 3*x[,"n_32"] - x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_32"] + x[,"n_42"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_40"])*(3*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cr)) + 3*log(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(2*log(pmax(1e-6,r_cr)) + log(3 - 6*r_cr + 4*r_cr^2)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_cr)) + log(1 - 2*r_cr + 4*r_cr^2)) + x[,"n_22"]*(log(pmax(1e-6,r_cr)) + log(7 - 13*r_cr + 12*r_cr^2 - 6*r_cr^3)) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,r_cr)) + log(3 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3)) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_cr)) + log(4 - 9*r_cr + 9*r_cr^2 - 4*r_cr^3)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(pmax(1e-6,r_cr)) + log(1 - 3*r_cr + 4*r_cr^2 - 2*r_cr^3)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(3 - 8*r_cr + 16*r_cr^2 - 16*r_cr^3 + 8*r_cr^4)
logL_rc <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 3*n10 - n11 - 3*n12 - n13 - 3*n14 - 3*n20 - 3*n21 - 2*n22 - 3*n23 - 3*n24 - 3*n30 - n31 - 3*n32 - n33 - 3*n34 - 3*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44)*log(2) + (n02 + n12 + n20 + n21 + n22 + n23 + n24 + n32 + n42)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n04 + n40)*(log(3 - r) + 3*log(pmax(1e-6,r))) + (n00 + n44)*(3*log(pmax(1e-6,1 - r)) + log(2 + r)) + (n03 + n14 + n30 + n41)*(2*log(pmax(1e-6,r)) + log(9 - 10*r + 4*r^2)) + (n01 + n10 + n34 + n43)*(2*log(pmax(1e-6,1 - r)) + log(3 + 2*r + 4*r^2)) + (n13 + n31)*(log(pmax(1e-6,r)) + log(9 - 9*r + 9*r^2 - 4*r^3)) + (n02 + n20 + n24 + n42)*(log(pmax(1e-6,r)) + log(3 - 5*r + 4*r^2 - 2*r^3)) + n22*log(6 + 5*r - 11*r^2 + 12*r^3 - 6*r^4) + (n11 + n33)*log(5 - 2*r - 6*r^2 + 7*r^3 - 4*r^4) + (n12 + n21 + n23 + n32)*log(9 - 8*r + 16*r^2 - 16*r^3 + 8*r^4)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rc <- -((x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-3*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_13"] + x[,"n_31"])*(-3*log10(2) + 2*log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-3*log10(2) + 2*log10(5)) - x[,"n_22"]*(-3*log10(2) + log10(5) + log10(11)) + (x[,"n_04"] + x[,"n_40"])*(log10(3 - r_rc) + 3*log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_44"])*(3*log10(pmax(1e-6,1 - r_rc)) + log10(2 + r_rc)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(2*log10(pmax(1e-6,r_rc)) + log10(9 - 10*r_rc + 4*r_rc^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(3 + 2*r_rc + 4*r_rc^2)) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(9 - 9*r_rc + 9*r_rc^2 - 4*r_rc^3)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(pmax(1e-6,r_rc)) + log10(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3)) + x[,"n_22"]*log10(6 + 5*r_rc - 11*r_rc^2 + 12*r_rc^3 - 6*r_rc^4) + (x[,"n_11"] + x[,"n_33"])*log10(5 - 2*r_rc - 6*r_rc^2 + 7*r_rc^3 - 4*r_rc^4) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(9 - 8*r_rc + 16*r_rc^2 - 16*r_rc^3 + 8*r_rc^4)
logL_rc <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_10"] - x[,"n_11"] - 3*x[,"n_12"] - x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 3*x[,"n_30"] - x[,"n_31"] - 3*x[,"n_32"] - x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_32"] + x[,"n_42"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_40"])*(log(3 - r_rc) + 3*log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_44"])*(3*log(pmax(1e-6,1 - r_rc)) + log(2 + r_rc)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(2*log(pmax(1e-6,r_rc)) + log(9 - 10*r_rc + 4*r_rc^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(2*log(pmax(1e-6,1 - r_rc)) + log(3 + 2*r_rc + 4*r_rc^2)) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(9 - 9*r_rc + 9*r_rc^2 - 4*r_rc^3)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(pmax(1e-6,r_rc)) + log(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3)) + x[,"n_22"]*log(6 + 5*r_rc - 11*r_rc^2 + 12*r_rc^3 - 6*r_rc^4) + (x[,"n_11"] + x[,"n_33"])*log(5 - 2*r_rc - 6*r_rc^2 + 7*r_rc^3 - 4*r_rc^4) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(9 - 8*r_rc + 16*r_rc^2 - 16*r_rc^3 + 8*r_rc^4)
logL_rm2c <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-3*n00 - 2*n01 - 3*n02 - 2*n03 - 3*n04 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 3*n20 - 2*n21 - n22 - 2*n23 - 3*n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 3*n40 - 2*n41 - 3*n42 - 2*n43 - 3*n44)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n13 - n14 - n20 - n22 - n24 - n30 - n31 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n04 + n40)*(log(pmax(1e-6,1 - r)) + log(3 - r) + 2*log(pmax(1e-6,r))) + (n00 + n44)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n14 + n30 + n41)*(log(pmax(1e-6,r)) + log(6 - 7*r + 6*r^2 - 2*r^3)) + (n12 + n21 + n23 + n32)*log(11 + 2*r - 6*r^2 + 8*r^3 - 4*r^4) + (n01 + n10 + n34 + n43)*log(3 - 2*r - r^2 + 2*r^3 - 2*r^4) + (n02 + n20 + n24 + n42)*log(9 - 10*r + 16*r^2 - 12*r^3 + 6*r^4) + (n13 + n31)*log(18 - 4*r + 19*r^2 - 20*r^3 + 8*r^4) + (n11 + n33)*log(21 - 6*r + 7*r^2 - 12*r^3 + 8*r^4) + n22*log(30 - r + 10*r^2 - 18*r^3 + 9*r^4)
return(L)}
interlogL_rm2c <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rm2c,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm2c <- -((x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_13"] + x[,"n_31"])*(-2*log10(2) + log10(3) + 2*log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-2*log10(2) + log10(3) + 2*log10(5)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-3*log10(2) + log10(5) + log10(11)) - x[,"n_22"]*(-4*log10(2) + log10(5) + log10(97)) + (x[,"n_04"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_rm2c)) + log10(3 - r_rm2c) + 2*log10(pmax(1e-6,r_rm2c))) + (x[,"n_00"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_rm2c)) + log10(pmax(1e-6,r_rm2c)) + log10(2 + r_rm2c)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log10(pmax(1e-6,r_rm2c)) + log10(6 - 7*r_rm2c + 6*r_rm2c^2 - 2*r_rm2c^3)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(11 + 2*r_rm2c - 6*r_rm2c^2 + 8*r_rm2c^3 - 4*r_rm2c^4) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*log10(3 - 2*r_rm2c - r_rm2c^2 + 2*r_rm2c^3 - 2*r_rm2c^4) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(9 - 10*r_rm2c + 16*r_rm2c^2 - 12*r_rm2c^3 + 6*r_rm2c^4) + (x[,"n_13"] + x[,"n_31"])*log10(18 - 4*r_rm2c + 19*r_rm2c^2 - 20*r_rm2c^3 + 8*r_rm2c^4) + (x[,"n_11"] + x[,"n_33"])*log10(21 - 6*r_rm2c + 7*r_rm2c^2 - 12*r_rm2c^3 + 8*r_rm2c^4) + x[,"n_22"]*log10(30 - r_rm2c + 10*r_rm2c^2 - 18*r_rm2c^3 + 9*r_rm2c^4)
logL_rm2c <- (-3*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 3*x[,"n_40"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 2*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_22"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_rm2c)) + log(3 - r_rm2c) + 2*log(pmax(1e-6,r_rm2c))) + (x[,"n_00"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_rm2c)) + log(pmax(1e-6,r_rm2c)) + log(2 + r_rm2c)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log(pmax(1e-6,r_rm2c)) + log(6 - 7*r_rm2c + 6*r_rm2c^2 - 2*r_rm2c^3)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(11 + 2*r_rm2c - 6*r_rm2c^2 + 8*r_rm2c^3 - 4*r_rm2c^4) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*log(3 - 2*r_rm2c - r_rm2c^2 + 2*r_rm2c^3 - 2*r_rm2c^4) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log(9 - 10*r_rm2c + 16*r_rm2c^2 - 12*r_rm2c^3 + 6*r_rm2c^4) + (x[,"n_13"] + x[,"n_31"])*log(18 - 4*r_rm2c + 19*r_rm2c^2 - 20*r_rm2c^3 + 8*r_rm2c^4) + (x[,"n_11"] + x[,"n_33"])*log(21 - 6*r_rm2c + 7*r_rm2c^2 - 12*r_rm2c^3 + 8*r_rm2c^4) + x[,"n_22"]*log(30 - r_rm2c + 10*r_rm2c^2 - 18*r_rm2c^3 + 9*r_rm2c^4)
logL_rm1c <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 3*n10 - n11 - 3*n12 - n13 - 3*n14 - 3*n20 - 3*n21 - 2*n22 - 3*n23 - 3*n24 - 3*n30 - n31 - 3*n32 - n33 - 3*n34 - 3*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n13 - n14 - n20 - n22 - n24 - n30 - n31 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n04 + n40)*(2*log(pmax(1e-6,1 - r)) + log(3 - r) + log(pmax(1e-6,r))) + (n00 + n44)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r)) + log(2 + r)) + (n01 + n10 + n34 + n43)*(log(pmax(1e-6,r)) + log(8 - r - 2*r^2 + 4*r^3)) + n22*log(60 + 7*r - 25*r^2 + 36*r^3 - 18*r^4) + (n02 + n20 + n24 + n42)*log(6 + 5*r - 11*r^2 + 12*r^3 - 6*r^4) + (n11 + n33)*log(6 + 7*r - 2*r^2 + 5*r^3 - 4*r^4) + (n13 + n31)*log(12 - 2*r - 11*r^2 + 11*r^3 - 4*r^4) + (n03 + n14 + n30 + n41)*log(9 - 16*r + 17*r^2 - 14*r^3 + 4*r^4) + (n12 + n21 + n23 + n32)*log(23 - 4*r + 12*r^2 - 16*r^3 + 8*r^4)
return(L)}
interlogL_rm1c <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rm1c,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm1c <- mapply(interlogL_rm1c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm1c <- -((x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + 2*log10(3) + log10(5)) - (x[,"n_13"] + x[,"n_31"])*(-3*log10(2) + log10(3) + 2*log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-3*log10(2) + log10(3) + 2*log10(5)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-3*log10(2) + log10(5) + log10(11)) - x[,"n_22"]*(-3*log10(2) + log10(5) + log10(97)) + (x[,"n_04"] + x[,"n_40"])*(2*log10(pmax(1e-6,1 - r_rm1c)) + log10(3 - r_rm1c) + log10(pmax(1e-6,r_rm1c))) + (x[,"n_00"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_rm1c)) + 2*log10(pmax(1e-6,r_rm1c)) + log10(2 + r_rm1c)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log10(pmax(1e-6,r_rm1c)) + log10(8 - r_rm1c - 2*r_rm1c^2 + 4*r_rm1c^3)) + x[,"n_22"]*log10(60 + 7*r_rm1c - 25*r_rm1c^2 + 36*r_rm1c^3 - 18*r_rm1c^4) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(6 + 5*r_rm1c - 11*r_rm1c^2 + 12*r_rm1c^3 - 6*r_rm1c^4) + (x[,"n_11"] + x[,"n_33"])*log10(6 + 7*r_rm1c - 2*r_rm1c^2 + 5*r_rm1c^3 - 4*r_rm1c^4) + (x[,"n_13"] + x[,"n_31"])*log10(12 - 2*r_rm1c - 11*r_rm1c^2 + 11*r_rm1c^3 - 4*r_rm1c^4) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*log10(9 - 16*r_rm1c + 17*r_rm1c^2 - 14*r_rm1c^3 + 4*r_rm1c^4) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(23 - 4*r_rm1c + 12*r_rm1c^2 - 16*r_rm1c^3 + 8*r_rm1c^4)
logL_rm1c <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_10"] - x[,"n_11"] - 3*x[,"n_12"] - x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 3*x[,"n_30"] - x[,"n_31"] - 3*x[,"n_32"] - x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_22"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_40"])*(2*log(pmax(1e-6,1 - r_rm1c)) + log(3 - r_rm1c) + log(pmax(1e-6,r_rm1c))) + (x[,"n_00"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_rm1c)) + 2*log(pmax(1e-6,r_rm1c)) + log(2 + r_rm1c)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log(pmax(1e-6,r_rm1c)) + log(8 - r_rm1c - 2*r_rm1c^2 + 4*r_rm1c^3)) + x[,"n_22"]*log(60 + 7*r_rm1c - 25*r_rm1c^2 + 36*r_rm1c^3 - 18*r_rm1c^4) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log(6 + 5*r_rm1c - 11*r_rm1c^2 + 12*r_rm1c^3 - 6*r_rm1c^4) + (x[,"n_11"] + x[,"n_33"])*log(6 + 7*r_rm1c - 2*r_rm1c^2 + 5*r_rm1c^3 - 4*r_rm1c^4) + (x[,"n_13"] + x[,"n_31"])*log(12 - 2*r_rm1c - 11*r_rm1c^2 + 11*r_rm1c^3 - 4*r_rm1c^4) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*log(9 - 16*r_rm1c + 17*r_rm1c^2 - 14*r_rm1c^3 + 4*r_rm1c^4) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(23 - 4*r_rm1c + 12*r_rm1c^2 - 16*r_rm1c^3 + 8*r_rm1c^4)
logL_rr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-3*n00 - 2*n01 - 2*n02 - 2*n03 - 3*n04 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 3*n40 - 2*n41 - 2*n42 - 2*n43 - 3*n44)*log(2) + (n02 + n12 + n20 + n21 + n22 + n23 + n24 + n32 + n42)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n04 + n40)*(3*log(pmax(1e-6,1 - r)) + log(3 - r)) + (n00 + n44)*(3*log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n14 + n30 + n41)*(2*log(pmax(1e-6,1 - r)) + log(1 + 4*r - 2*r^2)) + (n11 + n33)*(log(pmax(1e-6,r)) + log(12 + 3*r - 8*r^2 + 8*r^3)) + (n12 + n21 + n23 + n32)*log(3 + 4*r - 8*r^2 + 8*r^3 - 4*r^4) + (n01 + n10 + n34 + n43)*log(3*r^2 - 2*r^4) + (n02 + n20 + n24 + n42)*log(r - 2*r^3 + r^4) + n22*log(9 - 10*r + 16*r^2 - 12*r^3 + 6*r^4) + (n13 + n31)*log(15 - 26*r + 27*r^2 - 24*r^3 + 8*r^4)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rr <- (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(3*log10(2) - log10(5)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(4*log10(2) - log10(5)) - (x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(5)) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-3*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_13"] + x[,"n_31"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-2*log10(2) + 2*log10(5)) - x[,"n_22"]*(-3*log10(2) + log10(5) + log10(11)) + (x[,"n_04"] + x[,"n_40"])*(3*log10(pmax(1e-6,1 - r_rr)) + log10(3 - r_rr)) + (x[,"n_00"] + x[,"n_44"])*(3*log10(pmax(1e-6,r_rr)) + log10(2 + r_rr)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(1 + 4*r_rr - 2*r_rr^2)) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,r_rr)) + log10(12 + 3*r_rr - 8*r_rr^2 + 8*r_rr^3)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(3 + 4*r_rr - 8*r_rr^2 + 8*r_rr^3 - 4*r_rr^4) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*log10(3*r_rr^2 - 2*r_rr^4) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(r_rr - 2*r_rr^3 + r_rr^4) + x[,"n_22"]*log10(9 - 10*r_rr + 16*r_rr^2 - 12*r_rr^3 + 6*r_rr^4) + (x[,"n_13"] + x[,"n_31"])*log10(15 - 26*r_rr + 27*r_rr^2 - 24*r_rr^3 + 8*r_rr^4)
logL_rr <- (-3*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 3*x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_32"] + x[,"n_42"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_40"])*(3*log(pmax(1e-6,1 - r_rr)) + log(3 - r_rr)) + (x[,"n_00"] + x[,"n_44"])*(3*log(pmax(1e-6,r_rr)) + log(2 + r_rr)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_rr)) + log(1 + 4*r_rr - 2*r_rr^2)) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,r_rr)) + log(12 + 3*r_rr - 8*r_rr^2 + 8*r_rr^3)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(3 + 4*r_rr - 8*r_rr^2 + 8*r_rr^3 - 4*r_rr^4) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*log(3*r_rr^2 - 2*r_rr^4) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log(r_rr - 2*r_rr^3 + r_rr^4) + x[,"n_22"]*log(9 - 10*r_rr + 16*r_rr^2 - 12*r_rr^3 + 6*r_rr^4) + (x[,"n_13"] + x[,"n_31"])*log(15 - 26*r_rr + 27*r_rr^2 - 24*r_rr^3 + 8*r_rr^4)

return(list(
r_mat = cbind(r_cc,r_cm2c,r_cm1c,r_cr,r_rc,r_rm2c,r_rm1c,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm2c,LOD_cm1c,LOD_cr,LOD_rc,LOD_rm2c,LOD_rm1c,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm2c,logL_cm1c,logL_cr,logL_rc,logL_rm2c,logL_rm1c,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed2c","coupling mixed1c","coupling repulsion","repulsion coupling","repulsion mixed2c","repulsion mixed1c","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_1.4 <- function(x){
logL_cc <- function(r,n02,n03,n42,n43,n01,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n44) {
L <- (n02 + n03 + n42 + n43)*log(3) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(3*log(2) + log(5)) + (n21 + n24)*log(7) + 3*(n01 + n44)*log(pmax(1e-6,1 - r)) + 3*(n04 + n41)*log(pmax(1e-6,r)) + (n21 + n24)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n43)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n31)*(log(5 - 2*r) + 2*log(pmax(1e-6,r))) + (n03 + n42)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n11 + n34)*(2*log(pmax(1e-6,1 - r)) + log(3 + 2*r)) + (n13 + n32)*(log(pmax(1e-6,r)) + log(12 - 11*r + 6*r^2)) + (n22 + n23)*log(9 - 7*r + 7*r^2) + (n12 + n33)*log(7 - 8*r + 7*r^2 - 6*r^3)
return(L)}
interlogL_cc <- function(n02,n03,n42,n43,n01,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n44) {
optimize(logL_cc,c(0,0.5), n02,n03,n42,n43,n01,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_02"],x[,"n_03"],x[,"n_42"],x[,"n_43"],x[,"n_01"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_44"])
LOD_cc <- 2*(x[,"n_21"] + x[,"n_24"])*log10(2) - 2*(x[,"n_13"] + x[,"n_32"])*log10(2) - 2*(x[,"n_12"] + x[,"n_33"])*log10(2) + 3*(x[,"n_04"] + x[,"n_41"])*log10(2) + 3*(x[,"n_03"] + x[,"n_42"])*log10(2) + 3*(x[,"n_02"] + x[,"n_43"])*log10(2) + 3*(x[,"n_01"] + x[,"n_44"])*log10(2) - (x[,"n_22"] + x[,"n_23"])*(-2*log10(2) + log10(29)) + 3*(x[,"n_01"] + x[,"n_44"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_04"] + x[,"n_41"])*log10(pmax(1e-6,r_cc)) + (x[,"n_21"] + x[,"n_24"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_02"] + x[,"n_43"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_31"])*(log10(5 - 2*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_42"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_34"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(3 + 2*r_cc)) + (x[,"n_13"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(12 - 11*r_cc + 6*r_cc^2)) + (x[,"n_22"] + x[,"n_23"])*log10(9 - 7*r_cc + 7*r_cc^2) + (x[,"n_12"] + x[,"n_33"])*log10(7 - 8*r_cc + 7*r_cc^2 - 6*r_cc^3)
logL_cc <- (x[,"n_02"] + x[,"n_03"] + x[,"n_42"] + x[,"n_43"])*log(3) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(3*log(2) + log(5)) + (x[,"n_21"] + x[,"n_24"])*log(7) + 3*(x[,"n_01"] + x[,"n_44"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_04"] + x[,"n_41"])*log(pmax(1e-6,r_cc)) + (x[,"n_21"] + x[,"n_24"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_02"] + x[,"n_43"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_31"])*(log(5 - 2*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_42"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_34"])*(2*log(pmax(1e-6,1 - r_cc)) + log(3 + 2*r_cc)) + (x[,"n_13"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(12 - 11*r_cc + 6*r_cc^2)) + (x[,"n_22"] + x[,"n_23"])*log(9 - 7*r_cc + 7*r_cc^2) + (x[,"n_12"] + x[,"n_33"])*log(7 - 8*r_cc + 7*r_cc^2 - 6*r_cc^3)
logL_cm2c <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-2*n01 - 3*n02 - 3*n03 - 2*n04 - 2*n11 - 3*n12 - 3*n13 - 2*n14 - 2*n21 - 3*n22 - 3*n23 - 2*n24 - 2*n31 - 3*n32 - 3*n33 - 2*n34 - 2*n41 - 3*n42 - 3*n43 - 2*n44)*log(2) + (-n01 - n02 - n03 - n04 - n11 - n14 - n21 - n22 - n23 - n24 - n31 - n34 - n41 - n42 - n43 - n44)*log(3) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*log(5) + (n01 + n44)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n41)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n22 + n23)*log(21 + 2*r - 2*r^2) + (n21 + n24)*log(3 - r + r^2) + (n14 + n31)*(log(pmax(1e-6,r)) + log(4 - 3*r + 2*r^2)) + (n03 + n42)*(log(pmax(1e-6,r)) + log(5 - 8*r + 6*r^2)) + (n02 + n43)*log(3 - 7*r + 10*r^2 - 6*r^3) + (n13 + n32)*log(3 + 6*r^2 - 4*r^3) + (n11 + n34)*log(3 - 4*r + 3*r^2 - 2*r^3) + (n12 + n33)*log(5 - 6*r^2 + 4*r^3)
return(L)}
interlogL_cm2c <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_cm2c,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cm2c <- -2*(x[,"n_13"] + x[,"n_32"])*log10(2) - 2*(x[,"n_12"] + x[,"n_33"])*log10(2) + 3*(x[,"n_04"] + x[,"n_41"])*log10(2) + 3*(x[,"n_01"] + x[,"n_44"])*log10(2) - (x[,"n_14"] + x[,"n_31"])*(-log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_34"])*(-log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_42"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - (x[,"n_21"] + x[,"n_24"])*(-2*log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_23"])*(-log10(2) + log10(43)) + (x[,"n_01"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c))) + (x[,"n_04"] + x[,"n_41"])*(log10(pmax(1e-6,1 - r_cm2c)) + 2*log10(pmax(1e-6,r_cm2c))) + (x[,"n_22"] + x[,"n_23"])*log10(21 + 2*r_cm2c - 2*r_cm2c^2) + (x[,"n_21"] + x[,"n_24"])*log10(3 - r_cm2c + r_cm2c^2) + (x[,"n_14"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm2c)) + log10(4 - 3*r_cm2c + 2*r_cm2c^2)) + (x[,"n_03"] + x[,"n_42"])*(log10(pmax(1e-6,r_cm2c)) + log10(5 - 8*r_cm2c + 6*r_cm2c^2)) + (x[,"n_02"] + x[,"n_43"])*log10(3 - 7*r_cm2c + 10*r_cm2c^2 - 6*r_cm2c^3) + (x[,"n_13"] + x[,"n_32"])*log10(3 + 6*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_11"] + x[,"n_34"])*log10(3 - 4*r_cm2c + 3*r_cm2c^2 - 2*r_cm2c^3) + (x[,"n_12"] + x[,"n_33"])*log10(5 - 6*r_cm2c^2 + 4*r_cm2c^3)
logL_cm2c <- (-2*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_01"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c))) + (x[,"n_04"] + x[,"n_41"])*(log(pmax(1e-6,1 - r_cm2c)) + 2*log(pmax(1e-6,r_cm2c))) + (x[,"n_22"] + x[,"n_23"])*log(21 + 2*r_cm2c - 2*r_cm2c^2) + (x[,"n_21"] + x[,"n_24"])*log(3 - r_cm2c + r_cm2c^2) + (x[,"n_14"] + x[,"n_31"])*(log(pmax(1e-6,r_cm2c)) + log(4 - 3*r_cm2c + 2*r_cm2c^2)) + (x[,"n_03"] + x[,"n_42"])*(log(pmax(1e-6,r_cm2c)) + log(5 - 8*r_cm2c + 6*r_cm2c^2)) + (x[,"n_02"] + x[,"n_43"])*log(3 - 7*r_cm2c + 10*r_cm2c^2 - 6*r_cm2c^3) + (x[,"n_13"] + x[,"n_32"])*log(3 + 6*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_11"] + x[,"n_34"])*log(3 - 4*r_cm2c + 3*r_cm2c^2 - 2*r_cm2c^3) + (x[,"n_12"] + x[,"n_33"])*log(5 - 6*r_cm2c^2 + 4*r_cm2c^3)
logL_cm1c <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(3*log(2) + log(5)) + (n04 + n41)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n44)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n22 + n23)*log(6 + 5*r - 5*r^2) + (n11 + n34)*(log(pmax(1e-6,r)) + log(4 - 5*r + 2*r^2)) + (n02 + n43)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n21 + n24)*log(3 - 5*r + 5*r^2) + (n12 + n33)*log(6 - 8*r + 11*r^2 - 6*r^3) + (n03 + n42)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n14 + n31)*log(1 + r^2 - 2*r^3) + (n13 + n32)*log(3 + 4*r - 7*r^2 + 6*r^3)
return(L)}
interlogL_cm1c <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_cm1c,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm1c <- mapply(interlogL_cm1c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cm1c <- -2*(x[,"n_13"] + x[,"n_32"])*log10(2) - 2*(x[,"n_12"] + x[,"n_33"])*log10(2) + 3*(x[,"n_04"] + x[,"n_41"])*log10(2) + 3*(x[,"n_01"] + x[,"n_44"])*log10(2) + (x[,"n_03"] + x[,"n_42"])*(3*log10(2) - log10(3)) - (x[,"n_02"] + x[,"n_43"])*(-3*log10(2) + log10(3)) - (x[,"n_21"] + x[,"n_24"])*(-2*log10(2) + log10(7)) - (x[,"n_22"] + x[,"n_23"])*(-2*log10(2) + log10(29)) + (x[,"n_04"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_cm1c)) + log10(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cm1c)) + 2*log10(pmax(1e-6,r_cm1c))) + (x[,"n_22"] + x[,"n_23"])*log10(6 + 5*r_cm1c - 5*r_cm1c^2) + (x[,"n_11"] + x[,"n_34"])*(log10(pmax(1e-6,r_cm1c)) + log10(4 - 5*r_cm1c + 2*r_cm1c^2)) + (x[,"n_02"] + x[,"n_43"])*(log10(pmax(1e-6,r_cm1c)) + log10(2 - 4*r_cm1c + 3*r_cm1c^2)) + (x[,"n_21"] + x[,"n_24"])*log10(3 - 5*r_cm1c + 5*r_cm1c^2) + (x[,"n_12"] + x[,"n_33"])*log10(6 - 8*r_cm1c + 11*r_cm1c^2 - 6*r_cm1c^3) + (x[,"n_03"] + x[,"n_42"])*log10(1 - 3*r_cm1c + 5*r_cm1c^2 - 3*r_cm1c^3) + (x[,"n_14"] + x[,"n_31"])*log10(1 + r_cm1c^2 - 2*r_cm1c^3) + (x[,"n_13"] + x[,"n_32"])*log10(3 + 4*r_cm1c - 7*r_cm1c^2 + 6*r_cm1c^3)
logL_cm1c <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(3*log(2) + log(5)) + (x[,"n_04"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_cm1c)) + log(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cm1c)) + 2*log(pmax(1e-6,r_cm1c))) + (x[,"n_22"] + x[,"n_23"])*log(6 + 5*r_cm1c - 5*r_cm1c^2) + (x[,"n_11"] + x[,"n_34"])*(log(pmax(1e-6,r_cm1c)) + log(4 - 5*r_cm1c + 2*r_cm1c^2)) + (x[,"n_02"] + x[,"n_43"])*(log(pmax(1e-6,r_cm1c)) + log(2 - 4*r_cm1c + 3*r_cm1c^2)) + (x[,"n_21"] + x[,"n_24"])*log(3 - 5*r_cm1c + 5*r_cm1c^2) + (x[,"n_12"] + x[,"n_33"])*log(6 - 8*r_cm1c + 11*r_cm1c^2 - 6*r_cm1c^3) + (x[,"n_03"] + x[,"n_42"])*log(1 - 3*r_cm1c + 5*r_cm1c^2 - 3*r_cm1c^3) + (x[,"n_14"] + x[,"n_31"])*log(1 + r_cm1c^2 - 2*r_cm1c^3) + (x[,"n_13"] + x[,"n_32"])*log(3 + 4*r_cm1c - 7*r_cm1c^2 + 6*r_cm1c^3)
logL_rc <- function(r,n02,n03,n42,n43,n01,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n44) {
L <- (n02 + n03 + n42 + n43)*log(3) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(3*log(2) + 2*log(5)) + (n04 + n41)*(log(3 - r) + 2*log(pmax(1e-6,r))) + (n01 + n44)*(2*log(pmax(1e-6,1 - r)) + log(2 + r)) + (n22 + n23)*log(36 + r - r^2) + (n03 + n42)*(log(pmax(1e-6,r)) + log(2 - 2*r + r^2)) + (n21 + n24)*log(9 - r + r^2) + (n14 + n31)*(log(pmax(1e-6,r)) + log(12 - 5*r + 2*r^2)) + (n13 + n32)*log(18 + 11*r^2 - 6*r^3) + (n11 + n34)*log(9 - 8*r + r^2 - 2*r^3) + (n02 + n43)*log(1 - r + r^2 - r^3) + (n12 + n33)*log(23 - 4*r - 7*r^2 + 6*r^3)
return(L)}
interlogL_rc <- function(n02,n03,n42,n43,n01,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n44) {
optimize(logL_rc,c(0,0.5), n02,n03,n42,n43,n01,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_02"],x[,"n_03"],x[,"n_42"],x[,"n_43"],x[,"n_01"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_44"])
LOD_rc <- (x[,"n_02"] + x[,"n_43"])*(3*log10(2) - log10(5)) - (x[,"n_14"] + x[,"n_31"])*log10(5) - (x[,"n_11"] + x[,"n_34"])*log10(5) - (x[,"n_04"] + x[,"n_41"])*(-3*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_42"])*(-3*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_44"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_32"])*(2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_33"])*(2*log10(2) + log10(5)) - (x[,"n_21"] + x[,"n_24"])*(-2*log10(2) + log10(5) + log10(7)) - (x[,"n_22"] + x[,"n_23"])*(-2*log10(2) + log10(5) + log10(29)) + (x[,"n_04"] + x[,"n_41"])*(log10(3 - r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_01"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(2 + r_rc)) + (x[,"n_22"] + x[,"n_23"])*log10(36 + r_rc - r_rc^2) + (x[,"n_03"] + x[,"n_42"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 2*r_rc + r_rc^2)) + (x[,"n_21"] + x[,"n_24"])*log10(9 - r_rc + r_rc^2) + (x[,"n_14"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(12 - 5*r_rc + 2*r_rc^2)) + (x[,"n_13"] + x[,"n_32"])*log10(18 + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_11"] + x[,"n_34"])*log10(9 - 8*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_02"] + x[,"n_43"])*log10(1 - r_rc + r_rc^2 - r_rc^3) + (x[,"n_12"] + x[,"n_33"])*log10(23 - 4*r_rc - 7*r_rc^2 + 6*r_rc^3)
logL_rc <- (x[,"n_02"] + x[,"n_03"] + x[,"n_42"] + x[,"n_43"])*log(3) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(3*log(2) + 2*log(5)) + (x[,"n_04"] + x[,"n_41"])*(log(3 - r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_01"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_rc)) + log(2 + r_rc)) + (x[,"n_22"] + x[,"n_23"])*log(36 + r_rc - r_rc^2) + (x[,"n_03"] + x[,"n_42"])*(log(pmax(1e-6,r_rc)) + log(2 - 2*r_rc + r_rc^2)) + (x[,"n_21"] + x[,"n_24"])*log(9 - r_rc + r_rc^2) + (x[,"n_14"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(12 - 5*r_rc + 2*r_rc^2)) + (x[,"n_13"] + x[,"n_32"])*log(18 + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_11"] + x[,"n_34"])*log(9 - 8*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_02"] + x[,"n_43"])*log(1 - r_rc + r_rc^2 - r_rc^3) + (x[,"n_12"] + x[,"n_33"])*log(23 - 4*r_rc - 7*r_rc^2 + 6*r_rc^3)
logL_rm2c <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-2*n01 - 3*n02 - 3*n03 - 2*n04 - 2*n11 - 3*n12 - 3*n13 - 2*n14 - 2*n21 - 3*n22 - 3*n23 - 2*n24 - 2*n31 - 3*n32 - 3*n33 - 2*n34 - 2*n41 - 3*n42 - 3*n43 - 2*n44)*log(2) + (-n01 - n02 - n03 - n04 - n11 - n14 - n21 - n22 - n23 - n24 - n31 - n34 - n41 - n42 - n43 - n44)*log(3) + (-2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - n21 - n22 - n23 - n24 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n41 - 2*n42 - 2*n43 - 2*n44)*log(5) + (n04 + n41)*(log(pmax(1e-6,1 - r)) + log(3 - r) + log(pmax(1e-6,r))) + (n01 + n44)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 + r)) + (n22 + n23)*log(21 + 2*r - 2*r^2) + (n21 + n24)*log(3 - r + r^2) + (n03 + n42)*log(9 - 11*r + 14*r^2 - 6*r^3) + (n12 + n33)*log(19 + 6*r^2 - 4*r^3) + (n14 + n31)*log(9 - 4*r + 3*r^2 - 2*r^3) + (n11 + n34)*log(6 + 4*r - 3*r^2 + 2*r^3) + (n13 + n32)*log(21 - 6*r^2 + 4*r^3) + (n02 + n43)*log(6 + r - 4*r^2 + 6*r^3)
return(L)}
interlogL_rm2c <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_rm2c,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm2c <- -((x[,"n_04"] + x[,"n_41"])*(-3*log10(2) + log10(5))) - (x[,"n_01"] + x[,"n_44"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_32"])*(2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_33"])*(2*log10(2) + log10(5)) - (x[,"n_14"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_11"] + x[,"n_34"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_03"] + x[,"n_42"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_02"] + x[,"n_43"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_21"] + x[,"n_24"])*(-2*log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_23"])*(-log10(2) + log10(43)) + (x[,"n_04"] + x[,"n_41"])*(log10(pmax(1e-6,1 - r_rm2c)) + log10(3 - r_rm2c) + log10(pmax(1e-6,r_rm2c))) + (x[,"n_01"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_rm2c)) + log10(pmax(1e-6,r_rm2c)) + log10(2 + r_rm2c)) + (x[,"n_22"] + x[,"n_23"])*log10(21 + 2*r_rm2c - 2*r_rm2c^2) + (x[,"n_21"] + x[,"n_24"])*log10(3 - r_rm2c + r_rm2c^2) + (x[,"n_03"] + x[,"n_42"])*log10(9 - 11*r_rm2c + 14*r_rm2c^2 - 6*r_rm2c^3) + (x[,"n_12"] + x[,"n_33"])*log10(19 + 6*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_14"] + x[,"n_31"])*log10(9 - 4*r_rm2c + 3*r_rm2c^2 - 2*r_rm2c^3) + (x[,"n_11"] + x[,"n_34"])*log10(6 + 4*r_rm2c - 3*r_rm2c^2 + 2*r_rm2c^3) + (x[,"n_13"] + x[,"n_32"])*log10(21 - 6*r_rm2c^2 + 4*r_rm2c^3) + (x[,"n_02"] + x[,"n_43"])*log10(6 + r_rm2c - 4*r_rm2c^2 + 6*r_rm2c^3)
logL_rm2c <- (-2*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (-2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_41"])*(log(pmax(1e-6,1 - r_rm2c)) + log(3 - r_rm2c) + log(pmax(1e-6,r_rm2c))) + (x[,"n_01"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_rm2c)) + log(pmax(1e-6,r_rm2c)) + log(2 + r_rm2c)) + (x[,"n_22"] + x[,"n_23"])*log(21 + 2*r_rm2c - 2*r_rm2c^2) + (x[,"n_21"] + x[,"n_24"])*log(3 - r_rm2c + r_rm2c^2) + (x[,"n_03"] + x[,"n_42"])*log(9 - 11*r_rm2c + 14*r_rm2c^2 - 6*r_rm2c^3) + (x[,"n_12"] + x[,"n_33"])*log(19 + 6*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_14"] + x[,"n_31"])*log(9 - 4*r_rm2c + 3*r_rm2c^2 - 2*r_rm2c^3) + (x[,"n_11"] + x[,"n_34"])*log(6 + 4*r_rm2c - 3*r_rm2c^2 + 2*r_rm2c^3) + (x[,"n_13"] + x[,"n_32"])*log(21 - 6*r_rm2c^2 + 4*r_rm2c^3) + (x[,"n_02"] + x[,"n_43"])*log(6 + r_rm2c - 4*r_rm2c^2 + 6*r_rm2c^3)
logL_rm1c <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(3*log(2) + 2*log(5)) + (n04 + n41)*(2*log(pmax(1e-6,1 - r)) + log(3 - r)) + (n01 + n44)*(2*log(pmax(1e-6,r)) + log(2 + r)) + (n21 + n24)*log(6 + 11*r - 11*r^2) + (n02 + n43)*(log(pmax(1e-6,r)) + log(4 + r - 3*r^2)) + (n11 + n34)*(log(pmax(1e-6,r)) + log(8 + 5*r - 2*r^2)) + (n22 + n23)*log(39 - 11*r + 11*r^2) + (n13 + n32)*log(27 - 16*r + 7*r^2 - 6*r^3) + (n14 + n31)*log(11 - 12*r - r^2 + 2*r^3) + (n03 + n42)*log(2 + 3*r - 8*r^2 + 3*r^3) + (n12 + n33)*log(12 + 20*r - 11*r^2 + 6*r^3)
return(L)}
interlogL_rm1c <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_rm1c,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm1c <- mapply(interlogL_rm1c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm1c <- -((x[,"n_14"] + x[,"n_31"])*log10(5)) - (x[,"n_11"] + x[,"n_34"])*log10(5) - (x[,"n_04"] + x[,"n_41"])*(-3*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_44"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_32"])*(2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_33"])*(2*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_42"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_43"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_21"] + x[,"n_24"])*(-2*log10(2) + log10(5) + log10(7)) - (x[,"n_22"] + x[,"n_23"])*(-2*log10(2) + log10(5) + log10(29)) + (x[,"n_04"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_rm1c)) + log10(3 - r_rm1c)) + (x[,"n_01"] + x[,"n_44"])*(2*log10(pmax(1e-6,r_rm1c)) + log10(2 + r_rm1c)) + (x[,"n_21"] + x[,"n_24"])*log10(6 + 11*r_rm1c - 11*r_rm1c^2) + (x[,"n_02"] + x[,"n_43"])*(log10(pmax(1e-6,r_rm1c)) + log10(4 + r_rm1c - 3*r_rm1c^2)) + (x[,"n_11"] + x[,"n_34"])*(log10(pmax(1e-6,r_rm1c)) + log10(8 + 5*r_rm1c - 2*r_rm1c^2)) + (x[,"n_22"] + x[,"n_23"])*log10(39 - 11*r_rm1c + 11*r_rm1c^2) + (x[,"n_13"] + x[,"n_32"])*log10(27 - 16*r_rm1c + 7*r_rm1c^2 - 6*r_rm1c^3) + (x[,"n_14"] + x[,"n_31"])*log10(11 - 12*r_rm1c - r_rm1c^2 + 2*r_rm1c^3) + (x[,"n_03"] + x[,"n_42"])*log10(2 + 3*r_rm1c - 8*r_rm1c^2 + 3*r_rm1c^3) + (x[,"n_12"] + x[,"n_33"])*log10(12 + 20*r_rm1c - 11*r_rm1c^2 + 6*r_rm1c^3)
logL_rm1c <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(3*log(2) + 2*log(5)) + (x[,"n_04"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_rm1c)) + log(3 - r_rm1c)) + (x[,"n_01"] + x[,"n_44"])*(2*log(pmax(1e-6,r_rm1c)) + log(2 + r_rm1c)) + (x[,"n_21"] + x[,"n_24"])*log(6 + 11*r_rm1c - 11*r_rm1c^2) + (x[,"n_02"] + x[,"n_43"])*(log(pmax(1e-6,r_rm1c)) + log(4 + r_rm1c - 3*r_rm1c^2)) + (x[,"n_11"] + x[,"n_34"])*(log(pmax(1e-6,r_rm1c)) + log(8 + 5*r_rm1c - 2*r_rm1c^2)) + (x[,"n_22"] + x[,"n_23"])*log(39 - 11*r_rm1c + 11*r_rm1c^2) + (x[,"n_13"] + x[,"n_32"])*log(27 - 16*r_rm1c + 7*r_rm1c^2 - 6*r_rm1c^3) + (x[,"n_14"] + x[,"n_31"])*log(11 - 12*r_rm1c - r_rm1c^2 + 2*r_rm1c^3) + (x[,"n_03"] + x[,"n_42"])*log(2 + 3*r_rm1c - 8*r_rm1c^2 + 3*r_rm1c^3) + (x[,"n_12"] + x[,"n_33"])*log(12 + 20*r_rm1c - 11*r_rm1c^2 + 6*r_rm1c^3)

return(list(
r_mat = cbind(r_cc,r_cm2c,r_cm1c,r_rc,r_rm2c,r_rm1c,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm2c,LOD_cm1c,LOD_rc,LOD_rm2c,LOD_rm1c,0),
logL_mat = cbind(logL_cc,logL_cm2c,logL_cm1c,logL_rc,logL_rm2c,logL_rm1c,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed2c","coupling mixed1c","repulsion coupling","repulsion mixed2c","repulsion mixed1c","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_1.5 <- function(x){
logL_cc <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
L <- (-3*n02 - 2*n03 - 3*n04 - 2*n12 - n13 - 2*n14 - 3*n22 - n23 - 3*n24 - 2*n32 - n33 - 2*n34 - 3*n42 - 2*n43 - 3*n44)*log(2) + (n22 + n23 + n24)*log(3) + (-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*log(5) + 2*(n02 + n44)*log(pmax(1e-6,1 - r)) + 2*(n04 + n42)*log(pmax(1e-6,r)) + (n03 + n43)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n32)*(log(pmax(1e-6,r)) + log(2 + r)) + (n22 + n24)*log(1 + 2*r - 2*r^2) + (n13 + n33)*log(1 + r - r^2) + (n12 + n34)*log(3 - 4*r + r^2) + n23*log(1 - r + r^2)
return(L)}
interlogL_cc <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
optimize(logL_cc,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cc <- 2*(x[,"n_04"] + x[,"n_42"])*log10(2) + 2*(x[,"n_03"] + x[,"n_43"])*log10(2) + 2*(x[,"n_02"] + x[,"n_44"])*log10(2) + x[,"n_23"]*(2*log10(2) - log10(3)) - (x[,"n_22"] + x[,"n_24"])*(-log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_32"])*(-2*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_33"])*(-2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_34"])*(-2*log10(2) + log10(5)) + 2*(x[,"n_02"] + x[,"n_44"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_42"])*log10(pmax(1e-6,r_cc)) + (x[,"n_03"] + x[,"n_43"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_22"] + x[,"n_24"])*log10(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_13"] + x[,"n_33"])*log10(1 + r_cc - r_cc^2) + (x[,"n_12"] + x[,"n_34"])*log10(3 - 4*r_cc + r_cc^2) + x[,"n_23"]*log10(1 - r_cc + r_cc^2)
logL_cc <- (-3*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_22"] - x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_32"] - x[,"n_33"] - 2*x[,"n_34"] - 3*x[,"n_42"] - 2*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (x[,"n_22"] + x[,"n_23"] + x[,"n_24"])*log(3) + (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + 2*(x[,"n_02"] + x[,"n_44"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_42"])*log(pmax(1e-6,r_cc)) + (x[,"n_03"] + x[,"n_43"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_22"] + x[,"n_24"])*log(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_13"] + x[,"n_33"])*log(1 + r_cc - r_cc^2) + (x[,"n_12"] + x[,"n_34"])*log(3 - 4*r_cc + r_cc^2) + x[,"n_23"]*log(1 - r_cc + r_cc^2)
logL_cm2c <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
L <- (-3*n02 - 3*n03 - 3*n04 - 3*n12 - 2*n13 - 3*n14 - 2*n22 - 2*n23 - 2*n24 - 3*n32 - 2*n33 - 3*n34 - 3*n42 - 3*n43 - 3*n44)*log(2) + (n22 + n23 + n24)*log(3) + (-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*log(5) + (n02 + n04 + n42 + n44)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n34)*log(3 - 2*r^2) + n23*log(1 + 2*r - 2*r^2) + (n14 + n32)*log(1 + 4*r - 2*r^2) + (n22 + n24)*log(1 - r + r^2) + (n03 + n43)*log(1 - 2*r + 2*r^2) + (n13 + n33)*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_cm2c <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
optimize(logL_cm2c,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cm2c <- (x[,"n_03"] + x[,"n_43"])*log10(2) + 2*(x[,"n_02"] + x[,"n_04"] + x[,"n_42"] + x[,"n_44"])*log10(2) + (x[,"n_22"] + x[,"n_24"])*(2*log10(2) - log10(3)) - x[,"n_23"]*(-log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_32"])*(-log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_33"])*(-log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_34"])*(-log10(2) + log10(5)) + (x[,"n_02"] + x[,"n_04"] + x[,"n_42"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c))) + (x[,"n_12"] + x[,"n_34"])*log10(3 - 2*r_cm2c^2) + x[,"n_23"]*log10(1 + 2*r_cm2c - 2*r_cm2c^2) + (x[,"n_14"] + x[,"n_32"])*log10(1 + 4*r_cm2c - 2*r_cm2c^2) + (x[,"n_22"] + x[,"n_24"])*log10(1 - r_cm2c + r_cm2c^2) + (x[,"n_03"] + x[,"n_43"])*log10(1 - 2*r_cm2c + 2*r_cm2c^2) + (x[,"n_13"] + x[,"n_33"])*log10(3 - 2*r_cm2c + 2*r_cm2c^2)
logL_cm2c <- (-3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 3*x[,"n_32"] - 2*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (x[,"n_22"] + x[,"n_23"] + x[,"n_24"])*log(3) + (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_02"] + x[,"n_04"] + x[,"n_42"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c))) + (x[,"n_12"] + x[,"n_34"])*log(3 - 2*r_cm2c^2) + x[,"n_23"]*log(1 + 2*r_cm2c - 2*r_cm2c^2) + (x[,"n_14"] + x[,"n_32"])*log(1 + 4*r_cm2c - 2*r_cm2c^2) + (x[,"n_22"] + x[,"n_24"])*log(1 - r_cm2c + r_cm2c^2) + (x[,"n_03"] + x[,"n_43"])*log(1 - 2*r_cm2c + 2*r_cm2c^2) + (x[,"n_13"] + x[,"n_33"])*log(3 - 2*r_cm2c + 2*r_cm2c^2)
logL_rc <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
L <- (-3*n02 - 3*n03 - 3*n04 - 3*n12 - 2*n13 - 3*n14 - 2*n22 - 2*n23 - 2*n24 - 3*n32 - 2*n33 - 3*n34 - 3*n42 - 3*n43 - 3*n44)*log(2) + (n22 + n23 + n24)*log(3) + 2*(-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*log(5) + (n04 + n42)*(log(3 - r) + log(pmax(1e-6,r))) + (n02 + n44)*(log(pmax(1e-6,1 - r)) + log(2 + r)) + (n12 + n34)*log(15 - 4*r - 2*r^2) + n23*log(7 + 2*r - 2*r^2) + (n14 + n32)*log(9 + 8*r - 2*r^2) + (n22 + n24)*log(4 - r + r^2) + (n03 + n43)*log(3 - 2*r + 2*r^2) + (n13 + n33)*log(13 - 2*r + 2*r^2)
return(L)}
interlogL_rc <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
optimize(logL_rc,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rc <- -((x[,"n_04"] + x[,"n_42"])*(-2*log10(2) + log10(5))) - (x[,"n_02"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_43"])*(-log10(2) + log10(5)) - (x[,"n_22"] + x[,"n_24"])*(-2*log10(2) + log10(3) + log10(5)) - x[,"n_23"]*(-log10(2) + log10(3) + log10(5)) - (x[,"n_14"] + x[,"n_32"])*(-log10(2) + 2*log10(5)) - (x[,"n_13"] + x[,"n_33"])*(-log10(2) + 2*log10(5)) - (x[,"n_12"] + x[,"n_34"])*(-log10(2) + 2*log10(5)) + (x[,"n_04"] + x[,"n_42"])*(log10(3 - r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_02"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_rc)) + log10(2 + r_rc)) + (x[,"n_12"] + x[,"n_34"])*log10(15 - 4*r_rc - 2*r_rc^2) + x[,"n_23"]*log10(7 + 2*r_rc - 2*r_rc^2) + (x[,"n_14"] + x[,"n_32"])*log10(9 + 8*r_rc - 2*r_rc^2) + (x[,"n_22"] + x[,"n_24"])*log10(4 - r_rc + r_rc^2) + (x[,"n_03"] + x[,"n_43"])*log10(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_13"] + x[,"n_33"])*log10(13 - 2*r_rc + 2*r_rc^2)
logL_rc <- (-3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 3*x[,"n_32"] - 2*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (x[,"n_22"] + x[,"n_23"] + x[,"n_24"])*log(3) + 2*(-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_42"])*(log(3 - r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_02"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_rc)) + log(2 + r_rc)) + (x[,"n_12"] + x[,"n_34"])*log(15 - 4*r_rc - 2*r_rc^2) + x[,"n_23"]*log(7 + 2*r_rc - 2*r_rc^2) + (x[,"n_14"] + x[,"n_32"])*log(9 + 8*r_rc - 2*r_rc^2) + (x[,"n_22"] + x[,"n_24"])*log(4 - r_rc + r_rc^2) + (x[,"n_03"] + x[,"n_43"])*log(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_13"] + x[,"n_33"])*log(13 - 2*r_rc + 2*r_rc^2)
logL_rm2c <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
L <- (-3*n02 - 2*n03 - 3*n04 - 2*n12 - n13 - 2*n14 - 3*n22 - n23 - 3*n24 - 2*n32 - n33 - 2*n34 - 3*n42 - 2*n43 - 3*n44)*log(2) + (n22 + n23 + n24)*log(3) + 2*(-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*log(5) + (n04 + n42)*(log(pmax(1e-6,1 - r)) + log(3 - r)) + (n02 + n44)*(log(pmax(1e-6,r)) + log(2 + r)) + (n22 + n24)*log(7 + 2*r - 2*r^2) + (n03 + n43)*log(1 + r - r^2) + (n13 + n33)*log(6 + r - r^2) + (n14 + n32)*log(10 - 8*r + r^2) + n23*log(4 - r + r^2) + (n12 + n34)*log(3 + 6*r + r^2)
return(L)}
interlogL_rm2c <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
optimize(logL_rm2c,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm2c <- -((x[,"n_04"] + x[,"n_42"])*(-2*log10(2) + log10(5))) - (x[,"n_03"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - x[,"n_23"]*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_22"] + x[,"n_24"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_14"] + x[,"n_32"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_13"] + x[,"n_33"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_12"] + x[,"n_34"])*(-2*log10(2) + 2*log10(5)) + (x[,"n_04"] + x[,"n_42"])*(log10(pmax(1e-6,1 - r_rm2c)) + log10(3 - r_rm2c)) + (x[,"n_02"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm2c)) + log10(2 + r_rm2c)) + (x[,"n_22"] + x[,"n_24"])*log10(7 + 2*r_rm2c - 2*r_rm2c^2) + (x[,"n_03"] + x[,"n_43"])*log10(1 + r_rm2c - r_rm2c^2) + (x[,"n_13"] + x[,"n_33"])*log10(6 + r_rm2c - r_rm2c^2) + (x[,"n_14"] + x[,"n_32"])*log10(10 - 8*r_rm2c + r_rm2c^2) + x[,"n_23"]*log10(4 - r_rm2c + r_rm2c^2) + (x[,"n_12"] + x[,"n_34"])*log10(3 + 6*r_rm2c + r_rm2c^2)
logL_rm2c <- (-3*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_22"] - x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_32"] - x[,"n_33"] - 2*x[,"n_34"] - 3*x[,"n_42"] - 2*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (x[,"n_22"] + x[,"n_23"] + x[,"n_24"])*log(3) + 2*(-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_42"])*(log(pmax(1e-6,1 - r_rm2c)) + log(3 - r_rm2c)) + (x[,"n_02"] + x[,"n_44"])*(log(pmax(1e-6,r_rm2c)) + log(2 + r_rm2c)) + (x[,"n_22"] + x[,"n_24"])*log(7 + 2*r_rm2c - 2*r_rm2c^2) + (x[,"n_03"] + x[,"n_43"])*log(1 + r_rm2c - r_rm2c^2) + (x[,"n_13"] + x[,"n_33"])*log(6 + r_rm2c - r_rm2c^2) + (x[,"n_14"] + x[,"n_32"])*log(10 - 8*r_rm2c + r_rm2c^2) + x[,"n_23"]*log(4 - r_rm2c + r_rm2c^2) + (x[,"n_12"] + x[,"n_34"])*log(3 + 6*r_rm2c + r_rm2c^2)

return(list(
r_mat = cbind(r_cc,r_cm2c,r_rc,r_rm2c,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm2c,LOD_rc,LOD_rm2c,0),
logL_mat = cbind(logL_cc,logL_cm2c,logL_rc,logL_rm2c,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed2c","repulsion coupling","repulsion mixed2c","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_2.2 <- function(x){
logL_cc <- function(r,n00,n01,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n43,n44,n02,n42) {
L <- (-2*n00 - 3*n01 - 3*n03 - 2*n04 - 2*n10 - n11 - n12 - n13 - 2*n14 - 2*n20 - 3*n21 + n22 - 3*n23 - 2*n24 - 2*n30 - n31 - n32 - n33 - 2*n34 - 2*n40 - 3*n41 - 3*n43 - 2*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n20 + n24)*log(7) + 3*(n00 + n44)*log(pmax(1e-6,1 - r)) + 3*(n04 + n40)*log(pmax(1e-6,r)) + (n02 + n20 + n24 + n42)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n41)*(log(7 - 4*r) + 2*log(pmax(1e-6,r))) + (n14 + n30)*(log(5 - 2*r) + 2*log(pmax(1e-6,r))) + (n10 + n34)*(2*log(pmax(1e-6,1 - r)) + log(3 + 2*r)) + (n01 + n43)*(2*log(pmax(1e-6,1 - r)) + log(3 + 4*r)) + (n12 + n32)*log(5 + 2*r - 2*r^2) + (n21 + n23)*log(21 + 2*r - 2*r^2) + (n13 + n31)*(log(pmax(1e-6,r)) + log(7 - 3*r + 2*r^2)) + n22*log(3 - 2*r + 2*r^2) + (n11 + n33)*log(6 - 7*r + 3*r^2 - 2*r^3)
return(L)}
interlogL_cc <- function(n00,n01,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n43,n44,n02,n42) {
optimize(logL_cc,c(0,0.5), n00,n01,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n43,n44,n02,n42, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_43"],x[,"n_44"],x[,"n_02"],x[,"n_42"])
LOD_cc <- 3*(x[,"n_04"] + x[,"n_40"])*log10(2) + 2*(x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(2) + 3*(x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_13"] + x[,"n_31"])*log10(3) - (x[,"n_11"] + x[,"n_33"])*log10(3) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - x[,"n_22"]*(-log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_32"])*(-log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_23"])*(-log10(2) + log10(43)) + 3*(x[,"n_00"] + x[,"n_44"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_04"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_41"])*(log10(7 - 4*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_30"])*(log10(5 - 2*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_34"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(3 + 2*r_cc)) + (x[,"n_01"] + x[,"n_43"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(3 + 4*r_cc)) + (x[,"n_12"] + x[,"n_32"])*log10(5 + 2*r_cc - 2*r_cc^2) + (x[,"n_21"] + x[,"n_23"])*log10(21 + 2*r_cc - 2*r_cc^2) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_cc)) + log10(7 - 3*r_cc + 2*r_cc^2)) + x[,"n_22"]*log10(3 - 2*r_cc + 2*r_cc^2) + (x[,"n_11"] + x[,"n_33"])*log10(6 - 7*r_cc + 3*r_cc^2 - 2*r_cc^3)
logL_cc <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 3*x[,"n_21"] + x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_20"] + x[,"n_24"])*log(7) + 3*(x[,"n_00"] + x[,"n_44"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_04"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_03"] + x[,"n_41"])*(log(7 - 4*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_30"])*(log(5 - 2*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_34"])*(2*log(pmax(1e-6,1 - r_cc)) + log(3 + 2*r_cc)) + (x[,"n_01"] + x[,"n_43"])*(2*log(pmax(1e-6,1 - r_cc)) + log(3 + 4*r_cc)) + (x[,"n_12"] + x[,"n_32"])*log(5 + 2*r_cc - 2*r_cc^2) + (x[,"n_21"] + x[,"n_23"])*log(21 + 2*r_cc - 2*r_cc^2) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_cc)) + log(7 - 3*r_cc + 2*r_cc^2)) + x[,"n_22"]*log(3 - 2*r_cc + 2*r_cc^2) + (x[,"n_11"] + x[,"n_33"])*log(6 - 7*r_cc + 3*r_cc^2 - 2*r_cc^3)
logL_cm <- function(r,n00,n01,n02,n03,n04,n10,n12,n14,n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n11,n13,n31,n33) {
L <- (-n00 - 2*n01 - 3*n02 - 2*n03 - n04 - n10 - 2*n12 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - n30 - 2*n32 - n34 - n40 - 2*n41 - 3*n42 - 2*n43 - n44)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n12 + n32)*log(11) + (n00 + n44)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + n22*log(57 + 8*r - 8*r^2) + (n20 + n24)*log(3 - r + r^2) + (n14 + n30)*(log(pmax(1e-6,r)) + log(4 - 3*r + 2*r^2)) + (n21 + n23)*log(33 - 2*r + 2*r^2) + (n03 + n41)*(log(pmax(1e-6,r)) + log(6 - 7*r + 4*r^2)) + (n02 + n42)*log(9 - 8*r + 8*r^2) + (n01 + n43)*log(3 - 4*r + 5*r^2 - 4*r^3) + (n10 + n34)*log(3 - 4*r + 3*r^2 - 2*r^3) + (n13 + n31)*log(3 + 2*r + 3*r^2 - 2*r^3) + (n11 + n33)*log(6 - 2*r - 3*r^2 + 2*r^3)
return(L)}
interlogL_cm <- function(n00,n01,n02,n03,n04,n10,n12,n14,n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n11,n13,n31,n33) {
optimize(logL_cm,c(0,0.5), n00,n01,n02,n03,n04,n10,n12,n14,n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n11,n13,n31,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_12"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_32"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_11"],x[,"n_13"],x[,"n_31"],x[,"n_33"])
LOD_cm <- 3*(x[,"n_04"] + x[,"n_40"])*log10(2) + 3*(x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_14"] + x[,"n_30"])*(-log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_34"])*(-log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + 2*log10(3)) - (x[,"n_02"] + x[,"n_42"])*log10(7) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(7)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(7)) - (x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_23"])*(-log10(2) + log10(5) + log10(13)) - x[,"n_22"]*log10(59) + (x[,"n_00"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_cm)) + 2*log10(pmax(1e-6,r_cm))) + x[,"n_22"]*log10(57 + 8*r_cm - 8*r_cm^2) + (x[,"n_20"] + x[,"n_24"])*log10(3 - r_cm + r_cm^2) + (x[,"n_14"] + x[,"n_30"])*(log10(pmax(1e-6,r_cm)) + log10(4 - 3*r_cm + 2*r_cm^2)) + (x[,"n_21"] + x[,"n_23"])*log10(33 - 2*r_cm + 2*r_cm^2) + (x[,"n_03"] + x[,"n_41"])*(log10(pmax(1e-6,r_cm)) + log10(6 - 7*r_cm + 4*r_cm^2)) + (x[,"n_02"] + x[,"n_42"])*log10(9 - 8*r_cm + 8*r_cm^2) + (x[,"n_01"] + x[,"n_43"])*log10(3 - 4*r_cm + 5*r_cm^2 - 4*r_cm^3) + (x[,"n_10"] + x[,"n_34"])*log10(3 - 4*r_cm + 3*r_cm^2 - 2*r_cm^3) + (x[,"n_13"] + x[,"n_31"])*log10(3 + 2*r_cm + 3*r_cm^2 - 2*r_cm^3) + (x[,"n_11"] + x[,"n_33"])*log10(6 - 2*r_cm - 3*r_cm^2 + 2*r_cm^3)
logL_cm <- (-x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - 2*x[,"n_12"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - 2*x[,"n_32"] - x[,"n_34"] - x[,"n_40"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 2*x[,"n_43"] - x[,"n_44"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_12"] + x[,"n_32"])*log(11) + (x[,"n_00"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_cm)) + 2*log(pmax(1e-6,r_cm))) + x[,"n_22"]*log(57 + 8*r_cm - 8*r_cm^2) + (x[,"n_20"] + x[,"n_24"])*log(3 - r_cm + r_cm^2) + (x[,"n_14"] + x[,"n_30"])*(log(pmax(1e-6,r_cm)) + log(4 - 3*r_cm + 2*r_cm^2)) + (x[,"n_21"] + x[,"n_23"])*log(33 - 2*r_cm + 2*r_cm^2) + (x[,"n_03"] + x[,"n_41"])*(log(pmax(1e-6,r_cm)) + log(6 - 7*r_cm + 4*r_cm^2)) + (x[,"n_02"] + x[,"n_42"])*log(9 - 8*r_cm + 8*r_cm^2) + (x[,"n_01"] + x[,"n_43"])*log(3 - 4*r_cm + 5*r_cm^2 - 4*r_cm^3) + (x[,"n_10"] + x[,"n_34"])*log(3 - 4*r_cm + 3*r_cm^2 - 2*r_cm^3) + (x[,"n_13"] + x[,"n_31"])*log(3 + 2*r_cm + 3*r_cm^2 - 2*r_cm^3) + (x[,"n_11"] + x[,"n_33"])*log(6 - 2*r_cm - 3*r_cm^2 + 2*r_cm^3)
logL_cr <- function(r,n00,n01,n02,n03,n04,n10,n11,n13,n14,n20,n21,n22,n23,n24,n30,n31,n33,n34,n40,n41,n42,n43,n44,n12,n32) {
L <- (-2*n00 - 3*n01 - 2*n02 - 3*n03 - 2*n04 - 2*n10 - n11 - n13 - 2*n14 - 2*n20 - 3*n21 - n22 - 3*n23 - 2*n24 - 2*n30 - n31 - n33 - 2*n34 - 2*n40 - 3*n41 - 2*n42 - 3*n43 - 2*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n04 + n40)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n44)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + n22*log(9 + 4*r - 4*r^2) + (n21 + n23)*log(21 + 2*r - 2*r^2) + (n12 + n32)*log(3 - r + r^2) + (n10 + n34)*(log(pmax(1e-6,r)) + log(4 - 5*r + 2*r^2)) + (n01 + n43)*(log(pmax(1e-6,r)) + log(4 - 5*r + 4*r^2)) + (n20 + n24)*log(3 - 5*r + 5*r^2) + (n03 + n41)*log(3 - 6*r + 7*r^2 - 4*r^3) + (n14 + n30)*log(1 + r^2 - 2*r^3) + (n11 + n33)*log(3 - r + 3*r^2 - 2*r^3) + (n13 + n31)*log(3 + r - 3*r^2 + 2*r^3)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n10,n11,n13,n14,n20,n21,n22,n23,n24,n30,n31,n33,n34,n40,n41,n42,n43,n44,n12,n32) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n13,n14,n20,n21,n22,n23,n24,n30,n31,n33,n34,n40,n41,n42,n43,n44,n12,n32, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_12"],x[,"n_32"])
LOD_cr <- 3*(x[,"n_04"] + x[,"n_40"])*log10(2) + 3*(x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_13"] + x[,"n_31"])*log10(3) - (x[,"n_11"] + x[,"n_33"])*log10(3) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - x[,"n_22"]*(log10(2) + log10(5)) - (x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(7)) - (x[,"n_12"] + x[,"n_32"])*(-2*log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_23"])*(-log10(2) + log10(43)) + (x[,"n_04"] + x[,"n_40"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cr)) + 2*log10(pmax(1e-6,r_cr))) + x[,"n_22"]*log10(9 + 4*r_cr - 4*r_cr^2) + (x[,"n_21"] + x[,"n_23"])*log10(21 + 2*r_cr - 2*r_cr^2) + (x[,"n_12"] + x[,"n_32"])*log10(3 - r_cr + r_cr^2) + (x[,"n_10"] + x[,"n_34"])*(log10(pmax(1e-6,r_cr)) + log10(4 - 5*r_cr + 2*r_cr^2)) + (x[,"n_01"] + x[,"n_43"])*(log10(pmax(1e-6,r_cr)) + log10(4 - 5*r_cr + 4*r_cr^2)) + (x[,"n_20"] + x[,"n_24"])*log10(3 - 5*r_cr + 5*r_cr^2) + (x[,"n_03"] + x[,"n_41"])*log10(3 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3) + (x[,"n_14"] + x[,"n_30"])*log10(1 + r_cr^2 - 2*r_cr^3) + (x[,"n_11"] + x[,"n_33"])*log10(3 - r_cr + 3*r_cr^2 - 2*r_cr^3) + (x[,"n_13"] + x[,"n_31"])*log10(3 + r_cr - 3*r_cr^2 + 2*r_cr^3)
logL_cr <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 2*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 3*x[,"n_21"] - x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - x[,"n_31"] - x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 2*x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_40"])*(2*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cr)) + 2*log(pmax(1e-6,r_cr))) + x[,"n_22"]*log(9 + 4*r_cr - 4*r_cr^2) + (x[,"n_21"] + x[,"n_23"])*log(21 + 2*r_cr - 2*r_cr^2) + (x[,"n_12"] + x[,"n_32"])*log(3 - r_cr + r_cr^2) + (x[,"n_10"] + x[,"n_34"])*(log(pmax(1e-6,r_cr)) + log(4 - 5*r_cr + 2*r_cr^2)) + (x[,"n_01"] + x[,"n_43"])*(log(pmax(1e-6,r_cr)) + log(4 - 5*r_cr + 4*r_cr^2)) + (x[,"n_20"] + x[,"n_24"])*log(3 - 5*r_cr + 5*r_cr^2) + (x[,"n_03"] + x[,"n_41"])*log(3 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3) + (x[,"n_14"] + x[,"n_30"])*log(1 + r_cr^2 - 2*r_cr^3) + (x[,"n_11"] + x[,"n_33"])*log(3 - r_cr + 3*r_cr^2 - 2*r_cr^3) + (x[,"n_13"] + x[,"n_31"])*log(3 + r_cr - 3*r_cr^2 + 2*r_cr^3)
logL_rc <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 4*n20 - 3*n21 - 3*n22 - 3*n23 - 4*n24 - 3*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 4*n40 - 3*n41 - 4*n42 - 3*n43 - 4*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n04 + n40)*(log(3 - 2*r) + 2*log(pmax(1e-6,r))) + (n00 + n44)*(2*log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n02 + n42)*log(3 + 4*r - 4*r^2) + n22*log(39 + 4*r - 4*r^2) + (n21 + n23)*log(21 + 2*r - 2*r^2) + (n14 + n30)*(log(pmax(1e-6,r)) + log(6 - 5*r + 2*r^2)) + (n03 + n41)*(log(pmax(1e-6,r)) + log(3 - 2*r + 2*r^2)) + (n12 + n32)*log(23 - 4*r + 4*r^2) + (n20 + n24)*log(9 - 8*r + 8*r^2) + (n13 + n31)*log(9 + 4*r + 6*r^2 - 4*r^3) + (n10 + n34)*log(3 - 2*r + r^2 - 2*r^3) + (n01 + n43)*log(3 - 5*r + 4*r^2 - 2*r^3) + (n11 + n33)*log(15 - 4*r - 6*r^2 + 4*r^3)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rc <- -((x[,"n_14"] + x[,"n_30"])*log10(2)) - (x[,"n_10"] + x[,"n_34"])*log10(2) + (x[,"n_04"] + x[,"n_40"])*log10(2) - 2*(x[,"n_02"] + x[,"n_42"])*log10(2) + (x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_13"] + x[,"n_31"])*(2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_33"])*(2*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - x[,"n_22"]*(3*log10(2) + log10(5)) - (x[,"n_20"] + x[,"n_24"])*log10(7) - (x[,"n_12"] + x[,"n_32"])*(log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_23"])*(-log10(2) + log10(43)) + (x[,"n_04"] + x[,"n_40"])*(log10(3 - 2*r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_02"] + x[,"n_42"])*log10(3 + 4*r_rc - 4*r_rc^2) + x[,"n_22"]*log10(39 + 4*r_rc - 4*r_rc^2) + (x[,"n_21"] + x[,"n_23"])*log10(21 + 2*r_rc - 2*r_rc^2) + (x[,"n_14"] + x[,"n_30"])*(log10(pmax(1e-6,r_rc)) + log10(6 - 5*r_rc + 2*r_rc^2)) + (x[,"n_03"] + x[,"n_41"])*(log10(pmax(1e-6,r_rc)) + log10(3 - 2*r_rc + 2*r_rc^2)) + (x[,"n_12"] + x[,"n_32"])*log10(23 - 4*r_rc + 4*r_rc^2) + (x[,"n_20"] + x[,"n_24"])*log10(9 - 8*r_rc + 8*r_rc^2) + (x[,"n_13"] + x[,"n_31"])*log10(9 + 4*r_rc + 6*r_rc^2 - 4*r_rc^3) + (x[,"n_10"] + x[,"n_34"])*log10(3 - 2*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_01"] + x[,"n_43"])*log10(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3) + (x[,"n_11"] + x[,"n_33"])*log10(15 - 4*r_rc - 6*r_rc^2 + 4*r_rc^3)
logL_rc <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_40"])*(log(3 - 2*r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_02"] + x[,"n_42"])*log(3 + 4*r_rc - 4*r_rc^2) + x[,"n_22"]*log(39 + 4*r_rc - 4*r_rc^2) + (x[,"n_21"] + x[,"n_23"])*log(21 + 2*r_rc - 2*r_rc^2) + (x[,"n_14"] + x[,"n_30"])*(log(pmax(1e-6,r_rc)) + log(6 - 5*r_rc + 2*r_rc^2)) + (x[,"n_03"] + x[,"n_41"])*(log(pmax(1e-6,r_rc)) + log(3 - 2*r_rc + 2*r_rc^2)) + (x[,"n_12"] + x[,"n_32"])*log(23 - 4*r_rc + 4*r_rc^2) + (x[,"n_20"] + x[,"n_24"])*log(9 - 8*r_rc + 8*r_rc^2) + (x[,"n_13"] + x[,"n_31"])*log(9 + 4*r_rc + 6*r_rc^2 - 4*r_rc^3) + (x[,"n_10"] + x[,"n_34"])*log(3 - 2*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_01"] + x[,"n_43"])*log(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3) + (x[,"n_11"] + x[,"n_33"])*log(15 - 4*r_rc - 6*r_rc^2 + 4*r_rc^3)
logL_rm <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-3*n00 - 4*n01 - 3*n02 - 4*n03 - 3*n04 - 3*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 3*n30 - 2*n31 - 2*n32 - 2*n33 - 3*n34 - 3*n40 - 4*n41 - 3*n42 - 4*n43 - 3*n44)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n12 + n32)*log(11) + (n04 + n40)*(log(3 - 2*r) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n44)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(1 + 2*r)) + n22*log(57 + 8*r - 8*r^2) + (n20 + n24)*log(3 - r + r^2) + (n21 + n23)*log(33 - 2*r + 2*r^2) + (n02 + n42)*log(9 - 8*r + 8*r^2) + (n03 + n41)*log(9 - 6*r + 8*r^2 - 8*r^3) + (n14 + n30)*log(9 - 8*r + 6*r^2 - 4*r^3) + (n11 + n33)*log(15 + 4*r + 6*r^2 - 4*r^3) + (n13 + n31)*log(21 - 4*r - 6*r^2 + 4*r^3) + (n10 + n34)*log(3 + 8*r - 6*r^2 + 4*r^3) + (n01 + n43)*log(3 + 14*r - 16*r^2 + 8*r^3)
return(L)}
interlogL_rm <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rm,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm <- (x[,"n_04"] + x[,"n_40"])*log10(2) + (x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_14"] + x[,"n_30"])*(log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_34"])*(log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_31"])*(log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_33"])*(log10(2) + 2*log10(3)) - (x[,"n_03"] + x[,"n_41"])*log10(7) - (x[,"n_02"] + x[,"n_42"])*log10(7) - (x[,"n_01"] + x[,"n_43"])*log10(7) - (x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_23"])*(-log10(2) + log10(5) + log10(13)) - x[,"n_22"]*log10(59) + (x[,"n_04"] + x[,"n_40"])*(log10(3 - 2*r_rm) + log10(pmax(1e-6,1 - r_rm)) + log10(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_rm)) + log10(pmax(1e-6,r_rm)) + log10(1 + 2*r_rm)) + x[,"n_22"]*log10(57 + 8*r_rm - 8*r_rm^2) + (x[,"n_20"] + x[,"n_24"])*log10(3 - r_rm + r_rm^2) + (x[,"n_21"] + x[,"n_23"])*log10(33 - 2*r_rm + 2*r_rm^2) + (x[,"n_02"] + x[,"n_42"])*log10(9 - 8*r_rm + 8*r_rm^2) + (x[,"n_03"] + x[,"n_41"])*log10(9 - 6*r_rm + 8*r_rm^2 - 8*r_rm^3) + (x[,"n_14"] + x[,"n_30"])*log10(9 - 8*r_rm + 6*r_rm^2 - 4*r_rm^3) + (x[,"n_11"] + x[,"n_33"])*log10(15 + 4*r_rm + 6*r_rm^2 - 4*r_rm^3) + (x[,"n_13"] + x[,"n_31"])*log10(21 - 4*r_rm - 6*r_rm^2 + 4*r_rm^3) + (x[,"n_10"] + x[,"n_34"])*log10(3 + 8*r_rm - 6*r_rm^2 + 4*r_rm^3) + (x[,"n_01"] + x[,"n_43"])*log10(3 + 14*r_rm - 16*r_rm^2 + 8*r_rm^3)
logL_rm <- (-3*x[,"n_00"] - 4*x[,"n_01"] - 3*x[,"n_02"] - 4*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 3*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_40"] - 4*x[,"n_41"] - 3*x[,"n_42"] - 4*x[,"n_43"] - 3*x[,"n_44"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_12"] + x[,"n_32"])*log(11) + (x[,"n_04"] + x[,"n_40"])*(log(3 - 2*r_rm) + log(pmax(1e-6,1 - r_rm)) + log(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_rm)) + log(pmax(1e-6,r_rm)) + log(1 + 2*r_rm)) + x[,"n_22"]*log(57 + 8*r_rm - 8*r_rm^2) + (x[,"n_20"] + x[,"n_24"])*log(3 - r_rm + r_rm^2) + (x[,"n_21"] + x[,"n_23"])*log(33 - 2*r_rm + 2*r_rm^2) + (x[,"n_02"] + x[,"n_42"])*log(9 - 8*r_rm + 8*r_rm^2) + (x[,"n_03"] + x[,"n_41"])*log(9 - 6*r_rm + 8*r_rm^2 - 8*r_rm^3) + (x[,"n_14"] + x[,"n_30"])*log(9 - 8*r_rm + 6*r_rm^2 - 4*r_rm^3) + (x[,"n_11"] + x[,"n_33"])*log(15 + 4*r_rm + 6*r_rm^2 - 4*r_rm^3) + (x[,"n_13"] + x[,"n_31"])*log(21 - 4*r_rm - 6*r_rm^2 + 4*r_rm^3) + (x[,"n_10"] + x[,"n_34"])*log(3 + 8*r_rm - 6*r_rm^2 + 4*r_rm^3) + (x[,"n_01"] + x[,"n_43"])*log(3 + 14*r_rm - 16*r_rm^2 + 8*r_rm^3)
logL_rr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 4*n20 - 3*n21 - 3*n22 - 3*n23 - 4*n24 - 3*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 4*n40 - 3*n41 - 4*n42 - 3*n43 - 4*n44)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n20 - 2*n21 - n22 - 2*n23 - 2*n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n40 - 2*n41 - 2*n42 - 2*n43 - 2*n44)*log(5) + (n04 + n40)*(log(3 - 2*r) + 2*log(pmax(1e-6,1 - r))) + (n00 + n44)*(2*log(pmax(1e-6,r)) + log(1 + 2*r)) + (n20 + n24)*log(3 + 16*r - 16*r^2) + (n02 + n42)*log(1 + 12*r - 12*r^2) + (n12 + n32)*log(21 + 4*r - 4*r^2) + (n21 + n23)*log(21 + 2*r - 2*r^2) + (n10 + n34)*(log(pmax(1e-6,r)) + log(2 + 5*r - 2*r^2)) + n22*log(9 - 4*r + 4*r^2) + (n13 + n31)*log(21 - 20*r + 6*r^2 - 4*r^3) + (n01 + n43)*log(r + 4*r^2 - 2*r^3) + (n03 + n41)*log(3 - 3*r - 2*r^2 + 2*r^3) + (n14 + n30)*log(5 - 6*r - r^2 + 2*r^3) + (n11 + n33)*log(3 + 20*r - 6*r^2 + 4*r^3)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rr <- -3*x[,"n_22"]*log10(2) - (x[,"n_14"] + x[,"n_30"])*log10(2) - (x[,"n_10"] + x[,"n_34"])*log10(2) + (x[,"n_04"] + x[,"n_40"])*log10(2) - 2*(x[,"n_02"] + x[,"n_42"])*log10(2) + (x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_13"] + x[,"n_31"])*(2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_33"])*(2*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - (x[,"n_20"] + x[,"n_24"])*log10(7) - (x[,"n_12"] + x[,"n_32"])*(log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_23"])*(-log10(2) + log10(43)) + (x[,"n_04"] + x[,"n_40"])*(log10(3 - 2*r_rr) + 2*log10(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_44"])*(2*log10(pmax(1e-6,r_rr)) + log10(1 + 2*r_rr)) + (x[,"n_20"] + x[,"n_24"])*log10(3 + 16*r_rr - 16*r_rr^2) + (x[,"n_02"] + x[,"n_42"])*log10(1 + 12*r_rr - 12*r_rr^2) + (x[,"n_12"] + x[,"n_32"])*log10(21 + 4*r_rr - 4*r_rr^2) + (x[,"n_21"] + x[,"n_23"])*log10(21 + 2*r_rr - 2*r_rr^2) + (x[,"n_10"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(2 + 5*r_rr - 2*r_rr^2)) + x[,"n_22"]*log10(9 - 4*r_rr + 4*r_rr^2) + (x[,"n_13"] + x[,"n_31"])*log10(21 - 20*r_rr + 6*r_rr^2 - 4*r_rr^3) + (x[,"n_01"] + x[,"n_43"])*log10(r_rr + 4*r_rr^2 - 2*r_rr^3) + (x[,"n_03"] + x[,"n_41"])*log10(3 - 3*r_rr - 2*r_rr^2 + 2*r_rr^3) + (x[,"n_14"] + x[,"n_30"])*log10(5 - 6*r_rr - r_rr^2 + 2*r_rr^3) + (x[,"n_11"] + x[,"n_33"])*log10(3 + 20*r_rr - 6*r_rr^2 + 4*r_rr^3)
logL_rr <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"])*log(5) + (x[,"n_04"] + x[,"n_40"])*(log(3 - 2*r_rr) + 2*log(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_44"])*(2*log(pmax(1e-6,r_rr)) + log(1 + 2*r_rr)) + (x[,"n_20"] + x[,"n_24"])*log(3 + 16*r_rr - 16*r_rr^2) + (x[,"n_02"] + x[,"n_42"])*log(1 + 12*r_rr - 12*r_rr^2) + (x[,"n_12"] + x[,"n_32"])*log(21 + 4*r_rr - 4*r_rr^2) + (x[,"n_21"] + x[,"n_23"])*log(21 + 2*r_rr - 2*r_rr^2) + (x[,"n_10"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(2 + 5*r_rr - 2*r_rr^2)) + x[,"n_22"]*log(9 - 4*r_rr + 4*r_rr^2) + (x[,"n_13"] + x[,"n_31"])*log(21 - 20*r_rr + 6*r_rr^2 - 4*r_rr^3) + (x[,"n_01"] + x[,"n_43"])*log(r_rr + 4*r_rr^2 - 2*r_rr^3) + (x[,"n_03"] + x[,"n_41"])*log(3 - 3*r_rr - 2*r_rr^2 + 2*r_rr^3) + (x[,"n_14"] + x[,"n_30"])*log(5 - 6*r_rr - r_rr^2 + 2*r_rr^3) + (x[,"n_11"] + x[,"n_33"])*log(3 + 20*r_rr - 6*r_rr^2 + 4*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_2.3 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n05,n11,n12,n13,n14,n20,n21,n22,n23,n24,n25,n31,n32,n33,n34,n40,n41,n42,n43,n44,n45,n10,n15,n30,n35) {
L <- (-2*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 2*n05 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - n25 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + (n01 + n04 + n11 + n14 + n20 + n21 + n22 + n23 + n24 + n25 + n31 + n34 + n41 + n44)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + 4*(n00 + n45)*log(pmax(1e-6,1 - r)) + 4*(n05 + n40)*log(pmax(1e-6,r)) + (n10 + n35)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n25)*(2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n41)*(log(3 - 2*r) + 3*log(pmax(1e-6,r))) + (n15 + n30)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n01 + n44)*(3*log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n02 + n43)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(11 + 4*r)) + (n03 + n42)*(2*log(pmax(1e-6,r)) + log(15 - 19*r + 4*r^2)) + (n14 + n31)*(2*log(pmax(1e-6,r)) + log(13 - 14*r + 8*r^2)) + (n11 + n34)*(2*log(pmax(1e-6,1 - r)) + log(7 - 2*r + 8*r^2)) + (n13 + n32)*(log(pmax(1e-6,r)) + log(60 - 69*r + 54*r^2 - 16*r^3)) + (n21 + n24)*(log(pmax(1e-6,r)) + log(17 - 29*r + 24*r^2 - 12*r^3)) + (n12 + n33)*log(29 - 20*r - 3*r^2 + 10*r^3 - 16*r^4) + (n22 + n23)*log(15 - 17*r + 25*r^2 - 16*r^3 + 8*r^4)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n05,n11,n12,n13,n14,n20,n21,n22,n23,n24,n25,n31,n32,n33,n34,n40,n41,n42,n43,n44,n45,n10,n15,n30,n35) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n05,n11,n12,n13,n14,n20,n21,n22,n23,n24,n25,n31,n32,n33,n34,n40,n41,n42,n43,n44,n45,n10,n15,n30,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_10"],x[,"n_15"],x[,"n_30"],x[,"n_35"])
LOD_cc <- 4*(x[,"n_20"] + x[,"n_25"])*log10(2) + 4*(x[,"n_15"] + x[,"n_30"])*log10(2) - (x[,"n_14"] + x[,"n_31"])*log10(2) - (x[,"n_11"] + x[,"n_34"])*log10(2) + 4*(x[,"n_10"] + x[,"n_35"])*log10(2) + 4*(x[,"n_05"] + x[,"n_40"])*log10(2) + 2*(x[,"n_04"] + x[,"n_41"])*log10(2) + 2*(x[,"n_01"] + x[,"n_44"])*log10(2) + 4*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_22"] + x[,"n_23"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_21"] + x[,"n_24"])*(-log10(2) + log10(7)) - (x[,"n_03"] + x[,"n_42"])*(-3*log10(2) + log10(13)) - (x[,"n_02"] + x[,"n_43"])*(-3*log10(2) + log10(13)) - (x[,"n_13"] + x[,"n_32"])*(-log10(2) + log10(37)) - (x[,"n_12"] + x[,"n_33"])*(-log10(2) + log10(37)) + 4*(x[,"n_00"] + x[,"n_45"])*log10(pmax(1e-6,1 - r_cc)) + 4*(x[,"n_05"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_35"])*(3*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_20"] + x[,"n_25"])*(2*log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_41"])*(log10(3 - 2*r_cc) + 3*log10(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_cc)) + 3*log10(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_44"])*(3*log10(pmax(1e-6,1 - r_cc)) + log10(1 + 2*r_cc)) + (x[,"n_02"] + x[,"n_43"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc)) + log10(11 + 4*r_cc)) + (x[,"n_03"] + x[,"n_42"])*(2*log10(pmax(1e-6,r_cc)) + log10(15 - 19*r_cc + 4*r_cc^2)) + (x[,"n_14"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc)) + log10(13 - 14*r_cc + 8*r_cc^2)) + (x[,"n_11"] + x[,"n_34"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(7 - 2*r_cc + 8*r_cc^2)) + (x[,"n_13"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(60 - 69*r_cc + 54*r_cc^2 - 16*r_cc^3)) + (x[,"n_21"] + x[,"n_24"])*(log10(pmax(1e-6,r_cc)) + log10(17 - 29*r_cc + 24*r_cc^2 - 12*r_cc^3)) + (x[,"n_12"] + x[,"n_33"])*log10(29 - 20*r_cc - 3*r_cc^2 + 10*r_cc^3 - 16*r_cc^4) + (x[,"n_22"] + x[,"n_23"])*log10(15 - 17*r_cc + 25*r_cc^2 - 16*r_cc^3 + 8*r_cc^4)
logL_cc <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - x[,"n_25"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_25"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + 4*(x[,"n_00"] + x[,"n_45"])*log(pmax(1e-6,1 - r_cc)) + 4*(x[,"n_05"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + (x[,"n_10"] + x[,"n_35"])*(3*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_20"] + x[,"n_25"])*(2*log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_41"])*(log(3 - 2*r_cc) + 3*log(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_cc)) + 3*log(pmax(1e-6,r_cc))) + (x[,"n_01"] + x[,"n_44"])*(3*log(pmax(1e-6,1 - r_cc)) + log(1 + 2*r_cc)) + (x[,"n_02"] + x[,"n_43"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc)) + log(11 + 4*r_cc)) + (x[,"n_03"] + x[,"n_42"])*(2*log(pmax(1e-6,r_cc)) + log(15 - 19*r_cc + 4*r_cc^2)) + (x[,"n_14"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc)) + log(13 - 14*r_cc + 8*r_cc^2)) + (x[,"n_11"] + x[,"n_34"])*(2*log(pmax(1e-6,1 - r_cc)) + log(7 - 2*r_cc + 8*r_cc^2)) + (x[,"n_13"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(60 - 69*r_cc + 54*r_cc^2 - 16*r_cc^3)) + (x[,"n_21"] + x[,"n_24"])*(log(pmax(1e-6,r_cc)) + log(17 - 29*r_cc + 24*r_cc^2 - 12*r_cc^3)) + (x[,"n_12"] + x[,"n_33"])*log(29 - 20*r_cc - 3*r_cc^2 + 10*r_cc^3 - 16*r_cc^4) + (x[,"n_22"] + x[,"n_23"])*log(15 - 17*r_cc + 25*r_cc^2 - 16*r_cc^3 + 8*r_cc^4)
logL_cm2c <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 2*n05 - 2*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n00 + n45)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n05 + n40)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n15 + n30)*(2*log(pmax(1e-6,r)) + log(5 - 6*r + 4*r^2)) + (n10 + n35)*(2*log(pmax(1e-6,1 - r)) + log(3 - 2*r + 4*r^2)) + (n04 + n41)*(2*log(pmax(1e-6,r)) + log(11 - 11*r + 6*r^2)) + (n01 + n44)*(2*log(pmax(1e-6,1 - r)) + log(6 - r + 6*r^2)) + (n14 + n31)*(log(pmax(1e-6,r)) + log(44 - 33*r + 46*r^2 - 24*r^3)) + (n20 + n25)*(log(pmax(1e-6,r)) + log(7 - 13*r + 12*r^2 - 6*r^3)) + (n03 + n42)*(log(pmax(1e-6,r)) + log(18 - 20*r + 15*r^2 - 4*r^3)) + (n11 + n34)*log(33 - 20*r - 39*r^2 + 50*r^3 - 24*r^4) + (n22 + n23)*log(34 - r - 7*r^2 + 16*r^3 - 8*r^4) + (n02 + n43)*log(9 - 7*r + r^2 + r^3 - 4*r^4) + (n13 + n32)*log(42 + 16*r + 41*r^2 - 46*r^3 + 16*r^4) + (n12 + n33)*log(69 - 24*r - r^2 - 18*r^3 + 16*r^4) + (n21 + n24)*log(33 - 11*r + 47*r^2 - 72*r^3 + 36*r^4)
return(L)}
interlogL_cm2c <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cm2c,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cm2c <- 4*(x[,"n_05"] + x[,"n_40"])*log10(2) + 4*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_15"] + x[,"n_30"])*(-2*log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_35"])*(-2*log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_31"])*(log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_34"])*(log10(2) + 2*log10(3)) - (x[,"n_03"] + x[,"n_42"])*(-3*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_43"])*(-3*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_04"] + x[,"n_41"])*(-2*log10(2) + log10(7)) - (x[,"n_01"] + x[,"n_44"])*(-2*log10(2) + log10(7)) - (x[,"n_20"] + x[,"n_25"])*(-3*log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_24"])*(-log10(2) + log10(5) + log10(13)) - (x[,"n_22"] + x[,"n_23"])*(-2*log10(2) + log10(7) + log10(19)) - (x[,"n_13"] + x[,"n_32"])*(-log10(2) + log10(3) + log10(37)) - (x[,"n_12"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(37)) + (x[,"n_00"] + x[,"n_45"])*(3*log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c))) + (x[,"n_05"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_cm2c)) + 3*log10(pmax(1e-6,r_cm2c))) + (x[,"n_15"] + x[,"n_30"])*(2*log10(pmax(1e-6,r_cm2c)) + log10(5 - 6*r_cm2c + 4*r_cm2c^2)) + (x[,"n_10"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_cm2c)) + log10(3 - 2*r_cm2c + 4*r_cm2c^2)) + (x[,"n_04"] + x[,"n_41"])*(2*log10(pmax(1e-6,r_cm2c)) + log10(11 - 11*r_cm2c + 6*r_cm2c^2)) + (x[,"n_01"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_cm2c)) + log10(6 - r_cm2c + 6*r_cm2c^2)) + (x[,"n_14"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm2c)) + log10(44 - 33*r_cm2c + 46*r_cm2c^2 - 24*r_cm2c^3)) + (x[,"n_20"] + x[,"n_25"])*(log10(pmax(1e-6,r_cm2c)) + log10(7 - 13*r_cm2c + 12*r_cm2c^2 - 6*r_cm2c^3)) + (x[,"n_03"] + x[,"n_42"])*(log10(pmax(1e-6,r_cm2c)) + log10(18 - 20*r_cm2c + 15*r_cm2c^2 - 4*r_cm2c^3)) + (x[,"n_11"] + x[,"n_34"])*log10(33 - 20*r_cm2c - 39*r_cm2c^2 + 50*r_cm2c^3 - 24*r_cm2c^4) + (x[,"n_22"] + x[,"n_23"])*log10(34 - r_cm2c - 7*r_cm2c^2 + 16*r_cm2c^3 - 8*r_cm2c^4) + (x[,"n_02"] + x[,"n_43"])*log10(9 - 7*r_cm2c + r_cm2c^2 + r_cm2c^3 - 4*r_cm2c^4) + (x[,"n_13"] + x[,"n_32"])*log10(42 + 16*r_cm2c + 41*r_cm2c^2 - 46*r_cm2c^3 + 16*r_cm2c^4) + (x[,"n_12"] + x[,"n_33"])*log10(69 - 24*r_cm2c - r_cm2c^2 - 18*r_cm2c^3 + 16*r_cm2c^4) + (x[,"n_21"] + x[,"n_24"])*log10(33 - 11*r_cm2c + 47*r_cm2c^2 - 72*r_cm2c^3 + 36*r_cm2c^4)
logL_cm2c <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_00"] + x[,"n_45"])*(3*log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c))) + (x[,"n_05"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_cm2c)) + 3*log(pmax(1e-6,r_cm2c))) + (x[,"n_15"] + x[,"n_30"])*(2*log(pmax(1e-6,r_cm2c)) + log(5 - 6*r_cm2c + 4*r_cm2c^2)) + (x[,"n_10"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_cm2c)) + log(3 - 2*r_cm2c + 4*r_cm2c^2)) + (x[,"n_04"] + x[,"n_41"])*(2*log(pmax(1e-6,r_cm2c)) + log(11 - 11*r_cm2c + 6*r_cm2c^2)) + (x[,"n_01"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_cm2c)) + log(6 - r_cm2c + 6*r_cm2c^2)) + (x[,"n_14"] + x[,"n_31"])*(log(pmax(1e-6,r_cm2c)) + log(44 - 33*r_cm2c + 46*r_cm2c^2 - 24*r_cm2c^3)) + (x[,"n_20"] + x[,"n_25"])*(log(pmax(1e-6,r_cm2c)) + log(7 - 13*r_cm2c + 12*r_cm2c^2 - 6*r_cm2c^3)) + (x[,"n_03"] + x[,"n_42"])*(log(pmax(1e-6,r_cm2c)) + log(18 - 20*r_cm2c + 15*r_cm2c^2 - 4*r_cm2c^3)) + (x[,"n_11"] + x[,"n_34"])*log(33 - 20*r_cm2c - 39*r_cm2c^2 + 50*r_cm2c^3 - 24*r_cm2c^4) + (x[,"n_22"] + x[,"n_23"])*log(34 - r_cm2c - 7*r_cm2c^2 + 16*r_cm2c^3 - 8*r_cm2c^4) + (x[,"n_02"] + x[,"n_43"])*log(9 - 7*r_cm2c + r_cm2c^2 + r_cm2c^3 - 4*r_cm2c^4) + (x[,"n_13"] + x[,"n_32"])*log(42 + 16*r_cm2c + 41*r_cm2c^2 - 46*r_cm2c^3 + 16*r_cm2c^4) + (x[,"n_12"] + x[,"n_33"])*log(69 - 24*r_cm2c - r_cm2c^2 - 18*r_cm2c^3 + 16*r_cm2c^4) + (x[,"n_21"] + x[,"n_24"])*log(33 - 11*r_cm2c + 47*r_cm2c^2 - 72*r_cm2c^3 + 36*r_cm2c^4)
logL_cm1c <- function(r,n00,n01,n02,n03,n04,n05,n11,n12,n13,n14,n20,n21,n22,n23,n24,n25,n31,n32,n33,n34,n40,n41,n42,n43,n44,n45,n10,n15,n30,n35) {
L <- (-2*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 2*n05 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n00 + n05 + n40 + n45)*(2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n44)*(log(pmax(1e-6,r)) + log(8 - 13*r + 11*r^2 - 6*r^3)) + (n04 + n41)*(log(pmax(1e-6,r)) + log(9 - 16*r + 13*r^2 - 6*r^3)) + (n10 + n15 + n30 + n35)*(log(pmax(1e-6,r)) + log(1 - 2*r + 2*r^2 - r^3)) + (n21 + n24)*log(33 + 7*r - 43*r^2 + 72*r^3 - 36*r^4) + (n12 + n33)*log(60 - 8*r - 11*r^2 + 26*r^3 - 16*r^4) + (n13 + n32)*log(51 + 16*r - 29*r^2 + 38*r^3 - 16*r^4) + (n03 + n42)*log(9 - 13*r + 17*r^2 - 11*r^3 + 4*r^4) + (n02 + n43)*log(6 - 4*r + 8*r^2 - 5*r^3 + 4*r^4) + (n20 + n25)*log(3 - 8*r + 14*r^2 - 12*r^3 + 6*r^4) + (n22 + n23)*log(32 + 3*r + 5*r^2 - 16*r^3 + 8*r^4) + (n14 + n31)*log(15 - 4*r + 39*r^2 - 50*r^3 + 24*r^4) + (n11 + n34)*log(24 - 20*r + 33*r^2 - 46*r^3 + 24*r^4)
return(L)}
interlogL_cm1c <- function(n00,n01,n02,n03,n04,n05,n11,n12,n13,n14,n20,n21,n22,n23,n24,n25,n31,n32,n33,n34,n40,n41,n42,n43,n44,n45,n10,n15,n30,n35) {
optimize(logL_cm1c,c(0,0.5), n00,n01,n02,n03,n04,n05,n11,n12,n13,n14,n20,n21,n22,n23,n24,n25,n31,n32,n33,n34,n40,n41,n42,n43,n44,n45,n10,n15,n30,n35, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm1c <- mapply(interlogL_cm1c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_10"],x[,"n_15"],x[,"n_30"],x[,"n_35"])
LOD_cm1c <- 4*(x[,"n_00"] + x[,"n_05"] + x[,"n_40"] + x[,"n_45"])*log10(2) - (x[,"n_10"] + x[,"n_15"] + x[,"n_30"] + x[,"n_35"])*(-4*log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_31"])*(log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_34"])*(log10(2) + 2*log10(3)) - (x[,"n_03"] + x[,"n_42"])*(-3*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_43"])*(-3*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_04"] + x[,"n_41"])*(-2*log10(2) + log10(7)) - (x[,"n_01"] + x[,"n_44"])*(-2*log10(2) + log10(7)) - (x[,"n_20"] + x[,"n_25"])*(-3*log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_24"])*(-log10(2) + log10(5) + log10(13)) - (x[,"n_22"] + x[,"n_23"])*(-2*log10(2) + log10(7) + log10(19)) - (x[,"n_13"] + x[,"n_32"])*(-log10(2) + log10(3) + log10(37)) - (x[,"n_12"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(37)) + (x[,"n_00"] + x[,"n_05"] + x[,"n_40"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_cm1c)) + 2*log10(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_44"])*(log10(pmax(1e-6,r_cm1c)) + log10(8 - 13*r_cm1c + 11*r_cm1c^2 - 6*r_cm1c^3)) + (x[,"n_04"] + x[,"n_41"])*(log10(pmax(1e-6,r_cm1c)) + log10(9 - 16*r_cm1c + 13*r_cm1c^2 - 6*r_cm1c^3)) + (x[,"n_10"] + x[,"n_15"] + x[,"n_30"] + x[,"n_35"])*(log10(pmax(1e-6,r_cm1c)) + log10(1 - 2*r_cm1c + 2*r_cm1c^2 - r_cm1c^3)) + (x[,"n_21"] + x[,"n_24"])*log10(33 + 7*r_cm1c - 43*r_cm1c^2 + 72*r_cm1c^3 - 36*r_cm1c^4) + (x[,"n_12"] + x[,"n_33"])*log10(60 - 8*r_cm1c - 11*r_cm1c^2 + 26*r_cm1c^3 - 16*r_cm1c^4) + (x[,"n_13"] + x[,"n_32"])*log10(51 + 16*r_cm1c - 29*r_cm1c^2 + 38*r_cm1c^3 - 16*r_cm1c^4) + (x[,"n_03"] + x[,"n_42"])*log10(9 - 13*r_cm1c + 17*r_cm1c^2 - 11*r_cm1c^3 + 4*r_cm1c^4) + (x[,"n_02"] + x[,"n_43"])*log10(6 - 4*r_cm1c + 8*r_cm1c^2 - 5*r_cm1c^3 + 4*r_cm1c^4) + (x[,"n_20"] + x[,"n_25"])*log10(3 - 8*r_cm1c + 14*r_cm1c^2 - 12*r_cm1c^3 + 6*r_cm1c^4) + (x[,"n_22"] + x[,"n_23"])*log10(32 + 3*r_cm1c + 5*r_cm1c^2 - 16*r_cm1c^3 + 8*r_cm1c^4) + (x[,"n_14"] + x[,"n_31"])*log10(15 - 4*r_cm1c + 39*r_cm1c^2 - 50*r_cm1c^3 + 24*r_cm1c^4) + (x[,"n_11"] + x[,"n_34"])*log10(24 - 20*r_cm1c + 33*r_cm1c^2 - 46*r_cm1c^3 + 24*r_cm1c^4)
logL_cm1c <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_00"] + x[,"n_05"] + x[,"n_40"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_cm1c)) + 2*log(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_44"])*(log(pmax(1e-6,r_cm1c)) + log(8 - 13*r_cm1c + 11*r_cm1c^2 - 6*r_cm1c^3)) + (x[,"n_04"] + x[,"n_41"])*(log(pmax(1e-6,r_cm1c)) + log(9 - 16*r_cm1c + 13*r_cm1c^2 - 6*r_cm1c^3)) + (x[,"n_10"] + x[,"n_15"] + x[,"n_30"] + x[,"n_35"])*(log(pmax(1e-6,r_cm1c)) + log(1 - 2*r_cm1c + 2*r_cm1c^2 - r_cm1c^3)) + (x[,"n_21"] + x[,"n_24"])*log(33 + 7*r_cm1c - 43*r_cm1c^2 + 72*r_cm1c^3 - 36*r_cm1c^4) + (x[,"n_12"] + x[,"n_33"])*log(60 - 8*r_cm1c - 11*r_cm1c^2 + 26*r_cm1c^3 - 16*r_cm1c^4) + (x[,"n_13"] + x[,"n_32"])*log(51 + 16*r_cm1c - 29*r_cm1c^2 + 38*r_cm1c^3 - 16*r_cm1c^4) + (x[,"n_03"] + x[,"n_42"])*log(9 - 13*r_cm1c + 17*r_cm1c^2 - 11*r_cm1c^3 + 4*r_cm1c^4) + (x[,"n_02"] + x[,"n_43"])*log(6 - 4*r_cm1c + 8*r_cm1c^2 - 5*r_cm1c^3 + 4*r_cm1c^4) + (x[,"n_20"] + x[,"n_25"])*log(3 - 8*r_cm1c + 14*r_cm1c^2 - 12*r_cm1c^3 + 6*r_cm1c^4) + (x[,"n_22"] + x[,"n_23"])*log(32 + 3*r_cm1c + 5*r_cm1c^2 - 16*r_cm1c^3 + 8*r_cm1c^4) + (x[,"n_14"] + x[,"n_31"])*log(15 - 4*r_cm1c + 39*r_cm1c^2 - 50*r_cm1c^3 + 24*r_cm1c^4) + (x[,"n_11"] + x[,"n_34"])*log(24 - 20*r_cm1c + 33*r_cm1c^2 - 46*r_cm1c^3 + 24*r_cm1c^4)
logL_cr <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n00 - 3*n01 - 3*n02 - 3*n03 - 3*n04 - 2*n05 - 2*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + (n01 + n04 + n11 + n14 + n20 + n21 + n22 + n23 + n24 + n25 + n31 + n34 + n41 + n44)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n05 + n40)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n45)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n01 + n44)*(2*log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n04 + n41)*(2*log(pmax(1e-6,1 - r)) + log(1 - r + 2*r^2)) + (n10 + n35)*(2*log(pmax(1e-6,r)) + log(3 - 6*r + 4*r^2)) + (n15 + n30)*(2*log(pmax(1e-6,1 - r)) + log(1 - 2*r + 4*r^2)) + (n11 + n34)*(log(pmax(1e-6,r)) + log(8 - 13*r + 14*r^2 - 8*r^3)) + (n02 + n43)*(log(pmax(1e-6,r)) + log(6 - 9*r + 9*r^2 - 4*r^3)) + (n20 + n25)*(log(pmax(1e-6,r)) + log(1 - 3*r + 4*r^2 - 2*r^3)) + (n22 + n23)*log(9 + 11*r - 19*r^2 + 16*r^3 - 8*r^4) + (n14 + n31)*log(1 + 8*r - 19*r^2 + 18*r^3 - 8*r^4) + (n03 + n42)*log(2 + r - 6*r^2 + 7*r^3 - 4*r^4) + (n21 + n24)*log(6 - 13*r + 25*r^2 - 24*r^3 + 12*r^4) + (n12 + n33)*log(18 - 12*r + 39*r^2 - 34*r^3 + 16*r^4) + (n13 + n32)*log(27 - 28*r + 33*r^2 - 30*r^3 + 16*r^4)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cr <- 3*(x[,"n_20"] + x[,"n_25"])*log10(2) + 2*(x[,"n_15"] + x[,"n_30"])*log10(2) - (x[,"n_14"] + x[,"n_31"])*log10(2) - (x[,"n_11"] + x[,"n_34"])*log10(2) + 2*(x[,"n_10"] + x[,"n_35"])*log10(2) + 4*(x[,"n_05"] + x[,"n_40"])*log10(2) + 2*(x[,"n_04"] + x[,"n_41"])*log10(2) + 2*(x[,"n_01"] + x[,"n_44"])*log10(2) + 4*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_22"] + x[,"n_23"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_21"] + x[,"n_24"])*(-log10(2) + log10(7)) - (x[,"n_03"] + x[,"n_42"])*(-3*log10(2) + log10(13)) - (x[,"n_02"] + x[,"n_43"])*(-3*log10(2) + log10(13)) - (x[,"n_13"] + x[,"n_32"])*(-log10(2) + log10(37)) - (x[,"n_12"] + x[,"n_33"])*(-log10(2) + log10(37)) + (x[,"n_05"] + x[,"n_40"])*(3*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cr)) + 3*log10(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_44"])*(2*log10(pmax(1e-6,r_cr)) + log10(2 - 3*r_cr + 2*r_cr^2)) + (x[,"n_04"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(1 - r_cr + 2*r_cr^2)) + (x[,"n_10"] + x[,"n_35"])*(2*log10(pmax(1e-6,r_cr)) + log10(3 - 6*r_cr + 4*r_cr^2)) + (x[,"n_15"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(1 - 2*r_cr + 4*r_cr^2)) + (x[,"n_11"] + x[,"n_34"])*(log10(pmax(1e-6,r_cr)) + log10(8 - 13*r_cr + 14*r_cr^2 - 8*r_cr^3)) + (x[,"n_02"] + x[,"n_43"])*(log10(pmax(1e-6,r_cr)) + log10(6 - 9*r_cr + 9*r_cr^2 - 4*r_cr^3)) + (x[,"n_20"] + x[,"n_25"])*(log10(pmax(1e-6,r_cr)) + log10(1 - 3*r_cr + 4*r_cr^2 - 2*r_cr^3)) + (x[,"n_22"] + x[,"n_23"])*log10(9 + 11*r_cr - 19*r_cr^2 + 16*r_cr^3 - 8*r_cr^4) + (x[,"n_14"] + x[,"n_31"])*log10(1 + 8*r_cr - 19*r_cr^2 + 18*r_cr^3 - 8*r_cr^4) + (x[,"n_03"] + x[,"n_42"])*log10(2 + r_cr - 6*r_cr^2 + 7*r_cr^3 - 4*r_cr^4) + (x[,"n_21"] + x[,"n_24"])*log10(6 - 13*r_cr + 25*r_cr^2 - 24*r_cr^3 + 12*r_cr^4) + (x[,"n_12"] + x[,"n_33"])*log10(18 - 12*r_cr + 39*r_cr^2 - 34*r_cr^3 + 16*r_cr^4) + (x[,"n_13"] + x[,"n_32"])*log10(27 - 28*r_cr + 33*r_cr^2 - 30*r_cr^3 + 16*r_cr^4)
logL_cr <- (-2*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_25"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_05"] + x[,"n_40"])*(3*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cr)) + 3*log(pmax(1e-6,r_cr))) + (x[,"n_01"] + x[,"n_44"])*(2*log(pmax(1e-6,r_cr)) + log(2 - 3*r_cr + 2*r_cr^2)) + (x[,"n_04"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_cr)) + log(1 - r_cr + 2*r_cr^2)) + (x[,"n_10"] + x[,"n_35"])*(2*log(pmax(1e-6,r_cr)) + log(3 - 6*r_cr + 4*r_cr^2)) + (x[,"n_15"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_cr)) + log(1 - 2*r_cr + 4*r_cr^2)) + (x[,"n_11"] + x[,"n_34"])*(log(pmax(1e-6,r_cr)) + log(8 - 13*r_cr + 14*r_cr^2 - 8*r_cr^3)) + (x[,"n_02"] + x[,"n_43"])*(log(pmax(1e-6,r_cr)) + log(6 - 9*r_cr + 9*r_cr^2 - 4*r_cr^3)) + (x[,"n_20"] + x[,"n_25"])*(log(pmax(1e-6,r_cr)) + log(1 - 3*r_cr + 4*r_cr^2 - 2*r_cr^3)) + (x[,"n_22"] + x[,"n_23"])*log(9 + 11*r_cr - 19*r_cr^2 + 16*r_cr^3 - 8*r_cr^4) + (x[,"n_14"] + x[,"n_31"])*log(1 + 8*r_cr - 19*r_cr^2 + 18*r_cr^3 - 8*r_cr^4) + (x[,"n_03"] + x[,"n_42"])*log(2 + r_cr - 6*r_cr^2 + 7*r_cr^3 - 4*r_cr^4) + (x[,"n_21"] + x[,"n_24"])*log(6 - 13*r_cr + 25*r_cr^2 - 24*r_cr^3 + 12*r_cr^4) + (x[,"n_12"] + x[,"n_33"])*log(18 - 12*r_cr + 39*r_cr^2 - 34*r_cr^3 + 16*r_cr^4) + (x[,"n_13"] + x[,"n_32"])*log(27 - 28*r_cr + 33*r_cr^2 - 30*r_cr^3 + 16*r_cr^4)
logL_rc <- function(r,n01,n04,n11,n14,n20,n21,n22,n23,n24,n25,n31,n34,n41,n44,n00,n02,n03,n05,n10,n12,n13,n15,n30,n32,n33,n35,n40,n42,n43,n45) {
L <- (n01 + n04 + n11 + n14 + n20 + n21 + n22 + n23 + n24 + n25 + n31 + n34 + n41 + n44)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n05 + n40)*(log(3 - 2*r) + 3*log(pmax(1e-6,r))) + (n00 + n45)*(3*log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n04 + n41)*(2*log(pmax(1e-6,r)) + log(3 - 3*r + 2*r^2)) + (n01 + n44)*(2*log(pmax(1e-6,1 - r)) + log(2 - r + 2*r^2)) + (n15 + n30)*(2*log(pmax(1e-6,r)) + log(9 - 14*r + 8*r^2)) + (n10 + n35)*(2*log(pmax(1e-6,1 - r)) + log(3 - 2*r + 8*r^2)) + (n14 + n31)*(log(pmax(1e-6,r)) + log(12 - 13*r + 14*r^2 - 8*r^3)) + (n03 + n42)*(log(pmax(1e-6,r)) + log(9 - 6*r + 4*r^2 - 4*r^3)) + (n20 + n25)*(log(pmax(1e-6,r)) + log(3 - 7*r + 8*r^2 - 4*r^3)) + (n22 + n23)*log(21 + 8*r - 16*r^2 + 16*r^3 - 8*r^4) + (n11 + n34)*log(5 + 4*r - 19*r^2 + 18*r^3 - 8*r^4) + (n02 + n43)*log(3 + 7*r - 18*r^2 + 12*r^3 - 4*r^4) + (n21 + n24)*log(9 - 11*r + 23*r^2 - 24*r^3 + 12*r^4) + (n12 + n33)*log(55 - 52*r + 48*r^2 - 40*r^3 + 16*r^4) + (n13 + n32)*log(27 + 12*r + 24*r^2 - 24*r^3 + 16*r^4)
return(L)}
interlogL_rc <- function(n01,n04,n11,n14,n20,n21,n22,n23,n24,n25,n31,n34,n41,n44,n00,n02,n03,n05,n10,n12,n13,n15,n30,n32,n33,n35,n40,n42,n43,n45) {
optimize(logL_rc,c(0,0.5), n01,n04,n11,n14,n20,n21,n22,n23,n24,n25,n31,n34,n41,n44,n00,n02,n03,n05,n10,n12,n13,n15,n30,n32,n33,n35,n40,n42,n43,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_01"],x[,"n_04"],x[,"n_11"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_34"],x[,"n_41"],x[,"n_44"],x[,"n_00"],x[,"n_02"],x[,"n_03"],x[,"n_05"],x[,"n_10"],x[,"n_12"],x[,"n_13"],x[,"n_15"],x[,"n_30"],x[,"n_32"],x[,"n_33"],x[,"n_35"],x[,"n_40"],x[,"n_42"],x[,"n_43"],x[,"n_45"])
LOD_rc <- (x[,"n_20"] + x[,"n_25"])*log10(2) - 2*(x[,"n_14"] + x[,"n_31"])*log10(2) - 2*(x[,"n_11"] + x[,"n_34"])*log10(2) + 2*(x[,"n_05"] + x[,"n_40"])*log10(2) + (x[,"n_04"] + x[,"n_41"])*log10(2) + (x[,"n_01"] + x[,"n_44"])*log10(2) + 2*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_22"] + x[,"n_23"])*(-log10(2) + 2*log10(3) + log10(5)) - (x[,"n_21"] + x[,"n_24"])*log10(7) - (x[,"n_03"] + x[,"n_42"])*(-2*log10(2) + log10(13)) - (x[,"n_02"] + x[,"n_43"])*(-2*log10(2) + log10(13)) - (x[,"n_13"] + x[,"n_32"])*log10(37) - (x[,"n_12"] + x[,"n_33"])*log10(37) + (x[,"n_05"] + x[,"n_40"])*(log10(3 - 2*r_rc) + 3*log10(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_45"])*(3*log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_04"] + x[,"n_41"])*(2*log10(pmax(1e-6,r_rc)) + log10(3 - 3*r_rc + 2*r_rc^2)) + (x[,"n_01"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(2 - r_rc + 2*r_rc^2)) + (x[,"n_15"] + x[,"n_30"])*(2*log10(pmax(1e-6,r_rc)) + log10(9 - 14*r_rc + 8*r_rc^2)) + (x[,"n_10"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(3 - 2*r_rc + 8*r_rc^2)) + (x[,"n_14"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(12 - 13*r_rc + 14*r_rc^2 - 8*r_rc^3)) + (x[,"n_03"] + x[,"n_42"])*(log10(pmax(1e-6,r_rc)) + log10(9 - 6*r_rc + 4*r_rc^2 - 4*r_rc^3)) + (x[,"n_20"] + x[,"n_25"])*(log10(pmax(1e-6,r_rc)) + log10(3 - 7*r_rc + 8*r_rc^2 - 4*r_rc^3)) + (x[,"n_22"] + x[,"n_23"])*log10(21 + 8*r_rc - 16*r_rc^2 + 16*r_rc^3 - 8*r_rc^4) + (x[,"n_11"] + x[,"n_34"])*log10(5 + 4*r_rc - 19*r_rc^2 + 18*r_rc^3 - 8*r_rc^4) + (x[,"n_02"] + x[,"n_43"])*log10(3 + 7*r_rc - 18*r_rc^2 + 12*r_rc^3 - 4*r_rc^4) + (x[,"n_21"] + x[,"n_24"])*log10(9 - 11*r_rc + 23*r_rc^2 - 24*r_rc^3 + 12*r_rc^4) + (x[,"n_12"] + x[,"n_33"])*log10(55 - 52*r_rc + 48*r_rc^2 - 40*r_rc^3 + 16*r_rc^4) + (x[,"n_13"] + x[,"n_32"])*log10(27 + 12*r_rc + 24*r_rc^2 - 24*r_rc^3 + 16*r_rc^4)
logL_rc <- (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_25"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_05"] + x[,"n_40"])*(log(3 - 2*r_rc) + 3*log(pmax(1e-6,r_rc))) + (x[,"n_00"] + x[,"n_45"])*(3*log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_04"] + x[,"n_41"])*(2*log(pmax(1e-6,r_rc)) + log(3 - 3*r_rc + 2*r_rc^2)) + (x[,"n_01"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_rc)) + log(2 - r_rc + 2*r_rc^2)) + (x[,"n_15"] + x[,"n_30"])*(2*log(pmax(1e-6,r_rc)) + log(9 - 14*r_rc + 8*r_rc^2)) + (x[,"n_10"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_rc)) + log(3 - 2*r_rc + 8*r_rc^2)) + (x[,"n_14"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(12 - 13*r_rc + 14*r_rc^2 - 8*r_rc^3)) + (x[,"n_03"] + x[,"n_42"])*(log(pmax(1e-6,r_rc)) + log(9 - 6*r_rc + 4*r_rc^2 - 4*r_rc^3)) + (x[,"n_20"] + x[,"n_25"])*(log(pmax(1e-6,r_rc)) + log(3 - 7*r_rc + 8*r_rc^2 - 4*r_rc^3)) + (x[,"n_22"] + x[,"n_23"])*log(21 + 8*r_rc - 16*r_rc^2 + 16*r_rc^3 - 8*r_rc^4) + (x[,"n_11"] + x[,"n_34"])*log(5 + 4*r_rc - 19*r_rc^2 + 18*r_rc^3 - 8*r_rc^4) + (x[,"n_02"] + x[,"n_43"])*log(3 + 7*r_rc - 18*r_rc^2 + 12*r_rc^3 - 4*r_rc^4) + (x[,"n_21"] + x[,"n_24"])*log(9 - 11*r_rc + 23*r_rc^2 - 24*r_rc^3 + 12*r_rc^4) + (x[,"n_12"] + x[,"n_33"])*log(55 - 52*r_rc + 48*r_rc^2 - 40*r_rc^3 + 16*r_rc^4) + (x[,"n_13"] + x[,"n_32"])*log(27 + 12*r_rc + 24*r_rc^2 - 24*r_rc^3 + 16*r_rc^4)
logL_rm2c <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n05 + n40)*(log(3 - 2*r) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n45)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(1 + 2*r)) + (n15 + n30)*(log(pmax(1e-6,r)) + log(12 - 19*r + 18*r^2 - 8*r^3)) + (n04 + n41)*(log(pmax(1e-6,r)) + log(12 - 14*r + 11*r^2 - 6*r^3)) + (n21 + n24)*log(66 + 5*r - 41*r^2 + 72*r^3 - 36*r^4) + (n13 + n32)*log(111 - 4*r + 4*r^2 + 16*r^3 - 16*r^4) + (n12 + n33)*log(111 + 12*r - 44*r^2 + 48*r^3 - 16*r^4) + (n10 + n35)*log(3 + 4*r - 13*r^2 + 14*r^3 - 8*r^4) + (n01 + n44)*log(3 + 7*r - 17*r^2 + 13*r^3 - 6*r^4) + (n03 + n42)*log(9 + 3*r + 2*r^2 + 4*r^4) + (n02 + n43)*log(18 - 23*r + 26*r^2 - 16*r^3 + 4*r^4) + (n22 + n23)*log(65 + 4*r + 4*r^2 - 16*r^3 + 8*r^4) + (n20 + n25)*log(9 - 17*r + 29*r^2 - 24*r^3 + 12*r^4) + (n11 + n34)*log(39 - 16*r + 39*r^2 - 50*r^3 + 24*r^4) + (n14 + n31)*log(36 - 8*r + 33*r^2 - 46*r^3 + 24*r^4)
return(L)}
interlogL_rm2c <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23) {
optimize(logL_rm2c,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_22"],x[,"n_23"])
LOD_rm2c <- 2*(x[,"n_05"] + x[,"n_40"])*log10(2) + 2*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_15"] + x[,"n_30"])*log10(3) - (x[,"n_10"] + x[,"n_35"])*log10(3) - (x[,"n_14"] + x[,"n_31"])*(2*log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_34"])*(2*log10(2) + 2*log10(3)) - (x[,"n_03"] + x[,"n_42"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_43"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_04"] + x[,"n_41"])*(-log10(2) + log10(7)) - (x[,"n_01"] + x[,"n_44"])*(-log10(2) + log10(7)) - (x[,"n_20"] + x[,"n_25"])*(-log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_24"])*(log10(5) + log10(13)) - (x[,"n_22"] + x[,"n_23"])*(-log10(2) + log10(7) + log10(19)) - (x[,"n_13"] + x[,"n_32"])*(log10(3) + log10(37)) - (x[,"n_12"] + x[,"n_33"])*(log10(3) + log10(37)) + (x[,"n_05"] + x[,"n_40"])*(log10(3 - 2*r_rm2c) + log10(pmax(1e-6,1 - r_rm2c)) + 2*log10(pmax(1e-6,r_rm2c))) + (x[,"n_00"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_rm2c)) + log10(pmax(1e-6,r_rm2c)) + log10(1 + 2*r_rm2c)) + (x[,"n_15"] + x[,"n_30"])*(log10(pmax(1e-6,r_rm2c)) + log10(12 - 19*r_rm2c + 18*r_rm2c^2 - 8*r_rm2c^3)) + (x[,"n_04"] + x[,"n_41"])*(log10(pmax(1e-6,r_rm2c)) + log10(12 - 14*r_rm2c + 11*r_rm2c^2 - 6*r_rm2c^3)) + (x[,"n_21"] + x[,"n_24"])*log10(66 + 5*r_rm2c - 41*r_rm2c^2 + 72*r_rm2c^3 - 36*r_rm2c^4) + (x[,"n_13"] + x[,"n_32"])*log10(111 - 4*r_rm2c + 4*r_rm2c^2 + 16*r_rm2c^3 - 16*r_rm2c^4) + (x[,"n_12"] + x[,"n_33"])*log10(111 + 12*r_rm2c - 44*r_rm2c^2 + 48*r_rm2c^3 - 16*r_rm2c^4) + (x[,"n_10"] + x[,"n_35"])*log10(3 + 4*r_rm2c - 13*r_rm2c^2 + 14*r_rm2c^3 - 8*r_rm2c^4) + (x[,"n_01"] + x[,"n_44"])*log10(3 + 7*r_rm2c - 17*r_rm2c^2 + 13*r_rm2c^3 - 6*r_rm2c^4) + (x[,"n_03"] + x[,"n_42"])*log10(9 + 3*r_rm2c + 2*r_rm2c^2 + 4*r_rm2c^4) + (x[,"n_02"] + x[,"n_43"])*log10(18 - 23*r_rm2c + 26*r_rm2c^2 - 16*r_rm2c^3 + 4*r_rm2c^4) + (x[,"n_22"] + x[,"n_23"])*log10(65 + 4*r_rm2c + 4*r_rm2c^2 - 16*r_rm2c^3 + 8*r_rm2c^4) + (x[,"n_20"] + x[,"n_25"])*log10(9 - 17*r_rm2c + 29*r_rm2c^2 - 24*r_rm2c^3 + 12*r_rm2c^4) + (x[,"n_11"] + x[,"n_34"])*log10(39 - 16*r_rm2c + 39*r_rm2c^2 - 50*r_rm2c^3 + 24*r_rm2c^4) + (x[,"n_14"] + x[,"n_31"])*log10(36 - 8*r_rm2c + 33*r_rm2c^2 - 46*r_rm2c^3 + 24*r_rm2c^4)
logL_rm2c <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_05"] + x[,"n_40"])*(log(3 - 2*r_rm2c) + log(pmax(1e-6,1 - r_rm2c)) + 2*log(pmax(1e-6,r_rm2c))) + (x[,"n_00"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_rm2c)) + log(pmax(1e-6,r_rm2c)) + log(1 + 2*r_rm2c)) + (x[,"n_15"] + x[,"n_30"])*(log(pmax(1e-6,r_rm2c)) + log(12 - 19*r_rm2c + 18*r_rm2c^2 - 8*r_rm2c^3)) + (x[,"n_04"] + x[,"n_41"])*(log(pmax(1e-6,r_rm2c)) + log(12 - 14*r_rm2c + 11*r_rm2c^2 - 6*r_rm2c^3)) + (x[,"n_21"] + x[,"n_24"])*log(66 + 5*r_rm2c - 41*r_rm2c^2 + 72*r_rm2c^3 - 36*r_rm2c^4) + (x[,"n_13"] + x[,"n_32"])*log(111 - 4*r_rm2c + 4*r_rm2c^2 + 16*r_rm2c^3 - 16*r_rm2c^4) + (x[,"n_12"] + x[,"n_33"])*log(111 + 12*r_rm2c - 44*r_rm2c^2 + 48*r_rm2c^3 - 16*r_rm2c^4) + (x[,"n_10"] + x[,"n_35"])*log(3 + 4*r_rm2c - 13*r_rm2c^2 + 14*r_rm2c^3 - 8*r_rm2c^4) + (x[,"n_01"] + x[,"n_44"])*log(3 + 7*r_rm2c - 17*r_rm2c^2 + 13*r_rm2c^3 - 6*r_rm2c^4) + (x[,"n_03"] + x[,"n_42"])*log(9 + 3*r_rm2c + 2*r_rm2c^2 + 4*r_rm2c^4) + (x[,"n_02"] + x[,"n_43"])*log(18 - 23*r_rm2c + 26*r_rm2c^2 - 16*r_rm2c^3 + 4*r_rm2c^4) + (x[,"n_22"] + x[,"n_23"])*log(65 + 4*r_rm2c + 4*r_rm2c^2 - 16*r_rm2c^3 + 8*r_rm2c^4) + (x[,"n_20"] + x[,"n_25"])*log(9 - 17*r_rm2c + 29*r_rm2c^2 - 24*r_rm2c^3 + 12*r_rm2c^4) + (x[,"n_11"] + x[,"n_34"])*log(39 - 16*r_rm2c + 39*r_rm2c^2 - 50*r_rm2c^3 + 24*r_rm2c^4) + (x[,"n_14"] + x[,"n_31"])*log(36 - 8*r_rm2c + 33*r_rm2c^2 - 46*r_rm2c^3 + 24*r_rm2c^4)
logL_rm1c <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n05 + n40)*(log(3 - 2*r) + 2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n45)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r)) + log(1 + 2*r)) + (n01 + n44)*(log(pmax(1e-6,r)) + log(4 + 10*r - 11*r^2 + 6*r^3)) + (n10 + n35)*(log(pmax(1e-6,r)) + log(4 + 7*r - 10*r^2 + 8*r^3)) + (n11 + n34)*log(12 + 56*r - 33*r^2 + 46*r^3 - 24*r^4) + (n14 + n31)*log(57 - 32*r - 39*r^2 + 50*r^3 - 24*r^4) + (n20 + n25)*log(3 + 13*r - 25*r^2 + 24*r^3 - 12*r^4) + (n22 + n23)*log(67 - 8*r^2 + 16*r^3 - 8*r^4) + (n03 + n42)*log(18 - 17*r + 10*r^2 - 4*r^3 - 4*r^4) + (n02 + n43)*log(3 + 25*r - 26*r^2 + 20*r^3 - 4*r^4) + (n04 + n41)*log(9 - 15*r + 13*r^2 - 13*r^3 + 6*r^4) + (n15 + n30)*log(9 - 20*r + 25*r^2 - 22*r^3 + 8*r^4) + (n12 + n33)*log(93 + 20*r + 56*r^2 - 56*r^3 + 16*r^4) + (n13 + n32)*log(129 - 28*r - 16*r^2 - 8*r^3 + 16*r^4) + (n21 + n24)*log(66 - 13*r + 49*r^2 - 72*r^3 + 36*r^4)
return(L)}
interlogL_rm1c <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23) {
optimize(logL_rm1c,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm1c <- mapply(interlogL_rm1c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_22"],x[,"n_23"])
LOD_rm1c <- 2*(x[,"n_05"] + x[,"n_40"])*log10(2) + 2*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_15"] + x[,"n_30"])*log10(3) - (x[,"n_10"] + x[,"n_35"])*log10(3) - (x[,"n_14"] + x[,"n_31"])*(2*log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_34"])*(2*log10(2) + 2*log10(3)) - (x[,"n_03"] + x[,"n_42"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_43"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_04"] + x[,"n_41"])*(-log10(2) + log10(7)) - (x[,"n_01"] + x[,"n_44"])*(-log10(2) + log10(7)) - (x[,"n_20"] + x[,"n_25"])*(-log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_24"])*(log10(5) + log10(13)) - (x[,"n_22"] + x[,"n_23"])*(-log10(2) + log10(7) + log10(19)) - (x[,"n_13"] + x[,"n_32"])*(log10(3) + log10(37)) - (x[,"n_12"] + x[,"n_33"])*(log10(3) + log10(37)) + (x[,"n_05"] + x[,"n_40"])*(log10(3 - 2*r_rm1c) + 2*log10(pmax(1e-6,1 - r_rm1c)) + log10(pmax(1e-6,r_rm1c))) + (x[,"n_00"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_rm1c)) + 2*log10(pmax(1e-6,r_rm1c)) + log10(1 + 2*r_rm1c)) + (x[,"n_01"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm1c)) + log10(4 + 10*r_rm1c - 11*r_rm1c^2 + 6*r_rm1c^3)) + (x[,"n_10"] + x[,"n_35"])*(log10(pmax(1e-6,r_rm1c)) + log10(4 + 7*r_rm1c - 10*r_rm1c^2 + 8*r_rm1c^3)) + (x[,"n_11"] + x[,"n_34"])*log10(12 + 56*r_rm1c - 33*r_rm1c^2 + 46*r_rm1c^3 - 24*r_rm1c^4) + (x[,"n_14"] + x[,"n_31"])*log10(57 - 32*r_rm1c - 39*r_rm1c^2 + 50*r_rm1c^3 - 24*r_rm1c^4) + (x[,"n_20"] + x[,"n_25"])*log10(3 + 13*r_rm1c - 25*r_rm1c^2 + 24*r_rm1c^3 - 12*r_rm1c^4) + (x[,"n_22"] + x[,"n_23"])*log10(67 - 8*r_rm1c^2 + 16*r_rm1c^3 - 8*r_rm1c^4) + (x[,"n_03"] + x[,"n_42"])*log10(18 - 17*r_rm1c + 10*r_rm1c^2 - 4*r_rm1c^3 - 4*r_rm1c^4) + (x[,"n_02"] + x[,"n_43"])*log10(3 + 25*r_rm1c - 26*r_rm1c^2 + 20*r_rm1c^3 - 4*r_rm1c^4) + (x[,"n_04"] + x[,"n_41"])*log10(9 - 15*r_rm1c + 13*r_rm1c^2 - 13*r_rm1c^3 + 6*r_rm1c^4) + (x[,"n_15"] + x[,"n_30"])*log10(9 - 20*r_rm1c + 25*r_rm1c^2 - 22*r_rm1c^3 + 8*r_rm1c^4) + (x[,"n_12"] + x[,"n_33"])*log10(93 + 20*r_rm1c + 56*r_rm1c^2 - 56*r_rm1c^3 + 16*r_rm1c^4) + (x[,"n_13"] + x[,"n_32"])*log10(129 - 28*r_rm1c - 16*r_rm1c^2 - 8*r_rm1c^3 + 16*r_rm1c^4) + (x[,"n_21"] + x[,"n_24"])*log10(66 - 13*r_rm1c + 49*r_rm1c^2 - 72*r_rm1c^3 + 36*r_rm1c^4)
logL_rm1c <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_05"] + x[,"n_40"])*(log(3 - 2*r_rm1c) + 2*log(pmax(1e-6,1 - r_rm1c)) + log(pmax(1e-6,r_rm1c))) + (x[,"n_00"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_rm1c)) + 2*log(pmax(1e-6,r_rm1c)) + log(1 + 2*r_rm1c)) + (x[,"n_01"] + x[,"n_44"])*(log(pmax(1e-6,r_rm1c)) + log(4 + 10*r_rm1c - 11*r_rm1c^2 + 6*r_rm1c^3)) + (x[,"n_10"] + x[,"n_35"])*(log(pmax(1e-6,r_rm1c)) + log(4 + 7*r_rm1c - 10*r_rm1c^2 + 8*r_rm1c^3)) + (x[,"n_11"] + x[,"n_34"])*log(12 + 56*r_rm1c - 33*r_rm1c^2 + 46*r_rm1c^3 - 24*r_rm1c^4) + (x[,"n_14"] + x[,"n_31"])*log(57 - 32*r_rm1c - 39*r_rm1c^2 + 50*r_rm1c^3 - 24*r_rm1c^4) + (x[,"n_20"] + x[,"n_25"])*log(3 + 13*r_rm1c - 25*r_rm1c^2 + 24*r_rm1c^3 - 12*r_rm1c^4) + (x[,"n_22"] + x[,"n_23"])*log(67 - 8*r_rm1c^2 + 16*r_rm1c^3 - 8*r_rm1c^4) + (x[,"n_03"] + x[,"n_42"])*log(18 - 17*r_rm1c + 10*r_rm1c^2 - 4*r_rm1c^3 - 4*r_rm1c^4) + (x[,"n_02"] + x[,"n_43"])*log(3 + 25*r_rm1c - 26*r_rm1c^2 + 20*r_rm1c^3 - 4*r_rm1c^4) + (x[,"n_04"] + x[,"n_41"])*log(9 - 15*r_rm1c + 13*r_rm1c^2 - 13*r_rm1c^3 + 6*r_rm1c^4) + (x[,"n_15"] + x[,"n_30"])*log(9 - 20*r_rm1c + 25*r_rm1c^2 - 22*r_rm1c^3 + 8*r_rm1c^4) + (x[,"n_12"] + x[,"n_33"])*log(93 + 20*r_rm1c + 56*r_rm1c^2 - 56*r_rm1c^3 + 16*r_rm1c^4) + (x[,"n_13"] + x[,"n_32"])*log(129 - 28*r_rm1c - 16*r_rm1c^2 - 8*r_rm1c^3 + 16*r_rm1c^4) + (x[,"n_21"] + x[,"n_24"])*log(66 - 13*r_rm1c + 49*r_rm1c^2 - 72*r_rm1c^3 + 36*r_rm1c^4)
logL_rr <- function(r,n01,n04,n11,n14,n20,n21,n22,n23,n24,n25,n31,n34,n41,n44,n00,n02,n03,n05,n10,n12,n13,n15,n30,n32,n33,n35,n40,n42,n43,n45) {
L <- (n01 + n04 + n11 + n14 + n20 + n21 + n22 + n23 + n24 + n25 + n31 + n34 + n41 + n44)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n05 + n40)*(log(3 - 2*r) + 3*log(pmax(1e-6,1 - r))) + (n00 + n45)*(3*log(pmax(1e-6,r)) + log(1 + 2*r)) + (n10 + n35)*(2*log(pmax(1e-6,r)) + log(3 + 6*r - 8*r^2)) + (n15 + n30)*(2*log(pmax(1e-6,1 - r)) + log(1 + 10*r - 8*r^2)) + (n04 + n41)*(2*log(pmax(1e-6,1 - r)) + log(2 + r - 2*r^2)) + (n01 + n44)*(2*log(pmax(1e-6,r)) + log(1 + 3*r - 2*r^2)) + (n02 + n43)*(log(pmax(1e-6,r)) + log(3 + 18*r - 24*r^2 + 4*r^3)) + (n20 + n25)*(log(pmax(1e-6,r)) + log(1 + 3*r - 8*r^2 + 4*r^3)) + (n11 + n34)*(log(pmax(1e-6,r)) + log(4 + 13*r - 14*r^2 + 8*r^3)) + (n13 + n32)*log(57 - 44*r + 12*r^2 - 16*r^4) + (n12 + n33)*log(9 + 84*r - 84*r^2 + 64*r^3 - 16*r^4) + (n21 + n24)*log(3 + 19*r - 31*r^2 + 24*r^3 - 12*r^4) + (n03 + n42)*log(1 + 17*r - 30*r^2 + 8*r^3 + 4*r^4) + (n14 + n31)*log(11 - 20*r + 19*r^2 - 18*r^3 + 8*r^4) + (n22 + n23)*log(27 - 20*r + 28*r^2 - 16*r^3 + 8*r^4)
return(L)}
interlogL_rr <- function(n01,n04,n11,n14,n20,n21,n22,n23,n24,n25,n31,n34,n41,n44,n00,n02,n03,n05,n10,n12,n13,n15,n30,n32,n33,n35,n40,n42,n43,n45) {
optimize(logL_rr,c(0,0.5), n01,n04,n11,n14,n20,n21,n22,n23,n24,n25,n31,n34,n41,n44,n00,n02,n03,n05,n10,n12,n13,n15,n30,n32,n33,n35,n40,n42,n43,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_01"],x[,"n_04"],x[,"n_11"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_34"],x[,"n_41"],x[,"n_44"],x[,"n_00"],x[,"n_02"],x[,"n_03"],x[,"n_05"],x[,"n_10"],x[,"n_12"],x[,"n_13"],x[,"n_15"],x[,"n_30"],x[,"n_32"],x[,"n_33"],x[,"n_35"],x[,"n_40"],x[,"n_42"],x[,"n_43"],x[,"n_45"])
LOD_rr <- (x[,"n_20"] + x[,"n_25"])*log10(2) - 2*(x[,"n_14"] + x[,"n_31"])*log10(2) - 2*(x[,"n_11"] + x[,"n_34"])*log10(2) + 2*(x[,"n_05"] + x[,"n_40"])*log10(2) + (x[,"n_04"] + x[,"n_41"])*log10(2) + (x[,"n_01"] + x[,"n_44"])*log10(2) + 2*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_22"] + x[,"n_23"])*(-log10(2) + 2*log10(3) + log10(5)) - (x[,"n_21"] + x[,"n_24"])*log10(7) - (x[,"n_03"] + x[,"n_42"])*(-2*log10(2) + log10(13)) - (x[,"n_02"] + x[,"n_43"])*(-2*log10(2) + log10(13)) - (x[,"n_13"] + x[,"n_32"])*log10(37) - (x[,"n_12"] + x[,"n_33"])*log10(37) + (x[,"n_05"] + x[,"n_40"])*(log10(3 - 2*r_rr) + 3*log10(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_45"])*(3*log10(pmax(1e-6,r_rr)) + log10(1 + 2*r_rr)) + (x[,"n_10"] + x[,"n_35"])*(2*log10(pmax(1e-6,r_rr)) + log10(3 + 6*r_rr - 8*r_rr^2)) + (x[,"n_15"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(1 + 10*r_rr - 8*r_rr^2)) + (x[,"n_04"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(2 + r_rr - 2*r_rr^2)) + (x[,"n_01"] + x[,"n_44"])*(2*log10(pmax(1e-6,r_rr)) + log10(1 + 3*r_rr - 2*r_rr^2)) + (x[,"n_02"] + x[,"n_43"])*(log10(pmax(1e-6,r_rr)) + log10(3 + 18*r_rr - 24*r_rr^2 + 4*r_rr^3)) + (x[,"n_20"] + x[,"n_25"])*(log10(pmax(1e-6,r_rr)) + log10(1 + 3*r_rr - 8*r_rr^2 + 4*r_rr^3)) + (x[,"n_11"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(4 + 13*r_rr - 14*r_rr^2 + 8*r_rr^3)) + (x[,"n_13"] + x[,"n_32"])*log10(57 - 44*r_rr + 12*r_rr^2 - 16*r_rr^4) + (x[,"n_12"] + x[,"n_33"])*log10(9 + 84*r_rr - 84*r_rr^2 + 64*r_rr^3 - 16*r_rr^4) + (x[,"n_21"] + x[,"n_24"])*log10(3 + 19*r_rr - 31*r_rr^2 + 24*r_rr^3 - 12*r_rr^4) + (x[,"n_03"] + x[,"n_42"])*log10(1 + 17*r_rr - 30*r_rr^2 + 8*r_rr^3 + 4*r_rr^4) + (x[,"n_14"] + x[,"n_31"])*log10(11 - 20*r_rr + 19*r_rr^2 - 18*r_rr^3 + 8*r_rr^4) + (x[,"n_22"] + x[,"n_23"])*log10(27 - 20*r_rr + 28*r_rr^2 - 16*r_rr^3 + 8*r_rr^4)
logL_rr <- (x[,"n_01"] + x[,"n_04"] + x[,"n_11"] + x[,"n_14"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_25"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_05"] + x[,"n_40"])*(log(3 - 2*r_rr) + 3*log(pmax(1e-6,1 - r_rr))) + (x[,"n_00"] + x[,"n_45"])*(3*log(pmax(1e-6,r_rr)) + log(1 + 2*r_rr)) + (x[,"n_10"] + x[,"n_35"])*(2*log(pmax(1e-6,r_rr)) + log(3 + 6*r_rr - 8*r_rr^2)) + (x[,"n_15"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_rr)) + log(1 + 10*r_rr - 8*r_rr^2)) + (x[,"n_04"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_rr)) + log(2 + r_rr - 2*r_rr^2)) + (x[,"n_01"] + x[,"n_44"])*(2*log(pmax(1e-6,r_rr)) + log(1 + 3*r_rr - 2*r_rr^2)) + (x[,"n_02"] + x[,"n_43"])*(log(pmax(1e-6,r_rr)) + log(3 + 18*r_rr - 24*r_rr^2 + 4*r_rr^3)) + (x[,"n_20"] + x[,"n_25"])*(log(pmax(1e-6,r_rr)) + log(1 + 3*r_rr - 8*r_rr^2 + 4*r_rr^3)) + (x[,"n_11"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(4 + 13*r_rr - 14*r_rr^2 + 8*r_rr^3)) + (x[,"n_13"] + x[,"n_32"])*log(57 - 44*r_rr + 12*r_rr^2 - 16*r_rr^4) + (x[,"n_12"] + x[,"n_33"])*log(9 + 84*r_rr - 84*r_rr^2 + 64*r_rr^3 - 16*r_rr^4) + (x[,"n_21"] + x[,"n_24"])*log(3 + 19*r_rr - 31*r_rr^2 + 24*r_rr^3 - 12*r_rr^4) + (x[,"n_03"] + x[,"n_42"])*log(1 + 17*r_rr - 30*r_rr^2 + 8*r_rr^3 + 4*r_rr^4) + (x[,"n_14"] + x[,"n_31"])*log(11 - 20*r_rr + 19*r_rr^2 - 18*r_rr^3 + 8*r_rr^4) + (x[,"n_22"] + x[,"n_23"])*log(27 - 20*r_rr + 28*r_rr^2 - 16*r_rr^3 + 8*r_rr^4)

return(list(
r_mat = cbind(r_cc,r_cm2c,r_cm1c,r_cr,r_rc,r_rm2c,r_rm1c,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm2c,LOD_cm1c,LOD_cr,LOD_rc,LOD_rm2c,LOD_rm1c,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm2c,logL_cm1c,logL_cr,logL_rc,logL_rm2c,logL_rm1c,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed2c","coupling mixed1c","coupling repulsion","repulsion coupling","repulsion mixed2c","repulsion mixed1c","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_2.4 <- function(x){
logL_cc <- function(r,n01,n02,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n44,n45,n03,n43) {
L <- (-2*n01 - 3*n02 - 3*n04 - 2*n05 - 2*n11 - n12 - n13 - n14 - 2*n15 - 2*n21 - 3*n22 + n23 - 3*n24 - 2*n25 - 2*n31 - n32 - n33 - n34 - 2*n35 - 2*n41 - 3*n42 - 3*n44 - 2*n45)*log(2) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + (n21 + n25)*log(7) + 3*(n01 + n45)*log(pmax(1e-6,1 - r)) + 3*(n05 + n41)*log(pmax(1e-6,r)) + (n03 + n21 + n25 + n43)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n42)*(log(7 - 4*r) + 2*log(pmax(1e-6,r))) + (n15 + n31)*(log(5 - 2*r) + 2*log(pmax(1e-6,r))) + (n11 + n35)*(2*log(pmax(1e-6,1 - r)) + log(3 + 2*r)) + (n02 + n44)*(2*log(pmax(1e-6,1 - r)) + log(3 + 4*r)) + (n13 + n33)*log(5 + 2*r - 2*r^2) + (n22 + n24)*log(21 + 2*r - 2*r^2) + (n14 + n32)*(log(pmax(1e-6,r)) + log(7 - 3*r + 2*r^2)) + n23*log(3 - 2*r + 2*r^2) + (n12 + n34)*log(6 - 7*r + 3*r^2 - 2*r^3)
return(L)}
interlogL_cc <- function(n01,n02,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n44,n45,n03,n43) {
optimize(logL_cc,c(0,0.5), n01,n02,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n44,n45,n03,n43, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_01"],x[,"n_02"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_44"],x[,"n_45"],x[,"n_03"],x[,"n_43"])
LOD_cc <- 3*(x[,"n_05"] + x[,"n_41"])*log10(2) + 2*(x[,"n_03"] + x[,"n_21"] + x[,"n_25"] + x[,"n_43"])*log10(2) + 3*(x[,"n_01"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_32"])*log10(3) - (x[,"n_12"] + x[,"n_34"])*log10(3) - (x[,"n_04"] + x[,"n_42"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - x[,"n_23"]*(-log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_33"])*(-log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_24"])*(-log10(2) + log10(43)) + 3*(x[,"n_01"] + x[,"n_45"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_05"] + x[,"n_41"])*log10(pmax(1e-6,r_cc)) + (x[,"n_03"] + x[,"n_21"] + x[,"n_25"] + x[,"n_43"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_42"])*(log10(7 - 4*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_31"])*(log10(5 - 2*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(3 + 2*r_cc)) + (x[,"n_02"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(3 + 4*r_cc)) + (x[,"n_13"] + x[,"n_33"])*log10(5 + 2*r_cc - 2*r_cc^2) + (x[,"n_22"] + x[,"n_24"])*log10(21 + 2*r_cc - 2*r_cc^2) + (x[,"n_14"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(7 - 3*r_cc + 2*r_cc^2)) + x[,"n_23"]*log10(3 - 2*r_cc + 2*r_cc^2) + (x[,"n_12"] + x[,"n_34"])*log10(6 - 7*r_cc + 3*r_cc^2 - 2*r_cc^3)
logL_cc <- (-2*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 3*x[,"n_22"] + x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_21"] + x[,"n_25"])*log(7) + 3*(x[,"n_01"] + x[,"n_45"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_05"] + x[,"n_41"])*log(pmax(1e-6,r_cc)) + (x[,"n_03"] + x[,"n_21"] + x[,"n_25"] + x[,"n_43"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_42"])*(log(7 - 4*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_31"])*(log(5 - 2*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_cc)) + log(3 + 2*r_cc)) + (x[,"n_02"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_cc)) + log(3 + 4*r_cc)) + (x[,"n_13"] + x[,"n_33"])*log(5 + 2*r_cc - 2*r_cc^2) + (x[,"n_22"] + x[,"n_24"])*log(21 + 2*r_cc - 2*r_cc^2) + (x[,"n_14"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(7 - 3*r_cc + 2*r_cc^2)) + x[,"n_23"]*log(3 - 2*r_cc + 2*r_cc^2) + (x[,"n_12"] + x[,"n_34"])*log(6 - 7*r_cc + 3*r_cc^2 - 2*r_cc^3)
logL_cm2c <- function(r,n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n31,n33,n35,n41,n42,n43,n44,n45,n12,n14,n32,n34) {
L <- (-n01 - 2*n02 - 3*n03 - 2*n04 - n05 - n11 - 2*n13 - n15 - n21 - 2*n22 - 2*n23 - 2*n24 - n25 - n31 - 2*n33 - n35 - n41 - 2*n42 - 3*n43 - 2*n44 - n45)*log(2) + (-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(3) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + (n13 + n33)*log(11) + (n01 + n45)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n05 + n41)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + n23*log(57 + 8*r - 8*r^2) + (n21 + n25)*log(3 - r + r^2) + (n15 + n31)*(log(pmax(1e-6,r)) + log(4 - 3*r + 2*r^2)) + (n22 + n24)*log(33 - 2*r + 2*r^2) + (n04 + n42)*(log(pmax(1e-6,r)) + log(6 - 7*r + 4*r^2)) + (n03 + n43)*log(9 - 8*r + 8*r^2) + (n02 + n44)*log(3 - 4*r + 5*r^2 - 4*r^3) + (n11 + n35)*log(3 - 4*r + 3*r^2 - 2*r^3) + (n14 + n32)*log(3 + 2*r + 3*r^2 - 2*r^3) + (n12 + n34)*log(6 - 2*r - 3*r^2 + 2*r^3)
return(L)}
interlogL_cm2c <- function(n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n31,n33,n35,n41,n42,n43,n44,n45,n12,n14,n32,n34) {
optimize(logL_cm2c,c(0,0.5), n01,n02,n03,n04,n05,n11,n13,n15,n21,n22,n23,n24,n25,n31,n33,n35,n41,n42,n43,n44,n45,n12,n14,n32,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_13"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_33"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_12"],x[,"n_14"],x[,"n_32"],x[,"n_34"])
LOD_cm2c <- 3*(x[,"n_05"] + x[,"n_41"])*log10(2) + 3*(x[,"n_01"] + x[,"n_45"])*log10(2) - (x[,"n_15"] + x[,"n_31"])*(-log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_35"])*(-log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_32"])*(-log10(2) + 2*log10(3)) - (x[,"n_12"] + x[,"n_34"])*(-log10(2) + 2*log10(3)) - (x[,"n_03"] + x[,"n_43"])*log10(7) - (x[,"n_04"] + x[,"n_42"])*(-2*log10(2) + log10(7)) - (x[,"n_02"] + x[,"n_44"])*(-2*log10(2) + log10(7)) - (x[,"n_21"] + x[,"n_25"])*(-2*log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_24"])*(-log10(2) + log10(5) + log10(13)) - x[,"n_23"]*log10(59) + (x[,"n_01"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c))) + (x[,"n_05"] + x[,"n_41"])*(log10(pmax(1e-6,1 - r_cm2c)) + 2*log10(pmax(1e-6,r_cm2c))) + x[,"n_23"]*log10(57 + 8*r_cm2c - 8*r_cm2c^2) + (x[,"n_21"] + x[,"n_25"])*log10(3 - r_cm2c + r_cm2c^2) + (x[,"n_15"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm2c)) + log10(4 - 3*r_cm2c + 2*r_cm2c^2)) + (x[,"n_22"] + x[,"n_24"])*log10(33 - 2*r_cm2c + 2*r_cm2c^2) + (x[,"n_04"] + x[,"n_42"])*(log10(pmax(1e-6,r_cm2c)) + log10(6 - 7*r_cm2c + 4*r_cm2c^2)) + (x[,"n_03"] + x[,"n_43"])*log10(9 - 8*r_cm2c + 8*r_cm2c^2) + (x[,"n_02"] + x[,"n_44"])*log10(3 - 4*r_cm2c + 5*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_11"] + x[,"n_35"])*log10(3 - 4*r_cm2c + 3*r_cm2c^2 - 2*r_cm2c^3) + (x[,"n_14"] + x[,"n_32"])*log10(3 + 2*r_cm2c + 3*r_cm2c^2 - 2*r_cm2c^3) + (x[,"n_12"] + x[,"n_34"])*log10(6 - 2*r_cm2c - 3*r_cm2c^2 + 2*r_cm2c^3)
logL_cm2c <- (-x[,"n_01"] - 2*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - 2*x[,"n_13"] - x[,"n_15"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - 2*x[,"n_33"] - x[,"n_35"] - x[,"n_41"] - 2*x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"] - x[,"n_45"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_13"] + x[,"n_33"])*log(11) + (x[,"n_01"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c))) + (x[,"n_05"] + x[,"n_41"])*(log(pmax(1e-6,1 - r_cm2c)) + 2*log(pmax(1e-6,r_cm2c))) + x[,"n_23"]*log(57 + 8*r_cm2c - 8*r_cm2c^2) + (x[,"n_21"] + x[,"n_25"])*log(3 - r_cm2c + r_cm2c^2) + (x[,"n_15"] + x[,"n_31"])*(log(pmax(1e-6,r_cm2c)) + log(4 - 3*r_cm2c + 2*r_cm2c^2)) + (x[,"n_22"] + x[,"n_24"])*log(33 - 2*r_cm2c + 2*r_cm2c^2) + (x[,"n_04"] + x[,"n_42"])*(log(pmax(1e-6,r_cm2c)) + log(6 - 7*r_cm2c + 4*r_cm2c^2)) + (x[,"n_03"] + x[,"n_43"])*log(9 - 8*r_cm2c + 8*r_cm2c^2) + (x[,"n_02"] + x[,"n_44"])*log(3 - 4*r_cm2c + 5*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_11"] + x[,"n_35"])*log(3 - 4*r_cm2c + 3*r_cm2c^2 - 2*r_cm2c^3) + (x[,"n_14"] + x[,"n_32"])*log(3 + 2*r_cm2c + 3*r_cm2c^2 - 2*r_cm2c^3) + (x[,"n_12"] + x[,"n_34"])*log(6 - 2*r_cm2c - 3*r_cm2c^2 + 2*r_cm2c^3)
logL_cm1c <- function(r,n01,n02,n03,n04,n05,n11,n12,n14,n15,n21,n22,n23,n24,n25,n31,n32,n34,n35,n41,n42,n43,n44,n45,n13,n33) {
L <- (-2*n01 - 3*n02 - 2*n03 - 3*n04 - 2*n05 - 2*n11 - n12 - n14 - 2*n15 - 2*n21 - 3*n22 - n23 - 3*n24 - 2*n25 - 2*n31 - n32 - n34 - 2*n35 - 2*n41 - 3*n42 - 2*n43 - 3*n44 - 2*n45)*log(2) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + (n05 + n41)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n45)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + n23*log(9 + 4*r - 4*r^2) + (n22 + n24)*log(21 + 2*r - 2*r^2) + (n13 + n33)*log(3 - r + r^2) + (n11 + n35)*(log(pmax(1e-6,r)) + log(4 - 5*r + 2*r^2)) + (n02 + n44)*(log(pmax(1e-6,r)) + log(4 - 5*r + 4*r^2)) + (n21 + n25)*log(3 - 5*r + 5*r^2) + (n04 + n42)*log(3 - 6*r + 7*r^2 - 4*r^3) + (n15 + n31)*log(1 + r^2 - 2*r^3) + (n12 + n34)*log(3 - r + 3*r^2 - 2*r^3) + (n14 + n32)*log(3 + r - 3*r^2 + 2*r^3)
return(L)}
interlogL_cm1c <- function(n01,n02,n03,n04,n05,n11,n12,n14,n15,n21,n22,n23,n24,n25,n31,n32,n34,n35,n41,n42,n43,n44,n45,n13,n33) {
optimize(logL_cm1c,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n14,n15,n21,n22,n23,n24,n25,n31,n32,n34,n35,n41,n42,n43,n44,n45,n13,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm1c <- mapply(interlogL_cm1c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_13"],x[,"n_33"])
LOD_cm1c <- 3*(x[,"n_05"] + x[,"n_41"])*log10(2) + 3*(x[,"n_01"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_32"])*log10(3) - (x[,"n_12"] + x[,"n_34"])*log10(3) - (x[,"n_04"] + x[,"n_42"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - x[,"n_23"]*(log10(2) + log10(5)) - (x[,"n_21"] + x[,"n_25"])*(-2*log10(2) + log10(7)) - (x[,"n_13"] + x[,"n_33"])*(-2*log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_24"])*(-log10(2) + log10(43)) + (x[,"n_05"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_cm1c)) + log10(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cm1c)) + 2*log10(pmax(1e-6,r_cm1c))) + x[,"n_23"]*log10(9 + 4*r_cm1c - 4*r_cm1c^2) + (x[,"n_22"] + x[,"n_24"])*log10(21 + 2*r_cm1c - 2*r_cm1c^2) + (x[,"n_13"] + x[,"n_33"])*log10(3 - r_cm1c + r_cm1c^2) + (x[,"n_11"] + x[,"n_35"])*(log10(pmax(1e-6,r_cm1c)) + log10(4 - 5*r_cm1c + 2*r_cm1c^2)) + (x[,"n_02"] + x[,"n_44"])*(log10(pmax(1e-6,r_cm1c)) + log10(4 - 5*r_cm1c + 4*r_cm1c^2)) + (x[,"n_21"] + x[,"n_25"])*log10(3 - 5*r_cm1c + 5*r_cm1c^2) + (x[,"n_04"] + x[,"n_42"])*log10(3 - 6*r_cm1c + 7*r_cm1c^2 - 4*r_cm1c^3) + (x[,"n_15"] + x[,"n_31"])*log10(1 + r_cm1c^2 - 2*r_cm1c^3) + (x[,"n_12"] + x[,"n_34"])*log10(3 - r_cm1c + 3*r_cm1c^2 - 2*r_cm1c^3) + (x[,"n_14"] + x[,"n_32"])*log10(3 + r_cm1c - 3*r_cm1c^2 + 2*r_cm1c^3)
logL_cm1c <- (-2*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_11"] - x[,"n_12"] - x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 3*x[,"n_22"] - x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_31"] - x[,"n_32"] - x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 2*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_05"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_cm1c)) + log(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cm1c)) + 2*log(pmax(1e-6,r_cm1c))) + x[,"n_23"]*log(9 + 4*r_cm1c - 4*r_cm1c^2) + (x[,"n_22"] + x[,"n_24"])*log(21 + 2*r_cm1c - 2*r_cm1c^2) + (x[,"n_13"] + x[,"n_33"])*log(3 - r_cm1c + r_cm1c^2) + (x[,"n_11"] + x[,"n_35"])*(log(pmax(1e-6,r_cm1c)) + log(4 - 5*r_cm1c + 2*r_cm1c^2)) + (x[,"n_02"] + x[,"n_44"])*(log(pmax(1e-6,r_cm1c)) + log(4 - 5*r_cm1c + 4*r_cm1c^2)) + (x[,"n_21"] + x[,"n_25"])*log(3 - 5*r_cm1c + 5*r_cm1c^2) + (x[,"n_04"] + x[,"n_42"])*log(3 - 6*r_cm1c + 7*r_cm1c^2 - 4*r_cm1c^3) + (x[,"n_15"] + x[,"n_31"])*log(1 + r_cm1c^2 - 2*r_cm1c^3) + (x[,"n_12"] + x[,"n_34"])*log(3 - r_cm1c + 3*r_cm1c^2 - 2*r_cm1c^3) + (x[,"n_14"] + x[,"n_32"])*log(3 + r_cm1c - 3*r_cm1c^2 + 2*r_cm1c^3)
logL_rc <- function(r,n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
L <- (-4*n01 - 3*n02 - 4*n03 - 3*n04 - 4*n05 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n15 - 4*n21 - 3*n22 - 3*n23 - 3*n24 - 4*n25 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 3*n35 - 4*n41 - 3*n42 - 4*n43 - 3*n44 - 4*n45)*log(2) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + (n05 + n41)*(log(3 - 2*r) + 2*log(pmax(1e-6,r))) + (n01 + n45)*(2*log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n03 + n43)*log(3 + 4*r - 4*r^2) + n23*log(39 + 4*r - 4*r^2) + (n22 + n24)*log(21 + 2*r - 2*r^2) + (n15 + n31)*(log(pmax(1e-6,r)) + log(6 - 5*r + 2*r^2)) + (n04 + n42)*(log(pmax(1e-6,r)) + log(3 - 2*r + 2*r^2)) + (n13 + n33)*log(23 - 4*r + 4*r^2) + (n21 + n25)*log(9 - 8*r + 8*r^2) + (n14 + n32)*log(9 + 4*r + 6*r^2 - 4*r^3) + (n11 + n35)*log(3 - 2*r + r^2 - 2*r^3) + (n02 + n44)*log(3 - 5*r + 4*r^2 - 2*r^3) + (n12 + n34)*log(15 - 4*r - 6*r^2 + 4*r^3)
return(L)}
interlogL_rc <- function(n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
optimize(logL_rc,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rc <- -((x[,"n_15"] + x[,"n_31"])*log10(2)) - (x[,"n_11"] + x[,"n_35"])*log10(2) + (x[,"n_05"] + x[,"n_41"])*log10(2) - 2*(x[,"n_03"] + x[,"n_43"])*log10(2) + (x[,"n_01"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_32"])*(2*log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_34"])*(2*log10(2) + log10(3)) - (x[,"n_04"] + x[,"n_42"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - x[,"n_23"]*(3*log10(2) + log10(5)) - (x[,"n_21"] + x[,"n_25"])*log10(7) - (x[,"n_13"] + x[,"n_33"])*(log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_24"])*(-log10(2) + log10(43)) + (x[,"n_05"] + x[,"n_41"])*(log10(3 - 2*r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_01"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_03"] + x[,"n_43"])*log10(3 + 4*r_rc - 4*r_rc^2) + x[,"n_23"]*log10(39 + 4*r_rc - 4*r_rc^2) + (x[,"n_22"] + x[,"n_24"])*log10(21 + 2*r_rc - 2*r_rc^2) + (x[,"n_15"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(6 - 5*r_rc + 2*r_rc^2)) + (x[,"n_04"] + x[,"n_42"])*(log10(pmax(1e-6,r_rc)) + log10(3 - 2*r_rc + 2*r_rc^2)) + (x[,"n_13"] + x[,"n_33"])*log10(23 - 4*r_rc + 4*r_rc^2) + (x[,"n_21"] + x[,"n_25"])*log10(9 - 8*r_rc + 8*r_rc^2) + (x[,"n_14"] + x[,"n_32"])*log10(9 + 4*r_rc + 6*r_rc^2 - 4*r_rc^3) + (x[,"n_11"] + x[,"n_35"])*log10(3 - 2*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_02"] + x[,"n_44"])*log10(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3) + (x[,"n_12"] + x[,"n_34"])*log10(15 - 4*r_rc - 6*r_rc^2 + 4*r_rc^3)
logL_rc <- (-4*x[,"n_01"] - 3*x[,"n_02"] - 4*x[,"n_03"] - 3*x[,"n_04"] - 4*x[,"n_05"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 4*x[,"n_25"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_35"] - 4*x[,"n_41"] - 3*x[,"n_42"] - 4*x[,"n_43"] - 3*x[,"n_44"] - 4*x[,"n_45"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_05"] + x[,"n_41"])*(log(3 - 2*r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_01"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_03"] + x[,"n_43"])*log(3 + 4*r_rc - 4*r_rc^2) + x[,"n_23"]*log(39 + 4*r_rc - 4*r_rc^2) + (x[,"n_22"] + x[,"n_24"])*log(21 + 2*r_rc - 2*r_rc^2) + (x[,"n_15"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(6 - 5*r_rc + 2*r_rc^2)) + (x[,"n_04"] + x[,"n_42"])*(log(pmax(1e-6,r_rc)) + log(3 - 2*r_rc + 2*r_rc^2)) + (x[,"n_13"] + x[,"n_33"])*log(23 - 4*r_rc + 4*r_rc^2) + (x[,"n_21"] + x[,"n_25"])*log(9 - 8*r_rc + 8*r_rc^2) + (x[,"n_14"] + x[,"n_32"])*log(9 + 4*r_rc + 6*r_rc^2 - 4*r_rc^3) + (x[,"n_11"] + x[,"n_35"])*log(3 - 2*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_02"] + x[,"n_44"])*log(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3) + (x[,"n_12"] + x[,"n_34"])*log(15 - 4*r_rc - 6*r_rc^2 + 4*r_rc^3)
logL_rm2c <- function(r,n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
L <- (-3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n05 - 3*n11 - 2*n12 - 2*n13 - 2*n14 - 3*n15 - n21 - 2*n22 - 2*n23 - 2*n24 - n25 - 3*n31 - 2*n32 - 2*n33 - 2*n34 - 3*n35 - 3*n41 - 4*n42 - 3*n43 - 4*n44 - 3*n45)*log(2) + (-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(3) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + (n13 + n33)*log(11) + (n05 + n41)*(log(3 - 2*r) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n45)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(1 + 2*r)) + n23*log(57 + 8*r - 8*r^2) + (n21 + n25)*log(3 - r + r^2) + (n22 + n24)*log(33 - 2*r + 2*r^2) + (n03 + n43)*log(9 - 8*r + 8*r^2) + (n04 + n42)*log(9 - 6*r + 8*r^2 - 8*r^3) + (n15 + n31)*log(9 - 8*r + 6*r^2 - 4*r^3) + (n12 + n34)*log(15 + 4*r + 6*r^2 - 4*r^3) + (n14 + n32)*log(21 - 4*r - 6*r^2 + 4*r^3) + (n11 + n35)*log(3 + 8*r - 6*r^2 + 4*r^3) + (n02 + n44)*log(3 + 14*r - 16*r^2 + 8*r^3)
return(L)}
interlogL_rm2c <- function(n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
optimize(logL_rm2c,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rm2c <- (x[,"n_05"] + x[,"n_41"])*log10(2) + (x[,"n_01"] + x[,"n_45"])*log10(2) - (x[,"n_15"] + x[,"n_31"])*(log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_35"])*(log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_32"])*(log10(2) + 2*log10(3)) - (x[,"n_12"] + x[,"n_34"])*(log10(2) + 2*log10(3)) - (x[,"n_04"] + x[,"n_42"])*log10(7) - (x[,"n_03"] + x[,"n_43"])*log10(7) - (x[,"n_02"] + x[,"n_44"])*log10(7) - (x[,"n_21"] + x[,"n_25"])*(-2*log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_24"])*(-log10(2) + log10(5) + log10(13)) - x[,"n_23"]*log10(59) + (x[,"n_05"] + x[,"n_41"])*(log10(3 - 2*r_rm2c) + log10(pmax(1e-6,1 - r_rm2c)) + log10(pmax(1e-6,r_rm2c))) + (x[,"n_01"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_rm2c)) + log10(pmax(1e-6,r_rm2c)) + log10(1 + 2*r_rm2c)) + x[,"n_23"]*log10(57 + 8*r_rm2c - 8*r_rm2c^2) + (x[,"n_21"] + x[,"n_25"])*log10(3 - r_rm2c + r_rm2c^2) + (x[,"n_22"] + x[,"n_24"])*log10(33 - 2*r_rm2c + 2*r_rm2c^2) + (x[,"n_03"] + x[,"n_43"])*log10(9 - 8*r_rm2c + 8*r_rm2c^2) + (x[,"n_04"] + x[,"n_42"])*log10(9 - 6*r_rm2c + 8*r_rm2c^2 - 8*r_rm2c^3) + (x[,"n_15"] + x[,"n_31"])*log10(9 - 8*r_rm2c + 6*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_12"] + x[,"n_34"])*log10(15 + 4*r_rm2c + 6*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_14"] + x[,"n_32"])*log10(21 - 4*r_rm2c - 6*r_rm2c^2 + 4*r_rm2c^3) + (x[,"n_11"] + x[,"n_35"])*log10(3 + 8*r_rm2c - 6*r_rm2c^2 + 4*r_rm2c^3) + (x[,"n_02"] + x[,"n_44"])*log10(3 + 14*r_rm2c - 16*r_rm2c^2 + 8*r_rm2c^3)
logL_rm2c <- (-3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_05"] - 3*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_15"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_25"] - 3*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 3*x[,"n_35"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 3*x[,"n_45"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_13"] + x[,"n_33"])*log(11) + (x[,"n_05"] + x[,"n_41"])*(log(3 - 2*r_rm2c) + log(pmax(1e-6,1 - r_rm2c)) + log(pmax(1e-6,r_rm2c))) + (x[,"n_01"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_rm2c)) + log(pmax(1e-6,r_rm2c)) + log(1 + 2*r_rm2c)) + x[,"n_23"]*log(57 + 8*r_rm2c - 8*r_rm2c^2) + (x[,"n_21"] + x[,"n_25"])*log(3 - r_rm2c + r_rm2c^2) + (x[,"n_22"] + x[,"n_24"])*log(33 - 2*r_rm2c + 2*r_rm2c^2) + (x[,"n_03"] + x[,"n_43"])*log(9 - 8*r_rm2c + 8*r_rm2c^2) + (x[,"n_04"] + x[,"n_42"])*log(9 - 6*r_rm2c + 8*r_rm2c^2 - 8*r_rm2c^3) + (x[,"n_15"] + x[,"n_31"])*log(9 - 8*r_rm2c + 6*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_12"] + x[,"n_34"])*log(15 + 4*r_rm2c + 6*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_14"] + x[,"n_32"])*log(21 - 4*r_rm2c - 6*r_rm2c^2 + 4*r_rm2c^3) + (x[,"n_11"] + x[,"n_35"])*log(3 + 8*r_rm2c - 6*r_rm2c^2 + 4*r_rm2c^3) + (x[,"n_02"] + x[,"n_44"])*log(3 + 14*r_rm2c - 16*r_rm2c^2 + 8*r_rm2c^3)
logL_rm1c <- function(r,n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
L <- (-4*n01 - 3*n02 - 4*n03 - 3*n04 - 4*n05 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n15 - 4*n21 - 3*n22 - 3*n23 - 3*n24 - 4*n25 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 3*n35 - 4*n41 - 3*n42 - 4*n43 - 3*n44 - 4*n45)*log(2) + (-2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n05 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n21 - 2*n22 - n23 - 2*n24 - 2*n25 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n41 - 2*n42 - 2*n43 - 2*n44 - 2*n45)*log(5) + (n05 + n41)*(log(3 - 2*r) + 2*log(pmax(1e-6,1 - r))) + (n01 + n45)*(2*log(pmax(1e-6,r)) + log(1 + 2*r)) + (n21 + n25)*log(3 + 16*r - 16*r^2) + (n03 + n43)*log(1 + 12*r - 12*r^2) + (n13 + n33)*log(21 + 4*r - 4*r^2) + (n22 + n24)*log(21 + 2*r - 2*r^2) + (n11 + n35)*(log(pmax(1e-6,r)) + log(2 + 5*r - 2*r^2)) + n23*log(9 - 4*r + 4*r^2) + (n14 + n32)*log(21 - 20*r + 6*r^2 - 4*r^3) + (n02 + n44)*log(r + 4*r^2 - 2*r^3) + (n04 + n42)*log(3 - 3*r - 2*r^2 + 2*r^3) + (n15 + n31)*log(5 - 6*r - r^2 + 2*r^3) + (n12 + n34)*log(3 + 20*r - 6*r^2 + 4*r^3)
return(L)}
interlogL_rm1c <- function(n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
optimize(logL_rm1c,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm1c <- mapply(interlogL_rm1c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rm1c <- -3*x[,"n_23"]*log10(2) - (x[,"n_15"] + x[,"n_31"])*log10(2) - (x[,"n_11"] + x[,"n_35"])*log10(2) + (x[,"n_05"] + x[,"n_41"])*log10(2) - 2*(x[,"n_03"] + x[,"n_43"])*log10(2) + (x[,"n_01"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_32"])*(2*log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_34"])*(2*log10(2) + log10(3)) - (x[,"n_04"] + x[,"n_42"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - (x[,"n_21"] + x[,"n_25"])*log10(7) - (x[,"n_13"] + x[,"n_33"])*(log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_24"])*(-log10(2) + log10(43)) + (x[,"n_05"] + x[,"n_41"])*(log10(3 - 2*r_rm1c) + 2*log10(pmax(1e-6,1 - r_rm1c))) + (x[,"n_01"] + x[,"n_45"])*(2*log10(pmax(1e-6,r_rm1c)) + log10(1 + 2*r_rm1c)) + (x[,"n_21"] + x[,"n_25"])*log10(3 + 16*r_rm1c - 16*r_rm1c^2) + (x[,"n_03"] + x[,"n_43"])*log10(1 + 12*r_rm1c - 12*r_rm1c^2) + (x[,"n_13"] + x[,"n_33"])*log10(21 + 4*r_rm1c - 4*r_rm1c^2) + (x[,"n_22"] + x[,"n_24"])*log10(21 + 2*r_rm1c - 2*r_rm1c^2) + (x[,"n_11"] + x[,"n_35"])*(log10(pmax(1e-6,r_rm1c)) + log10(2 + 5*r_rm1c - 2*r_rm1c^2)) + x[,"n_23"]*log10(9 - 4*r_rm1c + 4*r_rm1c^2) + (x[,"n_14"] + x[,"n_32"])*log10(21 - 20*r_rm1c + 6*r_rm1c^2 - 4*r_rm1c^3) + (x[,"n_02"] + x[,"n_44"])*log10(r_rm1c + 4*r_rm1c^2 - 2*r_rm1c^3) + (x[,"n_04"] + x[,"n_42"])*log10(3 - 3*r_rm1c - 2*r_rm1c^2 + 2*r_rm1c^3) + (x[,"n_15"] + x[,"n_31"])*log10(5 - 6*r_rm1c - r_rm1c^2 + 2*r_rm1c^3) + (x[,"n_12"] + x[,"n_34"])*log10(3 + 20*r_rm1c - 6*r_rm1c^2 + 4*r_rm1c^3)
logL_rm1c <- (-4*x[,"n_01"] - 3*x[,"n_02"] - 4*x[,"n_03"] - 3*x[,"n_04"] - 4*x[,"n_05"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 4*x[,"n_25"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_35"] - 4*x[,"n_41"] - 3*x[,"n_42"] - 4*x[,"n_43"] - 3*x[,"n_44"] - 4*x[,"n_45"])*log(2) + (-2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 2*x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - 2*x[,"n_45"])*log(5) + (x[,"n_05"] + x[,"n_41"])*(log(3 - 2*r_rm1c) + 2*log(pmax(1e-6,1 - r_rm1c))) + (x[,"n_01"] + x[,"n_45"])*(2*log(pmax(1e-6,r_rm1c)) + log(1 + 2*r_rm1c)) + (x[,"n_21"] + x[,"n_25"])*log(3 + 16*r_rm1c - 16*r_rm1c^2) + (x[,"n_03"] + x[,"n_43"])*log(1 + 12*r_rm1c - 12*r_rm1c^2) + (x[,"n_13"] + x[,"n_33"])*log(21 + 4*r_rm1c - 4*r_rm1c^2) + (x[,"n_22"] + x[,"n_24"])*log(21 + 2*r_rm1c - 2*r_rm1c^2) + (x[,"n_11"] + x[,"n_35"])*(log(pmax(1e-6,r_rm1c)) + log(2 + 5*r_rm1c - 2*r_rm1c^2)) + x[,"n_23"]*log(9 - 4*r_rm1c + 4*r_rm1c^2) + (x[,"n_14"] + x[,"n_32"])*log(21 - 20*r_rm1c + 6*r_rm1c^2 - 4*r_rm1c^3) + (x[,"n_02"] + x[,"n_44"])*log(r_rm1c + 4*r_rm1c^2 - 2*r_rm1c^3) + (x[,"n_04"] + x[,"n_42"])*log(3 - 3*r_rm1c - 2*r_rm1c^2 + 2*r_rm1c^3) + (x[,"n_15"] + x[,"n_31"])*log(5 - 6*r_rm1c - r_rm1c^2 + 2*r_rm1c^3) + (x[,"n_12"] + x[,"n_34"])*log(3 + 20*r_rm1c - 6*r_rm1c^2 + 4*r_rm1c^3)

return(list(
r_mat = cbind(r_cc,r_cm2c,r_cm1c,r_rc,r_rm2c,r_rm1c,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm2c,LOD_cm1c,LOD_rc,LOD_rm2c,LOD_rm1c,0),
logL_mat = cbind(logL_cc,logL_cm2c,logL_cm1c,logL_rc,logL_rm2c,logL_rm1c,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed2c","coupling mixed1c","repulsion coupling","repulsion mixed2c","repulsion mixed1c","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_3.1 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 4*n20 - 3*n21 - 3*n22 - 3*n23 - 4*n24 - 3*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 4*n40 - 3*n41 - 4*n42 - 3*n43 - 4*n44)*log(2) + (n02 + n12 + n20 + n21 + 2*n22 + n23 + n24 + n32 + n42)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + 2*(n00 + n44)*log(pmax(1e-6,1 - r)) + 2*(n04 + n40)*log(pmax(1e-6,r)) + (n03 + n14 + n30 + n41)*(log(pmax(1e-6,r)) + log(2 + r)) + (n02 + n20 + n24 + n42)*log(1 + 2*r - 2*r^2) + (n12 + n21 + n23 + n32)*log(7 + 2*r - 2*r^2) + (n01 + n10 + n34 + n43)*log(3 - 4*r + r^2) + (n11 + n33)*log(21 - 18*r + 2*r^2) + n22*log(5 - 2*r + 2*r^2) + (n13 + n31)*log(5 + 14*r + 2*r^2)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cc <- 2*(x[,"n_04"] + x[,"n_40"])*log10(2) + 2*(x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-log10(2) + log10(3)) - x[,"n_22"]*(-log10(2) + 2*log10(3)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + 2*log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + 2*log10(5)) + 2*(x[,"n_00"] + x[,"n_44"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(7 + 2*r_cc - 2*r_cc^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*log10(3 - 4*r_cc + r_cc^2) + (x[,"n_11"] + x[,"n_33"])*log10(21 - 18*r_cc + 2*r_cc^2) + x[,"n_22"]*log10(5 - 2*r_cc + 2*r_cc^2) + (x[,"n_13"] + x[,"n_31"])*log10(5 + 14*r_cc + 2*r_cc^2)
logL_cc <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_20"] + x[,"n_21"] + 2*x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_32"] + x[,"n_42"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + 2*(x[,"n_00"] + x[,"n_44"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_04"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(7 + 2*r_cc - 2*r_cc^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*log(3 - 4*r_cc + r_cc^2) + (x[,"n_11"] + x[,"n_33"])*log(21 - 18*r_cc + 2*r_cc^2) + x[,"n_22"]*log(5 - 2*r_cc + 2*r_cc^2) + (x[,"n_13"] + x[,"n_31"])*log(5 + 14*r_cc + 2*r_cc^2)
logL_cr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-4*n00 - 4*n01 - 3*n02 - 4*n03 - 4*n04 - 4*n10 - 2*n11 - 2*n12 - 2*n13 - 4*n14 - 3*n20 - 2*n21 - 2*n22 - 2*n23 - 3*n24 - 4*n30 - 2*n31 - 2*n32 - 2*n33 - 4*n34 - 4*n40 - 4*n41 - 3*n42 - 4*n43 - 4*n44)*log(2) + (n02 + n12 + n20 + n21 + 2*n22 + n23 + n24 + n32 + n42)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n00 + n04 + n40 + n44)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n22*(log(2 - r) + log(1 + r)) + (n03 + n10 + n34 + n41)*log(3 - 2*r^2) + (n01 + n14 + n30 + n43)*log(1 + 4*r - 2*r^2) + (n11 + n13 + n31 + n33)*log(6 + r - r^2) + (n02 + n20 + n24 + n42)*log(1 - r + r^2) + (n12 + n21 + n23 + n32)*log(4 - r + r^2)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cr <- 2*(x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*log10(2) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(2*log10(2) - log10(3)) - 2*x[,"n_22"]*(-log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_10"] + x[,"n_34"] + x[,"n_41"])*(-log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_14"] + x[,"n_30"] + x[,"n_43"])*(-log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_11"] + x[,"n_13"] + x[,"n_31"] + x[,"n_33"])*(-2*log10(2) + 2*log10(5)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + x[,"n_22"]*(log10(2 - r_cr) + log10(1 + r_cr)) + (x[,"n_03"] + x[,"n_10"] + x[,"n_34"] + x[,"n_41"])*log10(3 - 2*r_cr^2) + (x[,"n_01"] + x[,"n_14"] + x[,"n_30"] + x[,"n_43"])*log10(1 + 4*r_cr - 2*r_cr^2) + (x[,"n_11"] + x[,"n_13"] + x[,"n_31"] + x[,"n_33"])*log10(6 + r_cr - r_cr^2) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(1 - r_cr + r_cr^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(4 - r_cr + r_cr^2)
logL_cr <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 3*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"] - 4*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 4*x[,"n_34"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 3*x[,"n_42"] - 4*x[,"n_43"] - 4*x[,"n_44"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_20"] + x[,"n_21"] + 2*x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_32"] + x[,"n_42"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + x[,"n_22"]*(log(2 - r_cr) + log(1 + r_cr)) + (x[,"n_03"] + x[,"n_10"] + x[,"n_34"] + x[,"n_41"])*log(3 - 2*r_cr^2) + (x[,"n_01"] + x[,"n_14"] + x[,"n_30"] + x[,"n_43"])*log(1 + 4*r_cr - 2*r_cr^2) + (x[,"n_11"] + x[,"n_13"] + x[,"n_31"] + x[,"n_33"])*log(6 + r_cr - r_cr^2) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log(1 - r_cr + r_cr^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(4 - r_cr + r_cr^2)
logL_rc <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-4*n00 - 4*n01 - 3*n02 - 4*n03 - 4*n04 - 4*n10 - 2*n11 - 2*n12 - 2*n13 - 4*n14 - 3*n20 - 2*n21 - 2*n22 - 2*n23 - 3*n24 - 4*n30 - 2*n31 - 2*n32 - 2*n33 - 4*n34 - 4*n40 - 4*n41 - 3*n42 - 4*n43 - 4*n44)*log(2) + (n02 + n12 + n20 + n21 + 2*n22 + n23 + n24 + n32 + n42)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n00 + n04 + n40 + n44)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n22*(log(2 - r) + log(1 + r)) + (n01 + n14 + n30 + n43)*log(3 - 2*r^2) + (n03 + n10 + n34 + n41)*log(1 + 4*r - 2*r^2) + (n11 + n13 + n31 + n33)*log(6 + r - r^2) + (n02 + n20 + n24 + n42)*log(1 - r + r^2) + (n12 + n21 + n23 + n32)*log(4 - r + r^2)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rc <- 2*(x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*log10(2) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(2*log10(2) - log10(3)) - 2*x[,"n_22"]*(-log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_10"] + x[,"n_34"] + x[,"n_41"])*(-log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_14"] + x[,"n_30"] + x[,"n_43"])*(-log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_11"] + x[,"n_13"] + x[,"n_31"] + x[,"n_33"])*(-2*log10(2) + 2*log10(5)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc))) + x[,"n_22"]*(log10(2 - r_rc) + log10(1 + r_rc)) + (x[,"n_01"] + x[,"n_14"] + x[,"n_30"] + x[,"n_43"])*log10(3 - 2*r_rc^2) + (x[,"n_03"] + x[,"n_10"] + x[,"n_34"] + x[,"n_41"])*log10(1 + 4*r_rc - 2*r_rc^2) + (x[,"n_11"] + x[,"n_13"] + x[,"n_31"] + x[,"n_33"])*log10(6 + r_rc - r_rc^2) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(1 - r_rc + r_rc^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(4 - r_rc + r_rc^2)
logL_rc <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 3*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"] - 4*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 4*x[,"n_34"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 3*x[,"n_42"] - 4*x[,"n_43"] - 4*x[,"n_44"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_20"] + x[,"n_21"] + 2*x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_32"] + x[,"n_42"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc))) + x[,"n_22"]*(log(2 - r_rc) + log(1 + r_rc)) + (x[,"n_01"] + x[,"n_14"] + x[,"n_30"] + x[,"n_43"])*log(3 - 2*r_rc^2) + (x[,"n_03"] + x[,"n_10"] + x[,"n_34"] + x[,"n_41"])*log(1 + 4*r_rc - 2*r_rc^2) + (x[,"n_11"] + x[,"n_13"] + x[,"n_31"] + x[,"n_33"])*log(6 + r_rc - r_rc^2) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log(1 - r_rc + r_rc^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(4 - r_rc + r_rc^2)
logL_rr <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 4*n20 - 3*n21 - 3*n22 - 3*n23 - 4*n24 - 3*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 4*n40 - 3*n41 - 4*n42 - 3*n43 - 4*n44)*log(2) + (n02 + n12 + n20 + n21 + 2*n22 + n23 + n24 + n32 + n42)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + 2*(n04 + n40)*log(pmax(1e-6,1 - r)) + 2*(n00 + n44)*log(pmax(1e-6,r)) + (n01 + n10 + n34 + n43)*(log(pmax(1e-6,r)) + log(2 + r)) + (n02 + n20 + n24 + n42)*log(1 + 2*r - 2*r^2) + (n12 + n21 + n23 + n32)*log(7 + 2*r - 2*r^2) + (n03 + n14 + n30 + n41)*log(3 - 4*r + r^2) + (n13 + n31)*log(21 - 18*r + 2*r^2) + n22*log(5 - 2*r + 2*r^2) + (n11 + n33)*log(5 + 14*r + 2*r^2)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rr <- 2*(x[,"n_04"] + x[,"n_40"])*log10(2) + 2*(x[,"n_00"] + x[,"n_44"])*log10(2) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-log10(2) + log10(3)) - x[,"n_22"]*(-log10(2) + 2*log10(3)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + 2*log10(5)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + 2*log10(5)) + 2*(x[,"n_04"] + x[,"n_40"])*log10(pmax(1e-6,1 - r_rr)) + 2*(x[,"n_00"] + x[,"n_44"])*log10(pmax(1e-6,r_rr)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log10(pmax(1e-6,r_rr)) + log10(2 + r_rr)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log10(1 + 2*r_rr - 2*r_rr^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(7 + 2*r_rr - 2*r_rr^2) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*log10(3 - 4*r_rr + r_rr^2) + (x[,"n_13"] + x[,"n_31"])*log10(21 - 18*r_rr + 2*r_rr^2) + x[,"n_22"]*log10(5 - 2*r_rr + 2*r_rr^2) + (x[,"n_11"] + x[,"n_33"])*log10(5 + 14*r_rr + 2*r_rr^2)
logL_rr <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"])*log(2) + (x[,"n_02"] + x[,"n_12"] + x[,"n_20"] + x[,"n_21"] + 2*x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_32"] + x[,"n_42"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + 2*(x[,"n_04"] + x[,"n_40"])*log(pmax(1e-6,1 - r_rr)) + 2*(x[,"n_00"] + x[,"n_44"])*log(pmax(1e-6,r_rr)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log(pmax(1e-6,r_rr)) + log(2 + r_rr)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*log(1 + 2*r_rr - 2*r_rr^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(7 + 2*r_rr - 2*r_rr^2) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*log(3 - 4*r_rr + r_rr^2) + (x[,"n_13"] + x[,"n_31"])*log(21 - 18*r_rr + 2*r_rr^2) + x[,"n_22"]*log(5 - 2*r_rr + 2*r_rr^2) + (x[,"n_11"] + x[,"n_33"])*log(5 + 14*r_rr + 2*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_3.2 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- 4*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n05 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n20 - n21 - 2*n22 - 2*n23 - n24 - 2*n25 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n40 - 2*n41 - 2*n42 - 2*n43 - 2*n44 - 2*n45)*log(5) + (n20 + n25)*log(7) + 3*(n00 + n45)*log(pmax(1e-6,1 - r)) + (n01 + n44)*(2*log(pmax(1e-6,1 - r)) + log(6 - r)) + 3*(n05 + n40)*log(pmax(1e-6,r)) + (n20 + n25)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n15 + n30)*(log(5 - 2*r) + 2*log(pmax(1e-6,r))) + (n04 + n41)*(2*log(pmax(1e-6,r)) + log(5 + r)) + (n10 + n35)*(2*log(pmax(1e-6,1 - r)) + log(3 + 2*r)) + (n03 + n42)*(log(pmax(1e-6,r)) + log(7 + 4*r - 8*r^2)) + (n21 + n24)*log(3 + 5*r - 5*r^2) + (n14 + n31)*(log(pmax(1e-6,r)) + log(20 + 9*r - 2*r^2)) + (n22 + n23)*log(75 - 32*r + 32*r^2) + (n12 + n33)*log(51 - 32*r + 16*r^2 - 16*r^3) + (n11 + n34)*log(27 - 32*r + 3*r^2 + 2*r^3) + (n02 + n43)*log(3 + 9*r - 20*r^2 + 8*r^3) + (n13 + n32)*log(19 + 48*r - 32*r^2 + 16*r^3)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cc <- 2*(x[,"n_20"] + x[,"n_25"])*log10(2) + 3*(x[,"n_05"] + x[,"n_40"])*log10(2) + 3*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_31"])*(2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_34"])*(2*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_42"])*(-log10(2) + log10(7)) - (x[,"n_02"] + x[,"n_43"])*(-log10(2) + log10(7)) - (x[,"n_04"] + x[,"n_41"])*(-3*log10(2) + log10(11)) - (x[,"n_01"] + x[,"n_44"])*(-3*log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_24"])*(-2*log10(2) + log10(17)) - (x[,"n_13"] + x[,"n_32"])*log10(37) - (x[,"n_12"] + x[,"n_33"])*log10(37) - (x[,"n_22"] + x[,"n_23"])*log10(67) + 3*(x[,"n_00"] + x[,"n_45"])*log10(pmax(1e-6,1 - r_cc)) + (x[,"n_01"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(6 - r_cc)) + 3*(x[,"n_05"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + (x[,"n_20"] + x[,"n_25"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_30"])*(log10(5 - 2*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_41"])*(2*log10(pmax(1e-6,r_cc)) + log10(5 + r_cc)) + (x[,"n_10"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(3 + 2*r_cc)) + (x[,"n_03"] + x[,"n_42"])*(log10(pmax(1e-6,r_cc)) + log10(7 + 4*r_cc - 8*r_cc^2)) + (x[,"n_21"] + x[,"n_24"])*log10(3 + 5*r_cc - 5*r_cc^2) + (x[,"n_14"] + x[,"n_31"])*(log10(pmax(1e-6,r_cc)) + log10(20 + 9*r_cc - 2*r_cc^2)) + (x[,"n_22"] + x[,"n_23"])*log10(75 - 32*r_cc + 32*r_cc^2) + (x[,"n_12"] + x[,"n_33"])*log10(51 - 32*r_cc + 16*r_cc^2 - 16*r_cc^3) + (x[,"n_11"] + x[,"n_34"])*log10(27 - 32*r_cc + 3*r_cc^2 + 2*r_cc^3) + (x[,"n_02"] + x[,"n_43"])*log10(3 + 9*r_cc - 20*r_cc^2 + 8*r_cc^3) + (x[,"n_13"] + x[,"n_32"])*log10(19 + 48*r_cc - 32*r_cc^2 + 16*r_cc^3)
logL_cc <- 4*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - 2*x[,"n_45"])*log(5) + (x[,"n_20"] + x[,"n_25"])*log(7) + 3*(x[,"n_00"] + x[,"n_45"])*log(pmax(1e-6,1 - r_cc)) + (x[,"n_01"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_cc)) + log(6 - r_cc)) + 3*(x[,"n_05"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + (x[,"n_20"] + x[,"n_25"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_30"])*(log(5 - 2*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_04"] + x[,"n_41"])*(2*log(pmax(1e-6,r_cc)) + log(5 + r_cc)) + (x[,"n_10"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_cc)) + log(3 + 2*r_cc)) + (x[,"n_03"] + x[,"n_42"])*(log(pmax(1e-6,r_cc)) + log(7 + 4*r_cc - 8*r_cc^2)) + (x[,"n_21"] + x[,"n_24"])*log(3 + 5*r_cc - 5*r_cc^2) + (x[,"n_14"] + x[,"n_31"])*(log(pmax(1e-6,r_cc)) + log(20 + 9*r_cc - 2*r_cc^2)) + (x[,"n_22"] + x[,"n_23"])*log(75 - 32*r_cc + 32*r_cc^2) + (x[,"n_12"] + x[,"n_33"])*log(51 - 32*r_cc + 16*r_cc^2 - 16*r_cc^3) + (x[,"n_11"] + x[,"n_34"])*log(27 - 32*r_cc + 3*r_cc^2 + 2*r_cc^3) + (x[,"n_02"] + x[,"n_43"])*log(3 + 9*r_cc - 20*r_cc^2 + 8*r_cc^3) + (x[,"n_13"] + x[,"n_32"])*log(19 + 48*r_cc - 32*r_cc^2 + 16*r_cc^3)
logL_cm <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23) {
L <- (-3*n00 - 4*n01 - 4*n02 - 4*n03 - 4*n04 - 3*n05 - 3*n10 - 4*n11 - 4*n12 - 4*n13 - 4*n14 - 3*n15 - 3*n20 - 4*n21 - 4*n24 - 3*n25 - 3*n30 - 4*n31 - 4*n32 - 4*n33 - 4*n34 - 3*n35 - 3*n40 - 4*n41 - 4*n42 - 4*n43 - 4*n44 - 3*n45)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(log(3) + 2*log(5)) + (n00 + n45)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n05 + n40)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n41)*(log(pmax(1e-6,r)) + log(9 - 4*r - 2*r^2)) + (n22 + n23)*log(13/80 + r/75 - r^2/75) + (n20 + n25)*log(3 - r + r^2) + (n15 + n30)*(log(pmax(1e-6,r)) + log(4 - 3*r + 2*r^2)) + (n21 + n24)*log(69 - 14*r + 14*r^2) + (n13 + n32)*log(99 + 8*r + 48*r^2 - 32*r^3) + (n02 + n43)*log(18 - 25*r + 32*r^2 - 16*r^3) + (n11 + n34)*log(51 - 32*r + 6*r^2 - 4*r^3) + (n10 + n35)*log(3 - 4*r + 3*r^2 - 2*r^3) + (n01 + n44)*log(3 + 5*r - 10*r^2 + 2*r^3) + (n14 + n31)*log(21 + 32*r - 6*r^2 + 4*r^3) + (n03 + n42)*log(9 + 9*r - 16*r^2 + 16*r^3) + (n12 + n33)*log(123 - 8*r - 48*r^2 + 32*r^3)
return(L)}
interlogL_cm <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23) {
optimize(logL_cm,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_22"],x[,"n_23"])
LOD_cm <- 3*(x[,"n_05"] + x[,"n_40"])*log10(2) + 3*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_15"] + x[,"n_30"])*(-log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_35"])*(-log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_31"])*(2*log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_34"])*(2*log10(2) + 2*log10(3)) - (x[,"n_20"] + x[,"n_25"])*(-2*log10(2) + log10(11)) - (x[,"n_04"] + x[,"n_41"])*(-2*log10(2) + log10(13)) - (x[,"n_01"] + x[,"n_44"])*(-2*log10(2) + log10(13)) - (x[,"n_03"] + x[,"n_42"])*(-log10(2) + log10(23)) - (x[,"n_02"] + x[,"n_43"])*(-log10(2) + log10(23)) - (x[,"n_13"] + x[,"n_32"])*(log10(3) + log10(37)) - (x[,"n_12"] + x[,"n_33"])*(log10(3) + log10(37)) - (x[,"n_21"] + x[,"n_24"])*(-log10(2) + log10(131)) + (x[,"n_22"] + x[,"n_23"])*(4*log10(2) + log10(3) + 2*log10(5) - log10(199)) + (x[,"n_00"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + (x[,"n_05"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_cm)) + 2*log10(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_41"])*(log10(pmax(1e-6,r_cm)) + log10(9 - 4*r_cm - 2*r_cm^2)) + (x[,"n_22"] + x[,"n_23"])*log10(13/80 + r_cm/75 - r_cm^2/75) + (x[,"n_20"] + x[,"n_25"])*log10(3 - r_cm + r_cm^2) + (x[,"n_15"] + x[,"n_30"])*(log10(pmax(1e-6,r_cm)) + log10(4 - 3*r_cm + 2*r_cm^2)) + (x[,"n_21"] + x[,"n_24"])*log10(69 - 14*r_cm + 14*r_cm^2) + (x[,"n_13"] + x[,"n_32"])*log10(99 + 8*r_cm + 48*r_cm^2 - 32*r_cm^3) + (x[,"n_02"] + x[,"n_43"])*log10(18 - 25*r_cm + 32*r_cm^2 - 16*r_cm^3) + (x[,"n_11"] + x[,"n_34"])*log10(51 - 32*r_cm + 6*r_cm^2 - 4*r_cm^3) + (x[,"n_10"] + x[,"n_35"])*log10(3 - 4*r_cm + 3*r_cm^2 - 2*r_cm^3) + (x[,"n_01"] + x[,"n_44"])*log10(3 + 5*r_cm - 10*r_cm^2 + 2*r_cm^3) + (x[,"n_14"] + x[,"n_31"])*log10(21 + 32*r_cm - 6*r_cm^2 + 4*r_cm^3) + (x[,"n_03"] + x[,"n_42"])*log10(9 + 9*r_cm - 16*r_cm^2 + 16*r_cm^3) + (x[,"n_12"] + x[,"n_33"])*log10(123 - 8*r_cm - 48*r_cm^2 + 32*r_cm^3)
logL_cm <- (-3*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_05"] - 3*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_15"] - 3*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_24"] - 3*x[,"n_25"] - 3*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"] - 4*x[,"n_34"] - 3*x[,"n_35"] - 3*x[,"n_40"] - 4*x[,"n_41"] - 4*x[,"n_42"] - 4*x[,"n_43"] - 4*x[,"n_44"] - 3*x[,"n_45"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + (x[,"n_05"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_cm)) + 2*log(pmax(1e-6,r_cm))) + (x[,"n_04"] + x[,"n_41"])*(log(pmax(1e-6,r_cm)) + log(9 - 4*r_cm - 2*r_cm^2)) + (x[,"n_22"] + x[,"n_23"])*log(13/80 + r_cm/75 - r_cm^2/75) + (x[,"n_20"] + x[,"n_25"])*log(3 - r_cm + r_cm^2) + (x[,"n_15"] + x[,"n_30"])*(log(pmax(1e-6,r_cm)) + log(4 - 3*r_cm + 2*r_cm^2)) + (x[,"n_21"] + x[,"n_24"])*log(69 - 14*r_cm + 14*r_cm^2) + (x[,"n_13"] + x[,"n_32"])*log(99 + 8*r_cm + 48*r_cm^2 - 32*r_cm^3) + (x[,"n_02"] + x[,"n_43"])*log(18 - 25*r_cm + 32*r_cm^2 - 16*r_cm^3) + (x[,"n_11"] + x[,"n_34"])*log(51 - 32*r_cm + 6*r_cm^2 - 4*r_cm^3) + (x[,"n_10"] + x[,"n_35"])*log(3 - 4*r_cm + 3*r_cm^2 - 2*r_cm^3) + (x[,"n_01"] + x[,"n_44"])*log(3 + 5*r_cm - 10*r_cm^2 + 2*r_cm^3) + (x[,"n_14"] + x[,"n_31"])*log(21 + 32*r_cm - 6*r_cm^2 + 4*r_cm^3) + (x[,"n_03"] + x[,"n_42"])*log(9 + 9*r_cm - 16*r_cm^2 + 16*r_cm^3) + (x[,"n_12"] + x[,"n_33"])*log(123 - 8*r_cm - 48*r_cm^2 + 32*r_cm^3)
logL_cr <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n05 + n40)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n45)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n22 + n23)*log(63 + 16*r - 16*r^2) + (n01 + n44)*(log(pmax(1e-6,r)) + log(2 + 2*r - r^2)) + (n10 + n35)*(log(pmax(1e-6,r)) + log(4 - 5*r + 2*r^2)) + (n20 + n25)*log(3 - 5*r + 5*r^2) + (n21 + n24)*log(24 - 11*r + 11*r^2) + (n12 + n33)*log(39 - 16*r + 32*r^2 - 16*r^3) + (n03 + n42)*log(6 - 9*r + 12*r^2 - 8*r^3) + (n14 + n31)*log(15 - 4*r - 3*r^2 - 2*r^3) + (n15 + n30)*log(1 + r^2 - 2*r^3) + (n04 + n41)*log(3 - 3*r - r^2 + r^3) + (n11 + n34)*log(6 + 16*r - 9*r^2 + 2*r^3) + (n02 + n43)*log(1 + 9*r - 12*r^2 + 8*r^3) + (n13 + n32)*log(39 - 16*r^2 + 16*r^3)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cr <- 3*(x[,"n_05"] + x[,"n_40"])*log10(2) + 3*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_31"])*(2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_34"])*(2*log10(2) + log10(3)) - (x[,"n_20"] + x[,"n_25"])*(-2*log10(2) + log10(7)) - (x[,"n_03"] + x[,"n_42"])*(-log10(2) + log10(7)) - (x[,"n_02"] + x[,"n_43"])*(-log10(2) + log10(7)) - (x[,"n_04"] + x[,"n_41"])*(-3*log10(2) + log10(11)) - (x[,"n_01"] + x[,"n_44"])*(-3*log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_24"])*(-2*log10(2) + log10(5) + log10(17)) - (x[,"n_13"] + x[,"n_32"])*log10(37) - (x[,"n_12"] + x[,"n_33"])*log10(37) - (x[,"n_22"] + x[,"n_23"])*log10(67) + (x[,"n_05"] + x[,"n_40"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cr)) + 2*log10(pmax(1e-6,r_cr))) + (x[,"n_22"] + x[,"n_23"])*log10(63 + 16*r_cr - 16*r_cr^2) + (x[,"n_01"] + x[,"n_44"])*(log10(pmax(1e-6,r_cr)) + log10(2 + 2*r_cr - r_cr^2)) + (x[,"n_10"] + x[,"n_35"])*(log10(pmax(1e-6,r_cr)) + log10(4 - 5*r_cr + 2*r_cr^2)) + (x[,"n_20"] + x[,"n_25"])*log10(3 - 5*r_cr + 5*r_cr^2) + (x[,"n_21"] + x[,"n_24"])*log10(24 - 11*r_cr + 11*r_cr^2) + (x[,"n_12"] + x[,"n_33"])*log10(39 - 16*r_cr + 32*r_cr^2 - 16*r_cr^3) + (x[,"n_03"] + x[,"n_42"])*log10(6 - 9*r_cr + 12*r_cr^2 - 8*r_cr^3) + (x[,"n_14"] + x[,"n_31"])*log10(15 - 4*r_cr - 3*r_cr^2 - 2*r_cr^3) + (x[,"n_15"] + x[,"n_30"])*log10(1 + r_cr^2 - 2*r_cr^3) + (x[,"n_04"] + x[,"n_41"])*log10(3 - 3*r_cr - r_cr^2 + r_cr^3) + (x[,"n_11"] + x[,"n_34"])*log10(6 + 16*r_cr - 9*r_cr^2 + 2*r_cr^3) + (x[,"n_02"] + x[,"n_43"])*log10(1 + 9*r_cr - 12*r_cr^2 + 8*r_cr^3) + (x[,"n_13"] + x[,"n_32"])*log10(39 - 16*r_cr^2 + 16*r_cr^3)
logL_cr <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_05"] + x[,"n_40"])*(2*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cr)) + 2*log(pmax(1e-6,r_cr))) + (x[,"n_22"] + x[,"n_23"])*log(63 + 16*r_cr - 16*r_cr^2) + (x[,"n_01"] + x[,"n_44"])*(log(pmax(1e-6,r_cr)) + log(2 + 2*r_cr - r_cr^2)) + (x[,"n_10"] + x[,"n_35"])*(log(pmax(1e-6,r_cr)) + log(4 - 5*r_cr + 2*r_cr^2)) + (x[,"n_20"] + x[,"n_25"])*log(3 - 5*r_cr + 5*r_cr^2) + (x[,"n_21"] + x[,"n_24"])*log(24 - 11*r_cr + 11*r_cr^2) + (x[,"n_12"] + x[,"n_33"])*log(39 - 16*r_cr + 32*r_cr^2 - 16*r_cr^3) + (x[,"n_03"] + x[,"n_42"])*log(6 - 9*r_cr + 12*r_cr^2 - 8*r_cr^3) + (x[,"n_14"] + x[,"n_31"])*log(15 - 4*r_cr - 3*r_cr^2 - 2*r_cr^3) + (x[,"n_15"] + x[,"n_30"])*log(1 + r_cr^2 - 2*r_cr^3) + (x[,"n_04"] + x[,"n_41"])*log(3 - 3*r_cr - r_cr^2 + r_cr^3) + (x[,"n_11"] + x[,"n_34"])*log(6 + 16*r_cr - 9*r_cr^2 + 2*r_cr^3) + (x[,"n_02"] + x[,"n_43"])*log(1 + 9*r_cr - 12*r_cr^2 + 8*r_cr^3) + (x[,"n_13"] + x[,"n_32"])*log(39 - 16*r_cr^2 + 16*r_cr^3)
logL_rc <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n00 + n45)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n05 + n40)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n22 + n23)*log(63 + 16*r - 16*r^2) + (n04 + n41)*(log(pmax(1e-6,r)) + log(2 + 2*r - r^2)) + (n15 + n30)*(log(pmax(1e-6,r)) + log(4 - 5*r + 2*r^2)) + (n20 + n25)*log(3 - 5*r + 5*r^2) + (n21 + n24)*log(24 - 11*r + 11*r^2) + (n13 + n32)*log(39 - 16*r + 32*r^2 - 16*r^3) + (n02 + n43)*log(6 - 9*r + 12*r^2 - 8*r^3) + (n11 + n34)*log(15 - 4*r - 3*r^2 - 2*r^3) + (n10 + n35)*log(1 + r^2 - 2*r^3) + (n01 + n44)*log(3 - 3*r - r^2 + r^3) + (n14 + n31)*log(6 + 16*r - 9*r^2 + 2*r^3) + (n03 + n42)*log(1 + 9*r - 12*r^2 + 8*r^3) + (n12 + n33)*log(39 - 16*r^2 + 16*r^3)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rc <- 3*(x[,"n_05"] + x[,"n_40"])*log10(2) + 3*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_31"])*(2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_34"])*(2*log10(2) + log10(3)) - (x[,"n_20"] + x[,"n_25"])*(-2*log10(2) + log10(7)) - (x[,"n_03"] + x[,"n_42"])*(-log10(2) + log10(7)) - (x[,"n_02"] + x[,"n_43"])*(-log10(2) + log10(7)) - (x[,"n_04"] + x[,"n_41"])*(-3*log10(2) + log10(11)) - (x[,"n_01"] + x[,"n_44"])*(-3*log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_24"])*(-2*log10(2) + log10(5) + log10(17)) - (x[,"n_13"] + x[,"n_32"])*log10(37) - (x[,"n_12"] + x[,"n_33"])*log10(37) - (x[,"n_22"] + x[,"n_23"])*log10(67) + (x[,"n_00"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc))) + (x[,"n_05"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_rc)) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_22"] + x[,"n_23"])*log10(63 + 16*r_rc - 16*r_rc^2) + (x[,"n_04"] + x[,"n_41"])*(log10(pmax(1e-6,r_rc)) + log10(2 + 2*r_rc - r_rc^2)) + (x[,"n_15"] + x[,"n_30"])*(log10(pmax(1e-6,r_rc)) + log10(4 - 5*r_rc + 2*r_rc^2)) + (x[,"n_20"] + x[,"n_25"])*log10(3 - 5*r_rc + 5*r_rc^2) + (x[,"n_21"] + x[,"n_24"])*log10(24 - 11*r_rc + 11*r_rc^2) + (x[,"n_13"] + x[,"n_32"])*log10(39 - 16*r_rc + 32*r_rc^2 - 16*r_rc^3) + (x[,"n_02"] + x[,"n_43"])*log10(6 - 9*r_rc + 12*r_rc^2 - 8*r_rc^3) + (x[,"n_11"] + x[,"n_34"])*log10(15 - 4*r_rc - 3*r_rc^2 - 2*r_rc^3) + (x[,"n_10"] + x[,"n_35"])*log10(1 + r_rc^2 - 2*r_rc^3) + (x[,"n_01"] + x[,"n_44"])*log10(3 - 3*r_rc - r_rc^2 + r_rc^3) + (x[,"n_14"] + x[,"n_31"])*log10(6 + 16*r_rc - 9*r_rc^2 + 2*r_rc^3) + (x[,"n_03"] + x[,"n_42"])*log10(1 + 9*r_rc - 12*r_rc^2 + 8*r_rc^3) + (x[,"n_12"] + x[,"n_33"])*log10(39 - 16*r_rc^2 + 16*r_rc^3)
logL_rc <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_00"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc))) + (x[,"n_05"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_rc)) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_22"] + x[,"n_23"])*log(63 + 16*r_rc - 16*r_rc^2) + (x[,"n_04"] + x[,"n_41"])*(log(pmax(1e-6,r_rc)) + log(2 + 2*r_rc - r_rc^2)) + (x[,"n_15"] + x[,"n_30"])*(log(pmax(1e-6,r_rc)) + log(4 - 5*r_rc + 2*r_rc^2)) + (x[,"n_20"] + x[,"n_25"])*log(3 - 5*r_rc + 5*r_rc^2) + (x[,"n_21"] + x[,"n_24"])*log(24 - 11*r_rc + 11*r_rc^2) + (x[,"n_13"] + x[,"n_32"])*log(39 - 16*r_rc + 32*r_rc^2 - 16*r_rc^3) + (x[,"n_02"] + x[,"n_43"])*log(6 - 9*r_rc + 12*r_rc^2 - 8*r_rc^3) + (x[,"n_11"] + x[,"n_34"])*log(15 - 4*r_rc - 3*r_rc^2 - 2*r_rc^3) + (x[,"n_10"] + x[,"n_35"])*log(1 + r_rc^2 - 2*r_rc^3) + (x[,"n_01"] + x[,"n_44"])*log(3 - 3*r_rc - r_rc^2 + r_rc^3) + (x[,"n_14"] + x[,"n_31"])*log(6 + 16*r_rc - 9*r_rc^2 + 2*r_rc^3) + (x[,"n_03"] + x[,"n_42"])*log(1 + 9*r_rc - 12*r_rc^2 + 8*r_rc^3) + (x[,"n_12"] + x[,"n_33"])*log(39 - 16*r_rc^2 + 16*r_rc^3)
logL_rm <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23) {
L <- (-3*n00 - 4*n01 - 4*n02 - 4*n03 - 4*n04 - 3*n05 - 3*n10 - 4*n11 - 4*n12 - 4*n13 - 4*n14 - 3*n15 - 3*n20 - 4*n21 - 4*n24 - 3*n25 - 3*n30 - 4*n31 - 4*n32 - 4*n33 - 4*n34 - 3*n35 - 3*n40 - 4*n41 - 4*n42 - 4*n43 - 4*n44 - 3*n45)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(log(3) + 2*log(5)) + (n05 + n40)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n45)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n44)*(log(pmax(1e-6,r)) + log(9 - 4*r - 2*r^2)) + (n22 + n23)*log(13/80 + r/75 - r^2/75) + (n20 + n25)*log(3 - r + r^2) + (n10 + n35)*(log(pmax(1e-6,r)) + log(4 - 3*r + 2*r^2)) + (n21 + n24)*log(69 - 14*r + 14*r^2) + (n12 + n33)*log(99 + 8*r + 48*r^2 - 32*r^3) + (n03 + n42)*log(18 - 25*r + 32*r^2 - 16*r^3) + (n14 + n31)*log(51 - 32*r + 6*r^2 - 4*r^3) + (n15 + n30)*log(3 - 4*r + 3*r^2 - 2*r^3) + (n04 + n41)*log(3 + 5*r - 10*r^2 + 2*r^3) + (n11 + n34)*log(21 + 32*r - 6*r^2 + 4*r^3) + (n02 + n43)*log(9 + 9*r - 16*r^2 + 16*r^3) + (n13 + n32)*log(123 - 8*r - 48*r^2 + 32*r^3)
return(L)}
interlogL_rm <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23) {
optimize(logL_rm,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_22"],x[,"n_23"])
LOD_rm <- 3*(x[,"n_05"] + x[,"n_40"])*log10(2) + 3*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_15"] + x[,"n_30"])*(-log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_35"])*(-log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_31"])*(2*log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_34"])*(2*log10(2) + 2*log10(3)) - (x[,"n_20"] + x[,"n_25"])*(-2*log10(2) + log10(11)) - (x[,"n_04"] + x[,"n_41"])*(-2*log10(2) + log10(13)) - (x[,"n_01"] + x[,"n_44"])*(-2*log10(2) + log10(13)) - (x[,"n_03"] + x[,"n_42"])*(-log10(2) + log10(23)) - (x[,"n_02"] + x[,"n_43"])*(-log10(2) + log10(23)) - (x[,"n_13"] + x[,"n_32"])*(log10(3) + log10(37)) - (x[,"n_12"] + x[,"n_33"])*(log10(3) + log10(37)) - (x[,"n_21"] + x[,"n_24"])*(-log10(2) + log10(131)) + (x[,"n_22"] + x[,"n_23"])*(4*log10(2) + log10(3) + 2*log10(5) - log10(199)) + (x[,"n_05"] + x[,"n_40"])*(2*log10(pmax(1e-6,1 - r_rm)) + log10(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_rm)) + 2*log10(pmax(1e-6,r_rm))) + (x[,"n_01"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm)) + log10(9 - 4*r_rm - 2*r_rm^2)) + (x[,"n_22"] + x[,"n_23"])*log10(13/80 + r_rm/75 - r_rm^2/75) + (x[,"n_20"] + x[,"n_25"])*log10(3 - r_rm + r_rm^2) + (x[,"n_10"] + x[,"n_35"])*(log10(pmax(1e-6,r_rm)) + log10(4 - 3*r_rm + 2*r_rm^2)) + (x[,"n_21"] + x[,"n_24"])*log10(69 - 14*r_rm + 14*r_rm^2) + (x[,"n_12"] + x[,"n_33"])*log10(99 + 8*r_rm + 48*r_rm^2 - 32*r_rm^3) + (x[,"n_03"] + x[,"n_42"])*log10(18 - 25*r_rm + 32*r_rm^2 - 16*r_rm^3) + (x[,"n_14"] + x[,"n_31"])*log10(51 - 32*r_rm + 6*r_rm^2 - 4*r_rm^3) + (x[,"n_15"] + x[,"n_30"])*log10(3 - 4*r_rm + 3*r_rm^2 - 2*r_rm^3) + (x[,"n_04"] + x[,"n_41"])*log10(3 + 5*r_rm - 10*r_rm^2 + 2*r_rm^3) + (x[,"n_11"] + x[,"n_34"])*log10(21 + 32*r_rm - 6*r_rm^2 + 4*r_rm^3) + (x[,"n_02"] + x[,"n_43"])*log10(9 + 9*r_rm - 16*r_rm^2 + 16*r_rm^3) + (x[,"n_13"] + x[,"n_32"])*log10(123 - 8*r_rm - 48*r_rm^2 + 32*r_rm^3)
logL_rm <- (-3*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_05"] - 3*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_15"] - 3*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_24"] - 3*x[,"n_25"] - 3*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"] - 4*x[,"n_34"] - 3*x[,"n_35"] - 3*x[,"n_40"] - 4*x[,"n_41"] - 4*x[,"n_42"] - 4*x[,"n_43"] - 4*x[,"n_44"] - 3*x[,"n_45"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(log(3) + 2*log(5)) + (x[,"n_05"] + x[,"n_40"])*(2*log(pmax(1e-6,1 - r_rm)) + log(pmax(1e-6,r_rm))) + (x[,"n_00"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_rm)) + 2*log(pmax(1e-6,r_rm))) + (x[,"n_01"] + x[,"n_44"])*(log(pmax(1e-6,r_rm)) + log(9 - 4*r_rm - 2*r_rm^2)) + (x[,"n_22"] + x[,"n_23"])*log(13/80 + r_rm/75 - r_rm^2/75) + (x[,"n_20"] + x[,"n_25"])*log(3 - r_rm + r_rm^2) + (x[,"n_10"] + x[,"n_35"])*(log(pmax(1e-6,r_rm)) + log(4 - 3*r_rm + 2*r_rm^2)) + (x[,"n_21"] + x[,"n_24"])*log(69 - 14*r_rm + 14*r_rm^2) + (x[,"n_12"] + x[,"n_33"])*log(99 + 8*r_rm + 48*r_rm^2 - 32*r_rm^3) + (x[,"n_03"] + x[,"n_42"])*log(18 - 25*r_rm + 32*r_rm^2 - 16*r_rm^3) + (x[,"n_14"] + x[,"n_31"])*log(51 - 32*r_rm + 6*r_rm^2 - 4*r_rm^3) + (x[,"n_15"] + x[,"n_30"])*log(3 - 4*r_rm + 3*r_rm^2 - 2*r_rm^3) + (x[,"n_04"] + x[,"n_41"])*log(3 + 5*r_rm - 10*r_rm^2 + 2*r_rm^3) + (x[,"n_11"] + x[,"n_34"])*log(21 + 32*r_rm - 6*r_rm^2 + 4*r_rm^3) + (x[,"n_02"] + x[,"n_43"])*log(9 + 9*r_rm - 16*r_rm^2 + 16*r_rm^3) + (x[,"n_13"] + x[,"n_32"])*log(123 - 8*r_rm - 48*r_rm^2 + 32*r_rm^3)
logL_rr <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- 4*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n05 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n20 - n21 - 2*n22 - 2*n23 - n24 - 2*n25 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n40 - 2*n41 - 2*n42 - 2*n43 - 2*n44 - 2*n45)*log(5) + (n20 + n25)*log(7) + 3*(n05 + n40)*log(pmax(1e-6,1 - r)) + (n04 + n41)*(2*log(pmax(1e-6,1 - r)) + log(6 - r)) + 3*(n00 + n45)*log(pmax(1e-6,r)) + (n20 + n25)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n35)*(log(5 - 2*r) + 2*log(pmax(1e-6,r))) + (n01 + n44)*(2*log(pmax(1e-6,r)) + log(5 + r)) + (n15 + n30)*(2*log(pmax(1e-6,1 - r)) + log(3 + 2*r)) + (n02 + n43)*(log(pmax(1e-6,r)) + log(7 + 4*r - 8*r^2)) + (n21 + n24)*log(3 + 5*r - 5*r^2) + (n11 + n34)*(log(pmax(1e-6,r)) + log(20 + 9*r - 2*r^2)) + (n22 + n23)*log(75 - 32*r + 32*r^2) + (n13 + n32)*log(51 - 32*r + 16*r^2 - 16*r^3) + (n14 + n31)*log(27 - 32*r + 3*r^2 + 2*r^3) + (n03 + n42)*log(3 + 9*r - 20*r^2 + 8*r^3) + (n12 + n33)*log(19 + 48*r - 32*r^2 + 16*r^3)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rr <- 2*(x[,"n_20"] + x[,"n_25"])*log10(2) + 3*(x[,"n_05"] + x[,"n_40"])*log10(2) + 3*(x[,"n_00"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_31"])*(2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_34"])*(2*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_42"])*(-log10(2) + log10(7)) - (x[,"n_02"] + x[,"n_43"])*(-log10(2) + log10(7)) - (x[,"n_04"] + x[,"n_41"])*(-3*log10(2) + log10(11)) - (x[,"n_01"] + x[,"n_44"])*(-3*log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_24"])*(-2*log10(2) + log10(17)) - (x[,"n_13"] + x[,"n_32"])*log10(37) - (x[,"n_12"] + x[,"n_33"])*log10(37) - (x[,"n_22"] + x[,"n_23"])*log10(67) + 3*(x[,"n_05"] + x[,"n_40"])*log10(pmax(1e-6,1 - r_rr)) + (x[,"n_04"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(6 - r_rr)) + 3*(x[,"n_00"] + x[,"n_45"])*log10(pmax(1e-6,r_rr)) + (x[,"n_20"] + x[,"n_25"])*(log10(pmax(1e-6,1 - r_rr)) + log10(pmax(1e-6,r_rr))) + (x[,"n_10"] + x[,"n_35"])*(log10(5 - 2*r_rr) + 2*log10(pmax(1e-6,r_rr))) + (x[,"n_01"] + x[,"n_44"])*(2*log10(pmax(1e-6,r_rr)) + log10(5 + r_rr)) + (x[,"n_15"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(3 + 2*r_rr)) + (x[,"n_02"] + x[,"n_43"])*(log10(pmax(1e-6,r_rr)) + log10(7 + 4*r_rr - 8*r_rr^2)) + (x[,"n_21"] + x[,"n_24"])*log10(3 + 5*r_rr - 5*r_rr^2) + (x[,"n_11"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(20 + 9*r_rr - 2*r_rr^2)) + (x[,"n_22"] + x[,"n_23"])*log10(75 - 32*r_rr + 32*r_rr^2) + (x[,"n_13"] + x[,"n_32"])*log10(51 - 32*r_rr + 16*r_rr^2 - 16*r_rr^3) + (x[,"n_14"] + x[,"n_31"])*log10(27 - 32*r_rr + 3*r_rr^2 + 2*r_rr^3) + (x[,"n_03"] + x[,"n_42"])*log10(3 + 9*r_rr - 20*r_rr^2 + 8*r_rr^3) + (x[,"n_12"] + x[,"n_33"])*log10(19 + 48*r_rr - 32*r_rr^2 + 16*r_rr^3)
logL_rr <- 4*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - 2*x[,"n_45"])*log(5) + (x[,"n_20"] + x[,"n_25"])*log(7) + 3*(x[,"n_05"] + x[,"n_40"])*log(pmax(1e-6,1 - r_rr)) + (x[,"n_04"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_rr)) + log(6 - r_rr)) + 3*(x[,"n_00"] + x[,"n_45"])*log(pmax(1e-6,r_rr)) + (x[,"n_20"] + x[,"n_25"])*(log(pmax(1e-6,1 - r_rr)) + log(pmax(1e-6,r_rr))) + (x[,"n_10"] + x[,"n_35"])*(log(5 - 2*r_rr) + 2*log(pmax(1e-6,r_rr))) + (x[,"n_01"] + x[,"n_44"])*(2*log(pmax(1e-6,r_rr)) + log(5 + r_rr)) + (x[,"n_15"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_rr)) + log(3 + 2*r_rr)) + (x[,"n_02"] + x[,"n_43"])*(log(pmax(1e-6,r_rr)) + log(7 + 4*r_rr - 8*r_rr^2)) + (x[,"n_21"] + x[,"n_24"])*log(3 + 5*r_rr - 5*r_rr^2) + (x[,"n_11"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(20 + 9*r_rr - 2*r_rr^2)) + (x[,"n_22"] + x[,"n_23"])*log(75 - 32*r_rr + 32*r_rr^2) + (x[,"n_13"] + x[,"n_32"])*log(51 - 32*r_rr + 16*r_rr^2 - 16*r_rr^3) + (x[,"n_14"] + x[,"n_31"])*log(27 - 32*r_rr + 3*r_rr^2 + 2*r_rr^3) + (x[,"n_03"] + x[,"n_42"])*log(3 + 9*r_rr - 20*r_rr^2 + 8*r_rr^3) + (x[,"n_12"] + x[,"n_33"])*log(19 + 48*r_rr - 32*r_rr^2 + 16*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_3.3 <- function(x){
logL_cc <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n05 - 4*n06 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n16 - 3*n20 - 3*n21 - 3*n22 - 2*n23 - 3*n24 - 3*n25 - 3*n26 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n36 - 4*n40 - 3*n41 - 4*n42 - 3*n43 - 4*n44 - 3*n45 - 4*n46)*log(2) + (n01 + n02 + n04 + n05 + n11 + n12 + n14 + n15 + n20 + n21 + n22 + n23 + n24 + n25 + n26 + n31 + n32 + n34 + n35 + n41 + n42 + n44 + n45)*log(3) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n05 - 2*n06 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n16 - 2*n20 - n21 - 2*n22 - 2*n23 - 2*n24 - n25 - 2*n26 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n36 - 2*n40 - 2*n41 - 2*n42 - 2*n43 - 2*n44 - 2*n45 - 2*n46)*log(5) + 3*(n01 + n45)*log(pmax(1e-6,1 - r)) + 4*(n00 + n46)*log(pmax(1e-6,1 - r)) + 4*(n06 + n40)*log(pmax(1e-6,r)) + 3*(n05 + n41)*log(pmax(1e-6,r)) + (n21 + n25)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n36)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n26)*(2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n15 + n31)*(log(2 - r) + 2*log(pmax(1e-6,r))) + (n16 + n30)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n11 + n35)*(2*log(pmax(1e-6,1 - r)) + log(1 + r)) + (n04 + n42)*(2*log(pmax(1e-6,r)) + log(4 - 3*r^2)) + (n02 + n44)*(2*log(pmax(1e-6,1 - r)) + log(1 + 6*r - 3*r^2)) + (n14 + n32)*(log(pmax(1e-6,r)) + log(4 + r - 3*r^2 + 3*r^3)) + (n03 + n43)*(log(pmax(1e-6,r)) + log(5 + 3*r - 16*r^2 + 8*r^3)) + (n13 + n33)*log(7 + 17*r - 33*r^2 + 32*r^3 - 16*r^4) + (n22 + n24)*log(6 + 8*r - 17*r^2 + 18*r^3 - 9*r^4) + (n12 + n34)*log(5 - 9*r + 10*r^2 - 9*r^3 + 3*r^4) + n23*log(9 - 13*r + 21*r^2 - 16*r^3 + 8*r^4)
return(L)}
interlogL_cc <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_cc <- 2*(x[,"n_21"] + x[,"n_25"])*log10(2) + 4*(x[,"n_20"] + x[,"n_26"])*log10(2) + 4*(x[,"n_16"] + x[,"n_30"])*log10(2) + 4*(x[,"n_10"] + x[,"n_36"])*log10(2) + 4*(x[,"n_06"] + x[,"n_40"])*log10(2) + 3*(x[,"n_05"] + x[,"n_41"])*log10(2) + 3*(x[,"n_01"] + x[,"n_45"])*log10(2) + 4*(x[,"n_00"] + x[,"n_46"])*log10(2) - (x[,"n_15"] + x[,"n_31"])*(-3*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_35"])*(-3*log10(2) + log10(3)) - x[,"n_23"]*(-2*log10(2) + 2*log10(5)) - (x[,"n_03"] + x[,"n_43"])*(-2*log10(2) + log10(7)) - (x[,"n_14"] + x[,"n_32"])*(-4*log10(2) + log10(3) + log10(11)) - (x[,"n_12"] + x[,"n_34"])*(-4*log10(2) + log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_42"])*(-4*log10(2) + log10(13)) - (x[,"n_02"] + x[,"n_44"])*(-4*log10(2) + log10(13)) - (x[,"n_22"] + x[,"n_24"])*(-4*log10(2) + log10(7) + log10(17)) - (x[,"n_13"] + x[,"n_33"])*(-2*log10(2) + log10(41)) + 3*(x[,"n_01"] + x[,"n_45"])*log10(pmax(1e-6,1 - r_cc)) + 4*(x[,"n_00"] + x[,"n_46"])*log10(pmax(1e-6,1 - r_cc)) + 4*(x[,"n_06"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + 3*(x[,"n_05"] + x[,"n_41"])*log10(pmax(1e-6,r_cc)) + (x[,"n_21"] + x[,"n_25"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_36"])*(3*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_20"] + x[,"n_26"])*(2*log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_31"])*(log10(2 - r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_16"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_cc)) + 3*log10(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(1 + r_cc)) + (x[,"n_04"] + x[,"n_42"])*(2*log10(pmax(1e-6,r_cc)) + log10(4 - 3*r_cc^2)) + (x[,"n_02"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(1 + 6*r_cc - 3*r_cc^2)) + (x[,"n_14"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(4 + r_cc - 3*r_cc^2 + 3*r_cc^3)) + (x[,"n_03"] + x[,"n_43"])*(log10(pmax(1e-6,r_cc)) + log10(5 + 3*r_cc - 16*r_cc^2 + 8*r_cc^3)) + (x[,"n_13"] + x[,"n_33"])*log10(7 + 17*r_cc - 33*r_cc^2 + 32*r_cc^3 - 16*r_cc^4) + (x[,"n_22"] + x[,"n_24"])*log10(6 + 8*r_cc - 17*r_cc^2 + 18*r_cc^3 - 9*r_cc^4) + (x[,"n_12"] + x[,"n_34"])*log10(5 - 9*r_cc + 10*r_cc^2 - 9*r_cc^3 + 3*r_cc^4) + x[,"n_23"]*log10(9 - 13*r_cc + 21*r_cc^2 - 16*r_cc^3 + 8*r_cc^4)
logL_cc <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_05"] - 4*x[,"n_06"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_16"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"] - 3*x[,"n_25"] - 3*x[,"n_26"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_36"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 3*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_04"] + x[,"n_05"] + x[,"n_11"] + x[,"n_12"] + x[,"n_14"] + x[,"n_15"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_25"] + x[,"n_26"] + x[,"n_31"] + x[,"n_32"] + x[,"n_34"] + x[,"n_35"] + x[,"n_41"] + x[,"n_42"] + x[,"n_44"] + x[,"n_45"])*log(3) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_06"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_16"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_25"] - 2*x[,"n_26"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_36"] - 2*x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - 2*x[,"n_45"] - 2*x[,"n_46"])*log(5) + 3*(x[,"n_01"] + x[,"n_45"])*log(pmax(1e-6,1 - r_cc)) + 4*(x[,"n_00"] + x[,"n_46"])*log(pmax(1e-6,1 - r_cc)) + 4*(x[,"n_06"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + 3*(x[,"n_05"] + x[,"n_41"])*log(pmax(1e-6,r_cc)) + (x[,"n_21"] + x[,"n_25"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_10"] + x[,"n_36"])*(3*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_20"] + x[,"n_26"])*(2*log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_31"])*(log(2 - r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_16"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_cc)) + 3*log(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_cc)) + log(1 + r_cc)) + (x[,"n_04"] + x[,"n_42"])*(2*log(pmax(1e-6,r_cc)) + log(4 - 3*r_cc^2)) + (x[,"n_02"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_cc)) + log(1 + 6*r_cc - 3*r_cc^2)) + (x[,"n_14"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(4 + r_cc - 3*r_cc^2 + 3*r_cc^3)) + (x[,"n_03"] + x[,"n_43"])*(log(pmax(1e-6,r_cc)) + log(5 + 3*r_cc - 16*r_cc^2 + 8*r_cc^3)) + (x[,"n_13"] + x[,"n_33"])*log(7 + 17*r_cc - 33*r_cc^2 + 32*r_cc^3 - 16*r_cc^4) + (x[,"n_22"] + x[,"n_24"])*log(6 + 8*r_cc - 17*r_cc^2 + 18*r_cc^3 - 9*r_cc^4) + (x[,"n_12"] + x[,"n_34"])*log(5 - 9*r_cc + 10*r_cc^2 - 9*r_cc^3 + 3*r_cc^4) + x[,"n_23"]*log(9 - 13*r_cc + 21*r_cc^2 - 16*r_cc^3 + 8*r_cc^4)
logL_cm2c <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n03 - 4*n04 - 4*n05 - 4*n06 - 4*n10 - 3*n11 - 4*n12 - 3*n13 - 4*n14 - 3*n15 - 4*n16 - 4*n20 - 4*n21 - 4*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n26 - 4*n30 - 3*n31 - 4*n32 - 3*n33 - 4*n34 - 3*n35 - 4*n36 - 4*n40 - 4*n41 - 4*n42 - 4*n43 - 4*n44 - 4*n45 - 4*n46)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(log(3) + 2*log(5)) + (n21 + n25)*log(7) + (n00 + n46)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n05 + n41)*(log(7 - 4*r) + 2*log(pmax(1e-6,r))) + (n06 + n40)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n01 + n45)*(2*log(pmax(1e-6,1 - r)) + log(3 + 4*r)) + (n21 + n25)*log(3 + 2*r - 2*r^2) + (n16 + n30)*(2*log(pmax(1e-6,r)) + log(5 - 6*r + 4*r^2)) + (n15 + n31)*(log(pmax(1e-6,r)) + log(14 - 3*r + 4*r^2)) + (n10 + n36)*(2*log(pmax(1e-6,1 - r)) + log(3 - 2*r + 4*r^2)) + (n20 + n26)*(log(pmax(1e-6,r)) + log(7 - 13*r + 12*r^2 - 6*r^3)) + (n11 + n35)*log(15 - 20*r + 9*r^2 - 4*r^3) + (n04 + n42)*(log(pmax(1e-6,r)) + log(15 + r - 7*r^2 + 9*r^3)) + n23*log(111 + 10*r - 58*r^2 + 96*r^3 - 48*r^4) + (n14 + n32)*log(39 + 76*r - 27*r^2 + 50*r^3 - 36*r^4) + (n12 + n34)*log(102 - 28*r - 93*r^2 + 94*r^3 - 36*r^4) + (n03 + n43)*log(9 + 18*r - 34*r^2 + 32*r^3 - 16*r^4) + (n02 + n44)*log(18 - 32*r + 34*r^2 - 29*r^3 + 9*r^4) + (n13 + n33)*log(63 - 14*r + 46*r^2 - 64*r^3 + 32*r^4) + (n22 + n24)*log(138 - 31*r + 85*r^2 - 108*r^3 + 54*r^4)
return(L)}
interlogL_cm2c <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_cm2c,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_cm2c <- 4*(x[,"n_06"] + x[,"n_40"])*log10(2) + 4*(x[,"n_00"] + x[,"n_46"])*log10(2) - (x[,"n_16"] + x[,"n_30"])*(-2*log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_36"])*(-2*log10(2) + log10(3)) - (x[,"n_15"] + x[,"n_31"])*(-2*log10(2) + 3*log10(3)) - (x[,"n_11"] + x[,"n_35"])*(-2*log10(2) + 3*log10(3)) - (x[,"n_05"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_45"])*(-2*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_43"])*(-log10(2) + 2*log10(5)) - (x[,"n_21"] + x[,"n_25"])*(-log10(2) + log10(7)) - (x[,"n_20"] + x[,"n_26"])*(-3*log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_32"])*(-2*log10(2) + 3*log10(3) + log10(11)) - (x[,"n_12"] + x[,"n_34"])*(-2*log10(2) + 3*log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_42"])*(-4*log10(2) + log10(7) + log10(17)) - (x[,"n_02"] + x[,"n_44"])*(-4*log10(2) + log10(7) + log10(17)) - x[,"n_23"]*(-log10(2) + log10(13) + log10(17)) - (x[,"n_13"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(41)) - (x[,"n_22"] + x[,"n_24"])*(-3*log10(2) + log10(1069)) + (x[,"n_00"] + x[,"n_46"])*(3*log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c))) + (x[,"n_05"] + x[,"n_41"])*(log10(7 - 4*r_cm2c) + 2*log10(pmax(1e-6,r_cm2c))) + (x[,"n_06"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_cm2c)) + 3*log10(pmax(1e-6,r_cm2c))) + (x[,"n_01"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_cm2c)) + log10(3 + 4*r_cm2c)) + (x[,"n_21"] + x[,"n_25"])*log10(3 + 2*r_cm2c - 2*r_cm2c^2) + (x[,"n_16"] + x[,"n_30"])*(2*log10(pmax(1e-6,r_cm2c)) + log10(5 - 6*r_cm2c + 4*r_cm2c^2)) + (x[,"n_15"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm2c)) + log10(14 - 3*r_cm2c + 4*r_cm2c^2)) + (x[,"n_10"] + x[,"n_36"])*(2*log10(pmax(1e-6,1 - r_cm2c)) + log10(3 - 2*r_cm2c + 4*r_cm2c^2)) + (x[,"n_20"] + x[,"n_26"])*(log10(pmax(1e-6,r_cm2c)) + log10(7 - 13*r_cm2c + 12*r_cm2c^2 - 6*r_cm2c^3)) + (x[,"n_11"] + x[,"n_35"])*log10(15 - 20*r_cm2c + 9*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_04"] + x[,"n_42"])*(log10(pmax(1e-6,r_cm2c)) + log10(15 + r_cm2c - 7*r_cm2c^2 + 9*r_cm2c^3)) + x[,"n_23"]*log10(111 + 10*r_cm2c - 58*r_cm2c^2 + 96*r_cm2c^3 - 48*r_cm2c^4) + (x[,"n_14"] + x[,"n_32"])*log10(39 + 76*r_cm2c - 27*r_cm2c^2 + 50*r_cm2c^3 - 36*r_cm2c^4) + (x[,"n_12"] + x[,"n_34"])*log10(102 - 28*r_cm2c - 93*r_cm2c^2 + 94*r_cm2c^3 - 36*r_cm2c^4) + (x[,"n_03"] + x[,"n_43"])*log10(9 + 18*r_cm2c - 34*r_cm2c^2 + 32*r_cm2c^3 - 16*r_cm2c^4) + (x[,"n_02"] + x[,"n_44"])*log10(18 - 32*r_cm2c + 34*r_cm2c^2 - 29*r_cm2c^3 + 9*r_cm2c^4) + (x[,"n_13"] + x[,"n_33"])*log10(63 - 14*r_cm2c + 46*r_cm2c^2 - 64*r_cm2c^3 + 32*r_cm2c^4) + (x[,"n_22"] + x[,"n_24"])*log10(138 - 31*r_cm2c + 85*r_cm2c^2 - 108*r_cm2c^3 + 54*r_cm2c^4)
logL_cm2c <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_06"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_15"] - 4*x[,"n_16"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"] - 4*x[,"n_30"] - 3*x[,"n_31"] - 4*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"] - 3*x[,"n_35"] - 4*x[,"n_36"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 4*x[,"n_42"] - 4*x[,"n_43"] - 4*x[,"n_44"] - 4*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(log(3) + 2*log(5)) + (x[,"n_21"] + x[,"n_25"])*log(7) + (x[,"n_00"] + x[,"n_46"])*(3*log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c))) + (x[,"n_05"] + x[,"n_41"])*(log(7 - 4*r_cm2c) + 2*log(pmax(1e-6,r_cm2c))) + (x[,"n_06"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_cm2c)) + 3*log(pmax(1e-6,r_cm2c))) + (x[,"n_01"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_cm2c)) + log(3 + 4*r_cm2c)) + (x[,"n_21"] + x[,"n_25"])*log(3 + 2*r_cm2c - 2*r_cm2c^2) + (x[,"n_16"] + x[,"n_30"])*(2*log(pmax(1e-6,r_cm2c)) + log(5 - 6*r_cm2c + 4*r_cm2c^2)) + (x[,"n_15"] + x[,"n_31"])*(log(pmax(1e-6,r_cm2c)) + log(14 - 3*r_cm2c + 4*r_cm2c^2)) + (x[,"n_10"] + x[,"n_36"])*(2*log(pmax(1e-6,1 - r_cm2c)) + log(3 - 2*r_cm2c + 4*r_cm2c^2)) + (x[,"n_20"] + x[,"n_26"])*(log(pmax(1e-6,r_cm2c)) + log(7 - 13*r_cm2c + 12*r_cm2c^2 - 6*r_cm2c^3)) + (x[,"n_11"] + x[,"n_35"])*log(15 - 20*r_cm2c + 9*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_04"] + x[,"n_42"])*(log(pmax(1e-6,r_cm2c)) + log(15 + r_cm2c - 7*r_cm2c^2 + 9*r_cm2c^3)) + x[,"n_23"]*log(111 + 10*r_cm2c - 58*r_cm2c^2 + 96*r_cm2c^3 - 48*r_cm2c^4) + (x[,"n_14"] + x[,"n_32"])*log(39 + 76*r_cm2c - 27*r_cm2c^2 + 50*r_cm2c^3 - 36*r_cm2c^4) + (x[,"n_12"] + x[,"n_34"])*log(102 - 28*r_cm2c - 93*r_cm2c^2 + 94*r_cm2c^3 - 36*r_cm2c^4) + (x[,"n_03"] + x[,"n_43"])*log(9 + 18*r_cm2c - 34*r_cm2c^2 + 32*r_cm2c^3 - 16*r_cm2c^4) + (x[,"n_02"] + x[,"n_44"])*log(18 - 32*r_cm2c + 34*r_cm2c^2 - 29*r_cm2c^3 + 9*r_cm2c^4) + (x[,"n_13"] + x[,"n_33"])*log(63 - 14*r_cm2c + 46*r_cm2c^2 - 64*r_cm2c^3 + 32*r_cm2c^4) + (x[,"n_22"] + x[,"n_24"])*log(138 - 31*r_cm2c + 85*r_cm2c^2 - 108*r_cm2c^3 + 54*r_cm2c^4)
logL_cm1c <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n05 - 4*n06 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n16 - 4*n20 - 3*n21 - 4*n22 - 2*n23 - 4*n24 - 3*n25 - 4*n26 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n36 - 4*n40 - 3*n41 - 4*n42 - 3*n43 - 4*n44 - 3*n45 - 4*n46)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(log(3) + 2*log(5)) + (n00 + n06 + n40 + n46)*(2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n45)*(log(pmax(1e-6,r)) + log(2 - r - r^2)) + (n05 + n41)*(log(pmax(1e-6,r)) + log(3 - 4*r + r^2)) + (n21 + n25)*log(15 - 11*r + 11*r^2) + (n15 + n31)*log(3 + r - r^3) + (n10 + n16 + n30 + n36)*(log(pmax(1e-6,r)) + log(1 - 2*r + 2*r^2 - r^3)) + (n11 + n35)*log(3 + 2*r - 3*r^2 + r^3) + (n22 + n24)*log(129 + 32*r - 86*r^2 + 108*r^3 - 54*r^4) + (n13 + n33)*log(30 + 7*r - 23*r^2 + 32*r^3 - 16*r^4) + (n04 + n42)*log(9 - 11*r^2 + 14*r^3 - 9*r^4) + (n02 + n44)*log(3 + 16*r - 23*r^2 + 22*r^3 - 9*r^4) + (n20 + n26)*log(3 - 8*r + 14*r^2 - 12*r^3 + 6*r^4) + (n03 + n43)*log(9 - 13*r + 21*r^2 - 16*r^3 + 8*r^4) + (n12 + n34)*log(21 - 10*r + 18*r^2 - 20*r^3 + 9*r^4) + (n14 + n32)*log(18 - 2*r + 12*r^2 - 16*r^3 + 9*r^4) + n23*log(54 - r + 25*r^2 - 48*r^3 + 24*r^4)
return(L)}
interlogL_cm1c <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_cm1c,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm1c <- mapply(interlogL_cm1c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_cm1c <- 4*(x[,"n_00"] + x[,"n_06"] + x[,"n_40"] + x[,"n_46"])*log10(2) - (x[,"n_10"] + x[,"n_16"] + x[,"n_30"] + x[,"n_36"])*(-4*log10(2) + log10(3)) - (x[,"n_15"] + x[,"n_31"])*(-3*log10(2) + 3*log10(3)) - (x[,"n_11"] + x[,"n_35"])*(-3*log10(2) + 3*log10(3)) - (x[,"n_05"] + x[,"n_41"])*(-3*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_45"])*(-3*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_43"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_21"] + x[,"n_25"])*(-2*log10(2) + 2*log10(7)) - (x[,"n_20"] + x[,"n_26"])*(-3*log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_32"])*(-4*log10(2) + 3*log10(3) + log10(11)) - (x[,"n_12"] + x[,"n_34"])*(-4*log10(2) + 3*log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_42"])*(-4*log10(2) + log10(7) + log10(17)) - (x[,"n_02"] + x[,"n_44"])*(-4*log10(2) + log10(7) + log10(17)) - x[,"n_23"]*(-2*log10(2) + log10(13) + log10(17)) - (x[,"n_13"] + x[,"n_33"])*(-2*log10(2) + log10(3) + log10(41)) - (x[,"n_22"] + x[,"n_24"])*(-3*log10(2) + log10(1069)) + (x[,"n_00"] + x[,"n_06"] + x[,"n_40"] + x[,"n_46"])*(2*log10(pmax(1e-6,1 - r_cm1c)) + 2*log10(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_45"])*(log10(pmax(1e-6,r_cm1c)) + log10(2 - r_cm1c - r_cm1c^2)) + (x[,"n_05"] + x[,"n_41"])*(log10(pmax(1e-6,r_cm1c)) + log10(3 - 4*r_cm1c + r_cm1c^2)) + (x[,"n_21"] + x[,"n_25"])*log10(15 - 11*r_cm1c + 11*r_cm1c^2) + (x[,"n_15"] + x[,"n_31"])*log10(3 + r_cm1c - r_cm1c^3) + (x[,"n_10"] + x[,"n_16"] + x[,"n_30"] + x[,"n_36"])*(log10(pmax(1e-6,r_cm1c)) + log10(1 - 2*r_cm1c + 2*r_cm1c^2 - r_cm1c^3)) + (x[,"n_11"] + x[,"n_35"])*log10(3 + 2*r_cm1c - 3*r_cm1c^2 + r_cm1c^3) + (x[,"n_22"] + x[,"n_24"])*log10(129 + 32*r_cm1c - 86*r_cm1c^2 + 108*r_cm1c^3 - 54*r_cm1c^4) + (x[,"n_13"] + x[,"n_33"])*log10(30 + 7*r_cm1c - 23*r_cm1c^2 + 32*r_cm1c^3 - 16*r_cm1c^4) + (x[,"n_04"] + x[,"n_42"])*log10(9 - 11*r_cm1c^2 + 14*r_cm1c^3 - 9*r_cm1c^4) + (x[,"n_02"] + x[,"n_44"])*log10(3 + 16*r_cm1c - 23*r_cm1c^2 + 22*r_cm1c^3 - 9*r_cm1c^4) + (x[,"n_20"] + x[,"n_26"])*log10(3 - 8*r_cm1c + 14*r_cm1c^2 - 12*r_cm1c^3 + 6*r_cm1c^4) + (x[,"n_03"] + x[,"n_43"])*log10(9 - 13*r_cm1c + 21*r_cm1c^2 - 16*r_cm1c^3 + 8*r_cm1c^4) + (x[,"n_12"] + x[,"n_34"])*log10(21 - 10*r_cm1c + 18*r_cm1c^2 - 20*r_cm1c^3 + 9*r_cm1c^4) + (x[,"n_14"] + x[,"n_32"])*log10(18 - 2*r_cm1c + 12*r_cm1c^2 - 16*r_cm1c^3 + 9*r_cm1c^4) + x[,"n_23"]*log10(54 - r_cm1c + 25*r_cm1c^2 - 48*r_cm1c^3 + 24*r_cm1c^4)
logL_cm1c <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_05"] - 4*x[,"n_06"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_16"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_25"] - 4*x[,"n_26"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_36"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 3*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_06"] + x[,"n_40"] + x[,"n_46"])*(2*log(pmax(1e-6,1 - r_cm1c)) + 2*log(pmax(1e-6,r_cm1c))) + (x[,"n_01"] + x[,"n_45"])*(log(pmax(1e-6,r_cm1c)) + log(2 - r_cm1c - r_cm1c^2)) + (x[,"n_05"] + x[,"n_41"])*(log(pmax(1e-6,r_cm1c)) + log(3 - 4*r_cm1c + r_cm1c^2)) + (x[,"n_21"] + x[,"n_25"])*log(15 - 11*r_cm1c + 11*r_cm1c^2) + (x[,"n_15"] + x[,"n_31"])*log(3 + r_cm1c - r_cm1c^3) + (x[,"n_10"] + x[,"n_16"] + x[,"n_30"] + x[,"n_36"])*(log(pmax(1e-6,r_cm1c)) + log(1 - 2*r_cm1c + 2*r_cm1c^2 - r_cm1c^3)) + (x[,"n_11"] + x[,"n_35"])*log(3 + 2*r_cm1c - 3*r_cm1c^2 + r_cm1c^3) + (x[,"n_22"] + x[,"n_24"])*log(129 + 32*r_cm1c - 86*r_cm1c^2 + 108*r_cm1c^3 - 54*r_cm1c^4) + (x[,"n_13"] + x[,"n_33"])*log(30 + 7*r_cm1c - 23*r_cm1c^2 + 32*r_cm1c^3 - 16*r_cm1c^4) + (x[,"n_04"] + x[,"n_42"])*log(9 - 11*r_cm1c^2 + 14*r_cm1c^3 - 9*r_cm1c^4) + (x[,"n_02"] + x[,"n_44"])*log(3 + 16*r_cm1c - 23*r_cm1c^2 + 22*r_cm1c^3 - 9*r_cm1c^4) + (x[,"n_20"] + x[,"n_26"])*log(3 - 8*r_cm1c + 14*r_cm1c^2 - 12*r_cm1c^3 + 6*r_cm1c^4) + (x[,"n_03"] + x[,"n_43"])*log(9 - 13*r_cm1c + 21*r_cm1c^2 - 16*r_cm1c^3 + 8*r_cm1c^4) + (x[,"n_12"] + x[,"n_34"])*log(21 - 10*r_cm1c + 18*r_cm1c^2 - 20*r_cm1c^3 + 9*r_cm1c^4) + (x[,"n_14"] + x[,"n_32"])*log(18 - 2*r_cm1c + 12*r_cm1c^2 - 16*r_cm1c^3 + 9*r_cm1c^4) + x[,"n_23"]*log(54 - r_cm1c + 25*r_cm1c^2 - 48*r_cm1c^3 + 24*r_cm1c^4)
logL_cr <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n03 - 4*n04 - 4*n05 - 4*n06 - 4*n10 - 3*n11 - 4*n12 - 3*n13 - 4*n14 - 3*n15 - 4*n16 - 4*n20 - 4*n21 - 4*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n26 - 4*n30 - 3*n31 - 4*n32 - 3*n33 - 4*n34 - 3*n35 - 4*n36 - 4*n40 - 4*n41 - 4*n42 - 4*n43 - 4*n44 - 4*n45 - 4*n46)*log(2) + (n01 + n02 + n04 + n05 + n11 + n12 + n14 + n15 + n20 + n21 + n22 + n23 + n24 + n25 + n26 + n31 + n32 + n34 + n35 + n41 + n42 + n44 + n45)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*log(5) + 2*(n05 + n41)*log(pmax(1e-6,1 - r)) + 2*(n01 + n45)*log(pmax(1e-6,r)) + (n06 + n40)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n11 + n35)*(log(2 - r) + log(pmax(1e-6,r))) + (n00 + n46)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n15 + n31)*log(1 - r^2) + (n21 + n25)*log(3 - 2*r + 2*r^2) + (n10 + n36)*(2*log(pmax(1e-6,r)) + log(3 - 6*r + 4*r^2)) + (n16 + n30)*(2*log(pmax(1e-6,1 - r)) + log(1 - 2*r + 4*r^2)) + (n20 + n26)*(log(pmax(1e-6,r)) + log(1 - 3*r + 4*r^2 - 2*r^3)) + (n02 + n44)*(log(pmax(1e-6,r)) + log(1 + 3*r - 5*r^2 + 3*r^3)) + n23*log(9 + 18*r - 34*r^2 + 32*r^3 - 16*r^4) + (n03 + n43)*log(1 + 14*r - 30*r^2 + 32*r^3 - 16*r^4) + (n12 + n34)*log(3 + 16*r - 19*r^2 + 22*r^3 - 12*r^4) + (n14 + n32)*log(10 + 4*r - 25*r^2 + 26*r^3 - 12*r^4) + (n04 + n42)*log(2 - 4*r + 6*r^2 - 7*r^3 + 3*r^4) + (n22 + n24)*log(18 - 17*r + 35*r^2 - 36*r^3 + 18*r^4) + (n13 + n33)*log(27 - 34*r + 66*r^2 - 64*r^3 + 32*r^4)
return(L)}
interlogL_cr <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_cr,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_cr <- 3*(x[,"n_20"] + x[,"n_26"])*log10(2) + 2*(x[,"n_16"] + x[,"n_30"])*log10(2) + 2*(x[,"n_10"] + x[,"n_36"])*log10(2) + 4*(x[,"n_06"] + x[,"n_40"])*log10(2) + 2*(x[,"n_05"] + x[,"n_41"])*log10(2) + 2*(x[,"n_01"] + x[,"n_45"])*log10(2) + 4*(x[,"n_00"] + x[,"n_46"])*log10(2) + (x[,"n_15"] + x[,"n_31"])*(2*log10(2) - log10(3)) - (x[,"n_11"] + x[,"n_35"])*(-2*log10(2) + log10(3)) - (x[,"n_21"] + x[,"n_25"])*(-log10(2) + log10(5)) - x[,"n_23"]*(-log10(2) + 2*log10(5)) - (x[,"n_03"] + x[,"n_43"])*(-log10(2) + log10(7)) - (x[,"n_14"] + x[,"n_32"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_12"] + x[,"n_34"])*(-2*log10(2) + log10(3) + log10(11)) + (x[,"n_04"] + x[,"n_42"])*(4*log10(2) - log10(13)) - (x[,"n_02"] + x[,"n_44"])*(-4*log10(2) + log10(13)) - (x[,"n_22"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_13"] + x[,"n_33"])*(-log10(2) + log10(41)) + 2*(x[,"n_05"] + x[,"n_41"])*log10(pmax(1e-6,1 - r_cr)) + 2*(x[,"n_01"] + x[,"n_45"])*log10(pmax(1e-6,r_cr)) + (x[,"n_06"] + x[,"n_40"])*(3*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_11"] + x[,"n_35"])*(log10(2 - r_cr) + log10(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_46"])*(log10(pmax(1e-6,1 - r_cr)) + 3*log10(pmax(1e-6,r_cr))) + (x[,"n_15"] + x[,"n_31"])*log10(1 - r_cr^2) + (x[,"n_21"] + x[,"n_25"])*log10(3 - 2*r_cr + 2*r_cr^2) + (x[,"n_10"] + x[,"n_36"])*(2*log10(pmax(1e-6,r_cr)) + log10(3 - 6*r_cr + 4*r_cr^2)) + (x[,"n_16"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(1 - 2*r_cr + 4*r_cr^2)) + (x[,"n_20"] + x[,"n_26"])*(log10(pmax(1e-6,r_cr)) + log10(1 - 3*r_cr + 4*r_cr^2 - 2*r_cr^3)) + (x[,"n_02"] + x[,"n_44"])*(log10(pmax(1e-6,r_cr)) + log10(1 + 3*r_cr - 5*r_cr^2 + 3*r_cr^3)) + x[,"n_23"]*log10(9 + 18*r_cr - 34*r_cr^2 + 32*r_cr^3 - 16*r_cr^4) + (x[,"n_03"] + x[,"n_43"])*log10(1 + 14*r_cr - 30*r_cr^2 + 32*r_cr^3 - 16*r_cr^4) + (x[,"n_12"] + x[,"n_34"])*log10(3 + 16*r_cr - 19*r_cr^2 + 22*r_cr^3 - 12*r_cr^4) + (x[,"n_14"] + x[,"n_32"])*log10(10 + 4*r_cr - 25*r_cr^2 + 26*r_cr^3 - 12*r_cr^4) + (x[,"n_04"] + x[,"n_42"])*log10(2 - 4*r_cr + 6*r_cr^2 - 7*r_cr^3 + 3*r_cr^4) + (x[,"n_22"] + x[,"n_24"])*log10(18 - 17*r_cr + 35*r_cr^2 - 36*r_cr^3 + 18*r_cr^4) + (x[,"n_13"] + x[,"n_33"])*log10(27 - 34*r_cr + 66*r_cr^2 - 64*r_cr^3 + 32*r_cr^4)
logL_cr <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_06"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_15"] - 4*x[,"n_16"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"] - 4*x[,"n_30"] - 3*x[,"n_31"] - 4*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"] - 3*x[,"n_35"] - 4*x[,"n_36"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 4*x[,"n_42"] - 4*x[,"n_43"] - 4*x[,"n_44"] - 4*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_04"] + x[,"n_05"] + x[,"n_11"] + x[,"n_12"] + x[,"n_14"] + x[,"n_15"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_25"] + x[,"n_26"] + x[,"n_31"] + x[,"n_32"] + x[,"n_34"] + x[,"n_35"] + x[,"n_41"] + x[,"n_42"] + x[,"n_44"] + x[,"n_45"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*log(5) + 2*(x[,"n_05"] + x[,"n_41"])*log(pmax(1e-6,1 - r_cr)) + 2*(x[,"n_01"] + x[,"n_45"])*log(pmax(1e-6,r_cr)) + (x[,"n_06"] + x[,"n_40"])*(3*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_11"] + x[,"n_35"])*(log(2 - r_cr) + log(pmax(1e-6,r_cr))) + (x[,"n_00"] + x[,"n_46"])*(log(pmax(1e-6,1 - r_cr)) + 3*log(pmax(1e-6,r_cr))) + (x[,"n_15"] + x[,"n_31"])*log(1 - r_cr^2) + (x[,"n_21"] + x[,"n_25"])*log(3 - 2*r_cr + 2*r_cr^2) + (x[,"n_10"] + x[,"n_36"])*(2*log(pmax(1e-6,r_cr)) + log(3 - 6*r_cr + 4*r_cr^2)) + (x[,"n_16"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_cr)) + log(1 - 2*r_cr + 4*r_cr^2)) + (x[,"n_20"] + x[,"n_26"])*(log(pmax(1e-6,r_cr)) + log(1 - 3*r_cr + 4*r_cr^2 - 2*r_cr^3)) + (x[,"n_02"] + x[,"n_44"])*(log(pmax(1e-6,r_cr)) + log(1 + 3*r_cr - 5*r_cr^2 + 3*r_cr^3)) + x[,"n_23"]*log(9 + 18*r_cr - 34*r_cr^2 + 32*r_cr^3 - 16*r_cr^4) + (x[,"n_03"] + x[,"n_43"])*log(1 + 14*r_cr - 30*r_cr^2 + 32*r_cr^3 - 16*r_cr^4) + (x[,"n_12"] + x[,"n_34"])*log(3 + 16*r_cr - 19*r_cr^2 + 22*r_cr^3 - 12*r_cr^4) + (x[,"n_14"] + x[,"n_32"])*log(10 + 4*r_cr - 25*r_cr^2 + 26*r_cr^3 - 12*r_cr^4) + (x[,"n_04"] + x[,"n_42"])*log(2 - 4*r_cr + 6*r_cr^2 - 7*r_cr^3 + 3*r_cr^4) + (x[,"n_22"] + x[,"n_24"])*log(18 - 17*r_cr + 35*r_cr^2 - 36*r_cr^3 + 18*r_cr^4) + (x[,"n_13"] + x[,"n_33"])*log(27 - 34*r_cr + 66*r_cr^2 - 64*r_cr^3 + 32*r_cr^4)
logL_rc <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n03 - 4*n04 - 4*n05 - 4*n06 - 4*n10 - 3*n11 - 4*n12 - 3*n13 - 4*n14 - 3*n15 - 4*n16 - 4*n20 - 4*n21 - 4*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n26 - 4*n30 - 3*n31 - 4*n32 - 3*n33 - 4*n34 - 3*n35 - 4*n36 - 4*n40 - 4*n41 - 4*n42 - 4*n43 - 4*n44 - 4*n45 - 4*n46)*log(2) + (n01 + n02 + n04 + n05 + n11 + n12 + n14 + n15 + n20 + n21 + n22 + n23 + n24 + n25 + n26 + n31 + n32 + n34 + n35 + n41 + n42 + n44 + n45)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*log(5) + 2*(n01 + n45)*log(pmax(1e-6,1 - r)) + 2*(n05 + n41)*log(pmax(1e-6,r)) + (n00 + n46)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n15 + n31)*(log(2 - r) + log(pmax(1e-6,r))) + (n06 + n40)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n11 + n35)*log(1 - r^2) + (n21 + n25)*log(3 - 2*r + 2*r^2) + (n16 + n30)*(2*log(pmax(1e-6,r)) + log(3 - 6*r + 4*r^2)) + (n10 + n36)*(2*log(pmax(1e-6,1 - r)) + log(1 - 2*r + 4*r^2)) + (n20 + n26)*(log(pmax(1e-6,r)) + log(1 - 3*r + 4*r^2 - 2*r^3)) + (n04 + n42)*(log(pmax(1e-6,r)) + log(1 + 3*r - 5*r^2 + 3*r^3)) + n23*log(9 + 18*r - 34*r^2 + 32*r^3 - 16*r^4) + (n03 + n43)*log(1 + 14*r - 30*r^2 + 32*r^3 - 16*r^4) + (n14 + n32)*log(3 + 16*r - 19*r^2 + 22*r^3 - 12*r^4) + (n12 + n34)*log(10 + 4*r - 25*r^2 + 26*r^3 - 12*r^4) + (n02 + n44)*log(2 - 4*r + 6*r^2 - 7*r^3 + 3*r^4) + (n22 + n24)*log(18 - 17*r + 35*r^2 - 36*r^3 + 18*r^4) + (n13 + n33)*log(27 - 34*r + 66*r^2 - 64*r^3 + 32*r^4)
return(L)}
interlogL_rc <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_rc,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_rc <- 3*(x[,"n_20"] + x[,"n_26"])*log10(2) + 2*(x[,"n_16"] + x[,"n_30"])*log10(2) + 2*(x[,"n_10"] + x[,"n_36"])*log10(2) + 4*(x[,"n_06"] + x[,"n_40"])*log10(2) + 2*(x[,"n_05"] + x[,"n_41"])*log10(2) + 2*(x[,"n_01"] + x[,"n_45"])*log10(2) + 4*(x[,"n_00"] + x[,"n_46"])*log10(2) + (x[,"n_11"] + x[,"n_35"])*(2*log10(2) - log10(3)) - (x[,"n_15"] + x[,"n_31"])*(-2*log10(2) + log10(3)) - (x[,"n_21"] + x[,"n_25"])*(-log10(2) + log10(5)) - x[,"n_23"]*(-log10(2) + 2*log10(5)) - (x[,"n_03"] + x[,"n_43"])*(-log10(2) + log10(7)) - (x[,"n_14"] + x[,"n_32"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_12"] + x[,"n_34"])*(-2*log10(2) + log10(3) + log10(11)) + (x[,"n_02"] + x[,"n_44"])*(4*log10(2) - log10(13)) - (x[,"n_04"] + x[,"n_42"])*(-4*log10(2) + log10(13)) - (x[,"n_22"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_13"] + x[,"n_33"])*(-log10(2) + log10(41)) + 2*(x[,"n_01"] + x[,"n_45"])*log10(pmax(1e-6,1 - r_rc)) + 2*(x[,"n_05"] + x[,"n_41"])*log10(pmax(1e-6,r_rc)) + (x[,"n_00"] + x[,"n_46"])*(3*log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc))) + (x[,"n_15"] + x[,"n_31"])*(log10(2 - r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_06"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_rc)) + 3*log10(pmax(1e-6,r_rc))) + (x[,"n_11"] + x[,"n_35"])*log10(1 - r_rc^2) + (x[,"n_21"] + x[,"n_25"])*log10(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_16"] + x[,"n_30"])*(2*log10(pmax(1e-6,r_rc)) + log10(3 - 6*r_rc + 4*r_rc^2)) + (x[,"n_10"] + x[,"n_36"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(1 - 2*r_rc + 4*r_rc^2)) + (x[,"n_20"] + x[,"n_26"])*(log10(pmax(1e-6,r_rc)) + log10(1 - 3*r_rc + 4*r_rc^2 - 2*r_rc^3)) + (x[,"n_04"] + x[,"n_42"])*(log10(pmax(1e-6,r_rc)) + log10(1 + 3*r_rc - 5*r_rc^2 + 3*r_rc^3)) + x[,"n_23"]*log10(9 + 18*r_rc - 34*r_rc^2 + 32*r_rc^3 - 16*r_rc^4) + (x[,"n_03"] + x[,"n_43"])*log10(1 + 14*r_rc - 30*r_rc^2 + 32*r_rc^3 - 16*r_rc^4) + (x[,"n_14"] + x[,"n_32"])*log10(3 + 16*r_rc - 19*r_rc^2 + 22*r_rc^3 - 12*r_rc^4) + (x[,"n_12"] + x[,"n_34"])*log10(10 + 4*r_rc - 25*r_rc^2 + 26*r_rc^3 - 12*r_rc^4) + (x[,"n_02"] + x[,"n_44"])*log10(2 - 4*r_rc + 6*r_rc^2 - 7*r_rc^3 + 3*r_rc^4) + (x[,"n_22"] + x[,"n_24"])*log10(18 - 17*r_rc + 35*r_rc^2 - 36*r_rc^3 + 18*r_rc^4) + (x[,"n_13"] + x[,"n_33"])*log10(27 - 34*r_rc + 66*r_rc^2 - 64*r_rc^3 + 32*r_rc^4)
logL_rc <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_06"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_15"] - 4*x[,"n_16"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"] - 4*x[,"n_30"] - 3*x[,"n_31"] - 4*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"] - 3*x[,"n_35"] - 4*x[,"n_36"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 4*x[,"n_42"] - 4*x[,"n_43"] - 4*x[,"n_44"] - 4*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_04"] + x[,"n_05"] + x[,"n_11"] + x[,"n_12"] + x[,"n_14"] + x[,"n_15"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_25"] + x[,"n_26"] + x[,"n_31"] + x[,"n_32"] + x[,"n_34"] + x[,"n_35"] + x[,"n_41"] + x[,"n_42"] + x[,"n_44"] + x[,"n_45"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*log(5) + 2*(x[,"n_01"] + x[,"n_45"])*log(pmax(1e-6,1 - r_rc)) + 2*(x[,"n_05"] + x[,"n_41"])*log(pmax(1e-6,r_rc)) + (x[,"n_00"] + x[,"n_46"])*(3*log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc))) + (x[,"n_15"] + x[,"n_31"])*(log(2 - r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_06"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_rc)) + 3*log(pmax(1e-6,r_rc))) + (x[,"n_11"] + x[,"n_35"])*log(1 - r_rc^2) + (x[,"n_21"] + x[,"n_25"])*log(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_16"] + x[,"n_30"])*(2*log(pmax(1e-6,r_rc)) + log(3 - 6*r_rc + 4*r_rc^2)) + (x[,"n_10"] + x[,"n_36"])*(2*log(pmax(1e-6,1 - r_rc)) + log(1 - 2*r_rc + 4*r_rc^2)) + (x[,"n_20"] + x[,"n_26"])*(log(pmax(1e-6,r_rc)) + log(1 - 3*r_rc + 4*r_rc^2 - 2*r_rc^3)) + (x[,"n_04"] + x[,"n_42"])*(log(pmax(1e-6,r_rc)) + log(1 + 3*r_rc - 5*r_rc^2 + 3*r_rc^3)) + x[,"n_23"]*log(9 + 18*r_rc - 34*r_rc^2 + 32*r_rc^3 - 16*r_rc^4) + (x[,"n_03"] + x[,"n_43"])*log(1 + 14*r_rc - 30*r_rc^2 + 32*r_rc^3 - 16*r_rc^4) + (x[,"n_14"] + x[,"n_32"])*log(3 + 16*r_rc - 19*r_rc^2 + 22*r_rc^3 - 12*r_rc^4) + (x[,"n_12"] + x[,"n_34"])*log(10 + 4*r_rc - 25*r_rc^2 + 26*r_rc^3 - 12*r_rc^4) + (x[,"n_02"] + x[,"n_44"])*log(2 - 4*r_rc + 6*r_rc^2 - 7*r_rc^3 + 3*r_rc^4) + (x[,"n_22"] + x[,"n_24"])*log(18 - 17*r_rc + 35*r_rc^2 - 36*r_rc^3 + 18*r_rc^4) + (x[,"n_13"] + x[,"n_33"])*log(27 - 34*r_rc + 66*r_rc^2 - 64*r_rc^3 + 32*r_rc^4)
logL_rm2c <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n05 - 4*n06 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n16 - 4*n20 - 3*n21 - 4*n22 - 2*n23 - 4*n24 - 3*n25 - 4*n26 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n36 - 4*n40 - 3*n41 - 4*n42 - 3*n43 - 4*n44 - 3*n45 - 4*n46)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(log(3) + 2*log(5)) + (n00 + n06 + n40 + n46)*(2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n05 + n41)*(log(pmax(1e-6,r)) + log(2 - r - r^2)) + (n01 + n45)*(log(pmax(1e-6,r)) + log(3 - 4*r + r^2)) + (n21 + n25)*log(15 - 11*r + 11*r^2) + (n11 + n35)*log(3 + r - r^3) + (n10 + n16 + n30 + n36)*(log(pmax(1e-6,r)) + log(1 - 2*r + 2*r^2 - r^3)) + (n15 + n31)*log(3 + 2*r - 3*r^2 + r^3) + (n22 + n24)*log(129 + 32*r - 86*r^2 + 108*r^3 - 54*r^4) + (n13 + n33)*log(30 + 7*r - 23*r^2 + 32*r^3 - 16*r^4) + (n02 + n44)*log(9 - 11*r^2 + 14*r^3 - 9*r^4) + (n04 + n42)*log(3 + 16*r - 23*r^2 + 22*r^3 - 9*r^4) + (n20 + n26)*log(3 - 8*r + 14*r^2 - 12*r^3 + 6*r^4) + (n03 + n43)*log(9 - 13*r + 21*r^2 - 16*r^3 + 8*r^4) + (n14 + n32)*log(21 - 10*r + 18*r^2 - 20*r^3 + 9*r^4) + (n12 + n34)*log(18 - 2*r + 12*r^2 - 16*r^3 + 9*r^4) + n23*log(54 - r + 25*r^2 - 48*r^3 + 24*r^4)
return(L)}
interlogL_rm2c <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_rm2c,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_rm2c <- 4*(x[,"n_00"] + x[,"n_06"] + x[,"n_40"] + x[,"n_46"])*log10(2) - (x[,"n_10"] + x[,"n_16"] + x[,"n_30"] + x[,"n_36"])*(-4*log10(2) + log10(3)) - (x[,"n_15"] + x[,"n_31"])*(-3*log10(2) + 3*log10(3)) - (x[,"n_11"] + x[,"n_35"])*(-3*log10(2) + 3*log10(3)) - (x[,"n_05"] + x[,"n_41"])*(-3*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_45"])*(-3*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_43"])*(-2*log10(2) + 2*log10(5)) - (x[,"n_21"] + x[,"n_25"])*(-2*log10(2) + 2*log10(7)) - (x[,"n_20"] + x[,"n_26"])*(-3*log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_32"])*(-4*log10(2) + 3*log10(3) + log10(11)) - (x[,"n_12"] + x[,"n_34"])*(-4*log10(2) + 3*log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_42"])*(-4*log10(2) + log10(7) + log10(17)) - (x[,"n_02"] + x[,"n_44"])*(-4*log10(2) + log10(7) + log10(17)) - x[,"n_23"]*(-2*log10(2) + log10(13) + log10(17)) - (x[,"n_13"] + x[,"n_33"])*(-2*log10(2) + log10(3) + log10(41)) - (x[,"n_22"] + x[,"n_24"])*(-3*log10(2) + log10(1069)) + (x[,"n_00"] + x[,"n_06"] + x[,"n_40"] + x[,"n_46"])*(2*log10(pmax(1e-6,1 - r_rm2c)) + 2*log10(pmax(1e-6,r_rm2c))) + (x[,"n_05"] + x[,"n_41"])*(log10(pmax(1e-6,r_rm2c)) + log10(2 - r_rm2c - r_rm2c^2)) + (x[,"n_01"] + x[,"n_45"])*(log10(pmax(1e-6,r_rm2c)) + log10(3 - 4*r_rm2c + r_rm2c^2)) + (x[,"n_21"] + x[,"n_25"])*log10(15 - 11*r_rm2c + 11*r_rm2c^2) + (x[,"n_11"] + x[,"n_35"])*log10(3 + r_rm2c - r_rm2c^3) + (x[,"n_10"] + x[,"n_16"] + x[,"n_30"] + x[,"n_36"])*(log10(pmax(1e-6,r_rm2c)) + log10(1 - 2*r_rm2c + 2*r_rm2c^2 - r_rm2c^3)) + (x[,"n_15"] + x[,"n_31"])*log10(3 + 2*r_rm2c - 3*r_rm2c^2 + r_rm2c^3) + (x[,"n_22"] + x[,"n_24"])*log10(129 + 32*r_rm2c - 86*r_rm2c^2 + 108*r_rm2c^3 - 54*r_rm2c^4) + (x[,"n_13"] + x[,"n_33"])*log10(30 + 7*r_rm2c - 23*r_rm2c^2 + 32*r_rm2c^3 - 16*r_rm2c^4) + (x[,"n_02"] + x[,"n_44"])*log10(9 - 11*r_rm2c^2 + 14*r_rm2c^3 - 9*r_rm2c^4) + (x[,"n_04"] + x[,"n_42"])*log10(3 + 16*r_rm2c - 23*r_rm2c^2 + 22*r_rm2c^3 - 9*r_rm2c^4) + (x[,"n_20"] + x[,"n_26"])*log10(3 - 8*r_rm2c + 14*r_rm2c^2 - 12*r_rm2c^3 + 6*r_rm2c^4) + (x[,"n_03"] + x[,"n_43"])*log10(9 - 13*r_rm2c + 21*r_rm2c^2 - 16*r_rm2c^3 + 8*r_rm2c^4) + (x[,"n_14"] + x[,"n_32"])*log10(21 - 10*r_rm2c + 18*r_rm2c^2 - 20*r_rm2c^3 + 9*r_rm2c^4) + (x[,"n_12"] + x[,"n_34"])*log10(18 - 2*r_rm2c + 12*r_rm2c^2 - 16*r_rm2c^3 + 9*r_rm2c^4) + x[,"n_23"]*log10(54 - r_rm2c + 25*r_rm2c^2 - 48*r_rm2c^3 + 24*r_rm2c^4)
logL_rm2c <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_05"] - 4*x[,"n_06"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_16"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_25"] - 4*x[,"n_26"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_36"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 3*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_06"] + x[,"n_40"] + x[,"n_46"])*(2*log(pmax(1e-6,1 - r_rm2c)) + 2*log(pmax(1e-6,r_rm2c))) + (x[,"n_05"] + x[,"n_41"])*(log(pmax(1e-6,r_rm2c)) + log(2 - r_rm2c - r_rm2c^2)) + (x[,"n_01"] + x[,"n_45"])*(log(pmax(1e-6,r_rm2c)) + log(3 - 4*r_rm2c + r_rm2c^2)) + (x[,"n_21"] + x[,"n_25"])*log(15 - 11*r_rm2c + 11*r_rm2c^2) + (x[,"n_11"] + x[,"n_35"])*log(3 + r_rm2c - r_rm2c^3) + (x[,"n_10"] + x[,"n_16"] + x[,"n_30"] + x[,"n_36"])*(log(pmax(1e-6,r_rm2c)) + log(1 - 2*r_rm2c + 2*r_rm2c^2 - r_rm2c^3)) + (x[,"n_15"] + x[,"n_31"])*log(3 + 2*r_rm2c - 3*r_rm2c^2 + r_rm2c^3) + (x[,"n_22"] + x[,"n_24"])*log(129 + 32*r_rm2c - 86*r_rm2c^2 + 108*r_rm2c^3 - 54*r_rm2c^4) + (x[,"n_13"] + x[,"n_33"])*log(30 + 7*r_rm2c - 23*r_rm2c^2 + 32*r_rm2c^3 - 16*r_rm2c^4) + (x[,"n_02"] + x[,"n_44"])*log(9 - 11*r_rm2c^2 + 14*r_rm2c^3 - 9*r_rm2c^4) + (x[,"n_04"] + x[,"n_42"])*log(3 + 16*r_rm2c - 23*r_rm2c^2 + 22*r_rm2c^3 - 9*r_rm2c^4) + (x[,"n_20"] + x[,"n_26"])*log(3 - 8*r_rm2c + 14*r_rm2c^2 - 12*r_rm2c^3 + 6*r_rm2c^4) + (x[,"n_03"] + x[,"n_43"])*log(9 - 13*r_rm2c + 21*r_rm2c^2 - 16*r_rm2c^3 + 8*r_rm2c^4) + (x[,"n_14"] + x[,"n_32"])*log(21 - 10*r_rm2c + 18*r_rm2c^2 - 20*r_rm2c^3 + 9*r_rm2c^4) + (x[,"n_12"] + x[,"n_34"])*log(18 - 2*r_rm2c + 12*r_rm2c^2 - 16*r_rm2c^3 + 9*r_rm2c^4) + x[,"n_23"]*log(54 - r_rm2c + 25*r_rm2c^2 - 48*r_rm2c^3 + 24*r_rm2c^4)
logL_rm1c <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n03 - 4*n04 - 4*n05 - 4*n06 - 4*n10 - 3*n11 - 4*n12 - 3*n13 - 4*n14 - 3*n15 - 4*n16 - 4*n20 - 4*n21 - 4*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n26 - 4*n30 - 3*n31 - 4*n32 - 3*n33 - 4*n34 - 3*n35 - 4*n36 - 4*n40 - 4*n41 - 4*n42 - 4*n43 - 4*n44 - 4*n45 - 4*n46)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(log(3) + 2*log(5)) + (n21 + n25)*log(7) + (n06 + n40)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n45)*(log(7 - 4*r) + 2*log(pmax(1e-6,r))) + (n00 + n46)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n05 + n41)*(2*log(pmax(1e-6,1 - r)) + log(3 + 4*r)) + (n21 + n25)*log(3 + 2*r - 2*r^2) + (n10 + n36)*(2*log(pmax(1e-6,r)) + log(5 - 6*r + 4*r^2)) + (n11 + n35)*(log(pmax(1e-6,r)) + log(14 - 3*r + 4*r^2)) + (n16 + n30)*(2*log(pmax(1e-6,1 - r)) + log(3 - 2*r + 4*r^2)) + (n20 + n26)*(log(pmax(1e-6,r)) + log(7 - 13*r + 12*r^2 - 6*r^3)) + (n15 + n31)*log(15 - 20*r + 9*r^2 - 4*r^3) + (n02 + n44)*(log(pmax(1e-6,r)) + log(15 + r - 7*r^2 + 9*r^3)) + n23*log(111 + 10*r - 58*r^2 + 96*r^3 - 48*r^4) + (n12 + n34)*log(39 + 76*r - 27*r^2 + 50*r^3 - 36*r^4) + (n14 + n32)*log(102 - 28*r - 93*r^2 + 94*r^3 - 36*r^4) + (n03 + n43)*log(9 + 18*r - 34*r^2 + 32*r^3 - 16*r^4) + (n04 + n42)*log(18 - 32*r + 34*r^2 - 29*r^3 + 9*r^4) + (n13 + n33)*log(63 - 14*r + 46*r^2 - 64*r^3 + 32*r^4) + (n22 + n24)*log(138 - 31*r + 85*r^2 - 108*r^3 + 54*r^4)
return(L)}
interlogL_rm1c <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_rm1c,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm1c <- mapply(interlogL_rm1c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_rm1c <- 4*(x[,"n_06"] + x[,"n_40"])*log10(2) + 4*(x[,"n_00"] + x[,"n_46"])*log10(2) - (x[,"n_16"] + x[,"n_30"])*(-2*log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_36"])*(-2*log10(2) + log10(3)) - (x[,"n_15"] + x[,"n_31"])*(-2*log10(2) + 3*log10(3)) - (x[,"n_11"] + x[,"n_35"])*(-2*log10(2) + 3*log10(3)) - (x[,"n_05"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_45"])*(-2*log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_43"])*(-log10(2) + 2*log10(5)) - (x[,"n_21"] + x[,"n_25"])*(-log10(2) + log10(7)) - (x[,"n_20"] + x[,"n_26"])*(-3*log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_32"])*(-2*log10(2) + 3*log10(3) + log10(11)) - (x[,"n_12"] + x[,"n_34"])*(-2*log10(2) + 3*log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_42"])*(-4*log10(2) + log10(7) + log10(17)) - (x[,"n_02"] + x[,"n_44"])*(-4*log10(2) + log10(7) + log10(17)) - x[,"n_23"]*(-log10(2) + log10(13) + log10(17)) - (x[,"n_13"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(41)) - (x[,"n_22"] + x[,"n_24"])*(-3*log10(2) + log10(1069)) + (x[,"n_06"] + x[,"n_40"])*(3*log10(pmax(1e-6,1 - r_rm1c)) + log10(pmax(1e-6,r_rm1c))) + (x[,"n_01"] + x[,"n_45"])*(log10(7 - 4*r_rm1c) + 2*log10(pmax(1e-6,r_rm1c))) + (x[,"n_00"] + x[,"n_46"])*(log10(pmax(1e-6,1 - r_rm1c)) + 3*log10(pmax(1e-6,r_rm1c))) + (x[,"n_05"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_rm1c)) + log10(3 + 4*r_rm1c)) + (x[,"n_21"] + x[,"n_25"])*log10(3 + 2*r_rm1c - 2*r_rm1c^2) + (x[,"n_10"] + x[,"n_36"])*(2*log10(pmax(1e-6,r_rm1c)) + log10(5 - 6*r_rm1c + 4*r_rm1c^2)) + (x[,"n_11"] + x[,"n_35"])*(log10(pmax(1e-6,r_rm1c)) + log10(14 - 3*r_rm1c + 4*r_rm1c^2)) + (x[,"n_16"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_rm1c)) + log10(3 - 2*r_rm1c + 4*r_rm1c^2)) + (x[,"n_20"] + x[,"n_26"])*(log10(pmax(1e-6,r_rm1c)) + log10(7 - 13*r_rm1c + 12*r_rm1c^2 - 6*r_rm1c^3)) + (x[,"n_15"] + x[,"n_31"])*log10(15 - 20*r_rm1c + 9*r_rm1c^2 - 4*r_rm1c^3) + (x[,"n_02"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm1c)) + log10(15 + r_rm1c - 7*r_rm1c^2 + 9*r_rm1c^3)) + x[,"n_23"]*log10(111 + 10*r_rm1c - 58*r_rm1c^2 + 96*r_rm1c^3 - 48*r_rm1c^4) + (x[,"n_12"] + x[,"n_34"])*log10(39 + 76*r_rm1c - 27*r_rm1c^2 + 50*r_rm1c^3 - 36*r_rm1c^4) + (x[,"n_14"] + x[,"n_32"])*log10(102 - 28*r_rm1c - 93*r_rm1c^2 + 94*r_rm1c^3 - 36*r_rm1c^4) + (x[,"n_03"] + x[,"n_43"])*log10(9 + 18*r_rm1c - 34*r_rm1c^2 + 32*r_rm1c^3 - 16*r_rm1c^4) + (x[,"n_04"] + x[,"n_42"])*log10(18 - 32*r_rm1c + 34*r_rm1c^2 - 29*r_rm1c^3 + 9*r_rm1c^4) + (x[,"n_13"] + x[,"n_33"])*log10(63 - 14*r_rm1c + 46*r_rm1c^2 - 64*r_rm1c^3 + 32*r_rm1c^4) + (x[,"n_22"] + x[,"n_24"])*log10(138 - 31*r_rm1c + 85*r_rm1c^2 - 108*r_rm1c^3 + 54*r_rm1c^4)
logL_rm1c <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_05"] - 4*x[,"n_06"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_15"] - 4*x[,"n_16"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"] - 4*x[,"n_30"] - 3*x[,"n_31"] - 4*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"] - 3*x[,"n_35"] - 4*x[,"n_36"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 4*x[,"n_42"] - 4*x[,"n_43"] - 4*x[,"n_44"] - 4*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(log(3) + 2*log(5)) + (x[,"n_21"] + x[,"n_25"])*log(7) + (x[,"n_06"] + x[,"n_40"])*(3*log(pmax(1e-6,1 - r_rm1c)) + log(pmax(1e-6,r_rm1c))) + (x[,"n_01"] + x[,"n_45"])*(log(7 - 4*r_rm1c) + 2*log(pmax(1e-6,r_rm1c))) + (x[,"n_00"] + x[,"n_46"])*(log(pmax(1e-6,1 - r_rm1c)) + 3*log(pmax(1e-6,r_rm1c))) + (x[,"n_05"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_rm1c)) + log(3 + 4*r_rm1c)) + (x[,"n_21"] + x[,"n_25"])*log(3 + 2*r_rm1c - 2*r_rm1c^2) + (x[,"n_10"] + x[,"n_36"])*(2*log(pmax(1e-6,r_rm1c)) + log(5 - 6*r_rm1c + 4*r_rm1c^2)) + (x[,"n_11"] + x[,"n_35"])*(log(pmax(1e-6,r_rm1c)) + log(14 - 3*r_rm1c + 4*r_rm1c^2)) + (x[,"n_16"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_rm1c)) + log(3 - 2*r_rm1c + 4*r_rm1c^2)) + (x[,"n_20"] + x[,"n_26"])*(log(pmax(1e-6,r_rm1c)) + log(7 - 13*r_rm1c + 12*r_rm1c^2 - 6*r_rm1c^3)) + (x[,"n_15"] + x[,"n_31"])*log(15 - 20*r_rm1c + 9*r_rm1c^2 - 4*r_rm1c^3) + (x[,"n_02"] + x[,"n_44"])*(log(pmax(1e-6,r_rm1c)) + log(15 + r_rm1c - 7*r_rm1c^2 + 9*r_rm1c^3)) + x[,"n_23"]*log(111 + 10*r_rm1c - 58*r_rm1c^2 + 96*r_rm1c^3 - 48*r_rm1c^4) + (x[,"n_12"] + x[,"n_34"])*log(39 + 76*r_rm1c - 27*r_rm1c^2 + 50*r_rm1c^3 - 36*r_rm1c^4) + (x[,"n_14"] + x[,"n_32"])*log(102 - 28*r_rm1c - 93*r_rm1c^2 + 94*r_rm1c^3 - 36*r_rm1c^4) + (x[,"n_03"] + x[,"n_43"])*log(9 + 18*r_rm1c - 34*r_rm1c^2 + 32*r_rm1c^3 - 16*r_rm1c^4) + (x[,"n_04"] + x[,"n_42"])*log(18 - 32*r_rm1c + 34*r_rm1c^2 - 29*r_rm1c^3 + 9*r_rm1c^4) + (x[,"n_13"] + x[,"n_33"])*log(63 - 14*r_rm1c + 46*r_rm1c^2 - 64*r_rm1c^3 + 32*r_rm1c^4) + (x[,"n_22"] + x[,"n_24"])*log(138 - 31*r_rm1c + 85*r_rm1c^2 - 108*r_rm1c^3 + 54*r_rm1c^4)
logL_rr <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n05 - 4*n06 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n16 - 3*n20 - 3*n21 - 3*n22 - 2*n23 - 3*n24 - 3*n25 - 3*n26 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n36 - 4*n40 - 3*n41 - 4*n42 - 3*n43 - 4*n44 - 3*n45 - 4*n46)*log(2) + (n01 + n02 + n04 + n05 + n11 + n12 + n14 + n15 + n20 + n21 + n22 + n23 + n24 + n25 + n26 + n31 + n32 + n34 + n35 + n41 + n42 + n44 + n45)*log(3) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n05 - 2*n06 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n16 - 2*n20 - n21 - 2*n22 - 2*n23 - 2*n24 - n25 - 2*n26 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n36 - 2*n40 - 2*n41 - 2*n42 - 2*n43 - 2*n44 - 2*n45 - 2*n46)*log(5) + 4*(n06 + n40)*log(pmax(1e-6,1 - r)) + 3*(n05 + n41)*log(pmax(1e-6,1 - r)) + 3*(n01 + n45)*log(pmax(1e-6,r)) + 4*(n00 + n46)*log(pmax(1e-6,r)) + (n21 + n25)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n16 + n30)*(3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n26)*(2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n11 + n35)*(log(2 - r) + 2*log(pmax(1e-6,r))) + (n10 + n36)*(log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n15 + n31)*(2*log(pmax(1e-6,1 - r)) + log(1 + r)) + (n02 + n44)*(2*log(pmax(1e-6,r)) + log(4 - 3*r^2)) + (n04 + n42)*(2*log(pmax(1e-6,1 - r)) + log(1 + 6*r - 3*r^2)) + (n12 + n34)*(log(pmax(1e-6,r)) + log(4 + r - 3*r^2 + 3*r^3)) + (n03 + n43)*(log(pmax(1e-6,r)) + log(5 + 3*r - 16*r^2 + 8*r^3)) + (n13 + n33)*log(7 + 17*r - 33*r^2 + 32*r^3 - 16*r^4) + (n22 + n24)*log(6 + 8*r - 17*r^2 + 18*r^3 - 9*r^4) + (n14 + n32)*log(5 - 9*r + 10*r^2 - 9*r^3 + 3*r^4) + n23*log(9 - 13*r + 21*r^2 - 16*r^3 + 8*r^4)
return(L)}
interlogL_rr <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_rr,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_rr <- 2*(x[,"n_21"] + x[,"n_25"])*log10(2) + 4*(x[,"n_20"] + x[,"n_26"])*log10(2) + 4*(x[,"n_16"] + x[,"n_30"])*log10(2) + 4*(x[,"n_10"] + x[,"n_36"])*log10(2) + 4*(x[,"n_06"] + x[,"n_40"])*log10(2) + 3*(x[,"n_05"] + x[,"n_41"])*log10(2) + 3*(x[,"n_01"] + x[,"n_45"])*log10(2) + 4*(x[,"n_00"] + x[,"n_46"])*log10(2) - (x[,"n_15"] + x[,"n_31"])*(-3*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_35"])*(-3*log10(2) + log10(3)) - x[,"n_23"]*(-2*log10(2) + 2*log10(5)) - (x[,"n_03"] + x[,"n_43"])*(-2*log10(2) + log10(7)) - (x[,"n_14"] + x[,"n_32"])*(-4*log10(2) + log10(3) + log10(11)) - (x[,"n_12"] + x[,"n_34"])*(-4*log10(2) + log10(3) + log10(11)) - (x[,"n_04"] + x[,"n_42"])*(-4*log10(2) + log10(13)) - (x[,"n_02"] + x[,"n_44"])*(-4*log10(2) + log10(13)) - (x[,"n_22"] + x[,"n_24"])*(-4*log10(2) + log10(7) + log10(17)) - (x[,"n_13"] + x[,"n_33"])*(-2*log10(2) + log10(41)) + 4*(x[,"n_06"] + x[,"n_40"])*log10(pmax(1e-6,1 - r_rr)) + 3*(x[,"n_05"] + x[,"n_41"])*log10(pmax(1e-6,1 - r_rr)) + 3*(x[,"n_01"] + x[,"n_45"])*log10(pmax(1e-6,r_rr)) + 4*(x[,"n_00"] + x[,"n_46"])*log10(pmax(1e-6,r_rr)) + (x[,"n_21"] + x[,"n_25"])*(log10(pmax(1e-6,1 - r_rr)) + log10(pmax(1e-6,r_rr))) + (x[,"n_16"] + x[,"n_30"])*(3*log10(pmax(1e-6,1 - r_rr)) + log10(pmax(1e-6,r_rr))) + (x[,"n_20"] + x[,"n_26"])*(2*log10(pmax(1e-6,1 - r_rr)) + 2*log10(pmax(1e-6,r_rr))) + (x[,"n_11"] + x[,"n_35"])*(log10(2 - r_rr) + 2*log10(pmax(1e-6,r_rr))) + (x[,"n_10"] + x[,"n_36"])*(log10(pmax(1e-6,1 - r_rr)) + 3*log10(pmax(1e-6,r_rr))) + (x[,"n_15"] + x[,"n_31"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(1 + r_rr)) + (x[,"n_02"] + x[,"n_44"])*(2*log10(pmax(1e-6,r_rr)) + log10(4 - 3*r_rr^2)) + (x[,"n_04"] + x[,"n_42"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(1 + 6*r_rr - 3*r_rr^2)) + (x[,"n_12"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(4 + r_rr - 3*r_rr^2 + 3*r_rr^3)) + (x[,"n_03"] + x[,"n_43"])*(log10(pmax(1e-6,r_rr)) + log10(5 + 3*r_rr - 16*r_rr^2 + 8*r_rr^3)) + (x[,"n_13"] + x[,"n_33"])*log10(7 + 17*r_rr - 33*r_rr^2 + 32*r_rr^3 - 16*r_rr^4) + (x[,"n_22"] + x[,"n_24"])*log10(6 + 8*r_rr - 17*r_rr^2 + 18*r_rr^3 - 9*r_rr^4) + (x[,"n_14"] + x[,"n_32"])*log10(5 - 9*r_rr + 10*r_rr^2 - 9*r_rr^3 + 3*r_rr^4) + x[,"n_23"]*log10(9 - 13*r_rr + 21*r_rr^2 - 16*r_rr^3 + 8*r_rr^4)
logL_rr <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_05"] - 4*x[,"n_06"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_16"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"] - 3*x[,"n_25"] - 3*x[,"n_26"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_36"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 3*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_04"] + x[,"n_05"] + x[,"n_11"] + x[,"n_12"] + x[,"n_14"] + x[,"n_15"] + x[,"n_20"] + x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"] + x[,"n_25"] + x[,"n_26"] + x[,"n_31"] + x[,"n_32"] + x[,"n_34"] + x[,"n_35"] + x[,"n_41"] + x[,"n_42"] + x[,"n_44"] + x[,"n_45"])*log(3) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_06"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_16"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_25"] - 2*x[,"n_26"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_36"] - 2*x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - 2*x[,"n_45"] - 2*x[,"n_46"])*log(5) + 4*(x[,"n_06"] + x[,"n_40"])*log(pmax(1e-6,1 - r_rr)) + 3*(x[,"n_05"] + x[,"n_41"])*log(pmax(1e-6,1 - r_rr)) + 3*(x[,"n_01"] + x[,"n_45"])*log(pmax(1e-6,r_rr)) + 4*(x[,"n_00"] + x[,"n_46"])*log(pmax(1e-6,r_rr)) + (x[,"n_21"] + x[,"n_25"])*(log(pmax(1e-6,1 - r_rr)) + log(pmax(1e-6,r_rr))) + (x[,"n_16"] + x[,"n_30"])*(3*log(pmax(1e-6,1 - r_rr)) + log(pmax(1e-6,r_rr))) + (x[,"n_20"] + x[,"n_26"])*(2*log(pmax(1e-6,1 - r_rr)) + 2*log(pmax(1e-6,r_rr))) + (x[,"n_11"] + x[,"n_35"])*(log(2 - r_rr) + 2*log(pmax(1e-6,r_rr))) + (x[,"n_10"] + x[,"n_36"])*(log(pmax(1e-6,1 - r_rr)) + 3*log(pmax(1e-6,r_rr))) + (x[,"n_15"] + x[,"n_31"])*(2*log(pmax(1e-6,1 - r_rr)) + log(1 + r_rr)) + (x[,"n_02"] + x[,"n_44"])*(2*log(pmax(1e-6,r_rr)) + log(4 - 3*r_rr^2)) + (x[,"n_04"] + x[,"n_42"])*(2*log(pmax(1e-6,1 - r_rr)) + log(1 + 6*r_rr - 3*r_rr^2)) + (x[,"n_12"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(4 + r_rr - 3*r_rr^2 + 3*r_rr^3)) + (x[,"n_03"] + x[,"n_43"])*(log(pmax(1e-6,r_rr)) + log(5 + 3*r_rr - 16*r_rr^2 + 8*r_rr^3)) + (x[,"n_13"] + x[,"n_33"])*log(7 + 17*r_rr - 33*r_rr^2 + 32*r_rr^3 - 16*r_rr^4) + (x[,"n_22"] + x[,"n_24"])*log(6 + 8*r_rr - 17*r_rr^2 + 18*r_rr^3 - 9*r_rr^4) + (x[,"n_14"] + x[,"n_32"])*log(5 - 9*r_rr + 10*r_rr^2 - 9*r_rr^3 + 3*r_rr^4) + x[,"n_23"]*log(9 - 13*r_rr + 21*r_rr^2 - 16*r_rr^3 + 8*r_rr^4)

return(list(
r_mat = cbind(r_cc,r_cm2c,r_cm1c,r_cr,r_rc,r_rm2c,r_rm1c,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm2c,LOD_cm1c,LOD_cr,LOD_rc,LOD_rm2c,LOD_rm1c,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm2c,logL_cm1c,logL_cr,logL_rc,logL_rm2c,logL_rm1c,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed2c","coupling mixed1c","coupling repulsion","repulsion coupling","repulsion mixed2c","repulsion mixed1c","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.3_4.1 <- function(x){
logL_cc <- function(r,n21,n22,n23,n24,n01,n02,n03,n04,n11,n12,n13,n14,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (n21 + n22 + n23 + n24)*log(3) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(4*log(2) + 2*log(5)) + (n01 + n44)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n04 + n41)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n12 + n33)*log(51 - 20*r - 4*r^2) + (n21 + n24)*log(5 + 4*r - 4*r^2) + (n13 + n32)*log(27 + 28*r - 4*r^2) + (n02 + n43)*log(6 - 3*r - 2*r^2) + (n03 + n42)*log(1 + 7*r - 2*r^2) + (n11 + n34)*log(19 - 20*r + 4*r^2) + (n22 + n23)*log(25 - 4*r + 4*r^2) + (n14 + n31)*log(3 + 12*r + 4*r^2)
return(L)}
interlogL_cc <- function(n21,n22,n23,n24,n01,n02,n03,n04,n11,n12,n13,n14,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_cc,c(0,0.5), n21,n22,n23,n24,n01,n02,n03,n04,n11,n12,n13,n14,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cc <- -2*(x[,"n_03"] + x[,"n_42"])*log10(2) - 2*(x[,"n_02"] + x[,"n_43"])*log10(2) - (x[,"n_21"] + x[,"n_24"])*(log10(2) + log10(3)) - (x[,"n_22"] + x[,"n_23"])*(3*log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_31"])*(log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_34"])*(log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_32"])*(3*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_33"])*(3*log10(2) + log10(5)) + (x[,"n_01"] + x[,"n_44"])*(log10(3 - 2*r_cc) + log10(pmax(1e-6,1 - r_cc))) + (x[,"n_04"] + x[,"n_41"])*(log10(pmax(1e-6,r_cc)) + log10(1 + 2*r_cc)) + (x[,"n_12"] + x[,"n_33"])*log10(51 - 20*r_cc - 4*r_cc^2) + (x[,"n_21"] + x[,"n_24"])*log10(5 + 4*r_cc - 4*r_cc^2) + (x[,"n_13"] + x[,"n_32"])*log10(27 + 28*r_cc - 4*r_cc^2) + (x[,"n_02"] + x[,"n_43"])*log10(6 - 3*r_cc - 2*r_cc^2) + (x[,"n_03"] + x[,"n_42"])*log10(1 + 7*r_cc - 2*r_cc^2) + (x[,"n_11"] + x[,"n_34"])*log10(19 - 20*r_cc + 4*r_cc^2) + (x[,"n_22"] + x[,"n_23"])*log10(25 - 4*r_cc + 4*r_cc^2) + (x[,"n_14"] + x[,"n_31"])*log10(3 + 12*r_cc + 4*r_cc^2)
logL_cc <- (x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"])*log(3) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(4*log(2) + 2*log(5)) + (x[,"n_01"] + x[,"n_44"])*(log(3 - 2*r_cc) + log(pmax(1e-6,1 - r_cc))) + (x[,"n_04"] + x[,"n_41"])*(log(pmax(1e-6,r_cc)) + log(1 + 2*r_cc)) + (x[,"n_12"] + x[,"n_33"])*log(51 - 20*r_cc - 4*r_cc^2) + (x[,"n_21"] + x[,"n_24"])*log(5 + 4*r_cc - 4*r_cc^2) + (x[,"n_13"] + x[,"n_32"])*log(27 + 28*r_cc - 4*r_cc^2) + (x[,"n_02"] + x[,"n_43"])*log(6 - 3*r_cc - 2*r_cc^2) + (x[,"n_03"] + x[,"n_42"])*log(1 + 7*r_cc - 2*r_cc^2) + (x[,"n_11"] + x[,"n_34"])*log(19 - 20*r_cc + 4*r_cc^2) + (x[,"n_22"] + x[,"n_23"])*log(25 - 4*r_cc + 4*r_cc^2) + (x[,"n_14"] + x[,"n_31"])*log(3 + 12*r_cc + 4*r_cc^2)
logL_cr <- function(r,n21,n22,n23,n24,n01,n02,n03,n04,n11,n12,n13,n14,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (n21 + n22 + n23 + n24)*log(3) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(4*log(2) + 2*log(5)) + (n01 + n44)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n04 + n41)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n11 + n14 + n31 + n34)*log(9 + 4*r - 4*r^2) + (n22 + n23)*log(23 + 4*r - 4*r^2) + (n03 + n42)*log(6 - 5*r + 2*r^2) + (n02 + n43)*log(3 + r + 2*r^2) + (n13 + n32)*log(45 - 12*r + 4*r^2) + (n21 + n24)*log(7 - 4*r + 4*r^2) + (n12 + n33)*log(37 + 4*r + 4*r^2)
return(L)}
interlogL_cr <- function(n21,n22,n23,n24,n01,n02,n03,n04,n11,n12,n13,n14,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_cr,c(0,0.5), n21,n22,n23,n24,n01,n02,n03,n04,n11,n12,n13,n14,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cr <- -2*(x[,"n_03"] + x[,"n_42"])*log10(2) - 2*(x[,"n_02"] + x[,"n_43"])*log10(2) - (x[,"n_21"] + x[,"n_24"])*(log10(2) + log10(3)) - (x[,"n_22"] + x[,"n_23"])*(3*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_14"] + x[,"n_31"] + x[,"n_34"])*(log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_32"])*(3*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_33"])*(3*log10(2) + log10(5)) + (x[,"n_01"] + x[,"n_44"])*(log10(3 - 2*r_cr) + log10(pmax(1e-6,r_cr))) + (x[,"n_04"] + x[,"n_41"])*(log10(pmax(1e-6,1 - r_cr)) + log10(1 + 2*r_cr)) + (x[,"n_11"] + x[,"n_14"] + x[,"n_31"] + x[,"n_34"])*log10(9 + 4*r_cr - 4*r_cr^2) + (x[,"n_22"] + x[,"n_23"])*log10(23 + 4*r_cr - 4*r_cr^2) + (x[,"n_03"] + x[,"n_42"])*log10(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_02"] + x[,"n_43"])*log10(3 + r_cr + 2*r_cr^2) + (x[,"n_13"] + x[,"n_32"])*log10(45 - 12*r_cr + 4*r_cr^2) + (x[,"n_21"] + x[,"n_24"])*log10(7 - 4*r_cr + 4*r_cr^2) + (x[,"n_12"] + x[,"n_33"])*log10(37 + 4*r_cr + 4*r_cr^2)
logL_cr <- (x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"])*log(3) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(4*log(2) + 2*log(5)) + (x[,"n_01"] + x[,"n_44"])*(log(3 - 2*r_cr) + log(pmax(1e-6,r_cr))) + (x[,"n_04"] + x[,"n_41"])*(log(pmax(1e-6,1 - r_cr)) + log(1 + 2*r_cr)) + (x[,"n_11"] + x[,"n_14"] + x[,"n_31"] + x[,"n_34"])*log(9 + 4*r_cr - 4*r_cr^2) + (x[,"n_22"] + x[,"n_23"])*log(23 + 4*r_cr - 4*r_cr^2) + (x[,"n_03"] + x[,"n_42"])*log(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_02"] + x[,"n_43"])*log(3 + r_cr + 2*r_cr^2) + (x[,"n_13"] + x[,"n_32"])*log(45 - 12*r_cr + 4*r_cr^2) + (x[,"n_21"] + x[,"n_24"])*log(7 - 4*r_cr + 4*r_cr^2) + (x[,"n_12"] + x[,"n_33"])*log(37 + 4*r_cr + 4*r_cr^2)
logL_rc <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-2*n01 - 3*n02 - 3*n03 - 2*n04 - 2*n11 - 3*n12 - 3*n13 - 2*n14 - n21 - 3*n22 - 3*n23 - n24 - 2*n31 - 3*n32 - 3*n33 - 2*n34 - 2*n41 - 3*n42 - 3*n43 - 2*n44)*log(2) + (n21 + n22 + n23 + n24)*log(3) + 2*(-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*log(5) + (n01 + n04 + n41 + n44)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n22 + n23)*log(11 + 4*r - 4*r^2) + (n14 + n31)*log(3 - 2*r^2) + (n11 + n34)*log(1 + 4*r - 2*r^2) + (n21 + n24)*log(1 - r + r^2) + (n02 + n43)*log(3 - 3*r + 2*r^2) + (n03 + n42)*log(2 - r + 2*r^2) + (n12 + n13 + n32 + n33)*log(21 - 4*r + 4*r^2)
return(L)}
interlogL_rc <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_rc,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rc <- -((x[,"n_03"] + x[,"n_42"])*log10(2)) - (x[,"n_02"] + x[,"n_43"])*log10(2) + 2*(x[,"n_01"] + x[,"n_04"] + x[,"n_41"] + x[,"n_44"])*log10(2) + (x[,"n_21"] + x[,"n_24"])*(2*log10(2) - log10(3)) - (x[,"n_22"] + x[,"n_23"])*(2*log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_31"])*(-log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_34"])*(-log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_13"] + x[,"n_32"] + x[,"n_33"])*(2*log10(2) + log10(5)) + (x[,"n_01"] + x[,"n_04"] + x[,"n_41"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc))) + (x[,"n_22"] + x[,"n_23"])*log10(11 + 4*r_rc - 4*r_rc^2) + (x[,"n_14"] + x[,"n_31"])*log10(3 - 2*r_rc^2) + (x[,"n_11"] + x[,"n_34"])*log10(1 + 4*r_rc - 2*r_rc^2) + (x[,"n_21"] + x[,"n_24"])*log10(1 - r_rc + r_rc^2) + (x[,"n_02"] + x[,"n_43"])*log10(3 - 3*r_rc + 2*r_rc^2) + (x[,"n_03"] + x[,"n_42"])*log10(2 - r_rc + 2*r_rc^2) + (x[,"n_12"] + x[,"n_13"] + x[,"n_32"] + x[,"n_33"])*log10(21 - 4*r_rc + 4*r_rc^2)
logL_rc <- (-2*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + (x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_01"] + x[,"n_04"] + x[,"n_41"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc))) + (x[,"n_22"] + x[,"n_23"])*log(11 + 4*r_rc - 4*r_rc^2) + (x[,"n_14"] + x[,"n_31"])*log(3 - 2*r_rc^2) + (x[,"n_11"] + x[,"n_34"])*log(1 + 4*r_rc - 2*r_rc^2) + (x[,"n_21"] + x[,"n_24"])*log(1 - r_rc + r_rc^2) + (x[,"n_02"] + x[,"n_43"])*log(3 - 3*r_rc + 2*r_rc^2) + (x[,"n_03"] + x[,"n_42"])*log(2 - r_rc + 2*r_rc^2) + (x[,"n_12"] + x[,"n_13"] + x[,"n_32"] + x[,"n_33"])*log(21 - 4*r_rc + 4*r_rc^2)
logL_rr <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-2*n01 - 3*n02 - 3*n03 - 2*n04 - n11 - 3*n12 - 3*n13 - n14 - 2*n21 - 3*n22 - 3*n23 - 2*n24 - n31 - 3*n32 - 3*n33 - n34 - 2*n41 - 3*n42 - 3*n43 - 2*n44)*log(2) + (n21 + n22 + n23 + n24)*log(3) + 2*(-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*log(5) + 2*(n04 + n41)*log(pmax(1e-6,1 - r)) + 2*(n01 + n44)*log(pmax(1e-6,r)) + (n02 + n43)*(log(5 - 2*r) + log(pmax(1e-6,r))) + (n11 + n34)*(log(pmax(1e-6,r)) + log(2 + r)) + (n13 + n32)*log(27 - 12*r - 4*r^2) + (n12 + n33)*log(11 + 20*r - 4*r^2) + (n03 + n42)*log(3 - r - 2*r^2) + (n21 + n24)*log(1 + 2*r - 2*r^2) + (n14 + n31)*log(3 - 4*r + r^2) + (n22 + n23)*log(13 - 4*r + 4*r^2)
return(L)}
interlogL_rr <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_rr,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rr <- 2*(x[,"n_04"] + x[,"n_41"])*log10(2) - (x[,"n_03"] + x[,"n_42"])*log10(2) - (x[,"n_02"] + x[,"n_43"])*log10(2) + 2*(x[,"n_01"] + x[,"n_44"])*log10(2) - (x[,"n_21"] + x[,"n_24"])*(-log10(2) + log10(3)) - (x[,"n_22"] + x[,"n_23"])*(2*log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_31"])*(-2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_34"])*(-2*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_32"])*(2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_33"])*(2*log10(2) + log10(5)) + 2*(x[,"n_04"] + x[,"n_41"])*log10(pmax(1e-6,1 - r_rr)) + 2*(x[,"n_01"] + x[,"n_44"])*log10(pmax(1e-6,r_rr)) + (x[,"n_02"] + x[,"n_43"])*(log10(5 - 2*r_rr) + log10(pmax(1e-6,r_rr))) + (x[,"n_11"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(2 + r_rr)) + (x[,"n_13"] + x[,"n_32"])*log10(27 - 12*r_rr - 4*r_rr^2) + (x[,"n_12"] + x[,"n_33"])*log10(11 + 20*r_rr - 4*r_rr^2) + (x[,"n_03"] + x[,"n_42"])*log10(3 - r_rr - 2*r_rr^2) + (x[,"n_21"] + x[,"n_24"])*log10(1 + 2*r_rr - 2*r_rr^2) + (x[,"n_14"] + x[,"n_31"])*log10(3 - 4*r_rr + r_rr^2) + (x[,"n_22"] + x[,"n_23"])*log10(13 - 4*r_rr + 4*r_rr^2)
logL_rr <- (-2*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - x[,"n_14"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - x[,"n_34"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + (x[,"n_21"] + x[,"n_22"] + x[,"n_23"] + x[,"n_24"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + 2*(x[,"n_04"] + x[,"n_41"])*log(pmax(1e-6,1 - r_rr)) + 2*(x[,"n_01"] + x[,"n_44"])*log(pmax(1e-6,r_rr)) + (x[,"n_02"] + x[,"n_43"])*(log(5 - 2*r_rr) + log(pmax(1e-6,r_rr))) + (x[,"n_11"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(2 + r_rr)) + (x[,"n_13"] + x[,"n_32"])*log(27 - 12*r_rr - 4*r_rr^2) + (x[,"n_12"] + x[,"n_33"])*log(11 + 20*r_rr - 4*r_rr^2) + (x[,"n_03"] + x[,"n_42"])*log(3 - r_rr - 2*r_rr^2) + (x[,"n_21"] + x[,"n_24"])*log(1 + 2*r_rr - 2*r_rr^2) + (x[,"n_14"] + x[,"n_31"])*log(3 - 4*r_rr + r_rr^2) + (x[,"n_22"] + x[,"n_23"])*log(13 - 4*r_rr + 4*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.4_1.4 <- function(x){
logL_cc <- function(r,n12,n13,n21,n24,n31,n34,n42,n43,n11,n14,n22,n23,n32,n33,n41,n44) {
L <- (n12 + n13 + n21 + n24 + n31 + n34 + n42 + n43)*log(3) + (-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(log(2) + log(5)) + 3*(n11 + n44)*log(pmax(1e-6,1 - r)) + 3*(n14 + n41)*log(pmax(1e-6,r)) + (n12 + n21 + n34 + n43)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n13 + n24 + n31 + n42)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n23 + n32)*(log(pmax(1e-6,r)) + log(7 - 12*r + 9*r^2)) + (n22 + n33)*log(4 - 10*r + 15*r^2 - 9*r^3)
return(L)}
interlogL_cc <- function(n12,n13,n21,n24,n31,n34,n42,n43,n11,n14,n22,n23,n32,n33,n41,n44) {
optimize(logL_cc,c(0,0.5), n12,n13,n21,n24,n31,n34,n42,n43,n11,n14,n22,n23,n32,n33,n41,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_24"],x[,"n_31"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_11"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_32"],x[,"n_33"],x[,"n_41"],x[,"n_44"])
LOD_cc <- 3*(x[,"n_14"] + x[,"n_41"])*log10(2) + 3*(x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log10(2) + 3*(x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log10(2) + 3*(x[,"n_11"] + x[,"n_44"])*log10(2) - (x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(13)) - (x[,"n_22"] + x[,"n_33"])*(-3*log10(2) + log10(13)) + 3*(x[,"n_11"] + x[,"n_44"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_14"] + x[,"n_41"])*log10(pmax(1e-6,r_cc)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(7 - 12*r_cc + 9*r_cc^2)) + (x[,"n_22"] + x[,"n_33"])*log10(4 - 10*r_cc + 15*r_cc^2 - 9*r_cc^3)
logL_cc <- (x[,"n_12"] + x[,"n_13"] + x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_42"] + x[,"n_43"])*log(3) + (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(log(2) + log(5)) + 3*(x[,"n_11"] + x[,"n_44"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_14"] + x[,"n_41"])*log(pmax(1e-6,r_cc)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(7 - 12*r_cc + 9*r_cc^2)) + (x[,"n_22"] + x[,"n_33"])*log(4 - 10*r_cc + 15*r_cc^2 - 9*r_cc^3)
logL_cm3c <- function(r,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(3*log(2) + log(5)) + (n14 + n41)*(log(2 - r) + 2*log(pmax(1e-6,r))) + (n11 + n44)*(log(pmax(1e-6,1 - r)) + log(1 - r^2)) + (n13 + n24 + n31 + n42)*(log(pmax(1e-6,r)) + log(5 - 5*r + 3*r^2)) + (n23 + n32)*log(6 - 4*r + 14*r^2 - 9*r^3) + (n12 + n21 + n34 + n43)*log(3 - 4*r + 4*r^2 - 3*r^3) + (n22 + n33)*log(7 + 3*r - 13*r^2 + 9*r^3)
return(L)}
interlogL_cm3c <- function(n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_cm3c,c(0,0.5), n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm3c <- mapply(interlogL_cm3c,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cm3c <- -((x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(3))) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(-3*log10(2) + log10(13)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*(-3*log10(2) + log10(13)) - (x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(3) + log10(17)) - (x[,"n_22"] + x[,"n_33"])*(-3*log10(2) + log10(3) + log10(17)) + (x[,"n_14"] + x[,"n_41"])*(log10(2 - r_cm3c) + 2*log10(pmax(1e-6,r_cm3c))) + (x[,"n_11"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cm3c)) + log10(1 - r_cm3c^2)) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(log10(pmax(1e-6,r_cm3c)) + log10(5 - 5*r_cm3c + 3*r_cm3c^2)) + (x[,"n_23"] + x[,"n_32"])*log10(6 - 4*r_cm3c + 14*r_cm3c^2 - 9*r_cm3c^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log10(3 - 4*r_cm3c + 4*r_cm3c^2 - 3*r_cm3c^3) + (x[,"n_22"] + x[,"n_33"])*log10(7 + 3*r_cm3c - 13*r_cm3c^2 + 9*r_cm3c^3)
logL_cm3c <- (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(3*log(2) + log(5)) + (x[,"n_14"] + x[,"n_41"])*(log(2 - r_cm3c) + 2*log(pmax(1e-6,r_cm3c))) + (x[,"n_11"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cm3c)) + log(1 - r_cm3c^2)) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(log(pmax(1e-6,r_cm3c)) + log(5 - 5*r_cm3c + 3*r_cm3c^2)) + (x[,"n_23"] + x[,"n_32"])*log(6 - 4*r_cm3c + 14*r_cm3c^2 - 9*r_cm3c^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log(3 - 4*r_cm3c + 4*r_cm3c^2 - 3*r_cm3c^3) + (x[,"n_22"] + x[,"n_33"])*log(7 + 3*r_cm3c - 13*r_cm3c^2 + 9*r_cm3c^3)
logL_cm2c <- function(r,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(2*log(2) + log(3) + log(5)) + (n11 + n44)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 + r)) + (n14 + n41)*(log(pmax(1e-6,r)) + log(3 - 4*r + r^2)) + (n22 + n33)*log(12 - 8*r + 11*r^2 - 9*r^3) + (n13 + n24 + n31 + n42)*log(3 - 4*r + 7*r^2 - 3*r^3) + (n12 + n21 + n34 + n43)*log(3 - r - 2*r^2 + 3*r^3) + (n23 + n32)*log(6 + 13*r - 16*r^2 + 9*r^3)
return(L)}
interlogL_cm2c <- function(n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_cm2c,c(0,0.5), n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cm2c <- -((x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(5))) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(7) + log10(11)) - (x[,"n_22"] + x[,"n_33"])*(-3*log10(2) + log10(7) + log10(11)) - (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(-3*log10(2) + log10(19)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*(-3*log10(2) + log10(19)) + (x[,"n_11"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c)) + log10(2 + r_cm2c)) + (x[,"n_14"] + x[,"n_41"])*(log10(pmax(1e-6,r_cm2c)) + log10(3 - 4*r_cm2c + r_cm2c^2)) + (x[,"n_22"] + x[,"n_33"])*log10(12 - 8*r_cm2c + 11*r_cm2c^2 - 9*r_cm2c^3) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log10(3 - 4*r_cm2c + 7*r_cm2c^2 - 3*r_cm2c^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log10(3 - r_cm2c - 2*r_cm2c^2 + 3*r_cm2c^3) + (x[,"n_23"] + x[,"n_32"])*log10(6 + 13*r_cm2c - 16*r_cm2c^2 + 9*r_cm2c^3)
logL_cm2c <- (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(2*log(2) + log(3) + log(5)) + (x[,"n_11"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c)) + log(2 + r_cm2c)) + (x[,"n_14"] + x[,"n_41"])*(log(pmax(1e-6,r_cm2c)) + log(3 - 4*r_cm2c + r_cm2c^2)) + (x[,"n_22"] + x[,"n_33"])*log(12 - 8*r_cm2c + 11*r_cm2c^2 - 9*r_cm2c^3) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log(3 - 4*r_cm2c + 7*r_cm2c^2 - 3*r_cm2c^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log(3 - r_cm2c - 2*r_cm2c^2 + 3*r_cm2c^3) + (x[,"n_23"] + x[,"n_32"])*log(6 + 13*r_cm2c - 16*r_cm2c^2 + 9*r_cm2c^3)
logL_rc <- function(r,n12,n13,n21,n24,n31,n34,n42,n43,n11,n14,n22,n23,n32,n33,n41,n44) {
L <- (n12 + n13 + n21 + n24 + n31 + n34 + n42 + n43)*log(3) + (-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(log(2) + 2*log(5)) + (n14 + n41)*(log(3 - r) + 2*log(pmax(1e-6,r))) + (n11 + n44)*(2*log(pmax(1e-6,1 - r)) + log(2 + r)) + (n13 + n24 + n31 + n42)*(log(pmax(1e-6,r)) + log(2 - 2*r + r^2)) + (n23 + n32)*log(9 - 7*r + 15*r^2 - 9*r^3) + (n12 + n21 + n34 + n43)*log(1 - r + r^2 - r^3) + (n22 + n33)*log(8 + 4*r - 12*r^2 + 9*r^3)
return(L)}
interlogL_rc <- function(n12,n13,n21,n24,n31,n34,n42,n43,n11,n14,n22,n23,n32,n33,n41,n44) {
optimize(logL_rc,c(0,0.5), n12,n13,n21,n24,n31,n34,n42,n43,n11,n14,n22,n23,n32,n33,n41,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_24"],x[,"n_31"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_11"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_32"],x[,"n_33"],x[,"n_41"],x[,"n_44"])
LOD_rc <- (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*(3*log10(2) - log10(5)) - (x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(-3*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(5) + log10(13)) - (x[,"n_22"] + x[,"n_33"])*(-3*log10(2) + log10(5) + log10(13)) + (x[,"n_14"] + x[,"n_41"])*(log10(3 - r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_11"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(2 + r_rc)) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 2*r_rc + r_rc^2)) + (x[,"n_23"] + x[,"n_32"])*log10(9 - 7*r_rc + 15*r_rc^2 - 9*r_rc^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log10(1 - r_rc + r_rc^2 - r_rc^3) + (x[,"n_22"] + x[,"n_33"])*log10(8 + 4*r_rc - 12*r_rc^2 + 9*r_rc^3)
logL_rc <- (x[,"n_12"] + x[,"n_13"] + x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_42"] + x[,"n_43"])*log(3) + (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(log(2) + 2*log(5)) + (x[,"n_14"] + x[,"n_41"])*(log(3 - r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_11"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_rc)) + log(2 + r_rc)) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(log(pmax(1e-6,r_rc)) + log(2 - 2*r_rc + r_rc^2)) + (x[,"n_23"] + x[,"n_32"])*log(9 - 7*r_rc + 15*r_rc^2 - 9*r_rc^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log(1 - r_rc + r_rc^2 - r_rc^3) + (x[,"n_22"] + x[,"n_33"])*log(8 + 4*r_rc - 12*r_rc^2 + 9*r_rc^3)
logL_rm3c <- function(r,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(3*log(2) + 2*log(5)) + (n14 + n41)*(log(2 - r) + log(3 - r) + log(pmax(1e-6,r))) + (n11 + n44)*(log(2 + r) + log(1 - r^2)) + (n22 + n33)*log(32 - 3*r + 10*r^2 - 9*r^3) + (n13 + n24 + n31 + n42)*log(9 - 5*r + 8*r^2 - 3*r^3) + (n12 + n21 + n34 + n43)*log(9 - 2*r - r^2 + 3*r^3) + (n23 + n32)*log(30 + 10*r - 17*r^2 + 9*r^3)
return(L)}
interlogL_rm3c <- function(n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_rm3c,c(0,0.5), n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm3c <- mapply(interlogL_rm3c,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm3c <- -((x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(3) + log10(5))) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(3) + log10(5)) - (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(-3*log10(2) + log10(5) + log10(13)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*(-3*log10(2) + log10(5) + log10(13)) - (x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(3) + log10(5) + log10(17)) - (x[,"n_22"] + x[,"n_33"])*(-3*log10(2) + log10(3) + log10(5) + log10(17)) + (x[,"n_14"] + x[,"n_41"])*(log10(2 - r_rm3c) + log10(3 - r_rm3c) + log10(pmax(1e-6,r_rm3c))) + (x[,"n_11"] + x[,"n_44"])*(log10(2 + r_rm3c) + log10(1 - r_rm3c^2)) + (x[,"n_22"] + x[,"n_33"])*log10(32 - 3*r_rm3c + 10*r_rm3c^2 - 9*r_rm3c^3) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log10(9 - 5*r_rm3c + 8*r_rm3c^2 - 3*r_rm3c^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log10(9 - 2*r_rm3c - r_rm3c^2 + 3*r_rm3c^3) + (x[,"n_23"] + x[,"n_32"])*log10(30 + 10*r_rm3c - 17*r_rm3c^2 + 9*r_rm3c^3)
logL_rm3c <- (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(3*log(2) + 2*log(5)) + (x[,"n_14"] + x[,"n_41"])*(log(2 - r_rm3c) + log(3 - r_rm3c) + log(pmax(1e-6,r_rm3c))) + (x[,"n_11"] + x[,"n_44"])*(log(2 + r_rm3c) + log(1 - r_rm3c^2)) + (x[,"n_22"] + x[,"n_33"])*log(32 - 3*r_rm3c + 10*r_rm3c^2 - 9*r_rm3c^3) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log(9 - 5*r_rm3c + 8*r_rm3c^2 - 3*r_rm3c^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log(9 - 2*r_rm3c - r_rm3c^2 + 3*r_rm3c^3) + (x[,"n_23"] + x[,"n_32"])*log(30 + 10*r_rm3c - 17*r_rm3c^2 + 9*r_rm3c^3)
logL_rm2c <- function(r,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(2*log(2) + log(3) + 2*log(5)) + (n11 + n44)*(log(pmax(1e-6,r)) + 2*log(2 + r)) + (n14 + n41)*(log(3 - r) + log(3 - 4*r + r^2)) + (n23 + n32)*log(57 - 25*r + 19*r^2 - 9*r^3) + (n12 + n21 + n34 + n43)*log(6 + 13*r - r^2 - 3*r^3) + (n13 + n24 + n31 + n42)*log(15 - 2*r - 10*r^2 + 3*r^3) + (n22 + n33)*log(42 + 14*r - 8*r^2 + 9*r^3)
return(L)}
interlogL_rm2c <- function(n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_rm2c,c(0,0.5), n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm2c <- -((x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + 2*log10(5))) - (x[,"n_11"] + x[,"n_44"])*(-log10(2) + 2*(-log10(2) + log10(5))) - (x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(5) + log10(7) + log10(11)) - (x[,"n_22"] + x[,"n_33"])*(-3*log10(2) + log10(5) + log10(7) + log10(11)) - (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*(-3*log10(2) + log10(5) + log10(19)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*(-3*log10(2) + log10(5) + log10(19)) + (x[,"n_11"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm2c)) + 2*log10(2 + r_rm2c)) + (x[,"n_14"] + x[,"n_41"])*(log10(3 - r_rm2c) + log10(3 - 4*r_rm2c + r_rm2c^2)) + (x[,"n_23"] + x[,"n_32"])*log10(57 - 25*r_rm2c + 19*r_rm2c^2 - 9*r_rm2c^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log10(6 + 13*r_rm2c - r_rm2c^2 - 3*r_rm2c^3) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log10(15 - 2*r_rm2c - 10*r_rm2c^2 + 3*r_rm2c^3) + (x[,"n_22"] + x[,"n_33"])*log10(42 + 14*r_rm2c - 8*r_rm2c^2 + 9*r_rm2c^3)
logL_rm2c <- (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(2*log(2) + log(3) + 2*log(5)) + (x[,"n_11"] + x[,"n_44"])*(log(pmax(1e-6,r_rm2c)) + 2*log(2 + r_rm2c)) + (x[,"n_14"] + x[,"n_41"])*(log(3 - r_rm2c) + log(3 - 4*r_rm2c + r_rm2c^2)) + (x[,"n_23"] + x[,"n_32"])*log(57 - 25*r_rm2c + 19*r_rm2c^2 - 9*r_rm2c^3) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log(6 + 13*r_rm2c - r_rm2c^2 - 3*r_rm2c^3) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log(15 - 2*r_rm2c - 10*r_rm2c^2 + 3*r_rm2c^3) + (x[,"n_22"] + x[,"n_33"])*log(42 + 14*r_rm2c - 8*r_rm2c^2 + 9*r_rm2c^3)

return(list(
r_mat = cbind(r_cc,r_cm3c,r_cm2c,r_rc,r_rm3c,r_rm2c,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm3c,LOD_cm2c,LOD_rc,LOD_rm3c,LOD_rm2c,0),
logL_mat = cbind(logL_cc,logL_cm3c,logL_cm2c,logL_rc,logL_rm3c,logL_rm2c,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed3c","coupling mixed2c","repulsion coupling","repulsion mixed3c","repulsion mixed2c","unknown")
)
)
}

#' @rdname r6_functions
r6_1.4_1.5 <- function(x){
logL_cc <- function(r,n12,n14,n22,n23,n24,n32,n33,n34,n42,n44,n13,n43) {
L <- (-n12 - n14 - 2*n22 - 2*n23 - 2*n24 - 2*n32 - 2*n33 - 2*n34 - n42 - n44)*log(2) + (-n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*log(5) + 2*(n12 + n44)*log(pmax(1e-6,1 - r)) + 2*(n14 + n42)*log(pmax(1e-6,r)) + (n24 + n32)*(log(5 - 2*r) + log(pmax(1e-6,r))) + (n13 + n43)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n22 + n34)*log(3 - r - 2*r^2) + (n23 + n33)*log(5 - 4*r + 4*r^2)
return(L)}
interlogL_cc <- function(n12,n14,n22,n23,n24,n32,n33,n34,n42,n44,n13,n43) {
optimize(logL_cc,c(0,0.5), n12,n14,n22,n23,n24,n32,n33,n34,n42,n44,n13,n43, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_12"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_44"],x[,"n_13"],x[,"n_43"])
LOD_cc <- -((x[,"n_24"] + x[,"n_32"])*log10(2)) - 2*(x[,"n_23"] + x[,"n_33"])*log10(2) - (x[,"n_22"] + x[,"n_34"])*log10(2) + 2*(x[,"n_14"] + x[,"n_42"])*log10(2) + 2*(x[,"n_13"] + x[,"n_43"])*log10(2) + 2*(x[,"n_12"] + x[,"n_44"])*log10(2) + 2*(x[,"n_12"] + x[,"n_44"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_14"] + x[,"n_42"])*log10(pmax(1e-6,r_cc)) + (x[,"n_24"] + x[,"n_32"])*(log10(5 - 2*r_cc) + log10(pmax(1e-6,r_cc))) + (x[,"n_13"] + x[,"n_43"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_22"] + x[,"n_34"])*log10(3 - r_cc - 2*r_cc^2) + (x[,"n_23"] + x[,"n_33"])*log10(5 - 4*r_cc + 4*r_cc^2)
logL_cc <- (-x[,"n_12"] - x[,"n_14"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - x[,"n_42"] - x[,"n_44"])*log(2) + (-x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + 2*(x[,"n_12"] + x[,"n_44"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_14"] + x[,"n_42"])*log(pmax(1e-6,r_cc)) + (x[,"n_24"] + x[,"n_32"])*(log(5 - 2*r_cc) + log(pmax(1e-6,r_cc))) + (x[,"n_13"] + x[,"n_43"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_22"] + x[,"n_34"])*log(3 - r_cc - 2*r_cc^2) + (x[,"n_23"] + x[,"n_33"])*log(5 - 4*r_cc + 4*r_cc^2)
logL_cm3c <- function(r,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
L <- (-n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*(3*log(2) + log(5)) + (n14 + n42)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n12 + n44)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n23 + n33)*log(7 + 4*r - 4*r^2) + (n22 + n34)*log(6 - 5*r + 2*r^2) + (n24 + n32)*log(3 + r + 2*r^2) + (n13 + n43)*log(3 - 4*r + 4*r^2)
return(L)}
interlogL_cm3c <- function(n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
optimize(logL_cm3c,c(0,0.5), n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm3c <- mapply(interlogL_cm3c,x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cm3c <- -2*(x[,"n_24"] + x[,"n_32"])*log10(2) - 3*(x[,"n_23"] + x[,"n_33"])*log10(2) - 2*(x[,"n_22"] + x[,"n_34"])*log10(2) - (x[,"n_13"] + x[,"n_43"])*log10(2) + (x[,"n_14"] + x[,"n_42"])*(log10(3 - 2*r_cm3c) + log10(pmax(1e-6,r_cm3c))) + (x[,"n_12"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cm3c)) + log10(1 + 2*r_cm3c)) + (x[,"n_23"] + x[,"n_33"])*log10(7 + 4*r_cm3c - 4*r_cm3c^2) + (x[,"n_22"] + x[,"n_34"])*log10(6 - 5*r_cm3c + 2*r_cm3c^2) + (x[,"n_24"] + x[,"n_32"])*log10(3 + r_cm3c + 2*r_cm3c^2) + (x[,"n_13"] + x[,"n_43"])*log10(3 - 4*r_cm3c + 4*r_cm3c^2)
logL_cm3c <- (-x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(3*log(2) + log(5)) + (x[,"n_14"] + x[,"n_42"])*(log(3 - 2*r_cm3c) + log(pmax(1e-6,r_cm3c))) + (x[,"n_12"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cm3c)) + log(1 + 2*r_cm3c)) + (x[,"n_23"] + x[,"n_33"])*log(7 + 4*r_cm3c - 4*r_cm3c^2) + (x[,"n_22"] + x[,"n_34"])*log(6 - 5*r_cm3c + 2*r_cm3c^2) + (x[,"n_24"] + x[,"n_32"])*log(3 + r_cm3c + 2*r_cm3c^2) + (x[,"n_13"] + x[,"n_43"])*log(3 - 4*r_cm3c + 4*r_cm3c^2)
logL_rc <- function(r,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
L <- (-n12 - n13 - n14 - 2*n22 - 2*n23 - 2*n24 - 2*n32 - 2*n33 - 2*n34 - n42 - n43 - n44)*log(2) + 2*(-n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*log(5) + (n14 + n42)*(log(3 - r) + log(pmax(1e-6,r))) + (n12 + n44)*(log(pmax(1e-6,1 - r)) + log(2 + r)) + (n23 + n33)*log(19 + 4*r - 4*r^2) + (n22 + n34)*log(12 - 5*r + 2*r^2) + (n13 + n43)*log(3 - 2*r + 2*r^2) + (n24 + n32)*log(9 + r + 2*r^2)
return(L)}
interlogL_rc <- function(n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
optimize(logL_rc,c(0,0.5), n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rc <- -((x[,"n_14"] + x[,"n_42"])*(-2*log10(2) + log10(5))) - (x[,"n_12"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_43"])*(-log10(2) + log10(5)) - (x[,"n_24"] + x[,"n_32"])*(log10(2) + log10(5)) - (x[,"n_22"] + x[,"n_34"])*(log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_33"])*(2*log10(2) + log10(5)) + (x[,"n_14"] + x[,"n_42"])*(log10(3 - r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_12"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_rc)) + log10(2 + r_rc)) + (x[,"n_23"] + x[,"n_33"])*log10(19 + 4*r_rc - 4*r_rc^2) + (x[,"n_22"] + x[,"n_34"])*log10(12 - 5*r_rc + 2*r_rc^2) + (x[,"n_13"] + x[,"n_43"])*log10(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_24"] + x[,"n_32"])*log10(9 + r_rc + 2*r_rc^2)
logL_rc <- (-x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(2) + 2*(-x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_14"] + x[,"n_42"])*(log(3 - r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_12"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_rc)) + log(2 + r_rc)) + (x[,"n_23"] + x[,"n_33"])*log(19 + 4*r_rc - 4*r_rc^2) + (x[,"n_22"] + x[,"n_34"])*log(12 - 5*r_rc + 2*r_rc^2) + (x[,"n_13"] + x[,"n_43"])*log(3 - 2*r_rc + 2*r_rc^2) + (x[,"n_24"] + x[,"n_32"])*log(9 + r_rc + 2*r_rc^2)
logL_rm3c <- function(r,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
L <- (-n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*(3*log(2) + 2*log(5)) + (n14 + n42)*(log(3 - 2*r) + log(3 - r)) + (n12 + n44)*(log(2 + r) + log(1 + 2*r)) + (n13 + n43)*log(9 + 4*r - 4*r^2) + (n24 + n32)*log(24 - 7*r - 2*r^2) + (n22 + n34)*log(15 + 11*r - 2*r^2) + (n23 + n33)*log(41 - 4*r + 4*r^2)
return(L)}
interlogL_rm3c <- function(n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
optimize(logL_rm3c,c(0,0.5), n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm3c <- mapply(interlogL_rm3c,x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm3c <- -((x[,"n_14"] + x[,"n_42"])*log10(5)) - (x[,"n_12"] + x[,"n_44"])*log10(5) - (x[,"n_13"] + x[,"n_43"])*(log10(2) + log10(5)) - (x[,"n_24"] + x[,"n_32"])*(2*log10(2) + log10(5)) - (x[,"n_22"] + x[,"n_34"])*(2*log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_33"])*(3*log10(2) + log10(5)) + (x[,"n_14"] + x[,"n_42"])*(log10(3 - 2*r_rm3c) + log10(3 - r_rm3c)) + (x[,"n_12"] + x[,"n_44"])*(log10(2 + r_rm3c) + log10(1 + 2*r_rm3c)) + (x[,"n_13"] + x[,"n_43"])*log10(9 + 4*r_rm3c - 4*r_rm3c^2) + (x[,"n_24"] + x[,"n_32"])*log10(24 - 7*r_rm3c - 2*r_rm3c^2) + (x[,"n_22"] + x[,"n_34"])*log10(15 + 11*r_rm3c - 2*r_rm3c^2) + (x[,"n_23"] + x[,"n_33"])*log10(41 - 4*r_rm3c + 4*r_rm3c^2)
logL_rm3c <- (-x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(3*log(2) + 2*log(5)) + (x[,"n_14"] + x[,"n_42"])*(log(3 - 2*r_rm3c) + log(3 - r_rm3c)) + (x[,"n_12"] + x[,"n_44"])*(log(2 + r_rm3c) + log(1 + 2*r_rm3c)) + (x[,"n_13"] + x[,"n_43"])*log(9 + 4*r_rm3c - 4*r_rm3c^2) + (x[,"n_24"] + x[,"n_32"])*log(24 - 7*r_rm3c - 2*r_rm3c^2) + (x[,"n_22"] + x[,"n_34"])*log(15 + 11*r_rm3c - 2*r_rm3c^2) + (x[,"n_23"] + x[,"n_33"])*log(41 - 4*r_rm3c + 4*r_rm3c^2)

return(list(
r_mat = cbind(r_cc,r_cm3c,r_rc,r_rm3c,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm3c,LOD_rc,LOD_rm3c,0),
logL_mat = cbind(logL_cc,logL_cm3c,logL_rc,logL_rm3c,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed3c","repulsion coupling","repulsion mixed3c","unknown")
)
)
}

#' @rdname r6_functions
r6_1.4_2.2 <- function(x){
logL_cc <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - n30 - 2*n31 - 2*n32 - 2*n33 - n34 - n40 - 2*n41 - 2*n42 - 2*n43 - n44)*log(2) + (-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*(log(3) + 2*log(5)) + (n14 + n40)*(2*log(pmax(1e-6,r)) + log(2 + r)) + (n12 + n42)*log(9 + 16*r - 16*r^2) + (n13 + n41)*(log(pmax(1e-6,r)) + log(12 + 7*r - 4*r^2)) + (n24 + n30)*(log(pmax(1e-6,r)) + log(5 + r - 3*r^2)) + (n10 + n44)*(log(pmax(1e-6,1 - r)) + log(3 - 4*r + r^2)) + (n22 + n32)*log(57 - 16*r + 16*r^2) + (n21 + n33)*log(42 - 32*r + 17*r^2 - 12*r^3) + (n20 + n34)*log(3 + 2*r - 8*r^2 + 3*r^3) + (n11 + n43)*log(15 - 14*r - 5*r^2 + 4*r^3) + (n23 + n31)*log(15 + 34*r - 19*r^2 + 12*r^3)
return(L)}
interlogL_cc <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_cc,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cc <- -((x[,"n_14"] + x[,"n_40"])*(-3*log10(2) + log10(5))) - (x[,"n_10"] + x[,"n_44"])*(-3*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_42"])*log10(13) - (x[,"n_24"] + x[,"n_30"])*(-3*log10(2) + log10(19)) - (x[,"n_20"] + x[,"n_34"])*(-3*log10(2) + log10(19)) - (x[,"n_23"] + x[,"n_31"])*(-2*log10(2) + log10(5) + log10(23)) - (x[,"n_21"] + x[,"n_33"])*(-2*log10(2) + log10(5) + log10(23)) - (x[,"n_13"] + x[,"n_41"])*(-2*log10(2) + log10(29)) - (x[,"n_11"] + x[,"n_43"])*(-2*log10(2) + log10(29)) - (x[,"n_22"] + x[,"n_32"])*log10(53) + (x[,"n_14"] + x[,"n_40"])*(2*log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_12"] + x[,"n_42"])*log10(9 + 16*r_cc - 16*r_cc^2) + (x[,"n_13"] + x[,"n_41"])*(log10(pmax(1e-6,r_cc)) + log10(12 + 7*r_cc - 4*r_cc^2)) + (x[,"n_24"] + x[,"n_30"])*(log10(pmax(1e-6,r_cc)) + log10(5 + r_cc - 3*r_cc^2)) + (x[,"n_10"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cc)) + log10(3 - 4*r_cc + r_cc^2)) + (x[,"n_22"] + x[,"n_32"])*log10(57 - 16*r_cc + 16*r_cc^2) + (x[,"n_21"] + x[,"n_33"])*log10(42 - 32*r_cc + 17*r_cc^2 - 12*r_cc^3) + (x[,"n_20"] + x[,"n_34"])*log10(3 + 2*r_cc - 8*r_cc^2 + 3*r_cc^3) + (x[,"n_11"] + x[,"n_43"])*log10(15 - 14*r_cc - 5*r_cc^2 + 4*r_cc^3) + (x[,"n_23"] + x[,"n_31"])*log10(15 + 34*r_cc - 19*r_cc^2 + 12*r_cc^3)
logL_cc <- (-x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - x[,"n_44"])*log(2) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(log(3) + 2*log(5)) + (x[,"n_14"] + x[,"n_40"])*(2*log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_12"] + x[,"n_42"])*log(9 + 16*r_cc - 16*r_cc^2) + (x[,"n_13"] + x[,"n_41"])*(log(pmax(1e-6,r_cc)) + log(12 + 7*r_cc - 4*r_cc^2)) + (x[,"n_24"] + x[,"n_30"])*(log(pmax(1e-6,r_cc)) + log(5 + r_cc - 3*r_cc^2)) + (x[,"n_10"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cc)) + log(3 - 4*r_cc + r_cc^2)) + (x[,"n_22"] + x[,"n_32"])*log(57 - 16*r_cc + 16*r_cc^2) + (x[,"n_21"] + x[,"n_33"])*log(42 - 32*r_cc + 17*r_cc^2 - 12*r_cc^3) + (x[,"n_20"] + x[,"n_34"])*log(3 + 2*r_cc - 8*r_cc^2 + 3*r_cc^3) + (x[,"n_11"] + x[,"n_43"])*log(15 - 14*r_cc - 5*r_cc^2 + 4*r_cc^3) + (x[,"n_23"] + x[,"n_31"])*log(15 + 34*r_cc - 19*r_cc^2 + 12*r_cc^3)
logL_cm <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-2*n10 - 3*n11 - 3*n12 - 3*n13 - 2*n14 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 2*n24 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 2*n34 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 2*n44)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n10 + n44)*(log(pmax(1e-6,1 - r)) + log(2 - r) + log(pmax(1e-6,r))) + (n22 + n32)*log(33 + 8*r - 8*r^2) + (n14 + n40)*(log(pmax(1e-6,r)) + log(1 - r^2)) + (n12 + n42)*log(11 - 8*r + 8*r^2) + (n23 + n31)*log(18 - 4*r + 19*r^2 - 12*r^3) + (n11 + n43)*log(6 - 4*r + 5*r^2 - 4*r^3) + (n20 + n34)*log(3 - 5*r + 6*r^2 - 3*r^3) + (n24 + n30)*log(1 + 2*r - 3*r^2 + 3*r^3) + (n13 + n41)*log(3 + 6*r - 7*r^2 + 4*r^3) + (n21 + n33)*log(21 + 2*r - 17*r^2 + 12*r^3)
return(L)}
interlogL_cm <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_cm,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cm <- -2*(x[,"n_12"] + x[,"n_42"])*log10(3) - (x[,"n_14"] + x[,"n_40"])*(-3*log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_44"])*(-3*log10(2) + log10(3)) - (x[,"n_22"] + x[,"n_32"])*(log10(5) + log10(7)) - (x[,"n_23"] + x[,"n_31"])*(-2*log10(2) + log10(7) + log10(11)) - (x[,"n_21"] + x[,"n_33"])*(-2*log10(2) + log10(7) + log10(11)) - (x[,"n_24"] + x[,"n_30"])*(-3*log10(2) + log10(13)) - (x[,"n_20"] + x[,"n_34"])*(-3*log10(2) + log10(13)) - (x[,"n_13"] + x[,"n_41"])*(-2*log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_43"])*(-2*log10(2) + log10(19)) + (x[,"n_10"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cm)) + log10(2 - r_cm) + log10(pmax(1e-6,r_cm))) + (x[,"n_22"] + x[,"n_32"])*log10(33 + 8*r_cm - 8*r_cm^2) + (x[,"n_14"] + x[,"n_40"])*(log10(pmax(1e-6,r_cm)) + log10(1 - r_cm^2)) + (x[,"n_12"] + x[,"n_42"])*log10(11 - 8*r_cm + 8*r_cm^2) + (x[,"n_23"] + x[,"n_31"])*log10(18 - 4*r_cm + 19*r_cm^2 - 12*r_cm^3) + (x[,"n_11"] + x[,"n_43"])*log10(6 - 4*r_cm + 5*r_cm^2 - 4*r_cm^3) + (x[,"n_20"] + x[,"n_34"])*log10(3 - 5*r_cm + 6*r_cm^2 - 3*r_cm^3) + (x[,"n_24"] + x[,"n_30"])*log10(1 + 2*r_cm - 3*r_cm^2 + 3*r_cm^3) + (x[,"n_13"] + x[,"n_41"])*log10(3 + 6*r_cm - 7*r_cm^2 + 4*r_cm^3) + (x[,"n_21"] + x[,"n_33"])*log10(21 + 2*r_cm - 17*r_cm^2 + 12*r_cm^3)
logL_cm <- (-2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_10"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cm)) + log(2 - r_cm) + log(pmax(1e-6,r_cm))) + (x[,"n_22"] + x[,"n_32"])*log(33 + 8*r_cm - 8*r_cm^2) + (x[,"n_14"] + x[,"n_40"])*(log(pmax(1e-6,r_cm)) + log(1 - r_cm^2)) + (x[,"n_12"] + x[,"n_42"])*log(11 - 8*r_cm + 8*r_cm^2) + (x[,"n_23"] + x[,"n_31"])*log(18 - 4*r_cm + 19*r_cm^2 - 12*r_cm^3) + (x[,"n_11"] + x[,"n_43"])*log(6 - 4*r_cm + 5*r_cm^2 - 4*r_cm^3) + (x[,"n_20"] + x[,"n_34"])*log(3 - 5*r_cm + 6*r_cm^2 - 3*r_cm^3) + (x[,"n_24"] + x[,"n_30"])*log(1 + 2*r_cm - 3*r_cm^2 + 3*r_cm^3) + (x[,"n_13"] + x[,"n_41"])*log(3 + 6*r_cm - 7*r_cm^2 + 4*r_cm^3) + (x[,"n_21"] + x[,"n_33"])*log(21 + 2*r_cm - 17*r_cm^2 + 12*r_cm^3)
logL_cr <- function(r,n11,n13,n21,n22,n23,n31,n32,n33,n41,n43,n10,n12,n14,n20,n24,n30,n34,n40,n42,n44) {
L <- (-n11 - n13 - n21 - n22 - n23 - n31 - n32 - n33 - n41 - n43)*log(2) + 2*(n22 + n32)*log(3) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n14 + n40)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n44)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n20 + n34)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n11 + n43)*(log(pmax(1e-6,r)) + log(4 - 5*r + 4*r^2)) + (n21 + n33)*log(6 - 8*r + 17*r^2 - 12*r^3) + (n13 + n41)*log(3 - 6*r + 7*r^2 - 4*r^3) + (n24 + n30)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n23 + n31)*log(3 + 10*r - 19*r^2 + 12*r^3)
return(L)}
interlogL_cr <- function(n11,n13,n21,n22,n23,n31,n32,n33,n41,n43,n10,n12,n14,n20,n24,n30,n34,n40,n42,n44) {
optimize(logL_cr,c(0,0.5), n11,n13,n21,n22,n23,n31,n32,n33,n41,n43,n10,n12,n14,n20,n24,n30,n34,n40,n42,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_11"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_41"],x[,"n_43"],x[,"n_10"],x[,"n_12"],x[,"n_14"],x[,"n_20"],x[,"n_24"],x[,"n_30"],x[,"n_34"],x[,"n_40"],x[,"n_42"],x[,"n_44"])
LOD_cr <- 3*(x[,"n_14"] + x[,"n_40"])*log10(2) + 3*(x[,"n_10"] + x[,"n_44"])*log10(2) + (x[,"n_24"] + x[,"n_30"])*(3*log10(2) - log10(3)) - (x[,"n_20"] + x[,"n_34"])*(-3*log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_31"])*(-2*log10(2) + log10(19)) - (x[,"n_21"] + x[,"n_33"])*(-2*log10(2) + log10(19)) + (x[,"n_14"] + x[,"n_40"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_10"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cr)) + 2*log10(pmax(1e-6,r_cr))) + (x[,"n_20"] + x[,"n_34"])*(log10(pmax(1e-6,r_cr)) + log10(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_11"] + x[,"n_43"])*(log10(pmax(1e-6,r_cr)) + log10(4 - 5*r_cr + 4*r_cr^2)) + (x[,"n_21"] + x[,"n_33"])*log10(6 - 8*r_cr + 17*r_cr^2 - 12*r_cr^3) + (x[,"n_13"] + x[,"n_41"])*log10(3 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3) + (x[,"n_24"] + x[,"n_30"])*log10(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_23"] + x[,"n_31"])*log10(3 + 10*r_cr - 19*r_cr^2 + 12*r_cr^3)
logL_cr <- (-x[,"n_11"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_41"] - x[,"n_43"])*log(2) + 2*(x[,"n_22"] + x[,"n_32"])*log(3) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_14"] + x[,"n_40"])*(2*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_10"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cr)) + 2*log(pmax(1e-6,r_cr))) + (x[,"n_20"] + x[,"n_34"])*(log(pmax(1e-6,r_cr)) + log(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_11"] + x[,"n_43"])*(log(pmax(1e-6,r_cr)) + log(4 - 5*r_cr + 4*r_cr^2)) + (x[,"n_21"] + x[,"n_33"])*log(6 - 8*r_cr + 17*r_cr^2 - 12*r_cr^3) + (x[,"n_13"] + x[,"n_41"])*log(3 - 6*r_cr + 7*r_cr^2 - 4*r_cr^3) + (x[,"n_24"] + x[,"n_30"])*log(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_23"] + x[,"n_31"])*log(3 + 10*r_cr - 19*r_cr^2 + 12*r_cr^3)
logL_rc <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*(3*log(2) + log(3) + 2*log(5)) + (n14 + n40)*(log(3 - 2*r) + log(pmax(1e-6,r)) + log(2 + r)) + (n22 + n32)*log(105 + 4*r - 4*r^2) + (n10 + n44)*(log(1 + 2*r) + log(3 - 4*r + r^2)) + (n12 + n42)*log(27 - 4*r + 4*r^2) + (n23 + n31)*log(57 - 4*r + 16*r^2 - 12*r^3) + (n20 + n34)*log(12 - 10*r + 13*r^2 - 6*r^3) + (n11 + n43)*log(21 - 16*r + 8*r^2 - 4*r^3) + (n13 + n41)*log(9 + 12*r - 4*r^2 + 4*r^3) + (n24 + n30)*log(9 + 2*r - 5*r^2 + 6*r^3) + (n21 + n33)*log(57 + 8*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rc <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rc,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rc <- -((x[,"n_14"] + x[,"n_40"])*(-log10(2) + log10(5))) - (x[,"n_10"] + x[,"n_44"])*(-log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_42"])*(log10(2) + log10(13)) - (x[,"n_24"] + x[,"n_30"])*(-log10(2) + log10(19)) - (x[,"n_20"] + x[,"n_34"])*(-log10(2) + log10(19)) - (x[,"n_23"] + x[,"n_31"])*(-log10(2) + log10(5) + log10(23)) - (x[,"n_21"] + x[,"n_33"])*(-log10(2) + log10(5) + log10(23)) - (x[,"n_13"] + x[,"n_41"])*(-log10(2) + log10(29)) - (x[,"n_11"] + x[,"n_43"])*(-log10(2) + log10(29)) - (x[,"n_22"] + x[,"n_32"])*(log10(2) + log10(53)) + (x[,"n_14"] + x[,"n_40"])*(log10(3 - 2*r_rc) + log10(pmax(1e-6,r_rc)) + log10(2 + r_rc)) + (x[,"n_22"] + x[,"n_32"])*log10(105 + 4*r_rc - 4*r_rc^2) + (x[,"n_10"] + x[,"n_44"])*(log10(1 + 2*r_rc) + log10(3 - 4*r_rc + r_rc^2)) + (x[,"n_12"] + x[,"n_42"])*log10(27 - 4*r_rc + 4*r_rc^2) + (x[,"n_23"] + x[,"n_31"])*log10(57 - 4*r_rc + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_20"] + x[,"n_34"])*log10(12 - 10*r_rc + 13*r_rc^2 - 6*r_rc^3) + (x[,"n_11"] + x[,"n_43"])*log10(21 - 16*r_rc + 8*r_rc^2 - 4*r_rc^3) + (x[,"n_13"] + x[,"n_41"])*log10(9 + 12*r_rc - 4*r_rc^2 + 4*r_rc^3) + (x[,"n_24"] + x[,"n_30"])*log10(9 + 2*r_rc - 5*r_rc^2 + 6*r_rc^3) + (x[,"n_21"] + x[,"n_33"])*log10(57 + 8*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rc <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(3*log(2) + log(3) + 2*log(5)) + (x[,"n_14"] + x[,"n_40"])*(log(3 - 2*r_rc) + log(pmax(1e-6,r_rc)) + log(2 + r_rc)) + (x[,"n_22"] + x[,"n_32"])*log(105 + 4*r_rc - 4*r_rc^2) + (x[,"n_10"] + x[,"n_44"])*(log(1 + 2*r_rc) + log(3 - 4*r_rc + r_rc^2)) + (x[,"n_12"] + x[,"n_42"])*log(27 - 4*r_rc + 4*r_rc^2) + (x[,"n_23"] + x[,"n_31"])*log(57 - 4*r_rc + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_20"] + x[,"n_34"])*log(12 - 10*r_rc + 13*r_rc^2 - 6*r_rc^3) + (x[,"n_11"] + x[,"n_43"])*log(21 - 16*r_rc + 8*r_rc^2 - 4*r_rc^3) + (x[,"n_13"] + x[,"n_41"])*log(9 + 12*r_rc - 4*r_rc^2 + 4*r_rc^3) + (x[,"n_24"] + x[,"n_30"])*log(9 + 2*r_rc - 5*r_rc^2 + 6*r_rc^3) + (x[,"n_21"] + x[,"n_33"])*log(57 + 8*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rm <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*(4*log(2) + 2*log(5)) + (n10 + n44)*(log(2 - r) + log(pmax(1e-6,r)) + log(1 + 2*r)) + (n22 + n32)*log(69 + 4*r - 4*r^2) + (n14 + n40)*(log(3 - 2*r) + log(1 - r^2)) + (n12 + n42)*log(19 - 4*r + 4*r^2) + (n21 + n33)*log(33 + 4*r + 20*r^2 - 12*r^3) + (n24 + n30)*log(10 - 10*r + 9*r^2 - 6*r^3) + (n13 + n41)*log(15 - 12*r + 4*r^2 - 4*r^3) + (n11 + n43)*log(3 + 16*r - 8*r^2 + 4*r^3) + (n20 + n34)*log(3 + 10*r - 9*r^2 + 6*r^3) + (n23 + n31)*log(45 - 8*r - 16*r^2 + 12*r^3)
return(L)}
interlogL_rm <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rm,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm <- -((x[,"n_14"] + x[,"n_40"])*(-log10(2) + log10(3))) - (x[,"n_10"] + x[,"n_44"])*(-log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_42"])*(log10(2) + 2*log10(3)) - (x[,"n_22"] + x[,"n_32"])*(log10(2) + log10(5) + log10(7)) - (x[,"n_23"] + x[,"n_31"])*(-log10(2) + log10(7) + log10(11)) - (x[,"n_21"] + x[,"n_33"])*(-log10(2) + log10(7) + log10(11)) - (x[,"n_24"] + x[,"n_30"])*(-log10(2) + log10(13)) - (x[,"n_20"] + x[,"n_34"])*(-log10(2) + log10(13)) - (x[,"n_13"] + x[,"n_41"])*(-log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_43"])*(-log10(2) + log10(19)) + (x[,"n_10"] + x[,"n_44"])*(log10(2 - r_rm) + log10(pmax(1e-6,r_rm)) + log10(1 + 2*r_rm)) + (x[,"n_22"] + x[,"n_32"])*log10(69 + 4*r_rm - 4*r_rm^2) + (x[,"n_14"] + x[,"n_40"])*(log10(3 - 2*r_rm) + log10(1 - r_rm^2)) + (x[,"n_12"] + x[,"n_42"])*log10(19 - 4*r_rm + 4*r_rm^2) + (x[,"n_21"] + x[,"n_33"])*log10(33 + 4*r_rm + 20*r_rm^2 - 12*r_rm^3) + (x[,"n_24"] + x[,"n_30"])*log10(10 - 10*r_rm + 9*r_rm^2 - 6*r_rm^3) + (x[,"n_13"] + x[,"n_41"])*log10(15 - 12*r_rm + 4*r_rm^2 - 4*r_rm^3) + (x[,"n_11"] + x[,"n_43"])*log10(3 + 16*r_rm - 8*r_rm^2 + 4*r_rm^3) + (x[,"n_20"] + x[,"n_34"])*log10(3 + 10*r_rm - 9*r_rm^2 + 6*r_rm^3) + (x[,"n_23"] + x[,"n_31"])*log10(45 - 8*r_rm - 16*r_rm^2 + 12*r_rm^3)
logL_rm <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(4*log(2) + 2*log(5)) + (x[,"n_10"] + x[,"n_44"])*(log(2 - r_rm) + log(pmax(1e-6,r_rm)) + log(1 + 2*r_rm)) + (x[,"n_22"] + x[,"n_32"])*log(69 + 4*r_rm - 4*r_rm^2) + (x[,"n_14"] + x[,"n_40"])*(log(3 - 2*r_rm) + log(1 - r_rm^2)) + (x[,"n_12"] + x[,"n_42"])*log(19 - 4*r_rm + 4*r_rm^2) + (x[,"n_21"] + x[,"n_33"])*log(33 + 4*r_rm + 20*r_rm^2 - 12*r_rm^3) + (x[,"n_24"] + x[,"n_30"])*log(10 - 10*r_rm + 9*r_rm^2 - 6*r_rm^3) + (x[,"n_13"] + x[,"n_41"])*log(15 - 12*r_rm + 4*r_rm^2 - 4*r_rm^3) + (x[,"n_11"] + x[,"n_43"])*log(3 + 16*r_rm - 8*r_rm^2 + 4*r_rm^3) + (x[,"n_20"] + x[,"n_34"])*log(3 + 10*r_rm - 9*r_rm^2 + 6*r_rm^3) + (x[,"n_23"] + x[,"n_31"])*log(45 - 8*r_rm - 16*r_rm^2 + 12*r_rm^3)
logL_rr <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
L <- (-2*n10 - n11 - 2*n12 - n13 - 2*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n40 - n41 - 2*n42 - n43 - 2*n44)*log(2) + (n22 + n32)*log(3) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n14 + n40)*(log(3 - 2*r) + 2*log(pmax(1e-6,1 - r))) + (n10 + n44)*(2*log(pmax(1e-6,r)) + log(1 + 2*r)) + (n12 + n42)*log(1 + 12*r - 12*r^2) + (n20 + n34)*(log(pmax(1e-6,r)) + log(2 + 5*r - 6*r^2)) + (n22 + n32)*log(7 - 4*r + 4*r^2) + (n23 + n31)*log(15 - 16*r + 16*r^2 - 12*r^3) + (n11 + n43)*log(r + 4*r^2 - 2*r^3) + (n13 + n41)*log(3 - 3*r - 2*r^2 + 2*r^3) + (n24 + n30)*log(1 + 6*r - 13*r^2 + 6*r^3) + (n21 + n33)*log(3 + 20*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rr <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44) {
optimize(logL_rr,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rr <- (x[,"n_14"] + x[,"n_40"])*log10(2) - 2*(x[,"n_12"] + x[,"n_42"])*log10(2) + (x[,"n_10"] + x[,"n_44"])*log10(2) - (x[,"n_24"] + x[,"n_30"])*(-log10(2) + log10(3)) - (x[,"n_20"] + x[,"n_34"])*(-log10(2) + log10(3)) - (x[,"n_22"] + x[,"n_32"])*(log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_41"])*(-2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_43"])*(-2*log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_31"])*(-log10(2) + log10(19)) - (x[,"n_21"] + x[,"n_33"])*(-log10(2) + log10(19)) + (x[,"n_14"] + x[,"n_40"])*(log10(3 - 2*r_rr) + 2*log10(pmax(1e-6,1 - r_rr))) + (x[,"n_10"] + x[,"n_44"])*(2*log10(pmax(1e-6,r_rr)) + log10(1 + 2*r_rr)) + (x[,"n_12"] + x[,"n_42"])*log10(1 + 12*r_rr - 12*r_rr^2) + (x[,"n_20"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(2 + 5*r_rr - 6*r_rr^2)) + (x[,"n_22"] + x[,"n_32"])*log10(7 - 4*r_rr + 4*r_rr^2) + (x[,"n_23"] + x[,"n_31"])*log10(15 - 16*r_rr + 16*r_rr^2 - 12*r_rr^3) + (x[,"n_11"] + x[,"n_43"])*log10(r_rr + 4*r_rr^2 - 2*r_rr^3) + (x[,"n_13"] + x[,"n_41"])*log10(3 - 3*r_rr - 2*r_rr^2 + 2*r_rr^3) + (x[,"n_24"] + x[,"n_30"])*log10(1 + 6*r_rr - 13*r_rr^2 + 6*r_rr^3) + (x[,"n_21"] + x[,"n_33"])*log10(3 + 20*r_rr - 20*r_rr^2 + 12*r_rr^3)
logL_rr <- (-2*x[,"n_10"] - x[,"n_11"] - 2*x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_40"] - x[,"n_41"] - 2*x[,"n_42"] - x[,"n_43"] - 2*x[,"n_44"])*log(2) + (x[,"n_22"] + x[,"n_32"])*log(3) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_14"] + x[,"n_40"])*(log(3 - 2*r_rr) + 2*log(pmax(1e-6,1 - r_rr))) + (x[,"n_10"] + x[,"n_44"])*(2*log(pmax(1e-6,r_rr)) + log(1 + 2*r_rr)) + (x[,"n_12"] + x[,"n_42"])*log(1 + 12*r_rr - 12*r_rr^2) + (x[,"n_20"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(2 + 5*r_rr - 6*r_rr^2)) + (x[,"n_22"] + x[,"n_32"])*log(7 - 4*r_rr + 4*r_rr^2) + (x[,"n_23"] + x[,"n_31"])*log(15 - 16*r_rr + 16*r_rr^2 - 12*r_rr^3) + (x[,"n_11"] + x[,"n_43"])*log(r_rr + 4*r_rr^2 - 2*r_rr^3) + (x[,"n_13"] + x[,"n_41"])*log(3 - 3*r_rr - 2*r_rr^2 + 2*r_rr^3) + (x[,"n_24"] + x[,"n_30"])*log(1 + 6*r_rr - 13*r_rr^2 + 6*r_rr^3) + (x[,"n_21"] + x[,"n_33"])*log(3 + 20*r_rr - 20*r_rr^2 + 12*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.4_2.3 <- function(x){
logL_cc <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + (n20 + n25 + n30 + n35)*log(3) + 2*(-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + 3*(n10 + n45)*log(pmax(1e-6,1 - r)) + 3*(n15 + n40)*log(pmax(1e-6,r)) + (n20 + n35)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n41)*(log(11 - 2*r) + 2*log(pmax(1e-6,r))) + (n25 + n30)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n11 + n44)*(2*log(pmax(1e-6,1 - r)) + log(9 + 2*r)) + (n13 + n42)*(log(pmax(1e-6,r)) + log(18 - 5*r - 4*r^2)) + (n24 + n31)*(log(pmax(1e-6,r)) + log(26 - 17*r + 6*r^2)) + (n22 + n33)*log(41 - 34*r + 29*r^2 - 12*r^3) + (n21 + n34)*log(15 - 10*r + r^2 - 6*r^3) + (n12 + n43)*log(9 + 4*r - 17*r^2 + 4*r^3) + (n23 + n32)*log(24 + 12*r - 7*r^2 + 12*r^3)
return(L)}
interlogL_cc <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cc,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cc <- 3*(x[,"n_25"] + x[,"n_30"])*log10(2) + 3*(x[,"n_20"] + x[,"n_35"])*log10(2) + 3*(x[,"n_15"] + x[,"n_40"])*log10(2) + 3*(x[,"n_10"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_41"])*(-log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_44"])*(-log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_22"] + x[,"n_33"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_24"] + x[,"n_31"])*(-log10(2) + log10(19)) - (x[,"n_21"] + x[,"n_34"])*(-log10(2) + log10(19)) - (x[,"n_13"] + x[,"n_42"])*(-2*log10(2) + log10(29)) - (x[,"n_12"] + x[,"n_43"])*(-2*log10(2) + log10(29)) + 3*(x[,"n_10"] + x[,"n_45"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_15"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + (x[,"n_20"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_41"])*(log10(11 - 2*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_25"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(9 + 2*r_cc)) + (x[,"n_13"] + x[,"n_42"])*(log10(pmax(1e-6,r_cc)) + log10(18 - 5*r_cc - 4*r_cc^2)) + (x[,"n_24"] + x[,"n_31"])*(log10(pmax(1e-6,r_cc)) + log10(26 - 17*r_cc + 6*r_cc^2)) + (x[,"n_22"] + x[,"n_33"])*log10(41 - 34*r_cc + 29*r_cc^2 - 12*r_cc^3) + (x[,"n_21"] + x[,"n_34"])*log10(15 - 10*r_cc + r_cc^2 - 6*r_cc^3) + (x[,"n_12"] + x[,"n_43"])*log10(9 + 4*r_cc - 17*r_cc^2 + 4*r_cc^3) + (x[,"n_23"] + x[,"n_32"])*log10(24 + 12*r_cc - 7*r_cc^2 + 12*r_cc^3)
logL_cc <- (-2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_20"] + x[,"n_25"] + x[,"n_30"] + x[,"n_35"])*log(3) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + 3*(x[,"n_10"] + x[,"n_45"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_15"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + (x[,"n_20"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_41"])*(log(11 - 2*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_25"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_cc)) + log(9 + 2*r_cc)) + (x[,"n_13"] + x[,"n_42"])*(log(pmax(1e-6,r_cc)) + log(18 - 5*r_cc - 4*r_cc^2)) + (x[,"n_24"] + x[,"n_31"])*(log(pmax(1e-6,r_cc)) + log(26 - 17*r_cc + 6*r_cc^2)) + (x[,"n_22"] + x[,"n_33"])*log(41 - 34*r_cc + 29*r_cc^2 - 12*r_cc^3) + (x[,"n_21"] + x[,"n_34"])*log(15 - 10*r_cc + r_cc^2 - 6*r_cc^3) + (x[,"n_12"] + x[,"n_43"])*log(9 + 4*r_cc - 17*r_cc^2 + 4*r_cc^3) + (x[,"n_23"] + x[,"n_32"])*log(24 + 12*r_cc - 7*r_cc^2 + 12*r_cc^3)
logL_cm2c <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35 - n40 - 2*n41 - 2*n42 - 2*n43 - 2*n44 - n45)*log(2) + (-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(3) + 2*(-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n10 + n45)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n15 + n40)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n14 + n41)*(log(pmax(1e-6,r)) + log(9 - 5*r + 2*r^2)) + (n25 + n30)*(log(pmax(1e-6,r)) + log(5 - 8*r + 6*r^2)) + (n24 + n31)*log(21 + 8*r + 22*r^2 - 12*r^3) + (n23 + n32)*log(27 + 2*r + 10*r^2 - 8*r^3) + (n20 + n35)*log(3 - 7*r + 10*r^2 - 6*r^3) + (n12 + n43)*log(15 - 11*r + 9*r^2 - 4*r^3) + (n11 + n44)*log(6 - 5*r + r^2 - 2*r^3) + (n13 + n42)*log(9 + 5*r - 3*r^2 + 4*r^3) + (n22 + n33)*log(31 + 2*r - 14*r^2 + 8*r^3) + (n21 + n34)*log(39 - 16*r - 14*r^2 + 12*r^3)
return(L)}
interlogL_cm2c <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cm2c,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cm2c <- 3*(x[,"n_15"] + x[,"n_40"])*log10(2) + 3*(x[,"n_10"] + x[,"n_45"])*log10(2) - (x[,"n_25"] + x[,"n_30"])*(-2*log10(2) + log10(5)) - (x[,"n_20"] + x[,"n_35"])*(-2*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_42"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_12"] + x[,"n_43"])*(-2*log10(2) + 2*log10(3) + log10(5)) - (x[,"n_14"] + x[,"n_41"])*(-log10(2) + log10(7)) - (x[,"n_11"] + x[,"n_44"])*(-log10(2) + log10(7)) - (x[,"n_24"] + x[,"n_31"])*log10(29) - (x[,"n_21"] + x[,"n_34"])*log10(29) - (x[,"n_23"] + x[,"n_32"])*(-log10(2) + log10(59)) - (x[,"n_22"] + x[,"n_33"])*(-log10(2) + log10(59)) + (x[,"n_10"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c))) + (x[,"n_15"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_cm2c)) + 2*log10(pmax(1e-6,r_cm2c))) + (x[,"n_14"] + x[,"n_41"])*(log10(pmax(1e-6,r_cm2c)) + log10(9 - 5*r_cm2c + 2*r_cm2c^2)) + (x[,"n_25"] + x[,"n_30"])*(log10(pmax(1e-6,r_cm2c)) + log10(5 - 8*r_cm2c + 6*r_cm2c^2)) + (x[,"n_24"] + x[,"n_31"])*log10(21 + 8*r_cm2c + 22*r_cm2c^2 - 12*r_cm2c^3) + (x[,"n_23"] + x[,"n_32"])*log10(27 + 2*r_cm2c + 10*r_cm2c^2 - 8*r_cm2c^3) + (x[,"n_20"] + x[,"n_35"])*log10(3 - 7*r_cm2c + 10*r_cm2c^2 - 6*r_cm2c^3) + (x[,"n_12"] + x[,"n_43"])*log10(15 - 11*r_cm2c + 9*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_11"] + x[,"n_44"])*log10(6 - 5*r_cm2c + r_cm2c^2 - 2*r_cm2c^3) + (x[,"n_13"] + x[,"n_42"])*log10(9 + 5*r_cm2c - 3*r_cm2c^2 + 4*r_cm2c^3) + (x[,"n_22"] + x[,"n_33"])*log10(31 + 2*r_cm2c - 14*r_cm2c^2 + 8*r_cm2c^3) + (x[,"n_21"] + x[,"n_34"])*log10(39 - 16*r_cm2c - 14*r_cm2c^2 + 12*r_cm2c^3)
logL_cm2c <- (-x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"] - x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - x[,"n_45"])*log(2) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(3) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_10"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c))) + (x[,"n_15"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_cm2c)) + 2*log(pmax(1e-6,r_cm2c))) + (x[,"n_14"] + x[,"n_41"])*(log(pmax(1e-6,r_cm2c)) + log(9 - 5*r_cm2c + 2*r_cm2c^2)) + (x[,"n_25"] + x[,"n_30"])*(log(pmax(1e-6,r_cm2c)) + log(5 - 8*r_cm2c + 6*r_cm2c^2)) + (x[,"n_24"] + x[,"n_31"])*log(21 + 8*r_cm2c + 22*r_cm2c^2 - 12*r_cm2c^3) + (x[,"n_23"] + x[,"n_32"])*log(27 + 2*r_cm2c + 10*r_cm2c^2 - 8*r_cm2c^3) + (x[,"n_20"] + x[,"n_35"])*log(3 - 7*r_cm2c + 10*r_cm2c^2 - 6*r_cm2c^3) + (x[,"n_12"] + x[,"n_43"])*log(15 - 11*r_cm2c + 9*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_11"] + x[,"n_44"])*log(6 - 5*r_cm2c + r_cm2c^2 - 2*r_cm2c^3) + (x[,"n_13"] + x[,"n_42"])*log(9 + 5*r_cm2c - 3*r_cm2c^2 + 4*r_cm2c^3) + (x[,"n_22"] + x[,"n_33"])*log(31 + 2*r_cm2c - 14*r_cm2c^2 + 8*r_cm2c^3) + (x[,"n_21"] + x[,"n_34"])*log(39 - 16*r_cm2c - 14*r_cm2c^2 + 12*r_cm2c^3)
logL_cm1c <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n15 + n40)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n45)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n11 + n44)*(log(pmax(1e-6,r)) + log(8 - 7*r + 2*r^2)) + (n20 + n35)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n22 + n33)*log(30 - 4*r + 13*r^2 - 12*r^3) + (n21 + n34)*log(12 - 10*r + 13*r^2 - 6*r^3) + (n13 + n42)*log(11 - 12*r + 11*r^2 - 4*r^3) + (n25 + n30)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n14 + n41)*log(3 - r^2 - 2*r^3) + (n12 + n43)*log(6 + 2*r - r^2 + 4*r^3) + (n24 + n31)*log(9 + 2*r - 5*r^2 + 6*r^3) + (n23 + n32)*log(27 + 14*r - 23*r^2 + 12*r^3)
return(L)}
interlogL_cm1c <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cm1c,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm1c <- mapply(interlogL_cm1c,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cm1c <- 3*(x[,"n_15"] + x[,"n_40"])*log10(2) + 3*(x[,"n_10"] + x[,"n_45"])*log10(2) + (x[,"n_25"] + x[,"n_30"])*(3*log10(2) - log10(3)) - (x[,"n_20"] + x[,"n_35"])*(-3*log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_41"])*(-log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_44"])*(-log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_22"] + x[,"n_33"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_24"] + x[,"n_31"])*(-log10(2) + log10(19)) - (x[,"n_21"] + x[,"n_34"])*(-log10(2) + log10(19)) - (x[,"n_13"] + x[,"n_42"])*(-2*log10(2) + log10(29)) - (x[,"n_12"] + x[,"n_43"])*(-2*log10(2) + log10(29)) + (x[,"n_15"] + x[,"n_40"])*(2*log10(pmax(1e-6,1 - r_cm1c)) + log10(pmax(1e-6,r_cm1c))) + (x[,"n_10"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cm1c)) + 2*log10(pmax(1e-6,r_cm1c))) + (x[,"n_11"] + x[,"n_44"])*(log10(pmax(1e-6,r_cm1c)) + log10(8 - 7*r_cm1c + 2*r_cm1c^2)) + (x[,"n_20"] + x[,"n_35"])*(log10(pmax(1e-6,r_cm1c)) + log10(2 - 4*r_cm1c + 3*r_cm1c^2)) + (x[,"n_22"] + x[,"n_33"])*log10(30 - 4*r_cm1c + 13*r_cm1c^2 - 12*r_cm1c^3) + (x[,"n_21"] + x[,"n_34"])*log10(12 - 10*r_cm1c + 13*r_cm1c^2 - 6*r_cm1c^3) + (x[,"n_13"] + x[,"n_42"])*log10(11 - 12*r_cm1c + 11*r_cm1c^2 - 4*r_cm1c^3) + (x[,"n_25"] + x[,"n_30"])*log10(1 - 3*r_cm1c + 5*r_cm1c^2 - 3*r_cm1c^3) + (x[,"n_14"] + x[,"n_41"])*log10(3 - r_cm1c^2 - 2*r_cm1c^3) + (x[,"n_12"] + x[,"n_43"])*log10(6 + 2*r_cm1c - r_cm1c^2 + 4*r_cm1c^3) + (x[,"n_24"] + x[,"n_31"])*log10(9 + 2*r_cm1c - 5*r_cm1c^2 + 6*r_cm1c^3) + (x[,"n_23"] + x[,"n_32"])*log10(27 + 14*r_cm1c - 23*r_cm1c^2 + 12*r_cm1c^3)
logL_cm1c <- (-2*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_15"] + x[,"n_40"])*(2*log(pmax(1e-6,1 - r_cm1c)) + log(pmax(1e-6,r_cm1c))) + (x[,"n_10"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cm1c)) + 2*log(pmax(1e-6,r_cm1c))) + (x[,"n_11"] + x[,"n_44"])*(log(pmax(1e-6,r_cm1c)) + log(8 - 7*r_cm1c + 2*r_cm1c^2)) + (x[,"n_20"] + x[,"n_35"])*(log(pmax(1e-6,r_cm1c)) + log(2 - 4*r_cm1c + 3*r_cm1c^2)) + (x[,"n_22"] + x[,"n_33"])*log(30 - 4*r_cm1c + 13*r_cm1c^2 - 12*r_cm1c^3) + (x[,"n_21"] + x[,"n_34"])*log(12 - 10*r_cm1c + 13*r_cm1c^2 - 6*r_cm1c^3) + (x[,"n_13"] + x[,"n_42"])*log(11 - 12*r_cm1c + 11*r_cm1c^2 - 4*r_cm1c^3) + (x[,"n_25"] + x[,"n_30"])*log(1 - 3*r_cm1c + 5*r_cm1c^2 - 3*r_cm1c^3) + (x[,"n_14"] + x[,"n_41"])*log(3 - r_cm1c^2 - 2*r_cm1c^3) + (x[,"n_12"] + x[,"n_43"])*log(6 + 2*r_cm1c - r_cm1c^2 + 4*r_cm1c^3) + (x[,"n_24"] + x[,"n_31"])*log(9 + 2*r_cm1c - 5*r_cm1c^2 + 6*r_cm1c^3) + (x[,"n_23"] + x[,"n_32"])*log(27 + 14*r_cm1c - 23*r_cm1c^2 + 12*r_cm1c^3)
logL_rc <- function(r,n20,n25,n30,n35,n10,n11,n12,n13,n14,n15,n21,n22,n23,n24,n31,n32,n33,n34,n40,n41,n42,n43,n44,n45) {
L <- (n20 + n25 + n30 + n35)*log(3) + (-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n15 + n40)*(log(3 - 2*r) + 2*log(pmax(1e-6,r))) + (n10 + n45)*(2*log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n14 + n41)*(log(pmax(1e-6,r)) + log(12 - 5*r + 2*r^2)) + (n25 + n30)*(log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n23 + n32)*log(57 + 16*r^2 - 12*r^3) + (n24 + n31)*log(18 - 2*r + 11*r^2 - 6*r^3) + (n12 + n43)*log(21 - 16*r + 8*r^2 - 4*r^3) + (n11 + n44)*log(9 - 8*r + r^2 - 2*r^3) + (n20 + n35)*log(1 - 2*r + 3*r^2 - 2*r^3) + (n13 + n42)*log(9 + 12*r - 4*r^2 + 4*r^3) + (n21 + n34)*log(21 - 2*r - 7*r^2 + 6*r^3) + (n22 + n33)*log(61 + 4*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rc <- function(n20,n25,n30,n35,n10,n11,n12,n13,n14,n15,n21,n22,n23,n24,n31,n32,n33,n34,n40,n41,n42,n43,n44,n45) {
optimize(logL_rc,c(0,0.5), n20,n25,n30,n35,n10,n11,n12,n13,n14,n15,n21,n22,n23,n24,n31,n32,n33,n34,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_20"],x[,"n_25"],x[,"n_30"],x[,"n_35"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rc <- (x[,"n_25"] + x[,"n_30"])*log10(2) + (x[,"n_20"] + x[,"n_35"])*log10(2) + (x[,"n_15"] + x[,"n_40"])*log10(2) + (x[,"n_10"] + x[,"n_45"])*log10(2) - (x[,"n_14"] + x[,"n_41"])*log10(5) - (x[,"n_11"] + x[,"n_44"])*log10(5) - (x[,"n_23"] + x[,"n_32"])*(-log10(2) + log10(7) + log10(17)) - (x[,"n_22"] + x[,"n_33"])*(-log10(2) + log10(7) + log10(17)) - (x[,"n_24"] + x[,"n_31"])*log10(19) - (x[,"n_21"] + x[,"n_34"])*log10(19) - (x[,"n_13"] + x[,"n_42"])*(-log10(2) + log10(29)) - (x[,"n_12"] + x[,"n_43"])*(-log10(2) + log10(29)) + (x[,"n_15"] + x[,"n_40"])*(log10(3 - 2*r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_10"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_14"] + x[,"n_41"])*(log10(pmax(1e-6,r_rc)) + log10(12 - 5*r_rc + 2*r_rc^2)) + (x[,"n_25"] + x[,"n_30"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 3*r_rc + 2*r_rc^2)) + (x[,"n_23"] + x[,"n_32"])*log10(57 + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_24"] + x[,"n_31"])*log10(18 - 2*r_rc + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_12"] + x[,"n_43"])*log10(21 - 16*r_rc + 8*r_rc^2 - 4*r_rc^3) + (x[,"n_11"] + x[,"n_44"])*log10(9 - 8*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_20"] + x[,"n_35"])*log10(1 - 2*r_rc + 3*r_rc^2 - 2*r_rc^3) + (x[,"n_13"] + x[,"n_42"])*log10(9 + 12*r_rc - 4*r_rc^2 + 4*r_rc^3) + (x[,"n_21"] + x[,"n_34"])*log10(21 - 2*r_rc - 7*r_rc^2 + 6*r_rc^3) + (x[,"n_22"] + x[,"n_33"])*log10(61 + 4*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rc <- (x[,"n_20"] + x[,"n_25"] + x[,"n_30"] + x[,"n_35"])*log(3) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_15"] + x[,"n_40"])*(log(3 - 2*r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_10"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_14"] + x[,"n_41"])*(log(pmax(1e-6,r_rc)) + log(12 - 5*r_rc + 2*r_rc^2)) + (x[,"n_25"] + x[,"n_30"])*(log(pmax(1e-6,r_rc)) + log(2 - 3*r_rc + 2*r_rc^2)) + (x[,"n_23"] + x[,"n_32"])*log(57 + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_24"] + x[,"n_31"])*log(18 - 2*r_rc + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_12"] + x[,"n_43"])*log(21 - 16*r_rc + 8*r_rc^2 - 4*r_rc^3) + (x[,"n_11"] + x[,"n_44"])*log(9 - 8*r_rc + r_rc^2 - 2*r_rc^3) + (x[,"n_20"] + x[,"n_35"])*log(1 - 2*r_rc + 3*r_rc^2 - 2*r_rc^3) + (x[,"n_13"] + x[,"n_42"])*log(9 + 12*r_rc - 4*r_rc^2 + 4*r_rc^3) + (x[,"n_21"] + x[,"n_34"])*log(21 - 2*r_rc - 7*r_rc^2 + 6*r_rc^3) + (x[,"n_22"] + x[,"n_33"])*log(61 + 4*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rm2c <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n15 - 4*n20 - 4*n21 - 2*n22 - 2*n23 - 4*n24 - 4*n25 - 4*n30 - 4*n31 - 2*n32 - 2*n33 - 4*n34 - 4*n35 - 3*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 3*n45)*log(2) + (-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n24 - n25 - n30 - n31 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(3) + 2*(-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n15 + n40)*(log(3 - 2*r) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n45)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(1 + 2*r)) + (n25 + n30)*log(9 - 16*r + 22*r^2 - 12*r^3) + (n21 + n34)*log(51 + 4*r + 26*r^2 - 12*r^3) + (n13 + n42)*log(27 - 14*r + 12*r^2 - 4*r^3) + (n14 + n41)*log(9 - 3*r - r^2 - 2*r^3) + (n22 + n33)*log(14 + r + 2*r^2 - 2*r^3) + (n11 + n44)*log(3 + 11*r - 7*r^2 + 2*r^3) + (n23 + n32)*log(15 + r - 4*r^2 + 2*r^3) + (n12 + n43)*log(21 + 2*r + 4*r^3) + (n20 + n35)*log(3 + 8*r - 14*r^2 + 12*r^3) + (n24 + n31)*log(69 - 20*r - 10*r^2 + 12*r^3)
return(L)}
interlogL_rm2c <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rm2c,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rm2c <- (x[,"n_15"] + x[,"n_40"])*log10(2) + (x[,"n_10"] + x[,"n_45"])*log10(2) - (x[,"n_25"] + x[,"n_30"])*log10(5) - (x[,"n_20"] + x[,"n_35"])*log10(5) - (x[,"n_13"] + x[,"n_42"])*(-log10(2) + 2*log10(3) + log10(5)) - (x[,"n_12"] + x[,"n_43"])*(-log10(2) + 2*log10(3) + log10(5)) - (x[,"n_14"] + x[,"n_41"])*log10(7) - (x[,"n_11"] + x[,"n_44"])*log10(7) - (x[,"n_24"] + x[,"n_31"])*(log10(2) + log10(29)) - (x[,"n_21"] + x[,"n_34"])*(log10(2) + log10(29)) - (x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(59)) - (x[,"n_22"] + x[,"n_33"])*(-2*log10(2) + log10(59)) + (x[,"n_15"] + x[,"n_40"])*(log10(3 - 2*r_rm2c) + log10(pmax(1e-6,1 - r_rm2c)) + log10(pmax(1e-6,r_rm2c))) + (x[,"n_10"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_rm2c)) + log10(pmax(1e-6,r_rm2c)) + log10(1 + 2*r_rm2c)) + (x[,"n_25"] + x[,"n_30"])*log10(9 - 16*r_rm2c + 22*r_rm2c^2 - 12*r_rm2c^3) + (x[,"n_21"] + x[,"n_34"])*log10(51 + 4*r_rm2c + 26*r_rm2c^2 - 12*r_rm2c^3) + (x[,"n_13"] + x[,"n_42"])*log10(27 - 14*r_rm2c + 12*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_14"] + x[,"n_41"])*log10(9 - 3*r_rm2c - r_rm2c^2 - 2*r_rm2c^3) + (x[,"n_22"] + x[,"n_33"])*log10(14 + r_rm2c + 2*r_rm2c^2 - 2*r_rm2c^3) + (x[,"n_11"] + x[,"n_44"])*log10(3 + 11*r_rm2c - 7*r_rm2c^2 + 2*r_rm2c^3) + (x[,"n_23"] + x[,"n_32"])*log10(15 + r_rm2c - 4*r_rm2c^2 + 2*r_rm2c^3) + (x[,"n_12"] + x[,"n_43"])*log10(21 + 2*r_rm2c + 4*r_rm2c^3) + (x[,"n_20"] + x[,"n_35"])*log10(3 + 8*r_rm2c - 14*r_rm2c^2 + 12*r_rm2c^3) + (x[,"n_24"] + x[,"n_31"])*log10(69 - 20*r_rm2c - 10*r_rm2c^2 + 12*r_rm2c^3)
logL_rm2c <- (-3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 4*x[,"n_34"] - 4*x[,"n_35"] - 3*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 3*x[,"n_45"])*log(2) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(3) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_15"] + x[,"n_40"])*(log(3 - 2*r_rm2c) + log(pmax(1e-6,1 - r_rm2c)) + log(pmax(1e-6,r_rm2c))) + (x[,"n_10"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_rm2c)) + log(pmax(1e-6,r_rm2c)) + log(1 + 2*r_rm2c)) + (x[,"n_25"] + x[,"n_30"])*log(9 - 16*r_rm2c + 22*r_rm2c^2 - 12*r_rm2c^3) + (x[,"n_21"] + x[,"n_34"])*log(51 + 4*r_rm2c + 26*r_rm2c^2 - 12*r_rm2c^3) + (x[,"n_13"] + x[,"n_42"])*log(27 - 14*r_rm2c + 12*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_14"] + x[,"n_41"])*log(9 - 3*r_rm2c - r_rm2c^2 - 2*r_rm2c^3) + (x[,"n_22"] + x[,"n_33"])*log(14 + r_rm2c + 2*r_rm2c^2 - 2*r_rm2c^3) + (x[,"n_11"] + x[,"n_44"])*log(3 + 11*r_rm2c - 7*r_rm2c^2 + 2*r_rm2c^3) + (x[,"n_23"] + x[,"n_32"])*log(15 + r_rm2c - 4*r_rm2c^2 + 2*r_rm2c^3) + (x[,"n_12"] + x[,"n_43"])*log(21 + 2*r_rm2c + 4*r_rm2c^3) + (x[,"n_20"] + x[,"n_35"])*log(3 + 8*r_rm2c - 14*r_rm2c^2 + 12*r_rm2c^3) + (x[,"n_24"] + x[,"n_31"])*log(69 - 20*r_rm2c - 10*r_rm2c^2 + 12*r_rm2c^3)
logL_rm1c <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n15 + n40)*(log(3 - 2*r) + 2*log(pmax(1e-6,1 - r))) + (n10 + n45)*(2*log(pmax(1e-6,r)) + log(1 + 2*r)) + (n20 + n35)*(log(pmax(1e-6,r)) + log(2 + 5*r - 6*r^2)) + (n11 + n44)*(log(pmax(1e-6,r)) + log(4 + 13*r - 2*r^2)) + (n23 + n32)*log(75 - 44*r + 32*r^2 - 12*r^3) + (n24 + n31)*log(27 - 14*r - r^2 - 6*r^3) + (n12 + n43)*log(3 + 28*r - 8*r^2 - 4*r^3) + (n14 + n41)*log(15 - 24*r + 7*r^2 + 2*r^3) + (n13 + n42)*log(19 - 20*r^2 + 4*r^3) + (n21 + n34)*log(6 + 34*r - 19*r^2 + 6*r^3) + (n25 + n30)*log(1 + 6*r - 13*r^2 + 6*r^3) + (n22 + n33)*log(51 + 16*r - 4*r^2 + 12*r^3)
return(L)}
interlogL_rm1c <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rm1c,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm1c <- mapply(interlogL_rm1c,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rm1c <- (x[,"n_15"] + x[,"n_40"])*log10(2) + (x[,"n_10"] + x[,"n_45"])*log10(2) - (x[,"n_25"] + x[,"n_30"])*(-log10(2) + log10(3)) - (x[,"n_20"] + x[,"n_35"])*(-log10(2) + log10(3)) - (x[,"n_14"] + x[,"n_41"])*log10(5) - (x[,"n_11"] + x[,"n_44"])*log10(5) - (x[,"n_23"] + x[,"n_32"])*(-log10(2) + log10(7) + log10(17)) - (x[,"n_22"] + x[,"n_33"])*(-log10(2) + log10(7) + log10(17)) - (x[,"n_24"] + x[,"n_31"])*log10(19) - (x[,"n_21"] + x[,"n_34"])*log10(19) - (x[,"n_13"] + x[,"n_42"])*(-log10(2) + log10(29)) - (x[,"n_12"] + x[,"n_43"])*(-log10(2) + log10(29)) + (x[,"n_15"] + x[,"n_40"])*(log10(3 - 2*r_rm1c) + 2*log10(pmax(1e-6,1 - r_rm1c))) + (x[,"n_10"] + x[,"n_45"])*(2*log10(pmax(1e-6,r_rm1c)) + log10(1 + 2*r_rm1c)) + (x[,"n_20"] + x[,"n_35"])*(log10(pmax(1e-6,r_rm1c)) + log10(2 + 5*r_rm1c - 6*r_rm1c^2)) + (x[,"n_11"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm1c)) + log10(4 + 13*r_rm1c - 2*r_rm1c^2)) + (x[,"n_23"] + x[,"n_32"])*log10(75 - 44*r_rm1c + 32*r_rm1c^2 - 12*r_rm1c^3) + (x[,"n_24"] + x[,"n_31"])*log10(27 - 14*r_rm1c - r_rm1c^2 - 6*r_rm1c^3) + (x[,"n_12"] + x[,"n_43"])*log10(3 + 28*r_rm1c - 8*r_rm1c^2 - 4*r_rm1c^3) + (x[,"n_14"] + x[,"n_41"])*log10(15 - 24*r_rm1c + 7*r_rm1c^2 + 2*r_rm1c^3) + (x[,"n_13"] + x[,"n_42"])*log10(19 - 20*r_rm1c^2 + 4*r_rm1c^3) + (x[,"n_21"] + x[,"n_34"])*log10(6 + 34*r_rm1c - 19*r_rm1c^2 + 6*r_rm1c^3) + (x[,"n_25"] + x[,"n_30"])*log10(1 + 6*r_rm1c - 13*r_rm1c^2 + 6*r_rm1c^3) + (x[,"n_22"] + x[,"n_33"])*log10(51 + 16*r_rm1c - 4*r_rm1c^2 + 12*r_rm1c^3)
logL_rm1c <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_15"] + x[,"n_40"])*(log(3 - 2*r_rm1c) + 2*log(pmax(1e-6,1 - r_rm1c))) + (x[,"n_10"] + x[,"n_45"])*(2*log(pmax(1e-6,r_rm1c)) + log(1 + 2*r_rm1c)) + (x[,"n_20"] + x[,"n_35"])*(log(pmax(1e-6,r_rm1c)) + log(2 + 5*r_rm1c - 6*r_rm1c^2)) + (x[,"n_11"] + x[,"n_44"])*(log(pmax(1e-6,r_rm1c)) + log(4 + 13*r_rm1c - 2*r_rm1c^2)) + (x[,"n_23"] + x[,"n_32"])*log(75 - 44*r_rm1c + 32*r_rm1c^2 - 12*r_rm1c^3) + (x[,"n_24"] + x[,"n_31"])*log(27 - 14*r_rm1c - r_rm1c^2 - 6*r_rm1c^3) + (x[,"n_12"] + x[,"n_43"])*log(3 + 28*r_rm1c - 8*r_rm1c^2 - 4*r_rm1c^3) + (x[,"n_14"] + x[,"n_41"])*log(15 - 24*r_rm1c + 7*r_rm1c^2 + 2*r_rm1c^3) + (x[,"n_13"] + x[,"n_42"])*log(19 - 20*r_rm1c^2 + 4*r_rm1c^3) + (x[,"n_21"] + x[,"n_34"])*log(6 + 34*r_rm1c - 19*r_rm1c^2 + 6*r_rm1c^3) + (x[,"n_25"] + x[,"n_30"])*log(1 + 6*r_rm1c - 13*r_rm1c^2 + 6*r_rm1c^3) + (x[,"n_22"] + x[,"n_33"])*log(51 + 16*r_rm1c - 4*r_rm1c^2 + 12*r_rm1c^3)

return(list(
r_mat = cbind(r_cc,r_cm2c,r_cm1c,r_rc,r_rm2c,r_rm1c,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm2c,LOD_cm1c,LOD_rc,LOD_rm2c,LOD_rm1c,0),
logL_mat = cbind(logL_cc,logL_cm2c,logL_cm1c,logL_rc,logL_rm2c,logL_rm1c,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed2c","coupling mixed1c","repulsion coupling","repulsion mixed2c","repulsion mixed1c","unknown")
)
)
}

#' @rdname r6_functions
r6_1.4_2.4 <- function(x){
logL_cc <- function(r,n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44,n21,n25,n31,n35,n11,n15,n41,n45) {
L <- (-n12 + 2*n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34 - n42 + 2*n43 - n44)*log(2) + (n21 + n25 + n31 + n35)*log(3) + 2*(-n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + 3*(n11 + n45)*log(pmax(1e-6,1 - r)) + 3*(n15 + n41)*log(pmax(1e-6,r)) + (n13 + n43)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n21 + n35)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n42)*(log(7 - 4*r) + 2*log(pmax(1e-6,r))) + (n25 + n31)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n12 + n44)*(2*log(pmax(1e-6,1 - r)) + log(3 + 4*r)) + (n23 + n33)*log(11 - 8*r + 8*r^2) + (n24 + n32)*(log(pmax(1e-6,r)) + log(16 - 19*r + 12*r^2)) + (n22 + n34)*log(9 - 14*r + 17*r^2 - 12*r^3)
return(L)}
interlogL_cc <- function(n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44,n21,n25,n31,n35,n11,n15,n41,n45) {
optimize(logL_cc,c(0,0.5), n12,n13,n14,n22,n23,n24,n32,n33,n34,n42,n43,n44,n21,n25,n31,n35,n11,n15,n41,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_21"],x[,"n_25"],x[,"n_31"],x[,"n_35"],x[,"n_11"],x[,"n_15"],x[,"n_41"],x[,"n_45"])
LOD_cc <- 3*(x[,"n_25"] + x[,"n_31"])*log10(2) + 3*(x[,"n_21"] + x[,"n_35"])*log10(2) + 3*(x[,"n_15"] + x[,"n_41"])*log10(2) + 2*(x[,"n_13"] + x[,"n_43"])*log10(2) + 3*(x[,"n_11"] + x[,"n_45"])*log10(2) - 2*(x[,"n_23"] + x[,"n_33"])*log10(3) - (x[,"n_14"] + x[,"n_42"])*(-2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - (x[,"n_24"] + x[,"n_32"])*(-2*log10(2) + log10(19)) - (x[,"n_22"] + x[,"n_34"])*(-2*log10(2) + log10(19)) + 3*(x[,"n_11"] + x[,"n_45"])*log10(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_15"] + x[,"n_41"])*log10(pmax(1e-6,r_cc)) + (x[,"n_13"] + x[,"n_43"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_21"] + x[,"n_35"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_42"])*(log10(7 - 4*r_cc) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_25"] + x[,"n_31"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_12"] + x[,"n_44"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(3 + 4*r_cc)) + (x[,"n_23"] + x[,"n_33"])*log10(11 - 8*r_cc + 8*r_cc^2) + (x[,"n_24"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(16 - 19*r_cc + 12*r_cc^2)) + (x[,"n_22"] + x[,"n_34"])*log10(9 - 14*r_cc + 17*r_cc^2 - 12*r_cc^3)
logL_cc <- (-x[,"n_12"] + 2*x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] + 2*x[,"n_43"] - x[,"n_44"])*log(2) + (x[,"n_21"] + x[,"n_25"] + x[,"n_31"] + x[,"n_35"])*log(3) + 2*(-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + 3*(x[,"n_11"] + x[,"n_45"])*log(pmax(1e-6,1 - r_cc)) + 3*(x[,"n_15"] + x[,"n_41"])*log(pmax(1e-6,r_cc)) + (x[,"n_13"] + x[,"n_43"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_21"] + x[,"n_35"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_14"] + x[,"n_42"])*(log(7 - 4*r_cc) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_25"] + x[,"n_31"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_12"] + x[,"n_44"])*(2*log(pmax(1e-6,1 - r_cc)) + log(3 + 4*r_cc)) + (x[,"n_23"] + x[,"n_33"])*log(11 - 8*r_cc + 8*r_cc^2) + (x[,"n_24"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(16 - 19*r_cc + 12*r_cc^2)) + (x[,"n_22"] + x[,"n_34"])*log(9 - 14*r_cc + 17*r_cc^2 - 12*r_cc^3)
logL_cm3c <- function(r,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
L <- (-2*n11 - 3*n12 - 3*n13 - 3*n14 - 2*n15 - 2*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35 - 2*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + 2*(n13 + n43)*log(3) + (-2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n21 - 2*n22 - n23 - 2*n24 - 2*n25 - 2*n31 - 2*n32 - n33 - 2*n34 - 2*n35 - 2*n41 - 2*n42 - 2*n43 - 2*n44 - 2*n45)*log(5) + (n23 + n33)*log(7) + (n15 + n41)*(log(2 - r) + 2*log(pmax(1e-6,r))) + (n11 + n45)*(log(pmax(1e-6,1 - r)) + log(1 - r^2)) + (n25 + n31)*(log(pmax(1e-6,r)) + log(5 - 5*r + 3*r^2)) + (n14 + n42)*(log(pmax(1e-6,r)) + log(12 - 7*r + 4*r^2)) + (n24 + n32)*log(15 + 2*r + 19*r^2 - 12*r^3) + (n12 + n44)*log(9 - 10*r + 5*r^2 - 4*r^3) + (n21 + n35)*log(3 - 4*r + 4*r^2 - 3*r^3) + (n22 + n34)*log(24 - 4*r - 17*r^2 + 12*r^3)
return(L)}
interlogL_cm3c <- function(n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
optimize(logL_cm3c,c(0,0.5), n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm3c <- mapply(interlogL_cm3c,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cm3c <- -((x[,"n_15"] + x[,"n_41"])*(-3*log10(2) + log10(3))) - (x[,"n_11"] + x[,"n_45"])*(-3*log10(2) + log10(3)) - (x[,"n_24"] + x[,"n_32"])*(-2*log10(2) + log10(7) + log10(11)) - (x[,"n_22"] + x[,"n_34"])*(-2*log10(2) + log10(7) + log10(11)) - (x[,"n_25"] + x[,"n_31"])*(-3*log10(2) + log10(13)) - (x[,"n_21"] + x[,"n_35"])*(-3*log10(2) + log10(13)) - (x[,"n_14"] + x[,"n_42"])*(-2*log10(2) + log10(19)) - (x[,"n_12"] + x[,"n_44"])*(-2*log10(2) + log10(19)) + (x[,"n_15"] + x[,"n_41"])*(log10(2 - r_cm3c) + 2*log10(pmax(1e-6,r_cm3c))) + (x[,"n_11"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cm3c)) + log10(1 - r_cm3c^2)) + (x[,"n_25"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm3c)) + log10(5 - 5*r_cm3c + 3*r_cm3c^2)) + (x[,"n_14"] + x[,"n_42"])*(log10(pmax(1e-6,r_cm3c)) + log10(12 - 7*r_cm3c + 4*r_cm3c^2)) + (x[,"n_24"] + x[,"n_32"])*log10(15 + 2*r_cm3c + 19*r_cm3c^2 - 12*r_cm3c^3) + (x[,"n_12"] + x[,"n_44"])*log10(9 - 10*r_cm3c + 5*r_cm3c^2 - 4*r_cm3c^3) + (x[,"n_21"] + x[,"n_35"])*log10(3 - 4*r_cm3c + 4*r_cm3c^2 - 3*r_cm3c^3) + (x[,"n_22"] + x[,"n_34"])*log10(24 - 4*r_cm3c - 17*r_cm3c^2 + 12*r_cm3c^3)
logL_cm3c <- (-2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + 2*(x[,"n_13"] + x[,"n_43"])*log(3) + (-2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 2*x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_31"] - 2*x[,"n_32"] - x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - 2*x[,"n_45"])*log(5) + (x[,"n_23"] + x[,"n_33"])*log(7) + (x[,"n_15"] + x[,"n_41"])*(log(2 - r_cm3c) + 2*log(pmax(1e-6,r_cm3c))) + (x[,"n_11"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cm3c)) + log(1 - r_cm3c^2)) + (x[,"n_25"] + x[,"n_31"])*(log(pmax(1e-6,r_cm3c)) + log(5 - 5*r_cm3c + 3*r_cm3c^2)) + (x[,"n_14"] + x[,"n_42"])*(log(pmax(1e-6,r_cm3c)) + log(12 - 7*r_cm3c + 4*r_cm3c^2)) + (x[,"n_24"] + x[,"n_32"])*log(15 + 2*r_cm3c + 19*r_cm3c^2 - 12*r_cm3c^3) + (x[,"n_12"] + x[,"n_44"])*log(9 - 10*r_cm3c + 5*r_cm3c^2 - 4*r_cm3c^3) + (x[,"n_21"] + x[,"n_35"])*log(3 - 4*r_cm3c + 4*r_cm3c^2 - 3*r_cm3c^3) + (x[,"n_22"] + x[,"n_34"])*log(24 - 4*r_cm3c - 17*r_cm3c^2 + 12*r_cm3c^3)
logL_cm2c <- function(r,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
L <- (-n11 - 2*n12 - 2*n13 - 2*n14 - n15 - n21 - 2*n22 - 2*n23 - 2*n24 - n25 - n31 - 2*n32 - 2*n33 - 2*n34 - n35 - n41 - 2*n42 - 2*n43 - 2*n44 - n45)*log(2) + (-n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*(log(3) + 2*log(5)) + (n11 + n45)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 + r)) + (n23 + n33)*log(51 + 8*r - 8*r^2) + (n15 + n41)*(log(pmax(1e-6,r)) + log(3 - 4*r + r^2)) + (n13 + n43)*log(15 - 8*r + 8*r^2) + (n22 + n34)*log(33 - 14*r + 17*r^2 - 12*r^3) + (n14 + n42)*log(9 - 6*r + 7*r^2 - 4*r^3) + (n25 + n31)*log(3 - 4*r + 7*r^2 - 3*r^3) + (n21 + n35)*log(3 - r - 2*r^2 + 3*r^3) + (n12 + n44)*log(6 + 4*r - 5*r^2 + 4*r^3) + (n24 + n32)*log(24 + 16*r - 19*r^2 + 12*r^3)
return(L)}
interlogL_cm2c <- function(n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
optimize(logL_cm2c,c(0,0.5), n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cm2c <- -((x[,"n_15"] + x[,"n_41"])*(-3*log10(2) + log10(5))) - (x[,"n_11"] + x[,"n_45"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_43"])*log10(13) - (x[,"n_25"] + x[,"n_31"])*(-3*log10(2) + log10(19)) - (x[,"n_21"] + x[,"n_35"])*(-3*log10(2) + log10(19)) - (x[,"n_24"] + x[,"n_32"])*(-2*log10(2) + log10(5) + log10(23)) - (x[,"n_22"] + x[,"n_34"])*(-2*log10(2) + log10(5) + log10(23)) - (x[,"n_14"] + x[,"n_42"])*(-2*log10(2) + log10(29)) - (x[,"n_12"] + x[,"n_44"])*(-2*log10(2) + log10(29)) - (x[,"n_23"] + x[,"n_33"])*log10(53) + (x[,"n_11"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c)) + log10(2 + r_cm2c)) + (x[,"n_23"] + x[,"n_33"])*log10(51 + 8*r_cm2c - 8*r_cm2c^2) + (x[,"n_15"] + x[,"n_41"])*(log10(pmax(1e-6,r_cm2c)) + log10(3 - 4*r_cm2c + r_cm2c^2)) + (x[,"n_13"] + x[,"n_43"])*log10(15 - 8*r_cm2c + 8*r_cm2c^2) + (x[,"n_22"] + x[,"n_34"])*log10(33 - 14*r_cm2c + 17*r_cm2c^2 - 12*r_cm2c^3) + (x[,"n_14"] + x[,"n_42"])*log10(9 - 6*r_cm2c + 7*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_25"] + x[,"n_31"])*log10(3 - 4*r_cm2c + 7*r_cm2c^2 - 3*r_cm2c^3) + (x[,"n_21"] + x[,"n_35"])*log10(3 - r_cm2c - 2*r_cm2c^2 + 3*r_cm2c^3) + (x[,"n_12"] + x[,"n_44"])*log10(6 + 4*r_cm2c - 5*r_cm2c^2 + 4*r_cm2c^3) + (x[,"n_24"] + x[,"n_32"])*log10(24 + 16*r_cm2c - 19*r_cm2c^2 + 12*r_cm2c^3)
logL_cm2c <- (-x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - x[,"n_45"])*log(2) + (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(log(3) + 2*log(5)) + (x[,"n_11"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c)) + log(2 + r_cm2c)) + (x[,"n_23"] + x[,"n_33"])*log(51 + 8*r_cm2c - 8*r_cm2c^2) + (x[,"n_15"] + x[,"n_41"])*(log(pmax(1e-6,r_cm2c)) + log(3 - 4*r_cm2c + r_cm2c^2)) + (x[,"n_13"] + x[,"n_43"])*log(15 - 8*r_cm2c + 8*r_cm2c^2) + (x[,"n_22"] + x[,"n_34"])*log(33 - 14*r_cm2c + 17*r_cm2c^2 - 12*r_cm2c^3) + (x[,"n_14"] + x[,"n_42"])*log(9 - 6*r_cm2c + 7*r_cm2c^2 - 4*r_cm2c^3) + (x[,"n_25"] + x[,"n_31"])*log(3 - 4*r_cm2c + 7*r_cm2c^2 - 3*r_cm2c^3) + (x[,"n_21"] + x[,"n_35"])*log(3 - r_cm2c - 2*r_cm2c^2 + 3*r_cm2c^3) + (x[,"n_12"] + x[,"n_44"])*log(6 + 4*r_cm2c - 5*r_cm2c^2 + 4*r_cm2c^3) + (x[,"n_24"] + x[,"n_32"])*log(24 + 16*r_cm2c - 19*r_cm2c^2 + 12*r_cm2c^3)
logL_rc <- function(r,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
L <- (-2*n11 - n12 - 2*n13 - n14 - 2*n15 - 2*n21 - 2*n22 - 2*n23 - 2*n24 - 2*n25 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n35 - 2*n41 - n42 - 2*n43 - n44 - 2*n45)*log(2) + (n21 + n25 + n31 + n35)*log(3) + 2*(-n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + (n15 + n41)*(log(3 - 2*r) + 2*log(pmax(1e-6,r))) + (n11 + n45)*(2*log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n13 + n43)*log(3 + 4*r - 4*r^2) + (n25 + n31)*(log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n14 + n42)*(log(pmax(1e-6,r)) + log(3 - 2*r + 2*r^2)) + (n23 + n33)*log(19 - 4*r + 4*r^2) + (n24 + n32)*log(9 - 4*r + 16*r^2 - 12*r^3) + (n21 + n35)*log(1 - 2*r + 3*r^2 - 2*r^3) + (n12 + n44)*log(3 - 5*r + 4*r^2 - 2*r^3) + (n22 + n34)*log(9 + 8*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rc <- function(n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
optimize(logL_rc,c(0,0.5), n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rc <- (x[,"n_25"] + x[,"n_31"])*log10(2) + (x[,"n_21"] + x[,"n_35"])*log10(2) + (x[,"n_15"] + x[,"n_41"])*log10(2) - 2*(x[,"n_13"] + x[,"n_43"])*log10(2) + (x[,"n_11"] + x[,"n_45"])*log10(2) - (x[,"n_23"] + x[,"n_33"])*(log10(2) + 2*log10(3)) - (x[,"n_14"] + x[,"n_42"])*(-2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - (x[,"n_24"] + x[,"n_32"])*(-log10(2) + log10(19)) - (x[,"n_22"] + x[,"n_34"])*(-log10(2) + log10(19)) + (x[,"n_15"] + x[,"n_41"])*(log10(3 - 2*r_rc) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_11"] + x[,"n_45"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_13"] + x[,"n_43"])*log10(3 + 4*r_rc - 4*r_rc^2) + (x[,"n_25"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 3*r_rc + 2*r_rc^2)) + (x[,"n_14"] + x[,"n_42"])*(log10(pmax(1e-6,r_rc)) + log10(3 - 2*r_rc + 2*r_rc^2)) + (x[,"n_23"] + x[,"n_33"])*log10(19 - 4*r_rc + 4*r_rc^2) + (x[,"n_24"] + x[,"n_32"])*log10(9 - 4*r_rc + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_21"] + x[,"n_35"])*log10(1 - 2*r_rc + 3*r_rc^2 - 2*r_rc^3) + (x[,"n_12"] + x[,"n_44"])*log10(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3) + (x[,"n_22"] + x[,"n_34"])*log10(9 + 8*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rc <- (-2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_41"] - x[,"n_42"] - 2*x[,"n_43"] - x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_21"] + x[,"n_25"] + x[,"n_31"] + x[,"n_35"])*log(3) + 2*(-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_15"] + x[,"n_41"])*(log(3 - 2*r_rc) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_11"] + x[,"n_45"])*(2*log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_13"] + x[,"n_43"])*log(3 + 4*r_rc - 4*r_rc^2) + (x[,"n_25"] + x[,"n_31"])*(log(pmax(1e-6,r_rc)) + log(2 - 3*r_rc + 2*r_rc^2)) + (x[,"n_14"] + x[,"n_42"])*(log(pmax(1e-6,r_rc)) + log(3 - 2*r_rc + 2*r_rc^2)) + (x[,"n_23"] + x[,"n_33"])*log(19 - 4*r_rc + 4*r_rc^2) + (x[,"n_24"] + x[,"n_32"])*log(9 - 4*r_rc + 16*r_rc^2 - 12*r_rc^3) + (x[,"n_21"] + x[,"n_35"])*log(1 - 2*r_rc + 3*r_rc^2 - 2*r_rc^3) + (x[,"n_12"] + x[,"n_44"])*log(3 - 5*r_rc + 4*r_rc^2 - 2*r_rc^3) + (x[,"n_22"] + x[,"n_34"])*log(9 + 8*r_rc - 20*r_rc^2 + 12*r_rc^3)
logL_rm3c <- function(r,n13,n43,n11,n12,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n44,n45) {
L <- (n13 + n43)*log(3) + (-n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n15 + n41)*(log(3 - 2*r) + log(2 - r) + log(pmax(1e-6,r))) + (n23 + n33)*log(67 + 12*r - 12*r^2) + (n11 + n45)*(log(1 + 2*r) + log(1 - r^2)) + (n13 + n43)*log(7 - 4*r + 4*r^2) + (n22 + n34)*log(39 - 8*r + 20*r^2 - 12*r^3) + (n25 + n31)*log(9 - 10*r + 13*r^2 - 6*r^3) + (n14 + n42)*log(9 + 4*r^2 - 4*r^3) + (n12 + n44)*log(9 + 4*r - 8*r^2 + 4*r^3) + (n21 + n35)*log(6 + 2*r - 5*r^2 + 6*r^3) + (n24 + n32)*log(39 + 4*r - 16*r^2 + 12*r^3)
return(L)}
interlogL_rm3c <- function(n13,n43,n11,n12,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n44,n45) {
optimize(logL_rm3c,c(0,0.5), n13,n43,n11,n12,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm3c <- mapply(interlogL_rm3c,x[,"n_13"],x[,"n_43"],x[,"n_11"],x[,"n_12"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_44"],x[,"n_45"])
LOD_rm3c <- -((x[,"n_15"] + x[,"n_41"])*(-log10(2) + log10(3))) - (x[,"n_11"] + x[,"n_45"])*(-log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_43"])*(log10(2) + log10(3)) - (x[,"n_23"] + x[,"n_33"])*(log10(2) + log10(5) + log10(7)) - (x[,"n_24"] + x[,"n_32"])*(-log10(2) + log10(7) + log10(11)) - (x[,"n_22"] + x[,"n_34"])*(-log10(2) + log10(7) + log10(11)) - (x[,"n_25"] + x[,"n_31"])*(-log10(2) + log10(13)) - (x[,"n_21"] + x[,"n_35"])*(-log10(2) + log10(13)) - (x[,"n_14"] + x[,"n_42"])*(-log10(2) + log10(19)) - (x[,"n_12"] + x[,"n_44"])*(-log10(2) + log10(19)) + (x[,"n_15"] + x[,"n_41"])*(log10(3 - 2*r_rm3c) + log10(2 - r_rm3c) + log10(pmax(1e-6,r_rm3c))) + (x[,"n_23"] + x[,"n_33"])*log10(67 + 12*r_rm3c - 12*r_rm3c^2) + (x[,"n_11"] + x[,"n_45"])*(log10(1 + 2*r_rm3c) + log10(1 - r_rm3c^2)) + (x[,"n_13"] + x[,"n_43"])*log10(7 - 4*r_rm3c + 4*r_rm3c^2) + (x[,"n_22"] + x[,"n_34"])*log10(39 - 8*r_rm3c + 20*r_rm3c^2 - 12*r_rm3c^3) + (x[,"n_25"] + x[,"n_31"])*log10(9 - 10*r_rm3c + 13*r_rm3c^2 - 6*r_rm3c^3) + (x[,"n_14"] + x[,"n_42"])*log10(9 + 4*r_rm3c^2 - 4*r_rm3c^3) + (x[,"n_12"] + x[,"n_44"])*log10(9 + 4*r_rm3c - 8*r_rm3c^2 + 4*r_rm3c^3) + (x[,"n_21"] + x[,"n_35"])*log10(6 + 2*r_rm3c - 5*r_rm3c^2 + 6*r_rm3c^3) + (x[,"n_24"] + x[,"n_32"])*log10(39 + 4*r_rm3c - 16*r_rm3c^2 + 12*r_rm3c^3)
logL_rm3c <- (x[,"n_13"] + x[,"n_43"])*log(3) + (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_15"] + x[,"n_41"])*(log(3 - 2*r_rm3c) + log(2 - r_rm3c) + log(pmax(1e-6,r_rm3c))) + (x[,"n_23"] + x[,"n_33"])*log(67 + 12*r_rm3c - 12*r_rm3c^2) + (x[,"n_11"] + x[,"n_45"])*(log(1 + 2*r_rm3c) + log(1 - r_rm3c^2)) + (x[,"n_13"] + x[,"n_43"])*log(7 - 4*r_rm3c + 4*r_rm3c^2) + (x[,"n_22"] + x[,"n_34"])*log(39 - 8*r_rm3c + 20*r_rm3c^2 - 12*r_rm3c^3) + (x[,"n_25"] + x[,"n_31"])*log(9 - 10*r_rm3c + 13*r_rm3c^2 - 6*r_rm3c^3) + (x[,"n_14"] + x[,"n_42"])*log(9 + 4*r_rm3c^2 - 4*r_rm3c^3) + (x[,"n_12"] + x[,"n_44"])*log(9 + 4*r_rm3c - 8*r_rm3c^2 + 4*r_rm3c^3) + (x[,"n_21"] + x[,"n_35"])*log(6 + 2*r_rm3c - 5*r_rm3c^2 + 6*r_rm3c^3) + (x[,"n_24"] + x[,"n_32"])*log(39 + 4*r_rm3c - 16*r_rm3c^2 + 12*r_rm3c^3)
logL_rm2c <- function(r,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
L <- (-n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*(3*log(2) + log(3) + 2*log(5)) + (n11 + n45)*(log(pmax(1e-6,r)) + log(2 + r) + log(1 + 2*r)) + (n13 + n43)*log(21 + 20*r - 20*r^2) + (n15 + n41)*(log(3 - 2*r) + log(3 - 4*r + r^2)) + (n23 + n33)*log(111 - 20*r + 20*r^2) + (n24 + n32)*log(75 - 40*r + 16*r^2 - 12*r^3) + (n21 + n35)*log(3 + 14*r + r^2 - 6*r^3) + (n12 + n44)*log(3 + 20*r + 8*r^2 - 4*r^3) + (n14 + n42)*log(27 - 24*r - 4*r^2 + 4*r^3) + (n25 + n31)*log(12 + 2*r - 17*r^2 + 6*r^3) + (n22 + n34)*log(39 + 44*r - 20*r^2 + 12*r^3)
return(L)}
interlogL_rm2c <- function(n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45) {
optimize(logL_rm2c,c(0,0.5), n11,n12,n13,n14,n15,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rm2c <- -((x[,"n_15"] + x[,"n_41"])*(-log10(2) + log10(5))) - (x[,"n_11"] + x[,"n_45"])*(-log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_43"])*(log10(2) + log10(13)) - (x[,"n_25"] + x[,"n_31"])*(-log10(2) + log10(19)) - (x[,"n_21"] + x[,"n_35"])*(-log10(2) + log10(19)) - (x[,"n_24"] + x[,"n_32"])*(-log10(2) + log10(5) + log10(23)) - (x[,"n_22"] + x[,"n_34"])*(-log10(2) + log10(5) + log10(23)) - (x[,"n_14"] + x[,"n_42"])*(-log10(2) + log10(29)) - (x[,"n_12"] + x[,"n_44"])*(-log10(2) + log10(29)) - (x[,"n_23"] + x[,"n_33"])*(log10(2) + log10(53)) + (x[,"n_11"] + x[,"n_45"])*(log10(pmax(1e-6,r_rm2c)) + log10(2 + r_rm2c) + log10(1 + 2*r_rm2c)) + (x[,"n_13"] + x[,"n_43"])*log10(21 + 20*r_rm2c - 20*r_rm2c^2) + (x[,"n_15"] + x[,"n_41"])*(log10(3 - 2*r_rm2c) + log10(3 - 4*r_rm2c + r_rm2c^2)) + (x[,"n_23"] + x[,"n_33"])*log10(111 - 20*r_rm2c + 20*r_rm2c^2) + (x[,"n_24"] + x[,"n_32"])*log10(75 - 40*r_rm2c + 16*r_rm2c^2 - 12*r_rm2c^3) + (x[,"n_21"] + x[,"n_35"])*log10(3 + 14*r_rm2c + r_rm2c^2 - 6*r_rm2c^3) + (x[,"n_12"] + x[,"n_44"])*log10(3 + 20*r_rm2c + 8*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_14"] + x[,"n_42"])*log10(27 - 24*r_rm2c - 4*r_rm2c^2 + 4*r_rm2c^3) + (x[,"n_25"] + x[,"n_31"])*log10(12 + 2*r_rm2c - 17*r_rm2c^2 + 6*r_rm2c^3) + (x[,"n_22"] + x[,"n_34"])*log10(39 + 44*r_rm2c - 20*r_rm2c^2 + 12*r_rm2c^3)
logL_rm2c <- (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(3*log(2) + log(3) + 2*log(5)) + (x[,"n_11"] + x[,"n_45"])*(log(pmax(1e-6,r_rm2c)) + log(2 + r_rm2c) + log(1 + 2*r_rm2c)) + (x[,"n_13"] + x[,"n_43"])*log(21 + 20*r_rm2c - 20*r_rm2c^2) + (x[,"n_15"] + x[,"n_41"])*(log(3 - 2*r_rm2c) + log(3 - 4*r_rm2c + r_rm2c^2)) + (x[,"n_23"] + x[,"n_33"])*log(111 - 20*r_rm2c + 20*r_rm2c^2) + (x[,"n_24"] + x[,"n_32"])*log(75 - 40*r_rm2c + 16*r_rm2c^2 - 12*r_rm2c^3) + (x[,"n_21"] + x[,"n_35"])*log(3 + 14*r_rm2c + r_rm2c^2 - 6*r_rm2c^3) + (x[,"n_12"] + x[,"n_44"])*log(3 + 20*r_rm2c + 8*r_rm2c^2 - 4*r_rm2c^3) + (x[,"n_14"] + x[,"n_42"])*log(27 - 24*r_rm2c - 4*r_rm2c^2 + 4*r_rm2c^3) + (x[,"n_25"] + x[,"n_31"])*log(12 + 2*r_rm2c - 17*r_rm2c^2 + 6*r_rm2c^3) + (x[,"n_22"] + x[,"n_34"])*log(39 + 44*r_rm2c - 20*r_rm2c^2 + 12*r_rm2c^3)

return(list(
r_mat = cbind(r_cc,r_cm3c,r_cm2c,r_rc,r_rm3c,r_rm2c,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm3c,LOD_cm2c,LOD_rc,LOD_rm3c,LOD_rm2c,0),
logL_mat = cbind(logL_cc,logL_cm3c,logL_cm2c,logL_rc,logL_rm3c,logL_rm2c,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed3c","coupling mixed2c","repulsion coupling","repulsion mixed3c","repulsion mixed2c","unknown")
)
)
}

#' @rdname r6_functions
r6_1.4_3.2 <- function(x){
logL_cc <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(3*log(2) + log(3) + 2*log(5)) + (n15 + n40)*(2*log(pmax(1e-6,r)) + log(2 + r)) + (n25 + n30)*(log(pmax(1e-6,r)) + log(5 + r - 3*r^2)) + (n10 + n45)*(log(pmax(1e-6,1 - r)) + log(3 - 4*r + r^2)) + (n14 + n41)*(log(pmax(1e-6,r)) + log(9 + 11*r + r^2)) + (n22 + n33)*log(111 - 64*r + 52*r^2 - 24*r^3) + (n13 + n42)*log(9 + 30*r - 4*r^2 - 8*r^3) + (n24 + n31)*log(12 + 38*r - 8*r^2 - 3*r^3) + (n11 + n44)*log(21 - 34*r + 14*r^2 - r^3) + (n21 + n34)*log(39 - 13*r - 17*r^2 + 3*r^3) + (n20 + n35)*log(3 + 2*r - 8*r^2 + 3*r^3) + (n12 + n43)*log(27 + 2*r - 28*r^2 + 8*r^3) + (n23 + n32)*log(75 + 32*r - 20*r^2 + 24*r^3)
return(L)}
interlogL_cc <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cc,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cc <- -((x[,"n_15"] + x[,"n_40"])*(-3*log10(2) + log10(5))) - (x[,"n_10"] + x[,"n_45"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_42"])*(log10(2) + log10(11)) - (x[,"n_12"] + x[,"n_43"])*(log10(2) + log10(11)) - (x[,"n_25"] + x[,"n_30"])*(-3*log10(2) + log10(19)) - (x[,"n_20"] + x[,"n_35"])*(-3*log10(2) + log10(19)) - (x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(59)) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(59)) - (x[,"n_23"] + x[,"n_32"])*log10(89) - (x[,"n_22"] + x[,"n_33"])*log10(89) - (x[,"n_24"] + x[,"n_31"])*(-3*log10(2) + log10(229)) - (x[,"n_21"] + x[,"n_34"])*(-3*log10(2) + log10(229)) + (x[,"n_15"] + x[,"n_40"])*(2*log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_25"] + x[,"n_30"])*(log10(pmax(1e-6,r_cc)) + log10(5 + r_cc - 3*r_cc^2)) + (x[,"n_10"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cc)) + log10(3 - 4*r_cc + r_cc^2)) + (x[,"n_14"] + x[,"n_41"])*(log10(pmax(1e-6,r_cc)) + log10(9 + 11*r_cc + r_cc^2)) + (x[,"n_22"] + x[,"n_33"])*log10(111 - 64*r_cc + 52*r_cc^2 - 24*r_cc^3) + (x[,"n_13"] + x[,"n_42"])*log10(9 + 30*r_cc - 4*r_cc^2 - 8*r_cc^3) + (x[,"n_24"] + x[,"n_31"])*log10(12 + 38*r_cc - 8*r_cc^2 - 3*r_cc^3) + (x[,"n_11"] + x[,"n_44"])*log10(21 - 34*r_cc + 14*r_cc^2 - r_cc^3) + (x[,"n_21"] + x[,"n_34"])*log10(39 - 13*r_cc - 17*r_cc^2 + 3*r_cc^3) + (x[,"n_20"] + x[,"n_35"])*log10(3 + 2*r_cc - 8*r_cc^2 + 3*r_cc^3) + (x[,"n_12"] + x[,"n_43"])*log10(27 + 2*r_cc - 28*r_cc^2 + 8*r_cc^3) + (x[,"n_23"] + x[,"n_32"])*log10(75 + 32*r_cc - 20*r_cc^2 + 24*r_cc^3)
logL_cc <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(3*log(2) + log(3) + 2*log(5)) + (x[,"n_15"] + x[,"n_40"])*(2*log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_25"] + x[,"n_30"])*(log(pmax(1e-6,r_cc)) + log(5 + r_cc - 3*r_cc^2)) + (x[,"n_10"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cc)) + log(3 - 4*r_cc + r_cc^2)) + (x[,"n_14"] + x[,"n_41"])*(log(pmax(1e-6,r_cc)) + log(9 + 11*r_cc + r_cc^2)) + (x[,"n_22"] + x[,"n_33"])*log(111 - 64*r_cc + 52*r_cc^2 - 24*r_cc^3) + (x[,"n_13"] + x[,"n_42"])*log(9 + 30*r_cc - 4*r_cc^2 - 8*r_cc^3) + (x[,"n_24"] + x[,"n_31"])*log(12 + 38*r_cc - 8*r_cc^2 - 3*r_cc^3) + (x[,"n_11"] + x[,"n_44"])*log(21 - 34*r_cc + 14*r_cc^2 - r_cc^3) + (x[,"n_21"] + x[,"n_34"])*log(39 - 13*r_cc - 17*r_cc^2 + 3*r_cc^3) + (x[,"n_20"] + x[,"n_35"])*log(3 + 2*r_cc - 8*r_cc^2 + 3*r_cc^3) + (x[,"n_12"] + x[,"n_43"])*log(27 + 2*r_cc - 28*r_cc^2 + 8*r_cc^3) + (x[,"n_23"] + x[,"n_32"])*log(75 + 32*r_cc - 20*r_cc^2 + 24*r_cc^3)
logL_cm <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n10 + n45)*(log(pmax(1e-6,1 - r)) + log(2 - r) + log(pmax(1e-6,r))) + (n15 + n40)*(log(pmax(1e-6,r)) + log(1 - r^2)) + (n23 + n32)*log(57 - 4*r + 28*r^2 - 24*r^3) + (n12 + n43)*log(19 - 16*r + 20*r^2 - 8*r^3) + (n20 + n35)*log(3 - 5*r + 6*r^2 - 3*r^3) + (n21 + n34)*log(24 - 14*r + 11*r^2 - 3*r^3) + (n14 + n41)*log(3 + 6*r - 5*r^2 - r^3) + (n11 + n44)*log(3 + 7*r - 8*r^2 + r^3) + (n25 + n30)*log(1 + 2*r - 3*r^2 + 3*r^3) + (n24 + n31)*log(18 + r + 2*r^2 + 3*r^3) + (n13 + n42)*log(15 - 4*r^2 + 8*r^3) + (n22 + n33)*log(57 + 20*r - 44*r^2 + 24*r^3)
return(L)}
interlogL_cm <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cm,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cm <- -((x[,"n_15"] + x[,"n_40"])*(-3*log10(2) + log10(3))) - (x[,"n_10"] + x[,"n_45"])*(-3*log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_42"])*(log10(3) + log10(5)) - (x[,"n_12"] + x[,"n_43"])*(log10(3) + log10(5)) - (x[,"n_25"] + x[,"n_30"])*(-3*log10(2) + log10(13)) - (x[,"n_20"] + x[,"n_35"])*(-3*log10(2) + log10(13)) - (x[,"n_24"] + x[,"n_31"])*(-3*log10(2) + log10(5) + log10(31)) - (x[,"n_21"] + x[,"n_34"])*(-3*log10(2) + log10(5) + log10(31)) - (x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(37)) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(37)) - (x[,"n_23"] + x[,"n_32"])*log10(59) - (x[,"n_22"] + x[,"n_33"])*log10(59) + (x[,"n_10"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cm)) + log10(2 - r_cm) + log10(pmax(1e-6,r_cm))) + (x[,"n_15"] + x[,"n_40"])*(log10(pmax(1e-6,r_cm)) + log10(1 - r_cm^2)) + (x[,"n_23"] + x[,"n_32"])*log10(57 - 4*r_cm + 28*r_cm^2 - 24*r_cm^3) + (x[,"n_12"] + x[,"n_43"])*log10(19 - 16*r_cm + 20*r_cm^2 - 8*r_cm^3) + (x[,"n_20"] + x[,"n_35"])*log10(3 - 5*r_cm + 6*r_cm^2 - 3*r_cm^3) + (x[,"n_21"] + x[,"n_34"])*log10(24 - 14*r_cm + 11*r_cm^2 - 3*r_cm^3) + (x[,"n_14"] + x[,"n_41"])*log10(3 + 6*r_cm - 5*r_cm^2 - r_cm^3) + (x[,"n_11"] + x[,"n_44"])*log10(3 + 7*r_cm - 8*r_cm^2 + r_cm^3) + (x[,"n_25"] + x[,"n_30"])*log10(1 + 2*r_cm - 3*r_cm^2 + 3*r_cm^3) + (x[,"n_24"] + x[,"n_31"])*log10(18 + r_cm + 2*r_cm^2 + 3*r_cm^3) + (x[,"n_13"] + x[,"n_42"])*log10(15 - 4*r_cm^2 + 8*r_cm^3) + (x[,"n_22"] + x[,"n_33"])*log10(57 + 20*r_cm - 44*r_cm^2 + 24*r_cm^3)
logL_cm <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_10"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cm)) + log(2 - r_cm) + log(pmax(1e-6,r_cm))) + (x[,"n_15"] + x[,"n_40"])*(log(pmax(1e-6,r_cm)) + log(1 - r_cm^2)) + (x[,"n_23"] + x[,"n_32"])*log(57 - 4*r_cm + 28*r_cm^2 - 24*r_cm^3) + (x[,"n_12"] + x[,"n_43"])*log(19 - 16*r_cm + 20*r_cm^2 - 8*r_cm^3) + (x[,"n_20"] + x[,"n_35"])*log(3 - 5*r_cm + 6*r_cm^2 - 3*r_cm^3) + (x[,"n_21"] + x[,"n_34"])*log(24 - 14*r_cm + 11*r_cm^2 - 3*r_cm^3) + (x[,"n_14"] + x[,"n_41"])*log(3 + 6*r_cm - 5*r_cm^2 - r_cm^3) + (x[,"n_11"] + x[,"n_44"])*log(3 + 7*r_cm - 8*r_cm^2 + r_cm^3) + (x[,"n_25"] + x[,"n_30"])*log(1 + 2*r_cm - 3*r_cm^2 + 3*r_cm^3) + (x[,"n_24"] + x[,"n_31"])*log(18 + r_cm + 2*r_cm^2 + 3*r_cm^3) + (x[,"n_13"] + x[,"n_42"])*log(15 - 4*r_cm^2 + 8*r_cm^3) + (x[,"n_22"] + x[,"n_33"])*log(57 + 20*r_cm - 44*r_cm^2 + 24*r_cm^3)
logL_cr <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n20 - 2*n21 - n22 - n23 - 2*n24 - 2*n25 - 2*n30 - 2*n31 - n32 - n33 - 2*n34 - 2*n35 - 2*n40 - 2*n41 - 2*n42 - 2*n43 - 2*n44 - 2*n45)*log(2) + (n22 + n23 + n32 + n33)*log(3) + 2*(-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n15 + n40)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n45)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n11 + n44)*(log(pmax(1e-6,r)) + log(2 + 2*r - r^2)) + (n20 + n35)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n13 + n42)*log(6 - 9*r + 12*r^2 - 8*r^3) + (n22 + n33)*log(3 - 3*r + 6*r^2 - 4*r^3) + (n24 + n31)*log(6 - 4*r + 4*r^2 - 3*r^3) + (n25 + n30)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n14 + n41)*log(3 - 3*r - r^2 + r^3) + (n21 + n34)*log(3 + 5*r - 5*r^2 + 3*r^3) + (n23 + n32)*log(2 + 3*r - 6*r^2 + 4*r^3) + (n12 + n43)*log(1 + 9*r - 12*r^2 + 8*r^3)
return(L)}
interlogL_cr <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cr,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cr <- 3*(x[,"n_15"] + x[,"n_40"])*log10(2) + 3*(x[,"n_10"] + x[,"n_45"])*log10(2) + (x[,"n_25"] + x[,"n_30"])*(3*log10(2) - log10(3)) - (x[,"n_20"] + x[,"n_35"])*(-3*log10(2) + log10(3)) - (x[,"n_23"] + x[,"n_32"])*(-log10(2) + log10(5)) - (x[,"n_22"] + x[,"n_33"])*(-log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_42"])*(-log10(2) + log10(7)) - (x[,"n_12"] + x[,"n_43"])*(-log10(2) + log10(7)) - (x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(11)) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(11)) - (x[,"n_24"] + x[,"n_31"])*(-3*log10(2) + log10(37)) - (x[,"n_21"] + x[,"n_34"])*(-3*log10(2) + log10(37)) + (x[,"n_15"] + x[,"n_40"])*(2*log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_10"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cr)) + 2*log10(pmax(1e-6,r_cr))) + (x[,"n_11"] + x[,"n_44"])*(log10(pmax(1e-6,r_cr)) + log10(2 + 2*r_cr - r_cr^2)) + (x[,"n_20"] + x[,"n_35"])*(log10(pmax(1e-6,r_cr)) + log10(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_13"] + x[,"n_42"])*log10(6 - 9*r_cr + 12*r_cr^2 - 8*r_cr^3) + (x[,"n_22"] + x[,"n_33"])*log10(3 - 3*r_cr + 6*r_cr^2 - 4*r_cr^3) + (x[,"n_24"] + x[,"n_31"])*log10(6 - 4*r_cr + 4*r_cr^2 - 3*r_cr^3) + (x[,"n_25"] + x[,"n_30"])*log10(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_14"] + x[,"n_41"])*log10(3 - 3*r_cr - r_cr^2 + r_cr^3) + (x[,"n_21"] + x[,"n_34"])*log10(3 + 5*r_cr - 5*r_cr^2 + 3*r_cr^3) + (x[,"n_23"] + x[,"n_32"])*log10(2 + 3*r_cr - 6*r_cr^2 + 4*r_cr^3) + (x[,"n_12"] + x[,"n_43"])*log10(1 + 9*r_cr - 12*r_cr^2 + 8*r_cr^3)
logL_cr <- (-2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 2*x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_22"] + x[,"n_23"] + x[,"n_32"] + x[,"n_33"])*log(3) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_15"] + x[,"n_40"])*(2*log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_10"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cr)) + 2*log(pmax(1e-6,r_cr))) + (x[,"n_11"] + x[,"n_44"])*(log(pmax(1e-6,r_cr)) + log(2 + 2*r_cr - r_cr^2)) + (x[,"n_20"] + x[,"n_35"])*(log(pmax(1e-6,r_cr)) + log(2 - 4*r_cr + 3*r_cr^2)) + (x[,"n_13"] + x[,"n_42"])*log(6 - 9*r_cr + 12*r_cr^2 - 8*r_cr^3) + (x[,"n_22"] + x[,"n_33"])*log(3 - 3*r_cr + 6*r_cr^2 - 4*r_cr^3) + (x[,"n_24"] + x[,"n_31"])*log(6 - 4*r_cr + 4*r_cr^2 - 3*r_cr^3) + (x[,"n_25"] + x[,"n_30"])*log(1 - 3*r_cr + 5*r_cr^2 - 3*r_cr^3) + (x[,"n_14"] + x[,"n_41"])*log(3 - 3*r_cr - r_cr^2 + r_cr^3) + (x[,"n_21"] + x[,"n_34"])*log(3 + 5*r_cr - 5*r_cr^2 + 3*r_cr^3) + (x[,"n_23"] + x[,"n_32"])*log(2 + 3*r_cr - 6*r_cr^2 + 4*r_cr^3) + (x[,"n_12"] + x[,"n_43"])*log(1 + 9*r_cr - 12*r_cr^2 + 8*r_cr^3)
logL_rc <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(3*log(2) + log(3) + 2*log(5)) + (n15 + n40)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 + r)) + (n10 + n45)*(log(pmax(1e-6,r)) + log(3 - 4*r + r^2)) + (n23 + n32)*log(93 - 16*r + 28*r^2 - 24*r^3) + (n12 + n43)*log(27 - 18*r + 20*r^2 - 8*r^3) + (n20 + n35)*log(3 - 4*r + 7*r^2 - 3*r^3) + (n21 + n34)*log(30 - 7*r + 10*r^2 - 3*r^3) + (n14 + n41)*log(3 + 11*r - 4*r^2 - r^3) + (n11 + n44)*log(9 - 7*r^2 + r^3) + (n25 + n30)*log(3 - r - 2*r^2 + 3*r^3) + (n24 + n31)*log(30 - 4*r + r^2 + 3*r^3) + (n13 + n42)*log(21 + 2*r - 4*r^2 + 8*r^3) + (n22 + n33)*log(81 + 32*r - 44*r^2 + 24*r^3)
return(L)}
interlogL_rc <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rc,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rc <- -((x[,"n_15"] + x[,"n_40"])*(-3*log10(2) + log10(5))) - (x[,"n_10"] + x[,"n_45"])*(-3*log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_42"])*(log10(2) + log10(11)) - (x[,"n_12"] + x[,"n_43"])*(log10(2) + log10(11)) - (x[,"n_25"] + x[,"n_30"])*(-3*log10(2) + log10(19)) - (x[,"n_20"] + x[,"n_35"])*(-3*log10(2) + log10(19)) - (x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(59)) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(59)) - (x[,"n_23"] + x[,"n_32"])*log10(89) - (x[,"n_22"] + x[,"n_33"])*log10(89) - (x[,"n_24"] + x[,"n_31"])*(-3*log10(2) + log10(229)) - (x[,"n_21"] + x[,"n_34"])*(-3*log10(2) + log10(229)) + (x[,"n_15"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc)) + log10(2 + r_rc)) + (x[,"n_10"] + x[,"n_45"])*(log10(pmax(1e-6,r_rc)) + log10(3 - 4*r_rc + r_rc^2)) + (x[,"n_23"] + x[,"n_32"])*log10(93 - 16*r_rc + 28*r_rc^2 - 24*r_rc^3) + (x[,"n_12"] + x[,"n_43"])*log10(27 - 18*r_rc + 20*r_rc^2 - 8*r_rc^3) + (x[,"n_20"] + x[,"n_35"])*log10(3 - 4*r_rc + 7*r_rc^2 - 3*r_rc^3) + (x[,"n_21"] + x[,"n_34"])*log10(30 - 7*r_rc + 10*r_rc^2 - 3*r_rc^3) + (x[,"n_14"] + x[,"n_41"])*log10(3 + 11*r_rc - 4*r_rc^2 - r_rc^3) + (x[,"n_11"] + x[,"n_44"])*log10(9 - 7*r_rc^2 + r_rc^3) + (x[,"n_25"] + x[,"n_30"])*log10(3 - r_rc - 2*r_rc^2 + 3*r_rc^3) + (x[,"n_24"] + x[,"n_31"])*log10(30 - 4*r_rc + r_rc^2 + 3*r_rc^3) + (x[,"n_13"] + x[,"n_42"])*log10(21 + 2*r_rc - 4*r_rc^2 + 8*r_rc^3) + (x[,"n_22"] + x[,"n_33"])*log10(81 + 32*r_rc - 44*r_rc^2 + 24*r_rc^3)
logL_rc <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(3*log(2) + log(3) + 2*log(5)) + (x[,"n_15"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc)) + log(2 + r_rc)) + (x[,"n_10"] + x[,"n_45"])*(log(pmax(1e-6,r_rc)) + log(3 - 4*r_rc + r_rc^2)) + (x[,"n_23"] + x[,"n_32"])*log(93 - 16*r_rc + 28*r_rc^2 - 24*r_rc^3) + (x[,"n_12"] + x[,"n_43"])*log(27 - 18*r_rc + 20*r_rc^2 - 8*r_rc^3) + (x[,"n_20"] + x[,"n_35"])*log(3 - 4*r_rc + 7*r_rc^2 - 3*r_rc^3) + (x[,"n_21"] + x[,"n_34"])*log(30 - 7*r_rc + 10*r_rc^2 - 3*r_rc^3) + (x[,"n_14"] + x[,"n_41"])*log(3 + 11*r_rc - 4*r_rc^2 - r_rc^3) + (x[,"n_11"] + x[,"n_44"])*log(9 - 7*r_rc^2 + r_rc^3) + (x[,"n_25"] + x[,"n_30"])*log(3 - r_rc - 2*r_rc^2 + 3*r_rc^3) + (x[,"n_24"] + x[,"n_31"])*log(30 - 4*r_rc + r_rc^2 + 3*r_rc^3) + (x[,"n_13"] + x[,"n_42"])*log(21 + 2*r_rc - 4*r_rc^2 + 8*r_rc^3) + (x[,"n_22"] + x[,"n_33"])*log(81 + 32*r_rc - 44*r_rc^2 + 24*r_rc^3)
logL_rm <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*(4*log(2) + 2*log(5)) + (n10 + n45)*(log(2 - r) + 2*log(pmax(1e-6,r))) + (n15 + n40)*(log(pmax(1e-6,1 - r)) + log(1 - r^2)) + (n11 + n44)*(log(pmax(1e-6,r)) + log(9 + r - r^2)) + (n20 + n35)*(log(pmax(1e-6,r)) + log(5 - 5*r + 3*r^2)) + (n22 + n33)*log(55 - 4*r + 36*r^2 - 24*r^3) + (n13 + n42)*log(21 - 16*r + 12*r^2 - 8*r^3) + (n25 + n30)*log(3 - 4*r + 4*r^2 - 3*r^3) + (n24 + n31)*log(27 - 17*r + 5*r^2 - 3*r^3) + (n14 + n41)*log(9 - 8*r - 2*r^2 + r^3) + (n21 + n34)*log(12 + 16*r - 4*r^2 + 3*r^3) + (n12 + n43)*log(9 + 16*r - 12*r^2 + 8*r^3) + (n23 + n32)*log(63 + 4*r - 36*r^2 + 24*r^3)
return(L)}
interlogL_rm <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rm,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rm <- -((x[,"n_15"] + x[,"n_40"])*(-3*log10(2) + log10(3))) - (x[,"n_10"] + x[,"n_45"])*(-3*log10(2) + log10(3)) - (x[,"n_13"] + x[,"n_42"])*(log10(3) + log10(5)) - (x[,"n_12"] + x[,"n_43"])*(log10(3) + log10(5)) - (x[,"n_25"] + x[,"n_30"])*(-3*log10(2) + log10(13)) - (x[,"n_20"] + x[,"n_35"])*(-3*log10(2) + log10(13)) - (x[,"n_24"] + x[,"n_31"])*(-3*log10(2) + log10(5) + log10(31)) - (x[,"n_21"] + x[,"n_34"])*(-3*log10(2) + log10(5) + log10(31)) - (x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(37)) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(37)) - (x[,"n_23"] + x[,"n_32"])*log10(59) - (x[,"n_22"] + x[,"n_33"])*log10(59) + (x[,"n_10"] + x[,"n_45"])*(log10(2 - r_rm) + 2*log10(pmax(1e-6,r_rm))) + (x[,"n_15"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_rm)) + log10(1 - r_rm^2)) + (x[,"n_11"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm)) + log10(9 + r_rm - r_rm^2)) + (x[,"n_20"] + x[,"n_35"])*(log10(pmax(1e-6,r_rm)) + log10(5 - 5*r_rm + 3*r_rm^2)) + (x[,"n_22"] + x[,"n_33"])*log10(55 - 4*r_rm + 36*r_rm^2 - 24*r_rm^3) + (x[,"n_13"] + x[,"n_42"])*log10(21 - 16*r_rm + 12*r_rm^2 - 8*r_rm^3) + (x[,"n_25"] + x[,"n_30"])*log10(3 - 4*r_rm + 4*r_rm^2 - 3*r_rm^3) + (x[,"n_24"] + x[,"n_31"])*log10(27 - 17*r_rm + 5*r_rm^2 - 3*r_rm^3) + (x[,"n_14"] + x[,"n_41"])*log10(9 - 8*r_rm - 2*r_rm^2 + r_rm^3) + (x[,"n_21"] + x[,"n_34"])*log10(12 + 16*r_rm - 4*r_rm^2 + 3*r_rm^3) + (x[,"n_12"] + x[,"n_43"])*log10(9 + 16*r_rm - 12*r_rm^2 + 8*r_rm^3) + (x[,"n_23"] + x[,"n_32"])*log10(63 + 4*r_rm - 36*r_rm^2 + 24*r_rm^3)
logL_rm <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*(4*log(2) + 2*log(5)) + (x[,"n_10"] + x[,"n_45"])*(log(2 - r_rm) + 2*log(pmax(1e-6,r_rm))) + (x[,"n_15"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_rm)) + log(1 - r_rm^2)) + (x[,"n_11"] + x[,"n_44"])*(log(pmax(1e-6,r_rm)) + log(9 + r_rm - r_rm^2)) + (x[,"n_20"] + x[,"n_35"])*(log(pmax(1e-6,r_rm)) + log(5 - 5*r_rm + 3*r_rm^2)) + (x[,"n_22"] + x[,"n_33"])*log(55 - 4*r_rm + 36*r_rm^2 - 24*r_rm^3) + (x[,"n_13"] + x[,"n_42"])*log(21 - 16*r_rm + 12*r_rm^2 - 8*r_rm^3) + (x[,"n_25"] + x[,"n_30"])*log(3 - 4*r_rm + 4*r_rm^2 - 3*r_rm^3) + (x[,"n_24"] + x[,"n_31"])*log(27 - 17*r_rm + 5*r_rm^2 - 3*r_rm^3) + (x[,"n_14"] + x[,"n_41"])*log(9 - 8*r_rm - 2*r_rm^2 + r_rm^3) + (x[,"n_21"] + x[,"n_34"])*log(12 + 16*r_rm - 4*r_rm^2 + 3*r_rm^3) + (x[,"n_12"] + x[,"n_43"])*log(9 + 16*r_rm - 12*r_rm^2 + 8*r_rm^3) + (x[,"n_23"] + x[,"n_32"])*log(63 + 4*r_rm - 36*r_rm^2 + 24*r_rm^3)
logL_rr <- function(r,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 2*n15 - 2*n20 - 2*n21 - n22 - n23 - 2*n24 - 2*n25 - 2*n30 - 2*n31 - n32 - n33 - 2*n34 - 2*n35 - 2*n40 - 2*n41 - 2*n42 - 2*n43 - 2*n44 - 2*n45)*log(2) + (n20 + n25 + n30 + n35)*log(3) + 2*(-n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + 3*(n15 + n40)*log(pmax(1e-6,1 - r)) + (n14 + n41)*(2*log(pmax(1e-6,1 - r)) + log(6 - r)) + 3*(n10 + n45)*log(pmax(1e-6,r)) + (n25 + n30)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n35)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n11 + n44)*(2*log(pmax(1e-6,r)) + log(5 + r)) + (n12 + n43)*(log(pmax(1e-6,r)) + log(7 + 4*r - 8*r^2)) + (n21 + n34)*(log(pmax(1e-6,r)) + log(11 - 2*r - 3*r^2)) + (n23 + n32)*log(12 - 17*r + 22*r^2 - 12*r^3) + (n24 + n31)*log(6 + 2*r - 11*r^2 + 3*r^3) + (n13 + n42)*log(3 + 9*r - 20*r^2 + 8*r^3) + (n22 + n33)*log(5 + 9*r - 14*r^2 + 12*r^3)
return(L)}
interlogL_rr <- function(n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rr,c(0,0.5), n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rr <- 3*(x[,"n_25"] + x[,"n_30"])*log10(2) + 3*(x[,"n_20"] + x[,"n_35"])*log10(2) + 3*(x[,"n_15"] + x[,"n_40"])*log10(2) + 3*(x[,"n_10"] + x[,"n_45"])*log10(2) - (x[,"n_23"] + x[,"n_32"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_22"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_13"] + x[,"n_42"])*(-log10(2) + log10(7)) - (x[,"n_12"] + x[,"n_43"])*(-log10(2) + log10(7)) - (x[,"n_14"] + x[,"n_41"])*(-3*log10(2) + log10(11)) - (x[,"n_11"] + x[,"n_44"])*(-3*log10(2) + log10(11)) - (x[,"n_24"] + x[,"n_31"])*(-3*log10(2) + log10(37)) - (x[,"n_21"] + x[,"n_34"])*(-3*log10(2) + log10(37)) + 3*(x[,"n_15"] + x[,"n_40"])*log10(pmax(1e-6,1 - r_rr)) + (x[,"n_14"] + x[,"n_41"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(6 - r_rr)) + 3*(x[,"n_10"] + x[,"n_45"])*log10(pmax(1e-6,r_rr)) + (x[,"n_25"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_rr)) + log10(pmax(1e-6,r_rr))) + (x[,"n_20"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_rr)) + 2*log10(pmax(1e-6,r_rr))) + (x[,"n_11"] + x[,"n_44"])*(2*log10(pmax(1e-6,r_rr)) + log10(5 + r_rr)) + (x[,"n_12"] + x[,"n_43"])*(log10(pmax(1e-6,r_rr)) + log10(7 + 4*r_rr - 8*r_rr^2)) + (x[,"n_21"] + x[,"n_34"])*(log10(pmax(1e-6,r_rr)) + log10(11 - 2*r_rr - 3*r_rr^2)) + (x[,"n_23"] + x[,"n_32"])*log10(12 - 17*r_rr + 22*r_rr^2 - 12*r_rr^3) + (x[,"n_24"] + x[,"n_31"])*log10(6 + 2*r_rr - 11*r_rr^2 + 3*r_rr^3) + (x[,"n_13"] + x[,"n_42"])*log10(3 + 9*r_rr - 20*r_rr^2 + 8*r_rr^3) + (x[,"n_22"] + x[,"n_33"])*log10(5 + 9*r_rr - 14*r_rr^2 + 12*r_rr^3)
logL_rr <- (-2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 2*x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_20"] + x[,"n_25"] + x[,"n_30"] + x[,"n_35"])*log(3) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + 3*(x[,"n_15"] + x[,"n_40"])*log(pmax(1e-6,1 - r_rr)) + (x[,"n_14"] + x[,"n_41"])*(2*log(pmax(1e-6,1 - r_rr)) + log(6 - r_rr)) + 3*(x[,"n_10"] + x[,"n_45"])*log(pmax(1e-6,r_rr)) + (x[,"n_25"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_rr)) + log(pmax(1e-6,r_rr))) + (x[,"n_20"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_rr)) + 2*log(pmax(1e-6,r_rr))) + (x[,"n_11"] + x[,"n_44"])*(2*log(pmax(1e-6,r_rr)) + log(5 + r_rr)) + (x[,"n_12"] + x[,"n_43"])*(log(pmax(1e-6,r_rr)) + log(7 + 4*r_rr - 8*r_rr^2)) + (x[,"n_21"] + x[,"n_34"])*(log(pmax(1e-6,r_rr)) + log(11 - 2*r_rr - 3*r_rr^2)) + (x[,"n_23"] + x[,"n_32"])*log(12 - 17*r_rr + 22*r_rr^2 - 12*r_rr^3) + (x[,"n_24"] + x[,"n_31"])*log(6 + 2*r_rr - 11*r_rr^2 + 3*r_rr^3) + (x[,"n_13"] + x[,"n_42"])*log(3 + 9*r_rr - 20*r_rr^2 + 8*r_rr^3) + (x[,"n_22"] + x[,"n_33"])*log(5 + 9*r_rr - 14*r_rr^2 + 12*r_rr^3)

return(list(
r_mat = cbind(r_cc,r_cm,r_cr,r_rc,r_rm,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_cr,LOD_rc,LOD_rm,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cm,logL_cr,logL_rc,logL_rm,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling repulsion","repulsion coupling","repulsion mixed","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.4_3.3 <- function(x){
logL_cc <- function(r,n20,n26,n30,n36,n10,n11,n12,n13,n14,n15,n16,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n46) {
L <- (n20 + n26 + n30 + n36)*log(3) + (-n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(4*log(2) + 2*log(5)) + 3*(n10 + n46)*log(pmax(1e-6,1 - r)) + (n11 + n45)*(log(9 - 2*r) + 2*log(pmax(1e-6,1 - r))) + 3*(n16 + n40)*log(pmax(1e-6,r)) + (n20 + n36)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n26 + n30)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n15 + n41)*(2*log(pmax(1e-6,r)) + log(7 + 2*r)) + (n13 + n43)*log(9 + 28*r - 28*r^2) + (n14 + n42)*(log(pmax(1e-6,r)) + log(15 + 13*r - 7*r^2)) + (n25 + n31)*(log(pmax(1e-6,r)) + log(16 - r - 6*r^2)) + (n23 + n33)*log(73 - 28*r + 28*r^2) + (n22 + n34)*log(57 - 44*r + 29*r^2 - 21*r^3) + (n21 + n35)*log(9 + 4*r - 19*r^2 + 6*r^3) + (n12 + n44)*log(21 - 20*r - 8*r^2 + 7*r^3) + (n24 + n32)*log(21 + 49*r - 34*r^2 + 21*r^3)
return(L)}
interlogL_cc <- function(n20,n26,n30,n36,n10,n11,n12,n13,n14,n15,n16,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_cc,c(0,0.5), n20,n26,n30,n36,n10,n11,n12,n13,n14,n15,n16,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_20"],x[,"n_26"],x[,"n_30"],x[,"n_36"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_cc <- 3*(x[,"n_26"] + x[,"n_30"])*log10(2) + 3*(x[,"n_20"] + x[,"n_36"])*log10(2) + 3*(x[,"n_16"] + x[,"n_40"])*log10(2) - (x[,"n_15"] + x[,"n_41"])*log10(2) - 4*(x[,"n_13"] + x[,"n_43"])*log10(2) - (x[,"n_11"] + x[,"n_45"])*log10(2) + 3*(x[,"n_10"] + x[,"n_46"])*log10(2) - (x[,"n_25"] + x[,"n_31"])*log10(7) - (x[,"n_21"] + x[,"n_35"])*log10(7) - (x[,"n_23"] + x[,"n_33"])*(log10(2) + log10(3) + log10(11)) - (x[,"n_14"] + x[,"n_42"])*(-3*log10(2) + log10(79)) - (x[,"n_12"] + x[,"n_44"])*(-3*log10(2) + log10(79)) - (x[,"n_24"] + x[,"n_32"])*(-3*log10(2) + log10(317)) - (x[,"n_22"] + x[,"n_34"])*(-3*log10(2) + log10(317)) + 3*(x[,"n_10"] + x[,"n_46"])*log10(pmax(1e-6,1 - r_cc)) + (x[,"n_11"] + x[,"n_45"])*(log10(9 - 2*r_cc) + 2*log10(pmax(1e-6,1 - r_cc))) + 3*(x[,"n_16"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + (x[,"n_20"] + x[,"n_36"])*(2*log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_26"] + x[,"n_30"])*(log10(pmax(1e-6,1 - r_cc)) + 2*log10(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_41"])*(2*log10(pmax(1e-6,r_cc)) + log10(7 + 2*r_cc)) + (x[,"n_13"] + x[,"n_43"])*log10(9 + 28*r_cc - 28*r_cc^2) + (x[,"n_14"] + x[,"n_42"])*(log10(pmax(1e-6,r_cc)) + log10(15 + 13*r_cc - 7*r_cc^2)) + (x[,"n_25"] + x[,"n_31"])*(log10(pmax(1e-6,r_cc)) + log10(16 - r_cc - 6*r_cc^2)) + (x[,"n_23"] + x[,"n_33"])*log10(73 - 28*r_cc + 28*r_cc^2) + (x[,"n_22"] + x[,"n_34"])*log10(57 - 44*r_cc + 29*r_cc^2 - 21*r_cc^3) + (x[,"n_21"] + x[,"n_35"])*log10(9 + 4*r_cc - 19*r_cc^2 + 6*r_cc^3) + (x[,"n_12"] + x[,"n_44"])*log10(21 - 20*r_cc - 8*r_cc^2 + 7*r_cc^3) + (x[,"n_24"] + x[,"n_32"])*log10(21 + 49*r_cc - 34*r_cc^2 + 21*r_cc^3)
logL_cc <- (x[,"n_20"] + x[,"n_26"] + x[,"n_30"] + x[,"n_36"])*log(3) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(4*log(2) + 2*log(5)) + 3*(x[,"n_10"] + x[,"n_46"])*log(pmax(1e-6,1 - r_cc)) + (x[,"n_11"] + x[,"n_45"])*(log(9 - 2*r_cc) + 2*log(pmax(1e-6,1 - r_cc))) + 3*(x[,"n_16"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + (x[,"n_20"] + x[,"n_36"])*(2*log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_26"] + x[,"n_30"])*(log(pmax(1e-6,1 - r_cc)) + 2*log(pmax(1e-6,r_cc))) + (x[,"n_15"] + x[,"n_41"])*(2*log(pmax(1e-6,r_cc)) + log(7 + 2*r_cc)) + (x[,"n_13"] + x[,"n_43"])*log(9 + 28*r_cc - 28*r_cc^2) + (x[,"n_14"] + x[,"n_42"])*(log(pmax(1e-6,r_cc)) + log(15 + 13*r_cc - 7*r_cc^2)) + (x[,"n_25"] + x[,"n_31"])*(log(pmax(1e-6,r_cc)) + log(16 - r_cc - 6*r_cc^2)) + (x[,"n_23"] + x[,"n_33"])*log(73 - 28*r_cc + 28*r_cc^2) + (x[,"n_22"] + x[,"n_34"])*log(57 - 44*r_cc + 29*r_cc^2 - 21*r_cc^3) + (x[,"n_21"] + x[,"n_35"])*log(9 + 4*r_cc - 19*r_cc^2 + 6*r_cc^3) + (x[,"n_12"] + x[,"n_44"])*log(21 - 20*r_cc - 8*r_cc^2 + 7*r_cc^3) + (x[,"n_24"] + x[,"n_32"])*log(21 + 49*r_cc - 34*r_cc^2 + 21*r_cc^3)
logL_cm2c <- function(r,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46,n23,n33) {
L <- (-3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n15 - 3*n16 - 4*n20 - 4*n21 - 4*n22 - 4*n24 - 4*n25 - 4*n26 - 4*n30 - 4*n31 - 4*n32 - 4*n34 - 4*n35 - 4*n36 - 3*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 3*n45 - 3*n46)*log(2) + (-n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(log(3) + 2*log(5)) + (n10 + n46)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n16 + n40)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n15 + n41)*(log(pmax(1e-6,r)) + log(6 - r - 2*r^2)) + (n23 + n33)*log(12 + r - r^2) + (n26 + n30)*(log(pmax(1e-6,r)) + log(5 - 8*r + 6*r^2)) + (n13 + n43)*log(27 - 8*r + 8*r^2) + (n24 + n32)*log(105 + 7*r + 62*r^2 - 42*r^3) + (n21 + n35)*log(33 - 32*r + 26*r^2 - 12*r^3) + (n12 + n44)*log(21 - 16*r + 11*r^2 - 7*r^3) + (n20 + n36)*log(3 - 7*r + 10*r^2 - 6*r^3) + (n11 + n45)*log(3 + 2*r - 7*r^2 + 2*r^3) + (n14 + n42)*log(9 + 15*r - 10*r^2 + 7*r^3) + (n25 + n31)*log(15 + 16*r - 10*r^2 + 12*r^3) + (n22 + n34)*log(132 - 5*r - 64*r^2 + 42*r^3)
return(L)}
interlogL_cm2c <- function(n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46,n23,n33) {
optimize(logL_cm2c,c(0,0.5), n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46,n23,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm2c <- mapply(interlogL_cm2c,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"],x[,"n_23"],x[,"n_33"])
LOD_cm2c <- 3*(x[,"n_16"] + x[,"n_40"])*log10(2) + 3*(x[,"n_10"] + x[,"n_46"])*log10(2) - 2*(x[,"n_13"] + x[,"n_43"])*log10(5) - (x[,"n_26"] + x[,"n_30"])*(-2*log10(2) + log10(5)) - (x[,"n_20"] + x[,"n_36"])*(-2*log10(2) + log10(5)) - (x[,"n_15"] + x[,"n_41"])*(-log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_45"])*(-log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_33"])*(-2*log10(2) + 2*log10(7)) - (x[,"n_25"] + x[,"n_31"])*(log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_35"])*(log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_42"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_12"] + x[,"n_44"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_24"] + x[,"n_32"])*(-2*log10(2) + 2*log10(5) + log10(19)) - (x[,"n_22"] + x[,"n_34"])*(-2*log10(2) + 2*log10(5) + log10(19)) + (x[,"n_10"] + x[,"n_46"])*(2*log10(pmax(1e-6,1 - r_cm2c)) + log10(pmax(1e-6,r_cm2c))) + (x[,"n_16"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_cm2c)) + 2*log10(pmax(1e-6,r_cm2c))) + (x[,"n_15"] + x[,"n_41"])*(log10(pmax(1e-6,r_cm2c)) + log10(6 - r_cm2c - 2*r_cm2c^2)) + (x[,"n_23"] + x[,"n_33"])*log10(12 + r_cm2c - r_cm2c^2) + (x[,"n_26"] + x[,"n_30"])*(log10(pmax(1e-6,r_cm2c)) + log10(5 - 8*r_cm2c + 6*r_cm2c^2)) + (x[,"n_13"] + x[,"n_43"])*log10(27 - 8*r_cm2c + 8*r_cm2c^2) + (x[,"n_24"] + x[,"n_32"])*log10(105 + 7*r_cm2c + 62*r_cm2c^2 - 42*r_cm2c^3) + (x[,"n_21"] + x[,"n_35"])*log10(33 - 32*r_cm2c + 26*r_cm2c^2 - 12*r_cm2c^3) + (x[,"n_12"] + x[,"n_44"])*log10(21 - 16*r_cm2c + 11*r_cm2c^2 - 7*r_cm2c^3) + (x[,"n_20"] + x[,"n_36"])*log10(3 - 7*r_cm2c + 10*r_cm2c^2 - 6*r_cm2c^3) + (x[,"n_11"] + x[,"n_45"])*log10(3 + 2*r_cm2c - 7*r_cm2c^2 + 2*r_cm2c^3) + (x[,"n_14"] + x[,"n_42"])*log10(9 + 15*r_cm2c - 10*r_cm2c^2 + 7*r_cm2c^3) + (x[,"n_25"] + x[,"n_31"])*log10(15 + 16*r_cm2c - 10*r_cm2c^2 + 12*r_cm2c^3) + (x[,"n_22"] + x[,"n_34"])*log10(132 - 5*r_cm2c - 64*r_cm2c^2 + 42*r_cm2c^3)
logL_cm2c <- (-3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 3*x[,"n_16"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_34"] - 4*x[,"n_35"] - 4*x[,"n_36"] - 3*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 3*x[,"n_45"] - 3*x[,"n_46"])*log(2) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(log(3) + 2*log(5)) + (x[,"n_10"] + x[,"n_46"])*(2*log(pmax(1e-6,1 - r_cm2c)) + log(pmax(1e-6,r_cm2c))) + (x[,"n_16"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_cm2c)) + 2*log(pmax(1e-6,r_cm2c))) + (x[,"n_15"] + x[,"n_41"])*(log(pmax(1e-6,r_cm2c)) + log(6 - r_cm2c - 2*r_cm2c^2)) + (x[,"n_23"] + x[,"n_33"])*log(12 + r_cm2c - r_cm2c^2) + (x[,"n_26"] + x[,"n_30"])*(log(pmax(1e-6,r_cm2c)) + log(5 - 8*r_cm2c + 6*r_cm2c^2)) + (x[,"n_13"] + x[,"n_43"])*log(27 - 8*r_cm2c + 8*r_cm2c^2) + (x[,"n_24"] + x[,"n_32"])*log(105 + 7*r_cm2c + 62*r_cm2c^2 - 42*r_cm2c^3) + (x[,"n_21"] + x[,"n_35"])*log(33 - 32*r_cm2c + 26*r_cm2c^2 - 12*r_cm2c^3) + (x[,"n_12"] + x[,"n_44"])*log(21 - 16*r_cm2c + 11*r_cm2c^2 - 7*r_cm2c^3) + (x[,"n_20"] + x[,"n_36"])*log(3 - 7*r_cm2c + 10*r_cm2c^2 - 6*r_cm2c^3) + (x[,"n_11"] + x[,"n_45"])*log(3 + 2*r_cm2c - 7*r_cm2c^2 + 2*r_cm2c^3) + (x[,"n_14"] + x[,"n_42"])*log(9 + 15*r_cm2c - 10*r_cm2c^2 + 7*r_cm2c^3) + (x[,"n_25"] + x[,"n_31"])*log(15 + 16*r_cm2c - 10*r_cm2c^2 + 12*r_cm2c^3) + (x[,"n_22"] + x[,"n_34"])*log(132 - 5*r_cm2c - 64*r_cm2c^2 + 42*r_cm2c^3)
logL_cm1c <- function(r,n23,n33,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (n23 + n33)*log(3) + (-n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(4*log(2) + 2*log(5)) + (n16 + n40)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n46)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n23 + n33)*log(21 + 4*r - 4*r^2) + (n11 + n45)*(log(pmax(1e-6,r)) + log(4 + r - 2*r^2)) + (n20 + n36)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n13 + n43)*log(19 - 12*r + 12*r^2) + (n22 + n34)*log(42 - 17*r + 35*r^2 - 21*r^3) + (n14 + n42)*log(15 - 12*r + 7*r^2 - 7*r^3) + (n25 + n31)*log(9 - 8*r + 11*r^2 - 6*r^3) + (n26 + n30)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n15 + n41)*log(3 - 5*r^2 + 2*r^3) + (n21 + n35)*log(6 + 4*r - 7*r^2 + 6*r^3) + (n12 + n44)*log(3 + 19*r - 14*r^2 + 7*r^3) + (n24 + n32)*log(39 + 10*r - 28*r^2 + 21*r^3)
return(L)}
interlogL_cm1c <- function(n23,n33,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_cm1c,c(0,0.5), n23,n33,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm1c <- mapply(interlogL_cm1c,x[,"n_23"],x[,"n_33"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_cm1c <- 3*(x[,"n_16"] + x[,"n_40"])*log10(2) - (x[,"n_15"] + x[,"n_41"])*log10(2) - 4*(x[,"n_13"] + x[,"n_43"])*log10(2) - (x[,"n_11"] + x[,"n_45"])*log10(2) + 3*(x[,"n_10"] + x[,"n_46"])*log10(2) + (x[,"n_26"] + x[,"n_30"])*(3*log10(2) - log10(3)) - (x[,"n_20"] + x[,"n_36"])*(-3*log10(2) + log10(3)) - (x[,"n_25"] + x[,"n_31"])*log10(7) - (x[,"n_21"] + x[,"n_35"])*log10(7) - (x[,"n_23"] + x[,"n_33"])*(log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_42"])*(-3*log10(2) + log10(79)) - (x[,"n_12"] + x[,"n_44"])*(-3*log10(2) + log10(79)) - (x[,"n_24"] + x[,"n_32"])*(-3*log10(2) + log10(317)) - (x[,"n_22"] + x[,"n_34"])*(-3*log10(2) + log10(317)) + (x[,"n_16"] + x[,"n_40"])*(2*log10(pmax(1e-6,1 - r_cm1c)) + log10(pmax(1e-6,r_cm1c))) + (x[,"n_10"] + x[,"n_46"])*(log10(pmax(1e-6,1 - r_cm1c)) + 2*log10(pmax(1e-6,r_cm1c))) + (x[,"n_23"] + x[,"n_33"])*log10(21 + 4*r_cm1c - 4*r_cm1c^2) + (x[,"n_11"] + x[,"n_45"])*(log10(pmax(1e-6,r_cm1c)) + log10(4 + r_cm1c - 2*r_cm1c^2)) + (x[,"n_20"] + x[,"n_36"])*(log10(pmax(1e-6,r_cm1c)) + log10(2 - 4*r_cm1c + 3*r_cm1c^2)) + (x[,"n_13"] + x[,"n_43"])*log10(19 - 12*r_cm1c + 12*r_cm1c^2) + (x[,"n_22"] + x[,"n_34"])*log10(42 - 17*r_cm1c + 35*r_cm1c^2 - 21*r_cm1c^3) + (x[,"n_14"] + x[,"n_42"])*log10(15 - 12*r_cm1c + 7*r_cm1c^2 - 7*r_cm1c^3) + (x[,"n_25"] + x[,"n_31"])*log10(9 - 8*r_cm1c + 11*r_cm1c^2 - 6*r_cm1c^3) + (x[,"n_26"] + x[,"n_30"])*log10(1 - 3*r_cm1c + 5*r_cm1c^2 - 3*r_cm1c^3) + (x[,"n_15"] + x[,"n_41"])*log10(3 - 5*r_cm1c^2 + 2*r_cm1c^3) + (x[,"n_21"] + x[,"n_35"])*log10(6 + 4*r_cm1c - 7*r_cm1c^2 + 6*r_cm1c^3) + (x[,"n_12"] + x[,"n_44"])*log10(3 + 19*r_cm1c - 14*r_cm1c^2 + 7*r_cm1c^3) + (x[,"n_24"] + x[,"n_32"])*log10(39 + 10*r_cm1c - 28*r_cm1c^2 + 21*r_cm1c^3)
logL_cm1c <- (x[,"n_23"] + x[,"n_33"])*log(3) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(4*log(2) + 2*log(5)) + (x[,"n_16"] + x[,"n_40"])*(2*log(pmax(1e-6,1 - r_cm1c)) + log(pmax(1e-6,r_cm1c))) + (x[,"n_10"] + x[,"n_46"])*(log(pmax(1e-6,1 - r_cm1c)) + 2*log(pmax(1e-6,r_cm1c))) + (x[,"n_23"] + x[,"n_33"])*log(21 + 4*r_cm1c - 4*r_cm1c^2) + (x[,"n_11"] + x[,"n_45"])*(log(pmax(1e-6,r_cm1c)) + log(4 + r_cm1c - 2*r_cm1c^2)) + (x[,"n_20"] + x[,"n_36"])*(log(pmax(1e-6,r_cm1c)) + log(2 - 4*r_cm1c + 3*r_cm1c^2)) + (x[,"n_13"] + x[,"n_43"])*log(19 - 12*r_cm1c + 12*r_cm1c^2) + (x[,"n_22"] + x[,"n_34"])*log(42 - 17*r_cm1c + 35*r_cm1c^2 - 21*r_cm1c^3) + (x[,"n_14"] + x[,"n_42"])*log(15 - 12*r_cm1c + 7*r_cm1c^2 - 7*r_cm1c^3) + (x[,"n_25"] + x[,"n_31"])*log(9 - 8*r_cm1c + 11*r_cm1c^2 - 6*r_cm1c^3) + (x[,"n_26"] + x[,"n_30"])*log(1 - 3*r_cm1c + 5*r_cm1c^2 - 3*r_cm1c^3) + (x[,"n_15"] + x[,"n_41"])*log(3 - 5*r_cm1c^2 + 2*r_cm1c^3) + (x[,"n_21"] + x[,"n_35"])*log(6 + 4*r_cm1c - 7*r_cm1c^2 + 6*r_cm1c^3) + (x[,"n_12"] + x[,"n_44"])*log(3 + 19*r_cm1c - 14*r_cm1c^2 + 7*r_cm1c^3) + (x[,"n_24"] + x[,"n_32"])*log(39 + 10*r_cm1c - 28*r_cm1c^2 + 21*r_cm1c^3)
logL_rc <- function(r,n23,n33,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (n23 + n33)*log(3) + (-n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(4*log(2) + 2*log(5)) + (n10 + n46)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n16 + n40)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n23 + n33)*log(21 + 4*r - 4*r^2) + (n15 + n41)*(log(pmax(1e-6,r)) + log(4 + r - 2*r^2)) + (n26 + n30)*(log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n13 + n43)*log(19 - 12*r + 12*r^2) + (n24 + n32)*log(42 - 17*r + 35*r^2 - 21*r^3) + (n12 + n44)*log(15 - 12*r + 7*r^2 - 7*r^3) + (n21 + n35)*log(9 - 8*r + 11*r^2 - 6*r^3) + (n20 + n36)*log(1 - 3*r + 5*r^2 - 3*r^3) + (n11 + n45)*log(3 - 5*r^2 + 2*r^3) + (n25 + n31)*log(6 + 4*r - 7*r^2 + 6*r^3) + (n14 + n42)*log(3 + 19*r - 14*r^2 + 7*r^3) + (n22 + n34)*log(39 + 10*r - 28*r^2 + 21*r^3)
return(L)}
interlogL_rc <- function(n23,n33,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_rc,c(0,0.5), n23,n33,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_23"],x[,"n_33"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_rc <- 3*(x[,"n_16"] + x[,"n_40"])*log10(2) - (x[,"n_15"] + x[,"n_41"])*log10(2) - 4*(x[,"n_13"] + x[,"n_43"])*log10(2) - (x[,"n_11"] + x[,"n_45"])*log10(2) + 3*(x[,"n_10"] + x[,"n_46"])*log10(2) + (x[,"n_20"] + x[,"n_36"])*(3*log10(2) - log10(3)) - (x[,"n_26"] + x[,"n_30"])*(-3*log10(2) + log10(3)) - (x[,"n_25"] + x[,"n_31"])*log10(7) - (x[,"n_21"] + x[,"n_35"])*log10(7) - (x[,"n_23"] + x[,"n_33"])*(log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_42"])*(-3*log10(2) + log10(79)) - (x[,"n_12"] + x[,"n_44"])*(-3*log10(2) + log10(79)) - (x[,"n_24"] + x[,"n_32"])*(-3*log10(2) + log10(317)) - (x[,"n_22"] + x[,"n_34"])*(-3*log10(2) + log10(317)) + (x[,"n_10"] + x[,"n_46"])*(2*log10(pmax(1e-6,1 - r_rc)) + log10(pmax(1e-6,r_rc))) + (x[,"n_16"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_rc)) + 2*log10(pmax(1e-6,r_rc))) + (x[,"n_23"] + x[,"n_33"])*log10(21 + 4*r_rc - 4*r_rc^2) + (x[,"n_15"] + x[,"n_41"])*(log10(pmax(1e-6,r_rc)) + log10(4 + r_rc - 2*r_rc^2)) + (x[,"n_26"] + x[,"n_30"])*(log10(pmax(1e-6,r_rc)) + log10(2 - 4*r_rc + 3*r_rc^2)) + (x[,"n_13"] + x[,"n_43"])*log10(19 - 12*r_rc + 12*r_rc^2) + (x[,"n_24"] + x[,"n_32"])*log10(42 - 17*r_rc + 35*r_rc^2 - 21*r_rc^3) + (x[,"n_12"] + x[,"n_44"])*log10(15 - 12*r_rc + 7*r_rc^2 - 7*r_rc^3) + (x[,"n_21"] + x[,"n_35"])*log10(9 - 8*r_rc + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_20"] + x[,"n_36"])*log10(1 - 3*r_rc + 5*r_rc^2 - 3*r_rc^3) + (x[,"n_11"] + x[,"n_45"])*log10(3 - 5*r_rc^2 + 2*r_rc^3) + (x[,"n_25"] + x[,"n_31"])*log10(6 + 4*r_rc - 7*r_rc^2 + 6*r_rc^3) + (x[,"n_14"] + x[,"n_42"])*log10(3 + 19*r_rc - 14*r_rc^2 + 7*r_rc^3) + (x[,"n_22"] + x[,"n_34"])*log10(39 + 10*r_rc - 28*r_rc^2 + 21*r_rc^3)
logL_rc <- (x[,"n_23"] + x[,"n_33"])*log(3) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(4*log(2) + 2*log(5)) + (x[,"n_10"] + x[,"n_46"])*(2*log(pmax(1e-6,1 - r_rc)) + log(pmax(1e-6,r_rc))) + (x[,"n_16"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_rc)) + 2*log(pmax(1e-6,r_rc))) + (x[,"n_23"] + x[,"n_33"])*log(21 + 4*r_rc - 4*r_rc^2) + (x[,"n_15"] + x[,"n_41"])*(log(pmax(1e-6,r_rc)) + log(4 + r_rc - 2*r_rc^2)) + (x[,"n_26"] + x[,"n_30"])*(log(pmax(1e-6,r_rc)) + log(2 - 4*r_rc + 3*r_rc^2)) + (x[,"n_13"] + x[,"n_43"])*log(19 - 12*r_rc + 12*r_rc^2) + (x[,"n_24"] + x[,"n_32"])*log(42 - 17*r_rc + 35*r_rc^2 - 21*r_rc^3) + (x[,"n_12"] + x[,"n_44"])*log(15 - 12*r_rc + 7*r_rc^2 - 7*r_rc^3) + (x[,"n_21"] + x[,"n_35"])*log(9 - 8*r_rc + 11*r_rc^2 - 6*r_rc^3) + (x[,"n_20"] + x[,"n_36"])*log(1 - 3*r_rc + 5*r_rc^2 - 3*r_rc^3) + (x[,"n_11"] + x[,"n_45"])*log(3 - 5*r_rc^2 + 2*r_rc^3) + (x[,"n_25"] + x[,"n_31"])*log(6 + 4*r_rc - 7*r_rc^2 + 6*r_rc^3) + (x[,"n_14"] + x[,"n_42"])*log(3 + 19*r_rc - 14*r_rc^2 + 7*r_rc^3) + (x[,"n_22"] + x[,"n_34"])*log(39 + 10*r_rc - 28*r_rc^2 + 21*r_rc^3)
logL_rm2c <- function(r,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46,n23,n33) {
L <- (-3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 3*n15 - 3*n16 - 4*n20 - 4*n21 - 4*n22 - 4*n24 - 4*n25 - 4*n26 - 4*n30 - 4*n31 - 4*n32 - 4*n34 - 4*n35 - 4*n36 - 3*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 3*n45 - 3*n46)*log(2) + (-n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(log(3) + 2*log(5)) + (n16 + n40)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n46)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n11 + n45)*(log(pmax(1e-6,r)) + log(6 - r - 2*r^2)) + (n23 + n33)*log(12 + r - r^2) + (n20 + n36)*(log(pmax(1e-6,r)) + log(5 - 8*r + 6*r^2)) + (n13 + n43)*log(27 - 8*r + 8*r^2) + (n22 + n34)*log(105 + 7*r + 62*r^2 - 42*r^3) + (n25 + n31)*log(33 - 32*r + 26*r^2 - 12*r^3) + (n14 + n42)*log(21 - 16*r + 11*r^2 - 7*r^3) + (n26 + n30)*log(3 - 7*r + 10*r^2 - 6*r^3) + (n15 + n41)*log(3 + 2*r - 7*r^2 + 2*r^3) + (n12 + n44)*log(9 + 15*r - 10*r^2 + 7*r^3) + (n21 + n35)*log(15 + 16*r - 10*r^2 + 12*r^3) + (n24 + n32)*log(132 - 5*r - 64*r^2 + 42*r^3)
return(L)}
interlogL_rm2c <- function(n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46,n23,n33) {
optimize(logL_rm2c,c(0,0.5), n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n24,n25,n26,n30,n31,n32,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46,n23,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm2c <- mapply(interlogL_rm2c,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"],x[,"n_23"],x[,"n_33"])
LOD_rm2c <- 3*(x[,"n_16"] + x[,"n_40"])*log10(2) + 3*(x[,"n_10"] + x[,"n_46"])*log10(2) - 2*(x[,"n_13"] + x[,"n_43"])*log10(5) - (x[,"n_26"] + x[,"n_30"])*(-2*log10(2) + log10(5)) - (x[,"n_20"] + x[,"n_36"])*(-2*log10(2) + log10(5)) - (x[,"n_15"] + x[,"n_41"])*(-log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_45"])*(-log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_33"])*(-2*log10(2) + 2*log10(7)) - (x[,"n_25"] + x[,"n_31"])*(log10(2) + log10(11)) - (x[,"n_21"] + x[,"n_35"])*(log10(2) + log10(11)) - (x[,"n_14"] + x[,"n_42"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_12"] + x[,"n_44"])*(-3*log10(2) + log10(7) + log10(17)) - (x[,"n_24"] + x[,"n_32"])*(-2*log10(2) + 2*log10(5) + log10(19)) - (x[,"n_22"] + x[,"n_34"])*(-2*log10(2) + 2*log10(5) + log10(19)) + (x[,"n_16"] + x[,"n_40"])*(2*log10(pmax(1e-6,1 - r_rm2c)) + log10(pmax(1e-6,r_rm2c))) + (x[,"n_10"] + x[,"n_46"])*(log10(pmax(1e-6,1 - r_rm2c)) + 2*log10(pmax(1e-6,r_rm2c))) + (x[,"n_11"] + x[,"n_45"])*(log10(pmax(1e-6,r_rm2c)) + log10(6 - r_rm2c - 2*r_rm2c^2)) + (x[,"n_23"] + x[,"n_33"])*log10(12 + r_rm2c - r_rm2c^2) + (x[,"n_20"] + x[,"n_36"])*(log10(pmax(1e-6,r_rm2c)) + log10(5 - 8*r_rm2c + 6*r_rm2c^2)) + (x[,"n_13"] + x[,"n_43"])*log10(27 - 8*r_rm2c + 8*r_rm2c^2) + (x[,"n_22"] + x[,"n_34"])*log10(105 + 7*r_rm2c + 62*r_rm2c^2 - 42*r_rm2c^3) + (x[,"n_25"] + x[,"n_31"])*log10(33 - 32*r_rm2c + 26*r_rm2c^2 - 12*r_rm2c^3) + (x[,"n_14"] + x[,"n_42"])*log10(21 - 16*r_rm2c + 11*r_rm2c^2 - 7*r_rm2c^3) + (x[,"n_26"] + x[,"n_30"])*log10(3 - 7*r_rm2c + 10*r_rm2c^2 - 6*r_rm2c^3) + (x[,"n_15"] + x[,"n_41"])*log10(3 + 2*r_rm2c - 7*r_rm2c^2 + 2*r_rm2c^3) + (x[,"n_12"] + x[,"n_44"])*log10(9 + 15*r_rm2c - 10*r_rm2c^2 + 7*r_rm2c^3) + (x[,"n_21"] + x[,"n_35"])*log10(15 + 16*r_rm2c - 10*r_rm2c^2 + 12*r_rm2c^3) + (x[,"n_24"] + x[,"n_32"])*log10(132 - 5*r_rm2c - 64*r_rm2c^2 + 42*r_rm2c^3)
logL_rm2c <- (-3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 3*x[,"n_16"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_34"] - 4*x[,"n_35"] - 4*x[,"n_36"] - 3*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 3*x[,"n_45"] - 3*x[,"n_46"])*log(2) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(log(3) + 2*log(5)) + (x[,"n_16"] + x[,"n_40"])*(2*log(pmax(1e-6,1 - r_rm2c)) + log(pmax(1e-6,r_rm2c))) + (x[,"n_10"] + x[,"n_46"])*(log(pmax(1e-6,1 - r_rm2c)) + 2*log(pmax(1e-6,r_rm2c))) + (x[,"n_11"] + x[,"n_45"])*(log(pmax(1e-6,r_rm2c)) + log(6 - r_rm2c - 2*r_rm2c^2)) + (x[,"n_23"] + x[,"n_33"])*log(12 + r_rm2c - r_rm2c^2) + (x[,"n_20"] + x[,"n_36"])*(log(pmax(1e-6,r_rm2c)) + log(5 - 8*r_rm2c + 6*r_rm2c^2)) + (x[,"n_13"] + x[,"n_43"])*log(27 - 8*r_rm2c + 8*r_rm2c^2) + (x[,"n_22"] + x[,"n_34"])*log(105 + 7*r_rm2c + 62*r_rm2c^2 - 42*r_rm2c^3) + (x[,"n_25"] + x[,"n_31"])*log(33 - 32*r_rm2c + 26*r_rm2c^2 - 12*r_rm2c^3) + (x[,"n_14"] + x[,"n_42"])*log(21 - 16*r_rm2c + 11*r_rm2c^2 - 7*r_rm2c^3) + (x[,"n_26"] + x[,"n_30"])*log(3 - 7*r_rm2c + 10*r_rm2c^2 - 6*r_rm2c^3) + (x[,"n_15"] + x[,"n_41"])*log(3 + 2*r_rm2c - 7*r_rm2c^2 + 2*r_rm2c^3) + (x[,"n_12"] + x[,"n_44"])*log(9 + 15*r_rm2c - 10*r_rm2c^2 + 7*r_rm2c^3) + (x[,"n_21"] + x[,"n_35"])*log(15 + 16*r_rm2c - 10*r_rm2c^2 + 12*r_rm2c^3) + (x[,"n_24"] + x[,"n_32"])*log(132 - 5*r_rm2c - 64*r_rm2c^2 + 42*r_rm2c^3)
logL_rm1c <- function(r,n20,n26,n30,n36,n10,n11,n12,n13,n14,n15,n16,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n46) {
L <- (n20 + n26 + n30 + n36)*log(3) + (-n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*(4*log(2) + 2*log(5)) + 3*(n16 + n40)*log(pmax(1e-6,1 - r)) + (n15 + n41)*(log(9 - 2*r) + 2*log(pmax(1e-6,1 - r))) + 3*(n10 + n46)*log(pmax(1e-6,r)) + (n26 + n30)*(2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n36)*(log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n11 + n45)*(2*log(pmax(1e-6,r)) + log(7 + 2*r)) + (n13 + n43)*log(9 + 28*r - 28*r^2) + (n12 + n44)*(log(pmax(1e-6,r)) + log(15 + 13*r - 7*r^2)) + (n21 + n35)*(log(pmax(1e-6,r)) + log(16 - r - 6*r^2)) + (n23 + n33)*log(73 - 28*r + 28*r^2) + (n24 + n32)*log(57 - 44*r + 29*r^2 - 21*r^3) + (n25 + n31)*log(9 + 4*r - 19*r^2 + 6*r^3) + (n14 + n42)*log(21 - 20*r - 8*r^2 + 7*r^3) + (n22 + n34)*log(21 + 49*r - 34*r^2 + 21*r^3)
return(L)}
interlogL_rm1c <- function(n20,n26,n30,n36,n10,n11,n12,n13,n14,n15,n16,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_rm1c,c(0,0.5), n20,n26,n30,n36,n10,n11,n12,n13,n14,n15,n16,n21,n22,n23,n24,n25,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm1c <- mapply(interlogL_rm1c,x[,"n_20"],x[,"n_26"],x[,"n_30"],x[,"n_36"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_rm1c <- 3*(x[,"n_26"] + x[,"n_30"])*log10(2) + 3*(x[,"n_20"] + x[,"n_36"])*log10(2) + 3*(x[,"n_16"] + x[,"n_40"])*log10(2) - (x[,"n_15"] + x[,"n_41"])*log10(2) - 4*(x[,"n_13"] + x[,"n_43"])*log10(2) - (x[,"n_11"] + x[,"n_45"])*log10(2) + 3*(x[,"n_10"] + x[,"n_46"])*log10(2) - (x[,"n_25"] + x[,"n_31"])*log10(7) - (x[,"n_21"] + x[,"n_35"])*log10(7) - (x[,"n_23"] + x[,"n_33"])*(log10(2) + log10(3) + log10(11)) - (x[,"n_14"] + x[,"n_42"])*(-3*log10(2) + log10(79)) - (x[,"n_12"] + x[,"n_44"])*(-3*log10(2) + log10(79)) - (x[,"n_24"] + x[,"n_32"])*(-3*log10(2) + log10(317)) - (x[,"n_22"] + x[,"n_34"])*(-3*log10(2) + log10(317)) + 3*(x[,"n_16"] + x[,"n_40"])*log10(pmax(1e-6,1 - r_rm1c)) + (x[,"n_15"] + x[,"n_41"])*(log10(9 - 2*r_rm1c) + 2*log10(pmax(1e-6,1 - r_rm1c))) + 3*(x[,"n_10"] + x[,"n_46"])*log10(pmax(1e-6,r_rm1c)) + (x[,"n_26"] + x[,"n_30"])*(2*log10(pmax(1e-6,1 - r_rm1c)) + log10(pmax(1e-6,r_rm1c))) + (x[,"n_20"] + x[,"n_36"])*(log10(pmax(1e-6,1 - r_rm1c)) + 2*log10(pmax(1e-6,r_rm1c))) + (x[,"n_11"] + x[,"n_45"])*(2*log10(pmax(1e-6,r_rm1c)) + log10(7 + 2*r_rm1c)) + (x[,"n_13"] + x[,"n_43"])*log10(9 + 28*r_rm1c - 28*r_rm1c^2) + (x[,"n_12"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm1c)) + log10(15 + 13*r_rm1c - 7*r_rm1c^2)) + (x[,"n_21"] + x[,"n_35"])*(log10(pmax(1e-6,r_rm1c)) + log10(16 - r_rm1c - 6*r_rm1c^2)) + (x[,"n_23"] + x[,"n_33"])*log10(73 - 28*r_rm1c + 28*r_rm1c^2) + (x[,"n_24"] + x[,"n_32"])*log10(57 - 44*r_rm1c + 29*r_rm1c^2 - 21*r_rm1c^3) + (x[,"n_25"] + x[,"n_31"])*log10(9 + 4*r_rm1c - 19*r_rm1c^2 + 6*r_rm1c^3) + (x[,"n_14"] + x[,"n_42"])*log10(21 - 20*r_rm1c - 8*r_rm1c^2 + 7*r_rm1c^3) + (x[,"n_22"] + x[,"n_34"])*log10(21 + 49*r_rm1c - 34*r_rm1c^2 + 21*r_rm1c^3)
logL_rm1c <- (x[,"n_20"] + x[,"n_26"] + x[,"n_30"] + x[,"n_36"])*log(3) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*(4*log(2) + 2*log(5)) + 3*(x[,"n_16"] + x[,"n_40"])*log(pmax(1e-6,1 - r_rm1c)) + (x[,"n_15"] + x[,"n_41"])*(log(9 - 2*r_rm1c) + 2*log(pmax(1e-6,1 - r_rm1c))) + 3*(x[,"n_10"] + x[,"n_46"])*log(pmax(1e-6,r_rm1c)) + (x[,"n_26"] + x[,"n_30"])*(2*log(pmax(1e-6,1 - r_rm1c)) + log(pmax(1e-6,r_rm1c))) + (x[,"n_20"] + x[,"n_36"])*(log(pmax(1e-6,1 - r_rm1c)) + 2*log(pmax(1e-6,r_rm1c))) + (x[,"n_11"] + x[,"n_45"])*(2*log(pmax(1e-6,r_rm1c)) + log(7 + 2*r_rm1c)) + (x[,"n_13"] + x[,"n_43"])*log(9 + 28*r_rm1c - 28*r_rm1c^2) + (x[,"n_12"] + x[,"n_44"])*(log(pmax(1e-6,r_rm1c)) + log(15 + 13*r_rm1c - 7*r_rm1c^2)) + (x[,"n_21"] + x[,"n_35"])*(log(pmax(1e-6,r_rm1c)) + log(16 - r_rm1c - 6*r_rm1c^2)) + (x[,"n_23"] + x[,"n_33"])*log(73 - 28*r_rm1c + 28*r_rm1c^2) + (x[,"n_24"] + x[,"n_32"])*log(57 - 44*r_rm1c + 29*r_rm1c^2 - 21*r_rm1c^3) + (x[,"n_25"] + x[,"n_31"])*log(9 + 4*r_rm1c - 19*r_rm1c^2 + 6*r_rm1c^3) + (x[,"n_14"] + x[,"n_42"])*log(21 - 20*r_rm1c - 8*r_rm1c^2 + 7*r_rm1c^3) + (x[,"n_22"] + x[,"n_34"])*log(21 + 49*r_rm1c - 34*r_rm1c^2 + 21*r_rm1c^3)

return(list(
r_mat = cbind(r_cc,r_cm2c,r_cm1c,r_rc,r_rm2c,r_rm1c,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm2c,LOD_cm1c,LOD_rc,LOD_rm2c,LOD_rm1c,0),
logL_mat = cbind(logL_cc,logL_cm2c,logL_cm1c,logL_rc,logL_rm2c,logL_rm1c,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed2c","coupling mixed1c","repulsion coupling","repulsion mixed2c","repulsion mixed1c","unknown")
)
)
}

#' @rdname r6_functions
r6_1.4_4.1 <- function(x){
logL_cc <- function(r,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*(4*log(2) + 2*log(5)) + 2*(n11 + n44)*log(3 - 2*r) + 2*(n14 + n41)*log(1 + 2*r) + (n12 + n21 + n34 + n43)*log(21 - 8*r - 4*r^2) + (n13 + n24 + n31 + n42)*log(9 + 16*r - 4*r^2) + (n22 + n33)*log(73 - 20*r + 4*r^2) + (n23 + n32)*log(57 + 12*r + 4*r^2)
return(L)}
interlogL_cc <- function(n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_cc,c(0,0.5), n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cc <- -6*(x[,"n_23"] + x[,"n_32"])*log10(2) - 6*(x[,"n_22"] + x[,"n_33"])*log10(2) - 2*(x[,"n_14"] + x[,"n_41"])*log10(2) - 4*(x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log10(2) - 4*(x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log10(2) - 2*(x[,"n_11"] + x[,"n_44"])*log10(2) + 2*(x[,"n_11"] + x[,"n_44"])*log10(3 - 2*r_cc) + 2*(x[,"n_14"] + x[,"n_41"])*log10(1 + 2*r_cc) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log10(21 - 8*r_cc - 4*r_cc^2) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log10(9 + 16*r_cc - 4*r_cc^2) + (x[,"n_22"] + x[,"n_33"])*log10(73 - 20*r_cc + 4*r_cc^2) + (x[,"n_23"] + x[,"n_32"])*log10(57 + 12*r_cc + 4*r_cc^2)
logL_cc <- (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*(4*log(2) + 2*log(5)) + 2*(x[,"n_11"] + x[,"n_44"])*log(3 - 2*r_cc) + 2*(x[,"n_14"] + x[,"n_41"])*log(1 + 2*r_cc) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log(21 - 8*r_cc - 4*r_cc^2) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log(9 + 16*r_cc - 4*r_cc^2) + (x[,"n_22"] + x[,"n_33"])*log(73 - 20*r_cc + 4*r_cc^2) + (x[,"n_23"] + x[,"n_32"])*log(57 + 12*r_cc + 4*r_cc^2)
logL_cr <- function(r,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-2*n11 - 2*n12 - 2*n13 - 2*n14 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n41 - 2*n42 - 2*n43 - 2*n44)*log(2) + 2*(-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*log(5) + (n11 + n44)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n14 + n41)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n23 + n32)*log(33 - 4*r^2) + (n22 + n33)*log(29 + 8*r - 4*r^2) + (n13 + n42)*log(6 - 5*r + 2*r^2) + (n12 + n43)*log(3 + r + 2*r^2) + (n21 + n24 + n31 + n34)*log(9 - 4*r + 4*r^2)
return(L)}
interlogL_cr <- function(n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_cr,c(0,0.5), n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_cr <- -5*(x[,"n_23"] + x[,"n_32"])*log10(2) - 5*(x[,"n_22"] + x[,"n_33"])*log10(2) - 3*(x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"])*log10(2) - 2*(x[,"n_13"] + x[,"n_42"])*log10(2) - 2*(x[,"n_12"] + x[,"n_43"])*log10(2) + (x[,"n_11"] + x[,"n_44"])*(log10(3 - 2*r_cr) + log10(pmax(1e-6,r_cr))) + (x[,"n_14"] + x[,"n_41"])*(log10(pmax(1e-6,1 - r_cr)) + log10(1 + 2*r_cr)) + (x[,"n_23"] + x[,"n_32"])*log10(33 - 4*r_cr^2) + (x[,"n_22"] + x[,"n_33"])*log10(29 + 8*r_cr - 4*r_cr^2) + (x[,"n_13"] + x[,"n_42"])*log10(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_12"] + x[,"n_43"])*log10(3 + r_cr + 2*r_cr^2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"])*log10(9 - 4*r_cr + 4*r_cr^2)
logL_cr <- (-2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_41"] - 2*x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"])*log(2) + 2*(-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_11"] + x[,"n_44"])*(log(3 - 2*r_cr) + log(pmax(1e-6,r_cr))) + (x[,"n_14"] + x[,"n_41"])*(log(pmax(1e-6,1 - r_cr)) + log(1 + 2*r_cr)) + (x[,"n_23"] + x[,"n_32"])*log(33 - 4*r_cr^2) + (x[,"n_22"] + x[,"n_33"])*log(29 + 8*r_cr - 4*r_cr^2) + (x[,"n_13"] + x[,"n_42"])*log(6 - 5*r_cr + 2*r_cr^2) + (x[,"n_12"] + x[,"n_43"])*log(3 + r_cr + 2*r_cr^2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"])*log(9 - 4*r_cr + 4*r_cr^2)
logL_rc <- function(r,n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
L <- (-2*n11 - 3*n12 - 3*n13 - 2*n14 - 2*n21 - 3*n22 - 3*n23 - 2*n24 - 2*n31 - 3*n32 - 3*n33 - 2*n34 - 2*n41 - 3*n42 - 3*n43 - 2*n44)*log(2) + 2*(-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*log(5) + (n11 + n44)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n14 + n41)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n23 + n32)*log(33 - 4*r^2) + (n22 + n33)*log(29 + 8*r - 4*r^2) + (n24 + n31)*log(6 - 5*r + 2*r^2) + (n21 + n34)*log(3 + r + 2*r^2) + (n12 + n13 + n42 + n43)*log(9 - 4*r + 4*r^2)
return(L)}
interlogL_rc <- function(n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44) {
optimize(logL_rc,c(0,0.5), n11,n12,n13,n14,n21,n22,n23,n24,n31,n32,n33,n34,n41,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rc <- -2*(x[,"n_24"] + x[,"n_31"])*log10(2) - 5*(x[,"n_23"] + x[,"n_32"])*log10(2) - 5*(x[,"n_22"] + x[,"n_33"])*log10(2) - 2*(x[,"n_21"] + x[,"n_34"])*log10(2) - 3*(x[,"n_12"] + x[,"n_13"] + x[,"n_42"] + x[,"n_43"])*log10(2) + (x[,"n_11"] + x[,"n_44"])*(log10(3 - 2*r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_14"] + x[,"n_41"])*(log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_23"] + x[,"n_32"])*log10(33 - 4*r_rc^2) + (x[,"n_22"] + x[,"n_33"])*log10(29 + 8*r_rc - 4*r_rc^2) + (x[,"n_24"] + x[,"n_31"])*log10(6 - 5*r_rc + 2*r_rc^2) + (x[,"n_21"] + x[,"n_34"])*log10(3 + r_rc + 2*r_rc^2) + (x[,"n_12"] + x[,"n_13"] + x[,"n_42"] + x[,"n_43"])*log10(9 - 4*r_rc + 4*r_rc^2)
logL_rc <- (-2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + 2*(-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_11"] + x[,"n_44"])*(log(3 - 2*r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_14"] + x[,"n_41"])*(log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_23"] + x[,"n_32"])*log(33 - 4*r_rc^2) + (x[,"n_22"] + x[,"n_33"])*log(29 + 8*r_rc - 4*r_rc^2) + (x[,"n_24"] + x[,"n_31"])*log(6 - 5*r_rc + 2*r_rc^2) + (x[,"n_21"] + x[,"n_34"])*log(3 + r_rc + 2*r_rc^2) + (x[,"n_12"] + x[,"n_13"] + x[,"n_42"] + x[,"n_43"])*log(9 - 4*r_rc + 4*r_rc^2)
logL_rr <- function(r,n12,n13,n21,n22,n23,n24,n31,n32,n33,n34,n42,n43,n11,n14,n41,n44) {
L <- (-n12 - n13 - n21 - 2*n22 - 2*n23 - n24 - n31 - 2*n32 - 2*n33 - n34 - n42 - n43)*log(2) + 2*(-n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24 - n31 - n32 - n33 - n34 - n41 - n42 - n43 - n44)*log(5) + 2*(n14 + n41)*log(pmax(1e-6,1 - r)) + 2*(n11 + n44)*log(pmax(1e-6,r)) + (n12 + n21 + n34 + n43)*(log(5 - 2*r) + log(pmax(1e-6,r))) + (n13 + n24 + n31 + n42)*log(3 - r - 2*r^2) + (n23 + n32)*log(21 - 12*r + 4*r^2) + (n22 + n33)*log(13 + 4*r + 4*r^2)
return(L)}
interlogL_rr <- function(n12,n13,n21,n22,n23,n24,n31,n32,n33,n34,n42,n43,n11,n14,n41,n44) {
optimize(logL_rr,c(0,0.5), n12,n13,n21,n22,n23,n24,n31,n32,n33,n34,n42,n43,n11,n14,n41,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rr <- mapply(interlogL_rr,x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_11"],x[,"n_14"],x[,"n_41"],x[,"n_44"])
LOD_rr <- -4*(x[,"n_23"] + x[,"n_32"])*log10(2) - 4*(x[,"n_22"] + x[,"n_33"])*log10(2) + 2*(x[,"n_14"] + x[,"n_41"])*log10(2) - (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log10(2) - (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*log10(2) + 2*(x[,"n_11"] + x[,"n_44"])*log10(2) + 2*(x[,"n_14"] + x[,"n_41"])*log10(pmax(1e-6,1 - r_rr)) + 2*(x[,"n_11"] + x[,"n_44"])*log10(pmax(1e-6,r_rr)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*(log10(5 - 2*r_rr) + log10(pmax(1e-6,r_rr))) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log10(3 - r_rr - 2*r_rr^2) + (x[,"n_23"] + x[,"n_32"])*log10(21 - 12*r_rr + 4*r_rr^2) + (x[,"n_22"] + x[,"n_33"])*log10(13 + 4*r_rr + 4*r_rr^2)
logL_rr <- (-x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"])*log(2) + 2*(-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + 2*(x[,"n_14"] + x[,"n_41"])*log(pmax(1e-6,1 - r_rr)) + 2*(x[,"n_11"] + x[,"n_44"])*log(pmax(1e-6,r_rr)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_34"] + x[,"n_43"])*(log(5 - 2*r_rr) + log(pmax(1e-6,r_rr))) + (x[,"n_13"] + x[,"n_24"] + x[,"n_31"] + x[,"n_42"])*log(3 - r_rr - 2*r_rr^2) + (x[,"n_23"] + x[,"n_32"])*log(21 - 12*r_rr + 4*r_rr^2) + (x[,"n_22"] + x[,"n_33"])*log(13 + 4*r_rr + 4*r_rr^2)

return(list(
r_mat = cbind(r_cc,r_cr,r_rc,r_rr,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr,LOD_rc,LOD_rr,0),
logL_mat = cbind(logL_cc,logL_cr,logL_rc,logL_rr,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","repulsion coupling","repulsion repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_1.5_1.5 <- function(x){
logL_cc <- function(r,n22,n23,n24,n32,n34,n42,n43,n44,n33) {
L <- (-2*n22 - n23 - 2*n24 - n32 - n34 - 2*n42 - n43 - 2*n44)*log(2) + 2*(n22 + n44)*log(pmax(1e-6,1 - r)) + 2*(n24 + n42)*log(pmax(1e-6,r)) + (n23 + n32 + n34 + n43)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n33*log(1/2 - r + r^2)
return(L)}
interlogL_cc <- function(n22,n23,n24,n32,n34,n42,n43,n44,n33) {
optimize(logL_cc,c(0,0.5), n22,n23,n24,n32,n34,n42,n43,n44,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_33"])
LOD_cc <- 2*x[,"n_33"]*log10(2) + 2*(x[,"n_24"] + x[,"n_42"])*log10(2) + 2*(x[,"n_23"] + x[,"n_32"] + x[,"n_34"] + x[,"n_43"])*log10(2) + 2*(x[,"n_22"] + x[,"n_44"])*log10(2) + 2*(x[,"n_22"] + x[,"n_44"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_24"] + x[,"n_42"])*log10(pmax(1e-6,r_cc)) + (x[,"n_23"] + x[,"n_32"] + x[,"n_34"] + x[,"n_43"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + x[,"n_33"]*log10(1/2 - r_cc + r_cc^2)
logL_cc <- (-2*x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - x[,"n_32"] - x[,"n_34"] - 2*x[,"n_42"] - x[,"n_43"] - 2*x[,"n_44"])*log(2) + 2*(x[,"n_22"] + x[,"n_44"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_24"] + x[,"n_42"])*log(pmax(1e-6,r_cc)) + (x[,"n_23"] + x[,"n_32"] + x[,"n_34"] + x[,"n_43"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + x[,"n_33"]*log(1/2 - r_cc + r_cc^2)
logL_cm <- function(r,n22,n23,n24,n32,n34,n42,n43,n44,n33) {
L <- 2*(-n22 - n23 - n24 - n32 - n34 - n42 - n43 - n44)*log(2) + (-n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*log(5) + (n24 + n42)*(log(3 - r) + log(pmax(1e-6,r))) + (n22 + n44)*(log(pmax(1e-6,1 - r)) + log(2 + r)) + n33*log(1 + r - r^2) + (n23 + n32 + n34 + n43)*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_cm <- function(n22,n23,n24,n32,n34,n42,n43,n44,n33) {
optimize(logL_cm,c(0,0.5), n22,n23,n24,n32,n34,n42,n43,n44,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_33"])
LOD_cm <- -(x[,"n_33"]*(-2*log10(2) + log10(5))) - (x[,"n_24"] + x[,"n_42"])*(-2*log10(2) + log10(5)) - (x[,"n_22"] + x[,"n_44"])*(-2*log10(2) + log10(5)) - (x[,"n_23"] + x[,"n_32"] + x[,"n_34"] + x[,"n_43"])*(-log10(2) + log10(5)) + (x[,"n_24"] + x[,"n_42"])*(log10(3 - r_cm) + log10(pmax(1e-6,r_cm))) + (x[,"n_22"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cm)) + log10(2 + r_cm)) + x[,"n_33"]*log10(1 + r_cm - r_cm^2) + (x[,"n_23"] + x[,"n_32"] + x[,"n_34"] + x[,"n_43"])*log10(3 - 2*r_cm + 2*r_cm^2)
logL_cm <- 2*(-x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(2) + (-x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_24"] + x[,"n_42"])*(log(3 - r_cm) + log(pmax(1e-6,r_cm))) + (x[,"n_22"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cm)) + log(2 + r_cm)) + x[,"n_33"]*log(1 + r_cm - r_cm^2) + (x[,"n_23"] + x[,"n_32"] + x[,"n_34"] + x[,"n_43"])*log(3 - 2*r_cm + 2*r_cm^2)
logL_rm <- function(r,n22,n23,n24,n32,n33,n34,n42,n43,n44) {
L <- (-2*n22 - n23 - 2*n24 - n32 - n33 - n34 - 2*n42 - n43 - 2*n44)*log(2) + 2*(-n22 - n23 - n24 - n32 - n33 - n34 - n42 - n43 - n44)*log(5) + 2*(n24 + n42)*log(3 - r) + 2*(n22 + n44)*log(2 + r) + (n23 + n32 + n34 + n43)*(log(3 - r) + log(2 + r)) + n33*log(13 - 2*r + 2*r^2)
return(L)}
interlogL_rm <- function(n22,n23,n24,n32,n33,n34,n42,n43,n44) {
optimize(logL_rm,c(0,0.5), n22,n23,n24,n32,n33,n34,n42,n43,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_42"],x[,"n_43"],x[,"n_44"])
LOD_rm <- -2*(x[,"n_24"] + x[,"n_42"])*(-log10(2) + log10(5)) - 2*(x[,"n_23"] + x[,"n_32"] + x[,"n_34"] + x[,"n_43"])*(-log10(2) + log10(5)) - 2*(x[,"n_22"] + x[,"n_44"])*(-log10(2) + log10(5)) - x[,"n_33"]*(-log10(2) + 2*log10(5)) + 2*(x[,"n_24"] + x[,"n_42"])*log10(3 - r_rm) + 2*(x[,"n_22"] + x[,"n_44"])*log10(2 + r_rm) + (x[,"n_23"] + x[,"n_32"] + x[,"n_34"] + x[,"n_43"])*(log10(3 - r_rm) + log10(2 + r_rm)) + x[,"n_33"]*log10(13 - 2*r_rm + 2*r_rm^2)
logL_rm <- (-2*x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - 2*x[,"n_42"] - x[,"n_43"] - 2*x[,"n_44"])*log(2) + 2*(-x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + 2*(x[,"n_24"] + x[,"n_42"])*log(3 - r_rm) + 2*(x[,"n_22"] + x[,"n_44"])*log(2 + r_rm) + (x[,"n_23"] + x[,"n_32"] + x[,"n_34"] + x[,"n_43"])*(log(3 - r_rm) + log(2 + r_rm)) + x[,"n_33"]*log(13 - 2*r_rm + 2*r_rm^2)

return(list(
r_mat = cbind(r_cc,r_cm,r_rm,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_rm,0),
logL_mat = cbind(logL_cc,logL_cm,logL_rm,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","repulsion mixed","unknown")
)
)
}

#' @rdname r6_functions
r6_1.5_2.2 <- function(x){
logL_cc <- function(r,n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n31,n33) {
L <- (-2*n20 - 3*n21 - n22 - 3*n23 - 2*n24 - 2*n30 + n32 - 2*n34 - 2*n40 - 3*n41 - n42 - 3*n43 - 2*n44)*log(2) + (n21 + n23 + n31 + n33 + n41 + n43)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n21 + n43)*log(7 - 6*r) + (n20 + n44)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n24 + n40)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n23 + n41)*log(1 + 6*r) + (n30 + n34)*log(1 + 4*r - 4*r^2) + (n22 + n42)*log(5 + 2*r - 2*r^2) + n32*log(3 - r + r^2)
return(L)}
interlogL_cc <- function(n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n31,n33) {
optimize(logL_cc,c(0,0.5), n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n31,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_32"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_31"],x[,"n_33"])
LOD_cc <- -((x[,"n_30"] + x[,"n_34"])*log10(2)) - 2*(x[,"n_23"] + x[,"n_41"])*log10(2) - 2*(x[,"n_21"] + x[,"n_43"])*log10(2) - x[,"n_32"]*(-2*log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_42"])*(-log10(2) + log10(11)) + (x[,"n_21"] + x[,"n_43"])*log10(7 - 6*r_cc) + (x[,"n_20"] + x[,"n_44"])*(log10(3 - 2*r_cc) + log10(pmax(1e-6,1 - r_cc))) + (x[,"n_24"] + x[,"n_40"])*(log10(pmax(1e-6,r_cc)) + log10(1 + 2*r_cc)) + (x[,"n_23"] + x[,"n_41"])*log10(1 + 6*r_cc) + (x[,"n_30"] + x[,"n_34"])*log10(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_22"] + x[,"n_42"])*log10(5 + 2*r_cc - 2*r_cc^2) + x[,"n_32"]*log10(3 - r_cc + r_cc^2)
logL_cc <- (-2*x[,"n_20"] - 3*x[,"n_21"] - x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] + x[,"n_32"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 3*x[,"n_41"] - x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + (x[,"n_21"] + x[,"n_23"] + x[,"n_31"] + x[,"n_33"] + x[,"n_41"] + x[,"n_43"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_21"] + x[,"n_43"])*log(7 - 6*r_cc) + (x[,"n_20"] + x[,"n_44"])*(log(3 - 2*r_cc) + log(pmax(1e-6,1 - r_cc))) + (x[,"n_24"] + x[,"n_40"])*(log(pmax(1e-6,r_cc)) + log(1 + 2*r_cc)) + (x[,"n_23"] + x[,"n_41"])*log(1 + 6*r_cc) + (x[,"n_30"] + x[,"n_34"])*log(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_22"] + x[,"n_42"])*log(5 + 2*r_cc - 2*r_cc^2) + x[,"n_32"]*log(3 - r_cc + r_cc^2)
logL_cm <- function(r,n21,n22,n23,n32,n41,n42,n43,n31,n33,n20,n24,n30,n34,n40,n44) {
L <- (-n21 - 2*n22 - n23 - n32 - n41 - 2*n42 - n43)*log(2) + (n21 + n23 + n31 + n33 + n41 + n43)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n20 + n24 + n40 + n44)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n32*log(9 + 8*r - 8*r^2) + (n30 + n34)*log(1 - 2*r + 2*r^2) + (n22 + n42)*log(13 - 8*r + 8*r^2)
return(L)}
interlogL_cm <- function(n21,n22,n23,n32,n41,n42,n43,n31,n33,n20,n24,n30,n34,n40,n44) {
optimize(logL_cm,c(0,0.5), n21,n22,n23,n32,n41,n42,n43,n31,n33,n20,n24,n30,n34,n40,n44, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_32"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_31"],x[,"n_33"],x[,"n_20"],x[,"n_24"],x[,"n_30"],x[,"n_34"],x[,"n_40"],x[,"n_44"])
LOD_cm <- (x[,"n_30"] + x[,"n_34"])*log10(2) + 2*(x[,"n_20"] + x[,"n_24"] + x[,"n_40"] + x[,"n_44"])*log10(2) - x[,"n_32"]*log10(11) - (x[,"n_22"] + x[,"n_42"])*log10(11) + (x[,"n_20"] + x[,"n_24"] + x[,"n_40"] + x[,"n_44"])*(log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + x[,"n_32"]*log10(9 + 8*r_cm - 8*r_cm^2) + (x[,"n_30"] + x[,"n_34"])*log10(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_22"] + x[,"n_42"])*log10(13 - 8*r_cm + 8*r_cm^2)
logL_cm <- (-x[,"n_21"] - 2*x[,"n_22"] - x[,"n_23"] - x[,"n_32"] - x[,"n_41"] - 2*x[,"n_42"] - x[,"n_43"])*log(2) + (x[,"n_21"] + x[,"n_23"] + x[,"n_31"] + x[,"n_33"] + x[,"n_41"] + x[,"n_43"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_20"] + x[,"n_24"] + x[,"n_40"] + x[,"n_44"])*(log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + x[,"n_32"]*log(9 + 8*r_cm - 8*r_cm^2) + (x[,"n_30"] + x[,"n_34"])*log(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_22"] + x[,"n_42"])*log(13 - 8*r_cm + 8*r_cm^2)
logL_rc <- function(r,n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n31,n33) {
L <- (-4*n20 - n21 - 3*n22 - n23 - 4*n24 - 3*n30 - 2*n32 - 3*n34 - 4*n40 - n41 - 3*n42 - n43 - 4*n44)*log(2) + (n21 + n23 + n31 + n33 + n41 + n43)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n20 + n24 + n40 + n44)*(log(3 - 2*r) + log(1 + 2*r)) + n32*log(21 + 4*r - 4*r^2) + (n30 + n34)*log(5 - 4*r + 4*r^2) + (n22 + n42)*log(23 - 4*r + 4*r^2)
return(L)}
interlogL_rc <- function(n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n31,n33) {
optimize(logL_rc,c(0,0.5), n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n31,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_32"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_31"],x[,"n_33"])
LOD_rc <- -2*(x[,"n_30"] + x[,"n_34"])*log10(2) - 2*(x[,"n_20"] + x[,"n_24"] + x[,"n_40"] + x[,"n_44"])*log10(2) - x[,"n_32"]*(log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_42"])*(log10(2) + log10(11)) + (x[,"n_20"] + x[,"n_24"] + x[,"n_40"] + x[,"n_44"])*(log10(3 - 2*r_rc) + log10(1 + 2*r_rc)) + x[,"n_32"]*log10(21 + 4*r_rc - 4*r_rc^2) + (x[,"n_30"] + x[,"n_34"])*log10(5 - 4*r_rc + 4*r_rc^2) + (x[,"n_22"] + x[,"n_42"])*log10(23 - 4*r_rc + 4*r_rc^2)
logL_rc <- (-4*x[,"n_20"] - x[,"n_21"] - 3*x[,"n_22"] - x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_30"] - 2*x[,"n_32"] - 3*x[,"n_34"] - 4*x[,"n_40"] - x[,"n_41"] - 3*x[,"n_42"] - x[,"n_43"] - 4*x[,"n_44"])*log(2) + (x[,"n_21"] + x[,"n_23"] + x[,"n_31"] + x[,"n_33"] + x[,"n_41"] + x[,"n_43"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_20"] + x[,"n_24"] + x[,"n_40"] + x[,"n_44"])*(log(3 - 2*r_rc) + log(1 + 2*r_rc)) + x[,"n_32"]*log(21 + 4*r_rc - 4*r_rc^2) + (x[,"n_30"] + x[,"n_34"])*log(5 - 4*r_rc + 4*r_rc^2) + (x[,"n_22"] + x[,"n_42"])*log(23 - 4*r_rc + 4*r_rc^2)
logL_rm <- function(r,n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n31,n33) {
L <- (-2*n20 - 3*n21 - n22 - 3*n23 - 2*n24 - 2*n30 + n32 - 2*n34 - 2*n40 - 3*n41 - n42 - 3*n43 - 2*n44)*log(2) + (n21 + n23 + n31 + n33 + n41 + n43)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(5) + (n23 + n41)*log(7 - 6*r) + (n24 + n40)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n20 + n44)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n21 + n43)*log(1 + 6*r) + (n30 + n34)*log(1 + 4*r - 4*r^2) + (n22 + n42)*log(5 + 2*r - 2*r^2) + n32*log(3 - r + r^2)
return(L)}
interlogL_rm <- function(n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n31,n33) {
optimize(logL_rm,c(0,0.5), n20,n21,n22,n23,n24,n30,n32,n34,n40,n41,n42,n43,n44,n31,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_32"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_31"],x[,"n_33"])
LOD_rm <- -((x[,"n_30"] + x[,"n_34"])*log10(2)) - 2*(x[,"n_23"] + x[,"n_41"])*log10(2) - 2*(x[,"n_21"] + x[,"n_43"])*log10(2) - x[,"n_32"]*(-2*log10(2) + log10(11)) - (x[,"n_22"] + x[,"n_42"])*(-log10(2) + log10(11)) + (x[,"n_23"] + x[,"n_41"])*log10(7 - 6*r_rm) + (x[,"n_24"] + x[,"n_40"])*(log10(3 - 2*r_rm) + log10(pmax(1e-6,1 - r_rm))) + (x[,"n_20"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm)) + log10(1 + 2*r_rm)) + (x[,"n_21"] + x[,"n_43"])*log10(1 + 6*r_rm) + (x[,"n_30"] + x[,"n_34"])*log10(1 + 4*r_rm - 4*r_rm^2) + (x[,"n_22"] + x[,"n_42"])*log10(5 + 2*r_rm - 2*r_rm^2) + x[,"n_32"]*log10(3 - r_rm + r_rm^2)
logL_rm <- (-2*x[,"n_20"] - 3*x[,"n_21"] - x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_30"] + x[,"n_32"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 3*x[,"n_41"] - x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"])*log(2) + (x[,"n_21"] + x[,"n_23"] + x[,"n_31"] + x[,"n_33"] + x[,"n_41"] + x[,"n_43"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(5) + (x[,"n_23"] + x[,"n_41"])*log(7 - 6*r_rm) + (x[,"n_24"] + x[,"n_40"])*(log(3 - 2*r_rm) + log(pmax(1e-6,1 - r_rm))) + (x[,"n_20"] + x[,"n_44"])*(log(pmax(1e-6,r_rm)) + log(1 + 2*r_rm)) + (x[,"n_21"] + x[,"n_43"])*log(1 + 6*r_rm) + (x[,"n_30"] + x[,"n_34"])*log(1 + 4*r_rm - 4*r_rm^2) + (x[,"n_22"] + x[,"n_42"])*log(5 + 2*r_rm - 2*r_rm^2) + x[,"n_32"]*log(3 - r_rm + r_rm^2)

return(list(
r_mat = cbind(r_cc,r_cm,r_rc,r_rm,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_rc,LOD_rm,0),
logL_mat = cbind(logL_cc,logL_cm,logL_rc,logL_rm,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","repulsion coupling","repulsion mixed","unknown")
)
)
}

#' @rdname r6_functions
r6_1.5_2.3 <- function(x){
logL_cc <- function(r,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - n35 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + (n21 + n24 + n31 + n34 + n41 + n44)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + 2*(n20 + n45)*log(pmax(1e-6,1 - r)) + 2*(n25 + n40)*log(pmax(1e-6,r)) + (n30 + n35)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n24 + n41)*(log(pmax(1e-6,r)) + log(3 + 2*r)) + (n22 + n43)*log(24 - 7*r - 8*r^2) + (n23 + n42)*log(9 + 23*r - 8*r^2) + (n31 + n34)*log(3 + 4*r - 4*r^2) + (n21 + n44)*log(5 - 7*r + 2*r^2) + (n32 + n33)*log(41 - 16*r + 16*r^2)
return(L)}
interlogL_cc <- function(n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cc,c(0,0.5), n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cc <- -2*(x[,"n_31"] + x[,"n_34"])*log10(2) + 2*(x[,"n_30"] + x[,"n_35"])*log10(2) + 2*(x[,"n_25"] + x[,"n_40"])*log10(2) - (x[,"n_24"] + x[,"n_41"])*log10(2) - (x[,"n_21"] + x[,"n_44"])*log10(2) + 2*(x[,"n_20"] + x[,"n_45"])*log10(2) - (x[,"n_32"] + x[,"n_33"])*log10(37) - (x[,"n_23"] + x[,"n_42"])*(-log10(2) + log10(37)) - (x[,"n_22"] + x[,"n_43"])*(-log10(2) + log10(37)) + 2*(x[,"n_20"] + x[,"n_45"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_25"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + (x[,"n_30"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_24"] + x[,"n_41"])*(log10(pmax(1e-6,r_cc)) + log10(3 + 2*r_cc)) + (x[,"n_22"] + x[,"n_43"])*log10(24 - 7*r_cc - 8*r_cc^2) + (x[,"n_23"] + x[,"n_42"])*log10(9 + 23*r_cc - 8*r_cc^2) + (x[,"n_31"] + x[,"n_34"])*log10(3 + 4*r_cc - 4*r_cc^2) + (x[,"n_21"] + x[,"n_44"])*log10(5 - 7*r_cc + 2*r_cc^2) + (x[,"n_32"] + x[,"n_33"])*log10(41 - 16*r_cc + 16*r_cc^2)
logL_cc <- (-2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - x[,"n_35"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + 2*(x[,"n_20"] + x[,"n_45"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_25"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + (x[,"n_30"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_24"] + x[,"n_41"])*(log(pmax(1e-6,r_cc)) + log(3 + 2*r_cc)) + (x[,"n_22"] + x[,"n_43"])*log(24 - 7*r_cc - 8*r_cc^2) + (x[,"n_23"] + x[,"n_42"])*log(9 + 23*r_cc - 8*r_cc^2) + (x[,"n_31"] + x[,"n_34"])*log(3 + 4*r_cc - 4*r_cc^2) + (x[,"n_21"] + x[,"n_44"])*log(5 - 7*r_cc + 2*r_cc^2) + (x[,"n_32"] + x[,"n_33"])*log(41 - 16*r_cc + 16*r_cc^2)
logL_cm <- function(r,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + (n21 + n24 + n31 + n34 + n41 + n44)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n20 + n25 + n40 + n45)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n32 + n33)*log(33 + 16*r - 16*r^2) + (n21 + n44)*log(2 + r - 2*r^2) + (n24 + n41)*log(1 + 3*r - 2*r^2) + (n30 + n35)*log(1 - 2*r + 2*r^2) + (n31 + n34)*log(5 - 4*r + 4*r^2) + (n22 + n43)*log(21 - 9*r + 8*r^2) + (n23 + n42)*log(20 - 7*r + 8*r^2)
return(L)}
interlogL_cm <- function(n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cm,c(0,0.5), n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cm <- -2*(x[,"n_31"] + x[,"n_34"])*log10(2) + (x[,"n_30"] + x[,"n_35"])*log10(2) - (x[,"n_24"] + x[,"n_41"])*log10(2) - (x[,"n_21"] + x[,"n_44"])*log10(2) + 2*(x[,"n_20"] + x[,"n_25"] + x[,"n_40"] + x[,"n_45"])*log10(2) - (x[,"n_32"] + x[,"n_33"])*log10(37) - (x[,"n_23"] + x[,"n_42"])*(-log10(2) + log10(37)) - (x[,"n_22"] + x[,"n_43"])*(-log10(2) + log10(37)) + (x[,"n_20"] + x[,"n_25"] + x[,"n_40"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + (x[,"n_32"] + x[,"n_33"])*log10(33 + 16*r_cm - 16*r_cm^2) + (x[,"n_21"] + x[,"n_44"])*log10(2 + r_cm - 2*r_cm^2) + (x[,"n_24"] + x[,"n_41"])*log10(1 + 3*r_cm - 2*r_cm^2) + (x[,"n_30"] + x[,"n_35"])*log10(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_31"] + x[,"n_34"])*log10(5 - 4*r_cm + 4*r_cm^2) + (x[,"n_22"] + x[,"n_43"])*log10(21 - 9*r_cm + 8*r_cm^2) + (x[,"n_23"] + x[,"n_42"])*log10(20 - 7*r_cm + 8*r_cm^2)
logL_cm <- (-2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_20"] + x[,"n_25"] + x[,"n_40"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + (x[,"n_32"] + x[,"n_33"])*log(33 + 16*r_cm - 16*r_cm^2) + (x[,"n_21"] + x[,"n_44"])*log(2 + r_cm - 2*r_cm^2) + (x[,"n_24"] + x[,"n_41"])*log(1 + 3*r_cm - 2*r_cm^2) + (x[,"n_30"] + x[,"n_35"])*log(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_31"] + x[,"n_34"])*log(5 - 4*r_cm + 4*r_cm^2) + (x[,"n_22"] + x[,"n_43"])*log(21 - 9*r_cm + 8*r_cm^2) + (x[,"n_23"] + x[,"n_42"])*log(20 - 7*r_cm + 8*r_cm^2)
logL_rc <- function(r,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-4*n20 - 4*n21 - 3*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n30 - 4*n31 - 3*n32 - 3*n33 - 4*n34 - 4*n35 - 4*n40 - 4*n41 - 3*n42 - 3*n43 - 4*n44 - 4*n45)*log(2) + (n21 + n24 + n31 + n34 + n41 + n44)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n25 + n40)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n20 + n45)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n32 + n33)*log(35 + 8*r - 8*r^2) + (n21 + n44)*log(4 + r - 2*r^2) + (n24 + n41)*log(3 + 3*r - 2*r^2) + (n22 + n43)*log(21 - 7*r + 4*r^2) + (n30 + n35)*log(3 - 4*r + 4*r^2) + (n31 + n34)*log(9 - 4*r + 4*r^2) + (n23 + n42)*log(18 - r + 4*r^2)
return(L)}
interlogL_rc <- function(n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rc,c(0,0.5), n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rc <- -3*(x[,"n_31"] + x[,"n_34"])*log10(2) - (x[,"n_30"] + x[,"n_35"])*log10(2) - 2*(x[,"n_24"] + x[,"n_41"])*log10(2) - 2*(x[,"n_21"] + x[,"n_44"])*log10(2) - (x[,"n_32"] + x[,"n_33"])*log10(37) - (x[,"n_23"] + x[,"n_42"])*(-log10(2) + log10(37)) - (x[,"n_22"] + x[,"n_43"])*(-log10(2) + log10(37)) + (x[,"n_25"] + x[,"n_40"])*(log10(3 - 2*r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_20"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_32"] + x[,"n_33"])*log10(35 + 8*r_rc - 8*r_rc^2) + (x[,"n_21"] + x[,"n_44"])*log10(4 + r_rc - 2*r_rc^2) + (x[,"n_24"] + x[,"n_41"])*log10(3 + 3*r_rc - 2*r_rc^2) + (x[,"n_22"] + x[,"n_43"])*log10(21 - 7*r_rc + 4*r_rc^2) + (x[,"n_30"] + x[,"n_35"])*log10(3 - 4*r_rc + 4*r_rc^2) + (x[,"n_31"] + x[,"n_34"])*log10(9 - 4*r_rc + 4*r_rc^2) + (x[,"n_23"] + x[,"n_42"])*log10(18 - r_rc + 4*r_rc^2)
logL_rc <- (-4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"] - 4*x[,"n_35"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 4*x[,"n_45"])*log(2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_25"] + x[,"n_40"])*(log(3 - 2*r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_20"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_32"] + x[,"n_33"])*log(35 + 8*r_rc - 8*r_rc^2) + (x[,"n_21"] + x[,"n_44"])*log(4 + r_rc - 2*r_rc^2) + (x[,"n_24"] + x[,"n_41"])*log(3 + 3*r_rc - 2*r_rc^2) + (x[,"n_22"] + x[,"n_43"])*log(21 - 7*r_rc + 4*r_rc^2) + (x[,"n_30"] + x[,"n_35"])*log(3 - 4*r_rc + 4*r_rc^2) + (x[,"n_31"] + x[,"n_34"])*log(9 - 4*r_rc + 4*r_rc^2) + (x[,"n_23"] + x[,"n_42"])*log(18 - r_rc + 4*r_rc^2)
logL_rm <- function(r,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-4*n20 - 4*n21 - 3*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n30 - 4*n31 - 3*n32 - 3*n33 - 4*n34 - 4*n35 - 4*n40 - 4*n41 - 3*n42 - 3*n43 - 4*n44 - 4*n45)*log(2) + (n21 + n24 + n31 + n34 + n41 + n44)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n25 + n40)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n20 + n45)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n23 + n42)*log(23 - 7*r - 4*r^2) + (n30 + n35)*log(1 + 4*r - 4*r^2) + (n31 + n34)*log(7 + 4*r - 4*r^2) + (n22 + n43)*log(12 + 15*r - 4*r^2) + (n24 + n41)*log(8 - 9*r + 2*r^2) + (n21 + n44)*log(1 + 5*r + 2*r^2) + (n32 + n33)*log(39 - 8*r + 8*r^2)
return(L)}
interlogL_rm <- function(n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rm,c(0,0.5), n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rm <- -3*(x[,"n_31"] + x[,"n_34"])*log10(2) - (x[,"n_30"] + x[,"n_35"])*log10(2) - 2*(x[,"n_24"] + x[,"n_41"])*log10(2) - 2*(x[,"n_21"] + x[,"n_44"])*log10(2) - (x[,"n_32"] + x[,"n_33"])*log10(37) - (x[,"n_23"] + x[,"n_42"])*(-log10(2) + log10(37)) - (x[,"n_22"] + x[,"n_43"])*(-log10(2) + log10(37)) + (x[,"n_25"] + x[,"n_40"])*(log10(3 - 2*r_rm) + log10(pmax(1e-6,1 - r_rm))) + (x[,"n_20"] + x[,"n_45"])*(log10(pmax(1e-6,r_rm)) + log10(1 + 2*r_rm)) + (x[,"n_23"] + x[,"n_42"])*log10(23 - 7*r_rm - 4*r_rm^2) + (x[,"n_30"] + x[,"n_35"])*log10(1 + 4*r_rm - 4*r_rm^2) + (x[,"n_31"] + x[,"n_34"])*log10(7 + 4*r_rm - 4*r_rm^2) + (x[,"n_22"] + x[,"n_43"])*log10(12 + 15*r_rm - 4*r_rm^2) + (x[,"n_24"] + x[,"n_41"])*log10(8 - 9*r_rm + 2*r_rm^2) + (x[,"n_21"] + x[,"n_44"])*log10(1 + 5*r_rm + 2*r_rm^2) + (x[,"n_32"] + x[,"n_33"])*log10(39 - 8*r_rm + 8*r_rm^2)
logL_rm <- (-4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"] - 4*x[,"n_35"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 4*x[,"n_45"])*log(2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_25"] + x[,"n_40"])*(log(3 - 2*r_rm) + log(pmax(1e-6,1 - r_rm))) + (x[,"n_20"] + x[,"n_45"])*(log(pmax(1e-6,r_rm)) + log(1 + 2*r_rm)) + (x[,"n_23"] + x[,"n_42"])*log(23 - 7*r_rm - 4*r_rm^2) + (x[,"n_30"] + x[,"n_35"])*log(1 + 4*r_rm - 4*r_rm^2) + (x[,"n_31"] + x[,"n_34"])*log(7 + 4*r_rm - 4*r_rm^2) + (x[,"n_22"] + x[,"n_43"])*log(12 + 15*r_rm - 4*r_rm^2) + (x[,"n_24"] + x[,"n_41"])*log(8 - 9*r_rm + 2*r_rm^2) + (x[,"n_21"] + x[,"n_44"])*log(1 + 5*r_rm + 2*r_rm^2) + (x[,"n_32"] + x[,"n_33"])*log(39 - 8*r_rm + 8*r_rm^2)

return(list(
r_mat = cbind(r_cc,r_cm,r_rc,r_rm,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_rc,LOD_rm,0),
logL_mat = cbind(logL_cc,logL_cm,logL_rc,logL_rm,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","repulsion coupling","repulsion mixed","unknown")
)
)
}

#' @rdname r6_functions
r6_1.5_2.4 <- function(x){
logL_cc <- function(r,n23,n31,n33,n35,n43,n22,n24,n32,n34,n42,n44,n21,n25,n41,n45) {
L <- (-2*n23 + n31 - n33 + n35 - 2*n43)*log(2) + (n22 + n24 + n32 + n34 + n42 + n44)*log(3) + 2*(-n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + (n22 + n44)*log(pmax(1e-6,1 - r)) + 2*(n21 + n45)*log(pmax(1e-6,1 - r)) + 2*(n25 + n41)*log(pmax(1e-6,r)) + (n24 + n42)*log(pmax(1e-6,r)) + (n31 + n35)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n23 + n43)*log(9 + 8*r - 8*r^2) + n33*log(13 - 8*r + 8*r^2)
return(L)}
interlogL_cc <- function(n23,n31,n33,n35,n43,n22,n24,n32,n34,n42,n44,n21,n25,n41,n45) {
optimize(logL_cc,c(0,0.5), n23,n31,n33,n35,n43,n22,n24,n32,n34,n42,n44,n21,n25,n41,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_23"],x[,"n_31"],x[,"n_33"],x[,"n_35"],x[,"n_43"],x[,"n_22"],x[,"n_24"],x[,"n_32"],x[,"n_34"],x[,"n_42"],x[,"n_44"],x[,"n_21"],x[,"n_25"],x[,"n_41"],x[,"n_45"])
LOD_cc <- 2*(x[,"n_31"] + x[,"n_35"])*log10(2) + 2*(x[,"n_25"] + x[,"n_41"])*log10(2) + (x[,"n_24"] + x[,"n_42"])*log10(2) + (x[,"n_22"] + x[,"n_44"])*log10(2) + 2*(x[,"n_21"] + x[,"n_45"])*log10(2) - x[,"n_33"]*log10(11) - (x[,"n_23"] + x[,"n_43"])*log10(11) + (x[,"n_22"] + x[,"n_44"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_21"] + x[,"n_45"])*log10(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_25"] + x[,"n_41"])*log10(pmax(1e-6,r_cc)) + (x[,"n_24"] + x[,"n_42"])*log10(pmax(1e-6,r_cc)) + (x[,"n_31"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_23"] + x[,"n_43"])*log10(9 + 8*r_cc - 8*r_cc^2) + x[,"n_33"]*log10(13 - 8*r_cc + 8*r_cc^2)
logL_cc <- (-2*x[,"n_23"] + x[,"n_31"] - x[,"n_33"] + x[,"n_35"] - 2*x[,"n_43"])*log(2) + (x[,"n_22"] + x[,"n_24"] + x[,"n_32"] + x[,"n_34"] + x[,"n_42"] + x[,"n_44"])*log(3) + 2*(-x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_22"] + x[,"n_44"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_21"] + x[,"n_45"])*log(pmax(1e-6,1 - r_cc)) + 2*(x[,"n_25"] + x[,"n_41"])*log(pmax(1e-6,r_cc)) + (x[,"n_24"] + x[,"n_42"])*log(pmax(1e-6,r_cc)) + (x[,"n_31"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_23"] + x[,"n_43"])*log(9 + 8*r_cc - 8*r_cc^2) + x[,"n_33"]*log(13 - 8*r_cc + 8*r_cc^2)
logL_cm <- function(r,n21,n22,n24,n25,n31,n35,n41,n42,n44,n45,n32,n34,n23,n33,n43) {
L <- (-2*n21 - 3*n22 - 3*n24 - 2*n25 - 2*n31 - 2*n35 - 2*n41 - 3*n42 - 3*n44 - 2*n45)*log(2) + (n22 + n24 + n32 + n34 + n42 + n44)*log(3) + 2*(-n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + (n22 + n44)*log(5 - 2*r) + (n25 + n41)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n21 + n45)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n24 + n42)*log(3 + 2*r) + n33*log(5 + 2*r - 2*r^2) + (n23 + n43)*log(3 - r + r^2) + (n31 + n35)*log(3 - 4*r + 4*r^2)
return(L)}
interlogL_cm <- function(n21,n22,n24,n25,n31,n35,n41,n42,n44,n45,n32,n34,n23,n33,n43) {
optimize(logL_cm,c(0,0.5), n21,n22,n24,n25,n31,n35,n41,n42,n44,n45,n32,n34,n23,n33,n43, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_21"],x[,"n_22"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_44"],x[,"n_45"],x[,"n_32"],x[,"n_34"],x[,"n_23"],x[,"n_33"],x[,"n_43"])
LOD_cm <- -((x[,"n_31"] + x[,"n_35"])*log10(2)) - 2*(x[,"n_24"] + x[,"n_42"])*log10(2) - 2*(x[,"n_22"] + x[,"n_44"])*log10(2) - (x[,"n_23"] + x[,"n_43"])*(-2*log10(2) + log10(11)) - x[,"n_33"]*(-log10(2) + log10(11)) + (x[,"n_22"] + x[,"n_44"])*log10(5 - 2*r_cm) + (x[,"n_25"] + x[,"n_41"])*(log10(3 - 2*r_cm) + log10(pmax(1e-6,r_cm))) + (x[,"n_21"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cm)) + log10(1 + 2*r_cm)) + (x[,"n_24"] + x[,"n_42"])*log10(3 + 2*r_cm) + x[,"n_33"]*log10(5 + 2*r_cm - 2*r_cm^2) + (x[,"n_23"] + x[,"n_43"])*log10(3 - r_cm + r_cm^2) + (x[,"n_31"] + x[,"n_35"])*log10(3 - 4*r_cm + 4*r_cm^2)
logL_cm <- (-2*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_31"] - 2*x[,"n_35"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_22"] + x[,"n_24"] + x[,"n_32"] + x[,"n_34"] + x[,"n_42"] + x[,"n_44"])*log(3) + 2*(-x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_22"] + x[,"n_44"])*log(5 - 2*r_cm) + (x[,"n_25"] + x[,"n_41"])*(log(3 - 2*r_cm) + log(pmax(1e-6,r_cm))) + (x[,"n_21"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cm)) + log(1 + 2*r_cm)) + (x[,"n_24"] + x[,"n_42"])*log(3 + 2*r_cm) + x[,"n_33"]*log(5 + 2*r_cm - 2*r_cm^2) + (x[,"n_23"] + x[,"n_43"])*log(3 - r_cm + r_cm^2) + (x[,"n_31"] + x[,"n_35"])*log(3 - 4*r_cm + 4*r_cm^2)
logL_rm <- function(r,n21,n22,n23,n24,n25,n31,n33,n35,n41,n42,n43,n44,n45,n32,n34) {
L <- (-4*n21 - 2*n22 - 3*n23 - 2*n24 - 4*n25 - 3*n31 - 2*n33 - 3*n35 - 4*n41 - 2*n42 - 3*n43 - 2*n44 - 4*n45)*log(2) + (n22 + n24 + n32 + n34 + n42 + n44)*log(3) + 2*(-n21 - n22 - n23 - n24 - n25 - n31 - n32 - n33 - n34 - n35 - n41 - n42 - n43 - n44 - n45)*log(5) + 2*(n25 + n41)*log(3 - 2*r) + (n24 + n42)*log(3 - 2*r) + (n22 + n44)*log(1 + 2*r) + 2*(n21 + n45)*log(1 + 2*r) + (n31 + n35)*(log(3 - 2*r) + log(1 + 2*r)) + (n23 + n43)*log(21 + 4*r - 4*r^2) + n33*log(23 - 4*r + 4*r^2)
return(L)}
interlogL_rm <- function(n21,n22,n23,n24,n25,n31,n33,n35,n41,n42,n43,n44,n45,n32,n34) {
optimize(logL_rm,c(0,0.5), n21,n22,n23,n24,n25,n31,n33,n35,n41,n42,n43,n44,n45,n32,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_31"],x[,"n_33"],x[,"n_35"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_32"],x[,"n_34"])
LOD_rm <- -2*(x[,"n_31"] + x[,"n_35"])*log10(2) - 2*(x[,"n_25"] + x[,"n_41"])*log10(2) - (x[,"n_24"] + x[,"n_42"])*log10(2) - (x[,"n_22"] + x[,"n_44"])*log10(2) - 2*(x[,"n_21"] + x[,"n_45"])*log10(2) - x[,"n_33"]*(log10(2) + log10(11)) - (x[,"n_23"] + x[,"n_43"])*(log10(2) + log10(11)) + 2*(x[,"n_25"] + x[,"n_41"])*log10(3 - 2*r_rm) + (x[,"n_24"] + x[,"n_42"])*log10(3 - 2*r_rm) + (x[,"n_22"] + x[,"n_44"])*log10(1 + 2*r_rm) + 2*(x[,"n_21"] + x[,"n_45"])*log10(1 + 2*r_rm) + (x[,"n_31"] + x[,"n_35"])*(log10(3 - 2*r_rm) + log10(1 + 2*r_rm)) + (x[,"n_23"] + x[,"n_43"])*log10(21 + 4*r_rm - 4*r_rm^2) + x[,"n_33"]*log10(23 - 4*r_rm + 4*r_rm^2)
logL_rm <- (-4*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 2*x[,"n_24"] - 4*x[,"n_25"] - 3*x[,"n_31"] - 2*x[,"n_33"] - 3*x[,"n_35"] - 4*x[,"n_41"] - 2*x[,"n_42"] - 3*x[,"n_43"] - 2*x[,"n_44"] - 4*x[,"n_45"])*log(2) + (x[,"n_22"] + x[,"n_24"] + x[,"n_32"] + x[,"n_34"] + x[,"n_42"] + x[,"n_44"])*log(3) + 2*(-x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + 2*(x[,"n_25"] + x[,"n_41"])*log(3 - 2*r_rm) + (x[,"n_24"] + x[,"n_42"])*log(3 - 2*r_rm) + (x[,"n_22"] + x[,"n_44"])*log(1 + 2*r_rm) + 2*(x[,"n_21"] + x[,"n_45"])*log(1 + 2*r_rm) + (x[,"n_31"] + x[,"n_35"])*(log(3 - 2*r_rm) + log(1 + 2*r_rm)) + (x[,"n_23"] + x[,"n_43"])*log(21 + 4*r_rm - 4*r_rm^2) + x[,"n_33"]*log(23 - 4*r_rm + 4*r_rm^2)

return(list(
r_mat = cbind(r_cc,r_cm,r_rm,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_rm,0),
logL_mat = cbind(logL_cc,logL_cm,logL_rm,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","repulsion mixed","unknown")
)
)
}

#' @rdname r6_functions
r6_1.5_3.2 <- function(x){
logL_cc <- function(r,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-4*n20 - 4*n21 - 3*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n30 - 4*n31 - 3*n32 - 3*n33 - 4*n34 - 4*n35 - 4*n40 - 4*n41 - 3*n42 - 3*n43 - 4*n44 - 4*n45)*log(2) + (n21 + n24 + n31 + n34 + n41 + n44)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n20 + n45)*(log(3 - 2*r) + log(pmax(1e-6,1 - r))) + (n25 + n40)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n22 + n43)*log(23 - 7*r - 4*r^2) + (n30 + n35)*log(1 + 4*r - 4*r^2) + (n31 + n34)*log(7 + 4*r - 4*r^2) + (n23 + n42)*log(12 + 15*r - 4*r^2) + (n21 + n44)*log(8 - 9*r + 2*r^2) + (n24 + n41)*log(1 + 5*r + 2*r^2) + (n32 + n33)*log(39 - 8*r + 8*r^2)
return(L)}
interlogL_cc <- function(n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cc,c(0,0.5), n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cc <- -3*(x[,"n_31"] + x[,"n_34"])*log10(2) - (x[,"n_30"] + x[,"n_35"])*log10(2) - 2*(x[,"n_24"] + x[,"n_41"])*log10(2) - 2*(x[,"n_21"] + x[,"n_44"])*log10(2) - (x[,"n_32"] + x[,"n_33"])*log10(37) - (x[,"n_23"] + x[,"n_42"])*(-log10(2) + log10(37)) - (x[,"n_22"] + x[,"n_43"])*(-log10(2) + log10(37)) + (x[,"n_20"] + x[,"n_45"])*(log10(3 - 2*r_cc) + log10(pmax(1e-6,1 - r_cc))) + (x[,"n_25"] + x[,"n_40"])*(log10(pmax(1e-6,r_cc)) + log10(1 + 2*r_cc)) + (x[,"n_22"] + x[,"n_43"])*log10(23 - 7*r_cc - 4*r_cc^2) + (x[,"n_30"] + x[,"n_35"])*log10(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_31"] + x[,"n_34"])*log10(7 + 4*r_cc - 4*r_cc^2) + (x[,"n_23"] + x[,"n_42"])*log10(12 + 15*r_cc - 4*r_cc^2) + (x[,"n_21"] + x[,"n_44"])*log10(8 - 9*r_cc + 2*r_cc^2) + (x[,"n_24"] + x[,"n_41"])*log10(1 + 5*r_cc + 2*r_cc^2) + (x[,"n_32"] + x[,"n_33"])*log10(39 - 8*r_cc + 8*r_cc^2)
logL_cc <- (-4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"] - 4*x[,"n_35"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 4*x[,"n_45"])*log(2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_20"] + x[,"n_45"])*(log(3 - 2*r_cc) + log(pmax(1e-6,1 - r_cc))) + (x[,"n_25"] + x[,"n_40"])*(log(pmax(1e-6,r_cc)) + log(1 + 2*r_cc)) + (x[,"n_22"] + x[,"n_43"])*log(23 - 7*r_cc - 4*r_cc^2) + (x[,"n_30"] + x[,"n_35"])*log(1 + 4*r_cc - 4*r_cc^2) + (x[,"n_31"] + x[,"n_34"])*log(7 + 4*r_cc - 4*r_cc^2) + (x[,"n_23"] + x[,"n_42"])*log(12 + 15*r_cc - 4*r_cc^2) + (x[,"n_21"] + x[,"n_44"])*log(8 - 9*r_cc + 2*r_cc^2) + (x[,"n_24"] + x[,"n_41"])*log(1 + 5*r_cc + 2*r_cc^2) + (x[,"n_32"] + x[,"n_33"])*log(39 - 8*r_cc + 8*r_cc^2)
logL_cm <- function(r,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - 2*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 2*n35 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + (n21 + n24 + n31 + n34 + n41 + n44)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n20 + n25 + n40 + n45)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n32 + n33)*log(33 + 16*r - 16*r^2) + (n24 + n41)*log(2 + r - 2*r^2) + (n21 + n44)*log(1 + 3*r - 2*r^2) + (n30 + n35)*log(1 - 2*r + 2*r^2) + (n31 + n34)*log(5 - 4*r + 4*r^2) + (n23 + n42)*log(21 - 9*r + 8*r^2) + (n22 + n43)*log(20 - 7*r + 8*r^2)
return(L)}
interlogL_cm <- function(n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_cm,c(0,0.5), n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_cm <- -2*(x[,"n_31"] + x[,"n_34"])*log10(2) + (x[,"n_30"] + x[,"n_35"])*log10(2) - (x[,"n_24"] + x[,"n_41"])*log10(2) - (x[,"n_21"] + x[,"n_44"])*log10(2) + 2*(x[,"n_20"] + x[,"n_25"] + x[,"n_40"] + x[,"n_45"])*log10(2) - (x[,"n_32"] + x[,"n_33"])*log10(37) - (x[,"n_23"] + x[,"n_42"])*(-log10(2) + log10(37)) - (x[,"n_22"] + x[,"n_43"])*(-log10(2) + log10(37)) + (x[,"n_20"] + x[,"n_25"] + x[,"n_40"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + (x[,"n_32"] + x[,"n_33"])*log10(33 + 16*r_cm - 16*r_cm^2) + (x[,"n_24"] + x[,"n_41"])*log10(2 + r_cm - 2*r_cm^2) + (x[,"n_21"] + x[,"n_44"])*log10(1 + 3*r_cm - 2*r_cm^2) + (x[,"n_30"] + x[,"n_35"])*log10(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_31"] + x[,"n_34"])*log10(5 - 4*r_cm + 4*r_cm^2) + (x[,"n_23"] + x[,"n_42"])*log10(21 - 9*r_cm + 8*r_cm^2) + (x[,"n_22"] + x[,"n_43"])*log10(20 - 7*r_cm + 8*r_cm^2)
logL_cm <- (-2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - 2*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 2*x[,"n_35"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_20"] + x[,"n_25"] + x[,"n_40"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + (x[,"n_32"] + x[,"n_33"])*log(33 + 16*r_cm - 16*r_cm^2) + (x[,"n_24"] + x[,"n_41"])*log(2 + r_cm - 2*r_cm^2) + (x[,"n_21"] + x[,"n_44"])*log(1 + 3*r_cm - 2*r_cm^2) + (x[,"n_30"] + x[,"n_35"])*log(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_31"] + x[,"n_34"])*log(5 - 4*r_cm + 4*r_cm^2) + (x[,"n_23"] + x[,"n_42"])*log(21 - 9*r_cm + 8*r_cm^2) + (x[,"n_22"] + x[,"n_43"])*log(20 - 7*r_cm + 8*r_cm^2)
logL_rc <- function(r,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-4*n20 - 4*n21 - 3*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n30 - 4*n31 - 3*n32 - 3*n33 - 4*n34 - 4*n35 - 4*n40 - 4*n41 - 3*n42 - 3*n43 - 4*n44 - 4*n45)*log(2) + (n21 + n24 + n31 + n34 + n41 + n44)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + (n20 + n45)*(log(3 - 2*r) + log(pmax(1e-6,r))) + (n25 + n40)*(log(pmax(1e-6,1 - r)) + log(1 + 2*r)) + (n32 + n33)*log(35 + 8*r - 8*r^2) + (n24 + n41)*log(4 + r - 2*r^2) + (n21 + n44)*log(3 + 3*r - 2*r^2) + (n23 + n42)*log(21 - 7*r + 4*r^2) + (n30 + n35)*log(3 - 4*r + 4*r^2) + (n31 + n34)*log(9 - 4*r + 4*r^2) + (n22 + n43)*log(18 - r + 4*r^2)
return(L)}
interlogL_rc <- function(n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rc,c(0,0.5), n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rc <- mapply(interlogL_rc,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rc <- -3*(x[,"n_31"] + x[,"n_34"])*log10(2) - (x[,"n_30"] + x[,"n_35"])*log10(2) - 2*(x[,"n_24"] + x[,"n_41"])*log10(2) - 2*(x[,"n_21"] + x[,"n_44"])*log10(2) - (x[,"n_32"] + x[,"n_33"])*log10(37) - (x[,"n_23"] + x[,"n_42"])*(-log10(2) + log10(37)) - (x[,"n_22"] + x[,"n_43"])*(-log10(2) + log10(37)) + (x[,"n_20"] + x[,"n_45"])*(log10(3 - 2*r_rc) + log10(pmax(1e-6,r_rc))) + (x[,"n_25"] + x[,"n_40"])*(log10(pmax(1e-6,1 - r_rc)) + log10(1 + 2*r_rc)) + (x[,"n_32"] + x[,"n_33"])*log10(35 + 8*r_rc - 8*r_rc^2) + (x[,"n_24"] + x[,"n_41"])*log10(4 + r_rc - 2*r_rc^2) + (x[,"n_21"] + x[,"n_44"])*log10(3 + 3*r_rc - 2*r_rc^2) + (x[,"n_23"] + x[,"n_42"])*log10(21 - 7*r_rc + 4*r_rc^2) + (x[,"n_30"] + x[,"n_35"])*log10(3 - 4*r_rc + 4*r_rc^2) + (x[,"n_31"] + x[,"n_34"])*log10(9 - 4*r_rc + 4*r_rc^2) + (x[,"n_22"] + x[,"n_43"])*log10(18 - r_rc + 4*r_rc^2)
logL_rc <- (-4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"] - 4*x[,"n_35"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 4*x[,"n_45"])*log(2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + (x[,"n_20"] + x[,"n_45"])*(log(3 - 2*r_rc) + log(pmax(1e-6,r_rc))) + (x[,"n_25"] + x[,"n_40"])*(log(pmax(1e-6,1 - r_rc)) + log(1 + 2*r_rc)) + (x[,"n_32"] + x[,"n_33"])*log(35 + 8*r_rc - 8*r_rc^2) + (x[,"n_24"] + x[,"n_41"])*log(4 + r_rc - 2*r_rc^2) + (x[,"n_21"] + x[,"n_44"])*log(3 + 3*r_rc - 2*r_rc^2) + (x[,"n_23"] + x[,"n_42"])*log(21 - 7*r_rc + 4*r_rc^2) + (x[,"n_30"] + x[,"n_35"])*log(3 - 4*r_rc + 4*r_rc^2) + (x[,"n_31"] + x[,"n_34"])*log(9 - 4*r_rc + 4*r_rc^2) + (x[,"n_22"] + x[,"n_43"])*log(18 - r_rc + 4*r_rc^2)
logL_rm <- function(r,n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
L <- (-2*n20 - 3*n21 - 3*n22 - 3*n23 - 3*n24 - 2*n25 - n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - n35 - 2*n40 - 3*n41 - 3*n42 - 3*n43 - 3*n44 - 2*n45)*log(2) + (n21 + n24 + n31 + n34 + n41 + n44)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n30 - n31 - n32 - n33 - n34 - n35 - n40 - n41 - n42 - n43 - n44 - n45)*log(5) + 2*(n25 + n40)*log(pmax(1e-6,1 - r)) + 2*(n20 + n45)*log(pmax(1e-6,r)) + (n30 + n35)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n21 + n44)*(log(pmax(1e-6,r)) + log(3 + 2*r)) + (n23 + n42)*log(24 - 7*r - 8*r^2) + (n22 + n43)*log(9 + 23*r - 8*r^2) + (n31 + n34)*log(3 + 4*r - 4*r^2) + (n24 + n41)*log(5 - 7*r + 2*r^2) + (n32 + n33)*log(41 - 16*r + 16*r^2)
return(L)}
interlogL_rm <- function(n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45) {
optimize(logL_rm,c(0,0.5), n20,n21,n22,n23,n24,n25,n30,n31,n32,n33,n34,n35,n40,n41,n42,n43,n44,n45, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"])
LOD_rm <- -2*(x[,"n_31"] + x[,"n_34"])*log10(2) + 2*(x[,"n_30"] + x[,"n_35"])*log10(2) + 2*(x[,"n_25"] + x[,"n_40"])*log10(2) - (x[,"n_24"] + x[,"n_41"])*log10(2) - (x[,"n_21"] + x[,"n_44"])*log10(2) + 2*(x[,"n_20"] + x[,"n_45"])*log10(2) - (x[,"n_32"] + x[,"n_33"])*log10(37) - (x[,"n_23"] + x[,"n_42"])*(-log10(2) + log10(37)) - (x[,"n_22"] + x[,"n_43"])*(-log10(2) + log10(37)) + 2*(x[,"n_25"] + x[,"n_40"])*log10(pmax(1e-6,1 - r_rm)) + 2*(x[,"n_20"] + x[,"n_45"])*log10(pmax(1e-6,r_rm)) + (x[,"n_30"] + x[,"n_35"])*(log10(pmax(1e-6,1 - r_rm)) + log10(pmax(1e-6,r_rm))) + (x[,"n_21"] + x[,"n_44"])*(log10(pmax(1e-6,r_rm)) + log10(3 + 2*r_rm)) + (x[,"n_23"] + x[,"n_42"])*log10(24 - 7*r_rm - 8*r_rm^2) + (x[,"n_22"] + x[,"n_43"])*log10(9 + 23*r_rm - 8*r_rm^2) + (x[,"n_31"] + x[,"n_34"])*log10(3 + 4*r_rm - 4*r_rm^2) + (x[,"n_24"] + x[,"n_41"])*log10(5 - 7*r_rm + 2*r_rm^2) + (x[,"n_32"] + x[,"n_33"])*log10(41 - 16*r_rm + 16*r_rm^2)
logL_rm <- (-2*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_25"] - x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - x[,"n_35"] - 2*x[,"n_40"] - 3*x[,"n_41"] - 3*x[,"n_42"] - 3*x[,"n_43"] - 3*x[,"n_44"] - 2*x[,"n_45"])*log(2) + (x[,"n_21"] + x[,"n_24"] + x[,"n_31"] + x[,"n_34"] + x[,"n_41"] + x[,"n_44"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"])*log(5) + 2*(x[,"n_25"] + x[,"n_40"])*log(pmax(1e-6,1 - r_rm)) + 2*(x[,"n_20"] + x[,"n_45"])*log(pmax(1e-6,r_rm)) + (x[,"n_30"] + x[,"n_35"])*(log(pmax(1e-6,1 - r_rm)) + log(pmax(1e-6,r_rm))) + (x[,"n_21"] + x[,"n_44"])*(log(pmax(1e-6,r_rm)) + log(3 + 2*r_rm)) + (x[,"n_23"] + x[,"n_42"])*log(24 - 7*r_rm - 8*r_rm^2) + (x[,"n_22"] + x[,"n_43"])*log(9 + 23*r_rm - 8*r_rm^2) + (x[,"n_31"] + x[,"n_34"])*log(3 + 4*r_rm - 4*r_rm^2) + (x[,"n_24"] + x[,"n_41"])*log(5 - 7*r_rm + 2*r_rm^2) + (x[,"n_32"] + x[,"n_33"])*log(41 - 16*r_rm + 16*r_rm^2)

return(list(
r_mat = cbind(r_cc,r_cm,r_rc,r_rm,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_rc,LOD_rm,0),
logL_mat = cbind(logL_cc,logL_cm,logL_rc,logL_rm,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","repulsion coupling","repulsion mixed","unknown")
)
)
}

#' @rdname r6_functions
r6_1.5_3.3 <- function(x){
logL_cc <- function(r,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n20 - 3*n21 - 4*n22 - 2*n23 - 4*n24 - 3*n25 - 4*n26 - 3*n30 - 3*n31 - 3*n32 - 2*n33 - 3*n34 - 3*n35 - 3*n36 - 4*n40 - 3*n41 - 4*n42 - 2*n43 - 4*n44 - 3*n45 - 4*n46)*log(2) + (n21 + n22 + n24 + n25 + n31 + n32 + n34 + n35 + n41 + n42 + n44 + n45)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*log(5) + 2*(n20 + n46)*log(pmax(1e-6,1 - r)) + (n21 + n45)*(log(pmax(1e-6,1 - r)) + log(2 - r)) + 2*(n26 + n40)*log(pmax(1e-6,r)) + (n30 + n36)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n25 + n41)*(log(pmax(1e-6,r)) + log(1 + r)) + (n23 + n43)*log(9 + 5*r - 5*r^2) + (n31 + n35)*log(1 + 2*r - 2*r^2) + (n32 + n34)*log(8 + r - r^2) + (n22 + n44)*log(14 - 12*r + r^2) + (n24 + n42)*log(3 + 10*r + r^2) + n33*log(23 - 10*r + 10*r^2)
return(L)}
interlogL_cc <- function(n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_cc,c(0,0.5), n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_cc <- 2*(x[,"n_30"] + x[,"n_36"])*log10(2) + 2*(x[,"n_26"] + x[,"n_40"])*log10(2) + 2*(x[,"n_20"] + x[,"n_46"])*log10(2) - (x[,"n_25"] + x[,"n_41"])*(-2*log10(2) + log10(3)) - (x[,"n_21"] + x[,"n_45"])*(-2*log10(2) + log10(3)) - (x[,"n_31"] + x[,"n_35"])*(-log10(2) + log10(3)) - (x[,"n_32"] + x[,"n_34"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_24"] + x[,"n_42"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_22"] + x[,"n_44"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_23"] + x[,"n_43"])*(-2*log10(2) + log10(41)) - x[,"n_33"]*(-log10(2) + log10(41)) + 2*(x[,"n_20"] + x[,"n_46"])*log10(pmax(1e-6,1 - r_cc)) + (x[,"n_21"] + x[,"n_45"])*(log10(pmax(1e-6,1 - r_cc)) + log10(2 - r_cc)) + 2*(x[,"n_26"] + x[,"n_40"])*log10(pmax(1e-6,r_cc)) + (x[,"n_30"] + x[,"n_36"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_25"] + x[,"n_41"])*(log10(pmax(1e-6,r_cc)) + log10(1 + r_cc)) + (x[,"n_23"] + x[,"n_43"])*log10(9 + 5*r_cc - 5*r_cc^2) + (x[,"n_31"] + x[,"n_35"])*log10(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_32"] + x[,"n_34"])*log10(8 + r_cc - r_cc^2) + (x[,"n_22"] + x[,"n_44"])*log10(14 - 12*r_cc + r_cc^2) + (x[,"n_24"] + x[,"n_42"])*log10(3 + 10*r_cc + r_cc^2) + x[,"n_33"]*log10(23 - 10*r_cc + 10*r_cc^2)
logL_cc <- (-4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_25"] - 4*x[,"n_26"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 2*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_35"] - 3*x[,"n_36"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 2*x[,"n_43"] - 4*x[,"n_44"] - 3*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (x[,"n_21"] + x[,"n_22"] + x[,"n_24"] + x[,"n_25"] + x[,"n_31"] + x[,"n_32"] + x[,"n_34"] + x[,"n_35"] + x[,"n_41"] + x[,"n_42"] + x[,"n_44"] + x[,"n_45"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*log(5) + 2*(x[,"n_20"] + x[,"n_46"])*log(pmax(1e-6,1 - r_cc)) + (x[,"n_21"] + x[,"n_45"])*(log(pmax(1e-6,1 - r_cc)) + log(2 - r_cc)) + 2*(x[,"n_26"] + x[,"n_40"])*log(pmax(1e-6,r_cc)) + (x[,"n_30"] + x[,"n_36"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_25"] + x[,"n_41"])*(log(pmax(1e-6,r_cc)) + log(1 + r_cc)) + (x[,"n_23"] + x[,"n_43"])*log(9 + 5*r_cc - 5*r_cc^2) + (x[,"n_31"] + x[,"n_35"])*log(1 + 2*r_cc - 2*r_cc^2) + (x[,"n_32"] + x[,"n_34"])*log(8 + r_cc - r_cc^2) + (x[,"n_22"] + x[,"n_44"])*log(14 - 12*r_cc + r_cc^2) + (x[,"n_24"] + x[,"n_42"])*log(3 + 10*r_cc + r_cc^2) + x[,"n_33"]*log(23 - 10*r_cc + 10*r_cc^2)
logL_cm <- function(r,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n20 - 4*n21 - 4*n22 - 3*n23 - 4*n24 - 4*n25 - 4*n26 - 4*n30 - 2*n31 - 4*n32 - n33 - 4*n34 - 2*n35 - 4*n36 - 4*n40 - 4*n41 - 4*n42 - 3*n43 - 4*n44 - 4*n45 - 4*n46)*log(2) + (n21 + n22 + n24 + n25 + n31 + n32 + n34 + n35 + n41 + n42 + n44 + n45)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*log(5) + (n20 + n26 + n40 + n46)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n33*log(9 + 5*r - 5*r^2) + (n21 + n25 + n41 + n45)*log(1 + 2*r - 2*r^2) + (n22 + n24 + n42 + n44)*log(8 + r - r^2) + (n31 + n35)*log(1 - r + r^2) + (n30 + n36)*log(1 - 2*r + 2*r^2) + (n32 + n34)*log(17 - 2*r + 2*r^2) + (n23 + n43)*log(23 - 10*r + 10*r^2)
return(L)}
interlogL_cm <- function(n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_cm,c(0,0.5), n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cm <- mapply(interlogL_cm,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_cm <- (x[,"n_30"] + x[,"n_36"])*log10(2) + 2*(x[,"n_20"] + x[,"n_26"] + x[,"n_40"] + x[,"n_46"])*log10(2) + (x[,"n_31"] + x[,"n_35"])*(2*log10(2) - log10(3)) - (x[,"n_21"] + x[,"n_25"] + x[,"n_41"] + x[,"n_45"])*(-log10(2) + log10(3)) - (x[,"n_22"] + x[,"n_24"] + x[,"n_42"] + x[,"n_44"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_32"] + x[,"n_34"])*(-log10(2) + log10(3) + log10(11)) - x[,"n_33"]*(-2*log10(2) + log10(41)) - (x[,"n_23"] + x[,"n_43"])*(-log10(2) + log10(41)) + (x[,"n_20"] + x[,"n_26"] + x[,"n_40"] + x[,"n_46"])*(log10(pmax(1e-6,1 - r_cm)) + log10(pmax(1e-6,r_cm))) + x[,"n_33"]*log10(9 + 5*r_cm - 5*r_cm^2) + (x[,"n_21"] + x[,"n_25"] + x[,"n_41"] + x[,"n_45"])*log10(1 + 2*r_cm - 2*r_cm^2) + (x[,"n_22"] + x[,"n_24"] + x[,"n_42"] + x[,"n_44"])*log10(8 + r_cm - r_cm^2) + (x[,"n_31"] + x[,"n_35"])*log10(1 - r_cm + r_cm^2) + (x[,"n_30"] + x[,"n_36"])*log10(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_32"] + x[,"n_34"])*log10(17 - 2*r_cm + 2*r_cm^2) + (x[,"n_23"] + x[,"n_43"])*log10(23 - 10*r_cm + 10*r_cm^2)
logL_cm <- (-4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_25"] - 4*x[,"n_26"] - 4*x[,"n_30"] - 2*x[,"n_31"] - 4*x[,"n_32"] - x[,"n_33"] - 4*x[,"n_34"] - 2*x[,"n_35"] - 4*x[,"n_36"] - 4*x[,"n_40"] - 4*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"] - 4*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (x[,"n_21"] + x[,"n_22"] + x[,"n_24"] + x[,"n_25"] + x[,"n_31"] + x[,"n_32"] + x[,"n_34"] + x[,"n_35"] + x[,"n_41"] + x[,"n_42"] + x[,"n_44"] + x[,"n_45"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*log(5) + (x[,"n_20"] + x[,"n_26"] + x[,"n_40"] + x[,"n_46"])*(log(pmax(1e-6,1 - r_cm)) + log(pmax(1e-6,r_cm))) + x[,"n_33"]*log(9 + 5*r_cm - 5*r_cm^2) + (x[,"n_21"] + x[,"n_25"] + x[,"n_41"] + x[,"n_45"])*log(1 + 2*r_cm - 2*r_cm^2) + (x[,"n_22"] + x[,"n_24"] + x[,"n_42"] + x[,"n_44"])*log(8 + r_cm - r_cm^2) + (x[,"n_31"] + x[,"n_35"])*log(1 - r_cm + r_cm^2) + (x[,"n_30"] + x[,"n_36"])*log(1 - 2*r_cm + 2*r_cm^2) + (x[,"n_32"] + x[,"n_34"])*log(17 - 2*r_cm + 2*r_cm^2) + (x[,"n_23"] + x[,"n_43"])*log(23 - 10*r_cm + 10*r_cm^2)
logL_rm <- function(r,n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
L <- (-4*n20 - 3*n21 - 4*n22 - 2*n23 - 4*n24 - 3*n25 - 4*n26 - 3*n30 - 3*n31 - 3*n32 - 2*n33 - 3*n34 - 3*n35 - 3*n36 - 4*n40 - 3*n41 - 4*n42 - 2*n43 - 4*n44 - 3*n45 - 4*n46)*log(2) + (n21 + n22 + n24 + n25 + n31 + n32 + n34 + n35 + n41 + n42 + n44 + n45)*log(3) + 2*(-n20 - n21 - n22 - n23 - n24 - n25 - n26 - n30 - n31 - n32 - n33 - n34 - n35 - n36 - n40 - n41 - n42 - n43 - n44 - n45 - n46)*log(5) + 2*(n26 + n40)*log(pmax(1e-6,1 - r)) + (n25 + n41)*(log(pmax(1e-6,1 - r)) + log(2 - r)) + 2*(n20 + n46)*log(pmax(1e-6,r)) + (n30 + n36)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n21 + n45)*(log(pmax(1e-6,r)) + log(1 + r)) + (n23 + n43)*log(9 + 5*r - 5*r^2) + (n31 + n35)*log(1 + 2*r - 2*r^2) + (n32 + n34)*log(8 + r - r^2) + (n24 + n42)*log(14 - 12*r + r^2) + (n22 + n44)*log(3 + 10*r + r^2) + n33*log(23 - 10*r + 10*r^2)
return(L)}
interlogL_rm <- function(n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46) {
optimize(logL_rm,c(0,0.5), n20,n21,n22,n23,n24,n25,n26,n30,n31,n32,n33,n34,n35,n36,n40,n41,n42,n43,n44,n45,n46, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_rm <- mapply(interlogL_rm,x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_35"],x[,"n_36"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"],x[,"n_45"],x[,"n_46"])
LOD_rm <- 2*(x[,"n_30"] + x[,"n_36"])*log10(2) + 2*(x[,"n_26"] + x[,"n_40"])*log10(2) + 2*(x[,"n_20"] + x[,"n_46"])*log10(2) - (x[,"n_25"] + x[,"n_41"])*(-2*log10(2) + log10(3)) - (x[,"n_21"] + x[,"n_45"])*(-2*log10(2) + log10(3)) - (x[,"n_31"] + x[,"n_35"])*(-log10(2) + log10(3)) - (x[,"n_32"] + x[,"n_34"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_24"] + x[,"n_42"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_22"] + x[,"n_44"])*(-2*log10(2) + log10(3) + log10(11)) - (x[,"n_23"] + x[,"n_43"])*(-2*log10(2) + log10(41)) - x[,"n_33"]*(-log10(2) + log10(41)) + 2*(x[,"n_26"] + x[,"n_40"])*log10(pmax(1e-6,1 - r_rm)) + (x[,"n_25"] + x[,"n_41"])*(log10(pmax(1e-6,1 - r_rm)) + log10(2 - r_rm)) + 2*(x[,"n_20"] + x[,"n_46"])*log10(pmax(1e-6,r_rm)) + (x[,"n_30"] + x[,"n_36"])*(log10(pmax(1e-6,1 - r_rm)) + log10(pmax(1e-6,r_rm))) + (x[,"n_21"] + x[,"n_45"])*(log10(pmax(1e-6,r_rm)) + log10(1 + r_rm)) + (x[,"n_23"] + x[,"n_43"])*log10(9 + 5*r_rm - 5*r_rm^2) + (x[,"n_31"] + x[,"n_35"])*log10(1 + 2*r_rm - 2*r_rm^2) + (x[,"n_32"] + x[,"n_34"])*log10(8 + r_rm - r_rm^2) + (x[,"n_24"] + x[,"n_42"])*log10(14 - 12*r_rm + r_rm^2) + (x[,"n_22"] + x[,"n_44"])*log10(3 + 10*r_rm + r_rm^2) + x[,"n_33"]*log10(23 - 10*r_rm + 10*r_rm^2)
logL_rm <- (-4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_25"] - 4*x[,"n_26"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 2*x[,"n_33"] - 3*x[,"n_34"] - 3*x[,"n_35"] - 3*x[,"n_36"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 2*x[,"n_43"] - 4*x[,"n_44"] - 3*x[,"n_45"] - 4*x[,"n_46"])*log(2) + (x[,"n_21"] + x[,"n_22"] + x[,"n_24"] + x[,"n_25"] + x[,"n_31"] + x[,"n_32"] + x[,"n_34"] + x[,"n_35"] + x[,"n_41"] + x[,"n_42"] + x[,"n_44"] + x[,"n_45"])*log(3) + 2*(-x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_35"] - x[,"n_36"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"] - x[,"n_45"] - x[,"n_46"])*log(5) + 2*(x[,"n_26"] + x[,"n_40"])*log(pmax(1e-6,1 - r_rm)) + (x[,"n_25"] + x[,"n_41"])*(log(pmax(1e-6,1 - r_rm)) + log(2 - r_rm)) + 2*(x[,"n_20"] + x[,"n_46"])*log(pmax(1e-6,r_rm)) + (x[,"n_30"] + x[,"n_36"])*(log(pmax(1e-6,1 - r_rm)) + log(pmax(1e-6,r_rm))) + (x[,"n_21"] + x[,"n_45"])*(log(pmax(1e-6,r_rm)) + log(1 + r_rm)) + (x[,"n_23"] + x[,"n_43"])*log(9 + 5*r_rm - 5*r_rm^2) + (x[,"n_31"] + x[,"n_35"])*log(1 + 2*r_rm - 2*r_rm^2) + (x[,"n_32"] + x[,"n_34"])*log(8 + r_rm - r_rm^2) + (x[,"n_24"] + x[,"n_42"])*log(14 - 12*r_rm + r_rm^2) + (x[,"n_22"] + x[,"n_44"])*log(3 + 10*r_rm + r_rm^2) + x[,"n_33"]*log(23 - 10*r_rm + 10*r_rm^2)

return(list(
r_mat = cbind(r_cc,r_cm,r_rm,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm,LOD_rm,0),
logL_mat = cbind(logL_cc,logL_cm,logL_rm,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","repulsion mixed","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_1.2 <- function(x){
logL_c <- function(r,n10,n11,n12,n13,n00,n01,n02,n03,n20,n21,n22,n23) {
L <- (-n10 + n11 + n12 - n13)*log(2) + (n10 + n11 + n12 + n13)*log(3) + 2*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(5) + (n01 + n22)*log(3 - 2*r) + (n00 + n23)*log(pmax(1e-6,1 - r)) + (n03 + n20)*log(pmax(1e-6,r)) + (n02 + n21)*log(1 + 2*r)
return(L)}
interlogL_c <- function(n10,n11,n12,n13,n00,n01,n02,n03,n20,n21,n22,n23) {
optimize(logL_c,c(0,0.5), n10,n11,n12,n13,n00,n01,n02,n03,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_c <- (x[,"n_03"] + x[,"n_20"])*log10(2) - (x[,"n_02"] + x[,"n_21"])*log10(2) - (x[,"n_01"] + x[,"n_22"])*log10(2) + (x[,"n_00"] + x[,"n_23"])*log10(2) + (x[,"n_01"] + x[,"n_22"])*log10(3 - 2*r_c) + (x[,"n_00"] + x[,"n_23"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_03"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_21"])*log10(1 + 2*r_c)
logL_c <- (-x[,"n_10"] + x[,"n_11"] + x[,"n_12"] - x[,"n_13"])*log(2) + (x[,"n_10"] + x[,"n_11"] + x[,"n_12"] + x[,"n_13"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(5) + (x[,"n_01"] + x[,"n_22"])*log(3 - 2*r_c) + (x[,"n_00"] + x[,"n_23"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_03"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_02"] + x[,"n_21"])*log(1 + 2*r_c)
logL_r <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
L <- (-2*n00 - n01 - n02 - 2*n03 - n10 + n11 + n12 - n13 - 2*n20 - n21 - n22 - 2*n23)*log(2) + (n10 + n11 + n12 + n13)*log(3) + 2*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(5) + (n03 + n20)*log(3 - 2*r) + (n02 + n21)*log(5 - 2*r) + (n00 + n23)*log(1 + 2*r) + (n01 + n22)*log(3 + 2*r)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_r <- -((x[,"n_03"] + x[,"n_20"])*log10(2)) - 2*(x[,"n_02"] + x[,"n_21"])*log10(2) - 2*(x[,"n_01"] + x[,"n_22"])*log10(2) - (x[,"n_00"] + x[,"n_23"])*log10(2) + (x[,"n_03"] + x[,"n_20"])*log10(3 - 2*r_r) + (x[,"n_02"] + x[,"n_21"])*log10(5 - 2*r_r) + (x[,"n_00"] + x[,"n_23"])*log10(1 + 2*r_r) + (x[,"n_01"] + x[,"n_22"])*log10(3 + 2*r_r)
logL_r <- (-2*x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - x[,"n_10"] + x[,"n_11"] + x[,"n_12"] - x[,"n_13"] - 2*x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"])*log(2) + (x[,"n_10"] + x[,"n_11"] + x[,"n_12"] + x[,"n_13"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(5) + (x[,"n_03"] + x[,"n_20"])*log(3 - 2*r_r) + (x[,"n_02"] + x[,"n_21"])*log(5 - 2*r_r) + (x[,"n_00"] + x[,"n_23"])*log(1 + 2*r_r) + (x[,"n_01"] + x[,"n_22"])*log(3 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_1.3 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 3*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24)*log(2) + (2*n02 + n10 + n11 + 3*n12 + n13 + n14 + 2*n22)*log(3) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n10 - n11 - 2*n12 - n13 - 2*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24)*log(5) + (n01 + n23)*log(9 - 8*r) + (n00 + n24)*log(pmax(1e-6,1 - r)) + (n04 + n20)*log(pmax(1e-6,r)) + (n03 + n21)*log(1 + 8*r)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_c <- (x[,"n_04"] + x[,"n_20"])*log10(2) + (x[,"n_00"] + x[,"n_24"])*log10(2) - (x[,"n_03"] + x[,"n_21"])*log10(5) - (x[,"n_01"] + x[,"n_23"])*log10(5) + (x[,"n_01"] + x[,"n_23"])*log10(9 - 8*r_c) + (x[,"n_00"] + x[,"n_24"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_21"])*log10(1 + 8*r_c)
logL_c <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(2) + (2*x[,"n_02"] + x[,"n_10"] + x[,"n_11"] + 3*x[,"n_12"] + x[,"n_13"] + x[,"n_14"] + 2*x[,"n_22"])*log(3) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - x[,"n_11"] - 2*x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(5) + (x[,"n_01"] + x[,"n_23"])*log(9 - 8*r_c) + (x[,"n_00"] + x[,"n_24"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_21"])*log(1 + 8*r_c)
logL_r <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-4*n00 - 2*n01 - 2*n02 - 2*n03 - 4*n04 - 3*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n14 - 4*n20 - 2*n21 - 2*n22 - 2*n23 - 4*n24)*log(2) + (2*n02 + n10 + n11 + 3*n12 + n13 + n14 + 2*n22)*log(3) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n10 - n11 - 2*n12 - n13 - 2*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24)*log(5) + (n03 + n21)*log(7 - 4*r) + (n04 + n20)*log(3 - 2*r) + (n00 + n24)*log(1 + 2*r) + (n01 + n23)*log(3 + 4*r)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_r <- -((x[,"n_04"] + x[,"n_20"])*log10(2)) - (x[,"n_00"] + x[,"n_24"])*log10(2) - (x[,"n_03"] + x[,"n_21"])*log10(5) - (x[,"n_01"] + x[,"n_23"])*log10(5) + (x[,"n_03"] + x[,"n_21"])*log10(7 - 4*r_r) + (x[,"n_04"] + x[,"n_20"])*log10(3 - 2*r_r) + (x[,"n_00"] + x[,"n_24"])*log10(1 + 2*r_r) + (x[,"n_01"] + x[,"n_23"])*log10(3 + 4*r_r)
logL_r <- (-4*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"])*log(2) + (2*x[,"n_02"] + x[,"n_10"] + x[,"n_11"] + 3*x[,"n_12"] + x[,"n_13"] + x[,"n_14"] + 2*x[,"n_22"])*log(3) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - x[,"n_11"] - 2*x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(5) + (x[,"n_03"] + x[,"n_21"])*log(7 - 4*r_r) + (x[,"n_04"] + x[,"n_20"])*log(3 - 2*r_r) + (x[,"n_00"] + x[,"n_24"])*log(1 + 2*r_r) + (x[,"n_01"] + x[,"n_23"])*log(3 + 4*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_1.4 <- function(x){
logL_c <- function(r,n11,n12,n13,n14,n01,n02,n03,n04,n21,n22,n23,n24) {
L <- (-n11 + n12 + n13 - n14)*log(2) + (n11 + n12 + n13 + n14)*log(3) + 2*(-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24)*log(5) + (n02 + n23)*log(3 - 2*r) + (n01 + n24)*log(pmax(1e-6,1 - r)) + (n04 + n21)*log(pmax(1e-6,r)) + (n03 + n22)*log(1 + 2*r)
return(L)}
interlogL_c <- function(n11,n12,n13,n14,n01,n02,n03,n04,n21,n22,n23,n24) {
optimize(logL_c,c(0,0.5), n11,n12,n13,n14,n01,n02,n03,n04,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_c <- (x[,"n_04"] + x[,"n_21"])*log10(2) - (x[,"n_03"] + x[,"n_22"])*log10(2) - (x[,"n_02"] + x[,"n_23"])*log10(2) + (x[,"n_01"] + x[,"n_24"])*log10(2) + (x[,"n_02"] + x[,"n_23"])*log10(3 - 2*r_c) + (x[,"n_01"] + x[,"n_24"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_21"])*log10(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_22"])*log10(1 + 2*r_c)
logL_c <- (-x[,"n_11"] + x[,"n_12"] + x[,"n_13"] - x[,"n_14"])*log(2) + (x[,"n_11"] + x[,"n_12"] + x[,"n_13"] + x[,"n_14"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_02"] + x[,"n_23"])*log(3 - 2*r_c) + (x[,"n_01"] + x[,"n_24"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_21"])*log(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_22"])*log(1 + 2*r_c)
logL_r <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
L <- (-2*n01 - n02 - n03 - 2*n04 - n11 + n12 + n13 - n14 - 2*n21 - n22 - n23 - 2*n24)*log(2) + (n11 + n12 + n13 + n14)*log(3) + 2*(-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24)*log(5) + (n04 + n21)*log(3 - 2*r) + (n03 + n22)*log(5 - 2*r) + (n01 + n24)*log(1 + 2*r) + (n02 + n23)*log(3 + 2*r)
return(L)}
interlogL_r <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
optimize(logL_r,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_r <- -((x[,"n_04"] + x[,"n_21"])*log10(2)) - 2*(x[,"n_03"] + x[,"n_22"])*log10(2) - 2*(x[,"n_02"] + x[,"n_23"])*log10(2) - (x[,"n_01"] + x[,"n_24"])*log10(2) + (x[,"n_04"] + x[,"n_21"])*log10(3 - 2*r_r) + (x[,"n_03"] + x[,"n_22"])*log10(5 - 2*r_r) + (x[,"n_01"] + x[,"n_24"])*log10(1 + 2*r_r) + (x[,"n_02"] + x[,"n_23"])*log10(3 + 2*r_r)
logL_r <- (-2*x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - 2*x[,"n_04"] - x[,"n_11"] + x[,"n_12"] + x[,"n_13"] - x[,"n_14"] - 2*x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"])*log(2) + (x[,"n_11"] + x[,"n_12"] + x[,"n_13"] + x[,"n_14"])*log(3) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_04"] + x[,"n_21"])*log(3 - 2*r_r) + (x[,"n_03"] + x[,"n_22"])*log(5 - 2*r_r) + (x[,"n_01"] + x[,"n_24"])*log(1 + 2*r_r) + (x[,"n_02"] + x[,"n_23"])*log(3 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_1.5 <- function(x){
r_c <- (x[,"n_04"] + x[,"n_22"])/(x[,"n_02"] + x[,"n_04"] + x[,"n_22"] + x[,"n_24"])
logL_c <- (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - 2*x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(2) + (x[,"n_12"] + x[,"n_13"] + x[,"n_14"])*log(3) + (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_02"] + x[,"n_24"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_22"])*log(pmax(1e-6,r_c))
LOD_c <- (x[,"n_04"] + x[,"n_22"])*log10(2) + (x[,"n_02"] + x[,"n_24"])*log10(2) + (x[,"n_02"] + x[,"n_24"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_04"] + x[,"n_22"])*log10(pmax(1e-6,r_c))
r_r <- (3*x[,"n_02"] - x[,"n_04"] - x[,"n_22"] + 3*x[,"n_24"])/(2*(x[,"n_02"] + x[,"n_04"] + x[,"n_22"] + x[,"n_24"]))
logL_r <- (-3*x[,"n_02"] - x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_22"] - x[,"n_23"] - 3*x[,"n_24"])*log(2) + (x[,"n_12"] + x[,"n_13"] + x[,"n_14"])*log(3) + (-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_04"] + x[,"n_22"])*log(3 - 2*r_r) + (x[,"n_02"] + x[,"n_24"])*log(1 + 2*r_r)
LOD_r <- -((x[,"n_04"] + x[,"n_22"])*log10(2)) - (x[,"n_02"] + x[,"n_24"])*log10(2) + (x[,"n_04"] + x[,"n_22"])*log10(3 - 2*r_r) + (x[,"n_02"] + x[,"n_24"])*log10(1 + 2*r_r)

return(list(
r_mat = cbind(r_c,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_r,0),
logL_mat = cbind(logL_c,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_2.0 <- function(x){
logL_c <- function(r,n01,n10,n12,n21,n00,n02,n11,n20,n22) {
L <- (n01 + n10 + n12 + n21)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(5) + 2*(n00 + n22)*log(pmax(1e-6,1 - r)) + 2*(n02 + n20)*log(pmax(1e-6,r)) + (n01 + n10 + n12 + n21)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n11*log(3 - 4*r + 4*r^2)
return(L)}
interlogL_c <- function(n01,n10,n12,n21,n00,n02,n11,n20,n22) {
optimize(logL_c,c(0,0.5), n01,n10,n12,n21,n00,n02,n11,n20,n22, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_01"],x[,"n_10"],x[,"n_12"],x[,"n_21"],x[,"n_00"],x[,"n_02"],x[,"n_11"],x[,"n_20"],x[,"n_22"])
LOD_c <- -(x[,"n_11"]*log10(2)) + 2*(x[,"n_02"] + x[,"n_20"])*log10(2) + 2*(x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(2) + 2*(x[,"n_00"] + x[,"n_22"])*log10(2) + 2*(x[,"n_00"] + x[,"n_22"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_02"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_c)) + log10(pmax(1e-6,r_c))) + x[,"n_11"]*log10(3 - 4*r_c + 4*r_c^2)
logL_c <- (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(5) + 2*(x[,"n_00"] + x[,"n_22"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_02"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_c)) + log(pmax(1e-6,r_c))) + x[,"n_11"]*log(3 - 4*r_c + 4*r_c^2)
logL_m <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n10 - n11 - 2*n12 - 2*n20 - 2*n21 - 2*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(5) + (n02 + n20)*(log(2 - r) + log(pmax(1e-6,r))) + n11*log(3 + 2*r - 2*r^2) + (n00 + n22)*log(1 - r^2) + (n01 + n10 + n12 + n21)*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_m <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22) {
optimize(logL_m,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"])
LOD_m <- (x[,"n_00"] + x[,"n_22"])*(2*log10(2) - log10(3)) - (x[,"n_02"] + x[,"n_20"])*(-2*log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(-log10(2) + log10(5)) - x[,"n_11"]*(-log10(2) + log10(7)) + (x[,"n_02"] + x[,"n_20"])*(log10(2 - r_m) + log10(pmax(1e-6,r_m))) + x[,"n_11"]*log10(3 + 2*r_m - 2*r_m^2) + (x[,"n_00"] + x[,"n_22"])*log10(1 - r_m^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(3 - 2*r_m + 2*r_m^2)
logL_m <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_10"] - x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(5) + (x[,"n_02"] + x[,"n_20"])*(log(2 - r_m) + log(pmax(1e-6,r_m))) + x[,"n_11"]*log(3 + 2*r_m - 2*r_m^2) + (x[,"n_00"] + x[,"n_22"])*log(1 - r_m^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(3 - 2*r_m + 2*r_m^2)
logL_r <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22) {
L <- (-n00 - n01 - n02 - n10 + n11 - n12 - n20 - n21 - n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*(log(3) + log(5)) + (n00 + n22)*(log(pmax(1e-6,r)) + log(2 + r)) + (n01 + n10 + n12 + n21)*log(3 + 2*r - 2*r^2) + (n02 + n20)*log(3 - 4*r + r^2) + n11*log(3 - r + r^2)
return(L)}
interlogL_r <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"])
LOD_r <- -((x[,"n_02"] + x[,"n_20"])*(-2*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_22"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(-log10(2) + log10(7)) - x[,"n_11"]*(-2*log10(2) + log10(11)) + (x[,"n_00"] + x[,"n_22"])*(log10(pmax(1e-6,r_r)) + log10(2 + r_r)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(3 + 2*r_r - 2*r_r^2) + (x[,"n_02"] + x[,"n_20"])*log10(3 - 4*r_r + r_r^2) + x[,"n_11"]*log10(3 - r_r + r_r^2)
logL_r <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] + x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*(log(3) + log(5)) + (x[,"n_00"] + x[,"n_22"])*(log(pmax(1e-6,r_r)) + log(2 + r_r)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(3 + 2*r_r - 2*r_r^2) + (x[,"n_02"] + x[,"n_20"])*log(3 - 4*r_r + r_r^2) + x[,"n_11"]*log(3 - r_r + r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_2.1 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n11,n12,n20,n21,n22,n23,n10,n13) {
L <- (-n00 - n01 - n02 - n03 - n11 - n12 - n20 - n21 - n22 - n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(5) + 2*(n00 + n23)*log(pmax(1e-6,1 - r)) + 2*(n03 + n20)*log(pmax(1e-6,r)) + (n10 + n13)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n21)*(log(2 - r) + log(pmax(1e-6,r))) + (n01 + n22)*log(1 - r^2) + (n11 + n12)*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n11,n12,n20,n21,n22,n23,n10,n13) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n11,n12,n20,n21,n22,n23,n10,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_10"],x[,"n_13"])
LOD_c <- 2*(x[,"n_10"] + x[,"n_13"])*log10(2) + 2*(x[,"n_03"] + x[,"n_20"])*log10(2) + 2*(x[,"n_00"] + x[,"n_23"])*log10(2) + (x[,"n_01"] + x[,"n_22"])*(2*log10(2) - log10(3)) - (x[,"n_02"] + x[,"n_21"])*(-2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_12"])*(-log10(2) + log10(5)) + 2*(x[,"n_00"] + x[,"n_23"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_03"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_13"])*(log10(pmax(1e-6,1 - r_c)) + log10(pmax(1e-6,r_c))) + (x[,"n_02"] + x[,"n_21"])*(log10(2 - r_c) + log10(pmax(1e-6,r_c))) + (x[,"n_01"] + x[,"n_22"])*log10(1 - r_c^2) + (x[,"n_11"] + x[,"n_12"])*log10(3 - 2*r_c + 2*r_c^2)
logL_c <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(5) + 2*(x[,"n_00"] + x[,"n_23"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_03"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_13"])*(log(pmax(1e-6,1 - r_c)) + log(pmax(1e-6,r_c))) + (x[,"n_02"] + x[,"n_21"])*(log(2 - r_c) + log(pmax(1e-6,r_c))) + (x[,"n_01"] + x[,"n_22"])*log(1 - r_c^2) + (x[,"n_11"] + x[,"n_12"])*log(3 - 2*r_c + 2*r_c^2)
logL_m <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(3*log(2) + log(5)) + (n03 + n20)*(log(2 - r) + log(pmax(1e-6,r))) + (n11 + n12)*log(9 + 2*r - 2*r^2) + (n00 + n23)*log(1 - r^2) + (n02 + n21)*log(3 + r^2) + (n01 + n22)*log(4 - 2*r + r^2) + (n10 + n13)*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_m <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
optimize(logL_m,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_m <- (x[,"n_00"] + x[,"n_23"])*(2*log10(2) - log10(3)) - (x[,"n_03"] + x[,"n_20"])*(-2*log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_13"])*(-log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_21"])*(-2*log10(2) + log10(13)) - (x[,"n_01"] + x[,"n_22"])*(-2*log10(2) + log10(13)) - (x[,"n_11"] + x[,"n_12"])*(-log10(2) + log10(19)) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - r_m) + log10(pmax(1e-6,r_m))) + (x[,"n_11"] + x[,"n_12"])*log10(9 + 2*r_m - 2*r_m^2) + (x[,"n_00"] + x[,"n_23"])*log10(1 - r_m^2) + (x[,"n_02"] + x[,"n_21"])*log10(3 + r_m^2) + (x[,"n_01"] + x[,"n_22"])*log10(4 - 2*r_m + r_m^2) + (x[,"n_10"] + x[,"n_13"])*log10(3 - 2*r_m + 2*r_m^2)
logL_m <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(3*log(2) + log(5)) + (x[,"n_03"] + x[,"n_20"])*(log(2 - r_m) + log(pmax(1e-6,r_m))) + (x[,"n_11"] + x[,"n_12"])*log(9 + 2*r_m - 2*r_m^2) + (x[,"n_00"] + x[,"n_23"])*log(1 - r_m^2) + (x[,"n_02"] + x[,"n_21"])*log(3 + r_m^2) + (x[,"n_01"] + x[,"n_22"])*log(4 - 2*r_m + r_m^2) + (x[,"n_10"] + x[,"n_13"])*log(3 - 2*r_m + 2*r_m^2)
logL_r <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(2*log(2) + log(3) + log(5)) + (n00 + n23)*(log(pmax(1e-6,r)) + log(2 + r)) + (n10 + n13)*log(3 + 2*r - 2*r^2) + (n02 + n21)*log(6 - 2*r - r^2) + (n01 + n22)*log(3 + 4*r - r^2) + (n03 + n20)*log(3 - 4*r + r^2) + (n11 + n12)*log(15 - 2*r + 2*r^2)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_r <- -((x[,"n_03"] + x[,"n_20"])*(-2*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_23"])*(-2*log10(2) + log10(5)) - (x[,"n_10"] + x[,"n_13"])*(-log10(2) + log10(7)) - (x[,"n_02"] + x[,"n_21"])*(-2*log10(2) + log10(19)) - (x[,"n_01"] + x[,"n_22"])*(-2*log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_12"])*(-log10(2) + log10(29)) + (x[,"n_00"] + x[,"n_23"])*(log10(pmax(1e-6,r_r)) + log10(2 + r_r)) + (x[,"n_10"] + x[,"n_13"])*log10(3 + 2*r_r - 2*r_r^2) + (x[,"n_02"] + x[,"n_21"])*log10(6 - 2*r_r - r_r^2) + (x[,"n_01"] + x[,"n_22"])*log10(3 + 4*r_r - r_r^2) + (x[,"n_03"] + x[,"n_20"])*log10(3 - 4*r_r + r_r^2) + (x[,"n_11"] + x[,"n_12"])*log10(15 - 2*r_r + 2*r_r^2)
logL_r <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(2*log(2) + log(3) + log(5)) + (x[,"n_00"] + x[,"n_23"])*(log(pmax(1e-6,r_r)) + log(2 + r_r)) + (x[,"n_10"] + x[,"n_13"])*log(3 + 2*r_r - 2*r_r^2) + (x[,"n_02"] + x[,"n_21"])*log(6 - 2*r_r - r_r^2) + (x[,"n_01"] + x[,"n_22"])*log(3 + 4*r_r - r_r^2) + (x[,"n_03"] + x[,"n_20"])*log(3 - 4*r_r + r_r^2) + (x[,"n_11"] + x[,"n_12"])*log(15 - 2*r_r + 2*r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_2.2 <- function(x){
logL_c <- function(r,n10,n14,n00,n01,n02,n03,n04,n11,n12,n13,n20,n21,n22,n23,n24) {
L <- (n10 + n14)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + 2*(n00 + n24)*log(pmax(1e-6,1 - r)) + 2*(n04 + n20)*log(pmax(1e-6,r)) + (n10 + n14)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n21)*(log(pmax(1e-6,r)) + log(2 + r)) + (n02 + n22)*log(1 + 4*r - 4*r^2) + (n11 + n13)*log(3 + 2*r - 2*r^2) + (n01 + n23)*log(3 - 4*r + r^2) + n12*log(9 - 8*r + 8*r^2)
return(L)}
interlogL_c <- function(n10,n14,n00,n01,n02,n03,n04,n11,n12,n13,n20,n21,n22,n23,n24) {
optimize(logL_c,c(0,0.5), n10,n14,n00,n01,n02,n03,n04,n11,n12,n13,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_10"],x[,"n_14"],x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_c <- 2*(x[,"n_10"] + x[,"n_14"])*log10(2) + 2*(x[,"n_04"] + x[,"n_20"])*log10(2) - (x[,"n_02"] + x[,"n_22"])*log10(2) + 2*(x[,"n_00"] + x[,"n_24"])*log10(2) - (x[,"n_03"] + x[,"n_21"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_23"])*(-2*log10(2) + log10(5)) - x[,"n_12"]*log10(7) - (x[,"n_11"] + x[,"n_13"])*(-log10(2) + log10(7)) + 2*(x[,"n_00"] + x[,"n_24"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_04"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_14"])*(log10(pmax(1e-6,1 - r_c)) + log10(pmax(1e-6,r_c))) + (x[,"n_03"] + x[,"n_21"])*(log10(pmax(1e-6,r_c)) + log10(2 + r_c)) + (x[,"n_02"] + x[,"n_22"])*log10(1 + 4*r_c - 4*r_c^2) + (x[,"n_11"] + x[,"n_13"])*log10(3 + 2*r_c - 2*r_c^2) + (x[,"n_01"] + x[,"n_23"])*log10(3 - 4*r_c + r_c^2) + x[,"n_12"]*log10(9 - 8*r_c + 8*r_c^2)
logL_c <- (x[,"n_10"] + x[,"n_14"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + 2*(x[,"n_00"] + x[,"n_24"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_04"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_14"])*(log(pmax(1e-6,1 - r_c)) + log(pmax(1e-6,r_c))) + (x[,"n_03"] + x[,"n_21"])*(log(pmax(1e-6,r_c)) + log(2 + r_c)) + (x[,"n_02"] + x[,"n_22"])*log(1 + 4*r_c - 4*r_c^2) + (x[,"n_11"] + x[,"n_13"])*log(3 + 2*r_c - 2*r_c^2) + (x[,"n_01"] + x[,"n_23"])*log(3 - 4*r_c + r_c^2) + x[,"n_12"]*log(9 - 8*r_c + 8*r_c^2)
logL_m <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 + n12 - 2*n13 - 2*n14 - 2*n20 - 2*n21 - n22 - 2*n23 - 2*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + (n04 + n20)*(log(2 - r) + log(pmax(1e-6,r))) + (n00 + n24)*log(1 - r^2) + (n01 + n23)*log(6 - 2*r - r^2) + n12*log(3 + r - r^2) + (n03 + n21)*log(3 + 4*r - r^2) + (n10 + n14)*log(3 - 2*r + 2*r^2) + (n02 + n22)*log(5 - 2*r + 2*r^2) + (n11 + n13)*log(15 - 2*r + 2*r^2)
return(L)}
interlogL_m <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_m,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_m <- (x[,"n_00"] + x[,"n_24"])*(2*log10(2) - log10(3)) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(3)) - (x[,"n_02"] + x[,"n_22"])*(-log10(2) + 2*log10(3)) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(5)) - x[,"n_12"]*(-2*log10(2) + log10(13)) - (x[,"n_03"] + x[,"n_21"])*(-2*log10(2) + log10(19)) - (x[,"n_01"] + x[,"n_23"])*(-2*log10(2) + log10(19)) - (x[,"n_11"] + x[,"n_13"])*(-log10(2) + log10(29)) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - r_m) + log10(pmax(1e-6,r_m))) + (x[,"n_00"] + x[,"n_24"])*log10(1 - r_m^2) + (x[,"n_01"] + x[,"n_23"])*log10(6 - 2*r_m - r_m^2) + x[,"n_12"]*log10(3 + r_m - r_m^2) + (x[,"n_03"] + x[,"n_21"])*log10(3 + 4*r_m - r_m^2) + (x[,"n_10"] + x[,"n_14"])*log10(3 - 2*r_m + 2*r_m^2) + (x[,"n_02"] + x[,"n_22"])*log10(5 - 2*r_m + 2*r_m^2) + (x[,"n_11"] + x[,"n_13"])*log10(15 - 2*r_m + 2*r_m^2)
logL_m <- (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] + x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_04"] + x[,"n_20"])*(log(2 - r_m) + log(pmax(1e-6,r_m))) + (x[,"n_00"] + x[,"n_24"])*log(1 - r_m^2) + (x[,"n_01"] + x[,"n_23"])*log(6 - 2*r_m - r_m^2) + x[,"n_12"]*log(3 + r_m - r_m^2) + (x[,"n_03"] + x[,"n_21"])*log(3 + 4*r_m - r_m^2) + (x[,"n_10"] + x[,"n_14"])*log(3 - 2*r_m + 2*r_m^2) + (x[,"n_02"] + x[,"n_22"])*log(5 - 2*r_m + 2*r_m^2) + (x[,"n_11"] + x[,"n_13"])*log(15 - 2*r_m + 2*r_m^2)
logL_r <- function(r,n00,n01,n02,n03,n04,n10,n11,n13,n14,n20,n21,n22,n23,n24,n12) {
L <- (-n00 - n01 + n02 - n03 - n04 - n10 - n11 - n13 - n14 - n20 - n21 + n22 - n23 - n24)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*(log(3) + 2*log(5)) + (n00 + n24)*(log(pmax(1e-6,r)) + log(2 + r)) + (n10 + n14)*log(3 + 2*r - 2*r^2) + (n11 + n13)*log(21 + 2*r - 2*r^2) + (n02 + n22)*log(3 + r - r^2) + (n03 + n21)*log(12 - 10*r + r^2) + (n04 + n20)*log(3 - 4*r + r^2) + (n01 + n23)*log(3 + 8*r + r^2) + n12*log(21 - 4*r + 4*r^2)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n04,n10,n11,n13,n14,n20,n21,n22,n23,n24,n12) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n13,n14,n20,n21,n22,n23,n24,n12, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_12"])
LOD_r <- -((x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - x[,"n_12"]*(2*log10(2) + log10(5)) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(7)) - (x[,"n_02"] + x[,"n_22"])*(-2*log10(2) + log10(13)) - (x[,"n_03"] + x[,"n_21"])*(-2*log10(2) + log10(29)) - (x[,"n_01"] + x[,"n_23"])*(-2*log10(2) + log10(29)) - (x[,"n_11"] + x[,"n_13"])*(-log10(2) + log10(43)) + (x[,"n_00"] + x[,"n_24"])*(log10(pmax(1e-6,r_r)) + log10(2 + r_r)) + (x[,"n_10"] + x[,"n_14"])*log10(3 + 2*r_r - 2*r_r^2) + (x[,"n_11"] + x[,"n_13"])*log10(21 + 2*r_r - 2*r_r^2) + (x[,"n_02"] + x[,"n_22"])*log10(3 + r_r - r_r^2) + (x[,"n_03"] + x[,"n_21"])*log10(12 - 10*r_r + r_r^2) + (x[,"n_04"] + x[,"n_20"])*log10(3 - 4*r_r + r_r^2) + (x[,"n_01"] + x[,"n_23"])*log10(3 + 8*r_r + r_r^2) + x[,"n_12"]*log10(21 - 4*r_r + 4*r_r^2)
logL_r <- (-x[,"n_00"] - x[,"n_01"] + x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] + x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*(log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_24"])*(log(pmax(1e-6,r_r)) + log(2 + r_r)) + (x[,"n_10"] + x[,"n_14"])*log(3 + 2*r_r - 2*r_r^2) + (x[,"n_11"] + x[,"n_13"])*log(21 + 2*r_r - 2*r_r^2) + (x[,"n_02"] + x[,"n_22"])*log(3 + r_r - r_r^2) + (x[,"n_03"] + x[,"n_21"])*log(12 - 10*r_r + r_r^2) + (x[,"n_04"] + x[,"n_20"])*log(3 - 4*r_r + r_r^2) + (x[,"n_01"] + x[,"n_23"])*log(3 + 8*r_r + r_r^2) + x[,"n_12"]*log(21 - 4*r_r + 4*r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_2.3 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n05 - n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - n15 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24 - 2*n25)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(5) + 2*(n00 + n25)*log(pmax(1e-6,1 - r)) + 2*(n05 + n20)*log(pmax(1e-6,r)) + (n10 + n15)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n21)*(log(pmax(1e-6,r)) + log(2 + 7*r)) + (n11 + n14)*log(3 + 14*r - 14*r^2) + (n02 + n23)*log(9 - 8*r^2) + (n03 + n22)*log(1 + 16*r - 8*r^2) + (n01 + n24)*log(9 - 16*r + 7*r^2) + (n12 + n13)*log(27 - 16*r + 16*r^2)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_c <- 2*(x[,"n_10"] + x[,"n_15"])*log10(2) + 2*(x[,"n_05"] + x[,"n_20"])*log10(2) + 2*(x[,"n_00"] + x[,"n_25"])*log10(2) - (x[,"n_03"] + x[,"n_22"])*log10(7) - (x[,"n_02"] + x[,"n_23"])*log10(7) - (x[,"n_04"] + x[,"n_21"])*(-2*log10(2) + log10(11)) - (x[,"n_01"] + x[,"n_24"])*(-2*log10(2) + log10(11)) - (x[,"n_11"] + x[,"n_14"])*(-log10(2) + log10(13)) - (x[,"n_12"] + x[,"n_13"])*log10(23) + 2*(x[,"n_00"] + x[,"n_25"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_05"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_15"])*(log10(pmax(1e-6,1 - r_c)) + log10(pmax(1e-6,r_c))) + (x[,"n_04"] + x[,"n_21"])*(log10(pmax(1e-6,r_c)) + log10(2 + 7*r_c)) + (x[,"n_11"] + x[,"n_14"])*log10(3 + 14*r_c - 14*r_c^2) + (x[,"n_02"] + x[,"n_23"])*log10(9 - 8*r_c^2) + (x[,"n_03"] + x[,"n_22"])*log10(1 + 16*r_c - 8*r_c^2) + (x[,"n_01"] + x[,"n_24"])*log10(9 - 16*r_c + 7*r_c^2) + (x[,"n_12"] + x[,"n_13"])*log10(27 - 16*r_c + 16*r_c^2)
logL_c <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_15"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + 2*(x[,"n_00"] + x[,"n_25"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_05"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_15"])*(log(pmax(1e-6,1 - r_c)) + log(pmax(1e-6,r_c))) + (x[,"n_04"] + x[,"n_21"])*(log(pmax(1e-6,r_c)) + log(2 + 7*r_c)) + (x[,"n_11"] + x[,"n_14"])*log(3 + 14*r_c - 14*r_c^2) + (x[,"n_02"] + x[,"n_23"])*log(9 - 8*r_c^2) + (x[,"n_03"] + x[,"n_22"])*log(1 + 16*r_c - 8*r_c^2) + (x[,"n_01"] + x[,"n_24"])*log(9 - 16*r_c + 7*r_c^2) + (x[,"n_12"] + x[,"n_13"])*log(27 - 16*r_c + 16*r_c^2)
logL_m <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- 2*(-2*n00 - 2*n01 - n02 - n03 - 2*n04 - 2*n05 - 2*n10 - 2*n11 - n12 - n13 - 2*n14 - 2*n15 - 2*n20 - 2*n21 - n22 - n23 - 2*n24 - 2*n25)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(5) + (n05 + n20)*(log(2 - r) + log(pmax(1e-6,r))) + (n01 + n24)*log(12 - 2*r - 7*r^2) + (n04 + n21)*log(3 + 16*r - 7*r^2) + (n12 + n13)*log(21 + 4*r - 4*r^2) + (n00 + n25)*log(1 - r^2) + (n03 + n22)*log(7 + 2*r^2) + (n02 + n23)*log(9 - 4*r + 2*r^2) + (n10 + n15)*log(3 - 2*r + 2*r^2) + (n11 + n14)*log(33 - 14*r + 14*r^2)
return(L)}
interlogL_m <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_m,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_m <- (x[,"n_00"] + x[,"n_25"])*(2*log10(2) - log10(3)) - (x[,"n_05"] + x[,"n_20"])*(-2*log10(2) + log10(3)) - (x[,"n_10"] + x[,"n_15"])*(-log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_22"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_02"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(5)) - (x[,"n_12"] + x[,"n_13"])*(log10(2) + log10(11)) - (x[,"n_04"] + x[,"n_21"])*(-2*log10(2) + log10(37)) - (x[,"n_01"] + x[,"n_24"])*(-2*log10(2) + log10(37)) - (x[,"n_11"] + x[,"n_14"])*(-log10(2) + log10(59)) + (x[,"n_05"] + x[,"n_20"])*(log10(2 - r_m) + log10(pmax(1e-6,r_m))) + (x[,"n_01"] + x[,"n_24"])*log10(12 - 2*r_m - 7*r_m^2) + (x[,"n_04"] + x[,"n_21"])*log10(3 + 16*r_m - 7*r_m^2) + (x[,"n_12"] + x[,"n_13"])*log10(21 + 4*r_m - 4*r_m^2) + (x[,"n_00"] + x[,"n_25"])*log10(1 - r_m^2) + (x[,"n_03"] + x[,"n_22"])*log10(7 + 2*r_m^2) + (x[,"n_02"] + x[,"n_23"])*log10(9 - 4*r_m + 2*r_m^2) + (x[,"n_10"] + x[,"n_15"])*log10(3 - 2*r_m + 2*r_m^2) + (x[,"n_11"] + x[,"n_14"])*log10(33 - 14*r_m + 14*r_m^2)
logL_m <- 2*(-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_05"] + x[,"n_20"])*(log(2 - r_m) + log(pmax(1e-6,r_m))) + (x[,"n_01"] + x[,"n_24"])*log(12 - 2*r_m - 7*r_m^2) + (x[,"n_04"] + x[,"n_21"])*log(3 + 16*r_m - 7*r_m^2) + (x[,"n_12"] + x[,"n_13"])*log(21 + 4*r_m - 4*r_m^2) + (x[,"n_00"] + x[,"n_25"])*log(1 - r_m^2) + (x[,"n_03"] + x[,"n_22"])*log(7 + 2*r_m^2) + (x[,"n_02"] + x[,"n_23"])*log(9 - 4*r_m + 2*r_m^2) + (x[,"n_10"] + x[,"n_15"])*log(3 - 2*r_m + 2*r_m^2) + (x[,"n_11"] + x[,"n_14"])*log(33 - 14*r_m + 14*r_m^2)
logL_r <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- (-3*n00 - 3*n01 - 2*n02 - 2*n03 - 3*n04 - 3*n05 - 3*n10 - 3*n11 - 2*n12 - 2*n13 - 3*n14 - 3*n15 - 3*n20 - 3*n21 - 2*n22 - 2*n23 - 3*n24 - 3*n25)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*(log(3) + 2*log(5)) + (n00 + n25)*(log(pmax(1e-6,r)) + log(2 + r)) + (n11 + n14)*log(39 + 14*r - 14*r^2) + (n03 + n22)*log(27 - 8*r - 4*r^2) + (n02 + n23)*log(15 + 16*r - 4*r^2) + (n10 + n15)*log(3 + 2*r - 2*r^2) + (n05 + n20)*log(3 - 4*r + r^2) + (n04 + n21)*log(30 - 34*r + 7*r^2) + (n01 + n24)*log(3 + 20*r + 7*r^2) + (n12 + n13)*log(69 - 8*r + 8*r^2)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_r <- -((x[,"n_05"] + x[,"n_20"])*(-2*log10(2) + log10(5))) - (x[,"n_00"] + x[,"n_25"])*(-2*log10(2) + log10(5)) - (x[,"n_10"] + x[,"n_15"])*(-log10(2) + log10(7)) - (x[,"n_03"] + x[,"n_22"])*(log10(2) + log10(11)) - (x[,"n_02"] + x[,"n_23"])*(log10(2) + log10(11)) - (x[,"n_11"] + x[,"n_14"])*(-log10(2) + log10(5) + log10(17)) - (x[,"n_04"] + x[,"n_21"])*(-2*log10(2) + log10(59)) - (x[,"n_01"] + x[,"n_24"])*(-2*log10(2) + log10(59)) - (x[,"n_12"] + x[,"n_13"])*log10(67) + (x[,"n_00"] + x[,"n_25"])*(log10(pmax(1e-6,r_r)) + log10(2 + r_r)) + (x[,"n_11"] + x[,"n_14"])*log10(39 + 14*r_r - 14*r_r^2) + (x[,"n_03"] + x[,"n_22"])*log10(27 - 8*r_r - 4*r_r^2) + (x[,"n_02"] + x[,"n_23"])*log10(15 + 16*r_r - 4*r_r^2) + (x[,"n_10"] + x[,"n_15"])*log10(3 + 2*r_r - 2*r_r^2) + (x[,"n_05"] + x[,"n_20"])*log10(3 - 4*r_r + r_r^2) + (x[,"n_04"] + x[,"n_21"])*log10(30 - 34*r_r + 7*r_r^2) + (x[,"n_01"] + x[,"n_24"])*log10(3 + 20*r_r + 7*r_r^2) + (x[,"n_12"] + x[,"n_13"])*log10(69 - 8*r_r + 8*r_r^2)
logL_r <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_05"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"] - 3*x[,"n_25"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*(log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_25"])*(log(pmax(1e-6,r_r)) + log(2 + r_r)) + (x[,"n_11"] + x[,"n_14"])*log(39 + 14*r_r - 14*r_r^2) + (x[,"n_03"] + x[,"n_22"])*log(27 - 8*r_r - 4*r_r^2) + (x[,"n_02"] + x[,"n_23"])*log(15 + 16*r_r - 4*r_r^2) + (x[,"n_10"] + x[,"n_15"])*log(3 + 2*r_r - 2*r_r^2) + (x[,"n_05"] + x[,"n_20"])*log(3 - 4*r_r + r_r^2) + (x[,"n_04"] + x[,"n_21"])*log(30 - 34*r_r + 7*r_r^2) + (x[,"n_01"] + x[,"n_24"])*log(3 + 20*r_r + 7*r_r^2) + (x[,"n_12"] + x[,"n_13"])*log(69 - 8*r_r + 8*r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_2.4 <- function(x){
logL_c <- function(r,n11,n15,n01,n02,n03,n04,n05,n12,n13,n14,n21,n22,n23,n24,n25) {
L <- (n11 + n15)*log(2) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25)*log(5) + 2*(n01 + n25)*log(pmax(1e-6,1 - r)) + 2*(n05 + n21)*log(pmax(1e-6,r)) + (n11 + n15)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n22)*(log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n23)*log(1 + 4*r - 4*r^2) + (n12 + n14)*log(3 + 2*r - 2*r^2) + (n02 + n24)*log(3 - 4*r + r^2) + n13*log(9 - 8*r + 8*r^2)
return(L)}
interlogL_c <- function(n11,n15,n01,n02,n03,n04,n05,n12,n13,n14,n21,n22,n23,n24,n25) {
optimize(logL_c,c(0,0.5), n11,n15,n01,n02,n03,n04,n05,n12,n13,n14,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_11"],x[,"n_15"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_c <- 2*(x[,"n_11"] + x[,"n_15"])*log10(2) + 2*(x[,"n_05"] + x[,"n_21"])*log10(2) - (x[,"n_03"] + x[,"n_23"])*log10(2) + 2*(x[,"n_01"] + x[,"n_25"])*log10(2) - (x[,"n_04"] + x[,"n_22"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - x[,"n_13"]*log10(7) - (x[,"n_12"] + x[,"n_14"])*(-log10(2) + log10(7)) + 2*(x[,"n_01"] + x[,"n_25"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_05"] + x[,"n_21"])*log10(pmax(1e-6,r_c)) + (x[,"n_11"] + x[,"n_15"])*(log10(pmax(1e-6,1 - r_c)) + log10(pmax(1e-6,r_c))) + (x[,"n_04"] + x[,"n_22"])*(log10(pmax(1e-6,r_c)) + log10(2 + r_c)) + (x[,"n_03"] + x[,"n_23"])*log10(1 + 4*r_c - 4*r_c^2) + (x[,"n_12"] + x[,"n_14"])*log10(3 + 2*r_c - 2*r_c^2) + (x[,"n_02"] + x[,"n_24"])*log10(3 - 4*r_c + r_c^2) + x[,"n_13"]*log10(9 - 8*r_c + 8*r_c^2)
logL_c <- (x[,"n_11"] + x[,"n_15"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + 2*(x[,"n_01"] + x[,"n_25"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_05"] + x[,"n_21"])*log(pmax(1e-6,r_c)) + (x[,"n_11"] + x[,"n_15"])*(log(pmax(1e-6,1 - r_c)) + log(pmax(1e-6,r_c))) + (x[,"n_04"] + x[,"n_22"])*(log(pmax(1e-6,r_c)) + log(2 + r_c)) + (x[,"n_03"] + x[,"n_23"])*log(1 + 4*r_c - 4*r_c^2) + (x[,"n_12"] + x[,"n_14"])*log(3 + 2*r_c - 2*r_c^2) + (x[,"n_02"] + x[,"n_24"])*log(3 - 4*r_c + r_c^2) + x[,"n_13"]*log(9 - 8*r_c + 8*r_c^2)
logL_m <- function(r,n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25) {
L <- (-2*n01 - 2*n02 - n03 - 2*n04 - 2*n05 - 2*n11 - 2*n12 + n13 - 2*n14 - 2*n15 - 2*n21 - 2*n22 - n23 - 2*n24 - 2*n25)*log(2) + 2*(-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25)*log(5) + (n05 + n21)*(log(2 - r) + log(pmax(1e-6,r))) + (n01 + n25)*log(1 - r^2) + (n02 + n24)*log(6 - 2*r - r^2) + n13*log(3 + r - r^2) + (n04 + n22)*log(3 + 4*r - r^2) + (n11 + n15)*log(3 - 2*r + 2*r^2) + (n03 + n23)*log(5 - 2*r + 2*r^2) + (n12 + n14)*log(15 - 2*r + 2*r^2)
return(L)}
interlogL_m <- function(n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25) {
optimize(logL_m,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n13,n14,n15,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_m <- (x[,"n_01"] + x[,"n_25"])*(2*log10(2) - log10(3)) - (x[,"n_05"] + x[,"n_21"])*(-2*log10(2) + log10(3)) - (x[,"n_03"] + x[,"n_23"])*(-log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_15"])*(-log10(2) + log10(5)) - x[,"n_13"]*(-2*log10(2) + log10(13)) - (x[,"n_04"] + x[,"n_22"])*(-2*log10(2) + log10(19)) - (x[,"n_02"] + x[,"n_24"])*(-2*log10(2) + log10(19)) - (x[,"n_12"] + x[,"n_14"])*(-log10(2) + log10(29)) + (x[,"n_05"] + x[,"n_21"])*(log10(2 - r_m) + log10(pmax(1e-6,r_m))) + (x[,"n_01"] + x[,"n_25"])*log10(1 - r_m^2) + (x[,"n_02"] + x[,"n_24"])*log10(6 - 2*r_m - r_m^2) + x[,"n_13"]*log10(3 + r_m - r_m^2) + (x[,"n_04"] + x[,"n_22"])*log10(3 + 4*r_m - r_m^2) + (x[,"n_11"] + x[,"n_15"])*log10(3 - 2*r_m + 2*r_m^2) + (x[,"n_03"] + x[,"n_23"])*log10(5 - 2*r_m + 2*r_m^2) + (x[,"n_12"] + x[,"n_14"])*log10(15 - 2*r_m + 2*r_m^2)
logL_m <- (-2*x[,"n_01"] - 2*x[,"n_02"] - x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_05"] - 2*x[,"n_11"] - 2*x[,"n_12"] + x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_15"] - 2*x[,"n_21"] - 2*x[,"n_22"] - x[,"n_23"] - 2*x[,"n_24"] - 2*x[,"n_25"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_05"] + x[,"n_21"])*(log(2 - r_m) + log(pmax(1e-6,r_m))) + (x[,"n_01"] + x[,"n_25"])*log(1 - r_m^2) + (x[,"n_02"] + x[,"n_24"])*log(6 - 2*r_m - r_m^2) + x[,"n_13"]*log(3 + r_m - r_m^2) + (x[,"n_04"] + x[,"n_22"])*log(3 + 4*r_m - r_m^2) + (x[,"n_11"] + x[,"n_15"])*log(3 - 2*r_m + 2*r_m^2) + (x[,"n_03"] + x[,"n_23"])*log(5 - 2*r_m + 2*r_m^2) + (x[,"n_12"] + x[,"n_14"])*log(15 - 2*r_m + 2*r_m^2)
logL_r <- function(r,n01,n02,n03,n04,n05,n11,n12,n14,n15,n21,n22,n23,n24,n25,n13) {
L <- (-n01 - n02 + n03 - n04 - n05 - n11 - n12 - n14 - n15 - n21 - n22 + n23 - n24 - n25)*log(2) + (-n01 - n02 - n03 - n04 - n05 - n11 - n12 - n13 - n14 - n15 - n21 - n22 - n23 - n24 - n25)*(log(3) + 2*log(5)) + (n01 + n25)*(log(pmax(1e-6,r)) + log(2 + r)) + (n11 + n15)*log(3 + 2*r - 2*r^2) + (n12 + n14)*log(21 + 2*r - 2*r^2) + (n03 + n23)*log(3 + r - r^2) + (n04 + n22)*log(12 - 10*r + r^2) + (n05 + n21)*log(3 - 4*r + r^2) + (n02 + n24)*log(3 + 8*r + r^2) + n13*log(21 - 4*r + 4*r^2)
return(L)}
interlogL_r <- function(n01,n02,n03,n04,n05,n11,n12,n14,n15,n21,n22,n23,n24,n25,n13) {
optimize(logL_r,c(0,0.5), n01,n02,n03,n04,n05,n11,n12,n14,n15,n21,n22,n23,n24,n25,n13, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_11"],x[,"n_12"],x[,"n_14"],x[,"n_15"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_13"])
LOD_r <- -((x[,"n_05"] + x[,"n_21"])*(-2*log10(2) + log10(5))) - (x[,"n_01"] + x[,"n_25"])*(-2*log10(2) + log10(5)) - x[,"n_13"]*(2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_15"])*(-log10(2) + log10(7)) - (x[,"n_03"] + x[,"n_23"])*(-2*log10(2) + log10(13)) - (x[,"n_04"] + x[,"n_22"])*(-2*log10(2) + log10(29)) - (x[,"n_02"] + x[,"n_24"])*(-2*log10(2) + log10(29)) - (x[,"n_12"] + x[,"n_14"])*(-log10(2) + log10(43)) + (x[,"n_01"] + x[,"n_25"])*(log10(pmax(1e-6,r_r)) + log10(2 + r_r)) + (x[,"n_11"] + x[,"n_15"])*log10(3 + 2*r_r - 2*r_r^2) + (x[,"n_12"] + x[,"n_14"])*log10(21 + 2*r_r - 2*r_r^2) + (x[,"n_03"] + x[,"n_23"])*log10(3 + r_r - r_r^2) + (x[,"n_04"] + x[,"n_22"])*log10(12 - 10*r_r + r_r^2) + (x[,"n_05"] + x[,"n_21"])*log10(3 - 4*r_r + r_r^2) + (x[,"n_02"] + x[,"n_24"])*log10(3 + 8*r_r + r_r^2) + x[,"n_13"]*log10(21 - 4*r_r + 4*r_r^2)
logL_r <- (-x[,"n_01"] - x[,"n_02"] + x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] + x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*(log(3) + 2*log(5)) + (x[,"n_01"] + x[,"n_25"])*(log(pmax(1e-6,r_r)) + log(2 + r_r)) + (x[,"n_11"] + x[,"n_15"])*log(3 + 2*r_r - 2*r_r^2) + (x[,"n_12"] + x[,"n_14"])*log(21 + 2*r_r - 2*r_r^2) + (x[,"n_03"] + x[,"n_23"])*log(3 + r_r - r_r^2) + (x[,"n_04"] + x[,"n_22"])*log(12 - 10*r_r + r_r^2) + (x[,"n_05"] + x[,"n_21"])*log(3 - 4*r_r + r_r^2) + (x[,"n_02"] + x[,"n_24"])*log(3 + 8*r_r + r_r^2) + x[,"n_13"]*log(21 - 4*r_r + 4*r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_3.0 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n03 - n10 - n11 - n12 - n13 - 2*n20 - 2*n21 - 2*n22 - 2*n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(5) + 2*(n00 + n23)*log(pmax(1e-6,1 - r)) + 2*(n03 + n20)*log(pmax(1e-6,r)) + (n10 + n13)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n21)*(log(4 - r) + log(pmax(1e-6,r))) + (n01 + n22)*log(3 - 2*r - r^2) + (n11 + n12)*log(3 - r + r^2)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_c <- 2*(x[,"n_10"] + x[,"n_13"])*log10(2) + 2*(x[,"n_03"] + x[,"n_20"])*log10(2) + 2*(x[,"n_00"] + x[,"n_23"])*log10(2) - (x[,"n_02"] + x[,"n_21"])*(-2*log10(2) + log10(7)) - (x[,"n_01"] + x[,"n_22"])*(-2*log10(2) + log10(7)) - (x[,"n_11"] + x[,"n_12"])*(-2*log10(2) + log10(11)) + 2*(x[,"n_00"] + x[,"n_23"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_03"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_13"])*(log10(pmax(1e-6,1 - r_c)) + log10(pmax(1e-6,r_c))) + (x[,"n_02"] + x[,"n_21"])*(log10(4 - r_c) + log10(pmax(1e-6,r_c))) + (x[,"n_01"] + x[,"n_22"])*log10(3 - 2*r_c - r_c^2) + (x[,"n_11"] + x[,"n_12"])*log10(3 - r_c + r_c^2)
logL_c <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(5) + 2*(x[,"n_00"] + x[,"n_23"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_03"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_13"])*(log(pmax(1e-6,1 - r_c)) + log(pmax(1e-6,r_c))) + (x[,"n_02"] + x[,"n_21"])*(log(4 - r_c) + log(pmax(1e-6,r_c))) + (x[,"n_01"] + x[,"n_22"])*log(3 - 2*r_c - r_c^2) + (x[,"n_11"] + x[,"n_12"])*log(3 - r_c + r_c^2)
logL_m <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
L <- (-n00 - n01 - n02 - n03 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - n20 - n21 - n22 - n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(log(3) + log(5)) + (n00 + n03 + n20 + n23)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n11 + n12)*log(15 + 4*r - 4*r^2) + (n01 + n02 + n21 + n22)*log(3 - r + r^2) + (n10 + n13)*log(3 - 4*r + 4*r^2)
return(L)}
interlogL_m <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
optimize(logL_m,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_m <- -4*(x[,"n_11"] + x[,"n_12"])*log10(2) - (x[,"n_10"] + x[,"n_13"])*log10(2) + 2*(x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*log10(2) - (x[,"n_01"] + x[,"n_02"] + x[,"n_21"] + x[,"n_22"])*(-2*log10(2) + log10(11)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_m)) + log10(pmax(1e-6,r_m))) + (x[,"n_11"] + x[,"n_12"])*log10(15 + 4*r_m - 4*r_m^2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_21"] + x[,"n_22"])*log10(3 - r_m + r_m^2) + (x[,"n_10"] + x[,"n_13"])*log10(3 - 4*r_m + 4*r_m^2)
logL_m <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(log(3) + log(5)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_m)) + log(pmax(1e-6,r_m))) + (x[,"n_11"] + x[,"n_12"])*log(15 + 4*r_m - 4*r_m^2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_21"] + x[,"n_22"])*log(3 - r_m + r_m^2) + (x[,"n_10"] + x[,"n_13"])*log(3 - 4*r_m + 4*r_m^2)
logL_r <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
L <- (-2*n00 - 2*n01 - 2*n02 - 2*n03 - n10 - n11 - n12 - n13 - 2*n20 - 2*n21 - 2*n22 - 2*n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(5) + 2*(n03 + n20)*log(pmax(1e-6,1 - r)) + 2*(n00 + n23)*log(pmax(1e-6,r)) + (n10 + n13)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n22)*(log(4 - r) + log(pmax(1e-6,r))) + (n02 + n21)*log(3 - 2*r - r^2) + (n11 + n12)*log(3 - r + r^2)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"])
LOD_r <- 2*(x[,"n_10"] + x[,"n_13"])*log10(2) + 2*(x[,"n_03"] + x[,"n_20"])*log10(2) + 2*(x[,"n_00"] + x[,"n_23"])*log10(2) - (x[,"n_02"] + x[,"n_21"])*(-2*log10(2) + log10(7)) - (x[,"n_01"] + x[,"n_22"])*(-2*log10(2) + log10(7)) - (x[,"n_11"] + x[,"n_12"])*(-2*log10(2) + log10(11)) + 2*(x[,"n_03"] + x[,"n_20"])*log10(pmax(1e-6,1 - r_r)) + 2*(x[,"n_00"] + x[,"n_23"])*log10(pmax(1e-6,r_r)) + (x[,"n_10"] + x[,"n_13"])*(log10(pmax(1e-6,1 - r_r)) + log10(pmax(1e-6,r_r))) + (x[,"n_01"] + x[,"n_22"])*(log10(4 - r_r) + log10(pmax(1e-6,r_r))) + (x[,"n_02"] + x[,"n_21"])*log10(3 - 2*r_r - r_r^2) + (x[,"n_11"] + x[,"n_12"])*log10(3 - r_r + r_r^2)
logL_r <- (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(5) + 2*(x[,"n_03"] + x[,"n_20"])*log(pmax(1e-6,1 - r_r)) + 2*(x[,"n_00"] + x[,"n_23"])*log(pmax(1e-6,r_r)) + (x[,"n_10"] + x[,"n_13"])*(log(pmax(1e-6,1 - r_r)) + log(pmax(1e-6,r_r))) + (x[,"n_01"] + x[,"n_22"])*(log(4 - r_r) + log(pmax(1e-6,r_r))) + (x[,"n_02"] + x[,"n_21"])*log(3 - 2*r_r - r_r^2) + (x[,"n_11"] + x[,"n_12"])*log(3 - r_r + r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_3.1 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-3*n00 - n01 - 3*n02 - n03 - 3*n04 - 2*n10 - 2*n11 - n12 - 2*n13 - 2*n14 - 3*n20 - n21 - 3*n22 - n23 - 3*n24)*log(2) + (n11 + n13)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + (n01 + n23)*log(pmax(1e-6,1 - r)) + 2*(n00 + n24)*log(pmax(1e-6,1 - r)) + 2*(n04 + n20)*log(pmax(1e-6,r)) + (n03 + n21)*log(pmax(1e-6,r)) + (n10 + n14)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n22)*log(3 + 2*r - 2*r^2) + n12*log(3 - r + r^2)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_c <- 2*(x[,"n_10"] + x[,"n_14"])*log10(2) + 2*(x[,"n_04"] + x[,"n_20"])*log10(2) + (x[,"n_03"] + x[,"n_21"])*log10(2) + (x[,"n_01"] + x[,"n_23"])*log10(2) + 2*(x[,"n_00"] + x[,"n_24"])*log10(2) - (x[,"n_02"] + x[,"n_22"])*(-log10(2) + log10(7)) - x[,"n_12"]*(-2*log10(2) + log10(11)) + (x[,"n_01"] + x[,"n_23"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_00"] + x[,"n_24"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_04"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_21"])*log10(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_14"])*(log10(pmax(1e-6,1 - r_c)) + log10(pmax(1e-6,r_c))) + (x[,"n_02"] + x[,"n_22"])*log10(3 + 2*r_c - 2*r_c^2) + x[,"n_12"]*log10(3 - r_c + r_c^2)
logL_c <- (-3*x[,"n_00"] - x[,"n_01"] - 3*x[,"n_02"] - x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_20"] - x[,"n_21"] - 3*x[,"n_22"] - x[,"n_23"] - 3*x[,"n_24"])*log(2) + (x[,"n_11"] + x[,"n_13"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_01"] + x[,"n_23"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_00"] + x[,"n_24"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_04"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_03"] + x[,"n_21"])*log(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_14"])*(log(pmax(1e-6,1 - r_c)) + log(pmax(1e-6,r_c))) + (x[,"n_02"] + x[,"n_22"])*log(3 + 2*r_c - 2*r_c^2) + x[,"n_12"]*log(3 - r_c + r_c^2)
logL_m <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 3*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n14 - 2*n20 - 2*n21 - n22 - 2*n23 - 2*n24)*log(2) + (-n00 - n02 - n04 - n10 + n11 - n12 + n13 - n14 - n20 - n22 - n24)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + (n00 + n04 + n20 + n24)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n12*log(15 + 4*r - 4*r^2) + (n02 + n22)*log(3 - r + r^2) + (n10 + n14)*log(3 - 4*r + 4*r^2)
return(L)}
interlogL_m <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_m,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_m <- -4*x[,"n_12"]*log10(2) - (x[,"n_10"] + x[,"n_14"])*log10(2) + 2*(x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*log10(2) - (x[,"n_02"] + x[,"n_22"])*(-2*log10(2) + log10(11)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(pmax(1e-6,1 - r_m)) + log10(pmax(1e-6,r_m))) + x[,"n_12"]*log10(15 + 4*r_m - 4*r_m^2) + (x[,"n_02"] + x[,"n_22"])*log10(3 - r_m + r_m^2) + (x[,"n_10"] + x[,"n_14"])*log10(3 - 4*r_m + 4*r_m^2)
logL_m <- (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(2) + (-x[,"n_00"] - x[,"n_02"] - x[,"n_04"] - x[,"n_10"] + x[,"n_11"] - x[,"n_12"] + x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_22"] - x[,"n_24"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(pmax(1e-6,1 - r_m)) + log(pmax(1e-6,r_m))) + x[,"n_12"]*log(15 + 4*r_m - 4*r_m^2) + (x[,"n_02"] + x[,"n_22"])*log(3 - r_m + r_m^2) + (x[,"n_10"] + x[,"n_14"])*log(3 - 4*r_m + 4*r_m^2)
logL_r <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
L <- (-3*n00 - n01 - 3*n02 - n03 - 3*n04 - 2*n10 - 2*n11 - n12 - 2*n13 - 2*n14 - 3*n20 - n21 - 3*n22 - n23 - 3*n24)*log(2) + (n11 + n13)*log(3) + (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(5) + 2*(n04 + n20)*log(pmax(1e-6,1 - r)) + (n03 + n21)*log(pmax(1e-6,1 - r)) + (n01 + n23)*log(pmax(1e-6,r)) + 2*(n00 + n24)*log(pmax(1e-6,r)) + (n10 + n14)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n22)*log(3 + 2*r - 2*r^2) + n12*log(3 - r + r^2)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_r <- 2*(x[,"n_10"] + x[,"n_14"])*log10(2) + 2*(x[,"n_04"] + x[,"n_20"])*log10(2) + (x[,"n_03"] + x[,"n_21"])*log10(2) + (x[,"n_01"] + x[,"n_23"])*log10(2) + 2*(x[,"n_00"] + x[,"n_24"])*log10(2) - (x[,"n_02"] + x[,"n_22"])*(-log10(2) + log10(7)) - x[,"n_12"]*(-2*log10(2) + log10(11)) + 2*(x[,"n_04"] + x[,"n_20"])*log10(pmax(1e-6,1 - r_r)) + (x[,"n_03"] + x[,"n_21"])*log10(pmax(1e-6,1 - r_r)) + (x[,"n_01"] + x[,"n_23"])*log10(pmax(1e-6,r_r)) + 2*(x[,"n_00"] + x[,"n_24"])*log10(pmax(1e-6,r_r)) + (x[,"n_10"] + x[,"n_14"])*(log10(pmax(1e-6,1 - r_r)) + log10(pmax(1e-6,r_r))) + (x[,"n_02"] + x[,"n_22"])*log10(3 + 2*r_r - 2*r_r^2) + x[,"n_12"]*log10(3 - r_r + r_r^2)
logL_r <- (-3*x[,"n_00"] - x[,"n_01"] - 3*x[,"n_02"] - x[,"n_03"] - 3*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_20"] - x[,"n_21"] - 3*x[,"n_22"] - x[,"n_23"] - 3*x[,"n_24"])*log(2) + (x[,"n_11"] + x[,"n_13"])*log(3) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + 2*(x[,"n_04"] + x[,"n_20"])*log(pmax(1e-6,1 - r_r)) + (x[,"n_03"] + x[,"n_21"])*log(pmax(1e-6,1 - r_r)) + (x[,"n_01"] + x[,"n_23"])*log(pmax(1e-6,r_r)) + 2*(x[,"n_00"] + x[,"n_24"])*log(pmax(1e-6,r_r)) + (x[,"n_10"] + x[,"n_14"])*(log(pmax(1e-6,1 - r_r)) + log(pmax(1e-6,r_r))) + (x[,"n_02"] + x[,"n_22"])*log(3 + 2*r_r - 2*r_r^2) + x[,"n_12"]*log(3 - r_r + r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_3.2 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- (-2*n00 - n01 - 2*n02 - 2*n03 - n04 - 2*n05 - n10 - n11 - n12 - n13 - n14 - n15 - 2*n20 - n21 - 2*n22 - 2*n23 - n24 - 2*n25)*log(2) + (n12 + n13)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(5) + 2*(n00 + n25)*log(pmax(1e-6,1 - r)) + 2*(n05 + n20)*log(pmax(1e-6,r)) + (n10 + n15)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n21)*(log(pmax(1e-6,r)) + log(2 + r)) + (n02 + n23)*log(10 - 4*r - 3*r^2) + (n03 + n22)*log(3 + 10*r - 3*r^2) + (n11 + n14)*log(3 + 2*r - 2*r^2) + (n01 + n24)*log(3 - 4*r + r^2) + (n12 + n13)*log(4 - r + r^2)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_c <- 2*(x[,"n_10"] + x[,"n_15"])*log10(2) + 2*(x[,"n_05"] + x[,"n_20"])*log10(2) + 2*(x[,"n_00"] + x[,"n_25"])*log10(2) - (x[,"n_04"] + x[,"n_21"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_13"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_11"] + x[,"n_14"])*(-log10(2) + log10(7)) - (x[,"n_03"] + x[,"n_22"])*(-2*log10(2) + log10(29)) - (x[,"n_02"] + x[,"n_23"])*(-2*log10(2) + log10(29)) + 2*(x[,"n_00"] + x[,"n_25"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_05"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_15"])*(log10(pmax(1e-6,1 - r_c)) + log10(pmax(1e-6,r_c))) + (x[,"n_04"] + x[,"n_21"])*(log10(pmax(1e-6,r_c)) + log10(2 + r_c)) + (x[,"n_02"] + x[,"n_23"])*log10(10 - 4*r_c - 3*r_c^2) + (x[,"n_03"] + x[,"n_22"])*log10(3 + 10*r_c - 3*r_c^2) + (x[,"n_11"] + x[,"n_14"])*log10(3 + 2*r_c - 2*r_c^2) + (x[,"n_01"] + x[,"n_24"])*log10(3 - 4*r_c + r_c^2) + (x[,"n_12"] + x[,"n_13"])*log10(4 - r_c + r_c^2)
logL_c <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - x[,"n_04"] - 2*x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_25"])*log(2) + (x[,"n_12"] + x[,"n_13"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + 2*(x[,"n_00"] + x[,"n_25"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_05"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_15"])*(log(pmax(1e-6,1 - r_c)) + log(pmax(1e-6,r_c))) + (x[,"n_04"] + x[,"n_21"])*(log(pmax(1e-6,r_c)) + log(2 + r_c)) + (x[,"n_02"] + x[,"n_23"])*log(10 - 4*r_c - 3*r_c^2) + (x[,"n_03"] + x[,"n_22"])*log(3 + 10*r_c - 3*r_c^2) + (x[,"n_11"] + x[,"n_14"])*log(3 + 2*r_c - 2*r_c^2) + (x[,"n_01"] + x[,"n_24"])*log(3 - 4*r_c + r_c^2) + (x[,"n_12"] + x[,"n_13"])*log(4 - r_c + r_c^2)
logL_m <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- (-n00 - n01 - n02 - n03 - n04 - n05 - 2*n10 + n11 - 2*n12 - 2*n13 + n14 - 2*n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(2) + (-n00 - n01 - n04 - n05 - n10 - n11 - n14 - n15 - n20 - n21 - n24 - n25)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(5) + (n00 + n05 + n20 + n25)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n13)*log(21 + 4*r - 4*r^2) + (n01 + n04 + n21 + n24)*log(3 + 2*r - 2*r^2) + (n11 + n14)*log(3 - r + r^2) + (n02 + n03 + n22 + n23)*log(4 - r + r^2) + (n10 + n15)*log(3 - 4*r + 4*r^2)
return(L)}
interlogL_m <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_m,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_m <- -((x[,"n_10"] + x[,"n_15"])*log10(2)) + 2*(x[,"n_00"] + x[,"n_05"] + x[,"n_20"] + x[,"n_25"])*log10(2) - (x[,"n_02"] + x[,"n_03"] + x[,"n_22"] + x[,"n_23"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_01"] + x[,"n_04"] + x[,"n_21"] + x[,"n_24"])*(-log10(2) + log10(7)) - (x[,"n_11"] + x[,"n_14"])*(-2*log10(2) + log10(11)) - (x[,"n_12"] + x[,"n_13"])*(log10(2) + log10(11)) + (x[,"n_00"] + x[,"n_05"] + x[,"n_20"] + x[,"n_25"])*(log10(pmax(1e-6,1 - r_m)) + log10(pmax(1e-6,r_m))) + (x[,"n_12"] + x[,"n_13"])*log10(21 + 4*r_m - 4*r_m^2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_21"] + x[,"n_24"])*log10(3 + 2*r_m - 2*r_m^2) + (x[,"n_11"] + x[,"n_14"])*log10(3 - r_m + r_m^2) + (x[,"n_02"] + x[,"n_03"] + x[,"n_22"] + x[,"n_23"])*log10(4 - r_m + r_m^2) + (x[,"n_10"] + x[,"n_15"])*log10(3 - 4*r_m + 4*r_m^2)
logL_m <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - 2*x[,"n_10"] + x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] + x[,"n_14"] - 2*x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_24"] - x[,"n_25"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + (x[,"n_00"] + x[,"n_05"] + x[,"n_20"] + x[,"n_25"])*(log(pmax(1e-6,1 - r_m)) + log(pmax(1e-6,r_m))) + (x[,"n_12"] + x[,"n_13"])*log(21 + 4*r_m - 4*r_m^2) + (x[,"n_01"] + x[,"n_04"] + x[,"n_21"] + x[,"n_24"])*log(3 + 2*r_m - 2*r_m^2) + (x[,"n_11"] + x[,"n_14"])*log(3 - r_m + r_m^2) + (x[,"n_02"] + x[,"n_03"] + x[,"n_22"] + x[,"n_23"])*log(4 - r_m + r_m^2) + (x[,"n_10"] + x[,"n_15"])*log(3 - 4*r_m + 4*r_m^2)
logL_r <- function(r,n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
L <- (-2*n00 - n01 - 2*n02 - 2*n03 - n04 - 2*n05 - n10 - n11 - n12 - n13 - n14 - n15 - 2*n20 - n21 - 2*n22 - 2*n23 - n24 - 2*n25)*log(2) + (n12 + n13)*log(3) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n10 - n11 - n12 - n13 - n14 - n15 - n20 - n21 - n22 - n23 - n24 - n25)*log(5) + 2*(n05 + n20)*log(pmax(1e-6,1 - r)) + 2*(n00 + n25)*log(pmax(1e-6,r)) + (n10 + n15)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n24)*(log(pmax(1e-6,r)) + log(2 + r)) + (n03 + n22)*log(10 - 4*r - 3*r^2) + (n02 + n23)*log(3 + 10*r - 3*r^2) + (n11 + n14)*log(3 + 2*r - 2*r^2) + (n04 + n21)*log(3 - 4*r + r^2) + (n12 + n13)*log(4 - r + r^2)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n04,n05,n10,n11,n12,n13,n14,n15,n20,n21,n22,n23,n24,n25, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"])
LOD_r <- 2*(x[,"n_10"] + x[,"n_15"])*log10(2) + 2*(x[,"n_05"] + x[,"n_20"])*log10(2) + 2*(x[,"n_00"] + x[,"n_25"])*log10(2) - (x[,"n_04"] + x[,"n_21"])*(-2*log10(2) + log10(5)) - (x[,"n_01"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_13"])*(-2*log10(2) + log10(3) + log10(5)) - (x[,"n_11"] + x[,"n_14"])*(-log10(2) + log10(7)) - (x[,"n_03"] + x[,"n_22"])*(-2*log10(2) + log10(29)) - (x[,"n_02"] + x[,"n_23"])*(-2*log10(2) + log10(29)) + 2*(x[,"n_05"] + x[,"n_20"])*log10(pmax(1e-6,1 - r_r)) + 2*(x[,"n_00"] + x[,"n_25"])*log10(pmax(1e-6,r_r)) + (x[,"n_10"] + x[,"n_15"])*(log10(pmax(1e-6,1 - r_r)) + log10(pmax(1e-6,r_r))) + (x[,"n_01"] + x[,"n_24"])*(log10(pmax(1e-6,r_r)) + log10(2 + r_r)) + (x[,"n_03"] + x[,"n_22"])*log10(10 - 4*r_r - 3*r_r^2) + (x[,"n_02"] + x[,"n_23"])*log10(3 + 10*r_r - 3*r_r^2) + (x[,"n_11"] + x[,"n_14"])*log10(3 + 2*r_r - 2*r_r^2) + (x[,"n_04"] + x[,"n_21"])*log10(3 - 4*r_r + r_r^2) + (x[,"n_12"] + x[,"n_13"])*log10(4 - r_r + r_r^2)
logL_r <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - x[,"n_04"] - 2*x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_25"])*log(2) + (x[,"n_12"] + x[,"n_13"])*log(3) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"])*log(5) + 2*(x[,"n_05"] + x[,"n_20"])*log(pmax(1e-6,1 - r_r)) + 2*(x[,"n_00"] + x[,"n_25"])*log(pmax(1e-6,r_r)) + (x[,"n_10"] + x[,"n_15"])*(log(pmax(1e-6,1 - r_r)) + log(pmax(1e-6,r_r))) + (x[,"n_01"] + x[,"n_24"])*(log(pmax(1e-6,r_r)) + log(2 + r_r)) + (x[,"n_03"] + x[,"n_22"])*log(10 - 4*r_r - 3*r_r^2) + (x[,"n_02"] + x[,"n_23"])*log(3 + 10*r_r - 3*r_r^2) + (x[,"n_11"] + x[,"n_14"])*log(3 + 2*r_r - 2*r_r^2) + (x[,"n_04"] + x[,"n_21"])*log(3 - 4*r_r + r_r^2) + (x[,"n_12"] + x[,"n_13"])*log(4 - r_r + r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_3.3 <- function(x){
logL_c <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
L <- (-4*n00 - 2*n01 - 4*n02 - 2*n03 - 4*n04 - 2*n05 - 4*n06 - 3*n10 - 3*n11 - 3*n12 - 2*n13 - 3*n14 - 3*n15 - 3*n16 - 4*n20 - 2*n21 - 4*n22 - 2*n23 - 4*n24 - 2*n25 - 4*n26)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26)*log(5) + 2*(n00 + n26)*log(pmax(1e-6,1 - r)) + 2*(n06 + n20)*log(pmax(1e-6,r)) + (n10 + n16)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n05 + n21)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n11 + n15)*log(3 + 8*r - 8*r^2) + (n03 + n23)*log(7 + 4*r - 4*r^2) + (n02 + n24)*log(36 - 32*r - r^2) + (n04 + n22)*log(3 + 34*r - r^2) + (n12 + n14)*log(30 - r + r^2) + (n01 + n25)*log(3 - 5*r + 2*r^2) + n13*log(27 - 8*r + 8*r^2)
return(L)}
interlogL_c <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
optimize(logL_c,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"])
LOD_c <- 2*(x[,"n_10"] + x[,"n_16"])*log10(2) + 2*(x[,"n_06"] + x[,"n_20"])*log10(2) - 3*(x[,"n_03"] + x[,"n_23"])*log10(2) + 2*(x[,"n_00"] + x[,"n_26"])*log10(2) - 2*x[,"n_13"]*log10(5) - (x[,"n_11"] + x[,"n_15"])*log10(5) - (x[,"n_12"] + x[,"n_14"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_04"] + x[,"n_22"])*(-2*log10(2) + log10(79)) - (x[,"n_02"] + x[,"n_24"])*(-2*log10(2) + log10(79)) + 2*(x[,"n_00"] + x[,"n_26"])*log10(pmax(1e-6,1 - r_c)) + 2*(x[,"n_06"] + x[,"n_20"])*log10(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_16"])*(log10(pmax(1e-6,1 - r_c)) + log10(pmax(1e-6,r_c))) + (x[,"n_05"] + x[,"n_21"])*(log10(pmax(1e-6,r_c)) + log10(1 + 2*r_c)) + (x[,"n_11"] + x[,"n_15"])*log10(3 + 8*r_c - 8*r_c^2) + (x[,"n_03"] + x[,"n_23"])*log10(7 + 4*r_c - 4*r_c^2) + (x[,"n_02"] + x[,"n_24"])*log10(36 - 32*r_c - r_c^2) + (x[,"n_04"] + x[,"n_22"])*log10(3 + 34*r_c - r_c^2) + (x[,"n_12"] + x[,"n_14"])*log10(30 - r_c + r_c^2) + (x[,"n_01"] + x[,"n_25"])*log10(3 - 5*r_c + 2*r_c^2) + x[,"n_13"]*log10(27 - 8*r_c + 8*r_c^2)
logL_c <- (-4*x[,"n_00"] - 2*x[,"n_01"] - 4*x[,"n_02"] - 2*x[,"n_03"] - 4*x[,"n_04"] - 2*x[,"n_05"] - 4*x[,"n_06"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 3*x[,"n_16"] - 4*x[,"n_20"] - 2*x[,"n_21"] - 4*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 2*x[,"n_25"] - 4*x[,"n_26"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"])*log(5) + 2*(x[,"n_00"] + x[,"n_26"])*log(pmax(1e-6,1 - r_c)) + 2*(x[,"n_06"] + x[,"n_20"])*log(pmax(1e-6,r_c)) + (x[,"n_10"] + x[,"n_16"])*(log(pmax(1e-6,1 - r_c)) + log(pmax(1e-6,r_c))) + (x[,"n_05"] + x[,"n_21"])*(log(pmax(1e-6,r_c)) + log(1 + 2*r_c)) + (x[,"n_11"] + x[,"n_15"])*log(3 + 8*r_c - 8*r_c^2) + (x[,"n_03"] + x[,"n_23"])*log(7 + 4*r_c - 4*r_c^2) + (x[,"n_02"] + x[,"n_24"])*log(36 - 32*r_c - r_c^2) + (x[,"n_04"] + x[,"n_22"])*log(3 + 34*r_c - r_c^2) + (x[,"n_12"] + x[,"n_14"])*log(30 - r_c + r_c^2) + (x[,"n_01"] + x[,"n_25"])*log(3 - 5*r_c + 2*r_c^2) + x[,"n_13"]*log(27 - 8*r_c + 8*r_c^2)
logL_m <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
L <- (-3*n00 - 3*n01 - 3*n02 - 2*n03 - 3*n04 - 3*n05 - 3*n06 - 4*n10 - 3*n11 - 4*n12 - 2*n13 - 4*n14 - 3*n15 - 4*n16 - 3*n20 - 3*n21 - 3*n22 - 2*n23 - 3*n24 - 3*n25 - 3*n26)*log(2) + (-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26)*(log(3) + 2*log(5)) + (n00 + n06 + n20 + n26)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n13*log(69 + 16*r - 16*r^2) + (n01 + n05 + n21 + n25)*log(3 + 8*r - 8*r^2) + (n12 + n14)*log(177 + 4*r - 4*r^2) + (n02 + n04 + n22 + n24)*log(30 - r + r^2) + (n10 + n16)*log(3 - 4*r + 4*r^2) + (n03 + n23)*log(27 - 8*r + 8*r^2) + (n11 + n15)*log(21 - 16*r + 16*r^2)
return(L)}
interlogL_m <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
optimize(logL_m,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"])
LOD_m <- -((x[,"n_10"] + x[,"n_16"])*log10(2)) + 2*(x[,"n_00"] + x[,"n_06"] + x[,"n_20"] + x[,"n_26"])*log10(2) - 2*(x[,"n_03"] + x[,"n_23"])*log10(5) - (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*log10(5) - (x[,"n_11"] + x[,"n_15"])*log10(17) - (x[,"n_02"] + x[,"n_04"] + x[,"n_22"] + x[,"n_24"])*(-2*log10(2) + log10(7) + log10(17)) - x[,"n_13"]*log10(73) - (x[,"n_12"] + x[,"n_14"])*(log10(2) + log10(89)) + (x[,"n_00"] + x[,"n_06"] + x[,"n_20"] + x[,"n_26"])*(log10(pmax(1e-6,1 - r_m)) + log10(pmax(1e-6,r_m))) + x[,"n_13"]*log10(69 + 16*r_m - 16*r_m^2) + (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*log10(3 + 8*r_m - 8*r_m^2) + (x[,"n_12"] + x[,"n_14"])*log10(177 + 4*r_m - 4*r_m^2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_22"] + x[,"n_24"])*log10(30 - r_m + r_m^2) + (x[,"n_10"] + x[,"n_16"])*log10(3 - 4*r_m + 4*r_m^2) + (x[,"n_03"] + x[,"n_23"])*log10(27 - 8*r_m + 8*r_m^2) + (x[,"n_11"] + x[,"n_15"])*log10(21 - 16*r_m + 16*r_m^2)
logL_m <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_03"] - 3*x[,"n_04"] - 3*x[,"n_05"] - 3*x[,"n_06"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 2*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_15"] - 4*x[,"n_16"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"] - 3*x[,"n_25"] - 3*x[,"n_26"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"])*(log(3) + 2*log(5)) + (x[,"n_00"] + x[,"n_06"] + x[,"n_20"] + x[,"n_26"])*(log(pmax(1e-6,1 - r_m)) + log(pmax(1e-6,r_m))) + x[,"n_13"]*log(69 + 16*r_m - 16*r_m^2) + (x[,"n_01"] + x[,"n_05"] + x[,"n_21"] + x[,"n_25"])*log(3 + 8*r_m - 8*r_m^2) + (x[,"n_12"] + x[,"n_14"])*log(177 + 4*r_m - 4*r_m^2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_22"] + x[,"n_24"])*log(30 - r_m + r_m^2) + (x[,"n_10"] + x[,"n_16"])*log(3 - 4*r_m + 4*r_m^2) + (x[,"n_03"] + x[,"n_23"])*log(27 - 8*r_m + 8*r_m^2) + (x[,"n_11"] + x[,"n_15"])*log(21 - 16*r_m + 16*r_m^2)
logL_r <- function(r,n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
L <- (-4*n00 - 2*n01 - 4*n02 - 2*n03 - 4*n04 - 2*n05 - 4*n06 - 3*n10 - 3*n11 - 3*n12 - 2*n13 - 3*n14 - 3*n15 - 3*n16 - 4*n20 - 2*n21 - 4*n22 - 2*n23 - 4*n24 - 2*n25 - 4*n26)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n05 - n06 - n10 - n11 - n12 - n13 - n14 - n15 - n16 - n20 - n21 - n22 - n23 - n24 - n25 - n26)*log(5) + 2*(n06 + n20)*log(pmax(1e-6,1 - r)) + 2*(n00 + n26)*log(pmax(1e-6,r)) + (n10 + n16)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n25)*(log(pmax(1e-6,r)) + log(1 + 2*r)) + (n11 + n15)*log(3 + 8*r - 8*r^2) + (n03 + n23)*log(7 + 4*r - 4*r^2) + (n04 + n22)*log(36 - 32*r - r^2) + (n02 + n24)*log(3 + 34*r - r^2) + (n12 + n14)*log(30 - r + r^2) + (n05 + n21)*log(3 - 5*r + 2*r^2) + n13*log(27 - 8*r + 8*r^2)
return(L)}
interlogL_r <- function(n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26) {
optimize(logL_r,c(0,0.5), n00,n01,n02,n03,n04,n05,n06,n10,n11,n12,n13,n14,n15,n16,n20,n21,n22,n23,n24,n25,n26, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_05"],x[,"n_06"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_15"],x[,"n_16"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_25"],x[,"n_26"])
LOD_r <- 2*(x[,"n_10"] + x[,"n_16"])*log10(2) + 2*(x[,"n_06"] + x[,"n_20"])*log10(2) - 3*(x[,"n_03"] + x[,"n_23"])*log10(2) + 2*(x[,"n_00"] + x[,"n_26"])*log10(2) - 2*x[,"n_13"]*log10(5) - (x[,"n_11"] + x[,"n_15"])*log10(5) - (x[,"n_12"] + x[,"n_14"])*(-2*log10(2) + log10(7) + log10(17)) - (x[,"n_04"] + x[,"n_22"])*(-2*log10(2) + log10(79)) - (x[,"n_02"] + x[,"n_24"])*(-2*log10(2) + log10(79)) + 2*(x[,"n_06"] + x[,"n_20"])*log10(pmax(1e-6,1 - r_r)) + 2*(x[,"n_00"] + x[,"n_26"])*log10(pmax(1e-6,r_r)) + (x[,"n_10"] + x[,"n_16"])*(log10(pmax(1e-6,1 - r_r)) + log10(pmax(1e-6,r_r))) + (x[,"n_01"] + x[,"n_25"])*(log10(pmax(1e-6,r_r)) + log10(1 + 2*r_r)) + (x[,"n_11"] + x[,"n_15"])*log10(3 + 8*r_r - 8*r_r^2) + (x[,"n_03"] + x[,"n_23"])*log10(7 + 4*r_r - 4*r_r^2) + (x[,"n_04"] + x[,"n_22"])*log10(36 - 32*r_r - r_r^2) + (x[,"n_02"] + x[,"n_24"])*log10(3 + 34*r_r - r_r^2) + (x[,"n_12"] + x[,"n_14"])*log10(30 - r_r + r_r^2) + (x[,"n_05"] + x[,"n_21"])*log10(3 - 5*r_r + 2*r_r^2) + x[,"n_13"]*log10(27 - 8*r_r + 8*r_r^2)
logL_r <- (-4*x[,"n_00"] - 2*x[,"n_01"] - 4*x[,"n_02"] - 2*x[,"n_03"] - 4*x[,"n_04"] - 2*x[,"n_05"] - 4*x[,"n_06"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 3*x[,"n_15"] - 3*x[,"n_16"] - 4*x[,"n_20"] - 2*x[,"n_21"] - 4*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 2*x[,"n_25"] - 4*x[,"n_26"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_05"] - x[,"n_06"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_15"] - x[,"n_16"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_25"] - x[,"n_26"])*log(5) + 2*(x[,"n_06"] + x[,"n_20"])*log(pmax(1e-6,1 - r_r)) + 2*(x[,"n_00"] + x[,"n_26"])*log(pmax(1e-6,r_r)) + (x[,"n_10"] + x[,"n_16"])*(log(pmax(1e-6,1 - r_r)) + log(pmax(1e-6,r_r))) + (x[,"n_01"] + x[,"n_25"])*(log(pmax(1e-6,r_r)) + log(1 + 2*r_r)) + (x[,"n_11"] + x[,"n_15"])*log(3 + 8*r_r - 8*r_r^2) + (x[,"n_03"] + x[,"n_23"])*log(7 + 4*r_r - 4*r_r^2) + (x[,"n_04"] + x[,"n_22"])*log(36 - 32*r_r - r_r^2) + (x[,"n_02"] + x[,"n_24"])*log(3 + 34*r_r - r_r^2) + (x[,"n_12"] + x[,"n_14"])*log(30 - r_r + r_r^2) + (x[,"n_05"] + x[,"n_21"])*log(3 - 5*r_r + 2*r_r^2) + x[,"n_13"]*log(27 - 8*r_r + 8*r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.0_4.1 <- function(x){
logL_c <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24)*(2*log(2) + log(3) + log(5)) + (n04 + n21)*(log(pmax(1e-6,r)) + log(2 + r)) + (n11 + n14)*log(3 + 2*r - 2*r^2) + (n02 + n23)*log(6 - 2*r - r^2) + (n03 + n22)*log(3 + 4*r - r^2) + (n01 + n24)*log(3 - 4*r + r^2) + (n12 + n13)*log(15 - 2*r + 2*r^2)
return(L)}
interlogL_c <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
optimize(logL_c,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_c <- mapply(interlogL_c,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_c <- -((x[,"n_04"] + x[,"n_21"])*(-2*log10(2) + log10(5))) - (x[,"n_01"] + x[,"n_24"])*(-2*log10(2) + log10(5)) - (x[,"n_11"] + x[,"n_14"])*(-log10(2) + log10(7)) - (x[,"n_03"] + x[,"n_22"])*(-2*log10(2) + log10(19)) - (x[,"n_02"] + x[,"n_23"])*(-2*log10(2) + log10(19)) - (x[,"n_12"] + x[,"n_13"])*(-log10(2) + log10(29)) + (x[,"n_04"] + x[,"n_21"])*(log10(pmax(1e-6,r_c)) + log10(2 + r_c)) + (x[,"n_11"] + x[,"n_14"])*log10(3 + 2*r_c - 2*r_c^2) + (x[,"n_02"] + x[,"n_23"])*log10(6 - 2*r_c - r_c^2) + (x[,"n_03"] + x[,"n_22"])*log10(3 + 4*r_c - r_c^2) + (x[,"n_01"] + x[,"n_24"])*log10(3 - 4*r_c + r_c^2) + (x[,"n_12"] + x[,"n_13"])*log10(15 - 2*r_c + 2*r_c^2)
logL_c <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*(2*log(2) + log(3) + log(5)) + (x[,"n_04"] + x[,"n_21"])*(log(pmax(1e-6,r_c)) + log(2 + r_c)) + (x[,"n_11"] + x[,"n_14"])*log(3 + 2*r_c - 2*r_c^2) + (x[,"n_02"] + x[,"n_23"])*log(6 - 2*r_c - r_c^2) + (x[,"n_03"] + x[,"n_22"])*log(3 + 4*r_c - r_c^2) + (x[,"n_01"] + x[,"n_24"])*log(3 - 4*r_c + r_c^2) + (x[,"n_12"] + x[,"n_13"])*log(15 - 2*r_c + 2*r_c^2)
logL_m <- function(r,n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
L <- (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24)*(3*log(2) + log(5)) + (n01 + n24)*(log(2 - r) + log(pmax(1e-6,r))) + (n12 + n13)*log(9 + 2*r - 2*r^2) + (n04 + n21)*log(1 - r^2) + (n02 + n23)*log(3 + r^2) + (n03 + n22)*log(4 - 2*r + r^2) + (n11 + n14)*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_m <- function(n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24) {
optimize(logL_m,c(0,0.5), n01,n02,n03,n04,n11,n12,n13,n14,n21,n22,n23,n24, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_m <- mapply(interlogL_m,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"])
LOD_m <- (x[,"n_04"] + x[,"n_21"])*(2*log10(2) - log10(3)) - (x[,"n_01"] + x[,"n_24"])*(-2*log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_14"])*(-log10(2) + log10(5)) - (x[,"n_03"] + x[,"n_22"])*(-2*log10(2) + log10(13)) - (x[,"n_02"] + x[,"n_23"])*(-2*log10(2) + log10(13)) - (x[,"n_12"] + x[,"n_13"])*(-log10(2) + log10(19)) + (x[,"n_01"] + x[,"n_24"])*(log10(2 - r_m) + log10(pmax(1e-6,r_m))) + (x[,"n_12"] + x[,"n_13"])*log10(9 + 2*r_m - 2*r_m^2) + (x[,"n_04"] + x[,"n_21"])*log10(1 - r_m^2) + (x[,"n_02"] + x[,"n_23"])*log10(3 + r_m^2) + (x[,"n_03"] + x[,"n_22"])*log10(4 - 2*r_m + r_m^2) + (x[,"n_11"] + x[,"n_14"])*log10(3 - 2*r_m + 2*r_m^2)
logL_m <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*(3*log(2) + log(5)) + (x[,"n_01"] + x[,"n_24"])*(log(2 - r_m) + log(pmax(1e-6,r_m))) + (x[,"n_12"] + x[,"n_13"])*log(9 + 2*r_m - 2*r_m^2) + (x[,"n_04"] + x[,"n_21"])*log(1 - r_m^2) + (x[,"n_02"] + x[,"n_23"])*log(3 + r_m^2) + (x[,"n_03"] + x[,"n_22"])*log(4 - 2*r_m + r_m^2) + (x[,"n_11"] + x[,"n_14"])*log(3 - 2*r_m + 2*r_m^2)
logL_r <- function(r,n01,n02,n03,n04,n12,n13,n21,n22,n23,n24,n11,n14) {
L <- (-n01 - n02 - n03 - n04 - n12 - n13 - n21 - n22 - n23 - n24)*log(2) + (-n01 - n02 - n03 - n04 - n11 - n12 - n13 - n14 - n21 - n22 - n23 - n24)*log(5) + 2*(n04 + n21)*log(pmax(1e-6,1 - r)) + 2*(n01 + n24)*log(pmax(1e-6,r)) + (n11 + n14)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n23)*(log(2 - r) + log(pmax(1e-6,r))) + (n03 + n22)*log(1 - r^2) + (n12 + n13)*log(3 - 2*r + 2*r^2)
return(L)}
interlogL_r <- function(n01,n02,n03,n04,n12,n13,n21,n22,n23,n24,n11,n14) {
optimize(logL_r,c(0,0.5), n01,n02,n03,n04,n12,n13,n21,n22,n23,n24,n11,n14, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_r <- mapply(interlogL_r,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_11"],x[,"n_14"])
LOD_r <- 2*(x[,"n_11"] + x[,"n_14"])*log10(2) + 2*(x[,"n_04"] + x[,"n_21"])*log10(2) + 2*(x[,"n_01"] + x[,"n_24"])*log10(2) + (x[,"n_03"] + x[,"n_22"])*(2*log10(2) - log10(3)) - (x[,"n_02"] + x[,"n_23"])*(-2*log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_13"])*(-log10(2) + log10(5)) + 2*(x[,"n_04"] + x[,"n_21"])*log10(pmax(1e-6,1 - r_r)) + 2*(x[,"n_01"] + x[,"n_24"])*log10(pmax(1e-6,r_r)) + (x[,"n_11"] + x[,"n_14"])*(log10(pmax(1e-6,1 - r_r)) + log10(pmax(1e-6,r_r))) + (x[,"n_02"] + x[,"n_23"])*(log10(2 - r_r) + log10(pmax(1e-6,r_r))) + (x[,"n_03"] + x[,"n_22"])*log10(1 - r_r^2) + (x[,"n_12"] + x[,"n_13"])*log10(3 - 2*r_r + 2*r_r^2)
logL_r <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(5) + 2*(x[,"n_04"] + x[,"n_21"])*log(pmax(1e-6,1 - r_r)) + 2*(x[,"n_01"] + x[,"n_24"])*log(pmax(1e-6,r_r)) + (x[,"n_11"] + x[,"n_14"])*(log(pmax(1e-6,1 - r_r)) + log(pmax(1e-6,r_r))) + (x[,"n_02"] + x[,"n_23"])*(log(2 - r_r) + log(pmax(1e-6,r_r))) + (x[,"n_03"] + x[,"n_22"])*log(1 - r_r^2) + (x[,"n_12"] + x[,"n_13"])*log(3 - 2*r_r + 2*r_r^2)

return(list(
r_mat = cbind(r_c,r_m,r_r,0.499),
LOD_mat = cbind(LOD_c,LOD_m,LOD_r,0),
logL_mat = cbind(logL_c,logL_m,logL_r,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","mixed","repulsion","unknown")
)
)
}

#' @rdname r6_functions
r6_2.1_1.5 <- function(x){
logL_cc <- function(r,n02,n04,n12,n13,n14,n22,n23,n24,n32,n34,n03,n33) {
L <- (-n02 - n04 - 2*n12 - 2*n13 - 2*n14 - 2*n22 - 2*n23 - 2*n24 - n32 - n34)*log(2) + 2*(-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34)*log(5) + (n02 + n34)*(log(pmax(1e-6,1 - r)) + log(3 - r)) + (n04 + n32)*(log(pmax(1e-6,r)) + log(2 + r)) + (n12 + n24)*log(13 - 5*r - 2*r^2) + (n14 + n22)*log(6 + 9*r - 2*r^2) + (n03 + n33)*log(1 + r - r^2) + (n13 + n23)*log(21 - 4*r + 4*r^2)
return(L)}
interlogL_cc <- function(n02,n04,n12,n13,n14,n22,n23,n24,n32,n34,n03,n33) {
optimize(logL_cc,c(0,0.5), n02,n04,n12,n13,n14,n22,n23,n24,n32,n34,n03,n33, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cc <- mapply(interlogL_cc,x[,"n_02"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_34"],x[,"n_03"],x[,"n_33"])
LOD_cc <- -((x[,"n_04"] + x[,"n_32"])*(-2*log10(2) + log10(5))) - (x[,"n_03"] + x[,"n_33"])*(-2*log10(2) + log10(5)) - (x[,"n_02"] + x[,"n_34"])*(-2*log10(2) + log10(5)) - (x[,"n_14"] + x[,"n_22"])*(log10(2) + log10(5)) - (x[,"n_12"] + x[,"n_24"])*(log10(2) + log10(5)) - (x[,"n_13"] + x[,"n_23"])*(2*log10(2) + log10(5)) + (x[,"n_02"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cc)) + log10(3 - r_cc)) + (x[,"n_04"] + x[,"n_32"])*(log10(pmax(1e-6,r_cc)) + log10(2 + r_cc)) + (x[,"n_12"] + x[,"n_24"])*log10(13 - 5*r_cc - 2*r_cc^2) + (x[,"n_14"] + x[,"n_22"])*log10(6 + 9*r_cc - 2*r_cc^2) + (x[,"n_03"] + x[,"n_33"])*log10(1 + r_cc - r_cc^2) + (x[,"n_13"] + x[,"n_23"])*log10(21 - 4*r_cc + 4*r_cc^2)
logL_cc <- (-x[,"n_02"] - x[,"n_04"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"] - x[,"n_32"] - x[,"n_34"])*log(2) + 2*(-x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(5) + (x[,"n_02"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_cc)) + log(3 - r_cc)) + (x[,"n_04"] + x[,"n_32"])*(log(pmax(1e-6,r_cc)) + log(2 + r_cc)) + (x[,"n_12"] + x[,"n_24"])*log(13 - 5*r_cc - 2*r_cc^2) + (x[,"n_14"] + x[,"n_22"])*log(6 + 9*r_cc - 2*r_cc^2) + (x[,"n_03"] + x[,"n_33"])*log(1 + r_cc - r_cc^2) + (x[,"n_13"] + x[,"n_23"])*log(21 - 4*r_cc + 4*r_cc^2)
logL_cr <- function(r,n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34) {
L <- (-n02 - n03 - n04 - 2*n12 - 2*n13 - 2*n14 - 2*n22 - 2*n23 - 2*n24 - n32 - n33 - n34)*log(2) + (-n02 - n03 - n04 - n12 - n13 - n14 - n22 - n23 - n24 - n32 - n33 - n34)*log(5) + (n02 + n04 + n32 + n34)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n13 + n23)*log(3 + 4*r - 4*r^2) + (n14 + n22)*log(3 - 3*r + 2*r^2) + (n03 + n33)*log(1 - 2*r + 2*r^2) + (n12 + n24)*log(2 - r + 2*r^2)
return(L)}
interlogL_cr <- function(n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34) {
optimize(logL_cr,c(0,0.5), n02,n03,n04,n12,n13,n14,n22,n23,n24,n32,n33,n34, maximum=TRUE, lower=0, upper=0.5)$maximum}
r_cr <- mapply(interlogL_cr,x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_32"],x[,"n_33"],x[,"n_34"])
LOD_cr <- -((x[,"n_14"] + x[,"n_22"])*log10(2)) - 2*(x[,"n_13"] + x[,"n_23"])*log10(2) - (x[,"n_12"] + x[,"n_24"])*log10(2) + (x[,"n_03"] + x[,"n_33"])*log10(2) + 2*(x[,"n_02"] + x[,"n_04"] + x[,"n_32"] + x[,"n_34"])*log10(2) + (x[,"n_02"] + x[,"n_04"] + x[,"n_32"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_13"] + x[,"n_23"])*log10(3 + 4*r_cr - 4*r_cr^2) + (x[,"n_1