% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/combined.R
\name{rfint}
\alias{rfint}
\title{rfint()}
\usage{
rfint(
  formula = formula,
  train_data = NULL,
  test_data = NULL,
  method = "Zhang",
  alpha = 0.1,
  symmetry = TRUE,
  seed = NULL,
  m_try = 2,
  num_trees = 500,
  min_node_size = 5,
  num_threads = parallel::detectCores(),
  calibrate = FALSE,
  Roy_method = "quantile",
  featureBias = FALSE,
  predictionBias = TRUE,
  Tung_R = 5,
  Tung_num_trees = 75,
  variant = 1,
  Ghosal_num_stages = 2,
  prop = 0.618,
  concise = TRUE,
  interval_type = "two-sided"
)
}
\arguments{
\item{formula}{Object of class formula or character describing the model to fit. Interaction terms supported only for numerical variables.}

\item{train_data}{Training data of class data.frame.}

\item{test_data}{Test data of class data.frame. Utilizes ranger::predict() to produce prediction intervals for test data.}

\item{method}{Choose what method to generate RF prediction intervals. Options are \code{method = c("Zhang", "quantile", "Romano", "Ghosal", "Roy", "Tung", "HDI")}. Defaults to \code{method = "Zhang"}.}

\item{alpha}{Significance level for prediction intervals. Defaults to \code{alpha = 0.1}.}

\item{symmetry}{True if constructing symmetric out-of-bag prediction intervals, False otherwise. Used only \code{method = "Zhang"}. Defaults to \code{symmetry = TRUE}.}

\item{seed}{Seed for random number generation. Currently not utilized.}

\item{m_try}{Number of variables to randomly select from at each split.}

\item{num_trees}{Number of trees used in the random forest.}

\item{min_node_size}{Minimum number of observations before split at a node.}

\item{num_threads}{The number of threads to use in parallel. Default is the current number of cores.}

\item{calibrate}{If \code{calibrate = TRUE}, intervals are calibrated to achieve nominal coverage. Currently uses quantiles to calibrate. Only for \code{method = "Roy"}.}

\item{Roy_method}{Interval method for \code{method = "Roy"}.
Options are \code{Roy_method = c("quantile", "HDI", "CHDI")}.}

\item{featureBias}{Remove feature bias. Only for \code{method = "Tung"}.}

\item{predictionBias}{Remove prediction bias. Only for \code{method = "Tung"}.}

\item{Tung_R}{Number of repetitions used in bias removal. Only for \code{method = "Tung"}.}

\item{Tung_num_trees}{Number of trees used in bias removal. Only for \code{method = "Tung"}.}

\item{variant}{Choose which variant to use. Options are \code{method = c("1", "2")}. Only for \code{method = "Ghosal"}.}

\item{Ghosal_num_stages}{Number of total stages. Only for \code{method = "Ghosal"}.}

\item{prop}{Proportion of training data to sample for each tree. Only for \code{method = "Ghosal"}.}

\item{concise}{If  concise = TRUE, only predictions output. Defaults to \code{concise = FALSE}.}

\item{interval_type}{Type of prediction interval to generate.
Options are \code{method = c("two-sided", "lower", "upper")}. Default is  \code{method = "two-sided"}.}
}
\value{
\item{\code{int}}{Default output. Includes prediction intervals for all methods in \code{methods}.}
   \item{\code{preds}}{Predictions for test data for all methods in \code{methods}. Output when \code{concise = FALSE}.}
}
\description{
Implements seven different random forest prediction interval methods.
}
\details{
The seven methods implemented are cited in the References section.
Additional information can be found within those references.
Each of these methods are implemented by utilizing the ranger package.
For \code{method = "Zhang"}, prediction intervals are generated using out-of-bag residuals.
\code{method = "Romano"} utilizes a split-conformal approach.
\code{method = "Roy"} uses a bag-of-predictors approach.
\code{method = "Ghosal"} performs boosting to reduce bias in the random forest, and estimates variance.
The authors provide multiple variants to their methodology.
\code{method = "Tung"} debiases feature selection and prediction. Prediction intervals are generated using quantile regression forests.
\code{method = "HDI"} delivers prediction intervals through highest-density interval regression forests.
\code{method = "quantile"} utilizes quantile regression forests.
}
\examples{
\donttest{
library(piRF)

#functions to get average length and average coverage of output
getPILength <- function(x){
#average PI length across each set of predictions
l <- x[,2] - x[,1]
avg_l <- mean(l)
return(avg_l)
}

getCoverage <- function(x, response){
  #output coverage for test data
  coverage <- sum((response >= x[,1]) * (response <= x[,2]))/length(response)
  return(coverage)
}

#import airfoil self noise dataset
data(airfoil)
method_vec <- c("quantile", "Zhang", "Tung", "Romano", "Roy", "HDI", "Ghosal")
#generate train and test data
ratio <- .975
nrow <- nrow(airfoil)
n <- floor(nrow*ratio)
samp <- sample(1:nrow, size = n)
train <- airfoil[samp,]
test <- airfoil[-samp,]

#generate prediction intervals
res <- rfint(pressure ~ . , train_data = train, test_data = test,
             method = method_vec,
             concise= FALSE,
             num_threads = 1)

#empirical coverage, and average prediction interval length for each method
coverage <- sapply(res$int, FUN = getCoverage, response = test$pressure)
coverage
length <- sapply(res$int, FUN = getPILength)
length

#get current mfrow setting
opar <- par(mfrow = c(2,2))

#plotting intervals and predictions
for(i in 1:7){
   col <- ((test$pressure >= res$int[[i]][,1]) *
   (test$pressure <= res$int[[i]][,2])-1)*(-1)+1
   plot(x = res$preds[[i]], y = test$pressure, pch = 20,
      col = "black", ylab = "true", xlab = "predicted", main = method_vec[i])
   abline(a = 0, b = 1)
   segments(x0 = res$int[[i]][,1], x1 = res$int[[i]][,2],
      y1 = test$pressure, y0 = test$pressure, lwd = 1, col = col)
}
par(opar)
}
}
\references{
\insertRef{breiman2001random}{piRF}

\insertRef{ghosal2018boosting}{piRF}

\insertRef{meinshausen2006quantile}{piRF}

\insertRef{romano2019conformalized}{piRF}

\insertRef{roy2019prediction}{piRF}

\insertRef{tung2014bias}{piRF}

\insertRef{zhang2019random}{piRF}

\insertRef{zhu2019hdi}{piRF}
}
\seealso{
\link[ranger]{ranger}

\link[rfinterval]{rfinterval}
}
\author{
Chancellor Johnstone

Haozhe Zhang
}
