% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ortho_optim.r
\name{ortho_optim}
\alias{ortho_optim}
\title{Orthogonality constrained optimization}
\usage{
ortho_optim(B, fn, grad = NULL, ..., maximize = FALSE, control = list(),
  maxitr = 500, verbose = FALSE)
}
\arguments{
\item{B}{Initial \code{B} values. Must be a matrix, and the columns are subject to the orthogonality constrains. Will be processed by Gram-Schmidt if not orthogonal}

\item{fn}{A function that calculate the objective function value. The first argument should be \code{B}. Returns a single value.}

\item{grad}{A function that calculate the gradient. The first argument should be \code{B}. Returns a matrix with the same dimension as \code{B}. If not specified, then numerical approximation is used.}

\item{...}{Arguments passed to \code{fn} and \code{grad}}

\item{maximize}{By default, the solver will try to minimize the objective function unless \code{maximize = TRUE}}

\item{control}{A list of tuning variables for optimization. \code{epsilon} is the size for numerically approximating the gradient. For others, see Wen and Yin (2013).}

\item{maxitr}{Maximum number of iterations}

\item{verbose}{Should information be displayed}
}
\value{
A \code{orthoDr} object; a list consisting of
\item{B}{The optimal \code{B} value}
\item{fn}{The final functional value}
\item{itr}{The number of iterations}
\item{converge}{convergence code}
}
\description{
A general purpose optimization solver with orthogonality constraint. The orthogonality constrained optimization method is a nearly direct translation from Wen and Yin (2010)'s Matlab code.
}
\examples{
# an eigen value problem
library(pracma)
set.seed(1)
n = 100; k = 6
A = matrix(rnorm(n*n), n, n)
A = t(A) \%*\% A
B = gramSchmidt(matrix(rnorm(n*k), n, k))$Q
fx <- function(B, A) -0.5 * sum(diag(t(B) \%*\% A \%*\% B ))
gx <- function(B, A) -A \%*\% B
fit = ortho_optim(B, fx, gx, A = A)
fx(fit$B, A)

# compare with the solution from the eigen function
sol = eigen(A)$vectors[, 1:k]
fx(sol, A)
}
\references{
Wen, Z. and Yin, W., "A feasible method for optimization with orthogonality constraints." Mathematical Programming 142.1-2 (2013): 397-434.
DOI: \url{https://doi.org/10.1007/s10107-012-0584-1}
}
