% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hash.R
\name{hashing}
\alias{hashing}
\alias{sha1}
\alias{hmac}
\alias{mac}
\alias{sha224}
\alias{sha256}
\alias{sha384}
\alias{sha512}
\alias{sha2}
\alias{md4}
\alias{md5}
\alias{blake2b}
\alias{blake2s}
\alias{ripemd160}
\alias{multihash}
\title{Vectorized hash/hmac functions}
\usage{
sha1(x, key = NULL)

sha224(x, key = NULL)

sha256(x, key = NULL)

sha384(x, key = NULL)

sha512(x, key = NULL)

sha2(x, size = 256, key = NULL)

md4(x, key = NULL)

md5(x, key = NULL)

blake2b(x, key = NULL)

blake2s(x, key = NULL)

ripemd160(x, key = NULL)

multihash(x, algos = c("md5", "sha1", "sha256", "sha384", "sha512"))
}
\arguments{
\item{x}{character vector, raw vector or connection object.}

\item{key}{string or raw vector used as the key for HMAC hashing}

\item{size}{must be equal to 224 256 384 or 512}

\item{algos}{string vector with names of hashing algorithms}
}
\description{
All hash functions either calculate a hash-digest for \code{key == NULL} or HMAC
(hashed message authentication code) when \code{key} is not \code{NULL}. Supported
inputs are binary (raw vector), strings (character vector) or a connection object.
}
\details{
The most efficient way to calculate hashes is by using input \link{connections},
such as a \link[base:connections]{file()} or \link[base:connections]{url()} object.
In this case the hash is calculated streamingly, using almost no memory or disk space,
regardless of the data size. When using a connection input in the \link{multihash}
function, the data is only read only once while streaming to multiple hash functions
simultaneously. Therefore several hashes are calculated simultanously, without the
need to store any data or download it multiple times.

Functions are vectorized for the case of character vectors: a vector with \code{n}
strings returns \code{n} hashes. When passing a connection object, the contents will
be stream-hashed which minimizes the amount of required memory. This is recommended
for hashing files from disk or network.

The sha2 family of algorithms (sha224, sha256, sha384 and sha512) is generally
recommended for sensitive information. While sha1 and md5 are usually sufficient for
collision-resistant identifiers, they are no longer considered secure for cryptographic
purposes.

In applications where hashes should be irreversible (such as names or passwords) it is
often recommended to use a random \emph{key} for HMAC hashing. This prevents attacks where
we can lookup hashes of common and/or short strings. See examples. A common special case
is adding a random salt to a large number of records to test for uniqueness within the
dataset, while simultaneously rendering the results incomparable to other datasets.

The \code{blake2b} and \code{blake2s} algorithms are only available if your system has
libssl 1.1 or newer.
}
\examples{
# Support both strings and binary
md5(c("foo", "bar"))
md5("foo", key = "secret")

hash <- md5(charToRaw("foo"))
as.character(hash, sep = ":")

# Compare to digest
digest::digest("foo", "md5", serialize = FALSE)

# Other way around
digest::digest(cars, skip = 0)
md5(serialize(cars, NULL))

# Stream-verify from connections (including files)
myfile <- system.file("CITATION")
md5(file(myfile))
md5(file(myfile), key = "secret")

\dontrun{check md5 from: http://cran.r-project.org/bin/windows/base/old/3.1.1/md5sum.txt
md5(url("http://cran.r-project.org/bin/windows/base/old/3.1.1/R-3.1.1-win.exe"))}

# Use a salt to prevent dictionary attacks
sha1("admin") # googleable
sha1("admin", key = "random_salt_value") #not googleable

# Use a random salt to identify duplicates while anonymizing values
sha256("john") # googleable
sha256(c("john", "mary", "john"), key = "random_salt_value")
}
\references{
Digest types: \url{https://www.openssl.org/docs/manmaster/man1/dgst.html}
}
