context("basic input output for neurons")

test_that("We can read neurons in rda or rds format", {
  rda=tempfile(fileext='.rda')
  rds=tempfile(fileext='.rds')
  on.exit(unlink(c(rda,rds)))
  
  n=Cell07PNs[[1]]
  save(n,file=rda)
  saveRDS(n,file=rds)
  expect_equivalent(n,read.neuron(rda))
  expect_equivalent(n,read.neuron(rds))
})

test_that("We can read neurons in swc format", {
  swc='../testdata/neuron/EBT7R.CNG.swc'
  expect_is(n<-read.neuron(swc),'neuron')
  expect_equal(n$NeuronName,'EBT7R.CNG.swc')
})

test_that("We can set the NeuronName field", {
  swc='../testdata/neuron/EBT7R.CNG.swc'
  n<-read.neuron(swc, NeuronName="rhubarb")
  expect_equal(n$NeuronName,'rhubarb')
  # check that we can use a user defined function to define the NeuronName
  nfun=function(x) sub("\\..*","",basename(x))
  n<-read.neuron(swc, NeuronName=nfun)
  expect_equal(n$NeuronName,'EBT7R')
})

test_that("We can read in neurons as a neuronlist",{
  expect_is(nl<-read.neurons(paths='../testdata/neuron/',pattern='\\.CNG\\.swc$',
               neuronnames=function(x) sub("\\..*","",basename(x))),'neuronlist')
  expect_equal(length(nl),2)
})

test_that("We can read hxlineset format neurons",{
  
  ebt7=structure(list(NeuronName = "EBT7R", InputFileName = "/GD/dev/R/nat/inst/testdata/neuron/EBT7R.am", 
      CreatedAt = structure(1390916617.76591, class = c("POSIXct", 
      "POSIXt")), NodeName = structure("mac1041-14.lmb.internal", .Names = "nodename"), 
      InputFileStat = structure(list(size = 9870, isdir = FALSE, 
          mode = structure(493L, class = "octmode"), mtime = structure(1390864527, class = c("POSIXct", 
          "POSIXt")), ctime = structure(1390864527, class = c("POSIXct", 
          "POSIXt")), atime = structure(1390864527, class = c("POSIXct", 
          "POSIXt")), uid = 501L, gid = 80L, uname = "jefferis", 
          grname = "admin"), .Names = c("size", "isdir", "mode", 
      "mtime", "ctime", "atime", "uid", "gid", "uname", "grname"
      ), row.names = "/GD/dev/R/nat/inst/testdata/neuron/EBT7R.am", class = "data.frame"), 
      InputFileMD5 = structure("9e5016e8722314537bb4344ab2877f03", .Names = "/GD/dev/R/nat/inst/testdata/neuron/EBT7R.am"), 
      NumPoints = 343L, StartPoint = 1, EndPoints = c(1L, 37L, 
      49L, 64L, 76L, 86L, 104L, 120L, 133L, 138L, 148L, 162L, 166L, 
      169L, 175L, 188L, 197L, 202L, 216L, 223L, 232L, 238L, 254L, 
      257L, 268L, 276L, 282L, 295L, 300L, 307L, 316L, 320L, 322L, 
      334L, 337L, 343L), BranchPoints = c(26L, 38L, 56L, 65L, 77L, 
      90L, 92L, 117L, 121L, 127L, 135L, 141L, 153L, 154L, 158L, 
      173L, 191L, 195L, 206L, 211L, 214L, 234L, 239L, 243L, 255L, 
      264L, 279L, 280L, 284L, 285L, 313L, 318L, 327L, 336L), NumSegs = 69L, 
      SegList = list(c(1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 
      14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26), c(26, 
      27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37), c(26, 38), c(38, 
      39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49), c(38, 50, 51, 
      52, 53, 54, 55, 56), c(56, 57, 58, 59, 60, 61, 62, 63, 64
      ), c(56, 65), c(65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 
      76), c(65, 77), c(77, 78, 79, 80, 81, 82, 83, 84, 85, 86), 
          c(77, 87, 88, 89, 90), c(90, 91, 92), c(92, 93, 94, 95, 
          96, 97, 98, 99, 100, 101, 102, 103, 104), c(92, 105, 
          106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 
          117), c(117, 118, 119, 120), c(117, 121), c(121, 122, 
          123, 124, 125, 126, 127), c(127, 128, 129, 130, 131, 
          132, 133), c(127, 134, 135), c(135, 136, 137, 138), c(135, 
          139, 140, 141), c(141, 142, 143, 144, 145, 146, 147, 
          148), c(141, 149, 150, 151, 152, 153), c(153, 154), c(154, 
          155, 156, 157, 158), c(158, 159, 160, 161, 162), c(158, 
          163, 164, 165, 166), c(154, 167, 168, 169), c(153, 170, 
          171, 172, 173), c(173, 174, 175), c(173, 176, 177, 178, 
          179, 180, 181, 182, 183, 184, 185, 186, 187, 188), c(121, 
          189, 190, 191), c(191, 192, 193, 194, 195), c(195, 196, 
          197), c(195, 198, 199, 200, 201, 202), c(191, 203, 204, 
          205, 206), c(206, 207, 208, 209, 210, 211), c(211, 212, 
          213, 214), c(214, 215, 216), c(214, 217, 218, 219, 220, 
          221, 222, 223), c(211, 224, 225, 226, 227, 228, 229, 
          230, 231, 232), c(206, 233, 234), c(234, 235, 236, 237, 
          238), c(234, 239), c(239, 240, 241, 242, 243), c(243, 
          244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254
          ), c(243, 255), c(255, 256, 257), c(255, 258, 259, 260, 
          261, 262, 263, 264), c(264, 265, 266, 267, 268), c(264, 
          269, 270, 271, 272, 273, 274, 275, 276), c(239, 277, 
          278, 279), c(279, 280), c(280, 281, 282), c(280, 283, 
          284), c(284, 285), c(285, 286, 287, 288, 289, 290, 291, 
          292, 293, 294, 295), c(285, 296, 297, 298, 299, 300), 
          c(284, 301, 302, 303, 304, 305, 306, 307), c(279, 308, 
          309, 310, 311, 312, 313), c(313, 314, 315, 316), c(313, 
          317, 318), c(318, 319, 320), c(318, 321, 322), c(90, 
          323, 324, 325, 326, 327), c(327, 328, 329, 330, 331, 
          332, 333, 334), c(327, 335, 336), c(336, 337), c(336, 
          338, 339, 340, 341, 342, 343)), d = structure(list(PointNo = 1:343, 
          Label = c(2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
          2, 2, 2, 2), X = c(12.75, 14.43, 15.84, 16.54, 17.53, 
          18.09, 18.65, 19.36, 19.5, 19.5, 19.22, 19.92, 20.76, 
          22.59, 24.28, 25.97, 28.36, 29.91, 32.72, 34.83, 37.36, 
          39.89, 42.28, 44.96, 46.93, 49.18, 51.99, 53.82, 55.23, 
          56.77, 58.04, 58.32, 58.32, 59.3, 61.84, 64.93, 66.2, 
          50.3, 49.46, 48.33, 47.49, 46.08, 45.66, 45.24, 45.94, 
          47.77, 48.61, 49.88, 50.02, 52.69, 54.38, 56.91, 59.45, 
          61.27, 62.54, 64.93, 65.49, 65.49, 65.78, 66.62, 65.92, 
          64.79, 63.67, 63.1, 66.06, 67.04, 69.85, 72.67, 76.89, 
          79.7, 82.37, 84.48, 86.59, 88.14, 89.41, 90.81, 69.43, 
          70.14, 71.12, 70.7, 69.43, 68.17, 67.32, 66.9, 67.18, 
          68.03, 70.98, 73.51, 75.62, 76.61, 79.28, 80.12, 79.42, 
          79.28, 79.98, 80.54, 81.25, 83.22, 84.62, 86.17, 90.11, 
          92.64, 94.05, 96.02, 81.67, 83.78, 86.03, 87.44, 89.69, 
          93.48, 97.56, 99.67, 102.21, 104.6, 106.28, 107.97, 108.96, 
          107.69, 106.28, 105.86, 110.93, 112.76, 114.02, 115.85, 
          117.82, 119.37, 120.63, 119.93, 118.94, 118.94, 118.1, 
          116.27, 115.01, 122.46, 123.87, 123.16, 122.46, 121.9, 
          125.84, 127.67, 128.93, 128.23, 127.95, 127.67, 126.96, 
          126.82, 126.12, 125.56, 130.76, 132.87, 134.56, 137.23, 
          138.64, 139.62, 141.31, 143.98, 145.39, 147.08, 148.34, 
          149.05, 149.47, 149.19, 148.34, 149.61, 150.73, 151.16, 
          140.47, 141.31, 141.87, 139.34, 142.15, 144.12, 145.25, 
          145.67, 145.67, 146.51, 147.92, 149.19, 150.73, 152, 
          153.12, 152.56, 151.58, 150.17, 149.47, 148.34, 147.5, 
          146.23, 112.33, 114.16, 115.01, 114.44, 112.05, 110.79, 
          110.08, 110.36, 110.5, 110.22, 109.52, 108.54, 107.69, 
          106.57, 115.85, 116.98, 118.24, 118.52, 117.26, 116.41, 
          115.15, 113.88, 113.18, 112.47, 111.07, 110.22, 111.35, 
          111.35, 109.66, 108.82, 107.55, 106.85, 105.44, 103.89, 
          103.33, 115.57, 116.27, 117.96, 119.93, 121.9, 124.01, 
          125.41, 127.95, 128.37, 119.65, 120.35, 119.79, 120.63, 
          121.62, 121.76, 121.48, 123.59, 125.27, 126.4, 127.1, 
          125.98, 125.84, 124.57, 124.15, 124.43, 125.13, 126.26, 
          127.24, 128.09, 130.76, 132.17, 128.09, 127.95, 127.38, 
          130.48, 132.31, 133.29, 135.96, 137.09, 138.92, 138.92, 
          137.51, 136.39, 136.1, 136.1, 140.04, 140.47, 141.59, 
          142.15, 142.72, 143.42, 144.4, 144.54, 122.32, 123.87, 
          124.57, 124.99, 124.57, 123.73, 126.12, 127.38, 127.95, 
          129.21, 130.48, 132.45, 133.43, 134.84, 136.81, 138.21, 
          139.06, 139.9, 140.47, 129.21, 130.9, 132.31, 133.99, 
          135.12, 126.82, 126.4, 124.99, 123.73, 122.74, 123.3, 
          123.59, 126.4, 129.35, 131.46, 132.73, 134.7, 137.51, 
          138.78, 139.48, 139.2, 139.48, 140.04, 140.47, 140.61, 
          141.45, 142.01, 77.45, 79.14, 80.83, 82.51, 83.22, 82.79, 
          82.37, 82.23, 84.34, 86.59, 88.7, 89.55, 84.62, 85.75, 
          85.75, 87.86, 90.25, 92.5, 94.47, 97.28, 98.41), Y = c(-121.51, 
          -119.4, -117.57, -115.89, -114.62, -113.21, -111.24, 
          -109.27, -107.31, -105.76, -103.93, -101.96, -99.57, 
          -97.46, -96.19, -94.79, -93.8, -92.68, -91.27, -90, -89.44, 
          -89.02, -88.74, -88.74, -88.88, -89.02, -90.57, -92.54, 
          -93.52, -95.07, -96.76, -99.01, -100.41, -102.38, -104.49, 
          -106.74, -107.31, -88.46, -86.63, -85.08, -83.67, -82.27, 
          -81, -79.17, -77.06, -76.92, -77.35, -78.89, -79.45, 
          -87.75, -86.91, -86.91, -88.32, -88.04, -87.33, -86.35, 
          -84.8, -83.82, -83.25, -82.55, -81.99, -81.56, -81.56, 
          -81.71, -86.21, -87.89, -91.13, -93.52, -95.21, -96.76, 
          -97.74, -100.41, -102.38, -103.79, -104.21, -104.63, 
          -85.08, -87.19, -87.61, -88.6, -89.72, -91.69, -94.08, 
          -97.32, -100.84, -102.1, -84.66, -83.67, -83.39, -83.11, 
          -82.13, -81.56, -80.16, -79.03, -78.05, -76.64, -76.08, 
          -75.24, -74.67, -75.09, -76.5, -77.35, -78.19, -80.72, 
          -80.72, -79.74, -79.6, -79.45, -79.45, -79.17, -79.03, 
          -78.89, -78.75, -78.75, -78.33, -78.05, -77.63, -76.92, 
          -76.22, -75.38, -77.63, -78.61, -79.45, -80.86, -81.56, 
          -82.41, -83.39, -82.41, -81.71, -81.14, -80.58, -79.88, 
          -79.31, -84.1, -84.8, -86.07, -86.49, -86.91, -85.36, 
          -86.21, -87.47, -87.33, -86.21, -84.8, -82.69, -80.58, 
          -78.33, -76.08, -86.91, -87.19, -88.46, -90, -90.71, 
          -90.15, -88.74, -87.61, -86.63, -85.78, -87.05, -89.02, 
          -91.13, -92.25, -84.24, -83.11, -82.69, -82.41, -89.72, 
          -89.58, -89.86, -92.25, -93.66, -93.94, -93.38, -94.08, 
          -95.07, -92.11, -91.41, -89.44, -87.75, -85.93, -84.24, 
          -82.27, -80.02, -77.06, -74.81, -72.98, -71.72, -69.89, 
          -76.08, -74.81, -74.11, -74.25, -74.95, -74.67, -74.81, 
          -73.41, -72.56, -75.8, -76.36, -76.5, -75.52, -74.95, 
          -72.84, -71.16, -68.34, -67.92, -67.78, -67.78, -67.5, 
          -67.64, -67.64, -67.5, -67.08, -66.51, -66.09, -65.25, 
          -67.22, -67.5, -67.22, -67.22, -67.64, -67.78, -67.78, 
          -67.64, -67.5, -67.64, -68.62, -69.19, -69.33, -68.76, 
          -68.76, -68.76, -67.22, -66.37, -65.81, -65.11, -64.69, 
          -64.69, -65.53, -66.09, -66.09, -66.65, -66.65, -69.05, 
          -70.17, -71.86, -73.13, -76.22, -77.35, -79.74, -81.56, 
          -82.97, -84.8, -85.36, -66.37, -65.25, -64.26, -66.37, 
          -66.23, -66.94, -67.78, -69.61, -71.44, -73.13, -73.97, 
          -75.24, -76.22, -76.64, -74.11, -75.8, -78.33, -80.58, 
          -82.41, -83.96, -85.22, -86.07, -63.84, -62.72, -62.58, 
          -61.59, -60.33, -59.76, -60.47, -59.34, -59.76, -61.17, 
          -60.89, -60.33, -59.9, -59.76, -60.18, -60.33, -61.03, 
          -62.01, -62.01, -58.78, -58.22, -58.64, -58.92, -59.48, 
          -58.64, -58.64, -58.78, -59.48, -60.04, -61.17, -61.31, 
          -62.72, -62.01, -61.31, -60.75, -60.47, -61.73, -62.86, 
          -63.84, -63.7, -62.15, -62.29, -62.29, -62.15, -62.86, 
          -63.28, -83.67, -84.1, -84.66, -85.64, -86.21, -87.47, 
          -88.32, -91.41, -93.52, -95.07, -95.07, -95.35, -86.35, 
          -86.49, -87.19, -86.49, -86.49, -87.75, -89.58, -90.29, 
          -90.85), Z = c(0, -2, -4, -6, -9, -12, -15, -18, -20, 
          -23, -27, -30, -33, -36, -37, -38, -39, -42, -44, -45, 
          -47, -48, -50, -51, -52, -55, -56, -57, -60, -59, -61, 
          -61, -61, -61, -61, -60, -60, -56, -57, -58, -59, -60, 
          -62, -63, -63, -66, -67, -67, -67, -56, -57, -57, -57, 
          -57, -56, -58, -59, -61, -63, -68, -70, -71, -69, -71, 
          -58, -58, -58, -60, -61, -61, -62, -62, -62, -64, -67, 
          -66, -59, -59, -62, -66, -68, -69, -69, -69, -71, -71, 
          -60, -60, -59, -59, -59, -59, -59, -60, -61, -62, -64, 
          -65, -66, -67, -67, -68, -70, -69, -59, -58, -57, -57, 
          -56, -54, -52, -51, -50, -49, -48, -47, -47, -47, -46, 
          -44, -47, -45, -44, -43, -42, -41, -40, -35, -34, -32, 
          -31, -30, -28, -38, -36, -33, -32, -31, -34, -32, -29, 
          -27, -26, -25, -23, -22, -21, -20, -29, -29, -29, -28, 
          -26, -26, -26, -26, -26, -27, -28, -31, -32, -35, -27, 
          -27, -27, -26, -23, -21, -18, -27, -26, -26, -25, -24, 
          -22, -25, -26, -27, -27, -27, -27, -26, -25, -24, -23, 
          -22, -21, -22, -46, -45, -41, -37, -36, -34, -34, -34, 
          -34, -33, -32, -29, -27, -26, -43, -41, -41, -41, -39, 
          -38, -36, -35, -35, -34, -33, -33, -32, -30, -32, -31, 
          -30, -30, -30, -29, -29, -32, -31, -29, -28, -26, -25, 
          -24, -25, -25, -42, -43, -41, -39, -38, -37, -43, -45, 
          -46, -47, -47, -48, -48, -50, -50, -51, -51, -51, -52, 
          -53, -54, -53, -49, -48, -47, -48, -48, -48, -49, -50, 
          -51, -51, -52, -53, -55, -54, -53, -52, -52, -51, -51, 
          -50, -49, -49, -44, -43, -42, -40, -40, -44, -38, -35, 
          -32, -31, -28, -25, -24, -24, -24, -25, -24, -24, -24, 
          -32, -31, -29, -28, -27, -34, -34, -33, -32, -31, -29, 
          -27, -40, -40, -40, -40, -40, -40, -41, -43, -47, -39, 
          -38, -35, -32, -39, -39, -62, -63, -65, -66, -67, -67, 
          -67, -68, -68, -70, -71, -70, -67, -68, -71, -70, -71, 
          -71, -70, -70, -69), W = c(1.27, 1.27, 1.27, 1.27, 1.27, 
          1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 
          1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 
          1.27, 1.27, 1.27, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 
          0.42, 0.42, 0.42, 0.42, 0.7, 1.13, 0.42, 0.42, 0.42, 
          0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.84, 
          0.84, 0.84, 0.84, 0.84, 0.84, 0.84, 0.42, 0.42, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.98, 0.14, 0.14, 0.14, 0.14, 
          0.14, 0.14, 0.14, 0.14, 0.14, 0.98, 0.98, 0.98, 0.42, 
          0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.84, 0.98, 
          0.98, 0.98, 0.98, 0.98, 0.98, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.42, 0.42, 0.42, 0.42, 0.42, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.7, 0.7, 0.7, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.7, 0.56, 0.56, 0.56, 
          0.56, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 1.41, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7)), .Names = c("PointNo", 
      "Label", "X", "Y", "Z", "W"), row.names = c(NA, -343L), class = "data.frame")), .Names = c("NeuronName", 
  "InputFileName", "CreatedAt", "NodeName", "InputFileStat", "InputFileMD5", 
  "NumPoints", "StartPoint", "EndPoints", "BranchPoints", "NumSegs", 
  "SegList", "d"), class = c("neuron", "list"))
  
  expect_error(read.neuron('../testdata/neuron/EBT7R.am'),
               regexp='Unable to identify')
})

test_that("we can read hxskel format neurons",{
  
  Neurites=neuron(NeuronName = "Neurites", NumPoints = 291L, StartPoint = 1L, 
      BranchPoints = c(98L, 256L, 272L), EndPoints = c(1L, 54L, 
      202L, 257L, 274L), NumSegs = 7L, SegList = list(c(1, 3, 4, 
      5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 
      21, 22, 2, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
      36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 
      51, 52, 53, 23, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 
      66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 
      81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 
      96, 97, 98), c(98, 54), c(98, 100, 101, 102, 103, 104, 105, 
      106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 
      118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129, 
      130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 
      142, 143, 144, 145, 146, 147, 99, 149, 150, 151, 152, 153, 
      154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 
      166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 
      178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 
      190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 
      148, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 
      214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 
      226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 
      238, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 
      250, 251, 252, 253, 254, 255, 256), c(256, 202), c(256, 258, 
      259, 260, 261, 262, 263, 264, 265, 266, 267, 268, 269, 270, 
      271, 272), c(272, 273, 257), c(272, 275, 276, 277, 278, 279, 
      280, 281, 282, 283, 284, 285, 286, 287, 288, 289, 290, 291, 
      274)), nTrees = 1L, d = structure(list(PointNo = 1:291, Label = c(2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2), X = c(374.888, 367.069, 374.368, 373.722, 
      373.54, 373.06, 372.566, 372.127, 371.78, 371.355, 371.035, 
      370.838, 370.688, 370.279, 370.065, 369.657, 369.463, 369.148, 
      368.773, 368.404, 367.915, 367.46, 353.928, 366.653, 366.212, 
      365.852, 365.353, 364.904, 364.445, 363.922, 363.397, 362.923, 
      362.431, 361.949, 361.538, 361.045, 360.716, 360.307, 359.857, 
      359.463, 359.048, 358.636, 358.247, 357.909, 357.534, 357.09, 
      356.569, 356.077, 355.622, 355.213, 354.876, 354.556, 354.252, 
      333.369, 353.573, 353.202, 352.765, 352.335, 351.875, 351.421, 
      350.954, 350.503, 350.076, 349.699, 349.23, 348.823, 348.355, 
      347.894, 347.434, 346.964, 346.519, 346.088, 345.621, 345.191, 
      344.733, 344.313, 343.89, 343.439, 342.975, 342.531, 342.051, 
      341.585, 341.088, 340.637, 340.216, 339.763, 339.265, 338.77, 
      338.271, 337.779, 337.29, 336.787, 336.276, 335.77, 335.291, 
      334.826, 334.337, 333.855, 314.925, 333.431, 332.992, 332.523, 
      332.064, 331.566, 331.112, 330.672, 330.289, 329.891, 329.427, 
      328.943, 328.547, 328.139, 327.73, 327.263, 326.831, 326.391, 
      325.979, 325.62, 325.355, 324.995, 324.667, 324.359, 324.053, 
      323.735, 323.446, 323.087, 322.742, 322.442, 322.189, 321.906, 
      321.57, 321.237, 320.855, 320.465, 320.063, 319.7, 319.292, 
      318.879, 318.48, 318.082, 317.704, 317.319, 316.905, 316.475, 
      316.045, 315.621, 315.262, 293.393, 314.521, 314.097, 313.667, 
      313.274, 312.86, 312.426, 311.999, 311.585, 311.14, 310.716, 
      310.291, 309.868, 309.473, 309.108, 308.732, 308.352, 307.986, 
      307.601, 307.212, 306.839, 306.433, 306.074, 305.695, 305.344, 
      305.029, 304.767, 304.485, 304.322, 303.999, 303.742, 303.387, 
      303.024, 302.64, 302.239, 301.846, 301.347, 300.981, 300.541, 
      300.122, 299.689, 299.236, 298.752, 298.364, 297.906, 297.534, 
      297.064, 296.574, 296.167, 295.678, 295.294, 294.802, 294.327, 
      293.862, 275.121, 292.979, 292.551, 292.082, 291.67, 291.281, 
      290.922, 290.559, 290.221, 289.845, 289.478, 289.128, 288.804, 
      288.504, 288.205, 287.898, 287.608, 287.347, 287.088, 286.884, 
      286.679, 286.425, 286.247, 286.058, 285.855, 285.57, 285.272, 
      284.972, 284.721, 284.576, 284.46, 284.221, 283.92, 283.517, 
      283.187, 282.757, 282.293, 281.837, 281.371, 280.97, 280.587, 
      280.223, 279.848, 279.444, 279.015, 278.557, 278.093, 277.641, 
      277.164, 276.729, 276.318, 275.959, 275.645, 275.467, 275.273, 
      268.316, 274.985, 274.684, 274.324, 273.95, 273.466, 273.011, 
      272.56, 272.122, 271.703, 271.273, 270.948, 270.647, 270.339, 
      269.921, 269.4, 268.847, 269.844, 269.25, 269.297, 269.457, 
      269.693, 269.842, 269.914, 269.975, 269.998, 270.004, 270.004, 
      270.016, 270.076, 270.148, 270.164, 270.134, 270.053, 269.952
      ), Y = c(127.498, 134.529, 127.91, 128.142, 128.732, 129.072, 
      129.291, 129.574, 129.973, 130.268, 130.671, 131.121, 131.576, 
      131.884, 132.313, 132.596, 133.006, 133.344, 133.633, 133.918, 
      134.038, 134.25, 141.004, 134.77, 134.832, 135.048, 135.129, 
      135.284, 135.427, 135.327, 135.326, 135.565, 135.806, 136.055, 
      136.402, 136.576, 136.97, 137.27, 137.475, 137.761, 137.995, 
      138.223, 138.474, 138.76, 139.007, 139.161, 139.258, 139.454, 
      139.667, 139.9, 140.188, 140.466, 140.748, 144.76, 141.07, 
      141.079, 141.028, 141.074, 141.077, 141.207, 141.391, 141.627, 
      141.889, 142.213, 142.355, 142.632, 142.791, 142.98, 143.181, 
      143.362, 143.584, 143.825, 143.991, 144.222, 144.314, 144.336, 
      144.445, 144.598, 144.775, 144.973, 145.076, 145.205, 145.151, 
      145.128, 145.099, 145.166, 145.118, 145.114, 145.121, 145.163, 
      145.167, 145.122, 145.106, 145.117, 144.984, 144.832, 144.796, 
      144.78, 132.196, 144.586, 144.405, 144.287, 144.112, 144.067, 
      143.929, 143.814, 143.593, 143.394, 143.345, 143.285, 143.041, 
      142.826, 142.584, 142.413, 142.206, 142.019, 141.737, 141.398, 
      140.983, 140.651, 140.291, 139.924, 139.559, 139.195, 138.793, 
      138.459, 138.112, 137.718, 137.289, 136.889, 136.541, 136.192, 
      135.892, 135.592, 135.299, 134.955, 134.669, 134.383, 134.109, 
      133.858, 133.599, 133.35, 133.113, 132.926, 132.761, 132.623, 
      132.417, 118.666, 132.05, 131.875, 131.696, 131.497, 131.345, 
      131.181, 131.079, 130.955, 130.861, 130.727, 130.591, 130.372, 
      130.103, 129.799, 129.519, 129.29, 129.051, 128.823, 128.543, 
      128.244, 128.004, 127.744, 127.441, 127.107, 126.75, 126.371, 
      126.005, 125.563, 125.197, 124.767, 124.393, 124.028, 123.677, 
      123.327, 122.961, 122.78, 122.412, 122.141, 121.865, 121.648, 
      121.473, 121.329, 121.03, 120.839, 120.505, 120.305, 120.158, 
      119.851, 119.705, 119.382, 119.272, 119.094, 118.883, 104.161, 
      118.403, 118.16, 118.009, 117.756, 117.471, 117.145, 116.817, 
      116.469, 116.167, 115.855, 115.531, 115.195, 114.832, 114.473, 
      114.196, 113.976, 113.711, 113.451, 113.188, 112.888, 112.588, 
      112.217, 111.876, 111.55, 111.339, 111.122, 110.939, 110.842, 
      110.675, 110.421, 110.144, 109.787, 109.507, 109.183, 109.025, 
      108.881, 108.722, 108.559, 108.269, 107.953, 107.623, 107.325, 
      107.117, 107.037, 106.931, 106.762, 106.549, 106.4, 106.154, 
      105.875, 105.555, 105.238, 104.867, 104.524, 102.149, 104.142, 
      103.75, 103.42, 103.083, 102.868, 102.637, 102.515, 102.415, 
      102.263, 102.174, 102.22, 102.307, 102.279, 102.152, 102.093, 
      102.156, 107.951, 102.476, 102.893, 103.305, 103.706, 104.127, 
      104.53, 104.874, 105.156, 105.423, 105.705, 105.992, 106.278, 
      106.555, 106.817, 107.076, 107.351, 107.644), Z = c(61.998, 
      64.303, 62.069, 62.07, 62.284, 62.274, 62.147, 62.031, 62.065, 
      62.097, 62.191, 62.366, 62.553, 62.613, 62.823, 62.974, 63.223, 
      63.461, 63.668, 63.891, 64.009, 64.134, 66.716, 64.483, 64.731, 
      65.029, 65.171, 65.387, 65.609, 65.631, 65.556, 65.504, 65.563, 
      65.589, 65.638, 65.562, 65.587, 65.555, 65.493, 65.525, 65.57, 
      65.667, 65.817, 66.041, 66.289, 66.543, 66.67, 66.729, 66.734, 
      66.714, 66.715, 66.71, 66.72, 65.032, 67.055, 67.383, 67.611, 
      67.856, 68.06, 68.244, 68.329, 68.308, 68.247, 68.2, 68.115, 
      68.059, 68.01, 67.972, 68.015, 67.979, 67.895, 67.792, 67.724, 
      67.613, 67.436, 67.166, 66.927, 66.787, 66.759, 66.641, 66.519, 
      66.353, 66.251, 66.024, 65.73, 65.497, 65.389, 65.269, 65.182, 
      65.067, 64.923, 64.843, 64.831, 64.829, 64.898, 64.969, 65.002, 
      65.038, 58.312, 64.847, 64.68, 64.522, 64.393, 64.303, 64.122, 
      63.901, 63.656, 63.416, 63.221, 63.095, 62.901, 62.704, 62.543, 
      62.467, 62.313, 62.154, 62.073, 61.966, 61.845, 61.721, 61.589, 
      61.437, 61.281, 61.153, 61.087, 60.977, 60.878, 60.814, 60.809, 
      60.734, 60.631, 60.532, 60.451, 60.401, 60.371, 60.339, 60.281, 
      60.232, 60.088, 59.909, 59.71, 59.524, 59.408, 59.272, 59.098, 
      58.881, 58.604, 53.007, 58.066, 57.882, 57.721, 57.505, 57.292, 
      57.14, 56.92, 56.681, 56.474, 56.235, 55.991, 55.803, 55.631, 
      55.472, 55.322, 55.117, 54.89, 54.675, 54.543, 54.421, 54.275, 
      54.058, 53.967, 53.864, 53.724, 53.527, 53.317, 53.11, 52.937, 
      52.788, 52.691, 52.585, 52.491, 52.499, 52.536, 52.554, 52.562, 
      52.604, 52.728, 52.881, 53.015, 53.033, 52.895, 52.773, 52.679, 
      52.679, 52.644, 52.671, 52.655, 52.772, 52.883, 52.946, 52.985, 
      44.643, 52.941, 52.88, 52.816, 52.706, 52.599, 52.505, 52.436, 
      52.341, 52.238, 52.126, 51.995, 51.827, 51.662, 51.476, 51.189, 
      50.844, 50.513, 50.18, 49.818, 49.492, 49.207, 48.95, 48.657, 
      48.352, 48.009, 47.676, 47.317, 46.889, 46.43, 45.996, 45.624, 
      45.383, 45.192, 44.947, 44.704, 44.549, 44.406, 44.321, 44.264, 
      44.268, 44.218, 44.1, 43.907, 43.673, 43.509, 43.442, 43.418, 
      43.389, 43.368, 43.415, 43.547, 43.766, 44.042, 44.342, 40.232, 
      44.142, 43.977, 43.783, 43.618, 43.573, 43.43, 43.181, 42.899, 
      42.608, 42.304, 41.875, 41.424, 40.954, 40.583, 40.343, 40.313, 
      33.754, 40.05, 39.77, 39.528, 39.328, 39.081, 38.771, 38.395, 
      37.969, 37.536, 37.113, 36.694, 36.279, 35.859, 35.424, 34.985, 
      34.56, 34.152), W = c(1.388, 2.782, 1.562, 1.582, 1.804, 
      1.994, 2.225, 2.517, 2.665, 2.612, 2.498, 2.337, 2.149, 2.115, 
      2.096, 2.165, 2.186, 2.272, 2.405, 2.483, 2.564, 2.688, 4.078, 
      2.772, 2.773, 2.756, 2.667, 2.443, 2.33, 2.182, 1.962, 1.948, 
      1.756, 1.696, 1.756, 1.681, 1.718, 1.692, 1.626, 1.684, 1.798, 
      1.992, 2.256, 2.664, 3.197, 3.621, 3.918, 4.21, 4.428, 4.558, 
      4.546, 4.436, 4.239, 4.527, 3.84, 3.883, 4.072, 4.135, 4.137, 
      4.001, 3.89, 3.81, 3.739, 3.701, 3.625, 3.569, 3.487, 3.468, 
      3.505, 3.666, 3.919, 4.13, 4.228, 4.114, 4.072, 4.114, 4.201, 
      4.28, 4.271, 4.184, 4.003, 3.854, 3.789, 3.754, 3.8, 3.896, 
      4.303, 4.598, 4.716, 4.534, 4.41, 4.331, 4.336, 4.358, 4.395, 
      4.455, 4.51, 4.52, 5.576, 4.37, 4.16, 4.027, 3.867, 3.817, 
      3.759, 3.697, 3.683, 3.695, 3.66, 3.622, 3.634, 3.727, 3.868, 
      4.121, 4.426, 4.606, 4.765, 4.664, 4.529, 4.289, 4.23, 4.249, 
      4.33, 4.437, 4.81, 4.984, 5.127, 5.274, 5.5, 5.607, 5.615, 
      5.627, 5.574, 5.557, 5.554, 5.608, 5.7, 5.884, 5.973, 5.966, 
      5.97, 5.966, 5.918, 5.83, 5.763, 5.596, 5.571, 3.689, 5.512, 
      5.423, 5.337, 5.5, 5.584, 5.72, 5.66, 5.818, 5.985, 6.223, 
      6.43, 6.657, 6.647, 6.397, 6.018, 5.635, 5.317, 5.102, 5.039, 
      4.92, 4.77, 4.518, 4.464, 4.485, 4.512, 4.53, 4.533, 4.452, 
      4.375, 4.322, 4.169, 4.081, 3.965, 3.996, 4.086, 4.123, 4.113, 
      4.046, 3.999, 3.893, 3.74, 3.619, 3.326, 3.179, 3.178, 3.233, 
      3.142, 3.164, 3.14, 3.267, 3.401, 3.538, 3.613, 2.862, 3.715, 
      3.743, 3.772, 3.803, 3.837, 3.874, 3.913, 3.956, 4, 4.047, 
      4.098, 4.152, 4.208, 4.263, 4.313, 4.35, 4.369, 4.364, 4.33, 
      4.267, 4.179, 4.069, 3.946, 3.818, 3.695, 3.587, 3.502, 3.444, 
      3.414, 3.408, 3.416, 3.428, 3.431, 3.417, 3.383, 3.328, 3.256, 
      3.176, 3.093, 3.015, 2.949, 2.897, 2.86, 2.839, 2.83, 2.829, 
      2.833, 2.839, 2.844, 2.848, 2.851, 2.855, 2.858, 2.861, 4.71, 
      3.006, 3.144, 3.273, 3.392, 3.505, 3.618, 3.736, 3.863, 3.999, 
      4.139, 4.277, 4.406, 4.517, 4.604, 4.664, 4.699, 3.977, 4.668, 
      4.662, 4.639, 4.594, 4.528, 4.442, 4.344, 4.242, 4.143, 4.057, 
      3.99, 3.948, 3.929, 3.929, 3.942, 3.958, 3.972), Parent = c(-1, 
      22, 1, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 
      18, 19, 20, 21, 53, 2, 24, 25, 26, 27, 28, 29, 30, 31, 32, 
      33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 
      48, 49, 50, 51, 52, 98, 23, 55, 56, 57, 58, 59, 60, 61, 62, 
      63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 
      78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 
      93, 94, 95, 96, 97, 147, 98, 100, 101, 102, 103, 104, 105, 
      106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 
      118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129, 
      130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 
      142, 143, 144, 145, 146, 201, 99, 149, 150, 151, 152, 153, 
      154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 
      166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 
      178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 
      190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 256, 
      148, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 
      214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 
      226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 
      238, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 
      250, 251, 252, 253, 254, 255, 273, 256, 258, 259, 260, 261, 
      262, 263, 264, 265, 266, 267, 268, 269, 270, 271, 272, 291, 
      272, 275, 276, 277, 278, 279, 280, 281, 282, 283, 284, 285, 
      286, 287, 288, 289, 290), NeighbourCount = c(1L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 3L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 1L, 2L, 
      2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 2L, 
      1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
      2L, 2L, 2L)), .Names = c("PointNo", "Label", "X", "Y", "Z", 
      "W", "Parent", "NeighbourCount"), class = "data.frame",
      row.names = c(NA, -291L)))
  expect_is(n<-read.neuron('../testdata/neuron/Neurites.am'),'neuron')
  g1<-as.ngraph(Neurites)
  g2<-as.ngraph(n)
  expect_true(igraph::graph.isomorphic(g1,g2))
  expect_equivalent(n,Neurites)
  tmpfile=tempfile(fileext='.wurgle')
  on.exit(unlink(tmpfile))
  file.copy('../testdata/neuron/Neurites.am',tmpfile)
  
  expect_equal(read.neuron(tmpfile),n)
})

test_that("we can identify amira hxskel neurons",{
  # hxlineset neuron
  expect_false(is.hxskel('../testdata/neuron/EBT7R.am'))
  # swc neuron
  expect_false(is.hxskel('../testdata/neuron/EBT7R.CNG.swc'))
  # hxskel neuron
  expect_true(is.hxskel('../testdata/neuron/Neurites.am'))
  # hxskel
  p='../testdata/neuron/Neurites.am'
  expect_true(is.hxskel(p,bytes=readBin(p,what=raw(),n=80)))
})
