% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/measures.R
\name{mt_calculate_measures}
\alias{mt_calculate_measures}
\title{Calculate mouse-tracking measures.}
\usage{
mt_calculate_measures(data, use = "trajectories", save_as = "measures",
  flip_threshold = 0, show_progress = TRUE)
}
\arguments{
\item{data}{a mousetrap data object created using one of the mt_import 
functions (see \link{mt_example} for details).}

\item{use}{a character string specifying which trajectory data should be 
used.}

\item{save_as}{a character string specifying where the calculated measures 
should be stored.}

\item{flip_threshold}{a numeric value specifying the distance that needs to 
be exceeded in one direction so that a change in direction counts as an x- 
or y-flip.}

\item{show_progress}{logical indicating whether function should report its 
progress.}
}
\value{
A mousetrap data object (see \link{mt_example}) where an 
  additional \link{data.frame} has been added (by default called "measures") 
  containing the per-trial mouse-tracking measures. Each row in the 
  data.frame corresponds to one trajectory (the corresponding trajectory is 
  by default identified in the column \link{mt_id}). Each column in the
  data.frame corresponds to one of the measures.
  
  Note that some measures are only returned if distance, velocity and
  acceleration are calculated using \link{mt_calculate_derivatives} before
  running \code{mt_calculate_measures}. Besides, the meaning of these
  measures depends on the values of the arguments in
  \link{mt_calculate_derivatives}.
  
  The following measures are computed for each trajectory: 
  \item{\link{mt_id}}{Trial ID (can be used for merging measures data.frame 
  with other trial-level data)}
  \item{x_max}{Maximum x-position} 
  \item{x_min}{Minimum x-position}
  \item{y_max}{Maximum y-position} 
  \item{y_min}{Minimum y-position}
  \item{MAD}{Maximum absolute deviation from the direct path connecting start
  and end point of the trajectory (straight line)}
  \item{MAD_time}{Time at which the maximum absolute deviation was reached
  first}
  \item{MD_above}{Maximum deviation above the direct path} 
  \item{MD_above_time}{Time at which the maximum deviation above was reached 
  first}
  \item{MD_below}{Maximum deviation below the direct path} 
  \item{MD_below_time}{Time at which the maximum deviation below was reached 
  first}
  \item{AD}{Average deviation from direct path}
  \item{AUC}{Area under curve, the geometric area between the actual
  trajectory and the direct path where areas below the direct path have been
  subtracted}
  \item{x_flips}{Number of directional changes along x-axis} 
  \item{y_flips}{Number of directional changes along y-axis} 
  \item{x_reversals}{Number of crossings of the y-axis} 
  \item{y_reversals}{Number of crossings of the y-axis}
  \item{RT}{Response time, the total time passed until a response was given}
  \item{initiation_time}{Time passed until first mouse movement was 
  initiated}
  \item{idle_time}{Total time without mouse movement across the entirety of
  the trial}
  \item{xy_dist}{Total distance covered by the trajectory}
  \item{vel_max}{Maximum velocity}
  \item{vel_max_time}{Time at which maximum velocity occurred first}
  \item{vel_min}{Minimum velocity} 
  \item{vel_min_time}{Time at which minimum velocity occurred first} 
  \item{acc_max}{Maximum acceleration}
  \item{acc_max_time}{Time at which maximum acceleration occurred first}
  \item{acc_min}{Minimum acceleration} 
  \item{acc_min_time}{Time where minimum acceleration occurred first}
}
\description{
Calculate a number of mouse-tracking measures for each trajectory.
}
\details{
The calculation of most measures can be deduced directly from their 
definition (see Value). For several more complex measures, a few details are 
provided in the following.

The \strong{maximum absolute deviation} (\code{MAD}) is the maximum 
perpendicular deviation from the straight path connecting start and end point
of the trajectory (e.g., Freeman & Ambady, 2010). If the \code{MAD} occurs
above the direct path, this is denoted by a positive value. If it occurs
below the direct path, this is denoted by a negative value. This assumes that
the complete movement in the trial was from bottom to top (i.e., the end
point has a higher y-position than the start point). In case the movement was
from top to bottom, \code{mt_calculate_measures} automatically flips the
signs. Both \code{MD_above} and  \code{MD_below} are also reported
separately. The \strong{average deviation} (\code{AD}) is the average of all
deviations across the trial.

The \code{AUC} represents the \strong{area under curve}, i.e., the geometric
area between the actual trajectory and the direct path. Areas above the
direct path are added and areas below are subtracted. The \code{AUC} is
calculated using the \link[pracma]{polyarea} function from the pracma
package.
}
\examples{
mt_example <- mt_calculate_derivatives(mt_example)
mt_example <- mt_calculate_measures(mt_example)

# Merge measures with trial data (adding "_raw"
# to columns already existing in the trial data)
mt_example_results <- merge(
  mt_example$data, mt_example$measures,
  by="mt_id",suffixes=c("_raw",""))
  
}
\references{
Mousetrap
  
  Freeman, J. B., & Ambady, N. (2010). MouseTracker: Software for studying 
  real-time mental processing using a computer mouse-tracking method. 
  \emph{Behavior Research Methods, 42}(1), 226-241.
}
\seealso{
\link{mt_sample_entropy} for calculating sample entropy.

\link{mt_movement_angle} for calculating the initial movement angle.

\link{mt_standardize} for standardizing the measures per subject.

\link{mt_check_bimodality} for checking bimodality of the measures using
different methods.

\link{mt_aggregate} and \link{mt_aggregate_per_subject} for aggregating the
measures.

\link{merge} for merging data.
}

