% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modeltime-recursive.R
\name{recursive}
\alias{recursive}
\title{Create a Recursive Time Series Model from a Parsnip or Workflow Regression Model}
\usage{
recursive(object, transform, train_tail, ...)
}
\arguments{
\item{object}{An object of \code{model_fit} or a fitted \code{workflow} class}

\item{transform}{A transformation performed on new_data after
each step of recursive algorithm. It can be an object of types:
\itemize{
\item \strong{Method 1, \code{recipe}:} A recipe generates lagged or sliding features (see examples)
\item \strong{Method 2, \code{function}:} Must have one argument \code{data} (see examples)
}}

\item{train_tail}{A tibble with tail of training data set.
In most cases it'll be required to create some variables
based on dependent variable.}

\item{...}{Not currently used.}
}
\value{
An object with added \code{recursive} class
}
\description{
Create a Recursive Time Series Model from a Parsnip or Workflow Regression Model
}
\details{
Recursive model can be used if some of the features used for training
is based of dependent variable we already are trying to forecast.
Typically, among these features we can find lags (e.g. created with \code{step_lag()})
or variables crated with sliding window.

When producing forecast, the following steps are performed:
\enumerate{
\item Computing forecast for first row of new data.
The first row cannot contain NA in any required column.
\item Filling i-th place of the dependent variable column with
already computed forecast.
\item Computing missing features for next step, based on
already calculated prediction. These features are computed
with on a tibble object made from binded \code{train_tail} (i.e. tail of
training data set) and \code{new_data} (which is an argument of predict function).
\item Jumping into point 2., and repeating rest of steps till the for-loop is ended.
}
}
\examples{
# Libraries & Setup ----
library(modeltime)
library(tidymodels)
library(tidyverse)
library(lubridate)
library(timetk)
library(slider)

m750

FORECAST_HORIZON <- 24

m750_extended <- m750 \%>\%
    group_by(id) \%>\%
    future_frame(
        .length_out = FORECAST_HORIZON,
        .bind_data  = TRUE
    ) \%>\%
    ungroup()

# METHOD 1: RECIPE ----
# - Used for recursive transformations via recipe prepeocessing steps

# Lag Recipe
recipe_lag <- recipe(value ~ date, m750_extended) \%>\%
    step_lag(value, lag = 1:FORECAST_HORIZON)

# Data Preparation
m750_lagged <- recipe_lag \%>\% prep() \%>\% juice()

m750_lagged

train_data <- m750_lagged \%>\%
    filter(!is.na(value)) \%>\%
    drop_na()

future_data <- m750_lagged \%>\%
    filter(is.na(value))

# Modeling
model_fit_lm <- linear_reg() \%>\%
    set_engine("lm") \%>\%
    fit(value ~ date, data = train_data)

model_fit_lm_recursive <- linear_reg() \%>\%
    set_engine("lm") \%>\%
    fit(value ~ ., data = train_data) \%>\%
    recursive(
        transform  = recipe_lag,
        train_tail = tail(train_data, FORECAST_HORIZON)
    )

model_fit_lm_recursive

# Forecasting
modeltime_table(
    model_fit_lm,
    model_fit_lm_recursive
) \%>\%
    update_model_description(2, "LM - Lag Roll") \%>\%
    modeltime_forecast(
        new_data    = future_data,
        actual_data = m750,
        keep_data   = TRUE
    ) \%>\%
    plot_modeltime_forecast(
        .interactive        = FALSE,
        .conf_interval_show = FALSE
    )


# METHOD 2: TRANSFORM FUNCTION ----
# - Used for complex transformations via transformation function

# Function run recursively that updates the forecasted dataset
lag_roll_transformer <- function(data){
    data \%>\%
        # Lags
        tk_augment_lags(value, .lags = 1:12) \%>\%
        # Rolling Features
        mutate(rolling_mean_12 = lag(slide_dbl(
            value, .f = mean, .before = 12, .complete = FALSE
        ), 1))
}

# Data Preparation
m750_rolling <- m750_extended \%>\%
    lag_roll_transformer() \%>\%
    select(-id)

train_data <- m750_rolling \%>\%
    filter(!is.na(value)) \%>\%
    drop_na()

future_data <- m750_rolling \%>\%
    filter(is.na(value))

# Modeling
model_fit_lm <- linear_reg() \%>\%
    set_engine("lm") \%>\%
    fit(value ~ date, data = train_data)

model_fit_lm_recursive <- linear_reg() \%>\%
    set_engine("lm") \%>\%
    fit(value ~ ., data = train_data) \%>\%
    recursive(
        transform  = lag_roll_transformer,
        train_tail = tail(train_data, FORECAST_HORIZON)
    )

model_fit_lm_recursive

# Forecasting
modeltime_table(
    model_fit_lm,
    model_fit_lm_recursive
) \%>\%
    update_model_description(2, "LM - Lag Roll") \%>\%
    modeltime_forecast(
        new_data    = future_data,
        actual_data = m750
    ) \%>\%
    plot_modeltime_forecast(
        .interactive        = FALSE,
        .conf_interval_show = FALSE
    )


}
