% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TunerSuccessiveHalving.R
\name{mlr_tuners_successive_halving}
\alias{mlr_tuners_successive_halving}
\alias{TunerSuccessiveHalving}
\title{Hyperparameter Tuning with Successive Halving}
\source{
Jamieson K, Talwalkar A (2016).
\dQuote{Non-stochastic Best Arm Identification and Hyperparameter Optimization.}
In Gretton A, Robert CC (eds.), \emph{Proceedings of the 19th International Conference on Artificial Intelligence and Statistics}, volume 51 series Proceedings of Machine Learning Research, 240-248.
\url{http://proceedings.mlr.press/v51/jamieson16.html}.
}
\description{
\code{TunerSuccessiveHalving} class that implements the successive halving
algorithm (SHA). SHA randomly samples \code{n} candidate
hyperparameter configurations and allocates a minimum budget (\code{r_min}) to all
candidates. The candidates are raced down in stages to a single best
candidate by repeatedly increasing the budget by a factor of \code{eta} and
promoting only the best \code{1 / eta } fraction to the next stage. This means
promising hyperparameter configurations are allocated a higher budget overall
and lower performing ones are discarded early on.

The budget hyperparameter must be tagged with \code{"budget"} in the search space.
The minimum budget (\code{r_min}) which is allocated in the base stage, is set by
the lower bound of the budget parameter. The upper bound  defines the maximum
budget (\code{r_max}) which is allocated to the candidates in the last stage. The
number of stages is computed so that each candidate in base stage is
allocated the minimum budget and the candidates in the last stage are not
evaluated on more than the maximum budget. The following table is the stage
layout for \code{eta = 2}, \code{r_min = 1} and \code{r_max = 8}.\tabular{rrr}{
   \code{i} \tab \code{n_i} \tab \code{r_i} \cr
   0 \tab 8 \tab 1 \cr
   1 \tab 4 \tab 2 \cr
   2 \tab 2 \tab 4 \cr
   3 \tab 1 \tab 8 \cr
}


\code{i} is stage number, \code{n_i} is the number of configurations and \code{r_i} is the
budget allocated to a single configuration.
}
\section{Subsample Budget}{

If the learner lacks a natural budget parameter,
\link[mlr3pipelines:mlr_pipeops_subsample]{mlr3pipelines::PipeOpSubsample} can be applied to use the subsampling rate
as budget parameter. The resulting \link[mlr3pipelines:mlr_learners_graph]{mlr3pipelines::GraphLearner} is fitted on
small proportions of the \link[mlr3:Task]{mlr3::Task} in the first stage, and on the complete
task in last stage.
}

\section{Parameters}{

\describe{
\item{\code{n}}{\code{integer(1)}\cr
Number of candidates in base stage.
}
\item{\code{eta}}{\code{numeric(1)}\cr
With every stage, the budget is increased by a factor of \code{eta}
and only the best \code{1 / eta} candidates are promoted to the next stage.
Non-integer values are supported, but \code{eta} is not allowed to be less or
equal 1.
}
\item{\code{sampler}}{\link[paradox:Sampler]{paradox::Sampler}\cr
Object defining how the samples of the parameter space should be drawn. The
default is uniform sampling.
}
\item{\code{repeats}}{\code{logical(1)}\cr
If \code{FALSE} (default), SHA terminates once all stages are evaluated.
Otherwise, SHA starts over again once the last stage is evaluated.
}
\item{\code{adjust_minimum_budget}}{\code{logical(1)}\cr
If \code{TRUE}, minimum budget is increased so that the last stage uses the
maximum budget defined in the search space.
}}
}

\section{Archive}{

The \link[mlr3tuning:ArchiveTuning]{mlr3tuning::ArchiveTuning} holds the following additional columns that
are specific to the successive halving algorithm:
\itemize{
\item \code{stage} (\verb{integer(1))}\cr
Stage index. Starts counting at 0.
\item \code{repetition} (\verb{integer(1))}\cr
Repetition index. Start counting at 1.
}
}

\section{Custom Sampler}{

Hyperband supports custom \link[paradox:Sampler]{paradox::Sampler} object for initial
configurations in each bracket.
A custom sampler may look like this (the full example is given in the
\emph{examples} section):

\if{html}{\out{<div class="sourceCode">}}\preformatted{# - beta distribution with alpha = 2 and beta = 5
# - categorical distribution with custom probabilities
sampler = SamplerJointIndep$new(list(
  Sampler1DRfun$new(params[[2]], function(n) rbeta(n, 2, 5)),
  Sampler1DCateg$new(params[[3]], prob = c(0.2, 0.3, 0.5))
))
}\if{html}{\out{</div>}}
}

\section{Progress Bars}{

\verb{$optimize()} supports progress bars via the package \CRANpkg{progressr}
combined with a \link{Terminator}. Simply wrap the function in
\code{progressr::with_progress()} to enable them. We recommend to use package
\CRANpkg{progress} as backend; enable with \code{progressr::handlers("progress")}.
}

\section{Parallelization}{

The hyperparameter configurations of one stage are evaluated in parallel with
the \CRANpkg{future} package. To select a parallel backend, use
\code{\link[future:plan]{future::plan()}}.
}

\section{Logging}{

Hyperband uses a logger (as implemented in \CRANpkg{lgr}) from package
\CRANpkg{bbotk}.
Use \code{lgr::get_logger("bbotk")} to access and control the logger.
}

\examples{
if(requireNamespace("xgboost")) {
  library(mlr3learners)

  # define hyperparameter and budget parameter
  search_space = ps(
    nrounds = p_int(lower = 1, upper = 16, tags = "budget"),
    eta = p_dbl(lower = 0, upper = 1),
    booster = p_fct(levels = c("gbtree", "gblinear", "dart"))
  )

  \donttest{
  # hyperparameter tuning on the pima indians diabetes data set
  instance = tune(
    method = "successive_halving",
    task = tsk("pima"),
    learner = lrn("classif.xgboost", eval_metric = "logloss"),
    resampling = rsmp("cv", folds = 3),
    measures = msr("classif.ce"),
    search_space = search_space,
    term_evals = 100
  )

  # best performing hyperparameter configuration
  instance$result
  }
}
}
\section{Super classes}{
\code{\link[mlr3tuning:Tuner]{mlr3tuning::Tuner}} -> \code{\link[mlr3tuning:TunerFromOptimizer]{mlr3tuning::TunerFromOptimizer}} -> \code{TunerSuccessiveHalving}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-TunerSuccessiveHalving-new}{\code{TunerSuccessiveHalving$new()}}
\item \href{#method-TunerSuccessiveHalving-clone}{\code{TunerSuccessiveHalving$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="format"><a href='../../mlr3tuning/html/Tuner.html#method-Tuner-format'><code>mlr3tuning::Tuner$format()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="help"><a href='../../mlr3tuning/html/Tuner.html#method-Tuner-help'><code>mlr3tuning::Tuner$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="print"><a href='../../mlr3tuning/html/Tuner.html#method-Tuner-print'><code>mlr3tuning::Tuner$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3tuning" data-topic="TunerFromOptimizer" data-id="optimize"><a href='../../mlr3tuning/html/TunerFromOptimizer.html#method-TunerFromOptimizer-optimize'><code>mlr3tuning::TunerFromOptimizer$optimize()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TunerSuccessiveHalving-new"></a>}}
\if{latex}{\out{\hypertarget{method-TunerSuccessiveHalving-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TunerSuccessiveHalving$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TunerSuccessiveHalving-clone"></a>}}
\if{latex}{\out{\hypertarget{method-TunerSuccessiveHalving-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TunerSuccessiveHalving$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
