% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mable-po-model.r
\name{maple.po}
\alias{maple.po}
\title{Mable fit of the PO model with given regression coefficients}
\usage{
maple.po(
  formula,
  data,
  M,
  g,
  tau,
  x0 = NULL,
  controls = mable.ctrl(),
  progress = TRUE
)
}
\arguments{
\item{formula}{regression formula. Response must be \code{cbind}. See 'Details'.}

\item{data}{a data frame containing variables in \code{formula}.}

\item{M}{a positive integer or a vector \code{(m0, m1)}. If \code{M = m} 
or \code{m0 = m1 = m},  then \code{m} is a preselected degree. 
If \code{m0 < m1} it specifies the set of consective candidate model degrees
  \code{m0:m1} for searching an optimal degree, where \code{m1-m0>3}.}

\item{g}{the given \eqn{d}-vector of regression coefficients}

\item{tau}{right endpoint of support \eqn{[0, \tau]} must be greater than 
or equal to the maximum observed time}

\item{x0}{a data frame specifying working baseline covariates on the right-hand-side of \code{formula}. See 'Details'.}

\item{controls}{Object of class \code{mable.ctrl()} specifying iteration limit
and other control options. Default is \code{\link{mable.ctrl}}.}

\item{progress}{if \code{TRUE} a text progressbar is displayed}
}
\value{
a class '\code{mable_reg}' object, a list with components
\itemize{
\item \code{M} the vector \code{(m0, m1)}, where \code{m1} is the last 
     candidate degree when the search stoped
\item \code{m} the selected optimal degree \code{m}
\item \code{p} the estimate of \eqn{p = (p_0, \dots, p_m,p_{m+1})}, 
      the coefficients of Bernstein polynomial of degree \code{m}
\item \code{coefficients} the given regression coefficients of the PH model
\item \code{mloglik} the maximum log-likelihood at an optimal degree \code{m}
\item \code{lk} log-likelihoods evaluated at \eqn{m \in \{m_0, \ldots, m_1\}}
\item \code{lr} likelihood ratios for change-points evaluated at 
   \eqn{m \in \{m_0+1, \ldots, m_1\}}
\item \code{tau.n} maximum observed time \eqn{\tau_n}
\item \code{tau} right endpoint of support \eqn{[0, \tau)}
\item \code{x0} the working baseline covariates
\item \code{egx0} the value of \eqn{e^{\gamma'x_0}}
\item \code{convergence} an integer code, 0 indicates successful 
   completion(the iteration is convergent), 1 indicates that 
  the maximum candidate degree had been reached in the calculation;
\item \code{delta} the final convergence criterion for EM iteration;
\item \code{chpts} the change-points among the candidate degrees;
\item \code{pom} the p-value of the selected optimal degree \code{m} 
      as a change-point;
}
}
\description{
Maximum approximate profile likelihood estimation of Bernstein
 polynomial model in proportional odds rate regression  based on interal
 censored event time data with given regression coefficients and select 
 an optimal degree m if coefficients are efficient
 estimates provided by other semiparametric methods.
}
\details{
Consider Generalized PO model with covariate for interval-censored failure time data:
\eqn{S(t|x) = S(t|x_0)^{\exp(\gamma'(x-x_0))}}, where \eqn{x_0} satisfies  
   \eqn{\gamma'(x-x_0)\ge 0}, where \eqn{x} and \eqn{x_0} may
contain dummy variables and interaction terms.  The working baseline \code{x0} in arguments
contains only the values of terms excluding dummy variables and interaction terms 
in the right-hand-side of \code{formula}. Thus \code{g} is the initial guess of 
the coefficients \eqn{\gamma} of \eqn{x-x_0} and could be longer than \code{x0}.
 Let \eqn{f(t|x)} and 
 \eqn{F(t|x) = 1-S(t|x)} be the density and cumulative distribution
functions of the event time given \eqn{X = x}, respectively.
Then \eqn{f(t|x_0)} on \eqn{[0,\tau_n]} can be approximated by
\eqn{f_m(t|x_0; p) = \tau^{-1}\sum_{i=0}^m p_i\beta_{mi}(t/\tau)},
where \eqn{p_i \ge 0}, \eqn{i = 0, \ldots, m}, \eqn{\sum_{i=0}^mp_i = 1},
\eqn{\beta_{mi}(u)} is the beta denity with shapes \eqn{i+1} and \eqn{m-i+1},
 and \eqn{\tau} is the right endpoint of support interval. So we can approximate  
   \eqn{S(t|x_0)} on \eqn{[0,\tau]} by
\eqn{S_m(t|x_0; p) = \sum_{i=0}^{m} p_i \bar B_{mi}(t/\tau)}, where
\eqn{\bar B_{mi}(u)}, \eqn{i = 0, \ldots, m}, is the beta survival function 
 with shapes \eqn{i+1} and \eqn{m-i+1}. 

Response variable should be of the form \code{cbind(l, u)}, where  
\code{(l, u)} is the interval containing the event time. Data are  
uncensored if \code{l = u}, right censored if \code{u = Inf} or   
\code{u = NA}, and left censored data if \code{l = 0}. The associated  
covariate contains \eqn{d} columns. The baseline \code{x0} should chosen 
so that \eqn{\gamma'(x-x_0)} is nonnegative for all the observed \eqn{x}.

The search for optimal degree \code{m} stops if either \code{m1} 
is reached or the test for change-point results in a p-value \code{pval} 
smaller than \code{sig.level}.
}
\examples{
\donttest{
## Simulated Weibull data
require(icenReg)
set.seed(111)
simdata<-simIC_weib(100, model = "po", inspections = 2, 
   inspectLength = 2.5, prob_cen=1)
sp<-ic_sp(cbind(l, u) ~ x1 + x2, data = simdata, model="po") 
gt<--sp$coefficients
res0<-maple.po(cbind(l, u) ~ x1 + x2, data = simdata, M=c(1,20), g=gt, tau=6)
op<-par(mfrow=c(1,2))
plot(res0,  which=c("likelihood","change-point"))
par(op)
res1<-mable.po(cbind(l, u) ~ x1 + x2, data = simdata, M=c(1,20),
   g=gt, tau=6, x0=data.frame(x1=max(simdata$x1),x2=-1))
op<-par(mfrow=c(2,2))    
plot(res1,  which=c("likelihood","change-point")) 
plot(res0, y=data.frame(x1=0,x2=0), which="density", add=FALSE, type="l",
    xlab="Time", main="Desnity Function")
plot(res1, y=data.frame(x1=0,x2=0), which="density", add=TRUE, lty=2, col=4)
lines(xx<-seq(0, 7, len=512), dweibull(xx, 2,2), lty=3, col=2, lwd=1.5)
legend("topright", bty="n", lty=1:3, col=c(1,4,2), c(expression(hat(f)[0]),
    expression(tilde(f)[0]), expression(f[0])))
plot(res0, y=data.frame(x1=0,x2=0), which="survival", add=FALSE, type="l",
    xlab="Time", main="Survival Function")
plot(res1, y=data.frame(x1=0,x2=0), which="survival", add=TRUE, lty=2, col=4)
lines(xx, 1-pweibull(xx, 2, 2), lty=2, col=2)
legend("topright", bty="n", lty=1:3, col=c(1,4,2), c(expression(hat(S)[0]),
    expression(tilde(S)[0]), expression(S[0])))
par(op)
}
}
\references{
Guan, Z. et al. (???) Maximum Approximate Bernstein Likelihood Estimation in 
  Generalized Proportional Odds Regression Model for Interval-Censored Data
}
\seealso{
\code{\link{mable.po}}
}
\author{
Zhong Guan <zguan@iu.edu>
}
\concept{interval censoring}
\concept{proportional odds model}
\keyword{distribution}
\keyword{models}
\keyword{nonparametric}
\keyword{regression}
\keyword{smooth}
\keyword{survival}
