% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllGenerics.R, R/aoristic.R
\docType{methods}
\name{aoristic}
\alias{aoristic}
\alias{aoristic-method}
\alias{aoristic,numeric,numeric-method}
\alias{aoristic,list,missing-method}
\title{Aoristic Analysis}
\usage{
aoristic(x, y, ...)

\S4method{aoristic}{numeric,numeric}(
  x,
  y,
  step = 1,
  start = min(x, na.rm = TRUE),
  stop = max(y, na.rm = TRUE),
  weight = TRUE,
  groups = NULL
)

\S4method{aoristic}{list,missing}(x, step = 1, start = NULL, stop = NULL, weight = FALSE, groups = NULL)
}
\arguments{
\item{x}{A \code{\link{numeric}} vector. If \code{y} is missing, must be a \code{\link{list}} (or a
\code{\link{data.frame}}) with \code{numeric} components (columns) \code{from} and \code{to}.}

\item{y}{A \code{\link{numeric}} vector. If missing, an attempt is made to interpret
\code{x} in a suitable way.}

\item{...}{Currently not used.}

\item{step}{A length-one \code{\link{integer}} vector giving the step size, i.e. the
width of each time step in the time series (in years CE; defaults to
\eqn{1} - i.e. annual level).}

\item{start}{A length-one \code{\link{numeric}} vector giving the beginning of the time
window (in years CE).}

\item{stop}{A length-one \code{\link{numeric}} vector giving the end of the time
window (in years CE).}

\item{weight}{A \code{\link{logical}} scalar: should the aoristic sum be weighted by
the length of periods. Defaults to \code{FALSE} (the aoristic sum
is the number of elements within a time step).}

\item{groups}{A \code{\link{factor}} vector in the sense that \code{as.factor(groups)}
defines the grouping. If \code{x} is a \code{list} (or a \code{data.frame}), \code{groups} can
be a length-one vector giving the index of the grouping component (column)
of \code{x}.}
}
\value{
An \linkS4class{AoristicSum} object.
}
\description{
Computes the aoristic sum.
}
\details{
Aoristic analysis is used to determine the probability of contemporaneity of
archaeological sites or assemblages. The aoristic analysis distributes the
probability of an event uniformly over each temporal fraction of the period
considered. The aoristic sum is then the distribution of the total number of
events to be assumed within this period.

Muller and Hinz (2018) pointed out that the overlapping of temporal
intervals related to period categorization and dating accuracy is likely to
bias the analysis. They proposed a weighting method to overcome this
problem. This method is not implemented here (for the moment), see the
\href{https://github.com/ISAAKiel/aoristAAR}{\pkg{aoristAAR} package}.
}
\examples{
## Aoristic Analysis
data("zuni", package = "folio")

## Set the start and end dates for each ceramic type
dates <- list(
  LINO = c(600, 875), KIAT = c(850, 950), RED = c(900, 1050),
  GALL = c(1025, 1125), ESC = c(1050, 1150), PUBW = c(1050, 1150),
  RES = c(1000, 1200), TULA = c(1175, 1300), PINE = c(1275, 1350),
  PUBR = c(1000, 1200), WING = c(1100, 1200), WIPO = c(1125, 1225),
  SJ = c(1200, 1300), LSJ = c(1250, 1300), SPR = c(1250, 1300),
  PINER = c(1275, 1325), HESH = c(1275, 1450), KWAK = c(1275, 1450)
)

## Keep only assemblages that have a sample size of at least 10
keep <- apply(X = zuni, MARGIN = 1, FUN = function(x) sum(x) >= 10)

## Calculate date ranges for each assemblage
span <- apply(
  X = zuni[keep, ],
  FUN = function(x, dates) {
    z <- range(unlist(dates[x > 0]))
    names(z) <- c("from", "to")
    z
  },
  MARGIN = 1,
  dates = dates
)

## Coerce to data.frame
span <- as.data.frame(t(span))

## Calculate aoristic sum (normal)
aorist_raw <- aoristic(span, step = 50, weight = FALSE)
plot(aorist_raw)

## Calculate aoristic sum (weights)
aorist_weigth <- aoristic(span, step = 50, weight = TRUE)
plot(aorist_weigth)

## Calculate aoristic sum (weights) by group
groups <- rep(c("A", "B", "C"), times = c(50, 90, 139))
aorist_groups <- aoristic(span, step = 50, weight = TRUE, groups = groups)
plot(aorist_groups)

## Rate of change
roc_weigth <- roc(aorist_weigth, n = 30)
plot(roc_weigth)

## Rate of change by group
roc_groups <- roc(aorist_groups, n = 30)
plot(roc_groups, facet = FALSE)
}
\references{
Crema, E. R. (2012). Modelling Temporal Uncertainty in Archaeological
Analysis. \emph{Journal of Archaeological Method and Theory}, 19(3): 440-61.
\doi{10.1007/s10816-011-9122-3}.

Johnson, I. (2004). Aoristic Analysis: Seeds of a New Approach to Mapping
Archaeological Distributions through Time. \emph{In} Ausserer, K. F., Börner, W.,
Goriany, M. & Karlhuber-Vöckl, L. (ed.), \emph{Enter the Past - The E-Way into
the Four Dimensions of Cultural Heritage}, Oxford: Archaeopress, p. 448-52.
BAR International Series 1227.
\doi{10.15496/publikation-2085}

Müller-Scheeßel, N. & Hinz, M. (2018). \emph{Aoristic Research in R: Correcting
Temporal Categorizations in Archaeology}. Presented at the Human History and
Digital Future (CAA 2018), Tubingen, March 21.
\url{https://www.youtube.com/watch?v=bUBukex30QI}.

Ratcliffe, J. H. (2000). Aoristic Analysis: The Spatial Interpretation of
Unspecific Temporal Events. \emph{International Journal of Geographical
Information Science}, 14(7): 669-79. \doi{10.1080/136588100424963}.

Ratcliffe, J. H. (2002). Aoristic Signatures and the Spatio-Temporal
Analysis of High Volume Crime Patterns. \emph{Journal of Quantitative
Criminology}, 18(1): 23-43. \doi{10.1023/A:1013240828824}.
}
\seealso{
\code{\link[=roc]{roc()}}, \link[=plot_aoristic]{plot()}

Other chronological analysis: 
\code{\link{apportion}()},
\code{\link{fit}()},
\code{\link{roc}()}
}
\author{
N. Frerebeau
}
\concept{chronological analysis}
