##################################################
# 'mNSeff': Modified Nash-sutcliffe Efficiency   #
##################################################
#    July 28th, 2009;  06-Sep-09                 #
##################################################
# Ref:
# Krause, P., Boyle, D. P., and Bäse, F.: Comparison of different efficiency criteria for hydrological model assessment, Adv. Geosci., 5, 89-97, 2005 
# Legates and McCabe, 1999. Evaluating the use of "goodness-of-fit" measures 
#                           in hydrologic and hydroclimatic model validation. 
#                           Water Resources Research. v35 i1. 233-241.

# Nash-Sutcliffe efficiency not "inflated" by squared values
# Essentially, the closer the model efficiency is to 1, the more accurate the model is.  

# 'obs' : numeric 'data.frame', 'matrix' or 'vector' with observed values
# 'sim' : numeric 'data.frame', 'matrix' or 'vector' with simulated values
# 'j'   : numeric, with the exponent to be used in the computation of the modified Nash-Sutcliffe effciency. The default value is j=1

# 'Result': Modified Nash-sutcliffe Efficiency between 'sim' and 'obs'

mNSeff <-function(sim, obs, ...) UseMethod("mNSeff")

mNSeff.default <- function (sim, obs, j=1, na.rm=TRUE, ...){ 

	 if ( is.na(match(class(sim), c("integer", "numeric", "ts", "zoo"))) |
          is.na(match(class(obs), c("integer", "numeric", "ts", "zoo")))
     ) stop("Invalid argument type: 'sim' & 'obs' have to be of class: c('integer', 'numeric', 'ts', 'zoo')")   
     
     # Checking that the provided exponent is positive
     if (j < 0 ) stop("Invalid argument: 'j' must be positive")           
   
     vi <- valindex(sim, obs)
	 
	 obs <- obs[vi]
	 sim <- sim[vi]
	 
	 denominator <- sum( abs(obs - mean(obs))^j )
	 
	 if (denominator != 0) {
	  
	 NS1 <- 1 - ( sum( abs(obs - sim)^j ) / denominator )
	 
	 } else stop("'sum(abs(obs - mean(obs))^j)=0', it is not possible to compute 'mNSeff'")  
	 
	 return(NS1)
     
} # 'mNSeff.default' end


mNSeff.matrix <- function (sim, obs, j=1, na.rm=TRUE, ...){ 

  NS1 <- rep(NA, ncol(obs))       
          
  NS1 <- sapply(1:ncol(obs), function(i,x,y) { 
                 NS1[i] <- mNSeff.default( x[,i], y[,i], j, na.rm=na.rm, ... )
               }, x=sim, y=obs )    
                     
  names(NS1) <- colnames(obs)
  return(NS1)
     
} # 'mNSeff.matrix' end


mNSeff.data.frame <- function (sim, obs, j=1, na.rm=TRUE, ...){ 
 
  sim <- as.matrix(sim)
  obs <- as.matrix(obs)
   
  mNSeff.matrix(sim, obs, j, na.rm=na.rm, ...)
     
} # 'mNSeff.data.frame' end
