/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include "hdf5.h"

#ifdef H5_HAVE_FLOCK

#include <sys/file.h>
#endif 

#ifdef H5_HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef H5_HAVE_WIN32_API

#define WIN32_LEAN_AND_MEAN 
#define NOGDI               

#include <windows.h>
#include <io.h>

#endif 

static bool H5FD_stdio_init_s = false;

static htri_t ignore_disabled_file_locks_s = -1;

static size_t H5_STDIO_MAX_IO_BYTES_g = (size_t)-1;

typedef enum {
    H5FD_STDIO_OP_UNKNOWN = 0,
    H5FD_STDIO_OP_READ    = 1,
    H5FD_STDIO_OP_WRITE   = 2,
    H5FD_STDIO_OP_SEEK    = 3
} H5FD_stdio_file_op;

typedef struct H5FD_stdio_t {
    H5FD_t             pub;          
    FILE              *fp;           
    int                fd;           
    haddr_t            eoa;          
    haddr_t            eof;          
    haddr_t            pos;          
    unsigned           write_access; 
    bool               ignore_disabled_file_locks;
    H5FD_stdio_file_op op; 
#ifndef H5_HAVE_WIN32_API
    
    dev_t device; 
    ino_t inode;  
#else
    
    DWORD nFileIndexLow;
    DWORD nFileIndexHigh;
    DWORD dwVolumeSerialNumber;

    HANDLE hFile; 
#endif 
} H5FD_stdio_t;

#ifdef H5_HAVE_WIN32_API

#define file_ftell _ftelli64
#else

#define file_ftell ftello
#endif

#if defined(H5_HAVE_WIN32_API) && !defined(H5_HAVE_MINGW)

#define file_fseek     _fseeki64
#define file_ftruncate _chsize_s
#else

#define file_fseek     fseeko
#define file_ftruncate ftruncate
#endif

#define MY_MAXADDR          (((haddr_t)1 << (8 * sizeof(HDoff_t) - 1)) - 1)
#define MY_ADDR_OVERFLOW(A) (HADDR_UNDEF == (A) || ((A) & ~(haddr_t)MY_MAXADDR))
#define MY_SIZE_OVERFLOW(Z) ((Z) & ~(hsize_t)MY_MAXADDR)
#define MY_REGION_OVERFLOW(A, Z)                                                                             \
    (MY_ADDR_OVERFLOW(A) || MY_SIZE_OVERFLOW(Z) || HADDR_UNDEF == (A) + (Z) ||                               \
     (HDoff_t)((A) + (Z)) < (HDoff_t)(A))

static H5FD_t *H5FD_stdio_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr);
static herr_t  H5FD_stdio_close(H5FD_t *lf);
static int     H5FD_stdio_cmp(const H5FD_t *_f1, const H5FD_t *_f2);
static herr_t  H5FD_stdio_query(const H5FD_t *_f1, unsigned long *flags);
static haddr_t H5FD_stdio_alloc(H5FD_t *_file, H5FD_mem_t type, hid_t dxpl_id, hsize_t size);
static haddr_t H5FD_stdio_get_eoa(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD_stdio_set_eoa(H5FD_t *_file, H5FD_mem_t type, haddr_t addr);
static haddr_t H5FD_stdio_get_eof(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD_stdio_get_handle(H5FD_t *_file, hid_t fapl, void **file_handle);
static herr_t  H5FD_stdio_read(H5FD_t *lf, H5FD_mem_t type, hid_t fapl_id, haddr_t addr, size_t size,
                               void *buf);
static herr_t  H5FD_stdio_write(H5FD_t *lf, H5FD_mem_t type, hid_t fapl_id, haddr_t addr, size_t size,
                                const void *buf);
static herr_t  H5FD_stdio_flush(H5FD_t *_file, hid_t dxpl_id, bool closing);
static herr_t  H5FD_stdio_truncate(H5FD_t *_file, hid_t dxpl_id, bool closing);
static herr_t  H5FD_stdio_lock(H5FD_t *_file, bool rw);
static herr_t  H5FD_stdio_unlock(H5FD_t *_file);
static herr_t  H5FD_stdio_delete(const char *filename, hid_t fapl_id);

const H5FD_class_t H5FD_stdio_g = {
    H5FD_CLASS_VERSION,    
    H5_VFD_STDIO,          
    "stdio",               
    MY_MAXADDR,            
    H5F_CLOSE_WEAK,        
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    0,                     
    NULL,                  
    NULL,                  
    NULL,                  
    0,                     
    NULL,                  
    NULL,                  
    H5FD_stdio_open,       
    H5FD_stdio_close,      
    H5FD_stdio_cmp,        
    H5FD_stdio_query,      
    NULL,                  
    H5FD_stdio_alloc,      
    NULL,                  
    H5FD_stdio_get_eoa,    
    H5FD_stdio_set_eoa,    
    H5FD_stdio_get_eof,    
    H5FD_stdio_get_handle, 
    H5FD_stdio_read,       
    H5FD_stdio_write,      
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    H5FD_stdio_flush,      
    H5FD_stdio_truncate,   
    H5FD_stdio_lock,       
    H5FD_stdio_unlock,     
    H5FD_stdio_delete,     
    NULL,                  
    H5FD_FLMAP_DICHOTOMY   
};

static herr_t
H5FD__stdio_init(void)
{
    char *lock_env_var = NULL; 

    
    lock_env_var = getenv(HDF5_USE_FILE_LOCKING);
    if (lock_env_var && !strcmp(lock_env_var, "BEST_EFFORT"))
        ignore_disabled_file_locks_s = 1; 
    else if (lock_env_var && (!strcmp(lock_env_var, "TRUE") || !strcmp(lock_env_var, "1")))
        ignore_disabled_file_locks_s = 0; 
    else
        ignore_disabled_file_locks_s = -1; 

    
    H5FD_stdio_init_s = true;

    return 0;
} 

herr_t
H5Pset_fapl_stdio(hid_t fapl_id)
{

    
    H5Eclear2(H5E_DEFAULT);

    if (0 == H5Pisa_class(fapl_id, H5P_FILE_ACCESS))
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_PLIST, H5E_BADTYPE, "not a file access property list", -1);

    return H5Pset_driver(fapl_id, H5FD_STDIO, NULL);
} 

static H5FD_t *
H5FD_stdio_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr)
{
    FILE         *f            = NULL;
    unsigned      write_access = 0; 
    H5FD_stdio_t *file         = NULL;
#ifdef H5_HAVE_WIN32_API
    struct _BY_HANDLE_FILE_INFORMATION fileinfo;
#else  
    struct stat sb;
#endif 

    
    assert(sizeof(HDoff_t) >= sizeof(size_t));

    
    (void)fapl_id;

    
    H5Eclear2(H5E_DEFAULT);

    
    if (!H5FD_stdio_init_s)
        if (H5FD__stdio_init() < 0)
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_VFL, H5E_CANTINIT, "can't initialize driver", NULL);

    
    if (!name || !*name)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_ARGS, H5E_BADVALUE, "invalid file name", NULL);
    if (0 == maxaddr || HADDR_UNDEF == maxaddr)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_ARGS, H5E_BADRANGE, "bogus maxaddr", NULL);
    if (MY_ADDR_OVERFLOW(maxaddr))
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_ARGS, H5E_OVERFLOW, "maxaddr too large", NULL);

    
    if (flags & H5F_ACC_RDWR)
        f = fopen(name, "rb+");
    else
        f = fopen(name, "rb");

    if (!f) {
        
        if (flags & H5F_ACC_CREAT) {
            assert(flags & H5F_ACC_RDWR);
            f            = fopen(name, "wb+");
            write_access = 1; 
        }
        else
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_CANTOPENFILE,
                        "file doesn't exist and CREAT wasn't specified", NULL);
    }
    else if (flags & H5F_ACC_EXCL) {
        
        assert(flags & H5F_ACC_CREAT);
        fclose(f);
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_FILEEXISTS,
                    "file exists but CREAT and EXCL were specified", NULL);
    }
    else if (flags & H5F_ACC_RDWR) {
        if (flags & H5F_ACC_TRUNC)
            f = freopen(name, "wb+", f);
        write_access = 1; 
    }                     
    

    if (!f)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_CANTOPENFILE, "fopen failed", NULL);

    
    if (NULL == (file = (H5FD_stdio_t *)calloc((size_t)1, sizeof(H5FD_stdio_t)))) {
        fclose(f);
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_RESOURCE, H5E_NOSPACE, "memory allocation failed", NULL);
    } 
    file->fp           = f;
    file->op           = H5FD_STDIO_OP_SEEK;
    file->pos          = HADDR_UNDEF;
    file->write_access = write_access; 
    if (file_fseek(file->fp, 0, SEEK_END) < 0) {
        file->op = H5FD_STDIO_OP_UNKNOWN;
    }
    else {
        HDoff_t x = file_ftell(file->fp);
        assert(x >= 0);
        file->eof = (haddr_t)x;
    }

    
    if (ignore_disabled_file_locks_s != -1)
        
        file->ignore_disabled_file_locks = ignore_disabled_file_locks_s;
    else {
        bool unused;

        
        if (H5Pget_file_locking(fapl_id, &unused, &file->ignore_disabled_file_locks) < 0) {
            free(file);
            fclose(f);
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_FILE, H5E_CANTGET,
                        "unable to get use disabled file locks property", NULL);
        }
    }

    
#ifdef H5_HAVE_WIN32_API
    file->fd = _fileno(file->fp);
#else  
    file->fd = fileno(file->fp);
#endif 
    if (file->fd < 0) {
        free(file);
        fclose(f);
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_FILE, H5E_CANTOPENFILE, "unable to get file descriptor", NULL);
    } 

#ifdef H5_HAVE_WIN32_API
    file->hFile = (HANDLE)_get_osfhandle(file->fd);
    if (INVALID_HANDLE_VALUE == file->hFile) {
        free(file);
        fclose(f);
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_FILE, H5E_CANTOPENFILE, "unable to get Windows file handle",
                    NULL);
    } 

    if (!GetFileInformationByHandle((HANDLE)file->hFile, &fileinfo)) {
        free(file);
        fclose(f);
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_FILE, H5E_CANTOPENFILE,
                    "unable to get Windows file descriptor information", NULL);
    } 

    file->nFileIndexHigh       = fileinfo.nFileIndexHigh;
    file->nFileIndexLow        = fileinfo.nFileIndexLow;
    file->dwVolumeSerialNumber = fileinfo.dwVolumeSerialNumber;
#else  
    if (fstat(file->fd, &sb) < 0) {
        free(file);
        fclose(f);
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_FILE, H5E_BADFILE, "unable to fstat file", NULL);
    } 
    file->device = sb.st_dev;
    file->inode  = sb.st_ino;
#endif 

    return ((H5FD_t *)file);
} 

static herr_t
H5FD_stdio_close(H5FD_t *_file)
{
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file;

    
    H5Eclear2(H5E_DEFAULT);

    if (fclose(file->fp) < 0)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_CLOSEERROR, "fclose failed", -1);

    free(file);

    return 0;
} 

static int
H5FD_stdio_cmp(const H5FD_t *_f1, const H5FD_t *_f2)
{
    const H5FD_stdio_t *f1 = (const H5FD_stdio_t *)_f1;
    const H5FD_stdio_t *f2 = (const H5FD_stdio_t *)_f2;

    
    H5Eclear2(H5E_DEFAULT);

#ifdef H5_HAVE_WIN32_API
    if (f1->dwVolumeSerialNumber < f2->dwVolumeSerialNumber)
        return -1;
    if (f1->dwVolumeSerialNumber > f2->dwVolumeSerialNumber)
        return 1;

    if (f1->nFileIndexHigh < f2->nFileIndexHigh)
        return -1;
    if (f1->nFileIndexHigh > f2->nFileIndexHigh)
        return 1;

    if (f1->nFileIndexLow < f2->nFileIndexLow)
        return -1;
    if (f1->nFileIndexLow > f2->nFileIndexLow)
        return 1;
#else  
    if (f1->device < f2->device)
        return -1;
    if (f1->device > f2->device)
        return 1;

    if (f1->inode < f2->inode)
        return -1;
    if (f1->inode > f2->inode)
        return 1;
#endif 

    return 0;
} 

static herr_t
H5FD_stdio_query(const H5FD_t *_f, unsigned long  *flags)
{
    
    (void)_f;

    
    if (flags) {
        *flags = 0;
        *flags |= H5FD_FEAT_AGGREGATE_METADATA;  
        *flags |= H5FD_FEAT_ACCUMULATE_METADATA; 
        *flags |= H5FD_FEAT_DATA_SIEVE; 
        *flags |= H5FD_FEAT_AGGREGATE_SMALLDATA;    
        *flags |= H5FD_FEAT_DEFAULT_VFD_COMPATIBLE; 
    }

    return 0;
} 

static haddr_t
H5FD_stdio_alloc(H5FD_t *_file, H5FD_mem_t  type, hid_t  dxpl_id, hsize_t size)
{
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file;
    haddr_t       addr;

    
    (void)type;
    (void)dxpl_id;

    
    H5Eclear2(H5E_DEFAULT);

    
    addr = file->eoa;

    file->eoa = addr + size;

    return addr;
} 

static haddr_t
H5FD_stdio_get_eoa(const H5FD_t *_file, H5FD_mem_t  type)
{
    const H5FD_stdio_t *file = (const H5FD_stdio_t *)_file;

    
    H5Eclear2(H5E_DEFAULT);

    
    (void)type;

    return file->eoa;
} 

static herr_t
H5FD_stdio_set_eoa(H5FD_t *_file, H5FD_mem_t  type, haddr_t addr)
{
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file;

    
    H5Eclear2(H5E_DEFAULT);

    
    (void)type;

    file->eoa = addr;

    return 0;
}

static haddr_t
H5FD_stdio_get_eof(const H5FD_t *_file, H5FD_mem_t  type)
{
    const H5FD_stdio_t *file = (const H5FD_stdio_t *)_file;

    
    (void)type;

    
    H5Eclear2(H5E_DEFAULT);

    
    (void)type;

    return (file->eof);
} 

static herr_t
H5FD_stdio_get_handle(H5FD_t *_file, hid_t  fapl, void **file_handle)
{
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file;

    
    (void)fapl;

    
    H5Eclear2(H5E_DEFAULT);

    *file_handle = &(file->fp);
    if (*file_handle == NULL)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_WRITEERROR, "get handle failed", -1);

    return 0;
} 

static herr_t
H5FD_stdio_read(H5FD_t *_file, H5FD_mem_t  type, hid_t  dxpl_id, haddr_t addr,
                size_t size, void  *buf)
{
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file;

    
    (void)type;
    (void)dxpl_id;

    
    H5Eclear2(H5E_DEFAULT);

    
    if (HADDR_UNDEF == addr)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_OVERFLOW, "file address overflowed", -1);
    if (MY_REGION_OVERFLOW(addr, size))
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_OVERFLOW, "file address overflowed", -1);

    
    if (0 == size)
        return 0;
    if ((haddr_t)addr >= file->eof) {
        memset(buf, 0, size);
        return 0;
    }

    
    if (!(file->op == H5FD_STDIO_OP_READ || file->op == H5FD_STDIO_OP_SEEK) || file->pos != addr) {
        if (file_fseek(file->fp, (HDoff_t)addr, SEEK_SET) < 0) {
            file->op  = H5FD_STDIO_OP_UNKNOWN;
            file->pos = HADDR_UNDEF;
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_SEEKERROR, "fseek failed", -1);
        }
        file->pos = addr;
    }

    
    if (addr + size > file->eof) {
        size_t nbytes = (size_t)(addr + size - file->eof);
        memset((unsigned char *)buf + size - nbytes, 0, nbytes);
        size -= nbytes;
    }

    
    while (size > 0) {

        size_t bytes_in   = 0; 
        size_t bytes_read = 0; 
        size_t item_size  = 1; 

        if (size > H5_STDIO_MAX_IO_BYTES_g)
            bytes_in = H5_STDIO_MAX_IO_BYTES_g;
        else
            bytes_in = size;

        bytes_read = fread(buf, item_size, bytes_in, file->fp);

        if (0 == bytes_read && ferror(file->fp)) { 
            file->op  = H5FD_STDIO_OP_UNKNOWN;
            file->pos = HADDR_UNDEF;
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_READERROR, "fread failed", -1);
        } 

        if (0 == bytes_read && feof(file->fp)) {
            
            memset((unsigned char *)buf, 0, size);
            break;
        } 

        size -= bytes_read;
        addr += (haddr_t)bytes_read;
        buf = (char *)buf + bytes_read;
    } 

    
    file->op  = H5FD_STDIO_OP_READ;
    file->pos = addr;

    return 0;
}

static herr_t
H5FD_stdio_write(H5FD_t *_file, H5FD_mem_t  type, hid_t  dxpl_id, haddr_t addr,
                 size_t size, const void *buf)
{
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file;

    
    (void)dxpl_id;
    (void)type;

    
    H5Eclear2(H5E_DEFAULT);

    
    if (HADDR_UNDEF == addr)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_OVERFLOW, "file address overflowed", -1);
    if (MY_REGION_OVERFLOW(addr, size))
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_OVERFLOW, "file address overflowed", -1);

    
    if ((file->op != H5FD_STDIO_OP_WRITE && file->op != H5FD_STDIO_OP_SEEK) || file->pos != addr) {
        if (file_fseek(file->fp, (HDoff_t)addr, SEEK_SET) < 0) {
            file->op  = H5FD_STDIO_OP_UNKNOWN;
            file->pos = HADDR_UNDEF;
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_SEEKERROR, "fseek failed", -1);
        }
        file->pos = addr;
    }

    
    while (size > 0) {

        size_t bytes_in    = 0; 
        size_t bytes_wrote = 0; 
        size_t item_size   = 1; 

        if (size > H5_STDIO_MAX_IO_BYTES_g)
            bytes_in = H5_STDIO_MAX_IO_BYTES_g;
        else
            bytes_in = size;

        bytes_wrote = fwrite(buf, item_size, bytes_in, file->fp);

        if (bytes_wrote != bytes_in || (0 == bytes_wrote && ferror(file->fp))) { 
            file->op  = H5FD_STDIO_OP_UNKNOWN;
            file->pos = HADDR_UNDEF;
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_WRITEERROR, "fwrite failed", -1);
        } 

        assert(bytes_wrote > 0);
        assert((size_t)bytes_wrote <= size);

        size -= bytes_wrote;
        addr += (haddr_t)bytes_wrote;
        buf = (const char *)buf + bytes_wrote;
    }

    
    file->op  = H5FD_STDIO_OP_WRITE;
    file->pos = addr;

    
    if (file->pos > file->eof)
        file->eof = file->pos;

    return 0;
}

static herr_t
H5FD_stdio_flush(H5FD_t *_file, hid_t  dxpl_id, bool closing)
{
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file;

    
    (void)dxpl_id;

    
    H5Eclear2(H5E_DEFAULT);

    
    if (file->write_access) {
        if (!closing) {
            if (fflush(file->fp) < 0)
                H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_WRITEERROR, "fflush failed", -1);

            
            file->pos = HADDR_UNDEF;
            file->op  = H5FD_STDIO_OP_UNKNOWN;
        } 
    }     

    return 0;
} 

static herr_t
H5FD_stdio_truncate(H5FD_t *_file, hid_t  dxpl_id, bool  closing)
{
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file;

    
    (void)dxpl_id;
    (void)closing;

    
    H5Eclear2(H5E_DEFAULT);

    
    if (file->write_access) {
        
        if (file->eoa != file->eof) {

#ifdef H5_HAVE_WIN32_API
            LARGE_INTEGER li;       
            DWORD         dwPtrLow; 
            DWORD dwError;          
            BOOL  bError;           

            
            rewind(file->fp);

            
            li.QuadPart = (LONGLONG)file->eoa;

            
            dwPtrLow = SetFilePointer(file->hFile, li.LowPart, &li.HighPart, FILE_BEGIN);
            if (INVALID_SET_FILE_POINTER == dwPtrLow) {
                dwError = GetLastError();
                if (dwError != NO_ERROR)
                    H5Epush_ret(__func__, H5E_ERR_CLS, H5E_FILE, H5E_FILEOPEN, "unable to set file pointer",
                                -1);
            }

            bError = SetEndOfFile(file->hFile);
            if (0 == bError)
                H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_SEEKERROR,
                            "unable to truncate/extend file properly", -1);
#else  
            
            rewind(file->fp);

            
            if (-1 == file_ftruncate(file->fd, (HDoff_t)file->eoa))
                H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_SEEKERROR,
                            "unable to truncate/extend file properly", -1);
#endif 

            
            file->eof = file->eoa;

            
            file->pos = HADDR_UNDEF;
            file->op  = H5FD_STDIO_OP_UNKNOWN;
        } 
    }     
    else {
        
        if (file->eoa > file->eof)
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_TRUNCATED, "eoa > eof!", -1);
    } 

    return 0;
} 

static herr_t
H5FD_stdio_lock(H5FD_t *_file, bool rw)
{
#ifdef H5_HAVE_FLOCK
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file; 
    int           lock_flags;                   

    
    H5Eclear2(H5E_DEFAULT);

    assert(file);

    
    lock_flags = rw ? LOCK_EX : LOCK_SH;

    
    if (flock(file->fd, lock_flags | LOCK_NB) < 0) {
        if (file->ignore_disabled_file_locks && ENOSYS == errno)
            
            errno = 0;
        else
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_VFL, H5E_CANTLOCKFILE, "file lock failed", -1);
    } 

    
    if (fflush(file->fp) < 0)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_WRITEERROR, "fflush failed", -1);

#endif 

    return 0;
} 

static herr_t
H5FD_stdio_unlock(H5FD_t *_file)
{
#ifdef H5_HAVE_FLOCK
    H5FD_stdio_t *file = (H5FD_stdio_t *)_file; 

    
    H5Eclear2(H5E_DEFAULT);

    assert(file);

    
    if (fflush(file->fp) < 0)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_IO, H5E_WRITEERROR, "fflush failed", -1);

    
    if (flock(file->fd, LOCK_UN) < 0) {
        if (file->ignore_disabled_file_locks && ENOSYS == errno)
            
            errno = 0;
        else
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_VFL, H5E_CANTUNLOCKFILE, "file unlock failed", -1);
    } 

#endif 

    return 0;
} 

static herr_t
H5FD_stdio_delete(const char *filename, hid_t  fapl_id)
{

    assert(filename);

    
    (void)fapl_id;

    
    H5Eclear2(H5E_DEFAULT);

    
    if (!H5FD_stdio_init_s)
        if (H5FD__stdio_init() < 0)
            H5Epush_ret(__func__, H5E_ERR_CLS, H5E_VFL, H5E_CANTINIT, "can't initialize driver", -1);

    if (remove(filename) < 0)
        H5Epush_ret(__func__, H5E_ERR_CLS, H5E_VFL, H5E_CANTDELETEFILE, "can't delete file)", -1);

    return 0;
} 

#ifdef H5private_H

#error "Do not use HDF5 private definitions"
#endif
