% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cloudbuild_schedule.R, R/cloudscheduler.R
\name{cr_build_schedule_http}
\alias{cr_build_schedule_http}
\alias{cr_schedule_http}
\alias{cr_schedule_pubsub}
\alias{cr_schedule}
\title{Create a Cloud Scheduler HTTP target from a Cloud Build object}
\usage{
cr_build_schedule_http(
  build,
  email = cr_email_get(),
  projectId = cr_project_get()
)

cr_schedule_http(build, email = cr_email_get(), projectId = cr_project_get())

cr_schedule_pubsub(
  topicName,
  PubsubMessage = NULL,
  data = NULL,
  attributes = NULL,
  projectId = cr_project_get()
)

cr_schedule(
  name,
  schedule = NULL,
  httpTarget = NULL,
  pubsubTarget = NULL,
  description = NULL,
  overwrite = FALSE,
  timeZone = Sys.timezone(),
  region = cr_region_get(),
  projectId = cr_project_get()
)
}
\arguments{
\item{build}{A \link{Build} object created via \link{cr_build_make} or \link{cr_build}}

\item{email}{The email that will authenticate the job set via \link{cr_email_set}}

\item{projectId}{The GCP project to run within usually set with \link{cr_project_set}}

\item{topicName}{The name of the Cloud Pub/Sub topic or a Topic object from \link[googlePubsubR]{topics_get}}

\item{PubsubMessage}{A \code{PubsubMessage} object generated via \link[googlePubsubR]{PubsubMessage}.  If used, then do not send in `data` or `attributes` arguments as will be redundant since this variable will hold the information.}

\item{data}{The message payload for PubsubMessage. An R object that will be turned into JSON via [jsonlite] and then base64 encoded into the PubSub format.}

\item{attributes}{Attributes for PubsubMessage.}

\item{name}{Name to call your scheduled job}

\item{schedule}{A cron schedule e.g. \code{"15 5 * * *"}}

\item{httpTarget}{A HTTP target object \link{HttpTarget}}

\item{pubsubTarget}{A Pub/Sub target object \link{PubsubTarget} such as created via \link{cr_schedule_pubsub}}

\item{description}{Optionally caller-specified in CreateJob or}

\item{overwrite}{If TRUE and an existing job with the same name exists, will overwrite it with the new parameters}

\item{timeZone}{Specifies the time zone to be used in interpreting schedule. If set to \code{NULL} will be "UTC". Note that some time zones include a provision for daylight savings time.}

\item{region}{The region usually set with \link{cr_region_set}}
}
\value{
\code{cr_schedule_http} returns a \link{HttpTarget} object for use in \link{cr_schedule}

\code{cr_schedule_pubsub} returns a \link{PubsubTarget} object for use within \link{cr_schedule} or \link{cr_schedule_build}

A \code{gar_scheduleJob} class object
}
\description{
This enables Cloud Scheduler to trigger Cloud Builds
}
\details{
Ensure you have a service email with \link{cr_email_set} of format \code{service-{project-number}@gcp-sa-cloudscheduler.iam.gserviceaccount.com} with Cloud Scheduler Service Agent role as per https://cloud.google.com/scheduler/docs/http-target-auth#add

You can parametrise builds by sending in values within PubSub. To read the data in the message set a substitution variable that picks up the data.  For example \code{_VAR1=$(body.message.data.var1)}

If your schedule to PubSub fails with a permission error, try turning the Cloud Scheduler API off and on again the Cloud Console, which will refresh the Google permissions.
}
\examples{
cloudbuild <- system.file("cloudbuild/cloudbuild.yaml", package = "googleCloudRunner")
build1 <- cr_build_make(cloudbuild)
build1
\dontrun{
cr_schedule("cloud-build-test1",
  schedule = "15 5 * * *",
  httpTarget = cr_schedule_http(build1)
)

# a cloud build you would like to schedule
itworks <- cr_build("cloudbuild.yaml", launch_browser = FALSE)

# once working, pass in the build to the scheduler
cr_schedule("itworks-schedule",
  schedule = "15 5 * * *",
  httpTarget = cr_schedule_http(itworks)
)
}
cr_project_set("my-project")
cr_bucket_set("my-bucket")
cloudbuild <- system.file("cloudbuild/cloudbuild.yaml",
  package = "googleCloudRunner"
)
bb <- cr_build_make(cloudbuild)
\dontrun{
# create a pubsub topic either in Google Console webUI or library(googlePubSubR)
library(googlePubsubR)
pubsub_auth()
topics_create("test-topic")
}

# create build trigger that will watch for messages to your created topic
pubsub_trigger <- cr_buildtrigger_pubsub("test-topic")
pubsub_trigger
\dontrun{
# create the build trigger with in-line build
cr_buildtrigger(bb, name = "pubsub-triggered", trigger = pubsub_trigger)


# create scheduler that calls the pub/sub topic
cr_schedule("cloud-build-pubsub",
  "15 5 * * *",
  pubsubTarget = cr_schedule_pubsub("test-topic")
)
}

# builds can be also parametrised to respond to parameters within your pubsub topic
# this cloudbuild echos back the value sent in 'var1'
cloudbuild <- system.file("cloudbuild/cloudbuild_substitutions.yml",
  package = "googleCloudRunner"
)
the_build <- cr_build_make(cloudbuild)

# var1 is sent via Pubsub to the buildtrigger
message <- list(var1 = "hello mum")
send_me <- jsonlite::base64_enc(jsonlite::toJSON(message))

# create build trigger that will work from pub/subscription
pubsub_trigger <- cr_buildtrigger_pubsub("test-topic")
\dontrun{
cr_buildtrigger(the_build, name = "pubsub-triggered-subs", trigger = pubsub_trigger)

# create scheduler that calls the pub/sub topic with a parameter
cr_schedule("cloud-build-pubsub",
  "15 5 * * *",
  pubsubTarget = cr_schedule_pubsub("test-topic",
    data = send_me
  )
)
}

\dontrun{
cr_project_set("my-project")
cr_region_set("europe-west1")
cr_schedule("test",
      "* * * * *",
      httpTarget = HttpTarget(uri="https://code.markedmondson.me"))

# schedule a cloud build (no source)
build1 <- cr_build_make("cloudbuild.yaml")
cr_schedule("cloud-build-test", "15 5 * * *",
             httpTarget = cr_schedule_http(build1))

# schedule a cloud build with code source from GCS bucket
my_gcs_source <- cr_build_upload_gcs("my_folder", bucket = cr_get_bucket())
build <- cr_build_make("cloudbuild.yaml", source = my_gcs_source)
cr_schedule("cloud-build-test2", "15 5 * * *",
            httpTarget = cr_schedule_http(build))

# update a schedule with the same name - only supply what you want to change
cr_schedule("cloud-build-test2", "12 6 * * *", overwrite=TRUE)

# By default will use the timezone as specified by Sys.timezone() - change
# this by supplying it directly
cr_schedule("timzone-utc", "12 2 * * *", timeZone = "UTC")

# schedule private Cloud Run app
# for authenticated Cloud Run apps - create with allowUnauthenticated=FALSE
cr_deploy_run("my-app", allowUnauthenticated = TRUE)

# deploying via R will help create a service email called my-app-invoker
cr_run_email("my-app")
#> "my-app-invoker@your-project.iam.gserviceaccount.com"

# schedule the endpoint
my_app <- cr_run_get("my-app")

endpoint <- paste0(my_app$status$url, "/fetch_stuff")

app_sched <- cr_run_schedule_http(endpoint, http_method = "GET",
                                  email = cr_run_email("my-app"))

cr_schedule("my-app-scheduled-1", schedule = "16 4 * * *",
            httpTarget = app_sched)


# creating build triggers that respond to pubsub events

\dontrun{
# create a pubsub topic either in webUI or via library(googlePubSubR)
library(googlePubsubR)
pubsub_auth()
topics_create("test-topic")
}

# create build trigger that will work from pub/subscription
pubsub_trigger <- cr_buildtrigger_pubsub("test-topic")
pubsub_trigger

\dontrun{
# create the build trigger with in-line build
cr_buildtrigger(bb, name = "pubsub-triggered", trigger = pubsub_trigger)
# create scheduler that calls the pub/sub topic

cr_schedule("cloud-build-pubsub",
            "15 5 * * *",
            pubsubTarget = cr_schedule_pubsub("test-topic"))

}

}
}
\seealso{
https://cloud.google.com/build/docs/api/reference/rest/v1/projects.builds/create

\href{https://cloud.google.com/scheduler/docs/reference/rest/v1/projects.locations.jobs/create}{Google Documentation for Cloud Scheduler}

Other Cloud Scheduler functions: 
\code{\link{HttpTarget}()},
\code{\link{Job}()},
\code{\link{PubsubTarget}()},
\code{\link{cr_run_schedule_http}()},
\code{\link{cr_schedule_delete}()},
\code{\link{cr_schedule_get}()},
\code{\link{cr_schedule_list}()},
\code{\link{cr_schedule_pause}()},
\code{\link{cr_schedule_run}()}

Other Cloud Scheduler functions: 
\code{\link{HttpTarget}()},
\code{\link{Job}()},
\code{\link{PubsubTarget}()},
\code{\link{cr_run_schedule_http}()},
\code{\link{cr_schedule_delete}()},
\code{\link{cr_schedule_get}()},
\code{\link{cr_schedule_list}()},
\code{\link{cr_schedule_pause}()},
\code{\link{cr_schedule_run}()}
}
\concept{Cloud Scheduler functions}
