% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read_plink.R
\name{read_plink}
\alias{read_plink}
\title{Read genotype and sample data in a Plink BED/BIM/FAM file set.}
\usage{
read_plink(file, verbose = TRUE)
}
\arguments{
\item{file}{Input file path, without extensions (each of .bed, .bim, .fam extensions will be added automatically as needed).
Alternatively, input file path may have .bed extension (but not .bim, .fam, or other extensions).}

\item{verbose}{If \code{TRUE} (default), function reports the paths of the files being read (after autocompleting the extensions).}
}
\value{
A named list with items in this order: \code{X} (genotype matrix, see description in return value of \code{\link[=read_bed]{read_bed()}}), \code{bim} (tibble, see \code{\link[=read_bim]{read_bim()}}), \code{fam} (tibble, see \code{\link[=read_fam]{read_fam()}}).
\code{X} has row and column names corresponding to the \code{id} values of the \code{bim} and \code{fam} tibbles.
}
\description{
This function reads a genotype matrix (\code{X}, encoded as reference allele dosages) and its associated locus (\code{bim}) and individual (\code{fam}) data tables in the three Plink files in BED, BIM, and FAM formats, respectively.
All inputs must exist or an error is thrown.
This function is a wrapper around the more basic functions
\code{\link[=read_bed]{read_bed()}},
\code{\link[=read_bim]{read_bim()}},
\code{\link[=read_fam]{read_fam()}},
which simplifies data parsing and additionally better guarantees data integrity.
Below suppose there are \code{m} loci and \code{n} individuals.
}
\examples{
# to read "data.bed" etc, run like this:
# obj <- read_plink("data")
# this also works
# obj <- read_plink("data.bed")
#
# you get a list with these three items:
# genotypes
# obj$X
# locus annotations
# obj$bim
# individual annotations
# obj$fam

# The following example is more awkward
# because package sample data has to be specified in this weird way:

# first get path to BED file
file <- system.file("extdata", 'sample.bed', package = "genio", mustWork = TRUE)

# read genotypes and annotation tables
plink_data <- read_plink(file)
# genotypes
plink_data$X
# locus annotations
plink_data$bim
# individual annotations
plink_data$fam

# the same works without .bed extension
file <- sub('\\\\.bed$', '', file) # remove extension
# it works!
plink_data <- read_plink(file)

}
\seealso{
\code{\link[=read_bed]{read_bed()}},
\code{\link[=read_bim]{read_bim()}}, and
\code{\link[=read_fam]{read_fam()}}
for individual parsers of each input table, including a description of each object returned.

\code{\link[=geno_to_char]{geno_to_char()}} for translating numerical genotypes into more human-readable character encodings.

Plink BED/BIM/FAM format reference:
\url{https://www.cog-genomics.org/plink/1.9/formats}
}
