% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fwb.R
\name{fwb}
\alias{fwb}
\alias{print.fwb}
\title{Fractional Weighted Bootstrap}
\usage{
fwb(
  data,
  statistic,
  R = 999,
  cluster = NULL,
  simple = FALSE,
  verbose = TRUE,
  cl = NULL,
  ...
)

\method{print}{fwb}(x, digits = getOption("digits"), index = 1L:ncol(x$t), ...)
}
\arguments{
\item{data}{the dataset used to compute the statistic}

\item{statistic}{a function, which, when applied to \code{data}, returns a vector containing the statistic(s) of interest. The function should take at least two arguments; the first argument should correspond to the dataset and the second argument should correspond to a vector of weights. Any further arguments can be passed to \code{statistic} through the \code{...} argument.}

\item{R}{the number of bootstrap replicates. Default is 999 but more is always better. For the percentile bootstrap confidence interval to be exact, it can be beneficial to use one less than a multiple of 100.}

\item{cluster}{optional; a vector containing cluster membership. If supplied, will run the cluster bootstrap. See Details. Evaluated first in \code{data} and then in the global environment.}

\item{simple}{\code{logical}; if \code{TRUE}, weights will be computed on-the-fly in each bootstrap replication rather than all at once. This can save memory at the cost of some speed.}

\item{verbose}{\code{logical}; whether to display a progress bar.}

\item{cl}{a cluster object created by \pkgfun{parallel}{makeCluster}, or an integer to indicate the number of child-processes (integer values are ignored on Windows) for parallel evaluations. See \pkgfun{pbapply}{pblapply} for details. If \code{NULL}, no parallelization will take place.}

\item{...}{other arguments passed to \code{statistic}.}

\item{x}{an \code{fwb} object; the output of a call to \code{fwb()}.}

\item{digits}{the number of significant digits to print}

\item{index}{the index or indices of the position of the quantity of interest in \code{x$t0} if more than one was specified in \code{fwb()}. Default is to print all quantities.}
}
\value{
A \code{fwb} object, which also inherits from \code{boot}, with the following components:
\item{t0}{The observed value of \code{statistic} applied to \code{data} with uniform weights.}
\item{t}{A matrix with \code{R} rows, each of which is a bootstrap replicate of the result of calling \code{statistic}.}
\item{R}{The value of \code{R} as passed to \code{fwb()}.}
\item{data}{The \code{data} as passed to \code{fwb()}.}
\item{seed}{The value of \code{.Random.seed} just prior to generating the weights (after the first call to \code{statistic} with uniform weights).}
\item{statistic}{The function \code{statistic} as passed to \code{fwb()}.}
\item{call}{The original call to \code{fwb()}.}
\item{cluster}{The vector passed to \code{cluster}, if any.}
}
\description{
\code{fwb()} implements the fractional (random) weighted bootstrap, also known as the Bayesian bootstrap. Rather than resampling units to include in bootstrap samples, weights are drawn to be applied to a weighted estimator.
}
\details{
\code{fwb()} implements the fractional weighted bootstrap and is meant to function as a drop-in for \code{boot::boot(., stype = "f")} (i.e., the usual bootstrap but with frequency weights representing the number of times each unit is drawn). In each bootstrap replication, the weights are sampled from independent exponential distributions with rate parameter 1 and then normalized to have a mean of 1, equivalent to drawing the weights from a Dirichlet distribution. The function supplied to \code{statistic} must incorporate the weights to compute a weighted statistic. For example, if the output is a regression coefficient, the weights supplied to the \code{w} argument of \code{statistic} should be supplied to the \code{weights} argument of \code{lm()}. These weights should be used any time frequency weights would be, since they are meant to function like frequency weights (which, in the case of the traditional bootstrap, would be integers). Unfortunately, there is no way for \code{fwb()} to know whether you are using the weights correctly, so care should be taken to ensure weights are correctly incorporated into the estimator.

When fitting binomial regression models (e.g., logistic) using \code{\link[=glm]{glm()}}, it may be useful to change the \code{family} to a "quasi" variety (e.g., \code{\link[=quasibinomial]{quasibinomial()}}) to avoid a spurious warning about "non-integer #successes".

The cluster bootstrap can be requested by supplying a vector of cluster membership to \code{cluster}. Rather than generating a weight for each unit, a weight is generated for each cluster and then applied to all units in that cluster.

Ideally, \code{statistic} should not involve a random element, or else it will not be straightforward to replicate the bootstrap results using the \code{seed} included in the output object. Setting a seed using \code{\link[=set.seed]{set.seed()}} is always advised.

The \code{print()} method displays the value of the statistics, the bias (the difference between the statistic and the mean of its bootstrap distribution), and the standard error (the standard deviation of the bootstrap distribution).
}
\section{Methods (by generic)}{
\itemize{
\item \code{print(fwb)}: Print an \code{fwb} object

}}
\examples{
\dontshow{if (requireNamespace("survival", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# Performing a Weibull analysis of the Bearing Cage
# failure data as done in Xu et al. (2020)
set.seed(123)
data("bearingcage")

weibull_est <- function(data, w) {
  fit <- survival::survreg(survival::Surv(hours, failure) ~ 1,
                           data = data, weights = w,
                           dist = "weibull")

  c(eta = unname(exp(coef(fit))), beta = 1/fit$scale)
}

boot_est <- fwb(bearingcage, statistic = weibull_est,
                R = 199, verbose = FALSE)
boot_est

#Get standard errors and CIs; uses bias-corrected
#percentile CI by default
summary(boot_est, ci.type = "bc")

#Plot statistic distributions
plot(boot_est, index = "beta", type = "hist")

\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=fwb.ci]{fwb.ci()}} for calculating confidence intervals; \code{\link[=summary.fwb]{summary.fwb()}} for displaying output in a clean way; \code{\link[=plot.fwb]{plot.fwb()}} for plotting the bootstrap distributions; \code{\link[=vcovFWB]{vcovFWB()}} for estimating the covariance matrix of estimates using the FWB; \pkgfun{boot}{boot} for the traditional bootstrap.
}
