#' Read a Lotek Acoustic Receiver File
#'
#' This function takes a raw acoustic detection file generated by a Lotek
#' JSATS receiver and processes it into a dataframe which can be used by
#' the filtering functions in this package.This is called within read_jsats().
#'
#' @param path the path to the folder containing the desired file
#' @param file the path of the desired file
#' @param timezone the Olsen Named time zone, default is "America/Los_Angeles"
#' @returns A dataframe converting the raw detection data into rows of detections
#' @export
#' @examples
#' # see read_ats or read_tekno for example usage
#'
read_lotek <- function(path, file, timezone="America/Los_Angeles"){
  LOT = read.csv(file.path(path,file),
                 colClasses = c(character(),
                                numeric(),
                                numeric(),
                                character(),
                                numeric()),
                 col.names = c("ReceiverSN","DateTime_Local","FS","Tag_Decimal","Tag_Hex","P"),
                 skip = 0)
  LOT$ReceiverSN <- as.numeric(gsub("WHS4K-","",LOT$ReceiverSN)) #Turn the file name into serial
  LOT$DateTime_Local =  as.POSIXct(LOT$DateTime_Local, #Convert serial DateTime to real DT
                                   tz = timezone,
                                   origin = "1899-12-30")
  LOT$FS = lubridate::seconds(LOT$FS) #Convert fractional seconds from number to seconds
  LOT$DateTime_Local = LOT$DateTime_Local+LOT$FS #Add them together
  LOT$Filename = stringr::str_split(file, pattern = '\\.')[[1]][1]
  LOT$Make = rep("Lotek", length(LOT$DateTime_Local))
  LOT <- dplyr::select(.data =  LOT, ReceiverSN, Make, DateTime_Local,
                       Tag_Decimal, Tag_Hex)
  LOT
}
